package Stat::ReportMaster;

=pod

    $Id$

=head1 NAME

    Stat::ReportMaster

=head1 DESCRIPTION

    Содержит функции необходимые для работы мастера отчетов статистики по аккаунту

=cut

use strict;
use warnings;
use Settings;
use Primitives;
use PrimitivesIds;
use User;
use HashingTools;
use RBACDirect;
use RBACElementary;
use RBAC2::Extended;
use Rbac;
use Campaign;
use Campaign::Types;
use Models::AdGroup;
use Models::AdGroupFilters;
use Client;
use GeoTools;
use Stat::Fields;
use Stat::Plot;
use Stat::CustomizedArray::StreamedExt;
use Stat::Const qw/:base %ATTRIBUTION_MODEL_TYPES/;
use Stat::Tools qw/norm_field_by_periods_suf periods_full_suffix_list periods_suffix_list/;
use Direct::Validation::Keywords;
use MinusWordsTools;

use Yandex::I18n;
use Yandex::DBTools;
use Yandex::DBShards;   
use Yandex::TimeCommon;
use Yandex::DateTime;
use Yandex::Validate;
use Yandex::ListUtils qw/xisect xminus xflatten/;
use Yandex::HashUtils;
use Yandex::Clone;
use Yandex::Trace;

use List::MoreUtils qw/uniq all any none first_index/;

use utf8;

# сколько шаблонов фильтров допускается сохранять на одного пользователя 
our $FILTERS_SETS_MAX_NUM = 100;
# сколько шаблонов отчетов допускается сохранять на одного пользователя 
our $REPORTS_MAX_NUM = 100;
# тип пользовательских опций, для мастера отчетов
our $USER_OPTION_TYPE = 'stat_report_master';
# название опции в user_typed_options со списком фильтров
our $FILTERS_OPTION_NAME = 'stat_master_filters';
# название опции в user_typed_options со списком отчетов
our $REPORTS_OPTION_NAME = 'stat_master_reports';
# сколько строк можем экспортировать в xls-файл (из-за ограничений по памяти)
our $XLS_ROWS_MAX_NUM = 1_000_000;
# поля (групировки, фильтры), которые представляют из себя регионы
my @REGION_FIELDS = qw/region physical_region/;
# поля (групировки, фильтры), которые представляют из себя ключевые слова
my @KEYWORD_FILTERS = qw/search_query phrase phrase_ext phrase_orig matched_phrase/;
# типы отчетов, различаемые на стороне серверсайда
my %SUPPORTED_STAT_TYPES = map {$_ => 1} qw/search_queries moc brand_safety brand_safety_mol/;
# Список символов, принимаемых для поиска по тексту в статистике
my $ALLOW_SEARCH_PHRASE_LETTERS = qq!${Settings::ALLOW_LETTERS}\\-\\"\\+\\ \!!;
# поддерживаемые фильтры для получения кампаний в режиме Мультиклиентности
my @SUPPORTED_FILTERS_FOR_GET_CIDS_IN_MULTI_CLIENTS_MODE = qw/client_id place_id page_id/;
# поля в базе по имени фильтра для получения кампаний в режиме Мультиклиентности
my %MULTI_CLIENTS_FILTER_TO_DB_FIELDS = (
    client_id => {eq => 'c.ClientID', ne => 'c.ClientID__not_in'},
    place_id => {eq => 'cint.place_id', ne => 'cint.place_id__not_in'},
);

my @EXTRA_COUNTABLE_FIELDS = (qw/avg_view_freq uniq_viewers avg_cpm eshows ectr avg_x avg_bid avg_cpm_bid avg_time_to_conv agoalnum agoalcost agoalroi agoalcrr 
                                agoalincome aconv agoals_profit auction_hits auction_win_rate imp_reach_rate imp_to_win_rate auction_wins served_impressions
                                pv_avg_time_to_conv pv_agoalnum pv_agoalcost pv_agoalroi pv_agoalcrr pv_agoalincome pv_aconv pv_agoals_profit/,
    Stat::Fields::get_extra_countable_fields()
);
my %EXTRA_COUNTABLE_FIELDS_MAP = map { $_ => 1 } @EXTRA_COUNTABLE_FIELDS;

my %FORBIDDEN_FIELDS_FOR_GROUPS = (
    # какие измеримые поля (столбцы, фильтры) запрещены при срезе по месту клика
    click_place => [qw/shows ctr fp_shows_avg_pos eshows ectr avg_x/],

    inventory_type => [qw/
        aprgoodmultigoal
        aprgoodmultigoal_cpa
        aprgoodmultigoal_conv_rate
        avg_time_to_conv

        imp_to_win_rate
    /],
);

=head2 get_user_filters_set_list

    Возвращает список шаблонов фильтров пользователя

=cut

sub get_user_filters_set_list {
    my ($UID, $uid, %O) = @_;

    $O{stat_type} = '' if defined $O{stat_type} && !$SUPPORTED_STAT_TYPES{$O{stat_type}};

    my $user_opts = get_user_typed_options($UID, $USER_OPTION_TYPE);
    my $list = $user_opts->{stat_master_filters} // [];

    if (defined $O{cid}) {
        @$list = grep { ($_->{cid} || 0) == ($O{cid} || 0) } @$list; 
    }
    if (defined $O{stat_type}) {
        @$list = grep { ($_->{stat_type} || '') eq $O{stat_type} } @$list; 
    }

    if ($uid) {
        for my $filters_set (@$list) {
            convert_deprecated_report_options({filters => $filters_set->{data} // {}});
            for my $region_field (@REGION_FIELDS) {
                if ($filters_set->{data}->{$region_field}) {
                    _convert_geo($filters_set->{data}->{region}, $uid, 'use');
                }
            }
        }
    }
    return $list;
}

=head2 save_user_filters_set 

    Сохраняет/обновляет шаблон фильтров пользователя

=cut

sub save_user_filters_set {
    my ($UID, $uid, $filters_set, %O) = @_;

    die "Incorrect uid: $UID" unless is_valid_id($UID);
    die "Expected HASHREF for filters_set" unless ref $filters_set eq 'HASH';
    $O{stat_type} = '' if defined $O{stat_type} && !$SUPPORTED_STAT_TYPES{$O{stat_type}};
    
    return _error_result(iget('Название фильтра - обязательное поле')) unless ($filters_set->{name} // '') =~ /\S+/;

    my @filters_set_list = grep { $_->{name} ne $filters_set->{name} 
                                    || ($O{cid} || 0) != ($_->{cid} || 0) 
                                    || ($O{stat_type} || '') ne ($_->{stat_type} || '')
                                } @{ get_user_filters_set_list($UID, $uid) };
    return _error_result(iget('Нельзя сохранять более %s шаблонов фильтров', $FILTERS_SETS_MAX_NUM)) 
                                                            if @filters_set_list >= $FILTERS_SETS_MAX_NUM;  

    $filters_set->{data} //= {};
    $filters_set->{cid} = $O{cid} if $O{cid};
    $filters_set->{stat_type} = $O{stat_type} if $O{stat_type};
    for my $region_field (@REGION_FIELDS) {
        if ($filters_set->{data}->{$region_field}) {
            _convert_geo($filters_set->{data}->{$region_field}, $uid, 'save');
        }
    }
    push @filters_set_list, $filters_set;

    update_user_typed_options($UID, $USER_OPTION_TYPE, {$FILTERS_OPTION_NAME => \@filters_set_list});
    return;
}

=head2 delete_user_filters_set 

    Удаляет шаблон фильтров пользователя

=cut

sub delete_user_filters_set {
    my ($UID, $filter_name, %O) = @_;
    $filter_name //= '';
    $O{stat_type} = '' if defined $O{stat_type} && !$SUPPORTED_STAT_TYPES{$O{stat_type}};

    die "Incorrect uid: $UID" unless is_valid_id($UID);
    
    my @filters_set_list = grep { $_->{name} ne $filter_name 
                                    || ($O{cid} || 0) != ($_->{cid} || 0)
                                    || ($O{stat_type} || '') ne ($_->{stat_type} || '')
                                      } @{ get_user_filters_set_list($UID) };
    
    update_user_typed_options($UID, $USER_OPTION_TYPE, {$FILTERS_OPTION_NAME => \@filters_set_list});
    return;
}

=head2 get_user_accessible_report

    Возвращает настройки сохраненного отчета, если у пользователя есть на него права, и отчет найден
    В противном случае возвращает undef

=cut

sub get_user_accessible_report {
    my ($UID, $uid, $report_id) = @_;

    return undef unless $UID && $report_id;

    my $uids = [$UID];
    my $rbac = RBAC2::Extended->get_singleton(1);
    if (rbac_who_is($rbac, $UID) eq 'client') {
        # каждый из представителей клиента имеет доступ к отчетам всех представителей клиента
        my $ClientID = get_user_data($UID, [qw/ClientID/])->{ClientID};
        $uids = rbac_get_main_reps_of_client($ClientID);
    } elsif (rbac_who_is($rbac, $uid) eq 'client' && rbac_is_owner($rbac, $UID, $uid)) {
        # сущности выше имеют доступ как к своим отчётам, так и к отчётам клиента
        my $ClientID = get_user_data($uid, [qw/ClientID/])->{ClientID};
        push @$uids, @{rbac_get_main_reps_of_client($ClientID)};
    }

    my $users_opts = mass_get_user_typed_options($uids, $USER_OPTION_TYPE);
    my @list = ();
    foreach (values %$users_opts) {
        push @list, @{$_->{$REPORTS_OPTION_NAME} // []};
    }

    my $report_options = (grep { $_->{report_id} eq $report_id } @list)[0];
    if ($report_options) {
        if ($report_options->{filters}) {
            for my $region_field (@REGION_FIELDS) {
                if ($report_options->{filters}->{$region_field}) {
                    _convert_geo($report_options->{filters}->{$region_field}, $uid, 'use');
                }
            }
        }
        convert_deprecated_report_options($report_options);
    }
    return $report_options;
}

=head2 convert_deprecated_report_options

    Заменить устаревшие фильтры и срезы на их новые аналоги в переданном хешрефе
    Принимает параметры:
        $report_options - опции отчета

=cut

sub convert_deprecated_report_options {
    my ($report_options) = @_;

    my $remove_winrate = 0;
    if ($report_options->{filters}) {
        if ($report_options->{filters}->{has_image}) {
            my $current_value = delete $report_options->{filters}->{has_image};
            for my $comparator (keys %{$current_value}) {
                $report_options->{filters}->{banner_image_type}->{$comparator} = $current_value->{$comparator} ? 'text_image' : 'text_only';
            }
            $remove_winrate = 1;
        }
        for my $phrase_filter (qw/phrase phrase_ext/) {
            $report_options->{filters}->{phrase_orig} = delete $report_options->{filters}->{$phrase_filter} if $report_options->{filters}->{$phrase_filter};
        }
        if (my $ct = $report_options->{filters}->{contexttype}) {
            my %deprecated_values = map { $_ => 1 } qw/auto-added-phrases synonym/;
            for my $comparator (keys %$ct) {
                $ct->{$comparator} = [ grep { !$deprecated_values{$_} } xflatten $ct->{$comparator} ];
            }
            $report_options->{filters}->{contexttype_orig} = delete $report_options->{filters}->{contexttype};
        }
    }
    if ($report_options->{group_by}) {
        my %old2new = (has_image => 'banner_image_type',
                       contextcond_ext => 'contextcond_orig',
                       contextcond => 'contextcond_orig',
                       contexttype => 'contexttype_orig');
        my $idx = first_index {$_ eq 'has_image'} @{$report_options->{group_by}};
        $remove_winrate = 1 if $idx >= 0;
        
        @{$report_options->{group_by}} = map { $old2new{$_} || $_ } @{$report_options->{group_by}};
        @{$report_options->{group_by_positions}} = map { $old2new{$_} || $_ } @{$report_options->{group_by_positions}};
    }
    # Новая группировка и фильтр по изображению не поддерживают winrate
    if ($remove_winrate && $report_options->{columns}) {
        my $idx = first_index {$_ eq 'winrate'} @{$report_options->{columns}};
        splice(@{$report_options->{columns}}, $idx, 1) if $idx >= 0;
    }
}

=head2 get_user_reports_list

    Возвращает список сохраненных шаблонов отчетов пользователя
    Принимает параметры:
        $UID - владелец отчета
        $uid - uid клиента, для транслокализации регионов
        %O - доп. опции
            no_data=1 - только report_id и report_name
            cid=123 - фильтровать отчеты по привязке к cid

=cut

sub get_user_reports_list {
    my ($UID, $uid, %O) = @_;

    $O{stat_type} = '' if defined $O{stat_type} && !$SUPPORTED_STAT_TYPES{$O{stat_type}};

    my $user_opts = get_user_typed_options($UID, $USER_OPTION_TYPE);
    my $list = $user_opts->{$REPORTS_OPTION_NAME} // [];
    if (defined $O{cid}) {
        @$list = grep { ($_->{cid} || 0) == ($O{cid} || 0) } @$list;
    }
    if (defined $O{stat_type}) {
        @$list = grep { ($_->{stat_type} // '') eq $O{stat_type} } @$list; 
    }

    if ($O{no_data}) {
        @$list = map { hash_cut $_, qw/report_id report_name compare_periods/ } @$list;
    } elsif ($uid) {
        for my $r (@$list) {
            if ($r->{filters} && $r->{filters}->{region}) {
                _convert_geo($r->{filters}->{region}, $uid, 'use');
            }
        }
    }

    return $list;
}


=head2 save_user_report

    Сохраняет новый/обновляет старый шаблон отчета

=cut

sub save_user_report {
    my ($UID, $uid, $report_options, %O) = @_;

    die "Incorrect uid: $UID" unless is_valid_id($UID);
    die "Expected HASHREF for report_options" unless ref $report_options eq 'HASH';
    $O{stat_type} = '' if defined $O{stat_type} && !$SUPPORTED_STAT_TYPES{$O{stat_type}};

    $report_options->{cid} = $O{cid} if $O{cid};
    $report_options->{stat_type} = $O{stat_type} if $O{stat_type};

    my $error = check_report_options($report_options, $uid, enable_internal_ads => $O{enable_internal_ads});
    return {result => 'error', error => $error} if $error;
    return {result => 'error', error => iget('Название отчета - обязательное поле')} unless ($report_options->{report_name} // '') =~ /\S+/;

    my $reports_list = get_user_reports_list($UID, $uid);
    if ($report_options->{report_id}) {
        return _error_result(iget('Вы пытаетесь сохранить отчет с некорректным номером')) 
                    if none { $_->{report_id} eq $report_options->{report_id} } @$reports_list;

        $reports_list = [grep { $_->{report_id} ne $report_options->{report_id} } @$reports_list];
    } else {
        $report_options->{report_id} = lc(join '', split /-/, HashingTools::generate_uuid());
    }

    return _error_result(iget('Отчет с таким названием уже существует'))
                    if any { $_->{report_name} eq $report_options->{report_name} &&
                             ($_->{cid} || 0) == ($report_options->{cid} || 0) &&
                             ($_->{stat_type} || '') == ($report_options->{stat_type} || '')} @$reports_list;

    return _error_result(iget('Нельзя сохранять более %s отчетов', $REPORTS_MAX_NUM))
                                                if @$reports_list >= $REPORTS_MAX_NUM;

    if ($report_options->{filters}) {
        for my $region_field (@REGION_FIELDS) {
            if ($report_options->{filters}->{$region_field}) {
                _convert_geo($report_options->{filters}->{$region_field}, $uid, 'save');
            }
        }
    }

    push @$reports_list, $report_options;

    update_user_typed_options($UID, $USER_OPTION_TYPE, {$REPORTS_OPTION_NAME => $reports_list});
    return {status => 'ok', report_id => $report_options->{report_id}};
}

=head2 delete_user_report 

    Удаляет сохраненный отчет пользователя

=cut

sub delete_user_report {
    my ($UID, $report_id) = @_;

    die "Incorrect uid: $UID" unless is_valid_id($UID);
    
    my @reports_list = grep { $_->{report_id} ne $report_id } @{ get_user_reports_list($UID) };
    
    update_user_typed_options($UID, $USER_OPTION_TYPE, {$REPORTS_OPTION_NAME => \@reports_list});
    return;
}

=head2 _convert_geo

    Конвертирует геодерево с настройками транслокальности клиента в формат с tree=api для сохранения,
    или в обратную сторону - для использования

    Входящие параметры:
        $filter - настройки фильтра по регионам
        $uid - uid клиента
        $convert_for - (save|use), для какой цели конвертить настройки

    В результате изменяется содержимое $filter (оно же и возвращается)

=cut

sub _convert_geo {
    my ($filter, $uid, $convert_for) = @_;

    die "Incorrect convert_for parameter: $convert_for" unless ($convert_for || '') =~ m/^(save|use)$/;
    my $ClientID = get_clientid(uid => $uid);

    for my $op (keys %$filter) {
        next unless defined $filter->{$op};
        my @values = ref $filter->{$op} ? @{$filter->{$op}} : $filter->{$op};

        @values = map { $convert_for eq 'save'
                            ? GeoTools::modify_translocal_region_before_save($_, {ClientID => $ClientID})
                            : GeoTools::modify_translocal_region_before_show($_, {ClientID => $ClientID}) } @values;
        $filter->{$op} = ref $filter->{$op} ? \@values : $values[0];
    }
    return $filter;
}

=head2 check_report_options

    Проверяет настройки отчета пришедшие из интерфейса на корректность
    
    %O - именованные параметры (не обязательные)
        enable_internal_ads => 1 - доступно ли внутренняя реклама в статистике для оператора 

=cut

sub check_report_options {
    my ($opts, $uid, %O) = @_;

    if ($opts->{compare_periods}) {
        my $err = check_period_dates($opts->{date_from}, $opts->{date_to}, 'A');
        return $err if $err;
        $err = check_period_dates($opts->{date_from_b}, $opts->{date_to_b}, 'B');
        return $err if $err;

        return iget("Дата окончания периода B должна быть раньше даты начала периода А") 
                            if normalize_date($opts->{date_to_b}) ge normalize_date($opts->{date_from});    

    } else {
        my $err = check_period_dates($opts->{date_from}, $opts->{date_to});
        return $err if $err;
    }

    return iget("Неправильный период агрегации") if $opts->{group_by_date} && $opts->{group_by_date} !~ $Stat::Const::GROUP_BY_DATE_RE;

    if ( $opts->{is_multi_clients_mode} ) {
        my $multi_clients_filters = _get_multi_clients_filters($opts->{filters});

        # Для мультиклиентности обязательно должен быть положительный(eq) фильтр
        unless( any { $multi_clients_filters->{$_}{eq} && scalar @{$multi_clients_filters->{$_}{eq}} } keys %$multi_clients_filters ) {
            my $required_filter_fields = $O{enable_internal_ads}
                ? [ iget('Название продукта'), iget('Номер плейса'), iget('Номер пейджа') ] 
                : [ iget('Идентификатор клиента') ];
            return iget("Для построения отчета выберите один из положительных фильтров: %s", (join ', ', @$required_filter_fields));
        }

        # ошибка при попытке запросить id в фильтрах больше чем допустимо
        for my $filter_name (keys %$multi_clients_filters) {
            for my $filter_type (qw/eq ne/) {
                next unless exists $multi_clients_filters->{$filter_name}{$filter_type};
                my $values = $multi_clients_filters->{$filter_name}{$filter_type};
                my $limit = $opts->{is_mcc_mode} ? $MCC_MULTI_CLIENTS_FILTER_IDS_MAX_NUM : $MULTI_CLIENTS_FILTER_IDS_MAX_NUM;
                if (scalar @$values > $limit) {
                    return iget('Можно указать не более %s значений в фильтрах Мультиклиентности', $limit);
                }
            }
        }
    }

    if ($opts->{filters}) {
        for my $region_field (@REGION_FIELDS) {
            if ($opts->{filters}->{$region_field}) {
                for my $geo_str (grep { defined $_ } values %{$opts->{filters}->{$region_field}}) {
                    my $error = validate_geo($geo_str, undef, {ClientID => get_clientid(uid => $uid)});
                    return $error if $error;
                }
            }
        }
        for my $keyword_field (@KEYWORD_FILTERS) {
            if ($opts->{filters}->{$keyword_field}) {
                for my $phrase_string (@{_split_text([ grep { defined $_ } xflatten values %{$opts->{filters}->{$keyword_field}} ])}) {
                    if ($phrase_string !~ /^[${ALLOW_SEARCH_PHRASE_LETTERS}]+$/ || $phrase_string =~ /''/) {
                        return iget('В поле фильтрации по тексту фразы разрешается использовать только буквы английского, турецкого, казахского, русского, украинского или белорусского алфавита, знаки "-", "+", "!", "[", "]", пробел и двойные кавычки');
                    }
                }
            }
        }
    }

    # ошибка при попытке запросить целей больше чем допустимо
    if (exists $opts->{filters}->{goal_ids}){
        my $goal_ids = $opts->{filters}->{goal_ids}->{eq};
        if(scalar @$goal_ids > $Stat::Const::GOALS_MAX_NUM_INTERNAL){
            return iget('Можно указать не более %s целей.', $Stat::Const::GOALS_MAX_NUM_INTERNAL);
        }
    }
    return;
}

=head2 check_period_dates

    Проверяет на корректность даты начала/окончания периода

=cut

sub check_period_dates {
    my ($df, $dt, $period_name) = @_;
    $period_name = $period_name ? " $period_name" : '';

    return iget("Неверно указана дата начала периода%s", $period_name) unless check_mysql_date($df);
    return iget("Неверно указана дата конца периода%s", $period_name) unless check_mysql_date($dt);
    return iget("Дата окончания периода%s должна быть не раньше даты начала", $period_name) if normalize_date($df) gt normalize_date($dt);  
}

=head2 extract_report_options_out_of_form

    Выделяет из данных формы поля, несущие информацию о требуемом отчете статистикой, 
    а также инициализирует нужные структуры при их отсутствии

    на вход принимает:
        $FORM - данные пришедшие от клиентской формы
        %O - доп. опции
            only_main_options - работаем только с основными параметрами отчета, которые можно сохранить как шаблон отчета (без пейджинга и сортировок)
            flat_params - в пришедшей форме json-параметры представлены в виде простых http-параметров, которые нужно преобразовать в классический формат
=cut

sub extract_report_options_out_of_form {
    my ($FORM, %O) = @_;

    my @form_fields = qw/date_from date_to date_range date_shift group_by_date with_nds report_name report_id 
                         compare_periods date_from_b date_to_b stat_type attribution_model region_level with_resources/;
    push @form_fields, qw/page_size/ unless $O{no_paging};
    push @form_fields, 'cid' if $FORM->{single_camp};
    unless ($O{only_main_options}) {
        push @form_fields, qw/sort reverse/;
        push @form_fields, qw/page/ unless $O{no_paging};
    }
    my $report_options = hash_cut $FORM, @form_fields;

    # Всегда показываем статистику со скидкой
    $report_options->{with_discount} = 1;
    # stat_type оставляем только тот который используется на серверсайде (чтоб не сохранять в шаблоны отчетов лишние поля, используемые только фронтендом)
    delete $report_options->{stat_type} if $report_options->{stat_type} && !$SUPPORTED_STAT_TYPES{$report_options->{stat_type}};

    if ($report_options->{sort}) {
        $report_options->{order_by} = delete $report_options->{sort};
        $report_options->{order_dir} = $report_options->{reverse} ? 'desc' : 'asc';
    }

    # В МОК добавили фильтрацию по цели, но переделать формат запроса для фронта проблемо, поэтому меняем тут
    my $goal_id = (defined $FORM->{'goals'} && $FORM->{'goals'} =~ /^[0-9]+$/) ? int($FORM->{'goals'}) : undef;
    my $goal_ids = (defined $FORM->{'goals'} && $FORM->{'goals'} =~ /^[0-9]+(,[0-9]+)+$/) ? [split /,/, $FORM->{'goals'}] : undef;

    if ($O{flat_params}) {
        $FORM->{'fl_goal_id__eq'} = $goal_id if $goal_id;
        $FORM->{'fl_goal_ids__eq'} = $goal_ids if $goal_ids;

        # преобразовуем пришедшие get-запросом параметры в формате fl_campaign_eq[]=1,2,3 и т.п. в формат который приходит посредством json-а в post-запросах
        $report_options->{filters} = {};
        foreach my $param (keys %$FORM) {
            if ($param =~ /^fl_(.+)__(.+?)(\[\])?$/) {
                $report_options->{filters}->{$1} //= {};
                $report_options->{filters}->{$1}->{$2} = $3 ? [split /,/, $FORM->{$param} // ''] : $FORM->{$param} // '';
            }
        }

        for my $field (qw/group_by group_by_positions columns columns_positions/) {
            next unless defined $FORM->{$field};
            $report_options->{$field} = [split /,/, $FORM->{$field} || ''];
        }
    } else {
        ($FORM->{json_filters} ||= {})->{goal_id}->{'eq'} = $goal_id if $goal_id;
        ($FORM->{json_filters} ||= {})->{goal_ids}->{'eq'} = $goal_ids if $goal_ids;

        $report_options->{filters} = $FORM->{json_filters} || {};
        for my $field (qw/group_by group_by_positions columns columns_positions/) {
            next unless defined $FORM->{"json_$field"};
            $report_options->{$field} = $FORM->{"json_$field"} || [];
        }
    }

    # Если это статистика по одной кампании, то тогда сразу для нее вычисляем тип,
    # так как дальше часто это пригождается.
    if ($report_options->{cid}) {
        $report_options->{camp_type} = get_camp_type(cid => $report_options->{cid});
    }

    $report_options->{is_multi_clients_mode} = $FORM->{multi_clients_mode} ? 1 : 0;
    $report_options->{is_mcc_mode} = $FORM->{is_mcc_mode} ? 1 : 0;

    convert_deprecated_report_options($report_options);

    return $report_options;
}

=head2 add_report_options_defaults

    К настройкам отчета (извлеченных их формы, или из сохраненного отчета) добавляет отсутствующим настройкам значения по-умолчанию

    на вход принимает:
        $report_options - ссылка на хеш с настройками отчете
        %O - доп. опции
            only_main_options - работаем только с основными параметрами отчета, которые можно сохранить как шаблон отчета (без пейджинга и сортировок)
            no_default_columns - не добавлять стандартные колонки, если список пустой
            stat_type - тип запрашиваемой статистики
            client_id - клиент

=cut

sub add_report_options_defaults {
    my ($report_options, %O) = @_;

    $report_options->{date_from} //= now()->subtract(days => 7)->ymd();
    $report_options->{date_to} //= now()->ymd();
    if ($report_options->{compare_periods}) {
        my $period_length_days = date($report_options->{date_to})->delta_days(date($report_options->{date_from}))->in_units('days');
        $report_options->{date_to_b} //= date($report_options->{date_from})->subtract(days => 1)->ymd();
        $report_options->{date_from_b} //= date($report_options->{date_to_b})->subtract(days => $period_length_days-1)->ymd();
        
    }
    $report_options->{group_by_date} ||= 'day';

    $report_options->{with_nds} //= 0;
    $report_options->{with_discount} //= 1;

    if ((($report_options->{stat_type} // '') eq 'search_queries') && $O{only_main_options} && !$report_options->{report_id}) {
        $report_options->{filters} = { clicks => { gt => 0 } }; # для отчета по поисковым запросам добавляем по умолчанию фильтрацию clicks > 0
    }

    $report_options->{filters} //= {};

    unless (defined $report_options->{group_by}) {
        if ($report_options->{stat_type} && $report_options->{stat_type} eq 'search_queries') {
            $report_options->{group_by_date} = 'none';
            $report_options->{group_by} = [qw/search_query adgroup contextcond_orig contexttype_orig matched_phrase match_type targeting_category/];
            unshift @{$report_options->{group_by}}, 'campaign' unless $report_options->{cid};
        } else {
            $report_options->{group_by} = [];
        }
    }

    my @default_column_order = qw/
        auction_hits
        auction_wins
        auction_win_rate
        served_impressions
        shows
        avg_nshow
        imp_to_win_rate
        imp_reach_rate
        cpm_winrate
        uniq_users_winrate
        eshows
        clicks
        ctr
        ectr
        sum
        av_sum
        avg_bid
        fp_shows_avg_pos
        video_mute
        video_unmute
        avg_x
        fp_clicks_avg_pos
        winrate
        bounce_ratio
        avg_cpm
        close_clicks
        close_ctr
        video_true_view
        video_true_view_rate
        video_avg_true_view_cost
        uniq_viewers
        uniq_completed_viewers
        avg_view_freq
        adepth
        aconv
        agoalcost
        agoalnum
        agoalroi
        agoalcrr
        agoalincome
        agoals_profit
        aprgoodmultigoal
        aprgoodmultigoal_cpa
        aprgoodmultigoal_conv_rate
        avg_time_to_conv
        video_first_quartile
        video_midpoint
        video_third_quartile
        video_complete
        video_first_quartile_rate
        video_midpoint_rate
        video_third_quartile_rate
        video_complete_rate
        avg_nshow_complete        
        ad_site_clicks
        cpcv
        viewable_impressions_mrc
        nonviewable_impressions_mrc
        undetermined_impressions_mrc
        measured_rate_mrc
        viewable_rate_mrc
        pv_bounce_ratio
        pv_adepth
        pv_aconv
        pv_agoalcost
        pv_agoalnum
        pv_agoalroi
        pv_agoalcrr
        pv_agoalincome
        pv_agoals_profit
        pv_aprgoodmultigoal
        pv_aprgoodmultigoal_cpa
        pv_aprgoodmultigoal_conv_rate
        pv_avg_time_to_conv        
    /;
    #Если МОК для кампаний типа: cpm_banner cpm_deals cpm_yndx_frontpage cpm_price
    if ($report_options->{camp_type} &&
         (any { $report_options->{camp_type} eq $_ } qw/cpm_banner cpm_deals cpm_yndx_frontpage cpm_price/)
         # Если это МОЛ, то проставлять последовательность
         || !defined $report_options->{camp_type}) {

        $report_options->{columns_positions} //= [@default_column_order];
    }

    $report_options->{group_by_positions} //= [];
    $report_options->{columns_positions} //= [];
    unless (@{$report_options->{columns} // []}) {
        unless ($O{no_default_columns}) {
            my %selected_column;
            my $camp_type = $report_options->{camp_type} || '';
            if ($report_options->{cid} && $camp_type eq 'performance') {
                $selected_column{$_} = 1 for qw/clicks sum av_sum/;
            } elsif ($report_options->{stat_type} && $report_options->{stat_type} eq 'search_queries') {
                $selected_column{$_} = 1 for qw/shows clicks ctr sum av_sum aconv agoalcost agoalnum/;
            } elsif ($report_options->{stat_type} && $report_options->{stat_type} eq 'moc') {
                $selected_column{$_} = 1 for qw/shows clicks ctr sum av_sum adepth agoalnum aconv agoalcost agoalroi agoalincome/;
                if ($camp_type eq 'mobile_content') {
                    delete $selected_column{adepth};
                }
                my $client_id = get_clientid(cid => $report_options->{cid});
                $selected_column{agoals_profit} = 1;
                if (Client::ClientFeatures::has_crr_strategy_feature($client_id)) {
                    $selected_column{agoalcrr} = 1;
                }
                if ($camp_type eq 'cpm_banner' || $camp_type eq 'cpm_deals') {
                    $selected_column{avg_cpm} = 1;
                    my $strategy = Campaign::campaign_strategy($report_options->{cid});
                    if (any {$strategy->{net}{name} eq $_} qw/autobudget_avg_cpv autobudget_avg_cpv_custom_period/) {
                        $selected_column{$_} = 1 for qw/video_true_view video_true_view_rate video_avg_true_view_cost/;
                    }
                }
                if ($camp_type eq 'cpm_banner' || $camp_type eq 'cpm_deals' || $camp_type eq 'cpm_yndx_frontpage') {
                    $selected_column{viewable_impressions_mrc} = 1;
                }
                if (Client::ClientFeatures::has_avg_nshow_for_strategy_in_moc_feature($client_id) ||
                    Client::ClientFeatures::has_avg_nshow_for_cpm_campaigns_in_moc_feature($client_id)) {
                    $selected_column{avg_nshow} = 1;
                }
                if (Client::ClientFeatures::has_avg_nshow_complete_for_strategy_in_moc_feature($client_id) ||
                    Client::ClientFeatures::has_avg_nshow_complete_for_cpm_campaigns_in_moc_feature($client_id)) {
                    $selected_column{avg_nshow_complete} = 1;
                }
            } else {
                $selected_column{$_} = 1 for qw/shows clicks ctr sum av_sum avg_cpm adepth aconv agoalcost agoalnum/;
            }

            my $column_order = @{$report_options->{columns_positions}} ? $report_options->{columns_positions} : \@default_column_order;
            $report_options->{columns} = [grep {$selected_column{$_}} @$column_order];
        }
    }


    unless ($O{no_paging}) {
        $report_options->{page_size} = 100 unless is_valid_int($report_options->{page_size}, 1, 10000);
        $report_options->{page} = 1 unless $O{only_main_options} || is_valid_int($report_options->{page}, 1);
    }

    # для внутренней рекламы всегда используем дефолтное значение
    if($O{enable_internal_campaigns} && !defined $report_options->{attribution_model}) {
        $report_options->{attribution_model} = get_attribution_model_internal_stat_default();
    }

    # Если получаем статистику по одной кампании, а модель атрибуции не указана,
    # то используем ту модель атрибуции, которая указана на кампании (или дефолт)

    if ($report_options->{cid} && !defined $report_options->{attribution_model}) {
        my $campaign = Campaign::get_camp_info($report_options->{cid}, undef, short => 1);
        $report_options->{attribution_model} = get_attribution_model_for_stat_request($campaign);
    }
    # При получении статистики по логину смотрим на атрибуцию на кампаниях клиента
    # если она везде одинаковая, используем значение установленное на кампаниях (иначе дефолт) DIRECT-101085
    if (defined $O{stat_type}
        && ($O{stat_type} eq 'mol'
        ||  ($O{stat_type} eq 'search_queries'  && !$report_options->{single_camp}))) {
        $report_options->{attribution_model} //= get_client_campaigns_attribution_type_or_default($O{client_id});
    }
}

=head2 apply_limits_for_options

    накладывает ограничения на некоторые группировки, фильтры, колонки и т.п. в зависимости от роли пользователя и группировок, фильтров и т.п.
    результат отдается в шаблон ($vars), т.е. "скрытые" от пользователя фильтры тут добавлять не стоит

=cut

sub apply_limits_for_options {
    my ($operator_client_id, $client_id, $UID, $uid, $report_options) = @_;

    my $rbac = RBAC2::Extended->get_singleton(1);
    my $user_role = rbac_who_is($rbac, $UID);

    my %limit_fields_by_role = (
        page                => { roles => [ qw/super superreader support/ ], opts => { group_by => 1 } },
        ext_phrase_status   => { roles => [ keys %RBACElementary::IS_INTERNAL_ROLE ] },
        search_query_status => { roles => [ keys %RBACElementary::IS_INTERNAL_ROLE ] },
        sim_distance        => { roles => [ qw/super superreader support/ ] },
        deal                => { roles => [ qw/super/ ], feature_name => "cpm_deals" },
        deal_export_id      => { roles => [ qw/super/ ], feature_name => "cpm_deals" },
        avg_bid             => { roles => [ qw/super superreader/ ], feature_name => "avg_bid_in_mol", opts => { columns => 1, filter => 1 } },
        avg_cpm_bid         => { roles => [ qw/super superreader/ ], feature_name => "avg_bid_in_mol", opts => { columns => 1, filter => 1 } },
        auction_hits        => { roles => [], feature_name => "cpm_banner_sov_in_stat", opts => { columns => 1, filter => 1 } },
        auction_win_rate    => { roles => [], feature_name => "cpm_banner_sov_in_stat", opts => { columns => 1, filter => 1 } },
        imp_reach_rate      => { roles => [], feature_name => "cpm_banner_sov_in_stat", opts => { columns => 1, filter => 1 } },
        imp_to_win_rate     => { roles => [], feature_name => "cpm_banner_sov_in_stat", opts => { columns => 1, filter => 1 } },
        auction_wins        => { roles => [], feature_name => "cpm_banner_sov_in_stat", opts => { columns => 1, filter => 1 } },
        avg_nshow           => { roles => [], feature_name => "avg_nshow_in_moc", opts => { columns => 1, filter => 1 } },
        avg_nshow_complete  => { roles => [], feature_name => "avg_nshow_in_moc_complete", opts => { columns => 1, filter => 1 } },
        strategy_id         => { roles => [], feature_name => "package_strategies_stage_two" },
        agoalcost           => { roles => [], feature_name => "single_goal_cost_enabled" }
    );

    for my $field (keys %limit_fields_by_role) {
        my $field_data = $limit_fields_by_role{$field};
        $field_data->{opts} //= {group_by => 1, filter => 1};
        next if $report_options->{cid} && $field_data->{opts}->{ignore_if_single_camp};
        if (none { $_ eq $user_role} @{$field_data->{roles}} ) {
            my $feature_name = $field_data->{feature_name};
            my $enabled_by_feature;
            if (defined($feature_name)) {
                if ($feature_name eq "cpm_deals") {
                    $enabled_by_feature = Client::ClientFeatures::has_cpm_deals_allowed_feature($operator_client_id);
                } elsif ($feature_name eq 'avg_bid_in_mol') {
                    $enabled_by_feature = Client::ClientFeatures::has_avg_bid_in_mol_feature($client_id);
                } elsif ($feature_name eq 'cpm_banner_sov_in_stat') {
                    # DIRECT-116629 - Для типов кампаний: Медийная кампания для Главной и кампании с фиксированным CPM в отчёте
                    # скрыть некоторые поля под отдельные фичи 
                    if (($report_options->{camp_type} // '') eq 'cpm_yndx_frontpage') {
                        $enabled_by_feature = Client::ClientFeatures::has_cpm_yndx_frontpage_sov_in_stat_feature($operator_client_id);
                    } elsif (($report_options->{camp_type} // '') eq 'cpm_price') {
                        $enabled_by_feature = Client::ClientFeatures::has_cpm_price_sov_in_stat_feature($operator_client_id);
                    } else {
                        $enabled_by_feature = Client::ClientFeatures::has_cpm_banner_sov_in_stat_feature($client_id);
                    }

                    $enabled_by_feature ||= Client::ClientFeatures::has_auction_stats_for_all_campaigns_feature($operator_client_id);
                } elsif ($feature_name eq 'avg_nshow_in_moc' and $report_options->{cid}) {
                    # Показывать поля если 
                    # включена фича avg_nshow_for_cpm_campaigns_in_moc и кампания CPM(Медийная, Медийная на Главной, Кампания с фиксированным CPM)
                    # ИЛИ
                    # включена фича avg_nshow_for_strategy_in_moc и стратегия autobudget_avg_cpv_custom_period.
                    $enabled_by_feature = Client::ClientFeatures::has_avg_nshow_for_cpm_campaigns_in_moc_feature($client_id) &&
                        camp_kind_in(type => ($report_options->{camp_type} // ''), 'cpm') ||
                        Client::ClientFeatures::has_avg_nshow_for_strategy_in_moc_feature($client_id) &&
                        Campaign::campaign_strategy($report_options->{cid})->{net}{name} eq 'autobudget_avg_cpv_custom_period';
                } elsif ($feature_name eq 'avg_nshow_in_moc_complete' and $report_options->{cid}) {
                    # Показывать поля если 
                    # включена фича avg_nshow_complete_for_cpm_campaigns_in_moc и кампания CPM(Медийная, Медийная на Главной, Кампания с фиксированным CPM)
                    # ИЛИ
                    # включена фича avg_nshow_complete_for_strategy_in_moc и стратегия autobudget_avg_cpv_custom_period.
                    $enabled_by_feature = Client::ClientFeatures::has_avg_nshow_complete_for_cpm_campaigns_in_moc_feature($client_id) &&
                        camp_kind_in(type => ($report_options->{camp_type} // ''), 'cpm') ||
                        Client::ClientFeatures::has_avg_nshow_complete_for_strategy_in_moc_feature($client_id) &&
                        Campaign::campaign_strategy($report_options->{cid})->{net}{name} eq 'autobudget_avg_cpv_custom_period';
                } elsif ($feature_name eq 'package_strategies_stage_two') {
                    $enabled_by_feature = Client::ClientFeatures::has_package_strategies_stage_two_enabled($client_id)
                } elsif ($feature_name eq 'single_goal_cost_enabled') {
                    $enabled_by_feature = Client::ClientFeatures::has_single_goal_cost_enabled($client_id);
                }
            }
            if (!$enabled_by_feature) {
                #прячем столбцы если они не разрешены ни через ролевые настройки, ни через настройки-'эксперименты'
                $report_options->{group_by} = xminus($report_options->{group_by}, [ $field ]) if $field_data->{opts}->{group_by};
                _remove_records_by_norm_field_name($report_options->{filters}, $field) if $field_data->{opts}->{filter};
                $report_options->{columns} = xminus($report_options->{columns}, [ $field ]) if $field_data->{opts}->{columns};
            }
        }
    }

    for my $group (keys %FORBIDDEN_FIELDS_FOR_GROUPS) {
        if (any { $_ eq $group } @{$report_options->{group_by} // []}) {
            my $forbidden_fields = $FORBIDDEN_FIELDS_FOR_GROUPS{$group};
            $report_options->{columns} = xminus $report_options->{columns}, $forbidden_fields;
            _remove_records_by_norm_field_name($report_options->{filters}, $_) for @$forbidden_fields;
        }

    }

    if (none { $_ eq 'click_place' } @{$report_options->{group_by} // []}) {
        # фильтра по месту клика доступен только при срезе по месту клика
        _remove_records_by_norm_field_name($report_options->{filters}, 'click_place');
    }

    my (@force_group_by_fields, @allow_group_by_fields, @forbid_fields);
    if (any {($report_options->{stat_type} || '') eq $_ } qw/search_queries/) {
        @forbid_fields = qw/winrate/;

        if ($report_options->{stat_type} eq 'search_queries') {
            @force_group_by_fields = qw/search_query/;
            @allow_group_by_fields = qw/campaign_type campaign adgroup banner search_query contexttype_orig contextcond_orig
                                        matched_phrase matched_phrase_id match_type sim_distance page page_group page_name targeting_category prisma_income_grade client_login/;
        }
    } else {
        @forbid_fields = qw/sim_distance/;
    }

    _limit_fields_by_lists($report_options, force_group_by_fields => \@force_group_by_fields,
                                            allow_group_by_fields => \@allow_group_by_fields,
                                            forbid_fields => \@forbid_fields);
    
    return $report_options;
}

=head2 _limit_fields_by_lists

    Применяет ограничения на поля по входящим спискам полей разного типа

    Входные параметры:
    $report_options
    %O - именованные параметры
        force_group_by_fields => [qw/.../], обязательные для группировки поля
        allow_group_by_fields => [qw/.../], разрешенные для группировки поля
        forbid_fields => [qw/.../], запрещенные для чего-либо поля (группровка/фильтрация/слолбцы)

=cut
    
sub _limit_fields_by_lists {
    my ($report_options, %O) = @_;

    my ($force_group_by_fields, $allow_group_by_fields, $forbid_fields) = map { $_ || [] } @O{qw/force_group_by_fields allow_group_by_fields forbid_fields/};
	# при манипуляциях ниже важно сохранить порядок полей в $report_options->{group_by}
	# добавляем обязательные группировки
	@{$report_options->{group_by}} = uniq @{$report_options->{group_by}}, @$force_group_by_fields;
	# оставляем только допустимые группировки
    if (@$allow_group_by_fields) {
        my %is_allowed_group_by_field = map { $_ => 1 } @$allow_group_by_fields;
        delete $is_allowed_group_by_field{$_} for @$forbid_fields;
        $report_options->{group_by} = [ grep { $is_allowed_group_by_field{$_} } @{$report_options->{group_by}} ];
        my $allowed_norm_filters = xminus([ 'goal_id', 'client_id',
                                            (map { @{_get_filter_aliases($_)} } @$allow_group_by_fields),
                                            @{ _get_countable_filters() }
                                          ],
                                          \@$forbid_fields );
        my $forbidden_norm_filters = xminus [map { norm_field_by_periods_suf($_) } keys %{$report_options->{filters}}],
                                            $allowed_norm_filters;

        for my $f (@$forbidden_norm_filters) {
            _remove_records_by_norm_field_name($report_options->{filters}, $f);
        }
    } else {
        my %is_forbidden_group_by_field = map { $_ => 1 } @$forbid_fields; 
        $report_options->{group_by} = [ grep { !$is_forbidden_group_by_field{$_} } @{$report_options->{group_by}} ];
        # оставляем только допустимые фильтры
        my $forbidden_norm_filters = [(map { @{_get_filter_aliases($_)} } @$forbid_fields)];
        for my $f (@$forbidden_norm_filters) {
            _remove_records_by_norm_field_name($report_options->{filters}, $f);
        }
    }
    
	# убираем недопустимые столбцы		
	$report_options->{columns} = xminus $report_options->{columns}, \@$forbid_fields;
}

=head2 _remove_records_by_norm_field_name
    по нормализованному названию поля (без суффиксов) удаляет все его упоминания в переданном хеше (сейчас используется для фильтров)
=cut

sub _remove_records_by_norm_field_name {
    my ($h, $field) = @_;

    for my $suf ('', periods_full_suffix_list()) {
        delete $h->{$field.$suf};
    }
}

=head2 remove_report_options_limits_for_xls
    
    убирает ограничение на объем выгружаемых в xls/xlsx/csv строк

=cut

sub remove_report_options_limits_for_xls {
    my ($report_options) = @_;

    delete $report_options->{$_} for qw/page_size page/;
}

=head2 add_report_options_limits_for_xls
    
    Добавляет ограничение на объем выгружаемых в xls/xlsx/csv строк

=cut

sub add_report_options_limits_for_xls {
    my ($report_options) = @_;

    hash_merge $report_options, {page_size => $XLS_ROWS_MAX_NUM, page => 1};
}

=head2 validate_report_options_for_plot

    Валидирует настройки отчета для построения графика по следующим правилам:

    1. Срез по дате нужен всегда.
    2. Если, помимо среза по дате, есть больше одного дополнительного среза, отдаем ошибку.
    3. Если у нас есть срез по дате и еще какой-то срез, но запрошено больше одной колонки данных, тоже отдаем ошибку.
    4. Если срез только по дате, даем выбрать любое количество колонок данных, но только колонки в двух размерностях (единицы/деньги/проценты).
       WARNING: этой фильтрации возможно не будет, так как фронт может захотеть кешировать у себя данные и загружать их разом
    5. Фильтры не проверяем

    Если что-то не так - возвращает строку с ошибкой, если все хорошо - ничего не возвращает. Входной $report_options не
    изменяется

=cut

sub validate_report_options_for_plot {
    my ($report_options) = @_;
    if (scalar @{$report_options->{columns}} == 0) {
        return iget("Не заданы колонки");
    }

    my $slices_num = scalar @{$report_options->{group_by}};
    if ($slices_num > 1) {
        return iget("Слишком много срезов");
    } elsif ($slices_num == 1) {
        if (scalar @{$report_options->{columns}} > 1) {
            return iget("Если задан срез, нельзя запросить данные больше чем по одному столбцу");
        }
    } else {
        my $used_dimensions = Stat::Plot::get_all_used_dimensions($report_options->{columns});
        if (scalar @$used_dimensions > 2) {
            return iget("Нельзя запрашивать данные о столбцах больше чем двух размерностей")
        }
    }
    return;
}

=head2 get_stat_report

    Получить данные статистики с заданными группировками, фильтрами и т.п.

=cut

sub get_stat_report {
    my ($UID, $uid, $report_options, %O) = @_;

    my $error = check_report_options($report_options, $uid);
    die $error if $error;
    
    my $client_id = get_clientid(uid => $uid);
    my $operator_client_id = get_clientid(uid => $UID);
    my $client_currency = get_client_currencies($client_id)->{work_currency};
    my $stat_currency = $client_currency;

    if ($report_options->{cid}) {
        my $camp_currency = ( get_camp_info($report_options->{cid}, undef, short => 1) // {} )->{currency};
        if ($camp_currency && $camp_currency eq 'YND_FIXED') {
            $stat_currency = $camp_currency;
        }
    } elsif ($report_options->{multi_clients_mode} && !$report_options->{is_mcc_mode}) {
        # для внутренней рекламы валюта всегда рубли
        $stat_currency = 'RUB';
        # TODO для остального Директа получать валюту по клиентам после вызова get_user_available_cids: DIRECT-125048
    }

    my %date_opts;
    if ($report_options->{compare_periods}) {
        %date_opts = ("start_date_a" => $report_options->{"date_from"},
                      "end_date_a" => $report_options->{"date_to"},
                      "start_date_b" => $report_options->{"date_from_b"},
                      "end_date_b" => $report_options->{"date_to_b"}, );
    } else {
        %date_opts = ("start_date" => $report_options->{"date_from"},
                      "end_date" => $report_options->{"date_to"}, );
    }
    my %stream_ext_opts = (%date_opts,
                           date_aggregation_by => $report_options->{group_by_date},
                           group_by   => [uniq(@{$report_options->{group_by}}, ($report_options->{group_by_date} // 'none') ne 'none' ? 'date' : ())],
                           options    => {
                                            with_nds => $report_options->{with_nds},
                                            with_discount => $report_options->{with_discount},
                                            currency => $stat_currency,
                                            partial_total_rows => 1,
                                            no_spec_and_all_prefix => 1,
                                            totals_separate => 1,
                                            compare_periods => $report_options->{compare_periods},
                                            region_level => $report_options->{region_level},
                                            operator_ClientID => $operator_client_id,
                                            is_mcc_mode => $report_options->{is_mcc_mode},
                                         },
                           translocal_params => {ClientID => $client_id},
                           ClientID_for_stat_experiments => $client_id,
                           stat_type => $report_options->{stat_type},
                       );

    $stream_ext_opts{options}->{use_page_id} = any { $_ eq 'page' } @{$stream_ext_opts{group_by}};
    @{$stream_ext_opts{group_by}} = map { $_ eq 'page_group' ? 'page' : $_ } @{$stream_ext_opts{group_by}};
    # для мастера отчетов на кампанию всегда добавляем группировку по кампаниям для нормальной работы колонки Процент полученных показов
    push @{$stream_ext_opts{group_by}}, 'campaign' if $report_options->{cid};

    if (Client::ClientFeatures::has_stat_4_digits_precision_feature($client_id)) {
        $stream_ext_opts{options}->{four_digits_precision} = 1;
    }

    $stream_ext_opts{options}->{with_winrate} = 1 if any { $_ eq 'winrate'} @{$report_options->{columns}};
    $stream_ext_opts{options}{extra_countable_fields} = [ grep { $EXTRA_COUNTABLE_FIELDS_MAP{$_} } @{$report_options->{columns}} ];

    my $custom_filters_is_empty = keys %{$report_options->{filters}} ? 0 : 1;

    # добавляем "скрытые" от пользователя фильтры     
    my $report_options_filters = yclone($report_options->{filters});
    if (any { $_ eq 'click_place' } @{$stream_ext_opts{group_by}}) {
        for my $suf ($report_options->{compare_periods} ? periods_suffix_list() : '') {
            # избавляемся от "неопределенного" места клика, если на нем висят только показы
            my $fl = $report_options_filters->{"clicks$suf"} //= {};
            if (($fl->{gt} // 0) <= 0) {
                $fl->{gt} = 0;
            }
        }
    }

    if (($report_options->{stat_type} || '') eq 'search_queries') {
        $report_options_filters->{targettype}->{eq} = 'search';
        # Костыль из DIRECT-66516
        # Ограничиваем площадки, если период в отчете залезает за 13-е июня
        # (дата после которой поисковые запросы на внешних площадках стали писаться корректно)
        my $compare_date = $date_opts{start_date};
        $compare_date =~ s/-//g;
        if ($compare_date le $BS_CORRUPTED_SEARCH_QUERIES_ON_PAGES_DATE) {
            my $page_filter = iget_noop('Яндекс');
            if (defined $report_options_filters->{page}->{eq}
                    && !ref $report_options_filters->{page}->{eq}
                    && $report_options_filters->{page}->{eq} ne $page_filter) {
                $report_options_filters->{page}->{eq} = '_NO_PAGE_';
            } else {
                $report_options_filters->{page}->{eq} = $page_filter;
            }
        }
        ###
        $report_options_filters->{search_query}->{ne} //= $Stat::Const::SIGNIFICANT_EMPTY_STRING;
    }
    ###

    my $cids_all = get_user_available_cids($UID, $uid, no_subjects => 1,
                                                       no_ecom_uc => $O{no_ecom_uc},
                                                       client_currency => $client_currency,
                                                       only_cid => $report_options->{cid},
                                                       is_multi_clients_mode => $report_options->{is_multi_clients_mode},
                                                       report_options_filters => $report_options_filters,
                                                       ($report_options->{cid} || $O{dont_check_camp_type_by_role}) ?
                                                           (dont_check_camp_type_by_role => 1,
                                                            dont_check_camp_currency => 1) : ());

    if (Client::ClientFeatures::has_mol_page_name_support($client_id)
        || Client::ClientFeatures::has_mol_page_name_support($operator_client_id)
    ) {
        my $new_filter_name = 'page_name';
        if ($report_options_filters->{page}) {
            # преобразуем фильтры eq/ne в contains/not_contains:
            # раньше названия площадок превращались в список `PageID` по вхождению подстроки, поэтому эти Фильтры
            # всегда работали как contains/not_contains
            if ($report_options_filters->{page}->{eq}) {
                $report_options_filters->{$new_filter_name} = {
                    contains => $report_options_filters->{page}->{eq}
                };
            }
            if ($report_options_filters->{page}->{ne}) {
                $report_options_filters->{$new_filter_name} = {
                    not_contains => $report_options_filters->{page}->{ne}
                };
            }
            delete $report_options_filters->{page};
        }
    }

    my $filters = prepare_filters_for_stream_ext($client_id, $cids_all, $report_options_filters);

    # Модель атрибуции приходит отдельным GET-параметром
    if ($report_options->{attribution_model}) {
        if (exists $filters->{goal_ids} && exists $filters->{goal_ids}->{eq}){
            $filters->{attribution_models} = {eq => $report_options->{attribution_model}};
        } else {
            $filters->{attribution_model} = {eq => $report_options->{attribution_model}};
        }
    }

    # https://st.yandex-team.ru/DIRECT-94152
    if ($report_options->{compare_periods} && exists $report_options->{filters}){
        if ($report_options->{filters}{goal_ids}){
            $filters->{single_goal_id}{eq} = $filters->{goal_ids}{eq}[0];
            delete $filters->{goal_ids};
        } elsif ($report_options->{filters}{goal_id}){
            $filters->{single_goal_id}{eq} = $filters->{goal_id}{eq}[0];
            delete $filters->{goal_id};
        }
    }

    my $filtered_cids = _merge_campaigns($cids_all, $filters);
    delete $filters->{campaign};
    if ($report_options->{is_multi_clients_mode}) {
        my $cids_limit = $report_options->{is_mcc_mode} ? $MCC_MULTI_CLIENTS_CAMPAIGNS_MAX_NUM : $MULTI_CLIENTS_CAMPAIGNS_MAX_NUM;
        return undef, 'TOO_MUCH_CAMPAIGNS_IN_FILTER', $cids_all if scalar @$filtered_cids > $cids_limit;
    }
    push @$filtered_cids, @{ get_one_column_sql(PPC(cid => $filtered_cids), [
        "SELECT sc.cid FROM subcampaigns sc",
        WHERE => {
            'sc.master_cid' => SHARD_IDS
        }
    ]) };

    $stream_ext_opts{filter} = $filters;
    $stream_ext_opts{oid} = get_orderids(cid => $filtered_cids);

    my $limits = $stream_ext_opts{limits} = {};
    
    if ($report_options->{order_by}) {
        $limits->{order_by} = [{field => $report_options->{order_by}, 
                                dir => lc($report_options->{order_dir} || 'asc')}];
    }

    if ($report_options->{page_size}) {
        $limits->{offset} = ($report_options->{page}-1) * $report_options->{page_size};
        $limits->{limit} = $report_options->{page_size};
    } elsif (exists $report_options->{limit} || exists $report_options->{offset}) {
        $limits->{limit} = $report_options->{limit};
        $limits->{offset} = $report_options->{offset};
    }

    if ($O{dynamic_data_ref}) {
        # заполняем при "расшифровке" условий нацеливания в статистике, для использования в $vars
        $stream_ext_opts{options}->{dynamic_data_ref} = $O{dynamic_data_ref};
    }

    $stream_ext_opts{options}->{external_countable_fields_override} =
        Stat::Fields::get_countable_fields_for_override_by_fields_array($report_options->{columns});
    my $stream_ext = new Stat::CustomizedArray::StreamedExt();
    $stream_ext->set_report_parameters(%stream_ext_opts);

    my $result = {};
    if ($O{use_stat_iterator}) {
        $result->{stat_iterator} = eval { $stream_ext->generate_streamed_ext_iterator() };
    } else {
        $result = eval { $stream_ext->generate_streamed_ext() };
    }

    if ($@) {
        if ($@ =~ /(Memory limit exceeded|Timeout exceeded|: 596)/) {
            return undef, 'BS_TOO_MUCH_STATISTICS', $cids_all;
        } else {
            die $@;
        }
    }
    $result->{order_ids} = $stream_ext_opts{oid};

    if ($report_options->{region_level}) {
        $result->{region_level} = $report_options->{region_level};
    }

    if (($report_options->{with_resources} // 0) == 1) {
        $result->{with_resources} = $report_options->{with_resources};
    }

    # для отчета по поисковым запросам, когда не заданы пользовательские фильтры, 
    # нужны также суммарные цифры по непопулярным поисковым запросам (без расшифровки)
    if (($report_options->{stat_type} || '') eq 'search_queries' && $custom_filters_is_empty) {
        my $new_stream_ext_opts = yclone \%stream_ext_opts;
        delete $new_stream_ext_opts->{filter}->{search_query};
        hash_merge $new_stream_ext_opts, { group_by => ['search_query'],
                                           date_aggregation_by => 'none',
                                           limits => { order_by => [{field => 'search_query', dir => 'asc'}],
                                                       limit => 1, }
                                         };
        my $local_stream_ext = Stat::CustomizedArray::StreamedExt->new();
        $local_stream_ext->set_report_parameters(%$new_stream_ext_opts);
        my $all_search_queries_result = $local_stream_ext->generate_streamed_ext();
        if ($all_search_queries_result->{data_array}->[0] && $all_search_queries_result->{data_array}->[0]->{search_query} eq '') {
            $result->{other_search_queries_totals} = $all_search_queries_result->{data_array}->[0];
            $result->{totals} = $all_search_queries_result->{totals};
        }
    }
    # для web передаем названия полей формата name_goal_attribution без attribution
    # например agoalnum_34141983_2 -> agoalnum_34141983
    my $attribution_model_bs_value = $ATTRIBUTION_MODEL_TYPES{$report_options->{attribution_model} // get_attribution_model_default()}{bs};
    for my $data ($result->{totals} // {}, @{$result->{data_array}}){
        for my $field (keys %$data) {
            if ($field =~ /^(\w+_\d+)_(\d+)$/) {
                if ($2 == $attribution_model_bs_value) {
                    $data->{$1} = $data->{$field};
                }
                delete $data->{$field};
            }
        }
    }
    return $result, undef, $cids_all;
}

=head2 check_incomplete_data

    Возвращяет список полей (срез/фильтр/колонка) с указанием даты, с которой это поле актуально в плане
    полноты статистики. Уситываются только поля из конкретного запроса статистики.

=cut

sub check_incomplete_data {
    my ($report_options) = @_;

    my %fields_by_date = (
        $Stat::Const::BS_MRC_VISIBILITY_BORDER_DATE => [qw/
            viewable_impressions_mrc
            nonviewable_impressions_mrc
            undetermined_impressions_mrc
            measured_rate_mrc
            viewable_rate_mrc
        /],
    );

    my %check_fields = ( fp_shows_avg_pos => $BS_STREAM_AVG_POS_BORDER_DATE,
                         fp_clicks_avg_pos => $BS_STREAM_AVG_POS_BORDER_DATE,
                         device_type => $BS_DEVICE_TYPE_BORDER_DATE,
                         age => $BS_AGE_GENDER_BORDER_DATE,
                         gender => $BS_AGE_GENDER_BORDER_DATE,
                         connection_type => $BS_CONNECTION_TYPE_BORDER_DATE,
                         detailed_device_type => $BS_DETAILED_DEVICE_TYPE_BORDER_DATE,
                         bounce_ratio => $BS_BOUNCES_BORDER_DATE,
                         contextcond_ext => { border_date => $BS_EXT_PHRASE_BORDER_DATE,
                                              add_stat_type_prefix => 1 },
                         search_query => { border_date => $BS_SEARCH_QUERIES_BORDER_DATE,
                                           days_before_now => $BS_SEARCH_QUERIES_LAST_DAYS,
                                           partly_incomplete_date => $BS_CORRUPTED_SEARCH_QUERIES_ON_PAGES_DATE,},
                         bm_type => $BS_BROADMATCH_TYPE_BORDER_DATE,
                         retargeting_coef => $BS_RETARGETING_COEF_BORDER_DATE,
                         aprgoodmultigoal => $BS_PRGOODMULTIGOAL_BORDER_DATE,
                         aprgoodmultigoal_cpa => $BS_PRGOODMULTIGOAL_BORDER_DATE,
                         aprgoodmultigoal_conv_rate => $BS_PRGOODMULTIGOAL_BORDER_DATE,
                         avg_bid => $BS_AVG_BID_BORDER_DATE,
                         avg_cpm_bid => $BS_AVG_BID_BORDER_DATE,
                         agoals_profit => $BS_PROFIT_BORDER_DATE,
                         turbo_page_type => $BS_TURBO_PAGE_TYPE_BORDER_DATE,
                         targeting_category => $BS_TARGETING_CATEGORY_BORDER_DATE,
                         prisma_income_grade => $BS_PRISMA_INCOME_GRADE_BORDER_DATE,
                         (map { my $date = $_; map { $_ => $date; } @{$fields_by_date{$date}}; } keys %fields_by_date)
                        );

    my %fields_alias = ( phrase_ext => 'contextcond_ext');

    my %incomplete_data = ();
    for my $suf ($report_options->{compare_periods} ? ('', '_b') : '') {
        my ($date_from, $date_to) = map { date($report_options->{$_.$suf})->ymd('') } qw/date_from date_to/;

        for my $fld (uniq map { $fields_alias{$_} // $_ }
                          @{$report_options->{group_by} // []}, 
                          @{$report_options->{columns} // []}, 
                          (map { (norm_field_by_periods_suf($_))[0] } keys %{$report_options->{filters} // {}}),
                          'available_date') {
            if ($check_fields{$fld}) {
                my $limits = ref $check_fields{$fld} ? $check_fields{$fld} : { border_date => $check_fields{$fld} };

                my ($applied_limit_type, $applied_border_date);
                for my $limit_type (qw/border_date days_before_now partly_incomplete_date/) {  # вначале идут более приоритетные типы ограничени
                    next unless defined $limits->{$limit_type};

                    my $border_date = $limit_type eq 'days_before_now' 
                                            ? date(today())->subtract(days => $limits->{$limit_type})->ymd('')
                                            : $limits->{$limit_type};
                    if ($date_from <= $border_date && (!$applied_border_date || $border_date > $applied_border_date)) {
                        $applied_limit_type = $limit_type;
                        $applied_border_date = $border_date;
                    }
                }

                if ($applied_limit_type) {
                    my $result_field = $limits->{add_stat_type_prefix} && $report_options->{stat_type} ? $report_options->{stat_type}."_$fld" : $fld;
                    $result_field .= ($applied_limit_type eq 'partly_incomplete_date' ? '_partly_incomplete' : '');
                    $incomplete_data{$result_field} = $applied_limit_type eq 'days_before_now' 
                                                    ? $limits->{$applied_limit_type}
                                                    : date($limits->{$applied_limit_type})->add(days => 1)->ymd();
                }
            }
        }
    }

    # частный случай, из-за специфического применения граничных дат для ДРФ и Поисковых запросов
    delete $incomplete_data{search_queries_contextcond_ext} if $incomplete_data{search_query};

    return \%incomplete_data;
}

=head2 get_user_available_cids

    Возвращает список доступных пользователю кампаний клиента
    %O - именованные параметры (не обязательные)
        client_currency => 'RUB' - валюта 
        only_cid => 123|[123,212] - ограничиться перечисленными кампаниями
        no_subjects => 1 - исключить подлежащие кампании
        no_ecom_uc => 1 - исключить Товарные кампании (Ecom UC)
        dont_check_camp_type_by_role => 1 - не проверять, доступен ли тип кампании конкретно пользователю
        dont_check_camp_currency => 1 - не отфильтровывать фишечные кампании у сконвертированного клиента
        is_multi_clients_mode => 1 - получение статистики в режиме Мультиклиентности
        report_options_filters => {...} - фильтры полученные из веб-интерфейса
=cut

sub get_user_available_cids {
    my ($UID, $uid, %O) = @_;

    my $rbac = RBAC2::Extended->get_singleton(1);

    my $cids_all;
    if ( $O{is_multi_clients_mode} )  {
        my $filters = _prepare_filters_for_get_cids_in_multi_clients_mode($O{report_options_filters});
        die "Prepared filters must be not empty" unless %$filters;

        my %conditions = ();
        my @join_tables = ();
        for my $filter_name (keys %$filters) {
            for my $filter_type (qw/eq ne/) {
                next unless exists $filters->{$filter_name}{$filter_type};
                my $values = $filters->{$filter_name}{$filter_type};

                my $db_field = $MULTI_CLIENTS_FILTER_TO_DB_FIELDS{$filter_name}->{$filter_type};
                $conditions{$db_field} = $values;
            }

            if ( $filter_name eq 'place_id' ) {
                push @join_tables, "JOIN campaigns_internal cint ON cint.cid = c.cid";
            }
        }
        die "Sql conditions must be not empty" unless %conditions;

        if ( $O{no_subjects} ) {
            push @join_tables, "LEFT JOIN subcampaigns sc_m ON sc_m.cid = c.cid";
            $conditions{'sc_m.master_cid__is_null'} = 1;
        }

        if ( $O{no_ecom_uc} ) {
            $conditions{'c.metatype__ne'} = 'ecom';
        }

        # если есть фильтрация по client_id, то шарды будем искать тоже по ним, иначе будем искать по всем шардам
        my ($key, $values) = $filters->{client_id}{eq} ? ('ClientID', $filters->{client_id}{eq}) : ('shard', 'all');
        $cids_all = get_one_column_sql(PPC($key => $values), [
            "SELECT c.cid FROM campaigns c",
            @join_tables,
            WHERE => {'c.statusEmpty' => 'No',
                'c.OrderID__gt' => 0,
                # доступные типы смотрим на оператора, чтобы новые типы которые скрыты за фичей по дефолту не были доступны
                'c.type' => get_user_available_camp_types($UID, client_client_id => get_clientid(uid => $UID)),
                %conditions,
            }
        ]);
    } else {
        $uid = rbac_get_chief_rep_of_client_rep($uid);

        my %conditions = ();
        my @join_tables = ();

        $O{client_currency} //= get_client_currencies(get_clientid(uid => $uid))->{work_currency};
        if ($O{client_currency} ne 'YND_FIXED' && !$O{dont_check_camp_currency}) {
            $conditions{'c.currency__is_not_null'} = 1;
            $conditions{'c.currency__ne'} = 'YND_FIXED';
        }

        if ($O{only_cid}) {
            $conditions{'c.cid'} = $O{only_cid};
        }

        if ($O{no_subjects})  {
            push @join_tables, "LEFT JOIN subcampaigns sc_m ON sc_m.cid = c.cid";
            $conditions{'sc_m.master_cid__is_null'} = 1;
        }

        if ($O{no_ecom_uc}) {
            $conditions{'c.metatype__ne'} = 'ecom';
        }

        $cids_all = get_one_column_sql(PPC(uid => $uid), ["SELECT c.cid FROM campaigns c",
                                                           @join_tables,
                                                           WHERE => {'c.statusEmpty' => 'No',
                                                                     'c.type' => get_user_available_camp_types($UID,
                                                                                client_client_id => get_clientid(uid => $uid),
                                                                                %{hash_cut \%O, 'dont_check_camp_type_by_role'},
                                                                             ),
                                                                     'c.OrderID__gt' => 0,
                                                                     'c.uid' => $uid,
                                                                     %conditions}]);
    }

    my $cids_availability = rbac_check_owner_of_camps($rbac, $UID, $cids_all);

    return [grep { $cids_availability->{$_} } @$cids_all];
}

=head2 get_user_available_camp_types 

    Возвращает список доступных пользователю типов кампаний, для просмотра статистики в МОЛ
    Нужно для постепенного открытия полей/значений связанных с новыми типами кампаний, сначала внутренним ролям, а затем на всех
    %O - именованные параметры (не обязательные)
        dont_check_camp_type_by_role => 1 - не проверять, доступен ли тип кампании конкретно пользователю
        client_client_id - для проверки доступности типа кампании по "фиче на клиента"

=cut

sub get_user_available_camp_types {
    my ($UID, %O) = @_;

    my $types = get_camp_kind_types('stat_stream_ext');

    # кроме гео-кампаний
    my @exclude_types = qw/geo/;
    $types = xminus $types, \@exclude_types;

    unless ($O{dont_check_camp_type_by_role}) { 
        my $rbac = RBAC2::Extended->get_singleton(1);
        my $user_role = rbac_who_is($rbac, $UID);

        if ($user_role ne 'super' && !Client::ClientFeatures::get_is_feature_cpm_banner_enabled(client_id => $O{client_client_id})) {
            $types = xminus $types, [qw/cpm_banner/]
        }

        if ($user_role ne 'super' && !Client::ClientFeatures::has_cpm_deals_allowed_feature($O{client_client_id})) {
            $types = xminus $types, [qw/cpm_deals/]
        }
        if (!content_promotion_in_stat_allowed_by_role($user_role) && !Client::ClientFeatures::has_any_content_promotion_feature_enabled($O{client_client_id})) {
            $types = xminus $types, [qw/content_promotion/]
        }
        my $perminfo = Rbac::get_perminfo( ClientID => $O{client_client_id} );
        unless ( internal_ads_in_stat_allowed_by_role($user_role) || Rbac::has_perm( $perminfo, 'internal_ad_product' ) ) {
            $types = xminus $types, [qw/internal_distrib internal_free internal_autobudget/]
        }
    }
    return $types;
}

=head2 check_client_has_brand_safety_campaigns

    Проверяем есть ли у клиента кампании с запущенным хотябы раз brand safety

=cut

sub check_client_has_brand_safety_campaigns {
    my $client_id = shift;
    return get_one_field_sql(PPC(ClientID => $client_id), ["SELECT COUNT(cid) FROM campaigns",
        where => { ClientID => $client_id, brandsafety_ret_cond_id__is_not_null => 1, sum_spent__gt => 0}]) > 0;
}

=head2 check_campaigns_have_brand_safety

    Проверяем есть были ли на кампаниях запущены brand safety

=cut

sub check_campaigns_have_brand_safety {
    my $cids = shift;
    return get_one_field_sql(PPC(cid => $cids), ["SELECT COUNT(cid) FROM campaigns",
        where => {cid => $cids, brandsafety_ret_cond_id__is_not_null => 1, sum_spent__gt => 0}]) > 0;
}

=head2 internal_ads_in_stat_allowed_by_role

    Разрешаем ли доступ к Внутренней рекламе в статистике для переданной роли

=cut

sub internal_ads_in_stat_allowed_by_role {
    my ($operator_role) = @_;
    
    return $operator_role =~ /^(?:super|superreader|internal_ad_(?:admin|manager|superreader))$/ ? 1 : 0;
}

=head2 content_promotion_in_stat_allowed_by_role

    Разрешаем ли доступ к кампаниям типа Продвижения контента в статистике для переданной роли

=cut

sub content_promotion_in_stat_allowed_by_role {
    my ($operator_role) = @_;
    
    return $operator_role =~ /^(super|superreader|support|limited_support|internal_ad_(admin|manager|superreader))$/ ? 1 : 0;
}

=head2 merge_minus_words_length_info

    В случае необходимости (если есть группировка по статусам ПЗ/ДРФ) вычисляет и добавляет в $vars информацию о длине минус-фраз на кампаниях/группах
    которые встречаются в статистике на странице

=cut

sub merge_minus_words_length_info {
    my ($vars, $report_options) = @_;
    
    my $profile = Yandex::Trace::new_profile('stat_report_master:merge_minus_words_length_info');

    return unless @{$vars->{data_array} // []} && any { m/^(search_query)$/ } @{$report_options->{group_by}};
    #EXPIRES after DIRECT-65257 && any { m/^(search_query_status|ext_phrase_status)$/ } @{$report_options->{group_by}};

    my (%adgroups, %campaigns);
    for my $row (@{$vars->{data_array}}) {
        $adgroups{$row->{adgroup_id}} = 0 if $row->{adgroup_id};
        $campaigns{$row->{cid}} = 0 if $row->{cid};
    }

    if (keys %campaigns) {
        my $mw_by_cid = get_hash_sql(PPC(cid => [keys \%campaigns]), ["select cid, minus_words from camp_options", where => {cid => SHARD_IDS}]);
        for my $cid (keys %$mw_by_cid) {
            my $mw_array = MinusWordsTools::minus_words_str2array($mw_by_cid->{$cid});
            $campaigns{$cid} = MinusWordsTools::minus_words_length_for_limit($mw_array); 
        }

        hash_merge $vars, {campaigns_minus_words_length => \%campaigns};
    }

    if (keys %adgroups) {
        my $mw_by_pid = get_hashes_hash_sql(PPC(pid => [keys \%adgroups]), ["select p.pid, p.mw_id, mw.mw_text 
                                                                               from phrases p
                                                                               join minus_words mw on p.mw_id = mw.mw_id", 
                                                                              where => {'p.pid' => SHARD_IDS}]);
        my %mw_len_by_mw_id = ();
        for my $pid (keys %$mw_by_pid) {
            my $mw_info = $mw_by_pid->{$pid};
            unless (defined $mw_len_by_mw_id{$mw_info->{mw_id}}) {
                my $mw_array = MinusWordsTools::minus_words_str2array($mw_info->{mw_text});
                $mw_len_by_mw_id{$mw_info->{mw_id}} = MinusWordsTools::minus_words_length_for_limit($mw_array);
            }
            
            $adgroups{$pid} = $mw_len_by_mw_id{$mw_info->{mw_id}};
        }

        hash_merge $vars, {adgroups_minus_words_length => \%adgroups};
    }
}

=head2 _error_result
=cut

sub _error_result {
    return {result => 'error', error => shift};
}

{
    
=head2 %filter_converter

    Структура, описывающая какие функции-конвертеры нужно применить к отдельным фильтрам, пришедшим из интерфейса,
    а также зависимости результатов конвертации от других фильтров (в таком случае сначала нужно обработать фильтры-зависимости)

=cut
my %filter_converter = (campaign          => undef,

                        campaign_strategy => {converters => \&_convert_camp_strategy_to_cid,
                                              field_name => 'campaign'},

                        campaign_status   => {converters => \&_convert_camp_status_to_cid,
                                              field_name => 'campaign'},

                        campaign_type     => {converters => \&_convert_camp_type_to_cid,
                                              field_name => 'campaign'},
                        strategy_id       => {converters => \&_split_num},

                        tags              => undef,

                        attribution_model => undef,
                        attribution_models => undef,

                        adgroup           => {converters => [\&_split_text, [\&_grep_num, \&_convert_adgroup_name_to_pid] ],
                                              operations => {starts_with => 'eq',
                                                             not_starts_with => 'ne'} },

                        adgroup_id        => {converters => \&_split_num},

                        adgroup_status    => {converters => \&_convert_adgroup_status_to_pid,
                                              field_name => 'adgroup',
                                              depends    => [qw/campaign campaign_strategy campaign_type campaign_status/]} ,

                        banner            => {converters => \&_split_num},

                        banner_status     => {converters => \&_convert_banner_status_to_bid,
                                              field_name => 'banner',
                                              depends    => [qw/campaign campaign_strategy campaign_type campaign_status/]},
                        banner_type       => undef,
                        phrase            => {converters => \&_split_text, linked_group_by => 'contextcond'},
                        phrase_ext        => {converters => \&_split_text, linked_group_by => 'contextcond_ext'},
                        phrase_orig       => {converters => \&_split_text, linked_group_by => 'contextcond_orig'},
                        rmp_interest      => {field_name => 'retargeting'},
                        retargeting       => {linked_group_by => [qw/contextcond contextcond_ext/]},
                        dynamic           => {converters => \&_convert_dyn_cond_uniq_id_to_ids, linked_group_by => [qw/contextcond contextcond_ext contextcond_orig/]},
                        performance       => {converters => \&_convert_perf_filter_uniq_id_to_ids, linked_group_by => [qw/contextcond contextcond_ext contextcond_orig/]},
                        matched_phrase    => {converters => \&_split_text},
                        matched_phrase_id => undef,
                        match_type        => undef,
                        retargeting_coef  => undef,
                        contexttype       => undef,
                        contexttype_orig  => undef,
                        criterion_type    => undef,
                        bm_type           => undef,
                        sim_distance      => {converters => \&_split_num},
                        region            => {converters => \&_convert_geo_str_to_region},
                        physical_region   => {converters => \&_convert_geo_str_to_region},
                        targettype        => undef,
                        page              => {converters => [\&_split_text, \&_convert_page_name_to_id]},
                        page_name         => {converters => [\&_split_text]},
                        position          => undef,
                        has_image         => undef,
                        image_size        => undef,
                        banner_image_type => undef,
                        device_type       => undef,
                        gender            => undef,
                        age               => undef,
                        detailed_device_type => undef,
                        connection_type   => undef,
                        click_place       => undef,
                        ssp               => {converters => \&_split_text},
                        search_query      => {converters => \&_split_text},
                        search_query_status => undef,
                        ext_phrase_status => undef,
                        goal_id           => undef,
                        goal_ids          => undef,
                        goal_id_list          => undef,
                        deal_export_id    => undef,
                        ab_segment    => {converters => \&_split_num},
                        inventory_type    => undef,
                        operating_system    => undef,
                        browser    => undef,
                        template_id    => {converters => \&_split_num},
                        client_id    => {converters => \&_split_num},
                        place_id    => {converters => \&_split_num},
                        place_description    => undef,
                        page_id    => {converters => \&_split_num},
                        content_targeting    => {converters => \&_split_num},
                        region_level    => undef,
                        turbo_page_type => undef,
                        targeting_category => undef,
                        prisma_income_grade => undef,
                        bs_criterion_id => undef,
                        region_source => undef,

                        map { $_ => {converters => \&_normalize_numbers, is_countable => 1} } (qw/
                            shows
                            eshows
                            clicks
                            ctr
                            ectr
                            avg_x
                            sum
                            av_sum
                            fp_shows_avg_pos
                            fp_clicks_avg_pos
                            winrate
                            bounce_ratio
                            adepth
                            aconv
                            agoalnum
                            agoalcost
                            agoalroi
                            agoalcrr
                            agoalincome
                            aprgoodmultigoal
                            avg_view_freq
                            uniq_viewers
                            avg_cpm
                            aprgoodmultigoal_cpa
                            aprgoodmultigoal_conv_rate
                            avg_bid
                            avg_cpm_bid
                            avg_time_to_conv
                            agoals_profit
                            auction_hits
                            auction_wins
                            auction_win_rate
                            imp_to_win_rate
                            imp_reach_rate
                            served_impressions
                            close_clicks
                            close_ctr
                            pv_bounce_ratio
                            pv_adepth
                            pv_aconv
                            pv_agoalcost
                            pv_agoalnum
                            pv_agoalroi
                            pv_agoalcrr
                            pv_agoalincome
                            pv_agoals_profit
                            pv_aprgoodmultigoal
                            pv_aprgoodmultigoal_cpa
                            pv_aprgoodmultigoal_conv_rate
                            pv_avg_time_to_conv 
                        /, Stat::Fields::get_countable_filter_fields()) );

=head2 _get_filter_aliases

    Возвращает ссылку на список фильтров интерфейса, соответствующих заданному фильтру или срезу, поддерживаемому БК (Stat::StreamExtended)

=cut

sub _get_filter_aliases {
    my $filter = shift;


    my @aliases = ();
    if (defined $filter) {
        for my $f (keys %filter_converter) {
            my $v = $filter_converter{$f} // {};
            if ($f eq $filter 
                || ($v->{field_name} // '') eq $filter
                || ($v->{linked_group_by} && any { $_ eq $filter } xflatten($v->{linked_group_by}))) {
                push @aliases, $f;
            } 
        }
    }

    return \@aliases;
}

=head2 _get_countable_filters

    Возвращает ссылку на список фильтров по исчислимым полям (показы, клики, и т.п.)

=cut

sub _get_countable_filters {
    return [ grep { $filter_converter{$_}->{is_countable} } keys %filter_converter ];
}

=head2 _sort_filter_fields_with_dependencies

    Сортирует список фильтров с учетом зависимости, для их дальнейшей обработки

=cut

sub _sort_filter_fields_with_dependencies {
    my %list = map { $_ => 1 } @_;

    my @sorted_list;
    while (keys %list) {
        my $changed = 0;
        for my $f (keys %list) {
            my $dependencies = $filter_converter{$f}->{depends} // [];
            if (none { $list{$_} } @$dependencies) {
                push @sorted_list, $f;
                delete $list{$f};
                $changed = 1;
            }
        }
        die "Unbreakable loop detected" unless $changed;
    }
    return @sorted_list;

}

=head2 _get_multi_clients_filters
    
    Получить фильтры Мультиклиентности
    На выходе id'шники разделенные пробелом будут в виде списка

=cut

sub _get_multi_clients_filters {
    my ($filters) = @_;

    my $supported_filters = hash_cut $filters || {}, @SUPPORTED_FILTERS_FOR_GET_CIDS_IN_MULTI_CLIENTS_MODE;

    # для конверторов передаваемых фильтров первые два параметра не нужны
    return prepare_filters_for_stream_ext(undef, undef, $supported_filters);
}    

=head2 _prepare_filters_for_get_cids_in_multi_clients_mode
    
    Подготовка фильтров пришедших из веб-интерфейса для использования в методе get_user_available_cids для Мультиклиентности

=cut

sub _prepare_filters_for_get_cids_in_multi_clients_mode {
    my ($filters) = @_;

    my $prepared_filters = _get_multi_clients_filters($filters);
    
    # для page_id получаем из ppcdict - place_id
    if ( $prepared_filters->{page_id} ) {
        my $filter_type = exists $prepared_filters->{page_id}{eq} ? 'eq' : 'ne';
        my $page_ids = $prepared_filters->{page_id}{$filter_type};
        
        my $place_ids = get_one_column_sql(PPCDICT, [
            "SELECT DISTINCT place_id",
            "FROM pages",
            "JOIN page_place ON page_place.page_id = pages.orig_page_id",
            WHERE => { 'PageID' => $page_ids }
        ]);

        $prepared_filters->{place_id} //= {};
        # если передавали фильтр по place_id, то нужно оставить пересечение этих id с полученными по page_id
        if ( exists $prepared_filters->{place_id}{$filter_type} ) {
            $prepared_filters->{place_id}{$filter_type} = xisect($prepared_filters->{place_id}{$filter_type}, $place_ids)
        } else {
            # пересечение id'шников в положительном и в отрицательном фильтре оставим - база разрулит, ожидается малое кол-во id
            $prepared_filters->{place_id}{$filter_type} = $place_ids;
        }
        
        delete $prepared_filters->{page_id};
    }
    
    return $prepared_filters;
}

=head2 prepare_filters_for_stream_ext

    Подготовка фильтров пришедших из веб-интерфейса для использования в Stat::StreamExtended

=cut

sub prepare_filters_for_stream_ext {
    my ($client_id, $cids_all, $filters) = @_;

    my $stream_ext_filters = {};
    for my $field (_sort_filter_fields_with_dependencies(grep { exists $filter_converter{norm_field_by_periods_suf($_)} } keys %$filters)) {
        my ($norm_field, $suf) = norm_field_by_periods_suf($field);
        my $field_opts = $filter_converter{$norm_field} // {};
        my $converters = $field_opts->{converters} // [];
        $converters = [$converters] unless ref $converters eq 'ARRAY';
        for my $op (keys %{$filters->{$field}}) {
            my @values = grep { m/\S+/ } xflatten $filters->{$field}->{$op};
            next unless @values;
            my $conv_op = ($field_opts->{operations} ? $field_opts->{operations}->{$op} : undef) // $op;
            for my $conv_group (@$converters) {
                my @conv_values;
                for my $conv (xflatten $conv_group) {
                    my $conv_values_part = $conv->(\@values, $op, $client_id, $cids_all, $stream_ext_filters);
                    push @conv_values, @$conv_values_part;
                }
                @values = @conv_values;
            }
            @values = uniq @values;

            my $value;
            my $field_filter = $stream_ext_filters->{($field_opts->{field_name} // $norm_field).$suf} //= {};
            if ($field_opts->{single_value} || any { $conv_op eq $_ } qw/lt gt/) {
                $value = $values[0];
            } elsif ($field_filter->{$conv_op} && any { $conv_op eq $_ } qw/eq starts_with contains contains_in_plus_part/) {
                # случай field IN (1,2,3) AND field IN (3,4,5), в результате должно остаться пересечение значений
                $value = xisect $field_filter->{$conv_op}, \@values;
            } elsif ($field_filter->{$conv_op} && any { $conv_op eq $_ } qw/ne not_starts_with not_contains not_contains_in_plus_part/) {
                # случай field NOT IN (1,2,3) AND field NOT IN (3,4,5), в результате должно остаться объединение значений
                $value = [uniq @{$field_filter->{$conv_op}}, @values];
            } else {
                $value = \@values;
            }
            $field_filter->{$conv_op} = $value;
        }
    }
    return $stream_ext_filters;
}

}

=head2 _split_text

=cut

sub _split_text {
    return [grep { m/\S+/ } map { split /[\n\r\t]+/, $_ // '' }  @{$_[0]}];
}

=head2 _split_num

=cut

sub _split_num {
    return [grep { $_ } map { split /\D+/, $_ // '' }  @{$_[0]}];
}

=head2 _grep_num

=cut

sub _grep_num {
    return [grep { is_valid_int($_) } @{$_[0]}];
}

=head2 _normalize_numbers

=cut

sub _normalize_numbers {
    return [map { s/,/./g; $_ } @{$_[0]}];
}

=head2 _campaign_stat_strategy_to_strategy_name

Определение названия стратегии кампании по описанию стратегии из mass_campaign_strategy

=cut

sub _campaign_stat_strategy_to_strategy_name {
    my $st = shift;

    # отдельно проверим смарты
    if (($st->{name} // '') =~ /^autobudget_avg_cp[ac]_per_/) {
        return $st->{name};
    }

    my $is_search_stop = ($st->{is_search_stop} || ($st->{search}->{name} // '') eq 'stop') ? 1 : 0;
    # my $is_net_stop = ($st->{is_net_stop} || $st->{net}->{name} // '' eq 'stop') ? 1 : 0;

    if ($is_search_stop) { # гарантирует раздельное управление
        # имя на сети должно быть определено при любом раздельном управлении
        if ($st->{net}->{name} eq 'autobudget') {
            return (defined $st->{net}->{goal_id}) ? 'week_autobudget_avg_cpa' : 'week_autobudget_avg_click';
        }
        if ($st->{net}->{name} eq 'maximum_coverage') {
            return 'manual_control';
        }
        return $st->{net}->{name};
    }
    # имя на поиске должно быть определено, если показы не отключены
    if ($st->{search}->{name} eq 'no_premium') {
        if (($st->{net}->{name} // '') eq 'maximum_coverage') {
            return 'manual_control_different_places';
        }
        return 'manual_control_no_premium';
    }
    if ($st->{search}->{name} eq 'default') {
        if (($st->{net}->{name} // '') eq 'maximum_coverage') {
           return 'manual_control_different_places';
        }
        return 'manual_control';
    }
    if ($st->{search}->{name} eq 'autobudget') {
        return (defined $st->{search}->{goal_id}) ? 'week_autobudget_avg_cpa' : 'week_autobudget_avg_click';
    }
    return $st->{search}->{name};
}

=head2 _convert_camp_strategy_to_cid

=cut

sub _convert_camp_strategy_to_cid {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my @cids_filtered = ();
    my $sn = mass_campaign_strategy($cids_all);

    for my $cid (keys %$sn) {
        my $camp_strategy_name = _campaign_stat_strategy_to_strategy_name($sn->{$cid});

        if (any { $_ eq $camp_strategy_name } @$values) {
            push @cids_filtered, $cid;
        }
    }

    return \@cids_filtered;
}

=head2 _convert_camp_status_to_cid

=cut

sub _convert_camp_status_to_cid {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my @conditions = ();
    foreach my $status (@$values) {
        if ($status eq 'arch') {
            push @conditions, "c.archived = 'Yes'";
        } elsif ($status eq 'off') {
            push @conditions, "c.archived = 'No' AND (c.statusShow = 'No' OR (c.finish_time > 0 AND c.finish_time < CURRENT_DATE()) )";
        } elsif ($status eq 'active') {
            push @conditions, "c.archived = 'No' AND c.statusShow = 'Yes' AND (c.finish_time = 0 OR c.finish_time >= CURRENT_DATE())";
        }
    }

    my $cids_filtered = [];
    if (@conditions) {
        $cids_filtered = get_one_column_sql(PPC(ClientID => $client_id), 
            "SELECT c.cid FROM campaigns c WHERE statusEmpty = 'No' AND ClientID = ? AND ".
            '(' . join(') OR (', @conditions) . ')', $client_id);
    }

    return $cids_filtered;
}

=head2 _convert_camp_type_to_cid

=cut

sub _convert_camp_type_to_cid {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my $types_re = join '|', @{get_camp_kind_types('stat_stream_ext')};

    my @types = grep { m/^($types_re)$/ } @$values;

    return get_one_column_sql(PPC(ClientID => $client_id), ["SELECT cid FROM campaigns ", 
                                                   WHERE => { statusEmpty => 'No',
                                                              ClientID => $client_id,
                                                              cid => $cids_all,
                                                              type => \@types } ]);
}

=head2 _merge_campaigns

    Объединить список всех кампаний и заданные фильтры на кампании, чтобы
    получить список кампаний, доступных для фильтрации.

=cut

sub _merge_campaigns {
    my ($all_cids, $prepared_filters) = @_;

    if (!exists $prepared_filters->{campaign} || !%{$prepared_filters->{campaign}} ) {
        return $all_cids;
    }

    if (none { exists $prepared_filters->{campaign}->{$_} } qw/eq ne/) {
        die "Unsupported operation";
    }

    my $filtered_cids = [ @$all_cids ];
    if (exists $prepared_filters->{campaign}->{eq}) {
        $filtered_cids = xisect($filtered_cids, $prepared_filters->{campaign}->{eq});
    }
    if (exists $prepared_filters->{campaign}->{ne}) {
        $filtered_cids = xminus($filtered_cids, $prepared_filters->{campaign}->{ne});
    }
    return $filtered_cids;
}

=head2 _convert_adgroup_name_to_pid

=cut

sub _convert_adgroup_name_to_pid {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my $pids = get_one_column_sql(PPC(ClientID => $client_id), ["SELECT pid FROM phrases", 
                                                       WHERE => {
                                                                    cid => $cids_all,
                                                                    ($op =~ /^(starts_with|not_starts_with)$/
                                                                        ? ("group_name__contains_any" => $values)
                                                                        : (group_name => $values)
                                                                    )
                                                                }]);

    return $pids;
}

=head2 _convert_adgroup_status_to_pid

=cut

sub _convert_adgroup_status_to_pid {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my $cids = _merge_campaigns($cids_all, $prepared_filters);
    my @pids = ();

    foreach my $status (grep { m/^(active|off|arch)$/ } @$values) {
        my %condition = Models::AdGroupFilters::get_status_condition($status, filter => {cid => $cids});

        push @pids, @{ get_one_column_sql(PPC(ClientID => $client_id), ["
                                                     SELECT DISTINCT g.pid
                                                       FROM phrases g",
                                                       $condition{tables},
                                                      WHERE => {'g.cid' => $cids,
                                                                _TEXT => sql_condition($condition{where})} ]) };
    }

    return [uniq @pids];
}

=head2 _convert_banner_status_to_bid

=cut

sub _convert_banner_status_to_bid {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my $cids = _merge_campaigns($cids_all, $prepared_filters);
    my @bids = ();

    foreach my $status (grep { m/^(active|off|arch|decline)$/ } @$values) {
        my %condition = Models::AdGroupFilters::get_status_condition($status, filter => {cid => $cids});

        push @bids, @{ get_one_column_sql(PPC(ClientID => $client_id), ["
                                                     SELECT DISTINCT b.bid
                                                       FROM phrases g",
                                                       $condition{tables},
                                                      WHERE => {'g.cid' => $cids,
                                                                _TEXT => sql_condition($condition{where})} ]) };
    }

    return [uniq @bids];
}


=head2 _convert_page_name_to_id

=cut

sub _convert_page_name_to_id {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;
    my %lang_aliases = (ua => 'uk');

    my $lang = Yandex::I18n::current_lang;
    $lang = $lang_aliases{$lang} || $lang;

    return Stat::Tools::convert_page_name_filter_to_ids($values, $op, $lang);
}

=head2 _convert_geo_str_to_region

=cut

sub _convert_geo_str_to_region {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    return Stat::Tools::get_plus_regions_by_geo($values, {ClientID => $client_id});
}

=head2 _convert_dyn_cond_uniq_id_to_ids

    конвертирует id условия нацеливания в список всех условий с таким же названием

=cut

sub _convert_dyn_cond_uniq_id_to_ids {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my $cids = _merge_campaigns($cids_all, $prepared_filters);
    my $dyn_conds = Stat::Tools::get_dynamic_data($values, cid => $cids);

    return Stat::Tools::get_dyn_cond_ids_by_names([ map { $_->{name } } values %$dyn_conds ], cid => $cids);
}

=head2 _convert_perf_filter_uniq_id_to_ids

    конвертирует id фильтров ДМО (перфоманс/смарт-баннеров) в список всех фильтров с таким же названием

=cut

sub _convert_perf_filter_uniq_id_to_ids {
    my ($values, $op, $client_id, $cids_all, $prepared_filters) = @_;

    my $cids = _merge_campaigns($cids_all, $prepared_filters);

    my $perf_filer_names = Stat::Tools::get_performance_text($values, cid => $cids);

    return Stat::Tools::get_perf_filter_ids_by_names([values %$perf_filer_names], cid => $cids);
}

1;
