package Stat::RollbackNotify;

=encoding utf8

=head1 NAME

    Stat::RollbackNotify

=head1 DESCRIPTION

    В модуль вынесены функции, использующиеся в скрипте protected/ppcStatRollbackNotify.pl

=cut

use Direct::Modern;

use Date::Calc qw(Delta_Days check_date);

use Yandex::DBShards qw/foreach_shard_parallel/;
use Yandex::DBTools;
use Yandex::HTTP;

use Settings;

=head2 $BS_STAT_ROLLBACK_URL

    URL по которому получаем данные об изменениях статистики

=cut

our $BS_STAT_ROLLBACK_URL //= $Settings::BS_EXPORT_PROXY_READONLY.'export/order-stat-return.cgi';

=head2 get_stat_rollback_data

    Получить из бк данные об изменениях статистики

    Параметры:
        $log    - объект Yandex::Log для логирования
        $start  - строка с датой/временем, с которого требуется получить данные
    Результат:
        $orders_data    - ссылка на хеш с данными по изменениям статистики, сгруппированным по OrderID
        $last_done_time - строка с датой/временем, с которыми нужно выполнить следующий запрос

=cut

sub get_stat_rollback_data {
    my ($log, $start) = @_;

    $log->die("start time is required parameter") unless $start;

    my $http_res = http_get("$BS_STAT_ROLLBACK_URL?start=$start");
    $log->die("$BS_STAT_ROLLBACK_URL not respond") unless defined $http_res;

    my @rows = split(/\n/, $http_res);
    my $begin_marker = shift @rows;
    my $end_marker = pop @rows;

    $log->die("End marker check failed") unless $end_marker && $end_marker =~ /^#End/;

    my $last_done_time = 0;
    my @titles = split /\t/, shift @rows;
    my %orders_data = ();
    my %date_ok = ();
    foreach my $row (@rows) {
        my %parsed = ();
        @parsed{qw/OrderID done_time update_time shows clicks sum sum_cur drop_type actions/} = split /\t/, $row;
        push @{$orders_data{$parsed{OrderID}}}, \%parsed;
        foreach my $k (qw(done_time update_time)) {
            next if exists $date_ok{$parsed{$k}};
            my @d = $parsed{$k} =~ /^(\d{4})(\d\d)(\d\d)(\d\d)(\d\d)(\d\d)$/;
            $log->die("Incorrect date $parsed{$k}") unless check_date(@d[0..2]);
            $date_ok{$parsed{$k}} = 1;
        }
        $last_done_time = $parsed{done_time} if $parsed{done_time} > $last_done_time;
    }

    return \%orders_data, $last_done_time;
}

=head2 update_direct_stat_rollbacks

    Записывает данные в таблицу ppcdict.stat_rollbacks

    Параметры:
        $orders - ссылка на хеш хешей с данными по заказам, сгруппированным по OrderID

=cut

sub update_direct_stat_rollbacks {
    my $orders = shift;

    my @data;
    foreach my $oid ( keys %$orders ) {
        push @data, map {[ $oid => $_->{update_time} ]} @{$orders->{$oid}};
    }

    return unless @data;

    do_mass_insert_sql(PPCDICT,
        'INSERT INTO stat_rollbacks (OrderID, border_date) VALUES %s',
        \@data
    );
    foreach_shard_parallel OrderID => [keys %$orders], sub {
        my ($shard, $order_ids) = @_;

        my @sharded_data;
        foreach my $oid (@$order_ids) {
            push @sharded_data, map {[$oid => $_->{update_time}]} @{$orders->{$oid}};
        }

        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT INTO stat_rollbacks (OrderID, border_date) VALUES %s',
            \@sharded_data
        );
    };
}

=head2 group_periods

    Группирует массив дат в массив последовательных периодов ("склеивает" соседние даты в периоды)

    Параметры:
        $periods    - ссылка на массив дат вида 'YYYY-MM-DD'
    Результат:
        $intervals  - ссылка на массив периодов - хешей вида:
                    {
                      start => 'YYYY-MM-DD',    # первый день непрерываного периода
                      stop  => 'YYYY-MM-DD',    # последний день непрерывного периода.
                                                # отсутствует, если период из одного дня
                    }

=cut

sub group_periods {
    my $periods = shift;
    my @intervals = ();
    return '' unless @$periods;

    my $current_start = shift @$periods;
    my $current_stop = $current_start;
    while (my $p = shift @$periods) {
        if (Delta_Days(split(/-/, $current_stop), split(/-/, $p)) > 1) {
            push @intervals, {start => $current_start,
                              stop => $current_stop};
            $current_start = $p;
            $current_stop = $current_start;
        } else {
            $current_stop = $p;
        }
    }
    push @intervals, {start => $current_start,
                      stop => $current_stop};
    foreach my $ivl (@intervals) {
        foreach my $k (keys %$ivl) {
            $ivl->{$k} = join('.', reverse split /-/, $ivl->{$k});
        }
        delete $ivl->{stop} if $ivl->{stop} eq $ivl->{start};
    }
    return \@intervals;
}

1;
