package Stat::SearchQuery::Queue;

# $Id$

=pod
=encoding utf-8
=head1 NAME
    Stat::SearchQuery::Queue 
    Модуль для работы с очередью для отчётов по поисковым фразам

=head1 DESCRIPTION

=cut

use Direct::Modern;

use JSON qw/to_json from_json/;
use Readonly;

use Yandex::DateTime;
use Yandex::DBTools;
use Yandex::TimeCommon;

use Settings;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    add_to_queue
    get_from_queue
    remove_from_queue
    update_queue
    update_queue_timeprocess
    clear_queue
    get_new_reports_options
    get_old_tasks_ids
    get_tasks_for_check_status
    calculate_error_status
/;

Readonly::Hash our %REPORT_TYPES => map { $_ => undef } qw/
    dynamic
    performance
/;

my $SECONDS_IN_A_DAY  = 86_400;
# время жизни отчёта с момента запроса его создания
our %REPORT_TTL = (
    dynamic => $SECONDS_IN_A_DAY * 91,
    performance => $SECONDS_IN_A_DAY * 91,
);
# время жизни отчёта с ошибкой после его последней обработки
our $ERRONEOUS_REPORT_HOLD_TIME = $SECONDS_IN_A_DAY * 7;

our %REPORT_STATUS_TO_CHECK = (
    dynamic => 'Process',
    performance => 'Process',
);

# отчеты с rank < $MAX_RANK еще могут быть обработаны заново
our $MAX_RANK = 5;

# время жизни записи об отчёте с момента запроса его создания до перекладывания в ppclog
our $RECORD_TTL ||= $SECONDS_IN_A_DAY * 7;

our $HOLD_TIME  ||= 4 * 60; # время, в течении которого статус отчёта не перепроверяется

=head2 load(string)

    Десериализация

=cut

sub load($) {
    my $string = shift;
    return from_json($string);
}

=head2 serialize(object)

    Сериализация

=cut

sub serialize($) {
    my $object = shift;
    return to_json($object);
}

=head2 add_to_queue(uid, type, request_options)

    Добавление запроса на отчёт в очередь

=cut

sub add_to_queue {
    my ($uid, $type, $options) = @_;
    my $id = eval { do_insert_into_table(PPCDICT, 'api_queue_search_query', {
                uid => $uid,
                type => $type,
                options => serialize($options),
            }) };
    if ($@) {
        warn $@;
        return 0;
    }
    return $id;
}

=head2 get_from_queue(uid, id)

    Получает информацию об одном неудалённом отчёте из очереди
    Принимает: uid, id
    Возвращает: хеш вида {  bs_id => ...,
                            status => ...,
                            timecreate => ...,
                            type => ...,
                            rank => ...  }
=cut

sub get_from_queue($$) {
    my ($uid, $id) = @_;
    my $task = get_one_line_sql(PPCDICT, [
        "select uid, id, bs_id, status, options, timecreate, rank, type from api_queue_search_query",
        where => {
            status => [qw/New Process Done/],
            id => $id,
            uid => $uid,
        }]);
    if (defined $task) {
        $task->{options} = load($task->{options});
        return $task;
    }
    return {};
}

###############################################################

=head2 remove_from_queue(id)

    Удаление записи об отчете из очереди

=cut

sub remove_from_queue($) {
    my ($id) = @_;
    my $result = do_delete_from_table(PPCDICT, 'api_queue_search_query', where => {id => $id});
    return (defined $result)? 1 : 0;
}

=head2 update_queue(id; fields_hash)

Обновление записи в очереди

=cut

sub update_queue($;%) {
    my ($id, %fields) = @_;
    $fields{timeprocess__dont_quote} = 'now()';
    $fields{options} = serialize($fields{options}) if ref $fields{options};

    do_update_table(PPCDICT, 'api_queue_search_query', \%fields, where => {id => $id});
}

=head2 update_queue_timeprocess(id)

    Обновление времени последней операции с отчётом по id

=cut

sub update_queue_timeprocess($) {
    my ($id) = @_;
    update_queue($id);
}

=head2 clear_queue(%params)

    Удаляет из очереди данные о старых отчетах, пишет в указанный лог

    Параметры именованные:
        type    - тип отчетов для выборки (обязательно)
        log     - объект $Yandex::Log для логирования (обязательно)
        id      - ограничить выборку указанным id-шником (опционально)
        task_cb - ссылка на метод, если передан то вызывается на каждое найденное задание

=cut

sub clear_queue {
    my %params = @_;

    die 'invalid type' unless $params{type} && exists $REPORT_TYPES{ $params{type} };
    die 'no log' unless $params{log};

    my $log = $params{log};
    my $time = time();
    my $where = {
        ($params{id} ? (id__int => $params{id}) : ()),
        type => $params{type},
        status => 'Deleted',
        timecreate__lt => unix2mysql($time - $RECORD_TTL),
    };
    my $old_tasks = get_hashes_hash_sql(PPCDICT, [ "select id, type, uid, options, rank, timecreate, timeready from api_queue_search_query", where => $where ]);
    return unless %$old_tasks;

    $log->out(sprintf("Deleting %d old record(s).", scalar keys %$old_tasks));
    foreach my $id (keys %$old_tasks) {
        my $task = $old_tasks->{$id};
        if(_write_in_log($task, $log)) {
            $task->{options} = load($task->{options});
            eval {
                $params{task_cb}->($task) if $params{task_cb};
                remove_from_queue($id);
                1;
            }
            or $log->out("Cleanup failed for id=$id: $@");
        } else {
            update_queue_timeprocess($id);
            $log->out("Log DB write failed for record $id.");
        }
    }
}

=head2 get_new_reports_options(%params)

    Получение информации о новых отчетах из очереди
    Параметры именованные:
        type    - тип отчетов для выборки (обязательно)
        id      - ограничить выборку указанным id-шником (опционально)
    Возвращает: массив хешей вида [ {   id => ...,
                                        type => ...,
                                        options => {...},
                                    } ]
    
=cut

sub get_new_reports_options(%) {
    my %params = @_;

    die 'invalid type' unless $params{type} && exists $REPORT_TYPES{ $params{type} };

    my $tasks = get_all_sql(PPCDICT, ["select id, type, options from api_queue_search_query",
                                where => {
                                    ($params{id} ? (id__int => $params{id}) : ()),
                                    type => $params{type},
                                    status => 'New',
                                    rank__lt => $MAX_RANK
                                },
                            ]);
    for my $task (@$tasks) {
        $task->{options} = load($task->{options});
    }
    return $tasks;
}

=head2 get_old_tasks_ids(%params)

    Получает id'шники БК для отчетов, которые
    либо удалены пользователем
    либо хранятся дольше чем максимальное время в течении которого могут храниться отчеты

    Параметры именованные:
        type    - тип отчетов для выборки (обязательно)
        id      - ограничить выборку указанным id-шником (опционально)
    Возвращает: массив вида [{ id => ...,
                              bs_id => ..., }, ...]

=cut

sub get_old_tasks_ids(%) {
    my %params = @_;

    die 'invalid type' unless $params{type} && exists $REPORT_TYPES{ $params{type} };

    my $time = time();
    my $tasks = get_all_sql(PPCDICT, [
            "SELECT id, type, bs_id FROM api_queue_search_query",
            WHERE => {
                ($params{id} ? (id__int => $params{id}) : ()),
                type => $params{type},
                _OR => [
                    status => 'Abandoned',
                    _AND => [
                        status => ['New', 'Process', 'Done'],
                        timecreate__lt => unix2mysql($time - $REPORT_TTL{ $params{type} }),
                    ],
                    _AND => [
                        status => 'Error',
                        timeprocess__lt => unix2mysql($time - $ERRONEOUS_REPORT_HOLD_TIME),
                    ],
                ],
            },
        ]);

    return $tasks;
}

=head2 get_tasks_for_check_status(%params)

    Получает информацию об отчетах, по которым пора запрашивать проверку статуса

    Параметры именованные:
        type    - тип отчетов для выборки (обязательно)
        id      - ограничить выборку указанным id-шником (опционально)
    Возвращает: массив хешей вида
    [ { id => ...,
        bs_id => ...,
        status => ... }]
=cut

sub get_tasks_for_check_status(%) {
    my %params = @_;

    die 'invalid type' unless $params{type} && exists $REPORT_TYPES{ $params{type} };

    my $time = time();
    my $tasks = get_all_sql(PPCDICT, 
        [
            "select id, uid, bs_id, rank, type, status from api_queue_search_query",
            where => {
                ($params{id} ? (id__int => $params{id}) : ()),
                type => $params{type},
                timeprocess__lt => unix2mysql($time - $HOLD_TIME),
                status => $REPORT_STATUS_TO_CHECK{ $params{type} },
                rank__lt => $MAX_RANK,
            },
        ]
    );
    return $tasks;
}

=head2 calculate_error_status(%params)

Переводит отчёты с rank больше 5 из статусов New и Process в статус Error

Параметры именованные:
    type - тип отчетов (обязательно)
    id   - ограничить проверку указанным id (опционально)
Возвращает количество затронутых записей

=cut

sub calculate_error_status(%) {
    my %params = @_;

    die 'invalid type' unless $params{type} && exists $REPORT_TYPES{ $params{type} };

    my $time = time();
    my $count = do_update_table(PPCDICT, 'api_queue_search_query',
        {
            status => 'Error',
        },
        where => {
            ($params{id} ? (id__int => $params{id}) : ()),
            type => $params{type},
            status => ['New', 'Process'],
            rank__ge => $MAX_RANK,
        },
    );

    return $count + 0;
}

=head2 _write_in_log($record, $log)

    Поместить запись об отчёте в журнал

=cut

sub _write_in_log($$) {
    my ($rec, $log) = @_;

    # для других (не search) отчетов - ничего не логируем.
    return 1;
}

1;
