package Stat::SearchQuery::Report;

# $Id$

=pod
=encoding utf-8
=head1 NAME

Stat::SearchQuery::Report - Модуль для работы с отчётами по поисковым фразам

=head1 DESCRIPTION

=cut

use warnings;
use strict;
use utf8;

use BS::SearchQuery;

use Settings;
use MailService qw/send_die_letters/;
use Stat::SearchQuery::Queue qw/
    update_queue
    update_queue_timeprocess
/;

use Mouse;

our $STREAM_TIMEOUT ||= 20; # таймаут для потоковых запросов в БК между передачами
our $RELOCK_PERIOD  ||= 15; # период переустановки семафора, должен быть меньше $STREAM_TIMEOUT


has log_object => ( is => 'rw', isa => 'Maybe[Object]', default => undef);
has bs => (is => 'rw', lazy_build => 1);


sub _build_bs {
    my ($self) = @_;
    return BS::SearchQuery->new(log_object => $self->log_object);
}


=head2 _log($string)

    Пишет в лог

=cut

sub _log {
    my ($self, $string) = @_;
    if (defined $self->log_object) {
        $self->log_object->out($string);
    }
}

=head2 create($bs_params)

    Запрос новых отчётов из БК

=cut

sub create {
    my ($self, $bs_params) = @_;
    $self->_log(sprintf("Requesting %d new report(s): %s.", scalar @$bs_params, join(',', map { $_->{id} } @$bs_params)));

    for my $params (@$bs_params) {
        my ($code, $bs_id) = $self->bs->create($params->{options});

        if (!defined $code) {
            update_queue($params->{id}, rank__dont_quote => "rank + 1");
        } else {
            if ($code eq '0') {
                update_queue($params->{id}, status => 'Process', bs_id => $bs_id, rank => 0);
            } else {
                update_queue($params->{id}, rank__dont_quote => "rank + 1");
                $self->_log("create() got response with no success: $code");
            }
        }
    }
}

=head2 delete($ids_list)

Удаление старых отчётов и отчётов, от которых отказался пользователь

=cut

sub delete {
    my ($self, $ids_list) = @_;
    my @ids = map { $_->{id} } @$ids_list;
    $self->_log(sprintf("Deleting %d record(s): %s.", scalar @ids, join(',', @ids)));

    for my $ids (@$ids_list) {
        my ($code, $status) = $self->bs->delete($ids->{bs_id});
        if (!defined $code) {
            update_queue_timeprocess($ids->{id});
        } else {
            update_queue($ids->{id}, status => 'Deleted')
                if $code eq '-2';
        }
    }
}

=head2 update(;log)

    Обновление статусов неудалённых отчётов
    Варианты ответов БК : "-2 Complete", "-1 Not ready", "-3 Bad act"

    Опции:
        on_ready - колбек для обработки готового отчёта

=cut



sub update {
    my ($self, $tasks, %opt) = @_;

    $self->_log(sprintf("Updating %d record(s): %s.", scalar @$tasks, join(',', map { $_->{id} } @$tasks)));

    for my $task (@$tasks) {
        my ($code, $status) = $self->bs->check($task->{bs_id});

        if (!defined $code) {
            update_queue_timeprocess($task->{id});
        } else {
            if ($code eq '-1') { # Not ready. Также возвращается для несуществующих отчётов
                if ($task->{status} eq 'Done') { # наверное, можно не ругаться, если отчёту больше нескольких дней
                    send_die_letters('Search Query report not found in BS. '.sprintf("Report %d", $task->{id}));
                    $self->_log('Search Query report not found in BS. '.sprintf("Report %d", $task->{id}));
                    update_queue($task->{id}, status => 'Deleted');
                    next;
                } else {
                    update_queue_timeprocess($task->{id});
                }
            } elsif ($code eq '-2') { # Complete
                if ($task->{status} ne 'Done') {
                    if ($opt{on_ready}) {
                        update_queue($task->{id}, rank__dont_quote => "rank + 1");
                        $opt{on_ready}->($task);
                        update_queue($task->{id}, rank__dont_quote => "rank - 1", status => 'Done', timeready__dont_quote => 'now()');
                    } else {
                        update_queue($task->{id}, status => 'Done', timeready__dont_quote => 'now()');
                    }
                } else {
                    update_queue_timeprocess($task->{id});
                }
            } elsif ($code eq '-3') {
                $self->_log("BS error in $task->{id}: '$code' '$status'");
                update_queue($task->{id}, rank__dont_quote => "rank + 1");
            } else {
                $self->_log("update() for $task->{id} got unexpected response: '$code' '$status'");
                update_queue($task->{id}, rank__dont_quote => "rank + 1");
            }
        }
    }
}

1;
