package Stat::StreamExtended;

=pod

    $Id$

=head1 NAME

    Stat::StreamExtended

=head1 DESCRIPTION

    Получение и обработка стримовой статистики из БК (с поддержкой фильтрации/группровки/сортировки/педжинга на стороне БК)

    Доступность статистики: начиная с 20140220 данные доступны в ClickHouse

=cut

use strict;
use warnings;
use feature qw/state/;

use JSON;
use List::Util qw/maxstr minstr min max/;
use List::MoreUtils qw/uniq all any none/;
use Data::Dumper;
use Digest::MD5 qw/md5/;

use Yandex::Trace;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xminus xisect xflatten/;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Log::Messages;
use Yandex::DateTime;
use Yandex::TimeCommon qw/today normalize_date/;
use Yandex::Validate qw/is_valid_float is_valid_int/;
use Yandex::I18n;
use Yandex::Clone qw/yclone/;
use Yandex::HTTP qw/http_parallel_request/;

use BS::URL;
use Models::AdGroup;
use PrimitivesIds;
use Settings;
use Stat::Const qw/:base :enums/;
use Stat::Fields;
use Stat::Tools qw/apply_suffixes periods_full_suffix_list periods_suffix_list periods_delta_suffix_list/;
use Stat::AddedPhrasesCache qw/get_phrases_from_cache_by_key/;
use Tag;
use Stat::StreamExtended::ParseIterator;
use HashingTools;
use Direct::PhraseTools;
use Stat::StreamExtended::HttpBodyIterator;
use Stat::StreamExtended::FakeHttpBodyIterator;
use Campaign::Types qw/camp_kind_in get_camp_type/;
use CampaignTools;
use MetrikaCounters ();
use Client::ClientFeatures ();
use Rbac qw/:const/;

use utf8;

our $DEBUG = 0;
my $FORCE_PREPROD = 0;
# Использовать простой, а не асинхронный обработчик запроса к БК
our $USE_SIMPLE_HTTP_REQUEST = 0;

my %countable_direct2bs = (
    Stat::Fields::get_countable_direct2bs(),
);

my %fields_opt = (  date        => {order => 'UpdateTime'},

                    campaign    => {group => 'cid',
                                    group_mappings => 'cid_by_OrderID',
                                    order => 'cid',
                                    order_mappings => 'cid_by_OrderID'},

                    campaign_type => {
                                    group => 'CampType',
                                    group_mappings => 'CampType_by_OrderID',
                                    order => 'CampType' },

                    camp_name   => {group_dependency => 'campaign',
                                    order => 'CampSortName',
                                    order_mappings => 'CampSortName_by_OrderID'
                                    },

                    strategy_id => {group => 'AutobudgetStrategyID',
                                    filter => 'AutobudgetStrategyID',
                                    filter_op => [qw/eq ne/],
                                    order => 'AutobudgetStrategyID'},

                    banner      => {group => 'bid',
                                    filter => 'bid',
                                    filter_op => [qw/eq ne/],
                                    order => 'bid'},

                    creative_id => {group => 'CreativeID',
                                    filter => 'CreativeID',
                                    filter_op => [qw/eq ne/],
                                    order => 'CreativeID'},

                    banner_type => {group => 'BannerType',
                                    common_dict_name => 'BANNER_TYPES',
                                    common_dict_default => undef,
                                    filter => 'BannerType',
                                    filter_op => [qw/eq/],
                                    order => 'BannerType'},

                    adgroup     => {group => 'GroupExportID',
                                    filter => 'GroupExportID',
                                    filter_op => [qw/eq ne/],
                                    order => 'GroupExportID'},

                    adgroup_id  => {group_dependency => 'adgroup',
                                    filter => 'GroupExportID',
                                    filter_op => [qw/eq ne/],
                                    order => 'GroupExportID'},

                    adgroup_name => {group_dependency => 'adgroup',
                                     order => 'GroupExportIDSort',
                                     order_mappings => 'GroupExportIDSort_by_GroupExportID'},

                    contextcond => {group => [qw/PhraseID mContextType/],
                                    group_mappings => [qw/mContextType_by_ContextTypeHideSynonym/],
                                    order => [qw/mContextType Phrase PhraseID/]},

                    contextcond_mod => {group => [qw/PhraseID CriterionID mContextType/],
                                    group_mappings => [qw/mContextType_by_ContextTypeHideSynonym/],
                                    order => [qw/mContextType Phrase CriterionID/]},

                    contextcond_ext => {
                                    group => [qw/PhraseWBM PhraseWBMID mContextType/],
                                    group_mappings => [qw/mContextType_by_ContextTypeHideSynonym/],
                                    order => [qw/mContextType PhraseWBM PhraseWBMID/]},

                    contextcond_ext_mod => {
                                    group => [qw/PhraseWBM PhraseWBMID CriterionID mContextType/],
                                    group_mappings => [qw/mContextType_by_ContextTypeHideSynonym/],
                                    order => [qw/mContextType PhraseWBM PhraseWBMID CriterionID/]},

                    contextcond_orig => {
                                    group => [qw/OriginalPhraseID ContextCondType OriginalPhrase/],
                                    group_mappings => [qw/ContextCondType_by_ContextTypeHideSynonym/],
                                    order => [qw/ContextCondType OriginalPhrase OriginalPhraseID/]},

                    contextcond_orig_mod => {
                                    group => [qw/OriginalPhraseID CriterionID ContextCondType OriginalPhrase/],
                                    group_mappings => [qw/ContextCondType_by_ContextTypeHideSynonym/],
                                    order => [qw/ContextCondType OriginalPhrase OriginalPhraseID CriterionID/]},

                    criterion_cond => {
                                    group => [qw/OriginalPhraseID mCriterionType ContextCondType OriginalPhrase/],
                                    group_mappings => [qw/mCriterionType_by_CriterionType ContextCondType_by_ContextTypeHideSynonym/],
                                    order => [qw/mCriterionType ContextCondType OriginalPhrase OriginalPhraseID/]},

                    attribution_model => {
                                    filter => 'AttributionType',
                                    filter_op => ['eq'],
                                    filter_single_value => 1,
                                    common_dict_name => 'ATTRIBUTION_MODEL_TYPES',
                                    common_dict_default => undef},

                    attribution_models => {
                                    filter => 'MultiGoalsAttributionType',
                                    filter_op => ['eq'],
                                    common_dict_name => 'ATTRIBUTION_MODEL_TYPES',
                                    common_dict_default => undef},


                    phrase      => {filter => 'Phrase',
                                    filter_op => [qw/eq ne starts_with not_starts_with
                                                     contains not_contains contains_in_plus_part not_contains_in_plus_part/]},

                    phrase_ext  => {filter => 'PhraseWBM',
                                    filter_op => [qw/eq ne starts_with not_starts_with
                                                     contains not_contains contains_in_plus_part not_contains_in_plus_part/]},

                    phrase_orig => {filter => 'OriginalPhrase',
                                    filter_op => [qw/eq ne starts_with not_starts_with
                                                     contains not_contains contains_in_plus_part not_contains_in_plus_part/]},

                    phraseid    => {filter => 'PhraseID',
                                    filter_op => [qw/eq ne/],
                                    group_dependency => 'contextcond',
                                    order => 'PhraseID'},

                    phraseid_orig => {
                                    filter => 'OriginalPhraseID',
                                    filter_op => [qw/eq ne/],
                                    group_dependency => 'contextcond_orig',
                                    order => 'OriginalPhraseID'},

                    criterion_id => {
                                    filter => 'OriginalPhraseID',
                                    filter_op => [qw/eq ne/],
                                    group_dependency => 'criterion_cond',
                                    order => 'OriginalPhraseID'},
                    bs_criterion_id => {
                                    group => 'CriterionID',
                                    filter => 'CriterionID',
                                    filter_op => [qw/eq ne/],
                                    order => 'CriterionID'},

                    retargeting => {filter => 'GoalContextID',
                                    filter_op => [qw/eq ne/]},

                    dynamic     => {filter => 'DynamicConditionID',
                                    filter_op => [qw/eq ne/]},

                    performance => {
                                    filter => 'PerfFilterID',
                                    filter_op => [qw/eq ne/] },

                    matched_phrase => {
                                    group => 'MatchedKeyword',
                                    filter => 'MatchedKeyword',
                                    filter_op => [qw/eq ne starts_with not_starts_with
                                                     contains not_contains contains_in_plus_part not_contains_in_plus_part/],
                                    order => 'MatchedKeyword'
                                   },

                    matched_phrase_id => {
                                    group => 'MatchedKeywordID',
                                    filter => 'MatchedKeywordID',
                                    filter_op => [qw/eq ne starts_with not_starts_with
                                                     contains not_contains contains_in_plus_part not_contains_in_plus_part/],
                                    order => 'MatchedKeywordID'
                                   },

                    match_type => {
                                    group => 'mMatchType',
                                    group_mappings => 'mMatchType_by_MatchType',
                                    common_dict_name => 'MATCH_TYPES',
                                    common_dict_default => 'none',
                                    filter => 'MatchType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mMatchType',
                                    order_mappings => 'mMatchType_by_MatchType',
                                   },

                    retargeting_coef => {
                                    group => 'CoefGoalContextID',
                                    filter => 'CoefGoalContextID',
                                    filter_op => [qw/eq ne/],
                                    order => 'CoefGoalContextIDSort',
                                    order_mappings => 'CoefGoalContextIDSort_by_CoefGoalContextID'},

                    coef_ret_cond_id => {
                                    group_dependency => 'retargeting_coef',
                                    order => 'CoefGoalContextID'},

                    contexttype => {group => 'mContextType',
                                    group_mappings => 'mContextType_by_ContextTypeSeparateSynonym',
                                    common_dict_name => 'CONTEXT_TYPES',
                                    common_dict_default => undef,
                                    filter => 'ContextTypeSeparateSynonym',
                                    filter_op => [qw/eq ne/],
                                    order => 'mContextType'},

                    contexttype_orig => {
                                    group => 'ContextCondType',
                                    group_mappings => 'ContextCondType_by_ContextTypeHideSynonym',
                                    common_dict_name => 'CONTEXT_COND_TYPES',
                                    common_dict_default => undef,
                                    filter => 'ContextTypeHideSynonym',
                                    filter_op => [qw/eq ne/],
                                    order => 'ContextCondType'},

                    criterion_type => {group => 'mCriterionType',
                                    group_mappings => 'mCriterionType_by_CriterionType',
                                    common_dict_name => 'CRITERION_TYPES',
                                    common_dict_default => undef,
                                    filter => 'CriterionType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mCriterionType'},

                    region      => {group => 'StatisticRegionID',
                                    filter => 'StatisticRegionID',
                                    filter_op => [qw/eq ne/],
                                    order => 'StatisticRegionName'},

                    physical_region => {
                                    group => 'RegionID',
                                    filter => 'RegionID',
                                    filter_op => [qw/eq ne/],
                                    order => 'RegionName'},

                    region_id   => {group_dependency => 'region',
                                    filter => 'StatisticRegionID',
                                    filter_op => [qw/eq ne/],
                                    order => 'StatisticRegionID'},

                    physical_region_id => {
                                    group_dependency => 'physical_region',
                                    filter => 'RegionID',
                                    filter_op => [qw/eq ne/],
                                    order => 'RegionID'},

                    page_group  => {group => [qw/PageGroupID mTargetType/],
                                    group_mappings => 'mTargetType_by_TargetType',
                                    order => 'PageSortName'},

                    page        => {group => [qw/PageID mTargetType/],
                                    group_mappings => 'mTargetType_by_TargetType',
                                    filter => 'PageID',
                                    filter_op => [qw/eq ne/],
                                    order => [qw/PageSortName PageID/]},

                    page_name   => {group => [qw/PageName mTargetType/],
                                    group_mappings => 'mTargetType_by_TargetType',
                                    filter => 'PageName',
                                    filter_op => [qw/eq ne contains not_contains/],
                                    order => [qw/PageName/]},

                    page_name_detailed   => {group => [qw/PageName PageID mTargetType/],
                                    group_mappings => 'mTargetType_by_TargetType',
                                    order => [qw/PageName PageID/]},

                    targettype  => {group => 'mTargetType',
                                    group_mappings => 'mTargetType_by_TargetType',
                                    common_dict_name => 'TARGET_TYPES',
                                    common_dict_default => 'search',
                                    filter => 'TargetType',
                                    filter_op => [qw/eq/],
                                    order => 'mTargetType'},

                    position    => {group => 'mTypeID',
                                    group_mappings => 'mTypeID_by_TypeID',
                                    common_dict_name => 'POSITION_TYPES',
                                    common_dict_default => 'non-prime',
                                    filter => 'TypeID',
                                    filter_op => [qw/eq ne/],
                                    order => 'mTypeID'},

                    tags        => {group => 'Tag',
                                    group_mappings => 'Tag_by_GroupExportID',
                                    filter => 'GroupExportID',
                                    filter_op => [qw/eq ne/],
                                    filter_prepare => \&_prepare_tags_filter,
                                    order => 'TagSort',
                                    order_mappings => 'TagSort_by_GroupExportID'},

                    tag_ids     => {filter => 'GroupExportID',
                                    filter_op => [qw/eq ne/],
                                    filter_prepare => \&_prepare_tag_ids_filter},

                    has_image   => {group => 'IsImage',
                                    filter => 'IsImage',
                                    filter_op => [qw/eq/],
                                    order => 'IsImage'},

                    banner_image_type  => {
                                    group => 'mBannerImageType',
                                    group_mappings => 'mBannerImageType_by_BannerImageType',
                                    common_dict_name => 'BANNER_IMAGE_TYPES',
                                    common_dict_default => undef,
                                    filter => 'BannerImageType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mBannerImageType'},

                    image_size  => {group => 'ImageSize',
                                    filter => 'ImageSize',
                                    filter_values => [uniq (@Direct::Model::Creative::AVAILABLE_SIZES,
                                                            map { join('x', @$_) } @Direct::Validation::Image::VALID_SIZES)],
                                    filter_op => [qw/eq ne/],
                                    order => 'ImageSize'},

                    device_type => {group => 'mDeviceType',
                                    group_mappings => 'mDeviceType_by_DeviceType',
                                    common_dict_name => 'DEVICE_TYPES',
                                    common_dict_default => 'desktop',
                                    filter => 'DeviceType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mDeviceType' },

                    click_place => {group => 'mClickPlace',
                                    group_mappings => 'mClickPlace_by_ClickPlace',
                                    common_dict_name => 'CLICK_PLACES',
                                    filter => 'ClickPlace',
                                    filter_op => [qw/eq ne/],
                                    order => 'mClickPlace' },
    
                    targeting_category => {group => 'mSemanticCorrespondenceType',
                                    group_mappings => 'mSemanticCorrespondenceType_by_SemanticCorrespondenceType',
                                    common_dict_name => 'TARGETING_CATEGORIES',
                                    filter => 'SemanticCorrespondenceType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mSemanticCorrespondenceType',
                                    order_mappings => 'mSemanticCorrespondenceType_by_SemanticCorrespondenceType'},
    
                    prisma_income_grade => {group => 'mPrismCluster',
                                    group_mappings => 'mPrismCluster_by_PrismCluster',
                                    common_dict_name => 'PRISMA_INCOME_GRADES',
                                    filter => 'PrismCluster',
                                    filter_op => [qw/eq ne/],
                                    order => 'mPrismCluster',
                                    order_mappings => 'mPrismCluster_by_PrismCluster'},

                    gender      => {group => 'mGender',
                                    group_mappings => 'mGender_by_Gender',
                                    common_dict_name => 'GENDERS',
                                    filter => 'Gender',
                                    filter_op => [qw/eq ne/],
                                    order => 'mGender' },

                    age         => {group => 'mAge',
                                    group_mappings => 'mAge_by_Age',
                                    common_dict_name => 'AGES',
                                    filter => 'Age',
                                    filter_op => [qw/eq ne/],
                                    order => 'mAge' },

                    detailed_device_type => {
                                    group => 'mDetailedDeviceType',
                                    group_mappings => 'mDetailedDeviceType_by_DetailedDeviceType',
                                    common_dict_name => 'DETAILED_DEVICE_TYPES',
                                    common_dict_default => 'other',
                                    filter => 'DetailedDeviceType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mDetailedDeviceType' },

                    connection_type => {
                                    group => 'mConnectionType',
                                    group_mappings => 'mConnectionType_by_ConnectionType',
                                    common_dict_name => 'CONNECTION_TYPES',
                                    filter => 'ConnectionType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mConnectionType' },

                    ssp         => {group => 'SSPTitle',
                                    filter => 'SSPTitle',
                                    filter_op => [qw/eq ne/],
                                    filter_prepare => \&_prepare_ssp_filter,
                                    order => 'SSPTitle'},

                    search_query => {
                                    group => 'SearchQuery',
                                    filter => 'SearchQuery',
                                    filter_op => [qw/eq ne starts_with not_starts_with/],
                                    order => 'SearchQuery'},

                    search_query_status => {group => 'mSearchQueryStatus',
                                            group_mappings => [qw/SearchQueryAndGroupExportID_by_SearchQueryAndGroupExportID
                                                                  SearchQueryAndOrderID_by_SearchQueryAndOrderID
                                                                  mSearchQueryStatus_by_SearchQueryStatus/],
                                            common_dict_name => 'BS_PHRASE_STATUSES',
                                            filter => 'SearchQueryStatus',
                                            filter_op => [qw/eq ne/],
                                            filter_mappings => [qw/SearchQueryAndGroupExportID_by_SearchQueryAndGroupExportID
                                                                   SearchQueryAndOrderID_by_SearchQueryAndOrderID/],
                                            order => 'mSearchQueryStatus'},

                    ext_phrase_status => {group => 'mPhraseWBMStatus',
                                          group_mappings => [qw/PhraseWBMAndGroupExportID_by_PhraseWBMAndGroupExportID
                                                                PhraseWBMAndOrderID_by_PhraseWBMAndOrderID
                                                                mPhraseWBMStatus_by_PhraseWBMStatus/],
                                          common_dict_name => 'BS_PHRASE_STATUSES',
                                          filter => 'PhraseWBMStatus',
                                          filter_op => [qw/eq ne/],
                                          filter_mappings => [qw/PhraseWBMAndGroupExportID_by_PhraseWBMAndGroupExportID
                                                                 PhraseWBMAndOrderID_by_PhraseWBMAndOrderID/],
                                          order => 'mPhraseWBMStatus'},

                    bm_type     => {group => 'mBroadmatchType',
                                    group_mappings => 'mBroadmatchType_by_BroadmatchType',
                                    common_dict_name => 'BROADMATCH_TYPES',
                                    filter => 'BroadmatchType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mBroadmatchType',
                                    order_mappings => 'mBroadmatchType_by_BroadmatchType' },

                    sim_distance => {
                                    group => 'SimDistance',
                                    filter => 'SimDistance',
                                    filter_op => [qw/eq ne/],
                                    order => 'SimDistance' },

                    goal_ids     => {filter => 'MultiGoalsID',
                                    filter_op => [qw/eq/],
                                    filter_prepare => \&_prepare_goal_id_filter,
                                    filter_mappings => ['MeaningfulGoalValue_by_OrderIDAndGoalID']},

                    goal_id     => {filter => 'MultiGoalsID',
                                    filter_op => [qw/eq/],
                                    filter_prepare => \&_prepare_goal_id_filter,
                                    filter_mappings => ['MeaningfulGoalValue_by_OrderIDAndGoalID']},

                    goal_id_list => {filter => 'GoalIDList',
                                    filter_op => [qw/eq/],
                                    filter_prepare => \&_prepare_goal_id_filter,
                    },

                    single_goal_id     => {
                                    filter => 'GoalID',
                                    filter_op => ['eq'],
                                    filter_single_value => 1,
                                    filter_prepare => \&_prepare_goal_id_filter},

                    deal        => {
                                      group => 'DealExportID',
                                      filter => 'DealExportID',
                                      filter_op => [qw/eq ne/],
                                      order => 'DealExportID' },
                    deal_name   => {group_dependency => 'deal',
                                    order => 'mDealExportID',
                                    order_mappings => 'mDealExportID_by_DealExportID'},
                    deal_export_id => {group_dependency => 'deal',
                                       filter => 'DealExportID',
                                       filter_op => [qw/eq ne/]},
                    ab_segment => {group_dependency => 'abSegments',
                                       group => 'ExpSegmentName',
                                       filter => 'ExpSegmentID',
                                       order => 'ExpSegmentName',
                                       filter_op => [qw/eq/]},
                    inventory_type => {group => 'InventoryType',
                                        filter => 'InventoryType',
                                        filter_op => [qw/eq ne/],
                                        common_dict_name => 'INVENTORY_TYPES',
                                        order => 'InventoryType',
                                        },
                    browser             => { group => 'Browser',
                                        filter                            => 'Browser',
                                        filter_op                         => [ qw/eq ne/ ],
                                        order                             => 'Browser' },

                    operating_system    => { group => 'OperatingSystem',
                                        filter                            => 'OperatingSystem',
                                        filter_op                         => [ qw/eq ne/ ],
                                        order                             => 'OperatingSystem' },

                    template_id         => { group => 'TemplateID',
                                        filter                            => 'TemplateID',
                                        filter_op                         => [ qw/eq ne/ ],
                                        order                             => 'TemplateID' },
                    client_id           => { group => 'ClientID',
                                        filter                            => 'ClientID',
                                        filter_op                         => [ qw/eq ne/ ],
                                        order                             => 'ClientID' },
                    client_login   => { group => 'mClientID',
                                        group_mappings => 'mClientID_by_ClientID_by_login',
                                        order                         => 'mClientID',
                                        order_mappings                => 'mClientID_by_ClientID_by_login' },

                    internal_product_name   => { group_dependency => 'client_id',
                                            order                         => 'mClientID',
                                            order_mappings                => 'mClientID_by_ClientID' },
                    place_id            => { group => 'PlaceID',
                                        filter                            => 'PlaceID',
                                        filter_op                         => [ qw/eq ne/ ],
                                        order                             => 'PlaceID' },
                    place_description   => { group => 'PlaceDescription',
                                        filter                            => 'PlaceDescription',
                                        filter_op                         => [ qw/eq ne contains not_contains/ ],
                                        order                             => 'PlaceDescription' },
                    page_id            => { group => 'PageID',
                                        filter                            => 'PageID',
                                        filter_op                         => [ qw/eq ne/ ],
                                        order                             => 'PageID' },

                    turbo_page_type     => {group => 'mTurboPageType',
                                    group_mappings => 'mTurboPageType_by_TurboPageType',
                                    common_dict_name => 'BS_TURBO_PAGE_TYPES',
                                    filter => 'TurboPageType',
                                    filter_op => [qw/eq ne/],
                                    order => 'mTurboPageType' },
                    content_targeting   => { group => 'ContentTargeting',
                                             order => 'ContentTargeting'},

                    region_source =>  { group                             => 'RegionSource',
                                        common_dict_name                  => 'REGION_SOURCES',
                                        common_dict_default               => 'unknown',
                                        filter                            => 'RegionSource',
                                        filter_op                         => [ qw/eq ne/ ],
                                        order                             => 'RegionSource' },

                    # измеримые поля, при изменениях нужно актуализировать Stat::Fields::countable_fields_dependencies
                    shows       => _countable_field_props('Shows'),
                    eshows      => _countable_field_props('eShows'),
                    clicks      => _countable_field_props('Clicks'),
                    ctr         => _countable_field_props('Ctr'),
                    ectr        => _countable_field_props('eCtr'),
                    avg_x       => _countable_field_props('AvgTrafficVolume'),
                    sum         => _countable_field_props('Cost'),
                    av_sum      => _countable_field_props('AvgClicksCost'),
                    fp_shows_avg_pos => _countable_field_props('AvgShowsPosition'),
                    fp_clicks_avg_pos => _countable_field_props('AvgClicksPosition'),
                    fp_shows    => _countable_field_props('FirstPageShows'),
                    fp_shows_pos => _countable_field_props('FirstPageSumPosShows'),
                    fp_clicks   => _countable_field_props('FirstPageClicks'),
                    fp_clicks_pos => _countable_field_props('FirstPageSumPosClicks'),
                    winrate     => _countable_field_props('PerformanceCoverage'),
                    bounces     => _countable_field_props('Bounces'),
                    bounce_ratio => _countable_field_props('BounceRatio'),
                    adepth      => _countable_field_props('AvgSessionDepth'),
                    aconv       => _countable_field_props('ConversionPct'),
                    agoalnum    => _countable_field_props('GoalsNum'),
                    agoalcost   => _countable_field_props('AvgGoalsCost'),
                    agoalroi    => _countable_field_props('GoalsROI'),
                    agoalcrr    => _countable_field_props('GoalsCrr'),
                    agoalincome => _countable_field_props('GoalsIncome'),
                    aseslen     => _countable_field_props('SessionDepth'),
                    asesnumlim  => _countable_field_props('SessionNumLimited'),
                    asesnum     => _countable_field_props('SessionNum'),
                    aprgoodmultigoal => _countable_field_props('PrGoodMultiGoal'),
                    avg_view_freq => _countable_field_props('RF'),
                    uniq_viewers => _countable_field_props('Reach'),
                    uniq_completed_viewers => _countable_field_props('CompletedReach'),
                    avg_cpm => _countable_field_props('CPM'),
                    aprgoodmultigoal_cpa => _countable_field_props('PrGoodMultiGoalCPA'),
                    aprgoodmultigoal_conv_rate => _countable_field_props('PrGoodMultiGoalConversionRate'),
                    avg_time_to_conv     => _countable_field_props('AvgTimeToConversion'),
                    avg_bid => _countable_field_props('AvgBid'),
                    avg_cpm_bid => _countable_field_props('AvgCPMBid'),
                    agoals_profit => _countable_field_props('GoalsProfit'),
                    auction_hits => _countable_field_props('CandidatesStatAuctionHits'),
                    auction_wins => _countable_field_props('CandidatesStatAuctionWins'),
                    auction_win_rate => _countable_field_props('CandidatesStatWinRate'),
                    imp_to_win_rate => _countable_field_props('CandidatesStatShowRate'),
                    imp_reach_rate => _countable_field_props('CandidatesStatReach'),
                    served_impressions => _countable_field_props('ServedImpressions'),
                    pv_adepth      => _countable_field_props('AvgSessionDepthPV'),
                    pv_bounce_ratio => _countable_field_props('BounceRatioPV'),
                    pv_agoalnum    => _countable_field_props('GoalsNumPV'),
                    pv_aconv       => _countable_field_props('ConversionPctPV'),
                    pv_agoalcost   => _countable_field_props('AvgGoalsCostPV'),
                    pv_agoalroi    => _countable_field_props('GoalsROIPV'),
                    pv_agoalcrr    => _countable_field_props('GoalsCrrPV'),
                    pv_agoalincome => _countable_field_props('GoalsIncomePV'),
                    pv_agoals_profit => _countable_field_props('GoalsProfitPV'),
                    pv_aseslen     => _countable_field_props('SessionDepthPV'),
                    pv_asesnumlim  => _countable_field_props('SessionNumLimitedPV'),
                    pv_asesnum     => _countable_field_props('SessionNumPV'),
                    pv_aprgoodmultigoal => _countable_field_props('PrGoodMultiGoalPV'),
                    pv_aprgoodmultigoal_cpa => _countable_field_props('PrGoodMultiGoalCPAPV'),
                    pv_aprgoodmultigoal_conv_rate => _countable_field_props('PrGoodMultiGoalConversionRatePV'),
                    pv_avg_time_to_conv => _countable_field_props('AvgTimeToConversionPV'),

                    (map { $_ => _countable_field_props($countable_direct2bs{$_}) } keys %countable_direct2bs),
                );

sub _countable_field_props {
    return {filter => $_[0],
            filter_op => [qw/eq ne gt lt/],
            filter_type => 'post',
            order => $_[0],
            is_countable => 1,
            field => $_[0]};
}

# статистика по % полученных показов собирается только для опредеденного набора срезов, потому при запросе информации по winrate
# нужно вводить ограничения на допустимые срезы/фильтры
my @winrate_supported_key_fields = qw/campaign_type campaign strategy_id adgroup contextcond phrase phraseid retargeting dynamic performance adgroup_id contexttype
                                      region physical_region region_id physical_region_id page_group page page_name targettype position has_image device_type click_place ssp
                                      contextcond_orig contexttype_orig match_type attribution_model criterion_type deal deal_export_id
                                      attribution_models goal_id goal_ids single_goal_id phrase_orig turbo_page_type targeting_category/;
$fields_opt{$_}->{support_winrate} = 1 for grep { exists $fields_opt{$_} } @winrate_supported_key_fields;

# изменения состава значений нужно вносить в @multigoal_fields в Stat::Fields
our %multi_goals_fields_bs2direct = (
        MultiGoalsNum     => 'agoalnum',
        MultiGoalsAvgGoalsCost => 'agoalcost',
        MultiGoalsROI     => 'agoalroi',
        MultiGoalsCrr     => 'agoalcrr',
        MultiGoalsIncome  => 'agoalincome',
        MultiGoalsConversionPct => 'aconv',

        MultiGoalsNumPV     => 'pv_agoalnum',
        MultiGoalsAvgGoalsCostPV => 'pv_agoalcost',
        MultiGoalsROIPV     => 'pv_agoalroi',
        MultiGoalsCrrPV     => 'pv_agoalcrr',
        MultiGoalsIncomePV  => 'pv_agoalincome',
        MultiGoalsConversionPctPV => 'pv_aconv',
    );

my %goals_fields = map { $_ => 1 } qw/agoalnum agoalcost agoalroi agoalcrr agoalincome aconv avg_time_to_conv
                                      pv_agoalnum pv_agoalcost pv_agoalroi pv_agoalcrr pv_agoalincome pv_aconv pv_avg_time_to_conv/;
my %bs_goals_fields = map { $_ => 1, 'Multi'.$_ => 1} 
                                    qw/GoalsNum GoalsAvgGoalsCost GoalsROI GoalsCrr GoalsIncome GoalsConversionPct GoalsAvgTimeToConversion
                                       GoalsNumPV GoalsAvgGoalsCostPV GoalsROIPV GoalsIncomePV GoalsConversionPctPV GoalsAvgTimeToConversionPV/;

# соответствие названий полей в ответе БК, и названий принятых в Директе
# из полей, сгенерированных в результате маппинга (/^m[A-Z].*/) предварительно удаляется префикс "m"
my %fields_bs2direct = (
                    AvgGoalsCost => 'agoalcost',
                    AutobudgetStrategyID => 'strategy_id',
                    UpdateTime        => 'date',
                    CreativeID        => 'creative_id',
                    GroupExportID     => 'adgroup_id',
                    IsImage           => 'has_image',
                    ImageSize         => 'image_size',
                    BannerImageType   => 'banner_image_type',
                    StatisticRegionID => 'region',
                    RegionID          => 'physical_region',
                    Phrase            => 'phrase',
                    PhraseWBMID       => 'PhraseID',
                    OriginalPhraseID  => 'PhraseID',
                    CriterionID       => 'bs_criterion_id',
                    MatchedKeyword    => 'matched_phrase',
                    MatchedKeywordID  => 'matched_phrase_id',
                    MatchType         => 'match_type',
                    Tag               => 'tags',
                    DeviceType        => 'device_type',
                    ClickPlace        => 'click_place',
                    SemanticCorrespondenceType => 'targeting_category',
                    PrismCluster => 'prisma_income_grade',
                    
                    CampType          => 'campaign_type',
                    BannerType        => 'banner_type',
                    Age               => 'age',
                    Gender            => 'gender',
                    ConnectionType    => 'connection_type',
                    DetailedDeviceType => 'detailed_device_type',
                    SSPTitle          => 'ssp',
                    SearchQuery       => 'search_query',
                    BroadmatchType    => 'bm_type',
                    CoefGoalContextID => 'coef_ret_cond_id',
                    TypeID            => 'position',
                    SimDistance       => 'sim_distance',
                    SearchQueryStatus => 'search_query_status',
                    PhraseWBMStatus   => 'ext_phrase_status',
                    Shows             => 'shows',
                    eShows            => 'eshows',
                    AvgTrafficVolume  => 'avg_x',
                    eCtr              => 'ectr',
                    Clicks            => 'clicks',
                    Cost              => 'sum',
                    BonusNoVAT        => 'bonus',
                    PerformanceCoverage => 'winrate',
                    GoalsNum          => 'agoalnum',
                    GoalsIncome       => 'agoalincome',
                    GoalsCrr          => 'agoalcrr',
                    SessionDepth      => 'aseslen',
                    SessionNum        => 'asesnum',
                    SessionNumLimited => 'asesnumlim',
                    FirstPageShows    => 'fp_shows',
                    FirstPageSumPosShows => 'fp_shows_pos',
                    FirstPageClicks   => 'fp_clicks',
                    FirstPageSumPosClicks => 'fp_clicks_pos',
                    # считаем эти два поля на своей стороне, чтобы можно было переагрегировать
                    # AvgClicksPosition => 'fp_clicks_avg_pos',
                    # AvgShowsPosition  => 'fp_shows_avg_pos',
                    Bounces           => 'bounces',
                    BounceRatio       => 'bounce_ratio',
                    PrGoodMultiGoal   => 'aprgoodmultigoal',
                    CriterionType     => 'criterion_type',
                    PrGoodMultiGoalCPA => 'aprgoodmultigoal_cpa',
                    PrGoodMultiGoalConversionRate => 'aprgoodmultigoal_conv_rate',
                    RF                => 'avg_view_freq',
                    Reach             => 'uniq_viewers',
                    CPM               => 'avg_cpm',
                    DealExportID      => 'deal_export_id',
                    ExpSegmentName    => 'ab_segment',
                    AvgBid            => 'avg_bid',
                    AvgCPMBid            => 'avg_cpm_bid',
                    AvgTimeToConversion  => 'avg_time_to_conv',
                    MeaningfulGoalValue => 'meaningfulgoalvalue',
                    GoalsProfit => 'agoals_profit',
                    CandidatesStatAuctionHits => 'auction_hits',
                    CandidatesStatAuctionWins => 'auction_wins',
                    CandidatesStatWinRate => 'auction_win_rate',
                    CandidatesStatShowRate => 'imp_to_win_rate',
                    CandidatesStatReach => 'imp_reach_rate',
                    ServedImpressions => 'served_impressions',
                    InventoryType     => 'inventory_type',
                    Browser                       => 'browser',
                    OperatingSystem               => 'operating_system',
                    TemplateID                    => 'template_id',
                    ClientID                    => 'client_id',
                    mClientID => 'client_login',
                    PlaceID                    => 'place_id',
                    PageID                    => 'page_id',
                    PlaceDescription                    => 'place_description',
                    TurboPageType => 'turbo_page_type',
                    ContentTargeting => 'content_targeting',
                    ConversionPctPV     => 'pv_aconv',
                    BounceRatioPV       => 'pv_bounce_ratio',
                    GoalsNumPV          => 'pv_agoalnum',
                    GoalsIncomePV       => 'pv_agoalincome',
                    GoalsCrrPV          => 'pv_agoalcrr',
                    SessionDepthPV      => 'pv_aseslen',
                    SessionNumPV        => 'pv_asesnum',
                    SessionNumLimitedPV => 'pv_asesnumlim',
                    RegionSource        => 'region_source',

                    %multi_goals_fields_bs2direct,
                    (map { $countable_direct2bs{$_} => $_ } keys %countable_direct2bs),
                );

# соответствие названий полей из множества "словарей" в директе, и названий в ответе БК, при этом поле пришедшее от БК нужно преобразовать 
# к первому соответствующему значению bs из словаря, а новое поле создать дополнительно, с необходимым преобразованием исходного поля
my %common_dict_fields_direct2bs_with_copy = (
    targettype => 'TargetType',
    contexttype => 'ContextType',
    contexttype_orig => 'ContextCondType', # на выходе перетирает contexttype (т.е. является более приоритетным срезом)
);

my %preserved_map_field_names = ( mClientID => 1 ); # для указанных полей не убираем префикс m

my @stat_required_fields = qw/Shows Clicks Cost BonusNoVAT
                              FirstPageShows FirstPageSumPosShows FirstPageClicks FirstPageSumPosClicks
                              SessionNum SessionNumLimited SessionDepth Bounces BounceRatio
                              PrGoodMultiGoal PrGoodMultiGoalCPA PrGoodMultiGoalConversionRate/;
our @goal_required_fields = qw/GoalsNum GoalsIncome GoalsProfit/;

my @winrate_required_fields = qw/PerformanceCoverage/;

my @fields_multipied_by_10x6   = uniq(
    qw/sum bonus agoalincome pv_agoalincome av_sum agoalcost pv_agoalcost avg_cpm aprgoodmultigoal_cpa avg_bid avg_cpm_bid agoals_profit pv_agoals_profit/,
    Stat::Fields::get_money_fields()
);
my @suffixes_no_multipied_by_10x6   = qw/_delta/;
my %suffix_by_targettype_direct2bs = (_0 => '_search', _1 => '_context');
my %suffix_by_targettype_bs2direct = reverse %suffix_by_targettype_direct2bs;

my %yandex_ssp_title = (bs => iget_noop('Яндекс'),
                        direct => iget_noop('Яндекс и РСЯ'));

my $NEGATIVE_PERIOD_LENGTH_ERROR = 'NEGATIVE_PERIOD_LENGTH_ERROR';

=head2 _get_bs_stream_stat()

    Запрашивает статистику в БК, парсит и возвращает в подготовленном виде

=cut

sub _get_bs_stream_stat
{
    my ($report_opts, %O) = @_;

    _debug(Dumper [ $report_opts, \%O ]);

    my $profile = Yandex::Trace::new_profile('stat_stream_extended:_get_bs_stream_stat', tags => 'bs_http_get_report');

    my $report_opts_for_request = yclone($report_opts);
    $report_opts_for_request->{countable_fields} = [_get_unsuffixed_stat_required_fields($report_opts)] unless defined $report_opts_for_request->{countable_fields};
    delete $report_opts_for_request->{external_countable_fields_override};
    if (ref $report_opts_for_request->{countable_fields} && $report_opts_for_request->{countable_fields_for_order_by}) {
        @{$report_opts_for_request->{countable_fields}} = uniq(@{$report_opts_for_request->{countable_fields}},
                                                               @{$report_opts_for_request->{countable_fields_for_order_by}}, @{$report_opts_for_request->{extra_countable_fields}});
    }

    delete $report_opts_for_request->{$_} for qw/countable_fields_for_order_by
                                                extra_countable_fields
                                                compare_periods
                                                consumer
                                                stat_type
                                                _real_limit
                                                goals_mapping
                                                using_mol_page_name
                                                is_perf_show_condition
                                                creative_free/;

    $report_opts_for_request->{countable_fields_by_targettype} = _get_countable_fields_by_targettype($report_opts);

    _print_log($report_opts_for_request);

    my $ClientID = $O{ClientID_for_stat_experiments};
    my $url = BS::URL::get('master_report', $ClientID ? (ClientID => $ClientID) : (OrderID => $report_opts->{order_ids}), $FORCE_PREPROD ? (force_preprod => 1) : ());
    my $report_opts_json = to_json($report_opts_for_request);
    
    # curl -H 'Content-type: application/json' '$url' -d '$report_opts_json'
    _debug("POST\t$url\t$report_opts_json");

    my $body_iterator;
    my $resp = {};
    
    if (!$USE_SIMPLE_HTTP_REQUEST) {
        $body_iterator = Stat::StreamExtended::HttpBodyIterator->new(
            url => $url,
            request_body => $report_opts_json,
            timeout => $O{timeout} // 590, # у nginx таймаут 600
            log => _log(),
            profile => $profile,
        );
        undef $profile;
    } else {
        $resp = http_parallel_request(
            POST    => { 1 => { url => $url,
                    body            => $report_opts_json,
                    headers         => { 'Content-Type' => 'application/json' } }
            },
            timeout => $O{timeout} // 590, # у nginx таймаут 600
        )->{1};

        undef $profile;

        unless ($resp->{is_success}) {
            utf8::decode($resp->{headers}->{Reason});
            _log()->die("http_parallel_request($url\t$report_opts_json): $resp->{headers}->{Status} $resp->{headers}->{Reason}");
        }

        $body_iterator = Stat::StreamExtended::FakeHttpBodyIterator->new(
            log => _log(),
            fake_response_body => \$resp->{content},
        );
    }

    my $parser;
    eval {
        $parser = _bs_stream_content_parse_iterator($body_iterator, $report_opts, four_digits_precision => $O{four_digits_precision});
    };
    if (my $err = $@) {
        if ($err =~ /Memory limit exceeded/ || ($resp->{content} // '') =~ /Memory limit exceeded/) {
            _log()->die("BS response: Memory limit exceeded");
        } elsif ($err =~ /Timeout exceeded/ || ($resp->{content} // '') =~ /Timeout exceeded/) {
            _log()->die("BS response: Timeout exceeded");
        } else {
            _log()->die($err);
        }
    }

    return $parser;
}



=head2 _bs_stream_content_parse_iterator($content_strref, $report_opts)

    Парсинг быстрого отчета БК

=cut

sub _bs_stream_content_parse_iterator
{
    my ($body_iterator, $report_opts, %O) = @_;

    my $parser = new Stat::StreamExtended::ParseIterator(body_iterator => $body_iterator,
                                                         log => _log(),
                                                         stat_required_fields => [_get_stat_required_fields($report_opts)],
                                                         report_opts => $report_opts,
                                                         _bs2direct_field => \&_bs2direct_field,
                                                         _norm_field_by_suf => \&_norm_field_by_suf,
                                                         _process_row => \&_process_row,
                                                         _process_row_cache => \&_process_row_cache,
                                                         four_digits_precision => $O{four_digits_precision},
                                                         );

    my $headers = $parser->headers;
    _log()->die("Bad stat: status $headers->{status}, error: $headers->{error_text}") if $headers->{status};

    return $parser;
}

=head2 _get_countable_fields_by_targettype

    на основе опций отчета, возвращает ссылку на список полей по которым нужна детализация по типам площадок

=cut

sub _get_countable_fields_by_targettype {
    my $report_opts = shift;

    if (defined $report_opts->{countable_fields_by_targettype}) {
        if (ref $report_opts->{countable_fields_by_targettype}) {
            my @unsupported_fields = grep { !( $fields_opt{$_} && $fields_opt{$_}->{is_countable} && $fields_opt{$_}->{order} ) } @{$report_opts->{countable_fields_by_targettype}};
            _log()->die("Unsupported fields in countable_fields_by_targettype: " . join(', ', @unsupported_fields)) if @unsupported_fields;

            return [ map { $fields_opt{$_}->{order} } @{$report_opts->{countable_fields_by_targettype}} ];
        } else {
            return $report_opts->{countable_fields_by_targettype} ? [_get_unsuffixed_stat_required_fields($report_opts)] : [];
        }
    }
    return [];
}

=head2 _get_unsuffixed_stat_required_fields

    в зависимости от настроек отчета отдает список обязательных измеримых полей в ответе БК, без добавления необходимых суффиксов

=cut

sub _get_unsuffixed_stat_required_fields {
    my $report_opts = shift;

    my @result;
    if ($report_opts->{external_countable_fields_override}) {
        my @countable_fields = grep {$_} map {$fields_opt{$_}->{field}} @{$report_opts->{external_countable_fields_override}};

        if ((scalar @countable_fields) == (scalar @{$report_opts->{external_countable_fields_override}})) {
            push @result, @countable_fields;
        } else {
            _log()->die(["couldn't convert some fields", $report_opts->{external_countable_fields_override}]);
        }
    } else {
        push @result, @stat_required_fields, @goal_required_fields;
    }
    push @result, @winrate_required_fields if $report_opts->{with_performance_coverage};

    state $required_when_filtering //= { map { $_ => 1 } Stat::Fields::get_bs_fields_required_when_filtering() };
    my @filters_post = keys %{$report_opts->{filters_post}};
    push @result, grep { $required_when_filtering->{$_} } @filters_post;

    return @result;
}

=head2 _get_stat_required_fields
    
    в зависимости от настроек отчета отдает список обязательных измеримых полей в ответе БК

=cut

sub _get_stat_required_fields {
    my $report_opts = shift;

    if ($report_opts->{compare_periods}) {
        return apply_suffixes([_get_unsuffixed_stat_required_fields($report_opts), @{$report_opts->{extra_countable_fields} // []}], [periods_full_suffix_list()])
    } else {
        return _get_unsuffixed_stat_required_fields($report_opts), @{$report_opts->{extra_countable_fields} // []},
           apply_suffixes(_get_countable_fields_by_targettype($report_opts), [keys %suffix_by_targettype_bs2direct]);
    }
}

=head2 _bs2direct_field

    преобразовывает название поля пришедшее от БК в название принятое в Директе
    предварительно разыменовуются результаты маппинга (в связи с тем что все поля в БК отдаются CamelCase-ом, поля-результаты маппингов
    формируем как m{Название реального поля})

=cut

sub _bs2direct_field {
    my ($f, $suf) = _norm_field_by_suf($_[0]);
    if ($f =~ /^m([A-Z].*)$/ && !$preserved_map_field_names{$f}) {
        $f = $1;
    }

    state $suffix_by_periods = { map { $_ => $_ } periods_full_suffix_list() };

    return ($fields_bs2direct{$f} || $f) . ($suffix_by_targettype_bs2direct{$suf} // $suffix_by_periods->{$suf} // '');
}

=head2 _norm_field_by_suf

    возвращает название поля без суффиксов разбивки по поиск/контекст и по периодам, и сам суффикс

=cut

sub _norm_field_by_suf {
    my $field = shift;

    my $suf = '';
    my $re_suf = join '|', keys %suffix_by_targettype_direct2bs, keys %suffix_by_targettype_bs2direct,
                           periods_full_suffix_list();
    if ($field =~ /^(.+?)($re_suf)$/) {
        ($field, $suf) =  ($1, $2 // '');
    }

    return wantarray ? ($field, $suf) : $field;
}

=head2 _process_row_cache

    возвращает хеш с некоторыми предрасчитанными "закешированными" данными, необходимыми для обработки строк статистики

=cut

sub _process_row_cache {
    my $header_fields = shift;

    my $_cache = {};
    $_cache->{fields_multipied_by_10x6} = [];
    $_cache->{fields_suffixes} = [];
    my @flat_header_fields = map {ref $_ ? @{$_} : $_} @$header_fields;
    for my $field_with_suf (@flat_header_fields) {
        my ($f, $suf) = _norm_field_by_suf($field_with_suf);
        if (any { $f eq $_ } @fields_multipied_by_10x6) {
            next if $suf && any { $_ eq $suf } @suffixes_no_multipied_by_10x6;
            push @{$_cache->{fields_multipied_by_10x6}}, $f.$suf;
        }

        push @{$_cache->{fields_suffixes}}, $suf;
    }
    @{$_cache->{fields_suffixes}} = uniq @{$_cache->{fields_suffixes}};

    return $_cache;
}

=head2 _process_row

=cut

sub _process_row {
    my ($row, $report_opts, $stat_extra_data, $_cache) = @_;
    $stat_extra_data //= {};
    $_cache //= {};

    # маппим обратно в стандартизованные значения словарные поля
    my $dict_fields_to_map = $_cache->{dict_fields_to_map} //= _get_dict_fields_to_map($report_opts->{consumer});

    for my $f (keys %$dict_fields_to_map) {
        if ($f eq 'position' && ($report_opts->{consumer} // 'default') eq 'default' && exists $row->{$f}) {
            # кастомное поведение чтоб долго не править фронт
            my $v = $dict_fields_to_map->{$f}->{map}->{$row->{$f}};
            $row->{$f} = $v ? (ref $v->{bs} ? $v->{bs}->[0] : $v->{bs}) : $Stat::Const::NON_PRIME_TYPE_ID;
        } else {
            if (my $src_field = $common_dict_fields_direct2bs_with_copy{$f}) {
                if (exists $row->{$src_field}) {
                    my $src_field_value = $row->{$src_field};
                    $row->{$src_field} = $dict_fields_to_map->{$f}->{map}->{$src_field_value}->{bs} // $row->{$src_field};
                    $row->{$f} = $dict_fields_to_map->{$f}->{map}->{$src_field_value}->{name} // $dict_fields_to_map->{$f}->{default};

                }
            } elsif (exists $row->{$f}) {
                $row->{$f} = $dict_fields_to_map->{$f}->{map}->{$row->{$f}}->{name} // $dict_fields_to_map->{$f}->{default};
            }
        }
    }

    # ContextCondType/contexttype_orig являются более приоритетными значениями, если они есть
    if (exists $row->{ContextCondType}) {
        $row->{ContextType} = delete $row->{ContextCondType};
        $row->{contexttype} = delete $row->{contexttype_orig};
    }

    # Проверяем кодировку у полей, которые теоретически могут иметь с ней проблемы
    for my $key (qw/search_query phrase PhraseWBM Phrase OriginalPhrase ssp matched_phrase PageName/) {
        if (exists $row->{$key}) {
            utf8::decode($row->{$key});
        }
    }

    ##
    if (exists $row->{PhraseID} || exists $row->{bs_criterion_id}) {
        if (exists $row->{PhraseWBM}) {
            $row->{phrase} = delete $row->{PhraseWBM};
            if ($row->{ContextType} == $CONTEXT_TYPE_BROADMATCH) {
                $row->{PhraseID} = 0 if exists $row->{PhraseID};
                $row->{bs_criterion_id} = undef if exists $row->{bs_criterion_id};
            }
        }
        if (exists $row->{OriginalPhrase}) {
            $row->{phrase} = delete $row->{OriginalPhrase};
        }
        if (any { $row->{ContextType} == $_ } ($CONTEXT_TYPE_RET, $CONTEXT_TYPE_DYNAMIC) ) {
            delete $row->{$_} for qw/phrase PhraseType/;
        } elsif (any { $row->{ContextType} == $_ } ($CONTEXT_TYPE_PHRASE, $CONTEXT_TYPE_BM_SYNONYM)) {
            if (($row->{PhraseType} // 0) == 1) {
                $row->{phrase} = '@' . $row->{phrase};
            } else {
                $row->{phrase} = Direct::PhraseTools::quoted_phrases_bs2direct($row->{phrase});
            }
            delete $row->{PhraseType};
        } elsif ($row->{ContextType} == $CONTEXT_TYPE_BROADMATCH) {
            $row->{phrase} = Direct::PhraseTools::quoted_phrases_bs2direct($row->{phrase});
        }
    }

    $row->{matched_phrase} = Direct::PhraseTools::quoted_phrases_bs2direct($row->{matched_phrase}) if defined $row->{matched_phrase};

    $row->{adgroup_id} = 0 if exists $row->{adgroup_id} && $row->{adgroup_id} < 0;

    $row->{OrderID} //= $stat_extra_data->{cid2OrderID}->{$row->{cid}} if exists $row->{cid};
    $row->{cid} //= $stat_extra_data->{OrderID2cid}->{$row->{OrderID}} if exists $row->{OrderID};

    my $page_id = $row->{page_id} // $row->{PageGroupID};
    if (defined $page_id && $stat_extra_data->{PageID_info}->{$page_id}) {
        hash_merge $row, hash_cut($stat_extra_data->{PageID_info}->{$page_id},
                                  qw/page_group page_name page_domain page_sorting/);
        $row->{page_name} = $row->{PageName} if exists $row->{PageName};
    } elsif (defined($row->{PageName}) && defined($row->{TargetType})) {
        # если запрашивалась группировка по PageName, а по PageID/PageGroupID - нет, выводим значения page_* полей
        # из того, что есть.
        # часть логики взята из
        # https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/src/main/java/ru/yandex/direct/jobs/bannersystem/dataimport/ImportPagesJob.java#L118
        my $page_name = $row->{PageName};
        # МОЛ присылает строчку "yandex" для площадок с флажком is ya group
        my $is_ya_group = $page_name eq 'yandex';
        # придумываем фейковый group_nick, т.к. он требуется в некоторых местах Директа. Точное значение не так важно,
        # но он должен заканчиваться на .0 или .3. А для яндексовых пейджей должен быть "yandex.[03]"
        my $page_group = $is_ya_group ? 'yandex' : $page_name.'.fake';
        $page_group .= '.'.($row->{TargetType} == 3 ? '3' : '0');
        $row->{page_group} = $page_group;
        $row->{page_name} = $page_name;
        $row->{page_domain} = $is_ya_group ? 'yandex.ru' : $page_name;
        $row->{page_sorting} = $is_ya_group ? 0 : 9;
    }
    delete $row->{$_} for qw/PageID PageGroupID PageName/;
    if (defined($row->{page_name})) {
        my $page_name = $row->{page_name};
        my $magic_page_names;
        # локализуем магические названия площадок
        if ($report_opts->{using_mol_page_name}) {
            # если запрашиваем название площадки напрямую из МОЛ, преобразуем "yandex" -> "Яндекс"/"Yandex"
            $magic_page_names = Stat::Tools::get_magic_page_name_translations($report_opts->{lang});
        } else {
            # если название площадки берём из своего справочника, преобразуем "Яндекс" -> "Яндекс"/"Yandex"
            $magic_page_names = Stat::Tools::get_old_magic_page_name_translations($report_opts->{lang});
        }
        if ($magic_page_names->{$page_name}) {
            $row->{page_name} = $magic_page_names->{$page_name};
        }
    }

    if (exists $row->{campaign_type}) {
        state $camp_type_bs2direct;
        unless ($camp_type_bs2direct) {
             my $cdict = Stat::Tools::consumer_dict_by_name('CAMP_TYPES', $report_opts->{consumer});
             $camp_type_bs2direct = { map {   $_ => $cdict->{need_reverse_translation} ? $cdict->{values}->{$_}->{name} : $_
                                          } keys %{$cdict->{values}} };
        }
        $row->{campaign_type} = $camp_type_bs2direct->{$row->{campaign_type}} if exists $camp_type_bs2direct->{$row->{campaign_type}};
    }

    if ($row->{ssp} && $row->{ssp} eq $yandex_ssp_title{bs}) {
        $row->{ssp} = iget($yandex_ssp_title{direct});
    }

    # костыль для API (группировка по StatGoals)
    $row->{goal_id} = $report_opts->{filters_pre}->{GoalID}->{eq} || 0;

    $_cache->{fields_multipied_by_10x6} //= [ apply_suffixes(\@fields_multipied_by_10x6, xminus([$report_opts->{compare_periods}
                                                                                                    ? periods_suffix_list()
                                                                                                    : ('', keys %suffix_by_targettype_bs2direct)],
                                                                                                  \@suffixes_no_multipied_by_10x6 )) ];
    my %fields_multipied_by_10x6 = map {$_ => 1} @{$_cache->{fields_multipied_by_10x6}};
    foreach my $f (keys %{$row}) {
        my $field_without_postfix = $f;
        $field_without_postfix =~ s/(\w+)_\d+_\d$/$1/;
        if (defined $fields_multipied_by_10x6{$field_without_postfix} && defined $row->{$f}) {
            $row->{$f} /= 1_000_000;
        }
    }
    $_cache->{fields_suffixes} //= [''];

    my $camp_type = '';
    if (defined $stat_extra_data->{OrderID2camp_type} && $row->{OrderID}) {
        $camp_type = $stat_extra_data->{OrderID2camp_type}->{$row->{OrderID}}
    } elsif (defined $stat_extra_data->{cid2camp_type} && $row->{cid}) {
        $camp_type = $stat_extra_data->{cid2camp_type}->{$row->{cid}}
    }

    my $banner_type = '';
    if (defined $stat_extra_data->{bid2banner_type} && $row->{bid}) {
        $banner_type = $stat_extra_data->{bid2banner_type}->{$row->{bid}}
    }

    # статистика по % полученных показов собирается только для ДМО (performance/смарт-баннер) кампаний, костыль чтоб отличать undef от 0
    if ($report_opts->{with_performance_coverage}) {
        if ($camp_type ne 'performance') {
            for my $suf (@{$_cache->{fields_suffixes}}) {
                if (defined $row->{"winrate$suf"}) {
                    $row->{"winrate$suf"} = undef;
                }
            }
        }
    }

    # до момента выкладки BSDEV-60668 сами подменяем показы на undef при фильтре/группировке по условиям показа
    if ($report_opts->{is_perf_show_condition}
        && $camp_type eq 'performance') {

        $row->{shows} = undef;
    }

    for my $suf ($report_opts->{compare_periods} ? periods_suffix_list() : '') {
        if ($row->{"date$suf"} || $report_opts->{"date_from$suf"}) {
            @{$row}{map { $_.$suf } qw/stat_date stat_date_end/} = Stat::Tools::get_date_subperiod($row->{"date$suf"} // $report_opts->{"date_from$suf"},
                                                                                                   $row->{"date$suf"} ? $report_opts->{group_by_date} : 'none',
                                                                                                   @{$report_opts}{map { $_.$suf } qw/date_from date_to/});
        }
    }

    for my $suf ($report_opts->{compare_periods} ? periods_suffix_list() : '') {
        if (my $stat_date_ymd = $row->{"stat_date$suf"}) {
            $stat_date_ymd =~ s/-//g;
            if ($stat_date_ymd le $BS_BOUNCES_BORDER_DATE) {
                for my $clean_suf ($report_opts->{compare_periods}
                                        ? ($suf, periods_delta_suffix_list())
                                        : @{$_cache->{fields_suffixes}}) {
                    if (defined $row->{"bounce_ratio$clean_suf"}) {
                        $row->{"bounce_ratio$clean_suf"} = undef;
                    }
                }
            }

            if ($stat_date_ymd le $BS_AVG_BID_BORDER_DATE) {
                for my $clean_suf ($report_opts->{compare_periods} ? ($suf, periods_delta_suffix_list()) : @{$_cache->{fields_suffixes}}) {
                    $row->{"avg_bid$clean_suf"} = undef if defined $row->{"avg_bid$clean_suf"};
                    $row->{"avg_cpm_bid$clean_suf"} = undef if defined $row->{"avg_cpm_bid$clean_suf"};
                }
            }

            if ($stat_date_ymd le $BS_PROFIT_BORDER_DATE) {
                for my $clean_suf ($report_opts->{compare_periods} ? ($suf, periods_delta_suffix_list()) : @{$_cache->{fields_suffixes}}) {
                    $row->{"agoals_profit$clean_suf"} = undef if defined $row->{"agoals_profit$clean_suf"};
                }
            }
        }
    }

    #Нужно для креатива менять размер 0x0 на текст 'Адаптивный'. Предполагаем, что только у креатива может быть такой размер.
    if (exists $row->{image_size} && $row->{image_size} eq $Direct::Model::Creative::ADAPTIVE_SIZE) {
        $row->{image_size} = $Direct::Model::Creative::ADAPTIVE_SIZE_TITLE;
    }

    if (exists $row->{content_targeting}) {
        Stat::Tools::add_content_targetings_goal_ids_text($row);
    }

    if ($banner_type =~ /^(performance|performance_main)$/ && $report_opts->{creative_free}) {
        $row->{bid} = undef;
    }

    return $row;
}


=head2 get_stream_stat

    Принимает на вход 
    order_ids => [1610872],
    date_from(''|_a|_b) => "20120819",
    date_to(''|_a|_b) => "20120820",
    group_by => [qw/ ... /], поля группировки
    group_by_date => (day|week|month|quarter|year|none)  по каким периодам группировка по дате
    order_by => [{field => 'bid', dir => 'asc'}, ...], поля для сортировки
    lang => 'ru', язык в котором извлекать мультиязычные данные (не обязательное поле), в частности для сортировки по регионам
    filters => {banner => {eq => [123,321]}},  поля для фильтрации статистики
    limits => {offset => 100, limit => 50}, смещение и лимит
    with_nds => 0|1, включать ли НДС в сумму (по-умолчанию = 1)
    with_discount => 0|1, учитывать ли скидки (по-умолчанию = 1)
    currency => <YND_FIXED|RUB|UAH|...>, в какой валюте отдавать статистику
    countable_fields_by_targettype => 0|1|[qw/field1 field2/] - нужно ли разбивать измеримые поля на всего/поиск/контекст
    with_winrate => 0|1 - нужно ли поле "% полученных показов"
    compare_periods => 0|1, запрашивается ли статистика в режиме сравнения двух периодов
    comsumer => api|web_ui|default - потребитель
    filter_by_consumer_values => 0|1 - если 1 значит значения в фильтрах по словарным полям заданы из набора потребителя
    external_countable_fields_override => [qw/.../], список полей в формате Директа. Если передан, то заменит собой список полей,
        спрашиваемых у БК по умолчанию. Все поля должны успешно транслироваться в конкретное поле формата БК.
    creative_free => 0|1, скрывать ли информацию о смарт-баннерах (фича creative_free_interface)

=cut

sub get_stream_stat {
    my %O = @_;


    _debug(Dumper { get_stream_stat => \%O });
    _log()->die("No OrderIDs defined") unless $O{order_ids} && ref $O{order_ids} eq 'ARRAY';

    my $stat;
    my $report_opts = { order_ids => $O{order_ids},
                        compare_periods => $O{compare_periods},
                        stat_type => $O{stat_type},
                        consumer => $O{consumer},
                        lang => $O{lang},
                        using_mol_page_name => $O{using_mol_page_name},
                    };

    if (@{$report_opts->{order_ids}}) {
        my $error = _add_period_opts($report_opts, hash_cut(\%O, qw/group_by/, apply_suffixes([qw/date_from date_to/], ['', periods_suffix_list()])));
        if ($error && $error eq $NEGATIVE_PERIOD_LENGTH_ERROR) {
            $stat = _get_empty_result($report_opts);
        } else {
            $report_opts->{request_cache} = {};
            _add_money_opts($report_opts, hash_cut(\%O, qw/with_discount with_nds currency/));
            _add_filter_opts($report_opts, hash_cut(\%O, qw/filters filter_by_consumer_values ClientID_for_stat_experiments/));
            _add_groupby_opts($report_opts, hash_cut(\%O, qw/group_by_date group_by region_level/));
            _add_extra_opts($report_opts, hash_cut(\%O, qw/countable_fields_by_targettype without_totals creative_free/));
            _add_orderby_opts($report_opts, hash_cut(\%O, qw/order_by group_by group_by_date/));
            _add_extra_countable_fields($report_opts, $O{extra_countable_fields}) if $O{extra_countable_fields} && @{$O{extra_countable_fields}};
            _add_limit_opts($report_opts, hash_cut(\%O, qw/limits partial_total_rows/));
            $report_opts->{external_countable_fields_override} = $O{external_countable_fields_override} if $O{external_countable_fields_override};
            _add_all_goals($report_opts, hash_cut(\%O, qw/filters ClientID_for_stat_experiments/));
            _add_available_goals($report_opts, hash_cut(\%O, qw/ClientID_for_stat_experiments/));

            _check_add_winrate_opts($report_opts, hash_cut(\%O, qw/group_by filters with_winrate is_mcc_mode/));

            delete $report_opts->{request_cache};
            $report_opts->{is_perf_show_condition} = any {$_ eq 'OriginalPhraseID' || $_ eq 'PhraseID' || $_ eq 'CriterionID'} @{$report_opts->{group_by}};
            $report_opts->{is_perf_show_condition} ||= any {$_ eq 'PerfFilterID'} keys %{$report_opts->{filters_pre}};

            if (_check_filter_empty_result(hash_merge {}, values %{ hash_cut($report_opts, qw/filters_pre filters_post/) }) ||
                (!$report_opts->{compare_periods} && $report_opts->{date_to} < $report_opts->{date_from}) ) {
                $stat = _get_empty_result($report_opts);
            } else {
                $stat = _get_bs_stream_stat($report_opts,
                    ClientID_for_stat_experiments => $O{ClientID_for_stat_experiments},
                    four_digits_precision => $O{four_digits_precision},
                    operator_ClientID => $O{operator_ClientID},
                );
            }
        }
    } else {
        $stat = _get_empty_result($report_opts);
    }

    return $stat;
}

=head2 _add_period_opts

    Проверяет, нормализует, и добавляет в настройки отчета период

=cut

sub _add_period_opts {
    my ($report_opts, $opts) = @_;

    my $date_from_limit = BS_STAT_START_DATE;
    if ((any { $_ eq 'search_query'} @{$opts->{group_by}}) ||
        (keys %{$opts->{filters}->{search_query} // {}}) ) {
        $date_from_limit = maxstr($date_from_limit,
                                  date($BS_SEARCH_QUERIES_BORDER_DATE)->add(days => 1)->ymd(''),
                                  date(today())->subtract(days => $BS_SEARCH_QUERIES_LAST_DAYS-1)->ymd(''));
    }
    if ((any { $_ eq 'contextcond_ext'} @{$opts->{group_by}}) ||
        (keys %{$opts->{filters}->{phrase_ext} // {}}) ) {
        $date_from_limit = maxstr($date_from_limit, date($BS_EXT_PHRASE_BORDER_DATE)->add(days => 1)->ymd(''));
    }

    for my $suf ($report_opts->{compare_periods} ? periods_suffix_list() : '') {
        $report_opts->{"date_from$suf"} = date(normalize_date($opts->{"date_from$suf"} // $BEGIN_OF_TIME_FOR_STAT))->ymd('');
        $report_opts->{"date_to$suf"} = date(normalize_date($opts->{"date_to$suf"} // today()))->ymd('');

        if ($report_opts->{"date_from$suf"} lt $date_from_limit) {
            $report_opts->{"date_from$suf"} = $date_from_limit;
        }

        if ($report_opts->{"date_to$suf"} < $report_opts->{"date_from$suf"}) {
            _log()->out(sprintf("Dates period %s is not correct: %s - %s", $suf ? qq/"$suf"/ : '', $report_opts->{"date_to$suf"}, $report_opts->{"date_from$suf"}));
            return $NEGATIVE_PERIOD_LENGTH_ERROR;
        }
    }

    if ($report_opts->{compare_periods} && $report_opts->{"date_from_a"} <= $report_opts->{"date_to_b"}) {
        _log()->die(qq/Dates period "a" must be after period "b"/);
    }
}

=head2 _add_money_opts

    Проверяет и добавляет в настройки отчета "денежные" опции (НДС, скидка, валюта)

=cut

sub _add_money_opts {
    my ($report_opts, $opts) = @_;

    $report_opts->{with_discount} = ($opts->{with_discount} // 1) ? 1 : 0;
    $report_opts->{with_vat} = ($opts->{with_nds} // 1) ? 1 : 0;

    if (defined $opts->{currency}) {
        _log()->die("Currency is incorrect: $opts->{currency}") unless $opts->{currency};
        $report_opts->{currency} = $opts->{currency};
    } else {
        my $orders_currencies = get_one_column_sql(PPC(OrderID => $report_opts->{order_ids}), ["
                                                        select distinct IFNULL(currency, 'YND_FIXED') 
                                                          from campaigns",
                                                         where => {OrderID => SHARD_IDS} ]);
        if (scalar(@$orders_currencies) <= 1) {
            $report_opts->{currency} = $orders_currencies->[0];
        } else {
            _log()->die("Can't determine one currency from order currencies: " . join(', ', @$orders_currencies));
        }
    }
}

=head2 _add_groupby_opts

    Проверяет и добавляет в настройки отчета поля для группировки, при необходимости инициирует добавление маппингов

=cut

sub _add_groupby_opts {
    my ($report_opts, $opts) = @_;

    _log()->die("Incorrect group_by_date parameter: $opts->{group_by_date}")
                if defined $opts->{group_by_date} && $opts->{group_by_date} !~ $Stat::Const::GROUP_BY_DATE_RE;
    $report_opts->{group_by_date} = $opts->{group_by_date} // 'none';

    _log()->die("Incorrect region_level parameter: $opts->{region_level}")
        if defined $opts->{region_level} && $opts->{region_level} !~ $Stat::Const::REGION_LEVEL_RE;
    if($opts->{region_level}){
        $report_opts->{region_level} = $opts->{region_level};
    }

    $report_opts->{group_by} //= [];
    $report_opts->{mapping} //= {};

    foreach my $field (@{$opts->{group_by} // []}) {
        my $field_opts = _get_field_opts($field);
        _log()->die("Incorrect field for groupping: $field") unless $field_opts && $field_opts->{group};

        push @{$report_opts->{group_by}}, @{$field_opts->{group}};
        foreach my $m (@{$field_opts->{group_mappings} // []}) {
            _add_mapping($report_opts->{mapping}, $m, hash_cut($report_opts, qw/order_ids filters_pre consumer request_cache/));
        }
    }
    @{$report_opts->{group_by}} = uniq @{$report_opts->{group_by}};
}

=head2 _add_filter_opts

    Проверяет и добавляет в настройки отчета опции для фильтрации

=cut

sub _add_filter_opts {
    my ($report_opts, $opts) = @_;

    $report_opts->{filters_pre} //= {};
    $report_opts->{filters_post} //= {};
    $report_opts->{mapping} //= {};
    $report_opts->{goals_mapping} = {};

    foreach my $field (keys %{$opts->{filters} // {}}) {
        my ($norm_field, $suf);
        my $field_opts = _get_field_opts($field);

        _log()->die("Incorrect field for filtering: $field") unless $field_opts && $field_opts->{filter};

        next unless keys %{$opts->{filters}->{$field}};

        my $wrong_op_list = xminus [keys %{$opts->{filters}->{$field}}], $field_opts->{filter_op};
        _log()->die("Unavailable filter operations for field $field: " . join(', ', @$wrong_op_list)) if @$wrong_op_list;

        my $common_dict;
        if ($field_opts->{common_dict_name}) {
            $common_dict = Stat::Tools::consumer_dict_by_name($field_opts->{common_dict_name},
                                                              $opts->{filter_by_consumer_values} ? $report_opts->{consumer} : undef);
        }
        while (my ($op, $value) = each %{$opts->{filters}->{$field}}) {
            if (ref $value && ($field_opts->{filter_single_value} || any { $op eq $_ } qw/lt gt/)) {
                _log()->die("Unexpected multiple filter values, operation: $op, field: $field") ;
            }

            if (any { $op eq $_ } qw/lt gt/) {
                my $value_to_check = ($value =~ s/^(-?)0+([1-9]|0\.|0$)/$1$2/r); #/ чиним подсветку
                if (is_valid_float($value_to_check)) {
                    $value = $value_to_check;
                } else {
                    _log()->die("Incorrect number value: $value, operation: $op, field: $field");
                }
            } else {
                $value = [uniq grep { defined $_ && length($_) > 0 } xflatten($value)];
            }

            my $filter_values = $field_opts->{filter_values};
            if (!$filter_values && $common_dict) {
                $filter_values = $common_dict->{need_reverse_translation} ? [map { $_->{name} } values %{$common_dict->{values}}] : [keys %{$common_dict->{values}}];
            }
            if ($filter_values) {
                my $wrong_values_list = xminus $value, $filter_values;
                _log()->die("Incorrect filter values for field $field: " . join(', ', @$wrong_values_list)) if @$wrong_values_list;
            }
            if ($field_opts->{filter_prepare}) {
                my $mapping;
                ($value, $op, $mapping) = $field_opts->{filter_prepare}->($value, $op, $report_opts->{order_ids}, $opts->{ClientID_for_stat_experiments});
                hash_merge($report_opts->{goals_mapping}, $mapping) if ($mapping && %$mapping);
                next if !defined $op;
            } elsif ($common_dict) {
                my $common_dict_default = exists $field_opts->{common_dict_default}
                                                ? $field_opts->{common_dict_default}
                                                : 'undefined';
                ($value, $op) = _prepare_common_dict_filter($value, $op, $common_dict, default => $common_dict_default);
                next if !defined $op;
            }
            my $field_filter = $report_opts->{'filters_' . $field_opts->{filter_type}}->{$field_opts->{filter}} //= {};
            if ($field_opts->{filter_single_value}) {
                $value = $value->[0] if ref $value eq 'ARRAY';
            } elsif ($field_filter->{$op}) {
                if (any { $op eq $_ } qw/eq starts_with contains contains_in_plus_part/) {
                    # случай field IN (1,2,3) AND field IN (3,4,5), в результате должно остаться пересечение значений
                    $value = xisect $field_filter->{$op}, $value;
                } elsif (any { $op eq $_ } qw/ne not_starts_with not_contains not_contains_in_plus_part/) {
                    # случай field NOT IN (1,2,3) AND field NOT IN (3,4,5), в результате должно остаться объединение значений
                    $value = [uniq @{$field_filter->{$op}}, @$value];
                }
            }
            unless (ref $value && !@$value && any { $op eq $_ } qw/ne not_starts_with not_contains not_contains_in_plus_part/) {
                unless (defined $norm_field) {
                    ($norm_field, $suf) = _norm_field_by_suf($field);
                }
                # поля, которые из БК приходят умноженными на 10x6, фильтруем также с таким коеффициентом
                if ((any { $_ eq $norm_field } @fields_multipied_by_10x6) &&
                    (none { $_ eq $suf } @suffixes_no_multipied_by_10x6) ) {
                    if (ref $value) {
                        @$value = map { $_ * 1_000_000 } @$value;
                    } else {
                        $value *= 1_000_000;
                    }
                }
                # фильтр по "значимым" пустым строкам передаем просто как пустую строку (они в БК не отсекаются)
                if (ref $value) {
                    @$value = map { $_ eq $Stat::Const::SIGNIFICANT_EMPTY_STRING ? '' : $_ } @$value;
                } else {
                    $value = '' if $value eq $Stat::Const::SIGNIFICANT_EMPTY_STRING;
                }

                $field_filter->{$op} = $value;
            }
        }
    }

    # маппинги добавляю тут, для оптимизации (чтоб все filters_pre/post уже были сформированы)
    foreach my $field (keys %{$opts->{filters} // {}}) {
        my $field_opts = _get_field_opts($field);
        next unless $report_opts->{'filters_' . $field_opts->{filter_type}}->{$field_opts->{filter}};

        foreach my $m (@{$field_opts->{filter_mappings} // []}) {
            _add_mapping($report_opts->{mapping}, $m, hash_cut($report_opts, qw/order_ids filters_pre consumer request_cache/));
        }
    }
}

=head2 _add_extra_opts

    Проверяет и добавляет в настройки отчета опции про то, нужно ли детализировать измеримые поля до всего/поиск/контекст
    и другие специфические опции

=cut

sub _add_extra_opts {
    my ($report_opts, $opts) = @_;

    $report_opts->{countable_fields_by_targettype} = $opts->{countable_fields_by_targettype};
    $report_opts->{without_totals} = 1 if $opts->{without_totals};
    $report_opts->{creative_free} = 1 if $opts->{creative_free};
}

=head2 _add_extra_countable_fields

    Проверяет и добавляет в настройки отчета возвращаемые расчетные поля

=cut

sub _add_extra_countable_fields {
    my ($report_opts, $extra_countable_fields) = @_;
    my %multi_goals_map = reverse %multi_goals_fields_bs2direct;

    foreach my $field (@$extra_countable_fields) {
        my $field_opts = _get_field_opts($field);
        if ($field_opts->{is_countable}) {
            if (defined $report_opts->{filters_pre} && defined $report_opts->{filters_pre}{MultiGoalsID} && exists $multi_goals_map{$field}) {
                push @{$report_opts->{extra_countable_fields}}, $multi_goals_map{$field};
            } else {
                push @{$report_opts->{extra_countable_fields}}, $field_opts->{field};
            }
        }
    }
}

=head2 _add_all_goals

    Если запрошены поля, значение которых зависит от целей и не заданы цели в фильтрации, добавляем в специальный фильтр все цели клиента

=cut

sub _add_all_goals {
    my ($report_opts, $opts) = @_;

    my $need_goal_list;
    if (exists $opts->{filters} && any {exists $opts->{filters}{$_}} qw/goal_ids goal_id single_goal_id/) {
        $need_goal_list = 0;
    } else {
        if ($report_opts->{external_countable_fields_override}
            && any { $goals_fields{$_} } @{$report_opts->{external_countable_fields_override}}) { # запросили целевое поле в API
            $need_goal_list = 1;
        } elsif ($report_opts->{extra_countable_fields}
            && any {$goals_fields{$_} || $bs_goals_fields{$_}} @{$report_opts->{extra_countable_fields}}) { # запросили целевое поле
            $need_goal_list = 1;
        } elsif (exists $opts->{filters} && any {$goals_fields{$_}} keys %{$opts->{filters}}) { # указали целевое поле в фильтрации
            $need_goal_list = 1;
        } elsif ($report_opts->{stat_type} && any { $report_opts->{stat_type} eq $_ } qw/campdate common phrase_date geo pages page_dates/) { # для указанных типов отчетов передаем цели всегда
            $need_goal_list = 1;
        }

    }

    if (Client::ClientFeatures::has_grid_goals_filtration_for_stat_feature($opts->{ClientID_for_stat_experiments}) && $need_goal_list) {
        my $orders_goal_ids = Stat::Tools::get_orders_goal_ids($report_opts->{order_ids});
        my $client_goal_ids = MetrikaCounters::get_client_goal_ids(
            $opts->{ClientID_for_stat_experiments},
            goal_ids_for_check_ecom => $orders_goal_ids,
            extra_goal_ids => [map { $_ } keys %Stat::Const::MOBILE_APP_SPECIAL_GOALS]
        );
        if ($client_goal_ids) {
            @$client_goal_ids = map { "$_" } @$client_goal_ids; # цели отправляем строковыми значениями в JSON
            my $filter_field_name = _get_field_opts('goal_id_list')->{filter};
            $report_opts->{filters_pre}{$filter_field_name} = { eq => $client_goal_ids };
        }
    }

    return;
}

=head2 _add_available_goals

    Если у клиента включена фича получения дохода только по доступным целям
    и при этом в запросе за конверсионными данными участвуют цели не обязательно доступные,
    то передаем дополнительное поле фильтрации AvailableGoalIDList, в котором
    лежат доступные цели из передаваемых в MultiGoalsID, GoalIDList или GoalID
    для того, чтобы БК собрала доход только по доступным целям,
    а остальные конверсионные данные - по всем переданным целям.

=cut

sub _add_available_goals {
    my ($report_opts, $opts) = @_;

    my $client_id = $opts->{ClientID_for_stat_experiments};
    my $order_ids = $report_opts->{order_ids};
    if (Client::ClientFeatures::has_get_revenue_only_by_available_goals_in_perl($client_id)
            && !_is_only_available_goals_for_filter($order_ids, $client_id)) {
        my $goal_ids;
        my $filters_pre = $report_opts->{filters_pre};
        if (exists $filters_pre->{MultiGoalsID}) {
            $goal_ids = $filters_pre->{MultiGoalsID}->{eq};
        } elsif (exists $filters_pre->{GoalIDList}) {
            $goal_ids = $filters_pre->{GoalIDList}->{eq};
        } elsif (exists $filters_pre->{GoalID}) {
            $goal_ids = $filters_pre->{GoalID}->{eq};
        }
        if ($goal_ids) {
            my @available_goal_ids = _get_available_requested_goal_ids($goal_ids, $order_ids, $client_id);
            $report_opts->{filters_pre}{AvailableGoalIDList} = { eq => \@available_goal_ids };
        }
    }
}

=head2 _add_orderby_opts

    Проверяет и добавляет в настройки отчета опции для сортировки

=cut

sub _add_orderby_opts {
    my ($report_opts, $opts) = @_;

    $report_opts->{order_by} //= [];
    $opts->{order_by} //= [];

    my @group_by = @{$opts->{group_by} // []};
    if ($opts->{group_by_date} && $opts->{group_by_date} ne 'none') {
        unshift @group_by, 'date';
    } else {
        $opts->{order_by} = [grep { $_->{field} ne 'date' } @{$opts->{order_by}}];
    }
    my %group_by_hash = map { $_ => 1 } @group_by;

    $report_opts->{countable_fields_for_order_by} //= [];

    foreach my $order_item ( @{$opts->{order_by}},
                             map { { field => $_ } } map { @$_ } (xminus \@group_by // [], [map { $_->{field} } @{$opts->{order_by}}])
                           ) {
        my ($norm_field, $suf) = _norm_field_by_suf($order_item->{field});

        my $field_opts = _get_field_opts($order_item->{field});
        _log()->die("Incorrect field for ordering: $order_item->{field}") unless $field_opts && $field_opts->{order} &&
                                                                                 !($suf && !$report_opts->{compare_periods} && any { $_ eq $suf } periods_full_suffix_list());

        next unless $group_by_hash{$order_item->{field}} || ($field_opts->{group_dependency} && $group_by_hash{$field_opts->{group_dependency}})
                        || $field_opts->{is_countable};

        push @{$report_opts->{order_by}}, map { {field => $_,
                                                 dir => $order_item->{dir} // 'asc'}
                                              } @{$field_opts->{order}};
        # перед отправкой запроса в БК добавится к countable_fields, если запрашиваются не все поля
        push @{$report_opts->{countable_fields_for_order_by}}, map { (_norm_field_by_suf($_))[0] } @{$field_opts->{order}} if $field_opts->{is_countable};

        foreach my $m (@{$field_opts->{order_mappings} // []}) {
            _add_mapping($report_opts->{mapping}, $m, hash_cut($report_opts, qw/order_ids filters_pre consumer request_cache/));
        }
    }
}

=head2 _add_limit_opts

   Проверяет и добавляет в настройки отчета опции по смещению и лимиту (для разбивки на страницы)

=cut

sub _add_limit_opts {
    my ($report_opts, $opts) = @_;

    $report_opts->{dont_group_and_filter_zeros_for_totals} = 1 if $opts->{partial_total_rows};

    $report_opts->{limits} //= {};
    return unless $opts->{limits};

    foreach (qw/offset limit/) {
        _log()->die("Incorrect value for $_: $opts->{limits}->{$_}") if !is_valid_int($opts->{limits}->{$_} // 0);
    }

    if (defined $opts->{limits}->{limit}) {
        $report_opts->{limits}->{offset} = $opts->{limits}->{offset} // 0;
        $report_opts->{limits}->{limit} = $opts->{limits}->{limit};
        if ($report_opts->{dont_group_and_filter_zeros_for_totals}) {
            $report_opts->{_real_limit} = $report_opts->{limits}->{limit};
            # коэффициент 1.1 введен потому что БК (если упростить) нам не отдает корректное общее количество строк (в тоталах), 
            # и нам надо понимать по полученному массиву строк, превышают ли они лимит
            # для этого, при выгрузке в xls, используется механизм прогноза количества полученных строк, до их парсинга и обработки, после получения очередного чанка статистики
            # и оверхед в 10% именно для того, чтоб этот механизм корректно работал
            $report_opts->{limits}->{limit} = $report_opts->{limits}->{limit} + (int($report_opts->{limits}->{limit} * 0.1) || 1);
        }
    } elsif (defined $opts->{limits}->{offset}) {
        $report_opts->{limits}->{offset} = $opts->{limits}->{offset};
    }
}

=head2 _check_add_winrate_opts

    Проверяет и добавляет в настройки отчета опции про "% полученных заказов"

=cut

sub _check_add_winrate_opts {
    my ($report_opts, $opts) = @_;

    $report_opts->{with_performance_coverage} = $opts->{with_winrate} || $opts->{filters}->{winrate} ? 1 : 0;

    if ($report_opts->{with_performance_coverage}) {
        _log()->die("Missed required groupping by campaign field when with_winrate=1")
                if none { $_ eq 'campaign' } @{$opts->{group_by} // []};
        foreach my $field (@{$opts->{group_by} // []}) {
            my $field_opts = _get_field_opts($field);
            _log()->die("Incorrect field for grouping when with_winrate=1: $field") unless $field_opts->{support_winrate};
        }

        foreach my $field (keys %{$opts->{filters} // {}}) {
            my $field_opts = _get_field_opts($field);
            _log()->die("Incorrect field for filtering when with_winrate=1: $field")
                        unless $field_opts->{support_winrate} || $field_opts->{is_countable}
                            || ($field eq 'client_id' && $opts->{is_mcc_mode});
        }

        my %group_by_hash = map { $_ => 1 } @{$opts->{group_by} // []};
        my %filter_need_group_by = (performance => 'contextcond_orig',
                                    adgroup => 'adgroup');
        for my $fl (keys %filter_need_group_by) {
            if ($opts->{filters} && $opts->{filters}->{$fl} && !$group_by_hash{$filter_need_group_by{$fl}}) {
                _log()->die(sprintf("Missed required groupping by %s field when filter by %s and with_winrate=1",
                                    $filter_need_group_by{$fl}, $fl));
            }
        }
    } elsif ($opts->{filters}->{winrate}) {
        _log()->die("Incorrect field for filtering when with_winrate=0: winrate");
    }

}

=head2 _prepare_tag_filter

    Пробразовывает фильтр по меткам (тексты) в фильтр по GroupExportID

=cut

sub _prepare_tags_filter {
    my ($values, $op, $order_ids) = @_;

    _log()->die(qq/order_ids is not defined/) unless $order_ids;

    my $tag_ids = Tag::get_tag_ids_by_tags_text($values, get_cids(OrderID => $order_ids)) || [];

    return _prepare_tag_ids_filter($tag_ids, $op, $order_ids);
}

=head2 _prepare_tag_ids_filter

    Пробразовывает фильтр по меткам (id-s) в фильтр по GroupExportID

=cut

sub _prepare_tag_ids_filter {
    my ($values, $op, $order_ids) = @_;

    my $groups_tags = Tag::get_groups_tags( tag_id => $values );

    return [keys %$groups_tags], $op;
}

=head2 _prepare_ssp_filter

    Преобразовывает локализованное название виртуальной SSP "Яндекс и РСЯ" в принятое единое название в БК

=cut

sub _prepare_ssp_filter {
    my ($values, $op, $order_ids) = @_;

    return [map { $_ eq iget($yandex_ssp_title{direct}) ? $yandex_ssp_title{bs} : $_  } @$values], $op;
}

=head2 _prepare_goal_id_filter

    Преобразовывает составной цели в id ее последнего шага. Для остальных целей - оставляет старое значение goal_id

=cut

sub _prepare_goal_id_filter {
    my ($values, $op, $order_ids, $client_id) = @_;

    my (%mapping, @goal_ids, @requested_goal_ids);

    if (_is_only_available_goals_for_filter($order_ids, $client_id)) {
        # фильтруем недоступные клиенту цели
        @requested_goal_ids = _get_available_requested_goal_ids($values, $order_ids, $client_id);
    } else {
        @requested_goal_ids = @$values;
    }
    foreach my $goal_id (@requested_goal_ids) {
        my $new_goal_id = Stat::Tools::get_valuable_goal_id($goal_id);
        if ($new_goal_id != $goal_id) {
            $mapping{$new_goal_id} = $goal_id;
        }
        push @goal_ids, $new_goal_id;
    }

    return \@goal_ids, (@goal_ids ? $op : undef), \%mapping;
}

sub _is_only_available_goals_for_filter {
    my ($order_ids, $client_id) = @_;

    my $unavailable_goals_allowed = Client::ClientFeatures::has_unavailable_goals_allowed($client_id)
                                    && _show_stats_by_unavailable_goals();
    return $client_id
            && Client::ClientFeatures::has_goals_only_with_campaign_counters_used($client_id)
            && !$unavailable_goals_allowed
            && !_is_uac_campaign_requested($order_ids);
}

sub _get_available_requested_goal_ids {
    my ($requested_goal_ids, $order_ids, $client_id) = @_;

    my $extra_counter_ids;
    if (Client::ClientFeatures::has_goals_from_all_orgs_allowed($client_id)) {
        my $cids = [ values %{get_orderid2cid(OrderID => $order_ids)} ];
        $extra_counter_ids = MetrikaCounters::get_spav_counter_ids_from_db_by_cids($cids);
    }
    my @external_mobile_goal_ids;
    if (Client::ClientFeatures::has_in_app_mobile_targeting_allowed($client_id)) {
        push @external_mobile_goal_ids, map {$_->{goal_id}} Stat::Tools::available_inapp_mobile_goals($client_id);
    }
    my $is_internal_ad = camp_kind_in(OrderID => $order_ids->[0], 'internal');
    return @{MetrikaCounters::filter_inaccessible_goal_ids($client_id, $requested_goal_ids,
        preserve_goal_ids => [ (keys (%Stat::Const::MOBILE_APP_SPECIAL_GOALS), @external_mobile_goal_ids) ],
        extra_counter_ids => $extra_counter_ids,
        is_internal_ad => $is_internal_ad)};
}

sub _is_uac_campaign_requested {
    my ($order_ids) = @_;

    if (scalar @$order_ids != 1) {
        return 0;
    }

    my $order_id = $order_ids->[0];

    return CampaignTools::is_uac_campaign_by_order_id($order_id);
}

=head2 _prepare_common_dict_filter

    Для фильтров по словарным полям, преобразовует принятые у потребителя идентификаторы к тем что понятны БК
    входные параметры:
        $values - значения по которым надо фильтровать
        $op - eq|ne - операция
        $cdict - подготовленный словарь потребителя
        %O - именованные параметры
            default => 'undefined', значение "по-умолчанию", которое при выборке отдается если в словаре соответствия не нашлось

=cut

sub _prepare_common_dict_filter {
    my ($values, $op, $cdict, %O) = @_;

    my %cdict_by_consumer_name = ();
    for my $k (keys %{$cdict->{values}}) {
        my $consumer_name = $cdict->{need_reverse_translation} ? $cdict->{values}->{$k}->{name} : $k;
        $cdict_by_consumer_name{$consumer_name} = hash_cut $cdict->{values}->{$k}, qw/bs orig_names/;
    }
    my $custom_default;
    if (defined $O{default}) {
        for my $k (keys %cdict_by_consumer_name) {
            if (any { $_ eq $O{default} } @{$cdict_by_consumer_name{$k}->{orig_names}}) {
                $custom_default = $k;
                last;
            }
        }
    }
    if ($custom_default && any { $_ eq $custom_default } @$values) {
        $values = xminus [keys %cdict_by_consumer_name], $values;
        $op = $op eq 'eq' ? 'ne' : 'eq';
    }

    return [map { @{ $cdict_by_consumer_name{$_}->{bs} // [] } } @$values], $op;
}


{
    my %mapping_subs = (
                mContextType_by_ContextTypeSeparateSynonym => \&_map_mContextType_by_ContextTypeSeparateSynonym,
                mContextType_by_ContextTypeHideSynonym => \&_map_mContextType_by_ContextTypeHideSynonym,
                ContextCondType_by_ContextTypeHideSynonym => \&_map_ContextCondType_by_ContextTypeHideSynonym,
                mTargetType_by_TargetType => \&_map_mTargetType_by_TargetType,
                mTypeID_by_TypeID => \&_map_mTypeID_by_TypeID,
                Tag_by_GroupExportID => \&_map_Tag_by_GroupExportID,
                GroupExportIDSort_by_GroupExportID => \&_map_GroupExportIDSort_by_GroupExportID,
                TagSort_by_GroupExportID => \&_map_TagSort_by_GroupExportID,
                mDeviceType_by_DeviceType => \&_map_mDeviceType_by_DeviceType,
                cid_by_OrderID => \&_map_cid_by_OrderID,
                CampSortName_by_OrderID => \&_map_CampSortName_by_OrderID,
                CampType_by_OrderID => \&_map_CampType_by_OrderID,
                mDetailedDeviceType_by_DetailedDeviceType => \&_map_mDetailedDeviceType_by_DetailedDeviceType,
                mConnectionType_by_ConnectionType => \&_map_mConnectionType_by_ConnectionType,
                mBroadmatchType_by_BroadmatchType => \&_map_mBroadmatchType_by_BroadmatchType,
                CoefGoalContextIDSort_by_CoefGoalContextID => \&_map_CoefGoalContextIDSort_by_CoefGoalContextID,
                mClickPlace_by_ClickPlace => \&_map_mClickPlace_by_ClickPlace,
                mGender_by_Gender => \&_map_mGender_by_Gender,
                mTurboPageType_by_TurboPageType => \&_map_mTurboPageType_by_TurboPageType,
                mAge_by_Age => \&_map_mAge_by_Age,
                mBannerImageType_by_BannerImageType => \&_map_mBannerImageType_by_BannerImageType,
                PhraseWBMAndGroupExportID_by_PhraseWBMAndGroupExportID => \&_map_PhraseGroupExportIDAndMD5,
                PhraseWBMAndOrderID_by_PhraseWBMAndOrderID => \&_map_PhraseOrderIDAndMD5,
                SearchQueryAndGroupExportID_by_SearchQueryAndGroupExportID => \&_map_PhraseGroupExportIDAndMD5,
                SearchQueryAndOrderID_by_SearchQueryAndOrderID => \&_map_PhraseOrderIDAndMD5,
                mSearchQueryStatus_by_SearchQueryStatus => \&_map_mPhraseStatus_by_PhraseStatus,
                mPhraseWBMStatus_by_PhraseWBMStatus => \&_map_mPhraseStatus_by_PhraseStatus,
                mMatchType_by_MatchType => \&_map_mMatchType_by_MatchType,
                mCriterionType_by_CriterionType => \&_map_mCriterionType_by_CriterionType,
                mDealExportID_by_DealExportID => \&_map_mDealExportID_by_DealExportID,
                MeaningfulGoalValue_by_OrderIDAndGoalID => \&_map_MeaningfulGoalValue_by_OrderIDAndGoalID,

                mClientID_by_ClientID => \&_map_ProductName_by_ClientID,
                mClientID_by_ClientID_by_login => \&_map_ChiefLogin_by_ClientID,
                mSemanticCorrespondenceType_by_SemanticCorrespondenceType => \&_map_mSemanticCorrespondenceType_by_SemanticCorrespondenceType,
                mPrismCluster_by_PrismCluster => \&_map_mPrismCluster_by_PrismCluster,
            );

    my %mapping_priorities = (  # первым в списке идет более приоритетный маппинг (который нужно оставить)
                mContextType => [qw/ContextTypeSeparateSynonym ContextTypeHideSynonym/],
            );

=head2 _add_mapping

    Добавляет маппинг в хранилище маппингов, по его названию.
    Если маппинг в хранилище уже присутствует - ничего не делает.

=cut

sub _add_mapping {
    my ($mappings, $map_name, $opts) = @_;

    _log()->die(qq/order_ids is not defined/) unless $opts->{order_ids};
    _log()->die("Unsuppored mapping: $map_name") if !$mapping_subs{$map_name};

    my ($map_res, $map_src) = split '_by_', $map_name;

    return if exists $mappings->{$map_res}
                && _detect_mapping_priority($map_res, $mappings->{$map_res}->{by}) >= _detect_mapping_priority($map_res, $map_src);

    my $map_data = $mapping_subs{$map_name}->($opts->{order_ids}, $opts->{filters_pre}, $opts->{consumer}, $opts->{request_cache});

    $mappings->{$map_res} = {by => $map_src,
                             map => $map_data->{map},
                             data_type => $map_data->{data_type} // 'number',
                             defined $map_data->{default} ? (default => $map_data->{default}) : () };

    # из-за особенностей CH на стороне БК
    # нужно чтобы мы в мапингах не посылали 0 в кажестве результата при дефолте отличным от нуля
    if (defined $mappings->{$map_res}->{default} && $mappings->{$map_res}->{default} ne '0') {
        _log()->die("Forbidden mapping result (0): " . $map_name) if any { $_ eq '0'} values %{$mappings->{$map_res}->{map}};
    }
}

=head2 _detect_mapping_priority

    Возвращает приоритет маппинга (более приоритетные маппинги перекрывают менее приоритетные, при одновременном использовании)

=cut

sub _detect_mapping_priority {
    my ($map_res, $map_src) = @_;

    my $priority = 0;
    if ($mapping_priorities{$map_res}) {
        for my $i (0 .. $#{$mapping_priorities{$map_res}}) {
            if ($mapping_priorities{$map_res}->[$i] eq $map_src) {
                $priority = $#{$mapping_priorities{$map_res}} - $i + 1;
                last;
            }
        }
    }
    return $priority;
}

}

=head2 _map_common_dict

    стандартный маппинг для словарных срезов
    входящие параметры:
        $dict_name - название словаря из Stat::Const
        %O - именованные параметры
            consumer => 'api' - потребитель, по-умолчанию 'default'
            default => 'undefined' - значение по умолчанию (один из основных ключей словаря)
            data_type => 'number|string' - тип значений на которые идет маппинг (для корректной сортировки в БК)

=cut

sub _map_common_dict {
    my ($dict_name, %O) = @_;

    my $pdict = Stat::Tools::consumer_dict_by_name($dict_name, $O{consumer});

    my %map = ();
    my $custom_default;
    for my $k (keys %{$pdict->{values}}) {
        if (defined $O{default} && any { $_ eq $O{default} } @{$pdict->{values}->{$k}->{orig_names}}) {
            $custom_default = $k;
            next;
        }
        for my $bs_value (@{$pdict->{values}->{$k}->{bs}}) {
            $map{$bs_value} = $k;
        }
    }

    return {map => \%map,
            (defined $custom_default ? (default   => $custom_default) : ()),
            ($O{data_type}           ? (data_type => $O{data_type}) : ()),
           };
}

=head2 _map_mContextType_by_ContextTypeSeparateSynonym

=cut

sub _map_mContextType_by_ContextTypeSeparateSynonym {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('CONTEXT_TYPES', consumer => $consumer);
}

=head2 _map_mContextType_by_ContextTypeHideSynonym

=cut

sub _map_mContextType_by_ContextTypeHideSynonym {
    return _map_mContextType_by_ContextTypeSeparateSynonym(@_);
}

=head2 _map_mContextType_by_ContextTypeSeparateSynonym

=cut

sub _map_ContextCondType_by_ContextTypeHideSynonym {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('CONTEXT_COND_TYPES', consumer => $consumer);
}

=head2 _map_mTargetType_by_TargetType

=cut

sub _map_mTargetType_by_TargetType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('TARGET_TYPES', default => 'search', consumer => $consumer);
}

=head2 _map_mTypeID_by_TypeID

=cut

sub _map_mTypeID_by_TypeID {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('POSITION_TYPES', default => 'non-prime', consumer => $consumer);
}

=head2 _map_Tag_by_GroupExportID

=cut

sub _map_Tag_by_GroupExportID {
    my ($order_ids, $filters) = @_;

    my $groups_tags = Tag::get_groups_tags( cid => get_cids(OrderID => $order_ids),
                                            ($filters->{GroupExportID} && $filters->{GroupExportID}->{eq} ? (pid => $filters->{GroupExportID}->{eq}) : ()) );
    my $tag_name_by_id = Tag::get_tags_by_tag_ids( [uniq map { @$_ } values %$groups_tags] );

    my %tag_id_by_name = ();
    while (my ($id, $name) = each %$tag_name_by_id) {
        my $name_lc = lc($name);
        push @{$tag_id_by_name{$name_lc}}, $id;
    }

    my %tag_id_map = ();
    for my $uniq_tag_name_ids ( grep { @$_ > 1 } values %tag_id_by_name) {
        @$uniq_tag_name_ids = sort @$uniq_tag_name_ids;
        my $main_tag_id = shift @$uniq_tag_name_ids;
        for my $tag_id (@$uniq_tag_name_ids) {
            $tag_id_map{$tag_id} = $main_tag_id;
        }
    }


    return { map => { map { $_ => join(' ', sort map { $tag_id_map{$_} // $_ } @{$groups_tags->{$_}}) } grep { @{$groups_tags->{$_}} } keys %$groups_tags },
             default => '',
             data_type => 'string' };
}

=head2 _map_TagSort_by_GroupExportID

=cut

sub _map_TagSort_by_GroupExportID {
    my ($order_ids, $filters) = @_;

    my $groups_tags = Tag::get_groups_tags( cid => get_cids(OrderID => $order_ids),
                                            ($filters->{GroupExportID} && $filters->{GroupExportID}->{eq} ? (pid => $filters->{GroupExportID}->{eq}) : ()) );
    my $names = Tag::get_tags_by_tag_ids([uniq map {@$_} values %$groups_tags]);

    my $groups_tag_names = {};
    foreach my $pid (grep { @{$groups_tags->{$_}} } keys %$groups_tags) {
        $groups_tag_names->{$pid} = join ' ', sort grep {length($_)} map { lc($names->{$_} // '') } @{$groups_tags->{$pid}};
    }

    my $map = {};
    my $i = 1;
    my $prev_tag_name = undef;
    foreach my $pid (sort { $groups_tag_names->{$a} cmp $groups_tag_names->{$b} } keys %$groups_tag_names) {
        if (!defined $prev_tag_name || $groups_tag_names->{$pid} ne $prev_tag_name) {
            $i++;
            $prev_tag_name = $groups_tag_names->{$pid};
        }
        $map->{$pid} = $i;
    }
    return {map => $map,
            default => 1};
}

=head2 _map_mDeviceType_by_DeviceType

=cut

sub _map_mDeviceType_by_DeviceType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('DEVICE_TYPES', default => 'desktop', consumer => $consumer);
}

=head2 _map_mDetailedDeviceType_by_DetailedDeviceType

=cut

sub _map_mDetailedDeviceType_by_DetailedDeviceType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('DETAILED_DEVICE_TYPES', default => 'other', consumer => $consumer);
}

=head2 _map_mConnectionType_by_ConnectionType

=cut

sub _map_mConnectionType_by_ConnectionType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('CONNECTION_TYPES', default => 'undefined', consumer => $consumer);
}

=head2 _map_mSemanticCorrespondenceType_by_SemanticCorrespondenceType

=cut

sub _map_mSemanticCorrespondenceType_by_SemanticCorrespondenceType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('TARGETING_CATEGORIES', default => 'undefined', consumer => $consumer);
}

=head2 _map_mPrismCluster_by_PrismCluster

=cut

sub _map_mPrismCluster_by_PrismCluster {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('PRISMA_INCOME_GRADES', default => 'undefined', consumer => $consumer);
}

=head2 _map_mClickPlace_by_ClickPlace

=cut

sub _map_mClickPlace_by_ClickPlace {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('CLICK_PLACES', default => 'undefined', consumer => $consumer);
}

=head2 _map_mGender_by_Gender

=cut

sub _map_mGender_by_Gender {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('GENDERS', default => 'undefined', consumer => $consumer);
}

=head2 _map_mAge_by_Age

=cut

sub _map_mAge_by_Age {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('AGES', default => 'undefined', consumer => $consumer);
}

=head2 mMatchType_by_MatchType

=cut

sub _map_mMatchType_by_MatchType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('MATCH_TYPES', default => 'none', consumer => $consumer);
}

=head2 mCriterionType_by_CriterionType

=cut

sub _map_mCriterionType_by_CriterionType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('CRITERION_TYPES', default => 'undefined', consumer => $consumer);
}

=head2 _map_mTurboPageType_by_TurboPageType

=cut

sub _map_mTurboPageType_by_TurboPageType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('BS_TURBO_PAGE_TYPES', default => 'none', consumer => $consumer);
}

=head2 _map_cid_by_OrderID

=cut

sub _map_cid_by_OrderID {
    my ($order_ids) = @_;

    my $map = get_hash_sql(PPC(OrderID => $order_ids), ["SELECT c.OrderID, IFNULL(sc.master_cid, c.cid) AS cid
                                                         FROM campaigns c
                                                           LEFT JOIN subcampaigns sc ON c.cid = sc.cid",
                                                         WHERE => { 'c.OrderID' => SHARD_IDS }]);

    return { map => $map, default => 0 };
}

=head2 _map_CampSortName_by_OrderID

=cut

sub _map_CampSortName_by_OrderID {
    my ($order_ids) = @_;

    my $order_ids_sorted = get_one_column_sql(PPC(OrderID => $order_ids), ["SELECT c.OrderID
                                                                            FROM campaigns c
                                                                              LEFT JOIN subcampaigns sc ON c.cid = sc.cid",
                                                                            WHERE => {
                                                                              'c.OrderID' => SHARD_IDS,
                                                                              'sc.master_cid__is_null' => 1 },
                                                                            'ORDER BY' => 'c.name']);
    my $map = {};
    my $i = 1;
    foreach my $OrderID (@$order_ids_sorted) {
        $map->{$OrderID} = $i++;
    }

    my $master_order_id_by_order_id = get_hash_sql(PPC(OrderID => $order_ids), ["SELECT c.OrderID, c_m.OrderID
                                                                                 FROM campaigns c
                                                                                   JOIN subcampaigns sc ON c.cid = sc.cid
                                                                                   JOIN campaigns c_m ON sc.master_cid = c_m.cid",
                                                                                 WHERE => { 'c.OrderID' => SHARD_IDS }]);
    foreach my $OrderID (keys %$master_order_id_by_order_id) {
        $map->{$OrderID} = $map->{$master_order_id_by_order_id->{$OrderID}};
    }

    return {map => $map,
            default => $i};
}

=head2 _map_GroupExportIDSort_by_GroupExportID

=cut

sub _map_GroupExportIDSort_by_GroupExportID {
    my ($order_ids) = @_;

    my $orders_adgroups = Stat::Tools::get_adgroups_info(cid => get_cids(OrderID => $order_ids));
    my $map = {};
    my $i = 0;
    my $prev_adgroup_name = '';
    foreach my $adgroup (sort { $a->{group_name} cmp $b->{group_name} } grep { $_->{group_name} } values %$orders_adgroups) {
        if ($prev_adgroup_name ne $adgroup->{group_name}) {
            $prev_adgroup_name = $adgroup->{group_name};
            $i++;
        }
        $map->{$adgroup->{pid}} = $i;
    }
    return {map => $map,
            default => $i};
}

=head2 _map_CampType_by_OrderID

=cut

sub _map_CampType_by_OrderID {
    my ($order_ids, $filters, $consumer) = @_;

    my $reverse_cdict = Stat::Tools::reverse_by_orig_names_dict( Stat::Tools::consumer_dict_by_name('CAMP_TYPES', $consumer) );

    my $default_type = 'text';
    my $db2consumer_type_sql_map = sql_case('type', $reverse_cdict, default__dont_quote => 'type');
    my $map = get_hash_sql(PPC(OrderID => $order_ids), ["select OrderID, $db2consumer_type_sql_map type from campaigns",
                                                          where => {OrderID => SHARD_IDS,
                                                                    type__ne => $default_type}]);

    my $master_order_id_by_order_id = get_hash_sql(PPC(OrderID => $order_ids), ["SELECT c.OrderID, c_m.OrderID
                                                                                 FROM campaigns c
                                                                                   JOIN subcampaigns sc ON c.cid = sc.cid
                                                                                   JOIN campaigns c_m ON sc.master_cid = c_m.cid",
                                                                                 WHERE => { 'c.OrderID' => SHARD_IDS }]);
    foreach my $OrderID (keys %$master_order_id_by_order_id) {
        $map->{$OrderID} = $map->{$master_order_id_by_order_id->{$OrderID}};
    }

    return {
        map => $map,
        default => $reverse_cdict->{$default_type} // $default_type,
        data_type => 'string'
    };
}

=head2 _map_mBroadmatchType_by_BroadmatchType

=cut

sub _map_mBroadmatchType_by_BroadmatchType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('BROADMATCH_TYPES', default => 'undefined', consumer => $consumer
                            , data_type => 'string' # костыль для обхода вероятного бага со стороны БК
                            );
}

=head2 _map_mPhraseStatus_by_PhraseStatus

=cut

sub _map_mPhraseStatus_by_PhraseStatus {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('BS_PHRASE_STATUSES', default => 'none', consumer => $consumer);
}

=head2 _map_CoefGoalContextIDSort_by_CoefGoalContextID

=cut

sub _map_CoefGoalContextIDSort_by_CoefGoalContextID {
    my ($order_ids) = @_;

    my $client_uids = get_uids(OrderID => $order_ids);
    my @retargetings;
    for my $uid (@$client_uids) {
        push @retargetings, values %{Retargeting::get_retargeting_conditions(uid => $uid, short => 1)};
    }

    my $map = {};
    my $i = 2;
    foreach my $ret (sort { $a->{condition_name} cmp $b->{condition_name} } @retargetings) {
        $map->{$ret->{ret_cond_id}} = $i++;
    }
    return {map => $map,
            default => 1};
}

sub _added_phrases_map {
    my ($order_ids, %O) = @_;

    if (! exists $O{cache}->{added_words_cache}) {
        $O{cache}->{added_words_cache} = get_phrases_from_cache_by_key(OrderID => $order_ids);
    }
    my $db_cached_data = $O{cache}->{added_words_cache};

    my %map;
    for my $order_id (keys %$db_cached_data) {
        for my $pid (keys %{$db_cached_data->{$order_id}}) {
            next if (($O{key} eq 'pid' && !$pid) || ($O{key} eq 'OrderID' && $pid));

            for my $type (keys %{ $db_cached_data->{$order_id}->{$pid} || {} }) {
                my $typed_add_status = $type eq 'minus' ? 1 : 2;
                for my $add_status (keys %{ $db_cached_data->{$order_id}->{$pid}->{$type} || {} }) {
                    for my $stored_hash (@{$db_cached_data->{$order_id}->{$pid}->{$type}->{$add_status} || []}) {
                        my $bs_key = $O{key} eq 'OrderID' ? $order_id : $pid;
                        # Для идентификации используем halfMD5 конкантенации id и хеша фразы
                        # https://st.yandex-team.ru/DIRECT-58934#1480430106000
                        my $bs_md5 = half_md5_hash(Digest::MD5::md5($bs_key . ',' . $stored_hash));
                        # Про статусы см. https://st.yandex-team.ru/BSDEV-58859#1485454353000
                        $map{$bs_md5} = $add_status ? $typed_add_status : 0;
                    }
                }
            }
        }
    }

    return {
        map => \%map,
        data_type => 'number',
    }
}

sub _map_PhraseOrderIDAndMD5 {
    my ($order_ids, $filters, $consumer, $cache) = @_;

    return _added_phrases_map($order_ids, key => 'OrderID', cache => $cache);
}

sub _map_PhraseGroupExportIDAndMD5 {
    my ($order_ids, $filters, $consumer, $cache) = @_;

    return _added_phrases_map($order_ids, key => 'pid', cache => $cache);
}

=head2 _map_mBannerImageType_by_BannerImageType

=cut

sub _map_mBannerImageType_by_BannerImageType {
    my ($order_ids, $filters, $consumer) = @_;

    return _map_common_dict('BANNER_IMAGE_TYPES', consumer => $consumer, data_type => 'string');
}

=head2 _map_mDealExportID_by_DealExportID

=cut

sub _map_mDealExportID_by_DealExportID {
    my ($order_ids) = @_;

    my $deals = CampaignTools::get_deals_by_cids(get_cids(OrderID => $order_ids));
    my $deals_hash = {map { $_->{deal_id} => $_->{name} } @$deals};
    my $map = {};
    my $i = 1;
    foreach my $deal_id (sort { $deals_hash->{$a} cmp $deals_hash->{$b} } keys %$deals_hash){
        $map->{$deal_id} = $i++;
    }
    return {map => $map,
        default => $i};
}

=head2 _map_ProductName_by_ClientID

=cut

sub _map_ProductName_by_ClientID {
    my ($order_ids) = @_;

    my $product_name_by_client_id = get_hash_sql(PPC(OrderID => $order_ids), [
        'SELECT ClientID, product_name',
        "FROM campaigns",
        'JOIN internal_ad_products USING (ClientID)',
        WHERE => {OrderID => SHARD_IDS}
    ]);
    my $map = {};
    my $i = 1;
    foreach my $client_id (sort { $product_name_by_client_id->{$a} cmp $product_name_by_client_id->{$b} } keys %$product_name_by_client_id){
        $map->{$client_id} = $i++;
    }

    return {map => $map,
        default => $i};
}

=head2 _map_ChiefLogin_by_ClientID

=cut

sub _map_ChiefLogin_by_ClientID {
    my (undef, $filters) = @_;
    my $client_ids = $filters->{ClientID}{eq} // [];

    my $chief_login_by_client_id = get_hash_sql(PPC(ClientID => $client_ids), [
        'SELECT ClientID, login',
        "FROM users",
        WHERE => {ClientID => SHARD_IDS, rep_type => $REP_CHIEF}
    ]);

    return {map => $chief_login_by_client_id, default => '', data_type => 'string'};
}

=head2 _map_MeaningfulGoalValue_by_OrderIDAndGoalID

    Для передачи в БК маппим в виде:

    "MeaningfulGoalValue": {
      "map": { "OrderID,GoalID": "Value", "OrderID,GoalID": "Value" },
      "by": "OrderIDAndGoalID",
      "data_type": "number",
      "default": "0"
    }

 	'goal_ids' на вход получаем в виде:

	{
	          'MultiGoalsAttributionType' => {
	                                           'eq' => [
	                                                     '2'
	                                                   ]
	                                         },
	          'MultiGoalsID' => {
	                              'eq' => [
	                                        '34141983',
	                                        '34141989'
	                                      ]
	                            }
	        };

=cut

sub _map_MeaningfulGoalValue_by_OrderIDAndGoalID {
    my ($order_ids, $goal_ids) = @_;
    my $multi_goal_ids;
    if (exists $goal_ids->{MultiGoalsID}){
        $multi_goal_ids = $goal_ids->{MultiGoalsID}->{eq};
    } elsif (exists $goal_ids->{GoalIDList}){         # https://st.yandex-team.ru/DIRECT-108651
        $multi_goal_ids = $goal_ids->{GoalIDList}->{eq};
    }
    my $co_meaningful_goals;
    my %map = ();
    for my $order_id (@$order_ids) {
        $co_meaningful_goals = get_one_field_sql(PPC(OrderID => $order_id),
            ["SELECT co.meaningful_goals FROM campaigns c JOIN camp_options co on co.cid = c.cid",
                WHERE => { "c.OrderID" => $order_id}]);

            if (defined $co_meaningful_goals){
            my @meaningful_goal_values = @{decode_json($co_meaningful_goals)};
                for my $meaningful_goal (@meaningful_goal_values) {
                    if (defined $meaningful_goal->{is_metrika_source_of_value}
                        && $meaningful_goal->{is_metrika_source_of_value}) {
                        next;
                    }
                    for my $goal_id (@$multi_goal_ids) {
                        if ($meaningful_goal->{goal_id} eq $goal_id) {
                            $map{join(",", ($order_id, $goal_id))} = $meaningful_goal->{value} * 1_000_000;
                        }
                    }
                }
            }
    }
    return {map => \%map,
        data_type => 'number',
        default => 0};
}


=head2 _get_field_opts

    отдает настройки по группировке/фильтрации/сортировке конкретного поля
    при этом все единичные значения преобразовывая в массивы

=cut

sub _get_field_opts {
    my $field = shift;
    ($field, my $suf) = _norm_field_by_suf($field);

    return undef unless $field && $fields_opt{$field};

    my $coderefs = hash_grep { ref($_) eq 'CODE' } $fields_opt{$field};
    my $field_opts = hash_merge yclone(hash_grep { ref($_) ne 'CODE' } $fields_opt{$field}), $coderefs;

    foreach my $opt (qw/group group_mappings filter_op order order_mappings/) {
        unless (defined $field_opts->{$opt}) {
            delete $field_opts->{$opt};
            next;
        }
        $field_opts->{$opt} = [$field_opts->{$opt}] unless ref $field_opts->{$opt};
        delete $field_opts->{$opt} unless @{$field_opts->{$opt}};
    }
    foreach my $opt (qw/filter/) {
        delete $field_opts->{$opt} unless $field_opts->{$opt};
    }
    $field_opts->{filter_type} //= 'pre' if $field_opts->{filter};
    
    # денормализуем при надобности названия полей в БК
    if ($suf) {
        my $bs_suf = $suffix_by_targettype_direct2bs{$suf} // $suf;
        @{$field_opts->{order}} = map { $_.$bs_suf } @{$field_opts->{order}} if $field_opts->{order};
        $field_opts->{filter} .= $bs_suf if $field_opts->{filter};
    }

    return $field_opts;
}

=head2 _check_filter_empty_result

    Возвращает 1, если единственные возможный результат с указанными фильтрами - отсутствие статистики

=cut

sub _check_filter_empty_result {
    my ($filters) = @_;

    foreach my $filter_key (keys %$filters) {
        next if $filter_key eq 'AvailableGoalIDList';
        my $f = $filters->{$filter_key};
        foreach my $op (keys %$f) {
            if ($op eq 'eq' &&
                (!defined $f->{$op} || (ref $f->{$op} eq 'ARRAY' && !@{$f->{$op}})) ) {
                return 1;
            }
        }
    }
}

=head2  _get_empty_result

    Эмулирует пустой результат от БК (нужно для случаев когда запрошена статистика по несуществующим баннерам, группам и т.п.)

=cut

sub _get_empty_result {
    my $report_opts = shift;
    # FIXME: Итератор, а не ссылка на строку
    my $bs_result =  to_json({
                                status => 0,
                                stat_time => now()->strftime('%Y%m%d%H%M%S'),
                                header => [_get_stat_required_fields($report_opts)],
                                data => [],
                                totals => { map { $_ => 0 } _get_stat_required_fields($report_opts)},
                                total_rows => 0
                            });
    my $fake_body_iterator = Stat::StreamExtended::FakeHttpBodyIterator->new(
        log => _log(),
        fake_response_body => \$bs_result,
    );
    my $stat = eval { _bs_stream_content_parse_iterator($fake_body_iterator, $report_opts) };
    _log()->die($@) if $@;

    return $stat;
}

=head2 _get_dict_fields_to_map

    возвращает хеш для обратного маппинга словарных полей, пришедших из БК

=cut

sub _get_dict_fields_to_map {
    my ($consumer) = @_;

    my $dict_fields_to_map = {};
    for my $f (keys %fields_opt) {
        next unless $fields_opt{$f}->{common_dict_name};

        my $pdict = Stat::Tools::consumer_dict_by_name($fields_opt{$f}->{common_dict_name}, $consumer);
        my $reverse_map;
        my $default_custom_value;
        if ($fields_opt{$f}->{common_dict_default}) {
            for my $k (keys %{$pdict->{values}}) {
                if (any { $_ eq $fields_opt{$f}->{common_dict_default} } @{$pdict->{values}->{$k}->{orig_names}}) {
                    $default_custom_value = $pdict->{need_reverse_translation} ? $pdict->{values}->{$k}->{name} : $k;
                    last;
                }
            }
        }

        if ($fields_opt{$f}->{group_mappings} && $pdict->{need_reverse_translation}) {
            $reverse_map = { map { $_ => {name => $pdict->{values}->{$_}->{name}, bs => ($pdict->{values}->{$_}->{bs} // [])->[0]} } keys %{$pdict->{values}} };
        } elsif (!$fields_opt{$f}->{group_mappings}) {
            $reverse_map = {};
            for my $k (keys %{$pdict->{values}}) {
                for my $bs_value (@{$pdict->{values}->{$k}->{bs}}) {
                    $reverse_map->{$bs_value} = {name => $pdict->{need_reverse_translation} ? $pdict->{values}->{$k}->{name} : $k,
                                                 bs => $bs_value};
                }
            }
        }

        $dict_fields_to_map->{$f} = {'map' => $reverse_map,
                                     'default' => $default_custom_value} if $reverse_map;
    }
    return $dict_fields_to_map;
}

sub _get_all_countable_fields {
    return sort grep {$fields_opt{$_}->{is_countable}} keys %fields_opt;
}

=head2 _show_stats_by_unavailable_goals

    Показывать статистику по недоступным целям в МОЛ/МОК

=cut

sub _show_stats_by_unavailable_goals {
    state $show_stats_by_unavailable_goals = Property->new('show_stats_by_unavailable_goals_in_mol');
    return $show_stats_by_unavailable_goals->get(60) ? 1 : 0;
}

=head2 _log

    возвращает ранее созданный объект Yandex::Log
    если еще не создавался - создает новый

=cut

sub _log {
    state $log;
    if (!$log) {
        $log = Yandex::Log::Messages->new();
        $log->msg_prefix('stat_stream_extended');
    }
    return $log;
}

=head2 _print_log

=cut

sub _print_log {
    _log()->out(@_);
}

=head2 _debug

=cut

sub _debug
{
    return unless $DEBUG;
    _print_log(@_);
}

1;
