package Stat::StreamExtended::FakeHttpBodyIterator;

=pod

    $Id$

=head1 NAME

    Stat::StreamExtended::FakeHttpBodyIterator

=head1 DESCRIPTION

    Класс, поддерживающий интерфейс Stat::StreamExtended::HttpBodyIterator, но имеющий одно важное отличие от него:
    этот класс не выполняет никаких HTTP-запросов, а отдает потребителю по частям строку, с которой он был  инициализирован.
    Это позволяет подкладывать парсеру ответа от БК данные для отладки/построения пустой статистики.

=cut

use Direct::Modern;
use utf8;

use IO::Scalar;

# размер чанка по-умолчанию
my $DEFAULT_CHUNK_SIZE = 1024 * 1024;

=head2 new(fake_response_body => \$json_str, chunk_size => 1024)

    Конструктор класса
    Возможные параметры
        fake_response_body - строка с json-ом, scalarref (обязателен)
        chunk_size - размер чанков, который будет возвращать итератор

=cut

sub new {
    my ($class, %params) = @_;

    my $self = {
        log => $params{log},
        fake_response_body => $params{fake_response_body},
        chunk_size => $params{chunk_size} // $DEFAULT_CHUNK_SIZE,
    };

    die "log is required parameter" unless $self->{log};

    $self->{log}->die("fake_response_body is required and should be a string ref")
        unless defined $self->{fake_response_body} && ref $self->{fake_response_body} eq 'SCALAR';

    bless $self, $class;
    return $self;
}

=head2 set_chunk_size($size)

    Установить новый размер чанка данных, возвращаемых методом next_chunk итератора

=cut

sub set_chunk_size {
    my ($self, $size) = @_;

    $self->{chunk_size} = $size;
}

=head2 next_chunk()

    Получить очередную порцию данных из строки с ответом сервера. Возвращаемое значение - ссылка на строку

=cut

sub next_chunk {
    my ($self) = @_;

    my $buf;
    if (defined $self->{fake_response_body}) {
        use bytes;
        unless ($self->{_source_json_io}) {
            $self->{_source_json_io} = new IO::Scalar($self->{fake_response_body});
            $self->{_source_json_bytes_length} = length(${$self->{fake_response_body}});
        }
        $self->{_source_json_io}->read($buf, $self->{chunk_size});
        # IO::Scalar->read похоже отдает не кол-во байт, а все же кол-во прочитанных символов
        $self->{_source_json_bytes_read} += length($buf) if defined $buf;
    }
    $buf //= '';

    if ($buf eq '') {
        delete $self->{_source_json_io};
        ${$self->{fake_response_body}} = undef;
        $self->{fake_response_body} = undef;
        $self->{log}->out({bs_size => ($self->content_length // 0)});
    }
    return \$buf;
}

=head2 content_length()

    Получить размер в байтах всего HTTP ответа

=cut

sub content_length {
    my ($self) = @_;

    return $self->{_source_json_bytes_length};
}

=head2 bytes_read()

    Получить размер в байтах данных, уже отданных итератором

=cut

sub bytes_read {
    my ($self) = @_;

    return $self->{_source_json_bytes_read};
}

1;
