package Stat::StreamExtended::HttpBodyIterator;

=pod

    $Id$

=head1 NAME

    Stat::StreamExtended::HttpBodyIterator

=head1 DESCRIPTION

    Класс, выполняющий запрос к ручке статистики БК и отдающий по частям ответ на этот запрос. Особенностью является то,
    что запрос читается асинхронно, то есть лишние и использованные данные не хранятся в памяти.

    Принцип работы такой:

    1. Когда потребитель запрашивает первый чанк с данными с помощью next_chunk, итератор запускает eventloop с помощью
        AnyEvent->condvar, отправляет HTTP-запрос и дожидается ответа.
    2. С помощью (иногда неоднократного) вызова _read_chunked_response в буфер считываются данные, равные или
        не сильно превышающие по размеру заданный размер чанка. О том как работает _read_chunked_response можно прочитать в
        описании этой функции и комментариях внутри.
    3. Существующий на текущий момент condvar сообщает о получении нужного количества данных, сообщение остнавливает eventloop
    4. Данные отдаются потребителю, буфер итератора очищается.
    5. При следующем вызове next_chunk, производится проверка, прочитан ли весь HTTP-запрос. Если запрос прочитан,
        возвращается ссылка на пустую строку, что означает что итератор больше не вернет никаких новых данных. Если запрос
        еще не прочитан, итератор запускает eventloop с помощью AnyEvent->condvar и продолжает работу с пункта 2.

    В итераторе может несколько раз запускаться и останавливаться eventloop. Это вызвано тем, что потребитель итератора не асинхронный.

=cut

use Direct::Modern;
use utf8;

use AnyEvent;
use AnyEvent::HTTP qw/http_request/;

use IO::Scalar;

# Размер чанка по-умолчанию
my $DEFAULT_CHUNK_SIZE = 1024 * 1024;
# Максимально допустимое превышение размера чанка
my $MAX_SIZE_MULTIPLIER = 2;

our $DEFAULT_CONNECT_TIMEOUT = 5; # дефолтное значение таймаута на установку соединения с сервером
our $DEFAULT_CONNECT_RETRY = 3; # дефолтное количество повторений попытки установить соединение с сервером

=head2 new(url => $url, request_body => $request_body, log => $log)

    Конструктор класса
    Возможные параметры
        url - Адрес ручки БК, к которой мы обращаемся. (обязательно)
        request_body - Строка, тело запроса. (обязательно)
        log - Объект логгера. (обязательно)
        timeout - Значение таимаута запроса. (обязательно)
        chunk_size - Размер чанка данных, отдаваемого итератором. Точный размер чанка не гарантируется, фактически он может как превышать
            заданное число в $MAX_SIZE_MULTIPLIER так и быть меньше его (последнее только в случае, если это последняя часть HTTP-ответа).
        use_simple_request_handling - использовать простой, но более затратный по памяти способ получения данных. По умолчанию не используется.
            Стоит использовать его, если возникнут проблемы с основным способом получения данных по чанкам или если захочется
            использовать HttpBodyIterator для запроса, в котором мы не уверены в том, что transfer-encoding будет chunked.
            Также стоит учитывать, что при несовпадении ожидаемого transfer-encoding с полученным, обработка фолбекается на упрощенную,
            но только после повторной попытки.

=cut

sub new {
    my ($class, %params) = @_;

    my $self = {
        log => $params{log},
        chunk_size => $params{chunk_size} || $DEFAULT_CHUNK_SIZE,
        use_simple_request_handling => $params{use_simple_request_handling} // 0,
        profile => $params{profile},
        _initiated => 0,
        _request_finished => 0,
        _body_bytes_read => 0,
        _body_bytes_expected => 0,
        _need_simple_retry => 0,
        buffer => '',
        _error => '',
        connect_timeout => $DEFAULT_CONNECT_TIMEOUT,
        connect_retry => $DEFAULT_CONNECT_RETRY,
    };

    die "log is required parameter" unless $self->{log};

    for my $f (qw/url request_body timeout/) {
        if ($params{$f}) {
            $self->{$f} = $params{$f};
        } else {
            $self->{log}->die("$f is required parameter");
        }
    }

    # В запросе отправляем байты, а не utf-строку
    utf8::encode($self->{request_body});
    bless $self, $class;
    return $self;
}

sub _do_request_simple {
    my ($self) = @_;

    # Опции запроса для AnyEvent::HTTP::http_request
    my %options = (
        timeout => $self->{timeout},
        body => $self->{request_body},
        headers => {
            'Content-Type' => 'application/json',
        },
        on_body => sub {
            my ($part, $headers) = @_;

            use bytes;
            if (!$self->{_headers}) {
                $self->{_headers} = $headers;
                if ($headers->{Status} != 200) {
                    if ($self->_need_retry_connection($headers->{Status})) { # определяем, что получили таймаут установки соединения и надо повторить попытку
                        $self->_log_retry();
                        $self->_do_request_simple();
                        return;
                    }
                    utf8::decode($headers->{Reason});
                    $self->_finish_request("$headers->{Status} $headers->{Reason}");
                    return;
                }
            }

            $self->{buffer} .= $part;
            $self->{_body_bytes_expected} += length($part);
            $self->_send_data_cv();
            return 1;
        },
        on_prepare => sub { return $self->{connect_timeout} }, # задаем таймаут на установку соединения с сервером
    );

    http_request(
        POST => $self->{url},
        %options,
        sub {
            my ($body, $headers) = @_;

            # передаём $self->{_error}, чтобы, если обработчик on_body установил это вызовом _finish_request, здесь не перезатереть
            # см. тж. https://st.yandex-team.ru/DIRECT-91729
            $self->_finish_request($self->{_error});
        },
    );
}

=head3 _read_chunked_response()

    Функция - модифицированный аналог обработчика chunked ответа из AnyEvent::HTTP::http_request.
    Мы, к сожалению, не можем использовать стандартный обработчик, потому что когда используется Transfer-Encoding: chunked,
    он возвращает ответ по частям равным размеру чанка, который устанавливает ручка статистики. Фактически этот размер варьируется
    от нескольких килобайт до нескольких сотен мегабайт. Во втором случае это приводит к значительному переиспользованию памяти
    при потоковом чтении JSON и замедлению работы, что неприемлемо в нашем случае.

    Моифицированный обработчик отличается от стандартного только тем, что он дополнительно бьет на части слишком большие чанки и
    не использует анонимные функции.

=cut

sub _read_chunked_response {
    my ($handle, $line) = @_;

    if ($line =~ m/^([0-9a-fA-F]+)$/) {
        # Получаем число в начале чанка, оно в шестнадцатеричной системе, может быть нулем
        # Затем считываем положенное количество данных
        my $len = hex $1;
        $handle->{_it}->{_body_bytes_expected} += $len;

        if ($len) {
            if ($len <= $handle->{_it}->{chunk_size}) {
                # Если чанк ответа меньше чанка итератора, читаем его весь и запускаем $read_chunked заново для
                # получения следующей порции ответа
                $handle->push_read(chunk => $len, \&_read_full_chunk);
            } else {
                # Если чанк ответа больше чанка итератора, считываем его кусочками в несколько итераций
                # и запускаем $read_chunked заново для получения следующей порции ответа
                $handle->{_ch_len} = $len;
                $handle->push_read(chunk => $handle->{_it}->{chunk_size}, \&_read_chunk_part);
            }
        } else {
            # Ноль в длине чанка означает, что данные закончились.
            $handle->{_it}->_finish_request();
        }
    } else {
        # Если мы не нашли числа в начале чанка, значит есть два варианта
        # Первый - тело ответа повреждено и нам пора завершаться с ошибкой
        # Второй - мы получили пустую строку, такое бывает, попробуем еще раз
        if (length($line)) {
            $handle->{_it}->_finish_request("bad transfer encoding");
        } else {
            $handle->push_read(line => \&_read_chunked_response);
        }
    }
}

=head3 _read_full_chunk($handle, $data)

    Сохранить чанк данных целиком в буфер итератора. После этого поставить обработчику запроса задачу прочитать метаданные о
    следующем чанке (_read_chunked_response).

    Если после сохранения данных количество байт в буфере превысило необходимый размер чанка, то перед постановкой следующей задачи
    выполняется остановка eventloop и отправка сообщения о том, что данные готовы. Дальнейшая работа по чтению HTTP-ответа будет
    произведена только при следующем запуске eventloop.

=cut

sub _read_full_chunk {
    my ($handle, $data) = @_;

    $handle->{_it}->{buffer} .= $data;
    $handle->{_it}->_send_data_cv();
    $handle->push_read(line => \&_read_chunked_response);  # -> начинаем новую итерацию чтения
}

=head3 _read_chunk_part($handle, $data)

    Сохранить полученные данные в буфер итератора. После этого:
        - если чанк HTTP-ответа уже прочитан полностью, поставить обработчику запроса задачу прочитать метаданные о
          следующем чанке HTTP-ответа (_read_chunked_response).
        - если чанк HTTP-ответа еще не прочитан до конца, поставить задачу на чтение следующей его части (_read_chunk_part).

    Если после сохранения данных количество байт в буфере превысило необходимый размер чанка, то перед постановкой следующей задачи
    выполняется остановка eventloop и отправка сообщения о том, что данные готовы. Дальнейшая работа по чтению HTTP-ответа будет
    произведена только при следующем запуске eventloop.

=cut

sub _read_chunk_part {
    my ($handle, $data) = @_;

    $handle->{_it}->{buffer} .= $data;
    $handle->{_ch_len} -= $handle->{_it}->{chunk_size};
    $handle->{_it}->_send_data_cv();
    if ($handle->{_ch_len} <= 0) {
        $handle->push_read(line => \&_read_chunked_response);  # -> начинаем новую итерацию чтения
    } else {
        $handle->push_read(
            chunk => ($handle->{_ch_len} > $handle->{_it}->{chunk_size} ? $handle->{_it}->{chunk_size} : $handle->{_ch_len}),
            \&_read_chunk_part
        );  # -> продолжаем загружать HTTP-чанк
    }
}

=head3 _process_response_error()

=cut

sub _process_response_error {
    my ($hdl, $ftl, $msg) = @_;

    utf8::decode($msg);
    $hdl->{_it}->_finish_request($msg);
}

=head3 _do_request()

    Выполнить HTTP-запрос к БК, получить обработчик запроса и подготовить его для почанкового чтения ответа из БК.

    В подготовке полагаемся на то, что ответ всегда будет в формате Transfer-Encoding: chunked.

=cut

sub _do_request {
    my ($self) = @_;

    # Опции запроса для AnyEvent::HTTP::http_request
    my %options = (
        timeout => $self->{timeout},
        body => $self->{request_body},
        headers => {
            'Content-Type' => 'application/json',
        },
        on_header => sub {
            my ($headers) = @_;

            unless ($headers->{"transfer-encoding"} =~ /\bchunked\b/i) {
                $self->{_need_simple_retry} = 1;
                return 0;
            }
            return 1;
        },
        on_prepare => sub { return $self->{connect_timeout} }, # задаем таймаут на установку соединения с сервером
        want_body_handle => 1,  # Не читать ответ самостоятельно, а передать нам объект обрабочика ответа
    );

    http_request(
        POST => $self->{url},
        %options,
        sub {
            my ($handle, $headers) = @_;

            # Если код ответа не 200, сразу завершаем обработку
            if ($headers->{Status} != 200) {
                if ($self->_need_retry_connection($headers->{Status})) { # определяем, что получили таймаут установки соединения и надо повторить попытку
                    $self->_log_retry();
                    $self->_do_request();
                    return;
                }
                utf8::decode($headers->{Reason});
                $self->_finish_request("$headers->{Status} $headers->{Reason}");
                return;
            }

            # Если не Transfer-Encoding: chunked, тоже завершаем обработку
            my $chunked = $headers->{"transfer-encoding"} =~ /\bchunked\b/i;
            if (!$chunked) {
                $self->_finish_request("unexpected transfer encoding");
            }

            $handle->{_it} = $self;
            $self->{_handle} = $handle;

            # Готовимся завершить обработку на ошибке любой степени фатальности
            $handle->on_error(\&_process_response_error);

            # Если по хедерам все хорошо, значит можем начать читать тело ответа
            # _read_chunked_response - функция, которая читает информацию о том, сколько данных нужно прочитать
            # и ставит задачу на чтение этих данных.
            $handle->push_read(line => \&_read_chunked_response);
        }
    );
    return;
}

=head3 _need_retry_connection()

    Определяем, что получили таймаут установки соединения и нужно повторять попытку

=cut

sub _need_retry_connection {
    my ($self, $status) = @_;

    return $status == 595 && $self->{connect_tries_left} && --$self->{connect_tries_left} >= 0 ? 1 : 0;
}

=head3 _log_retry()

    Пишем в лог информацию о повторении попытки установки соединения

=cut

sub _log_retry {
    my ($self) = @_;

    $self->{log}->out("connect timeout ($self->{connect_timeout}), retry ".($self->{connect_retry} - $self->{connect_tries_left})." of $self->{connect_retry}");

    return;
}


=head3 _send_data_cv()

    Проверить, готовы ли данные к выдаче и, если необходимо, оповестить итератор о том, что данные можно отдавать.

=cut

sub _send_data_cv {
    my ($self) = @_;

    if (defined $self->{_data_cv} && ($self->{_request_finished} || length($self->{buffer}) >= $self->{chunk_size})) {
        $self->{_data_cv}->send();
    }
}

=head3 _finish_request($error)

    Завершить запрос к БК и освободить все ресурсы

    error - Текст ошибки, если она была

=cut

sub _finish_request {
    my ($self, $error) = @_;

    $self->{_request_finished} = 1;
    $self->{_error} = $error;
    if (defined $self->{_handle}) {
        # Если этого не сделать, начнем течь по памяти
        $self->{_handle}->{_it} = undef;
        $self->{_handle}->destroy();
        $self->{_handle} = undef;
    }
    $self->{_init_cv}->send();
    $self->_send_data_cv();
    return;
}

=head3 _init()

    Инициировать процесс загрузки данных из ручки статистики БК

=cut

sub _init {
    my ($self) = @_;

    die "Already initiated" if $self->{_initiated};
    $self->{_initiated} = 1;

    $self->{_init_cv} = AnyEvent->condvar();
    $self->{connect_tries_left} = $self->{connect_retry} // 0;
    if ($self->{use_simple_request_handling}) {
        $self->_do_request_simple();
    } else {
        $self->_do_request();
    }
    return;
}

=head3 _wait_for_data()

    Ожидать новую порцию данных до тех пор, пока она не будет получена

=cut

sub _wait_for_data {
    my ($self) = @_;

    $self->{_data_cv} = AnyEvent->condvar();
    $self->{_data_cv}->recv();
    $self->{_data_cv} = undef;

    if ($self->{_error}) {
        # Если были проблемы с transfer-encoding - сделаем ретрай со стандартным обработчиком
        unless ($self->{_need_simple_retry}) {
            $self->{log}->die("http_request($self->{url}\t$self->{request_body}): $self->{_error}");
        }
        $self->{log}->out("http_request($self->{url}\t$self->{request_body}): $self->{_error}");
        $self->{log}->out("Starting simple retry because of transfer encoding error");

        $self->{_need_simple_retry} = 0;
        $self->{_initiated} = 0;
        $self->{_request_finished} = 0;
        $self->{_error} = '';
        $self->{use_simple_request_handling} = 1;

        $self->_init();
        $self->_wait_for_data();
    }

    return;
}

=head2 set_chunk_size($size)

    Установить новый размер чанка данных, возвращаемых методом next_chunk итератора

=cut

sub set_chunk_size {
    my ($self, $size) = @_;

    $self->{chunk_size} = $size;
}

=head2 next_chunk()

    Получить очередную порцию данных ответа сервера. Возвращаемое значение - ссылка на строку.
    Если метод возвращает ссылку на пустую строку, значит запрос обработан и данных больше не будет

=cut

sub next_chunk {
    my ($self) = @_;
    use bytes;

    if (! $self->{_initiated}) {
        $self->_init();
    };

    my $buf;
    my $buflen = length($self->{buffer});
    if (defined $self->{_buffer_io}) {
        $self->{_buffer_io}->read($buf, $self->{chunk_size});
        if (!defined $buf) {
            # по состоянию на 2019-07-30 кажется, что в эту ветку (и такую же ниже) никогда не попадаем
            # в теории это возможно, если при чтении произошла ошибка (но как можно не прочитать из строки?)
            $self->{_buffer_io} = undef;
        }
        if ($self->{_buffer_io}->eof) {
            # иногда так случается, что буфер вычитали весь, но данные в него были записаны еще не все.
            # текущий чанк данных отдадим ниже, но на следующей итерации провалимся в _wait_for_data
            $self->{_buffer_io} = undef;
            $self->{buffer} = ''
        }
    } elsif ($buflen > $self->{chunk_size} || ! $self->{_request_finished}) {
        if ($buflen < $self->{chunk_size}) {
            $self->_wait_for_data();
        }

        if ($self->{profile}) {
            delete $self->{profile};
        }

        if (length($self->{buffer}) < $self->{chunk_size} * $MAX_SIZE_MULTIPLIER) {
            $buf = $self->{buffer};
            $self->{buffer} = '';
        } else {
            $self->{_buffer_io} = new IO::Scalar(\$self->{buffer});
            $self->{_buffer_io}->read($buf, $self->{chunk_size});
            if (!defined $buf) {
                $self->{_buffer_io} = undef;
            }
        }
    } else {
        if (defined $self->{_init_cv}) {
            $self->{_init_cv}->recv();
            $self->{_init_cv} = undef;
        }

        if ($self->{buffer}) {
            $buf = $self->{buffer};
            $self->{buffer} = '';
        }
    }

    $buf //= '';
    $self->{_body_bytes_read} += length($buf);
    if ($buf eq '') {
        $self->{log}->out({bs_size => ($self->content_length // 0)});
    }

    # Если не сделать этого ПЗ и ДРФ будут не в той кодировке
    utf8::decode($buf);
    return \$buf;
}

=head2 content_length()

    Получить текущую оценку размера в байтах всего HTTP ответа. Поскольку данные отдаются потоково, результат может
    очень сильно отличаться от истины, не стоит полагаться на него в важных расчетах

=cut

sub content_length {
    my ($self) = @_;

    return $self->{_body_bytes_expected};
}

=head2 bytes_read()

    Получить размер в байтах данных, уже отданных итератором

=cut

sub bytes_read {
    my ($self) = @_;

    return $self->{_body_bytes_read};
}

1;
