package Stat::StreamExtended::ParseIterator;

=pod

    $Id$

=head1 NAME

    Stat::StreamExtended::ParseIterator

=head1 DESCRIPTION

    Парсинг json-а полученного от БК "по частям", приведение строк статистики к нужному формату, отдача пачками заданного размера полученных данных
    
    Обязательные методы:
        headers - возвращает накопленные общие данные по всей полученной статистике (все кроме основного массива со строками статистики)
        next_chunk(chunk_size => 123) - возвращает очередную пачку строк статистики, размер пачки можно регулировать параметром chunk_size

=cut

use Direct::Modern;

use JSON::SL;
use List::MoreUtils qw/any uniq each_array/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xminus xisect xflatten/;
use Yandex::Trace;

use Settings;
use Stat::Fields;
use Stat::Tools;
use TextTools;
use PrimitivesIds;
use DirectCache;

# поле MultiGoalsAvgTimeToConversion на фронте не поддержано (DIRECT-97876)
our @MULTI_GOALS_FIELDS = qw/MultiGoalsNum MultiGoalsAvgGoalsCost MultiGoalsROI MultiGoalsCrr MultiGoalsIncome
                             MultiGoalsConversionPct MultiGoalsAvgTimeToConversion MultiGoalsCrrPV
                             MultiGoalsNumPV MultiGoalsAvgGoalsCostPV MultiGoalsROIPV MultiGoalsIncomePV MultiGoalsConversionPctPV MultiGoalsAvgTimeToConversionPV/;

# размер чанка json, для парсинга (байт)
our $JSON_CHUNK_SIZE_BYTE = 2*1024*1024;

# дефолтное количество строк, отдаваемое методом next_chunk итератора
our $ITERATOR_DEFAULT_CHUNK_SIZE = 50_000;

# ключи заголовков, которые мы ожидаем получить от БК
my @header_keys = qw/status error_text stat_time header totals total_rows/;
my @fields_multipied_by_10x6   = Stat::Fields::get_all_money_fields();
my %fields_multipied_by_10x6   = map { $_ => 1 } @fields_multipied_by_10x6;
# все поля в %goals_fields_to_round вычислимые и вообще не должны встречаться в ответе БК
my %goals_fields_to_round   = map { $_ => 1 } qw/agoalroi aconv agoalcost pv_agoalroi pv_aconv pv_agoalcost/;
my @suffixes = qw/_a _b _absdelta _0 _1/;

=head2 new(body_iterator => \$body_iterator, log => Stat::StreamExtended::_log())

    Конструктор класса
    Возможные параметры
        body_iterator - Итератор, возвращающий по частям HTTP-ответ от БК (обязателен)
        log - ссылка на объект Yandex::Log (обязателен)
        four_digits_precision — флаг округления денежных полей до 4 знаков
        stat_required_fields - список обязательных полей в статистике, arrayref (обязателен)
        report_opts - настройки отчета, по которым была получена статистика, hashref (обязателен)
        ссылки на соответствующие функции Stat::StreamExtended (обязательные):
            _bs2direct_field
            _norm_field_by_suf
            _process_row
            _process_row_cache


=cut

sub new {
    my $class = shift;
    my %params = @_;

    my $self = hash_cut \%params, qw/log four_digits_precision/;

    die "log is required parameter" unless $self->{log};

    for my $f (qw/body_iterator stat_required_fields report_opts
                  _bs2direct_field _norm_field_by_suf _process_row _process_row_cache/) {
        if ($params{$f}) {
            $self->{$f} = $params{$f};
        } else {
            $self->{log}->die("$f is required parameter");
        }
    }

    $self->{_direct_cache} = new DirectCache(groups => ['stat_date_format']);
    $self->{body_iterator}->set_chunk_size($JSON_CHUNK_SIZE_BYTE);

    bless $self, $class;
    return $self;
}

=head2 headers

    Возвращает ссылку на хеш с уже полученными/вычисленными заголовками полученной статистики (totals, stat_stream_ts ...)

=cut

sub headers {
    my $self = shift;

    $self->_parse_headers() unless $self->_headers_is_finished;
    return $self->{headers};
}

=head2 _parse_headers

    Вытягивает из json-а хедеры (все перечисленные в @header_keys поля до начала данных)

    про totals:
    парсим структуру полученную из БК
    	{
	          'BonusNoVAT' => 0,
	          'BounceRatio' => 55,
	          'Bounces' => '209',
	          'Clicks' => '1062',
	          'ClicksNonmobile' => '1062',
	          'Cost' => '20885934830.5085',
	          'CostNonmobile' => '20885934830.5085',
	          'FirstPageClicks' => '0',
	          'FirstPageShows' => '0',
	          'FirstPageSumPosClicks' => '0',
	          'FirstPageSumPosShows' => '0',
	          'GoalsIncome' => 0,
	          'GoalsNum' => '13',
	          'GoalsProfit' => '-20885934830.5085',
	          'PrGoodMultiGoal' => undef,
	          'PrGoodMultiGoalCPA' => undef,
	          'PrGoodMultiGoalConversionRate' => undef,
	          'PrGoodMultiGoalPure' => '0.385298062872607',
	          'SessionDepth' => '535',
	          'SessionNum' => '380',
	          'SessionNumLimited' => '380',
	          'Shows' => '3418919',
	          '["MultiGoalsAttributionType","MultiGoalsID","MultiGoalsNum","MultiGoalsAvgGoalsCost"]' => [
	                                                                                                       [
	                                                                                                         '2',
	                                                                                                         '34141983',
	                                                                                                         10,
	                                                                                                         '34312607221.5496'
	                                                                                                       ],
	                                                                                                       [
	                                                                                                         '2',
	                                                                                                         '34141989',
	                                                                                                         3,
	                                                                                                         '6961978276.83616'
	                                                                                                       ]
	                                                                                                     ]
	        };

	        в структуру для директа:

            {
	          'ClicksNonmobile' => '1062',
	          'CostNonmobile' => '20885934830.5085',
	          'PrGoodMultiGoalPure' => '0.385298062872607',
	          'agoalcost_34141983_3' => '34312.61', -- _3 здесь модель атрибуции
	          'agoalcost_34141989_3' => '6961.98',
	          'agoalincome' => '0',
	          'agoalnum' => '13',
	          'agoalnum_34141983_3' => 10,
	          'agoalnum_34141989_3' => 3,
	          'agoals_profit' => '-20885.9348305085',
	          'aprgoodmultigoal' => undef,
	          'aprgoodmultigoal_conv_rate' => undef,
	          'aprgoodmultigoal_cpa' => '0',
	          'aseslen' => '535',
	          'asesnum' => '380',
	          'asesnumlim' => '380',
	          'bonus' => '0',
	          'bounce_ratio' => 55,
	          'bounces' => '209',
	          'clicks' => '1062',
	          'fp_clicks' => '0',
	          'fp_clicks_pos' => '0',
	          'fp_shows' => '0',
	          'fp_shows_pos' => '0',
	          'shows' => '3418919',
	          'stat_date' => '2018-01-03',
	          'stat_date_end' => '2019-01-10',
	          'sum' => '20885.9348305085'
	        };

	        1) нужно смапить названия в директовские (_bs2direct_field)
	        2) отдельно парсится сложный ключ мультицелей из БК '["MultiGoalsAttributionType","MultiGoalsID","MultiGoalsNum","MultiGoalsAvgGoalsCost"]'
	        и его значение в значения понятные директу, например, 'agoalcost_34141989' => '6961.98', где agoalcost - директовское название
	        34141989 - id цели
=cut

sub _parse_headers {
    my $self = shift;
    my $rounding_mode = $self->{four_digits_precision} ? 'four_digits_precision' : '';

    return if $self->_headers_is_finished;

    $self->{headers} = {};
    my $has_error = 0;
    my $p = $self->_json_sl_parser;
    my $goals_mapping = $self->{report_opts}{goals_mapping} // {};
    OUTER_LOOP: while (!$self->_headers_is_finished) {
        while (my $obj = $p->fetch) {
            if ($obj->{JSONPointer} eq '/data/^') {
                $self->_headers_is_finished(1);
                $self->_accumulate_data($obj->{Value});
                last OUTER_LOOP;
            } elsif ($obj->{JSONPointer} eq '/status' && $obj->{Value}) {
                $has_error = 1;
            } elsif ($has_error && $obj->{JSONPointer} eq '/error_text') {
                my $error_msg = $obj->{Value} ? $obj->{Value} : "Unknown error";
                $self->_log()->die("BS response: $error_msg");
            } else {
                my $key = substr $obj->{JSONPointer}, 1;
                $self->{headers}->{$key} = $obj->{Value};

                if ($key eq 'totals') {
                    my $fields_available_for_totals = xminus $self->{stat_required_fields}, \@MULTI_GOALS_FIELDS;
                    my $missed_fields = xminus $fields_available_for_totals, [keys %{$self->{headers}->{totals}}];
                    $self->_log()->die("Missed fields in stat totals: " . join(', ', @$missed_fields)) if @$missed_fields;

                    my @keys_totals = keys %{$self->{headers}->{totals}};
                    my @keys_values = values %{$self->{headers}->{totals}};
                    my @keys_totals_fields;
                    for my $total (@keys_totals) {
                        if ($total =~ /^\[\"(.+)\"\]$/) {
                            my @total = split /","/, $1;
                            my @multi_goals_fields = map { $self->{_bs2direct_field}->($_) } @total;
                            push @keys_totals_fields, \@multi_goals_fields;
                        } else {
                            push @keys_totals_fields, $self->{_bs2direct_field}->($total);
                        }
                    }
                    #этот массив понадобится чуть ниже, чтобы округлить поля с суфиксами _a _b _absdelta _0 _1 DIRECT-92762 DIRECT-93859
                    my @fields_with_suffixes;
                    for my $suf (@suffixes) {
                        push @fields_with_suffixes, map { $_.$suf } @fields_multipied_by_10x6;
                    }
                    my %parsed;
                    my $i = 0;
                    for my $keys_totals_field (@keys_totals_fields) {
                        if (ref $keys_totals_field) {
                            my $multi_goals_values_sets = $keys_values[$i] || [];
                            for my $values (@$multi_goals_values_sets) {
                                my ($attribution_model, $goal_id, @multi_goals_values) = @$values;
                                (undef, undef, my @multi_goals_headers) = @$keys_totals_field;
                                my $ea = each_array(@multi_goals_headers, @multi_goals_values);
                                while (my ($h, $v) = $ea->()) {
                                    if (exists $fields_multipied_by_10x6{$h}) {
                                        # Income и ROI по составным целям могут приходить как null
                                        $parsed{"${h}_${goal_id}_$attribution_model"} = ($v // 0) / 1_000_000;
                                    }
                                    else {
                                        $parsed{"${h}_${goal_id}_$attribution_model"} = $v;
                                    }
                                    if (exists $goals_fields_to_round{$h}) {
                                        $parsed{"${h}_${goal_id}_$attribution_model"} = sprintf("%.2f", round2s($parsed{"${h}_${goal_id}_$attribution_model"}));
                                    }

                                    if (exists $goals_mapping->{$goal_id}) {
                                        $parsed{"${h}_$goals_mapping->{$goal_id}_$attribution_model"} = $parsed{"${h}_${goal_id}_$attribution_model"};
                                    }
                                }
                            }
                        } else {
                            if (exists $fields_multipied_by_10x6{$keys_totals_field}){
                                $parsed{$keys_totals_field} = ($keys_values[$i] // 0)/1_000_000;
                            } else {
                                $parsed{$keys_totals_field} = $keys_values[$i];
                            }
                            if (exists $goals_fields_to_round{$keys_totals_field}) {
                                $parsed{$keys_totals_field} = sprintf("%.2f", round2s($parsed{$keys_totals_field}));
                            }

                            # округлить поля с суфиксами _a _b _absdelta _0 _1 (DIRECT-92762 DIRECT-93859)
                            # поля с суффиксами получаются из полей, домноженных на миллион — это всё денежные поля
                            if (any {$keys_totals_field eq $_} @fields_with_suffixes) {
                                $parsed{$keys_totals_field} = Stat::Tools::sprintf_round(($parsed{$keys_totals_field} // 0)/1_000_000, $rounding_mode);
                            }
                        }
                        $i++;
                    }
                    $parsed{stat_date} = $self->{report_opts}->{date_from};
                    $parsed{stat_date_end} = $self->{report_opts}->{date_to};
                    s/^(\d{4})(\d{2})(\d{2})$/$1-$2-$3/ foreach (grep {defined $_} @parsed{qw/stat_date stat_date_end/});

                    $self->{headers}->{totals} = \%parsed;

                } elsif ($key eq 'header') {
                    my $headers = $self->{headers}->{header};
                    my @flatten_headers = xflatten @$headers;
                    my $missed_fields = xminus $self->{stat_required_fields},  \@flatten_headers;
                    $self->_log()->die("Missed fields in stat data: " . join(', ', @$missed_fields)) if @$missed_fields;

                    for my $header (@{$self->{headers}->{header}}) {
                        if (ref $header) {
                            my @multi_goals_fields = map { $self->{_bs2direct_field}->($_) } @$header;
                            push @{$self->{headers}->{_header_fields}}, \@multi_goals_fields;
                        } else {
                            push @{$self->{headers}->{_header_fields}}, $self->{_bs2direct_field}->($header);
                        }
                    }
                }
            }
        }
        last OUTER_LOOP unless $self->_init_next_chunk_feed;
    }
    # на случай если данные были пустыми
    $self->_headers_is_finished(1);

    $self->{_total_rows_accumulated} //= 0;
}

=head2 _headers_is_finished

=cut
sub _headers_is_finished {
    my ($self, $value) = @_;
    $self->{_headers_is_finished} = $value if defined $value;
    return $self->{_headers_is_finished} ? 1 : 0;
}

=head2 _when_data_is_finished

=cut
sub _when_data_is_finished {
    my $self = shift;

    if ($self->{report_opts}->{without_totals} || $self->{report_opts}->{dont_group_and_filter_zeros_for_totals}) {
        # нужно вручную проверить и применить лимиты к полученным данным, и пересчитать общее количество строк
        $self->_check_request_limits();
        if (!$self->{_total_rows_accumulated}) {
            $self->{headers}->{total_rows} = 0;
        } else {
            $self->{headers}->{total_rows} = ($self->{report_opts}->{limits}->{offset} || 0)
                                                    + $self->{_total_rows_accumulated}
                                                    + ($self->{_rows_limit_exceeded} ? 1 : 0);
        }
    }
}

=head2 next_chunk

    Возвращает очередную пачку строк статистики или undef
    входные параметры:
    %O - именованные параметры
        chunk_size => 1000 - переопределяем размер пачки запрашиваемых данных

=cut

sub next_chunk {
    my ($self, %O) = @_;

    my $profile = Yandex::Trace::new_profile('stat_stream_extended:parse_iterator_next_chunk');
    my $chunk_size = int($O{chunk_size} // 0) || $ITERATOR_DEFAULT_CHUNK_SIZE;

    my $p = $self->_json_sl_parser;
    $self->_parse_headers();

    $self->{_accum_data} //= [];
    while (scalar(@{$self->{_accum_data}}) < $chunk_size) {
        my @buf;
        while (my $obj = $p->fetch) {
            push @buf, $obj->{Value};
        }
        $self->_accumulate_data(@buf);

        # если дальше аккумулировать данные для чанка не будем - выходим из цикла, чтоб не портить прогнозное значение количества строк
        last if scalar(@{$self->{_accum_data}}) >= $chunk_size;

        unless ($self->_init_next_chunk_feed) {
            $self->_when_data_is_finished;
            last;
        }
    }

    my @chunk = splice @{$self->{_accum_data}}, 0, $chunk_size;

    my $stat_extra_data = $self->_stat_extra_data(\@chunk);
    for my $row ( @chunk ) {
         $self->{_process_row}->($row, $self->{report_opts}, $stat_extra_data, $self->_process_row_cache);
    }
    $self->_update_recieved_rows_forecast;

    return scalar(@chunk) ? \@chunk : undef;
}

=head2 _check_request_limits

    проверяет достижение лимитов из запроса потребителя (возвращает 1 если лимиты достигнуты)

=cut

sub _check_request_limits {
    my $self = shift;

    $self->_parse_headers;
    my $real_limit = $self->_real_limit;

    if (defined $real_limit && defined $self->{_total_rows_accumulated} && $self->{_total_rows_accumulated} > $real_limit) {
        my $extra_rows_num = $self->{_total_rows_accumulated} - $real_limit;
        splice @{$self->{_accum_data}}, $#{$self->{_accum_data}} - $extra_rows_num + 1, $extra_rows_num;
        $self->{_total_rows_accumulated} = $self->{_total_rows_accumulated} - $extra_rows_num;
        $self->{_rows_limit_exceeded} = 1;
    }
}

=head2 _real_limit

=cut

sub _real_limit {
    my $self = shift;

    return ($self->{report_opts}->{dont_group_and_filter_zeros_for_totals} && $self->{report_opts}->{limits}->{limit}
                                    ? $self->{report_opts}->{_real_limit}
                                    : undef);
}

=head2 _stat_extra_data

    делаем массовую выборку информации для отдельных полей в массиве статистики,
    для последующей расшифровки/расширения информации об этих полях

=cut

sub _stat_extra_data {
    my ($self, $stat_data) = @_;

    my $stat_extra_data = {};

    my %need_extraction_fields = map {$_ => []} @{ xisect $self->headers->{_header_fields}, [qw/cid OrderID page_id PageGroupID bid/] };
    if (keys %need_extraction_fields) {
        for my $row (@$stat_data) {
            foreach my $f (keys %need_extraction_fields) {
                push @{$need_extraction_fields{$f}}, $row->{$f} if $row->{$f};
            }
        }
        foreach my $f (keys %need_extraction_fields) {
            @{$need_extraction_fields{$f}} = uniq @{$need_extraction_fields{$f}};
        }
    }

    if (my $page_ids = ($need_extraction_fields{page_id} || $need_extraction_fields{PageGroupID})) {
        $self->{_pages_info_cache} //= {};
        $stat_extra_data->{PageID_info} = Stat::Tools::get_pages_info( $page_ids, $self->{_pages_info_cache} );
    }

    if ($need_extraction_fields{cid}) {
        $self->{_cid2OrderID_cache} //= {};
        hash_merge $self->{_cid2OrderID_cache}, get_cid2orderid( cid => [grep { !exists $self->{_cid2OrderID_cache}->{$_} } @{$need_extraction_fields{cid}} ] );
        $stat_extra_data->{cid2OrderID} = $self->{_cid2OrderID_cache};
        if ($self->{report_opts}->{with_performance_coverage}
            || $self->{report_opts}->{is_perf_show_condition}) {

            $stat_extra_data->{cid2camp_type} = get_hash_sql(PPC(cid => $need_extraction_fields{cid}),
                ["select cid, type from campaigns", where => {cid => SHARD_IDS}]);
        }
    }

    if ($need_extraction_fields{OrderID}) {
        $self->{_OrderID2cid_cache} //= {};
        hash_merge $self->{_OrderID2cid_cache}, get_orderid2cid( OrderID => [grep { !exists $self->{_OrderID2cid_cache}->{$_} } @{$need_extraction_fields{OrderID}} ] );
        $stat_extra_data->{OrderID2cid} = $self->{_OrderID2cid_cache};
        if ($self->{report_opts}->{with_performance_coverage}
            || $self->{report_opts}->{is_perf_show_condition}) {

            $stat_extra_data->{OrderID2camp_type} = get_hash_sql(PPC(OrderID => $need_extraction_fields{OrderID}),
                                                                 ["select OrderID, type from campaigns", where => {OrderID => SHARD_IDS}]);
        }
    }

    if ($need_extraction_fields{bid} && $self->{report_opts}->{creative_free}) {
        $stat_extra_data->{bid2banner_type} = get_hash_sql(PPC(bid => $need_extraction_fields{bid}),
            ["select bid, banner_type from banners", where => {bid => SHARD_IDS}]);
    }

    return $stat_extra_data;
}


=head2 _accumulate_data

    преобразовует строки статистики "массив -> хеш", и аккумулирует их в специальный массив/буфер

=cut

sub _accumulate_data {
    my $self = shift;

    my $_header_fields = $self->headers->{_header_fields};
    my $goals_mapping = $self->{report_opts}{goals_mapping} // {};

    for my $row (@_) {
        my %parsed;
        my $i = 0;
        for my $header (@{$_header_fields}) {
            if (ref $header) {
                my $multi_goals_values_sets = $row->[$i];
                for my $values (@$multi_goals_values_sets) {
                    my ($attribution_model, $goal_id, @multi_goals_values) = @$values;
                    (undef, undef, my @multi_goals_headers) = @$header;
                    my $ea = each_array(@multi_goals_headers, @multi_goals_values);
                    while ( my ($h, $v) = $ea->() )   {
                        $parsed{"${h}_${goal_id}_${attribution_model}"} = $v;
                        if (exists $goals_mapping->{$goal_id}) {
                            $parsed{"${h}_$goals_mapping->{$goal_id}_$attribution_model"} = $v;
                        }
                    }
                }
            } else {
                $parsed{$header} = $row->[$i];
            }
            $i++;
        }
        push @{$self->{_accum_data}}, \%parsed;
        $self->{_total_rows_accumulated}++;
    }
    $self->_check_request_limits;
    return 1;
}

=head2 _process_row_cache

    возвращает хеш с некоторыми предрасчитанными "закешированными" данными, необходимыми для обработки строк статистики

=cut

sub _process_row_cache {
    my $self = shift;

    $self->{_process_row_cache_obj} //= $self->{_process_row_cache}->($self->headers->{_header_fields});
    return $self->{_process_row_cache_obj};
}

=head2 _update_recieved_rows_forecast

    обновляет прогнозное количество полученных от БК строк
    рассчитывается на основе общего объема полученного json-a, объема прочитанных данны, и кол-ва уже обработанных строк

=cut

sub _update_recieved_rows_forecast {
    my $self = shift;

    if ($self->{body_iterator}->content_length &&
        $self->{body_iterator}->bytes_read &&
        $self->{_total_rows_accumulated} ) {
        $self->headers->{recieved_rows_forecast} = int($self->{body_iterator}->content_length / $self->{body_iterator}->bytes_read * $self->{_total_rows_accumulated});

        my $real_limit = $self->_real_limit;
        if (defined $real_limit && $self->headers->{recieved_rows_forecast} > $real_limit) {
            $self->headers->{recieved_rows_forecast} = $real_limit+1;
        }
    } else {
        $self->headers->{recieved_rows_forecast} = undef;
    }
}

=head2 _json_sl_parser

    Возвращает готовый к парсингу данных объект SL::JSON
    При необходимости - создает его, и инициирует парсинг первой части JSON-а
    Входные параметры:
        $must_exists - если объект еще не создан - умираем (для исключения циклических вызовов)

=cut

sub _json_sl_parser {
    my ($self, $must_exists) = @_;

    unless ($self->{_json_sl_parser}) {
        $self->_log()->die("_json_sl_parser is not exists yet") if $must_exists;

        my $p = JSON::SL->new();
        $p->set_jsonpointer([(map {"/$_"} @header_keys), "/data/^"]);

        $self->{_json_sl_parser} = $p;
        $self->_init_next_chunk_feed;
    }

    return $self->{_json_sl_parser};
}

=head2 _init_next_chunk_feed
    
    Добавляет в фид следующий чанк json-а, и инициирует фид для работы (запускает процесс парсинга)

=cut

sub _init_next_chunk_feed {
    my $self = shift;

    return 0 if $self->{_http_iterator_empty};
    my $p = $self->_json_sl_parser('must_exists');

    my $next_json_chunk = $self->_next_json_chunk;
    if ($$next_json_chunk eq '') {
        $self->{_http_iterator_empty} = 1;
        return 0;
    }

    eval { $p->feed($$next_json_chunk) };
    my $error = $@;
    if ($error) {
        my $content = $$next_json_chunk;
        while (1) {
            $next_json_chunk = $self->_next_json_chunk;
            last if $$next_json_chunk eq '';

            $content .= $$next_json_chunk;
        }

        if ($error =~ /WEIRD_WHITESPACE/) {
            $self->_log()->out("BS response: $content");
            $self->_log()->die("BS response is unparsable");
        }

        if ($content =~ /Memory limit exceeded/) {
            $self->_log()->die("BS response: Memory limit exceeded");
        } elsif ($content =~ /Timeout exceeded/) {
            $self->_log()->die("BS response: Timeout exceeded");
        } else {
            $self->_log()->out("BS response: $content");
            $self->_log()->die($error);
        }
    }
    return 1;
}

=head2 _next_json_chunk
    
    Возвращает очередной кусок JSON-а, в виде ссылки на скаляр (строку)

=cut

sub _next_json_chunk {
    my $self = shift;

    return $self->{body_iterator}->next_chunk();
}

=head2 _log

=cut

sub _log {
    my $self = shift;

    return $self->{log};
}

1;
