package Stat::Tools;

=pod

    $Id$

=head1 NAME

    Stat::Tools

=head1 DESCRIPTION

    Содержит общие функции, используемые при работе со статистикой

=cut

use strict;
use warnings;
use feature 'state';

use List::Util qw/min minstr maxstr max/;
use List::MoreUtils qw/uniq any firstval/;
use Storable qw/dclone/;

use Settings;
use PrimitivesIds;
use Primitives;
use ShardingTools qw/choose_shard_param/;

use Stat::Fields;
use Stat::Const qw/:base :enums/;
use geo_regions;
use GeoTools;
use CampaignTools;
use Campaign::Types qw/camp_kind_in/;
use MetrikaCounters;
use JSON;
use TextTools;
use Tag;
use Sitelinks;
use BannerTemplates;
use Direct::AdGroups2::MobileContent;
use Direct::Model::Creative;
use Direct::BannersAdditions;
use Phrase;
use Client::ClientFeatures ();

use Yandex::I18n;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::TimeCommon;
use Yandex::DateTime;
use Yandex::Validate;
use Yandex::ListUtils qw/xminus xisect xflatten chunks nsort/;
use Yandex::HashUtils;
use Yandex::Clone;
use Yandex::Log;
use Yandex::Log::Messages;

use utf8;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    field_list_sum
    field_list_sum_if_exists
    field_list_avg
    suffix_list
    apply_suffixes
    periods_suffix_list
    periods_delta_suffix_list
    periods_full_suffix_list
    norm_field_by_periods_suf
    consumer_dict_by_name
    is_need_criterion_id
/;

# Источники для определения технического счетчика DIRECT-147995
# turbodirect заменяется на turbo в enrich_counter_source_by_id
our @TECHNICAL_COUNTERS_SOURCE = qw/
    market
    turbo
    sprav
    eda
/;

our $SPEC_PREFIX = "spec_";
our $ALL_PREFIX  = "all_";

my %money_fields = map {$_ => undef} Stat::Fields::get_all_money_fields();

=head2 special_phrase_title

    Возвращает перевод заголовка для фраз с определенным специальным ID

=cut
sub special_phrase_title {
    my $phrase_id = shift;
    return iget($SPECIAL_PHRASES{$phrase_id});
}

=head2 get_retargetings_text

    получаем названия условий ретаргетинга по их id
    для шардинга можно (нужно) передавать именнованным параметром что-то замапленное на шарды (pid, OrderID)
    если к названиям условий ретаргетинга не нужно добавлять текстовый суффикс - есть опция without_text_suffix => 1
        my $ret_names = get_retargetings_text([1,2,4], pid => [2,3,4]);
        result: {
            1 => 'ret number one (retargeting)',
            2 => 'two (retargeting)',
            ...
        }

=cut

sub get_retargetings_text {
    my ($ret_ids) = shift;
    my %O = @_;

    my $ret_data = get_retargetings_data($ret_ids, %O);

    return {map { ($_ => $ret_data->{$_}->{name}) } keys %$ret_data};
}

=head2 get_retargetings_data

    получаем названия и тип (retargeting или interest) условий ретаргетинга по их id
    для шардинга можно (нужно) передавать именнованным параметром что-то замапленное на шарды (pid, OrderID)
    если к названиям условий ретаргетинга не нужно добавлять текстовый суффикс - есть опция without_text_suffix => 1
        my $ret_names = get_retargetings_data([1,2,4], pid => [2,3,4]);
        result: {
            1 => {name => 'ret number one (retargeting)', type => 'retargeting'},
            2 => {name => 'two (interest)', type => 'interest'},
            ...
        }

=cut

sub get_retargetings_data {
    my ($ret_ids) = shift;
    my %O = @_;

    my %shard_cond = (shard => 'all');
    foreach (qw/pid OrderID/) {
        if ($O{$_}) {
            %shard_cond = ($_ => $O{$_});
            last;
        }
    }
    my $conditions = get_all_sql(PPC(%shard_cond),
        ["SELECT ret_cond_id, condition_name, FIND_IN_SET('interest', properties) as has_interest
            FROM retargeting_conditions",
           WHERE => {ret_cond_id => $ret_ids}]);

    # Суффикс для ретаргетингов оторван в DIRECT-66943
    my $text_suffix = "";
    my %id2name;
    my %rmp_interests;
    for my $cond (@$conditions) {
        if ($cond->{has_interest}) {
            $rmp_interests{$cond->{ret_cond_id}} = $cond;
            next;
             }
        $id2name{$cond->{ret_cond_id}} = {name => $cond->{condition_name} . $text_suffix, type => 'retargeting'};
    }

    if (%rmp_interests) {
        my $interests = get_hash_sql(PPC(%shard_cond),
            ["SELECT ret_cond_id, condition_json FROM retargeting_conditions",
             WHERE => {ret_cond_id => [keys %rmp_interests]}]);
        my $goal2name = get_hash_sql(PPCDICT, ["SELECT import_id, name FROM targeting_categories WHERE targeting_type='rmp_interest'"]);

        $text_suffix = $O{without_text_suffix} ? '' : " (" . iget("Интерес") . ")";
        for my $cond (values %rmp_interests) {
            my $goal_name = _calc_rmp_interest_condition_name($interests->{$cond->{ret_cond_id}}, $goal2name);
            $id2name{$cond->{ret_cond_id}} = {name => $goal_name . $text_suffix, type => 'interest'};
        }
    }

    return \%id2name;
}

=head3 _calc_rmp_interest_condition_name($cond_json, $goal2name)

    Получить имя интереса РМП по строке с JSON-описанием вида
    "[{"goals":[{"goal_id":118654,"time":90}],"type":"all"}]"

    By design там всегда долен быть список из одного объекта с одной целью
    внутри

=cut

sub _calc_rmp_interest_condition_name {
    my ($cond_json, $goal2name) = @_;

    my $cond = from_json($cond_json);
    my @name_parts;
    for my $item (@$cond) {
        my @goals;
        for my $goal (@{$item->{goals}}) {
            if (exists $goal2name->{$goal->{goal_id}}) {
                push @goals, iget($goal2name->{$goal->{goal_id}});
            }
        }
        if (@goals) {
            push @name_parts, join(', ', @goals);
        }
    }
    return join('; ', @name_parts);
}


=head2 get_performance_text($perf_ids, %options)

    my $performance_text = get_performance_text($perf_ids, pid => [...]);

    Получить словарь названий перфоманс-фильтров по их id.

    Параметры:
        $perf_ids    - ссылка на массив id фильтров (их "PhraseID")
    Параметры именованные:
      # для определения шарда выборки данных
        OrderID => $oid     - номер заказа
      # или
        cid => $cid         - номер кампании
      # или
        pid => $pids        - номера групп
    Результат:
        $performance_text   - ссылка на хеш, ключами которого являются bids_preformance.id,
                              а значениями - соответствующие им названия фильтров

=cut

sub get_performance_text {
    my ($perf_ids, %options) = @_;
    return {} unless $perf_ids && @$perf_ids;

    my @shard = choose_shard_param(\%options, [qw/OrderID cid pid/]);

    return get_hash_sql(PPC(@shard), ['SELECT bi_perf.perf_filter_id, bi_perf.name',
                                      'FROM bids_performance bi_perf',
                                      WHERE => {
                                            'bi_perf.perf_filter_id' => $perf_ids,
                                      },
                        ]);
}

=head2 get_perf_filter_ids_by_names

    Получить список id фильтров ДМО (перфоманс/смарт-баннеров), по их названиям и списку заказов

=cut

sub get_perf_filter_ids_by_names {
    my ($perf_filter_names, $camp_selector, $camp_selector_ids) = @_;

    die "Only cid & OrderID allowed as camp_selector" unless $camp_selector && any { $camp_selector eq $_ } qw/cid OrderID/;
    return [] unless $perf_filter_names && $camp_selector_ids;
    
    return get_one_column_sql(PPC($camp_selector => $camp_selector_ids), ["
                                                        SELECT bp.perf_filter_id
                                                          FROM campaigns c
                                                          JOIN phrases p ON p.cid = c.cid
                                                          JOIN bids_performance bp ON bp.pid = p.pid",
                                                         where => {
                                                               'c.'.$camp_selector => SHARD_IDS,
                                                               'bp.name' => $perf_filter_names
                                                         },
                                                  ]);
}

=head2 get_dynamic_data($dyn_ids, %params)

    my $dynamic_data = get_dynamic_data([keys %dyn_cond_ids], OrderID => $oid, with_condition => 1);

    Получить словарь названий условий нацеливания по их id
    Параметры:
        $dyn_ids    - ссылка на массив id условий нацеливания (их "PhraseID")
    Параметры именованные:
        with_condition  - флажок, что нужно выбирать не только имя, но и цели условия нацеливания
      # для определения шарда выборки данных
        OrderID => $oid     - номер заказа
      или
        pid => $pids        - номера групп
    Результат:
    {
        dyn_cond_id1 => {
            name => 'condition_name_1',     # название условия
            condition => [                  # расшифровка условия нацеливания
                {                           # присутствует, если был указан $params{with_condition}
                    kind => ...,
                    type => ...,
                    value => ...
                },
                ...
            ],
        },
        dyn_cond_id2 => ...,
    }

=cut
sub get_dynamic_data {
    my ($dyn_ids, %params) = @_;
    return {} unless $dyn_ids && @$dyn_ids;

    my %result;
    my $dyn_conditions = Direct::DynamicConditions->get_by(dyn_cond_id => $dyn_ids,
                                                           %{hash_cut(\%params, qw/OrderID pid cid/)}, # для определения шарда
                                                           with_deleted => 1, # включая старые условия, уже без фраз
                                                           )->items;
    for my $dynamic_condition (@$dyn_conditions) {
        next if exists $result{$dynamic_condition->dyn_cond_id};

        my $short_dynamic_data = { name => $dynamic_condition->condition_name };
        if ($params{with_condition}) {
            $short_dynamic_data->{condition} = [ map { $_->to_hash } @{ $dynamic_condition->condition } ],
        }
        $result{$dynamic_condition->dyn_cond_id} = $short_dynamic_data;
    }
    return \%result;
}

=head2 get_dyn_cond_ids_by_names

    Получить список id условий нацеливания, по их названиям и списку заказов

=cut

sub get_dyn_cond_ids_by_names {
    my ($dyn_names, $camp_selector, $camp_selector_ids) = @_;

    die "Only cid & OrderID allowed as camp_selector" unless $camp_selector && any { $camp_selector eq $_ } qw/cid OrderID/;
    return [] unless $dyn_names && $camp_selector_ids;

    return get_one_column_sql(PPC($camp_selector => $camp_selector_ids), [
                                                        'SELECT dc.dyn_cond_id',
                                                        'FROM campaigns c',
                                                        'JOIN phrases p ON p.cid = c.cid',
                                                        'JOIN dynamic_conditions dc ON dc.pid = p.pid',
                                                      where => {
                                                        'c.'.$camp_selector => $camp_selector_ids,
                                                        'dc.condition_name' => $dyn_names
                                                      },
                                                  ]);
}

=head2 calc_avg

    Для каждой строчки вычисляет "вычислимые" поля
    Входные параметры:
        $h - ссылка на хеш со строкой статистики
        $suf - суффикс полей (_0, _1, _a, _b и т. п.)
        %options - именованные поля
            prefix => 't',
            spec_and_all_prefix => 0|1 - по умолчанию 1
            without_round => 0|1 - возвращать все поля без округлений
            preserve_existing => 0|1 - не пересчитывать поля, в которых уже есть значения
            four_digits_precision => 0|1 – округлять денежные поля до 4 знаков

    При изменениях нужно актуализировать список запрашиваемых из БК полей в Stat::Fields::countable_fields_dependencies

=cut

sub calc_avg {
    my ($h, $suf, %options) = @_;
    no warnings "uninitialized";

    my $prefix = $options{prefix}? 't' : '';
    $options{spec_and_all_prefix} //= 1;
    $suf ||= '';

    my @goal_postfixes = _get_goal_postfixes($h);

    # инициализируем основные аггрегируемые поля, на случай если по какой-то причине они undef-ы
    my @fields_to_check = qw/clicks sum bonus agoalincome pv_agoalincome/;
    if (!$options{allow_null_shows}) {
        push @fields_to_check, 'shows';
    }
    for my $field (@fields_to_check) {
        $h->{"${prefix}$field$suf"}  ||= 0;
    }

    for ( '', ($options{spec_and_all_prefix} ? ${SPEC_PREFIX} : ()) ) {
        my $asesnumlim = $h->{"$_${prefix}asesnumlim$suf"} // $h->{"$_${prefix}asesnum$suf"};
        if (!defined($h->{"$_${prefix}shows$suf"})) {
            $h->{"$_${prefix}ctr$suf"} = undef;
        } else {
            $h->{"$_${prefix}ctr$suf"}   = $h->{"$_${prefix}shows$suf"} ? 100 * $h->{"$_${prefix}clicks$suf"}/$h->{"$_${prefix}shows$suf"} : 0;
        }
        $h->{"$_${prefix}fp_shows_avg_pos$suf"}     = $h->{"$_${prefix}fp_shows_pos$suf"}/$h->{"$_${prefix}fp_shows$suf"} if $h->{"$_${prefix}fp_shows$suf"} && $h->{"$_${prefix}fp_shows_pos$suf"};
        $h->{"$_${prefix}fp_clicks_avg_pos$suf"}     = $h->{"$_${prefix}fp_clicks_pos$suf"}/$h->{"$_${prefix}fp_clicks$suf"} if $h->{"$_${prefix}fp_clicks$suf"} && $h->{"$_${prefix}fp_clicks_pos$suf"};
        $h->{"$_${prefix}av_sum$suf"}    = $h->{"$_${prefix}sum$suf"}/$h->{"$_${prefix}clicks$suf"} if $h->{"$_${prefix}clicks$suf"};
        $h->{"$_${prefix}avg_cpm$suf"}   = 1000 * $h->{"$_${prefix}sum$suf"}/$h->{"$_${prefix}shows$suf"} if $h->{"$_${prefix}shows$suf"} && (!$options{preserve_existing} || !exists $h->{"$_${prefix}avg_cpm$suf"});
        $h->{"$_${prefix}adepth$suf"}    = $h->{"$_${prefix}aseslen$suf"}/$asesnumlim if $asesnumlim && $h->{"$_${prefix}clicks$suf"};
        $h->{"$_${prefix}agoalcost$suf"} = $h->{"$_${prefix}sum$suf"}/$h->{"$_${prefix}agoalnum$suf"} if $h->{"$_${prefix}agoalnum$suf"};
        # Нам от Статистики приходит aconv(ConversionPct) если мы его запрашиваем.
        # Но для промежуточной статистики где-то в веб-интерфейсе мы вычисляем дополнительно.
        $h->{"$_${prefix}aconv$suf"}     = 100 * $h->{"$_${prefix}agoalnum$suf"}/$h->{"$_${prefix}clicks$suf"} 
            if $h->{"$_${prefix}clicks$suf"} && $h->{"$_${prefix}agoalnum$suf"} > 0;
        delete $h->{"$_${prefix}agoalnum$suf"} unless defined $h->{"$_${prefix}aconv$suf"};
        $h->{"$_${prefix}agoalroi$suf"}  = ($h->{"$_${prefix}agoalincome$suf"} - $h->{"$_${prefix}sum$suf"}) / $h->{"$_${prefix}sum$suf"} if $h->{"$_${prefix}sum$suf"} > 0 && $h->{"$_${prefix}agoalincome$suf"} > 0;

        $h->{"$_${prefix}pv_agoalcost$suf"} = $h->{"$_${prefix}sum$suf"}/$h->{"$_${prefix}pv_agoalnum$suf"} if $h->{"$_${prefix}pv_agoalnum$suf"};
        $h->{"$_${prefix}pv_agoalroi$suf"}  = ($h->{"$_${prefix}pv_agoalincome$suf"} - $h->{"$_${prefix}sum$suf"}) / $h->{"$_${prefix}sum$suf"} if $h->{"$_${prefix}sum$suf"} > 0 && $h->{"$_${prefix}pv_agoalincome$suf"} > 0;

        if (defined $h->{"$_${prefix}eshows$suf"}) {
            $h->{"$_${prefix}ectr$suf"}  = $h->{"$_${prefix}eshows$suf"} > 0 ? (100 * $h->{"$_${prefix}clicks$suf"}/$h->{"$_${prefix}eshows$suf"}) : 0;
            $h->{"$_${prefix}avg_x$suf"} = $h->{"$_${prefix}shows$suf"} ? (100 * $h->{"$_${prefix}eshows$suf"}/$h->{"$_${prefix}shows$suf"}) : 0;
        }
        if ($options{attribution_model}//'' eq "last_yandex_direct_view_cross_device") {
            $h->{"$_${prefix}all_agoalnum$suf"} = ($h->{"$_${prefix}agoalnum$suf"} // 0) + ($h->{"$_${prefix}pv_agoalnum$suf"} // 0);
        }
    }

    if ($options{spec_and_all_prefix}) {
        if (!defined($h->{"${prefix}shows$suf"}) || !defined($h->{"${SPEC_PREFIX}${prefix}shows$suf"})) {
            $h->{"${ALL_PREFIX}${prefix}shows$suf"} = undef;
        } else {
            $h->{"${ALL_PREFIX}${prefix}shows$suf"} = $h->{"${prefix}shows$suf"} + $h->{"${SPEC_PREFIX}${prefix}shows$suf"};
        }
        foreach my $fld (qw/clicks sum bonus fp_shows fp_shows_pos fp_clicks fp_clicks_pos agoalnum agoalincome pv_agoalnum pv_agoalincome/) {
            $h->{"${ALL_PREFIX}${prefix}$fld$suf"} = ($h->{"${prefix}$fld$suf"} || 0) + ($h->{"${SPEC_PREFIX}${prefix}$fld$suf"} || 0);
        }

        my $asesnumlim = $h->{"${prefix}asesnumlim$suf"} // $h->{"${prefix}asesnum$suf"};
        my $asesnumlim_spec = $h->{"${SPEC_PREFIX}${prefix}asesnumlim$suf"} // $h->{"${SPEC_PREFIX}${prefix}asesnum$suf"};
        my $pv_asesnumlim = $h->{"${prefix}pv_asesnumlim$suf"} // $h->{"${prefix}pv_asesnum$suf"};
        my $pv_asesnumlim_spec = $h->{"${SPEC_PREFIX}${prefix}pv_asesnumlim$suf"} // $h->{"${SPEC_PREFIX}${prefix}pv_asesnum$suf"};

        if (!defined($h->{"${ALL_PREFIX}${prefix}shows$suf"})) {
            $h->{"${ALL_PREFIX}${prefix}ctr$suf"} = undef;
        } else {
            $h->{"${ALL_PREFIX}${prefix}ctr$suf"} = ($h->{"${ALL_PREFIX}${prefix}shows$suf"}) ? 
                                                            100 * $h->{"${ALL_PREFIX}${prefix}clicks$suf"} /
                                                            $h->{"${ALL_PREFIX}${prefix}shows$suf"} : 0;
        }
        
        $h->{"${ALL_PREFIX}${prefix}fp_shows_avg_pos$suf"} =  $h->{"${ALL_PREFIX}${prefix}fp_shows_pos$suf"}/$h->{"${ALL_PREFIX}${prefix}fp_shows$suf"} 
                                                        if $h->{"${ALL_PREFIX}${prefix}fp_shows$suf"} && $h->{"${ALL_PREFIX}${prefix}fp_shows_pos$suf"};

        $h->{"${ALL_PREFIX}${prefix}fp_clicks_avg_pos$suf"} = $h->{"${ALL_PREFIX}${prefix}fp_clicks_pos$suf"}/$h->{"${ALL_PREFIX}${prefix}fp_clicks$suf"}
                                                        if $h->{"${ALL_PREFIX}${prefix}fp_clicks$suf"} && $h->{"${ALL_PREFIX}${prefix}fp_clicks_pos$suf"};

        $h->{"${ALL_PREFIX}${prefix}agoalcost$suf"} = $h->{"${ALL_PREFIX}${prefix}sum$suf"} / $h->{"${ALL_PREFIX}${prefix}agoalnum$suf"}
                                                        if ($h->{"${ALL_PREFIX}${prefix}agoalnum$suf"});

        $h->{"${ALL_PREFIX}${prefix}pv_agoalcost$suf"} = $h->{"${ALL_PREFIX}${prefix}sum$suf"} / $h->{"${ALL_PREFIX}${prefix}pv_agoalnum$suf"}
                                                        if ($h->{"${ALL_PREFIX}${prefix}pv_agoalnum$suf"});

        $h->{"${ALL_PREFIX}${prefix}av_sum$suf"}    = $h->{"${ALL_PREFIX}${prefix}sum$suf"} / $h->{"${ALL_PREFIX}${prefix}clicks$suf"}
                                                        if ($h->{"${ALL_PREFIX}${prefix}clicks$suf"});

        $h->{"${ALL_PREFIX}${prefix}avg_cpm$suf"}    = 1000 * $h->{"${ALL_PREFIX}${prefix}sum$suf"} / $h->{"${ALL_PREFIX}${prefix}shows$suf"}
                                                        if ($h->{"${ALL_PREFIX}${prefix}shows$suf"}) && (!$options{preserve_existing} || !exists $h->{"${ALL_PREFIX}${prefix}avg_cpm$suf"});

        $h->{"${ALL_PREFIX}${prefix}adepth$suf"}    = (($h->{"${prefix}aseslen$suf"} || 0) + ($h->{"${SPEC_PREFIX}${prefix}aseslen$suf"} || 0)) /
                                                       (($asesnumlim || 0) + ($asesnumlim_spec || 0))
                                                        if ($asesnumlim && $h->{"${prefix}clicks$suf"} || $asesnumlim_spec && $h->{"${SPEC_PREFIX}${prefix}clicks$suf"});
        $h->{"${ALL_PREFIX}${prefix}pv_adepth$suf"}    = (($h->{"${prefix}pv_aseslen$suf"} || 0) + ($h->{"${SPEC_PREFIX}${prefix}pv_aseslen$suf"} || 0)) /
                                                       (($pv_asesnumlim || 0) + ($pv_asesnumlim_spec || 0))
                                                        if ($pv_asesnumlim && $h->{"${prefix}shows$suf"} || $pv_asesnumlim_spec && $h->{"${SPEC_PREFIX}${prefix}shows$suf"});


        $h->{"${ALL_PREFIX}${prefix}aconv$suf"}     = 100 * $h->{"${ALL_PREFIX}${prefix}agoalnum$suf"} /
                                                        (($h->{"${prefix}clicks$suf"} || 0) + ($h->{"${SPEC_PREFIX}${prefix}clicks$suf"} || 0))
                                                        if ($h->{"${prefix}clicks$suf"} || $h->{"${SPEC_PREFIX}${prefix}clicks$suf"});
        $h->{"${ALL_PREFIX}${prefix}pv_aconv$suf"}     = 100 * $h->{"${ALL_PREFIX}${prefix}pv_agoalnum$suf"} /
                                                        (($h->{"${prefix}shows$suf"} || 0) + ($h->{"${SPEC_PREFIX}${prefix}shows$suf"} || 0))
                                                        if ($h->{"${prefix}shows$suf"} || $h->{"${SPEC_PREFIX}${prefix}shows$suf"});


        delete $h->{"${ALL_PREFIX}${prefix}agoalnum$suf"} unless defined $h->{"${ALL_PREFIX}${prefix}aconv$suf"};

        delete $h->{"${ALL_PREFIX}${prefix}pv_agoalnum$suf"} unless defined $h->{"${ALL_PREFIX}${prefix}pv_aconv$suf"};

        $h->{"${ALL_PREFIX}${prefix}agoalroi$suf"}  = ($h->{"${ALL_PREFIX}${prefix}agoalincome$suf"} - $h->{"${ALL_PREFIX}${prefix}sum$suf"}) / $h->{"${ALL_PREFIX}${prefix}sum$suf"}
                                                        if $h->{"${ALL_PREFIX}${prefix}sum$suf"} > 0 && $h->{"${ALL_PREFIX}${prefix}agoalincome$suf"} > 0;
        $h->{"${ALL_PREFIX}${prefix}pv_agoalroi$suf"}  = ($h->{"${ALL_PREFIX}${prefix}pv_agoalincome$suf"} - $h->{"${ALL_PREFIX}${prefix}sum$suf"}) / $h->{"${ALL_PREFIX}${prefix}sum$suf"}
                                                        if $h->{"${ALL_PREFIX}${prefix}sum$suf"} > 0 && $h->{"${ALL_PREFIX}${prefix}pv_agoalincome$suf"} > 0;
        if ($options{attribution_model}//'' eq "last_yandex_direct_view_cross_device") {
            $h->{"${ALL_PREFIX}${prefix}all_agoalnum$suf"} = ($h->{"${ALL_PREFIX}${prefix}agoalnum$suf"} // 0) + ($h->{"${ALL_PREFIX}${prefix}pv_agoalnum$suf"} // 0);
        }
    }

    _round_avg($h, $suf, \@goal_postfixes, %options) unless $options{without_round};
}

=head2 _round_avg 

    Округляет нужные поля для посчитанной строки статистики

=cut

sub _round_avg {
    my ($h, $suf, $goal_postfixes, %options) = @_;
    my $rounding_mode = $options{four_digits_precision} ? 'four_digits_precision' : '';

    my $prefix = $options{prefix}? 't' : '';
    state $floating_point_fields //= [Stat::Fields::get_countable_floating_point_fields()];
    state $extended_precision_fields //= {map { $_ => 1 } Stat::Fields::get_countable_extended_precision_fields()};

    for my $sprefix ( '', ($options{spec_and_all_prefix} ? ($SPEC_PREFIX, $ALL_PREFIX)  : ()) ) {
        # округляем денежные значения, чтобы сходились отображаемые промежуточные суммы и итоговые
        for my $field (qw/sum bonus agoalincome agoals_profit pv_agoalincome pv_agoals_profit/) {
            if (any { $field eq $_} qw/agoalincome agoals_profit pv_agoalincome pv_agoals_profit/){
                for my $goal_postfix ( '', @$goal_postfixes) {
                    $h->{"$sprefix$prefix$field${goal_postfix}$suf"} = sprintf_round($h->{"$sprefix$prefix$field${goal_postfix}$suf"}, $rounding_mode)
                        if defined $h->{"$sprefix$prefix$field${goal_postfix}$suf"};
                }
            } else {
                $h->{"$sprefix$prefix$field$suf"} = sprintf_round($h->{"$sprefix$prefix$field$suf"}, $rounding_mode) if defined $h->{"$sprefix$prefix$field$suf"}; # двойное округление - DIRECT-49406
            }
        }
        for my $field (qw/ctr ectr fp_shows_avg_pos fp_clicks_avg_pos
                          av_sum adepth winrate bounce_ratio pv_bounce_ratio
                          aprgoodmultigoal aprgoodmultigoal_cpa aprgoodmultigoal_conv_rate
                          pv_aprgoodmultigoal pv_aprgoodmultigoal_cpa pv_aprgoodmultigoal_conv_rate
                          avg_view_freq avg_cpm eshows avg_x avg_bid avg_cpm_bid agoals_profit pv_agoals_profit
                          auction_win_rate imp_reach_rate imp_to_win_rate/, @$floating_point_fields
        ) {
            my $sprintf_fmt = "%.2f";
            if ($extended_precision_fields->{$field} || ($rounding_mode eq 'four_digits_precision' && exists $money_fields{$field})) {
                $sprintf_fmt = "%.4f";
            }
            my $prefixed_field = "$sprefix$prefix$field$suf";
            $h->{$prefixed_field} = sprintf($sprintf_fmt, $h->{$prefixed_field}) if defined $h->{$prefixed_field};
        }

        for my $field (qw/agoalcost agoalroi agoalcrr aconv avg_time_to_conv pv_agoalcost pv_agoalroi pv_agoalcrr pv_aconv pv_avg_time_to_conv/) {
            for my $goal_postfix ( '', @$goal_postfixes) {
                my $sprintf_fmt = "%.2f";
                if ($rounding_mode eq 'four_digits_precision' && exists $money_fields{$field}) {
                    $sprintf_fmt = "%.4f";
                }
                $h->{"$sprefix$prefix$field${goal_postfix}$suf"} = sprintf($sprintf_fmt, $h->{"$sprefix$prefix$field${goal_postfix}$suf"})
                    if defined $h->{"$sprefix$prefix$field${goal_postfix}$suf"};
            }
        }
    }
}

sub _get_goal_postfixes {
    my ($h) = @_;
    my @goal_postfixes;
    for my $field (keys %$h) {
        if ( $field =~ /\w+(_\d+_\d)$/ ) {
            push @goal_postfixes, $1;
        }
    }
    return uniq @goal_postfixes;
}

=head2 calc_avg_compare_periods

    Для каждой строчки вычисляет "вычислимые" поля с суффиксами каждого периода, а также дельты

=cut

sub calc_avg_compare_periods {
    my ($h, %options) = @_;
    my $rounding_mode = $options{four_digits_precision} ? 'four_digits_precision' : '';
    no warnings "uninitialized";

    for my $suf (periods_suffix_list()) {
        # если в показах за период от БК пришел undef, значит статистики по данному периоду вовсе нет
        # пропускаем чтоб не заменить в основных аггрегируемых полях undef на 0
        # будет актуально когда в БК реализуют поддержку группировки по дням/неделям/...
        next unless defined $h->{"shows$suf"} || defined $h->{"clicks$suf"};

        calc_avg($h, $suf, %options, spec_and_all_prefix => 0, without_round => 1);
    }

    # целочисленные поля (для правильного формата абсолютных дельт)
    state $add_integer_fields //= [Stat::Fields::get_countable_integer_fields()];
    my %integer_fields = map { $_ => 1 } qw/shows clicks agoalnum pv_agoalnum uniq_viewers auction_hits auction_wins served_impressions/, @$add_integer_fields;

    state $extended_precision_fields //= {map { $_ => 1 } Stat::Fields::get_countable_extended_precision_fields()};

    state $add_floating_point_fields //= [Stat::Fields::get_countable_floating_point_fields()];
    foreach my $fld (qw/shows eshows clicks sum ctr ectr bonus avg_x fp_shows_avg_pos fp_clicks_avg_pos av_sum winrate
                       adepth agoalnum agoalcost agoalincome aconv agoalroi bounce_ratio 
                       aprgoodmultigoal aprgoodmultigoal_cpa aprgoodmultigoal_conv_rate
                       pv_adepth pv_agoalnum pv_agoalcost pv_agoalincome pv_aconv pv_agoalroi pv_bounce_ratio 
                       pv_aprgoodmultigoal pv_aprgoodmultigoal_cpa pv_aprgoodmultigoal_conv_rate
                       avg_view_freq avg_cpm uniq_viewers avg_bid avg_cpm_bid avg_time_to_conv agoals_profit
                       auction_win_rate imp_reach_rate imp_to_win_rate/, @$add_floating_point_fields,
                     ($options{total_av} ? qw/av_day av_day_shows av_grouping/ : ()) ) {
        # исключение для agoalnum и ctr - https://st.yandex-team.ru/DIRECT-54748#1465299501000
        my ($va, $vb);
        unless (exists $h->{$fld.'_delta'} && exists $h->{$fld.'_absdelta'}) {
            if ($fld eq 'agoalnum') {
                ($va, $vb) = ($h->{$fld.'_a'} // 0, $h->{$fld.'_b'} // 0)
            } elsif ($fld eq 'ctr') {
                $va = $h->{'shows_a'} ? $h->{$fld.'_a'} : undef;
                $vb = $h->{'shows_b'} ? $h->{$fld.'_b'} : undef;
            } else {
                ($va, $vb) = ($h->{$fld.'_a'}, $h->{$fld.'_b'});
            }
        }

        my $sprintf_fmt = "%.2f";
        my $abs_sprintf_fmt = $sprintf_fmt;
        if ($extended_precision_fields->{$fld}) {
            $abs_sprintf_fmt = "%.4f";
        }
        if ($rounding_mode eq 'four_digits_precision' && exists $money_fields{$fld}) {
            $sprintf_fmt = "%.4f";
            $abs_sprintf_fmt = "%.4f";
        }
        if ($integer_fields{$fld}) {
            $abs_sprintf_fmt = "%d";
        }


        if (exists $h->{$fld.'_delta'}) {
            $h->{$fld.'_delta'} = sprintf("%.2f", $h->{$fld.'_delta'}) if defined $h->{$fld.'_delta'};
        } else {
            if (defined $va && defined $vb && $vb != 0) {
                $h->{$fld.'_delta'} = sprintf($sprintf_fmt, ($va - $vb)/abs($vb)*100);
            }
        }
        if (exists $h->{$fld.'_absdelta'}) {
            $h->{$fld.'_absdelta'} = sprintf($abs_sprintf_fmt, $h->{$fld.'_absdelta'}) if defined $h->{$fld.'_absdelta'};
        } else {
            if (defined $va && defined $vb) {
                $h->{$fld.'_absdelta'} = sprintf($abs_sprintf_fmt, $va - $vb);
            }
        }
    }

    for my $suf (periods_suffix_list()) {
        next unless defined $h->{"shows$suf"} || defined $h->{"clicks$suf"};

        _round_avg($h, $suf, [], %options, spec_and_all_prefix => 0);
    }
}

=head2 calc_average_stat

    алиас к calc_avg

=cut

sub calc_average_stat {
    my ($h, $suf, %options) = @_;
    calc_avg($h, $suf, %options);
}

=head2 field_list_sum

    Аддитивные поля статистики (можно суммировать в зависимости от нужных группировок)

=cut

sub field_list_sum {
    return qw(shows clicks sum bonus fp_shows fp_shows_pos fp_clicks fp_clicks_pos aseslen asesnum agoalnum gsum agoalincome 
              pv_aseslen pv_asesnum pv_agoalnum pv_agoalincome);
}

=head2 field_list_sum_if_exists

    Необязательные аддитивные поля статистики (можно суммировать в зависимости от нужных группировок, если поле есть в суммируемых данных)

=cut

sub field_list_sum_if_exists {
    return qw(asesnumlim pv_asesnumlim eshows);
}

=head2 field_list_avg

    Поля которые значение которых высчитывается из других полей

=cut

sub field_list_avg {
    return qw(ctr ectr fp_shows_avg_pos fp_clicks_avg_pos av_sum adepth aconv agoalcost agoalroi agoalcrr avg_cpm avg_x
    pv_adepth pv_aconv pv_agoalcost pv_agoalroi pv_agoalcrr);
}

=head2 suffix_list

    Набор возможных суффиксов

=cut

sub suffix_list {
    return ('', '_0', '_1');
}

=head2 periods_suffix_list

    Набор возможных суффиксов для обозначения периодов при сравнении двух периодов

=cut

sub periods_suffix_list {
    return ('_a', '_b');
}

=head2 periods_delta_suffix_list

    Набор возможных суффиксов для обозначения дельт при сравнении двух периодов

=cut

sub periods_delta_suffix_list {
    return ('_delta', '_absdelta');
}

=head2 periods_full_suffix_list

    Набор возможных суффиксов для обозначения различных полей при сравнении двух периодов

=cut

sub periods_full_suffix_list {
    return (periods_suffix_list(), periods_delta_suffix_list());
}


=head2 norm_field_by_periods_suf

    возвращает название поля без суффиксов разбивки по поиск/контекст и по периодам, и сам суффикс

=cut

sub norm_field_by_periods_suf {
    my $field = shift;

    my $suf = '';
    my $re_suf = join '|', periods_full_suffix_list();
    if ($field =~ /^(.+?)($re_suf)$/) {
        ($field, $suf) =  ($1, $2 // '');
    }

    return wantarray ? ($field, $suf) : $field;
}

=head2 apply_suffixes 

    Добавляет к списку полей, список суффиксов

=cut

sub apply_suffixes
{
    my ($fields, $suffixes) = @_;

    my @sfx_fields;
    for my $field (xflatten $fields) {
        push @sfx_fields, map { $field.$_ } xflatten $suffixes
    }
    return @sfx_fields;
}

=head2 rename_field

    Добавление префикса "t" ко всем полям статистики (для тоталов в $vars)

=cut

sub rename_field {
    my $data = shift;
    
    for my $suf ( suffix_list() ) {
        for my $f ( field_list_sum(), field_list_sum_if_exists(), field_list_avg() ) {
            if (exists $data->{"$f$suf"}) {
                $data->{"t$f$suf"} = $data->{"$f$suf"};
                delete $data->{"$f$suf"};
            }
        }
    }     
}

=head2 calc_average_campaign_stat

Вычисление средних значений (в день, в период группировки)

    $days_num -- кол-во дней статистики
    $periods_num -- кол-во периодов группировки по которым есть статистика (для вычисления среднего значения)

=cut

sub calc_average_campaign_stat {
    my ($h, $suf, $days_num, $periods_num, $prefix, %options) = @_;

    $suf ||= '';
    $days_num ||= 1;
    $prefix //= 't';
    calc_avg($h, $suf, prefix=>$prefix ? 1 : 0, %options);
    my $sprintf_fmt = $options{four_digits_precision} ? "%.4f" : "%.2f";
    $h->{"av_day$suf"} = sprintf($sprintf_fmt, $h->{"${prefix}sum$suf"} / $days_num);
    $h->{"av_day_shows$suf"} = sprintf("%.02f", $h->{"${prefix}shows$suf"} / $days_num);

    # если передали кол-во периодов группировки, то считаем средний расход за такой период
    if (defined $periods_num && $periods_num > 0) {
        $h->{"av_grouping$suf"} = sprintf($sprintf_fmt, $h->{"${prefix}sum$suf"} / $periods_num);
    }
}

=head2 sprintf_round

Округлить $value переданным в $mode образом (или до двух знаков) и отдать строку

=cut

sub sprintf_round {
    my ($value, $mode) = @_;
    if ($mode // '' eq 'four_digits_precision') {
        return sprintf("%.4f", round4s($value));
    }
    return sprintf("%.2f", round2s($value));
}

=head2 get_group_by_as_hash(\@group_by, $dateagg)
    
    Преобразует список параметров группировки и тип агреграции по дате в 
    \%group_by

=cut

sub get_group_by_as_hash {
    my ($group_by, $dateagg) = @_;

    my $group_hash = {};
    for my $group ( @$group_by ) {
        if ( lc($group) eq 'date' ) {
            $group_hash->{$group} = $dateagg;
        } else {
            $group_hash->{$group} = 1;
        }
    }

    return $group_hash;
}

=head2 page_target_type_number_from_filter

    Номер типа площадки для использования в отчетах

=cut

sub page_target_type_number_from_filter {
    my $filter = shift;
    if ( $filter->{page_target} && $filter->{page_target} eq 'search' ) {
        return '!3';
    } elsif ( $filter->{page_target} && $filter->{page_target} eq 'context' ) {
        return '3';
    }
    return;
}

=head2 $BANNERS_CACHE

    хеш с закешированной информацией о баннерах

=cut

my $BANNERS_CACHE;

=head2 get_banner_info

    Получить информацию о баннере

=cut

sub get_banner_info {
    my ( $id_key, $banner_id, %O ) = @_;

    return mass_get_banner_info($id_key, [$banner_id], %O)->{$banner_id};
}

=head2 mass_get_banner_info

    Получить массово информацию о cписке баннеров

=cut

sub mass_get_banner_info {
    my ( $ids_key, $banner_ids, %O ) = @_;

    die "Expected bid or BannerID in \$ids_key" unless $ids_key && any { $ids_key eq $_} qw/BannerID bid/;
    die "ARRAYREF expected in \$banner_ids" unless ref $banner_ids eq 'ARRAY';
    die "HASHREF expected on \$O{sharding_params}" unless ref $O{sharding_params} eq 'HASH';

    my ($sharding_key, $sharding_values) = %{$O{sharding_params}};
    my $id2shard;
    my $bannerid2bid;
    if ($ids_key eq 'bid') {
        $id2shard = get_shard_multi('bid' => [ xflatten $banner_ids ]);
    } else {
        $bannerid2bid = get_bannerid2bid($banner_ids, $sharding_key, $sharding_values);
        $id2shard = get_shard_multi('bid' => [ xflatten values $bannerid2bid ]);
    }
    my @shards = uniq values %$id2shard;

    $BANNERS_CACHE //= {BannerID => {}, bid => {}};
    for my $chunk (chunks($banner_ids, 1_000)) {
        for my $shard (@shards) {
            my $chunk_banner_ids = [grep { is_valid_int($_, 1) && !$BANNERS_CACHE->{$ids_key}->{$_} && ($id2shard->{$_} == $shard || $id2shard->{$bannerid2bid->{$_}} ) } @$chunk];
            next unless @$chunk_banner_ids;

            my @fields = (
                'b.bid', 'b.title', 'b.body', 'b.href', 'b.domain', 'b.sitelinks_set_id',
                'b.statusModerate bStatusModerate', 'b.statusPostModerate bStatusPostModerate',
                'b.statusShow statusShow', 'b.statusArch', 'b.type AS banner_type', 'b.banner_type AS real_banner_type',
                'b.flags',
                'domains.domain AS adgroup_main_domain', 'p.geo', 'p.pid', 'p.group_name', 
                'p.statusModerate pStatusModerate', 'p.statusPostModerate pStatusPostModerate', 
                '(vcard_id IS NOT NULL) AS with_phone', 'mw.mw_text as banner_minus_words', 
                'bim.image_hash as image', 'u.ClientID', 'p.cid', 'p.adgroup_type', 
                'gmc.store_content_href', 'gmc.mobile_content_id', 'gmc.device_type_targeting', 
                'gmc.network_targeting', 'bmc.reflected_attrs', 'bmc.primary_action',
                'bdh.display_href',
                'br.used_resources', 'cbr.categories_bs',
                'bpml.permalink',
                'acp.content_promotion_type AS content_promotion_content_type', 'cp.preview_url AS content_promotion_preview_url',
                'bcp.visit_url AS content_promotion_visit_url'
            );
            push @fields, map {
                sprintf "perfc.%s AS %s", sql_quote_identifier($_), sql_quote_identifier("perfc_${_}")
            } Direct::Model::Creative->get_db_columns_list('perf_creatives', [qw/id name width height alt_text preview_url template_id creative_type source_media_type
                                                                                 live_preview_url has_packshot is_adaptive layout_id duration _additional_data/]);
            my @images_formats_columns = Direct::Model::ImageFormat->get_db_columns_list('banner_images_formats', [qw/hash mds_group_id namespace avatars_host width height/]);
            push @fields, map {
                sprintf "bif.%s AS %s", sql_quote_identifier($_), sql_quote_identifier("imad_${_}")
            } @images_formats_columns;
            push @fields, map {
                sprintf "bim_f.%s AS %s", sql_quote_identifier($_), sql_quote_identifier("image_${_}")
            } @images_formats_columns;

            my $banners = get_hashes_hash_sql(
                PPC(shard => $shard),[
                "SELECT b.$ids_key, b.BannerID, " . join(',', @fields) . "
                FROM
                    banners b JOIN phrases p on p.pid = b.pid
                    LEFT JOIN adgroups_dynamic ad ON ad.pid = p.pid AND p.adgroup_type = 'dynamic'
                    LEFT JOIN domains ON domains.domain_id = ad.main_domain_id
                    LEFT JOIN minus_words mw on mw.mw_id = p.mw_id
                    LEFT JOIN banner_images bim on b.bid = bim.bid
                    LEFT JOIN banner_images_formats bim_f ON bim.image_hash = bim_f.image_hash
                    LEFT JOIN images im on b.bid = im.bid
                    LEFT JOIN banner_images_formats bif on bif.image_hash = im.image_hash
                    JOIN campaigns c on c.cid = p.cid
                    JOIN users u on u.uid = c.uid
                    LEFT JOIN adgroups_mobile_content gmc on p.pid = gmc.pid AND p.adgroup_type = 'mobile_content'
                    LEFT JOIN banners_mobile_content bmc on bmc.bid = b.bid AND b.banner_type = 'mobile_content'
                    LEFT JOIN banners_performance bperf on bperf.bid = b.bid
                    LEFT JOIN perf_creatives perfc on perfc.creative_id = bperf.creative_id
                    LEFT JOIN banner_display_hrefs bdh ON bdh.bid = b.bid
                    LEFT JOIN banner_resources br on b.bid = br.bid
                    LEFT JOIN catalogia_banners_rubrics cbr on cbr.bid = b.bid
                    LEFT JOIN banner_permalinks bpml ON bpml.bid = b.bid AND bpml.permalink_assign_type = 'manual'
                    LEFT JOIN adgroups_content_promotion acp ON acp.pid = p.pid AND p.adgroup_type = 'content_promotion'
                    LEFT JOIN banners_content_promotion bcp ON bcp.bid = b.bid
                    LEFT JOIN content_promotion cp ON cp.id = bcp.content_promotion_id
                ",
                WHERE => { 
                    "b.$ids_key" => $chunk_banner_ids 
                    } 
                ]
            );
            my $not_selected_banner_ids = [grep { !exists $banners->{$_} } @$chunk_banner_ids];
            if (@$not_selected_banner_ids) {
                hash_merge $banners, get_hashes_hash_sql(PPC(shard => $shard), [
                    "SELECT bim.$ids_key, bim.BannerID, " . join(',', @fields) . "
                    FROM
                        banners b JOIN phrases p on p.pid = b.pid
                        LEFT JOIN adgroups_dynamic ad ON ad.pid = p.pid AND p.adgroup_type = 'dynamic'
                        LEFT JOIN domains ON domains.domain_id = ad.main_domain_id
                        LEFT JOIN minus_words mw on mw.mw_id = p.mw_id
                        LEFT JOIN images im on b.bid = im.bid
                        LEFT JOIN banner_images_formats bif on bif.image_hash = im.image_hash
                        JOIN banner_images bim on b.bid = bim.bid
                        LEFT JOIN banner_images_formats bim_f ON bim.image_hash = bim_f.image_hash
                        JOIN campaigns c on c.cid = p.cid
                        JOIN users u on u.uid = c.uid
                        LEFT JOIN adgroups_mobile_content gmc on p.pid = gmc.pid AND p.adgroup_type = 'mobile_content'
                        LEFT JOIN banners_mobile_content bmc on bmc.bid = b.bid AND b.banner_type = 'mobile_content'
                        LEFT JOIN banners_performance bperf on bperf.bid = b.bid
                        LEFT JOIN perf_creatives perfc on perfc.creative_id = bperf.creative_id
                        LEFT JOIN banner_display_hrefs bdh ON bdh.bid = b.bid
                        LEFT JOIN banner_resources br on b.bid = br.bid
                        LEFT JOIN catalogia_banners_rubrics cbr on cbr.bid = b.bid
                        LEFT JOIN banner_permalinks bpml ON bpml.bid = b.bid AND bpml.permalink_assign_type = 'manual'
                        LEFT JOIN adgroups_content_promotion acp ON acp.pid = p.pid AND p.adgroup_type = 'content_promotion'
                        LEFT JOIN banners_content_promotion bcp ON bcp.bid = b.bid
                        LEFT JOIN content_promotion cp ON cp.id = bcp.content_promotion_id
                    ",
                    WHERE => {
                        "bim.$ids_key" => $not_selected_banner_ids
                        }
                    ]
                );
            }

            foreach my $banner_id (grep { !exists $banners->{$_} } @$chunk_banner_ids) {
                # баннеры были удалены
                $BANNERS_CACHE->{$ids_key}->{$banner_id} ||= {};
            }
            
            # adgroup tags
            my $banner_id2pid = { map { $_ => $banners->{$_}->{pid} } grep {defined $banners->{$_}->{pid}} keys %$banners };
            my $pid2tags = get_groups_tags(pid => [uniq values %$banner_id2pid]);
            while (my ($banner_id, $pid) = each %$banner_id2pid) {
                $banners->{$banner_id}->{tags} = {map {$_ => 1} @{$pid2tags->{$pid} // []}};
            }

            my $pid2banner_ids = {};
            while (my($banner_id, $pid) = each %$banner_id2pid) {
                push @{$pid2banner_ids->{$pid}}, $banner_id;
            }
            # Выбираем фразы
            if (keys %$pid2banner_ids) {
                my $phrases = get_all_sql(PPC(shard => $shard), [
                                                "SELECT bi.pid, bi.PhraseID, bi.phrase, IFNULL(auct.rank, 2) as rank
                                                   FROM bids bi
                                                        left join bs_auction_stat auct on auct.pid = bi.pid and auct.PhraseID = bi.PhraseID ",
                                                  WHERE => { 'bi.pid' => [keys %$pid2banner_ids] }
                                                  ] );
                my %phrases_by_pid = ();
                foreach my $ph (@$phrases) {
                    push @{$phrases_by_pid{$ph->{pid}}}, $ph;
                }
                undef $phrases;

                while (my ($pid, $phrases) = each %phrases_by_pid) {
                    for my $banner (@{$banners}{@{$pid2banner_ids->{$pid} || []}}) {
                        hash_merge $banner, add_banner_template_fields($banner, $phrases);

                        for my $ph ( @$phrases ) {
                            $banner->{phrases_hash}->{$ph->{PhraseID}} = $ph;
                        }
                    }
                }
            }

            my $sl_set_id2banner_ids = {};
            foreach (values %$banners) {
                push @{$sl_set_id2banner_ids->{$_->{sitelinks_set_id}}}, $_->{$ids_key} if $_->{sitelinks_set_id};
            }

            if (keys %$sl_set_id2banner_ids) {
                my $sl_set_id2sitelinks = Sitelinks::get_sitelinks_by_set_id_multi([keys %$sl_set_id2banner_ids]);
                while (my ($sitelinks_set_id, $sitelinks) = each %$sl_set_id2sitelinks) {
                    foreach my $banner_id (@{$sl_set_id2banner_ids->{$sitelinks_set_id}}) {
                        $banners->{$banner_id}->{sitelinks} = $sitelinks;
                    }
                }
            }

            while (my ($banner_id, $banner) = each %$banners) {
                $banner->{geo_ids} = $banner->{geo} = GeoTools::modify_translocal_region_before_show(
                    $banner->{geo},
                    $O{translocal_params}
                );
                $banner->{geo_names} = get_geo_names($banner->{geo}, ', ');
                $banner->{is_template_banner_href} = 1 if ($banner->{href}||'') =~ /$TEMPLATE_METKA/si;

                # У графических объявлений и объявлений охватного продукта нет заголовка и текста, в базе лежат технические, их показывать нельзя
                $banner->{title} = $banner->{body} = '' if $banner->{real_banner_type} =~ /^(image_ad|cpm_banner|cpm_outdoor|cpm_indoor|cpm_audio)$/;

                # Для текстовых баннеров с картинкой mds_group_id хранится в image_mds_group_id, а не в imad_mds_group_id
                $banner->{mds_group_id} = $banner->{image_mds_group_id} if $banner->{real_banner_type} eq 'text';

                if ($banner->{real_banner_type} ne 'dynamic') {
                    delete $banner->{adgroup_main_domain};
                }

                $banner->{disable_videomotion} = Models::Banner::is_videomotion_disabled($banner) ? 1 : 0;
                $BANNERS_CACHE->{$ids_key}->{$banner_id} = $banner;
            }

            my $mobile_content_banners = [grep {$_->{adgroup_type} eq 'mobile_content'} values %$banners];
            if (@$mobile_content_banners) {
                my $mobile_content = Direct::AdGroups2::MobileContent->get_mobile_content_by(adgroup_id => [map {$_->{pid}} @$mobile_content_banners]);
                for my $banner (@$mobile_content_banners) {
                    $banner->{reflected_attrs} = [split(/,/, $banner->{reflected_attrs})];
                    $banner->{mobile_content} = exists $mobile_content->{$banner->{pid}}
                                                ? $mobile_content->{$banner->{pid}}->to_template_hash
                                                : {};
                }
            }

            # заполняем видео
            Direct::BannersResources::add_video_resources_to_banners([values %$banners]);

            my $cache = \{};
            for my $banner (values %$banners) {
                my ($field, $prefix, $class, $meth);
                if ($banner->{adgroup_type} eq 'performance' ||
                    # TODO cpm_video
                    ($banner->{adgroup_type} ne 'cpm_banner' && $banner->{real_banner_type} eq 'image_ad' && defined $banner->{perfc_creative_id})
                ) {
                    $field = 'creative';
                    $prefix = 'perfc_';
                    if ($banner->{real_banner_type} eq 'image_ad') {
                        $class = 'Direct::Model::CanvasCreative';
                        $meth = 'to_template_hash';
                        $banner->{"${prefix}moderate_info"} = '';  # иначе to_template_hash падает
                        if ($banner->{perfc_creative_type} eq 'html5_creative') {
                            $class = 'Direct::Model::CanvasHtml5Creative';
                        }
                    } else {
                        $class = 'Direct::Model::Creative';
                        $meth = 'to_hash';
                    }
                } elsif ($banner->{real_banner_type} eq 'cpm_banner' || $banner->{real_banner_type} eq 'cpc_video'
                    || $banner->{real_banner_type} eq 'cpm_outdoor' || $banner->{real_banner_type} eq 'cpm_indoor'
            || $banner->{real_banner_type} eq 'cpm_audio') {
                    $field = 'creative';
                    $prefix = 'perfc_';
                    if ($banner->{perfc_creative_type} eq 'canvas') {
                        $class = 'Direct::Model::CanvasCreative';
                        $meth = 'to_template_hash';
                        $banner->{"${prefix}moderate_info"} = '';
                    } elsif ($banner->{perfc_creative_type} eq 'html5_creative') {
                        $class = 'Direct::Model::CanvasHtml5Creative';
                        $meth = 'to_template_hash';
                        $banner->{"${prefix}moderate_info"} = '';
                    } elsif ($banner->{perfc_creative_type} eq 'video_addition') {
                        $class = 'Direct::Model::VideoAddition';
                        $meth = 'to_template_hash';
                        $banner->{"${prefix}moderate_info"} = '';
                    } else {
                        $class = 'Direct::Model::Creative';
                        $meth = 'to_hash';
                    }
                } elsif ($banner->{real_banner_type} eq 'image_ad') {
                    $field = 'image_ad';
                    $prefix = 'imad_';
                    $class = 'Direct::Model::ImageFormat';
                    $meth = 'to_stat_hash';
                    $banner->{"${prefix}image_type"} = 'image_ad';
                    $banner->{"${prefix}namespace"} = 'direct-picture';
                } elsif ($banner->{perfc_creative_type} && $banner->{perfc_creative_type} eq 'video_addition') {
                    $field = 'video_resources';
                    $prefix = 'perfc_';
                    $class = 'Direct::Model::VideoAddition';
                    $meth = 'to_template_hash';
                } else {
                    next;
                }

                $banner->{$field} = $class->from_db_hash(
                    { map { $_ => delete $banner->{$_} } grep { /^$prefix/ } keys %$banner },
                            $cache,
                            prefix => $prefix,
                )->${\$meth};
            }

            # заполняем уточнения
            Direct::BannersAdditions::add_callouts_to_banners([values %$banners]);
        }
    }

    my $result_banners = {};
    foreach my $banner_id (@$banner_ids) {
        next unless $BANNERS_CACHE->{$ids_key}->{$banner_id};
        $result_banners->{$banner_id} = dclone($BANNERS_CACHE->{$ids_key}->{$banner_id});
    }
    return $result_banners;
}

=head2 clear_banner_info_cache

    Очищает кеш с информацией о баннерах    

=cut

sub clear_banner_info_cache {
    $BANNERS_CACHE = undef; 
}

=head2 format_date

    Форматирование даты с учётом аггрегации

=cut

sub format_date {
    my ( $date, $dateagg, %O ) = @_;

    my @month_abbr   = iget_noop("янв", "фев", "мар", "апр", "май", "июн", "июл", "авг", "сен", "окт", "ноя", "дек");
    my @month_full   = iget_noop("Январь", "Февраль", "Март", "Апрель", "Май", "Июнь", "Июль", "Август", "Сентябрь", "Октябрь", "Ноябрь", "Декабрь");

    $O{period_separator} //= "\x{2013}"; # тире
    $O{month_text} //= 'abbr';

    my $dt = date(normalize_date($date));
    
    if ( $dateagg && $dateagg eq 'week' ) {
        $dt->truncate(to => 'week');
        return $dt->strftime("%d.%m.%y") . " $O{period_separator} " . $dt->add(days => 6)->strftime("%d.%m.%y");
    } elsif ( $dateagg && $dateagg eq 'month' ) {
        my $month_texts = $O{month_text} eq 'abbr' ? \@month_abbr : \@month_full;
        return iget($month_texts->[$dt->month - 1])." ".$dt->year;
    } elsif ( $dateagg && $dateagg eq 'quarter' ) {
        return iget("%s квартал %d", $dt->quarter, $dt->year);
    } elsif ( $dateagg && $dateagg eq 'year' ) {
        return iget("%d г.", $dt->year);
    } else {
        return $dt->dmy('.');
    }
}

=head2 format_date_pdf

    Форматирование даты с учётом аггрегации для Pdf-отчётов

=cut

sub format_date_pdf {
    my ( $date, $dateagg ) = @_;
    return format_date($date, $dateagg, period_separator => '--', month_text => 'full');
}

=head2 get_date_subperiod 

    В зависимости от периода группировки, общего периода, и даты из него - возвращает начало и окончание подпериода в который входит заданная дата

=cut

sub get_date_subperiod {
    my ($date, $dateagg, $period_from, $period_to) = @_;

    state %_dates_cache;
    for ($period_from, $period_to) {
        $_dates_cache{$_} //= date(normalize_date($_))->ymd();
    }

    my ($date_from, $date_to) = ($period_from, $period_to) = map { $_dates_cache{$_} } ($period_from, $period_to);

    if ($dateagg && $dateagg ne 'none') {
        my $dt = date(normalize_date($date));

        if ( $dateagg && $dateagg eq 'week' ) {
            $date_from = $dt->truncate(to => 'week');
            $date_to = $date_from->clone()->add(days => 6);
        } elsif ( $dateagg && $dateagg eq 'month' ) {
            $date_from = $dt->truncate(to => 'month');
            $date_to = $date_from->clone()->add(months => 1)->subtract(days => 1);
        } elsif ( $dateagg && $dateagg eq 'quarter' ) {
            $date_from = $dt->subtract(days => $dt->day_of_quarter()-1);
            $date_to = $date_from->clone()->add(months => 3)->subtract(days => 1);
        } elsif ( $dateagg && $dateagg eq 'year' ) {
            $date_from = $dt->truncate(to => 'year');
            $date_to = $date_from->clone()->add(years => 1)->subtract(days => 1);
        } else {
            $date_from = $date_to = $dt;
        }
        $date_from = maxstr $date_from->ymd(), $period_from;
        $date_to = minstr $date_to->ymd(), $period_to;
    }

    return ($date_from, $date_to);
}

=head2 get_pages_info

    Вернуть хеш с необходимой информацией о площадках
    Параметры:
        $pages - 123|[123,222,333] - PageID площадок
        $cache - hashref для кеширования (опциональный)
                 если указан - возвращается все что накопилось в кеше, а не только информация по запрошенным $pages

=cut

sub get_pages_info
{
    my ($pages, $cache) = @_;
    $pages = [$pages] unless ref($pages);

    my $pages_to_process = $cache ? [grep { !exists $cache->{$_} } @$pages] : $pages;
    my $processed_pages_info = {};
    if (@$pages_to_process) {
        $processed_pages_info =  get_hashes_hash_sql(PPCDICT, [" 
                                               select PageID
                                                    , group_nick as page_group
                                                    , sorting as page_sorting
                                                    , name as page_name
                                                    , domain as page_domain
                                                    , TargetType
                                                 from pages", 
                                                where => {PageID => $pages_to_process}] );
        foreach my $p (@$pages_to_process) {
            if (exists $processed_pages_info->{$p} && $processed_pages_info->{$p}->{page_group}) {
                $processed_pages_info->{'pg_' . $processed_pages_info->{$p}->{page_group}} = $processed_pages_info->{$p};
            }
        }
    }
    if ($cache) {
        hash_merge $cache, $processed_pages_info;
        return $cache;
    } else {
        return $processed_pages_info;
    }
}

=head2 get_orders_goal_ids

    По списку идентификаторов заказов получаем цели, по которым были достижения

=cut

sub get_orders_goal_ids {
    my ($order_ids) = @_;

    return get_one_column_sql(
        PPC(OrderID => $order_ids),
        ["SELECT distinct g.goal_id FROM camp_metrika_goals g JOIN campaigns c USING (cid)", WHERE => { "c.OrderID" => SHARD_IDS }]
    );
}

=head2 orders_goals

    По cid или OrderID возвращаем список целей метрики, находим и группируем по составным целям если нужно
    $vars->{goals_list} = DBStat::orders_goals(OrderID => $OrderID);

    результат:
    [
        {goal_id => 125, name => 'goal name',  goal_type => 'step',   status => 'Active', parent_goal_id => undef, subgoal_index => 0, is_meaningful_goal => 1},
        {goal_id => 124, name => ' - name',    goal_type => 'url',    status => 'Active', parent_goal_id => 123,   subgoal_index => 1},
        {goal_id => 125, name => ' - Шаг 2',   goal_type => 'url',    status => 'Active', parent_goal_id => 123,   subgoal_index => 2},
        {goal_id => 126, name => 'goal name3', goal_type => 'number', status => 'Deleted, parent_goal_id => undef, subgoal_index => 0, is_meaningful_goal => 1},
    ]

    отладка:
    perl -ME -MDBStat -e 'p DBStat::orders_goals(848769)'

    Опции:
    is_need_mobile_content_goals - добавлять ли в список цели для РМП

=cut

sub orders_goals {
    my ($key, $oids, %O) = @_;

    my $goal_ids = [uniq @{get_one_column_sql(PPC($key => $oids), ["
                                                   SELECT distinct g.goal_id 
                                                     FROM camp_metrika_goals g
                                                     JOIN campaigns c USING (cid)",

                                                    WHERE => { "c.$key" => SHARD_IDS }] )} ];

    my $goals = CampaignTools::get_metrika_goals(where => {goal_id => $goal_ids});
    my @parent_goal_ids = uniq
                          map {$_->{parent_goal_id}}
                          grep {$_->{parent_goal_id}}
                          values %$goals;


    # находим составные цели и их остальные под-цели
    my $other_complex_goals = CampaignTools::get_metrika_goals(where => {goal_id => \@parent_goal_ids}
                                                 , OR => {parent_goal_id => \@parent_goal_ids}
                                               );

    hash_merge $goals, $other_complex_goals;

    # находим ключевые цели по cid, необходимо для формирования флага is_meaningful_goal => 1 в goals_list
    # который используется в форме "данные по цели" во фронте

    my @meaningful_goals = @{get_one_column_sql(PPC($key => $oids), ["
                    SELECT co.meaningful_goals
                    FROM campaigns c JOIN camp_options co on co.cid = c.cid",
        WHERE => { "c.$key" => SHARD_IDS }] )};
    my @meaningful_goal_ids;
    for my $mgoals (@meaningful_goals) {
        next unless $mgoals;
        push(@meaningful_goal_ids, map { $_->{goal_id} } @{decode_json($mgoals)});
    }

    # https://st.yandex-team.ru/DIRECT-108651
    my $cids;
    if ($key eq 'OrderID'){
        my $order2cid = get_orderid2cid(OrderID => $oids);
        $cids = [values %$order2cid]
    } else {
        $cids = ref $cids eq 'ARRAY' ? $oids : [$oids];
    }

    # $cids может быть пустым
    my $is_internal_ad = scalar @$cids > 0 ? camp_kind_in(cid => $cids->[0], 'internal') : 0;

    my @goals_with_camp_counters;
    my $client_id = get_clientid('cid' => $cids->[0]);
    my $has_goals_only_with_campaign_counters = Client::ClientFeatures::has_goals_only_with_campaign_counters_used($client_id);
    my $is_unavailable_goals_allowed = Client::ClientFeatures::has_unavailable_goals_allowed($client_id)
                                       && Client::ClientFeatures::is_send_unavailable_goals_in_perl_enabled($client_id);

    if ($has_goals_only_with_campaign_counters && $O{goals_for_stat} == 1) {
        my $campaign_counter_ids = MetrikaCounters::get_counter_ids_from_db_by_cids($cids);
        my $client_counters = MetrikaCounters::get_client_id_reps_counters_by_ids($client_id, $campaign_counter_ids,
            skip_errors => $O{skip_errors}, is_internal_ad => $is_internal_ad);

        if (Client::ClientFeatures::has_goals_from_all_orgs_allowed($client_id)) {
            @$client_counters = uniq @$client_counters, @{MetrikaCounters::get_spav_counter_ids_from_db_by_cids($cids)};
        }

        my %client_counters_hash = map { $_ => 1 } @{$client_counters};
        my $counters_to_fetch_goals = $client_counters;
        if ($is_unavailable_goals_allowed) {
            my @unavailable_counter_ids = grep { !exists $client_counters_hash{$_} } @$campaign_counter_ids;
            if (@unavailable_counter_ids) {
                my $allow_to_use_by_counter_id = get_allow_use_counter_without_access_by_id(\@unavailable_counter_ids);
                my @allowed_unavailable_counter_ids =
                    grep { $allow_to_use_by_counter_id->{$_} } keys %$allow_to_use_by_counter_id;
                @$counters_to_fetch_goals = uniq @$client_counters, @allowed_unavailable_counter_ids;
            }
        }

        # цели по номерам счетчиков
        my $counters_goals = MetrikaCounters::get_counters_goals($counters_to_fetch_goals, skip_errors => 1, get_steps => 1);
        push(@goals_with_camp_counters, map {$_->{goal_id}} map {@{$_ || []}} values %$counters_goals);

        # виртуальные цели ecommerce добавляем отдельно чтобы при фильтрации по счетчикам не потеряли из списка доступных целей
        for my $goal_id (keys %$goals) {
            if ($goals->{$goal_id}->{goal_type} eq 'ecommerce'
                && exists($client_counters_hash{$goals->{$goal_id}->{ecommerce_counter_id}})) {
                push @goals_with_camp_counters, $goal_id;
            }
        }
    }
    my $has_in_app_mobile_targeting_allowed = Client::ClientFeatures::has_in_app_mobile_targeting_allowed($client_id);

    my @mobile_goals;
    if ($has_in_app_mobile_targeting_allowed) {
        push @mobile_goals, available_inapp_mobile_goals($client_id);
    }

    my %mobile_goals_hash = map { $_->{goal_id} => $_ } @mobile_goals;

    # считаем статус счетчика и добавляем цели если их нет в metrika_goals, выносим подцели
    my %subgoals;
    my %mgoal_ids = map { $_ => 1 } @meaningful_goal_ids;
    for my $goal_id (uniq @$goal_ids, keys %$goals) {
        if (exists $goals->{$goal_id}) {
            $goals->{$goal_id}->{status} = $goals->{$goal_id}->{counter_status} eq 'Active' && $goals->{$goal_id}->{goal_status} eq 'Active'
                                           ? 'Active' : 'Deleted';
            delete $goals->{$goal_id}->{counter_status};
            delete $goals->{$goal_id}->{goal_status};

            if (exists $mgoal_ids{$goal_id}) {
                $goals->{$goal_id}->{is_meaningful_goal} = '1';
            }

            if ($goals->{$goal_id}->{parent_goal_id}) {
                my $parent_goal_id = $goals->{$goal_id}->{parent_goal_id};
                push @{$subgoals{$parent_goal_id}}, delete $goals->{$goal_id};
            }
        }  else {
            unless ($MOBILE_APP_SPECIAL_GOALS{$goal_id} ||  $mobile_goals_hash{$goal_id}) {
                $goals->{$goal_id} = {
                    name => "$goal_id",
                    goal_id => $goal_id,
                    status => 'Deleted'
                };
            }
        }
    }

    # сортируем под-цели по индексу, меняем имя если нужно
    for my $goal_id (keys %subgoals) {
        for my $row (@{ $subgoals{$goal_id} }) {
            $row->{name} = iget('Шаг') . ' ' . $row->{subgoal_index} if ! defined $row->{name} || $row->{name} eq '';
            $row->{name} = (chr(0xA0) x 2) . " • " . $row->{name}; # chr(0xA0) - &nbsp; который не нужно искейпить (нужно для отступа в select-е для составных целей)
        }
        $subgoals{$goal_id} = [sort {$a->{subgoal_index} <=> $b->{subgoal_index}}
                               @{$subgoals{$goal_id}}
                              ];
    }

    if ($is_internal_ad) {
        # для внутренней рекламы оставляем цели из campaigns_internal.rotation_goal_id
        @goals_with_camp_counters = @{get_one_column_sql(PPC($key => $oids), [ "
            SELECT distinct c_internal.rotation_goal_id
            FROM campaigns_internal c_internal
            JOIN campaigns c USING (cid)",
            WHERE => { "c.$key" => SHARD_IDS, "c_internal.rotation_goal_id__is_not_null" => 1 } ])};
    }

    # сортируем цели по статусу и по id, добавляем под-цели, формируем плоский список
    my @result_goals;
    my %goal_ids_with_camp_counters = map { $_ => 1 } @goals_with_camp_counters;
    for my $goal_id (sort {$goals->{$a}->{status} cmp $goals->{$b}->{status} || $a <=> $b} keys %$goals) {
        if ($has_goals_only_with_campaign_counters && exists $goal_ids_with_camp_counters{$goal_id}
            || !$has_goals_only_with_campaign_counters
            || !$O{goals_for_stat})
        { # https://st.yandex-team.ru/DIRECT-108651
            push @result_goals, $goals->{$goal_id};
            if (exists $subgoals{$goal_id}) {
                push @result_goals, @{ $subgoals{$goal_id} }
            }
        }
    }

    if ($O{is_need_mobile_content_goals}) {
        my $mobile_content_goals = mobile_content_orders_goals();
        push @result_goals, @$mobile_content_goals;
    } elsif ($is_internal_ad) {
        # в статистике для внутренней рекламы цель c goal_id = 3 должна быть всегда доступна
        my $mobile_content_goals = mobile_content_orders_goals();
        for my $goal (@$mobile_content_goals) {
            if ($goal->{goal_id} eq '3') {
                push @result_goals, $goal;
            }
        }
    }

    if ($has_in_app_mobile_targeting_allowed){
        push @result_goals, @mobile_goals;
    }

    return \@result_goals;
}

=head2 available_inapp_mobile_goals

    Возвращаем список целей для всех доступных мобильных приложений

=cut

sub available_inapp_mobile_goals {
    my ($client_id) = @_;
    my @result_goals;
    push @result_goals, client_inapp_mobile_goals($client_id);
    push @result_goals, shared_inapp_mobile_goals($client_id);
    return @result_goals
}

=head2 client_inapp_mobile_goals

    Возвращаем список целей для мобильных приложений клиента

=cut

sub client_inapp_mobile_goals {
    my ($client_id) = @_;
    my @result_goals;
    push @result_goals, mobile_app_external_tracker_goals($client_id);
    push @result_goals, mobile_app_app_metrika_goals($client_id);
    return @result_goals
}

=head2 shared_inapp_mobile_goals

    Возвращаем список целей для чужих, доступных клиенту мобильных приложений

=cut

sub shared_inapp_mobile_goals {
    my ($client_id) = @_;
    my $owners_of_mobile_apps = get_all_sql(PPC(ClientID => $client_id), [
        'SELECT client_id_to as client_id FROM reverse_clients_relations', WHERE => { client_id_from => $client_id }
    ]);

    my @result_goals;
    for my $owner (@$owners_of_mobile_apps) {
        push @result_goals, client_inapp_mobile_goals($owner->{client_id})
    }
    return @result_goals
}

=head2 mobile_app_external_tracker_goals

    Возвращаем список целей для событий от внешних трекеров

=cut

sub mobile_app_external_tracker_goals {
    my ($client_id) = @_;

    my $external_tracker_goals = get_all_sql(PPC(ClientID => $client_id),
        ['SELECT g.goal_id, g.mobile_app_id, g.event_name, g.custom_name, m.name as mobile_app_name
            FROM mobile_apps m
            JOIN mobile_app_goals_external_tracker g USING (mobile_app_id)',
            WHERE => {ClientID => $client_id, 'g.is_deleted' => 0}]);

    my @result_goals;
    for my $goal (@$external_tracker_goals) {
        my $result_goal = {
            goal_id => $goal->{goal_id},
            name => $goal->{mobile_app_id} . ' ' . $goal->{mobile_app_name} . ': '
                . ($goal->{custom_name} ? $goal->{custom_name} : $goal->{event_name}),
            status => 'Active'
        };
        push @result_goals, $result_goal;
    }

    return @result_goals;
}

=head2 mobile_app_app_metrika_goals

    Возвращаем список целей для событий от аппметрики

=cut

sub mobile_app_app_metrika_goals {
    my ($client_id) = @_;

    my $mobile_apps = get_hash_sql(PPC(ClientID => $client_id),
        'SELECT app_metrika_application_id as id, name FROM mobile_apps WHERE ClientID = ? AND app_metrika_application_id IS NOT NULL', $client_id);
    my @mobile_app_ids = keys %$mobile_apps;

    my $app_metrika_goals = get_all_sql(PPCDICT(),
        ['SELECT goal_id, app_metrika_application_id as app_id, event_type, event_subtype, event_name, custom_name
            FROM mobile_app_goals_appmetrika',
            WHERE => {app_metrika_application_id => \@mobile_app_ids, 'is_deleted' => 0}]);

    my @result_goals;
    for my $goal (@$app_metrika_goals) {
        my $result_goal = {
            goal_id => $goal->{goal_id},
            name => $goal->{app_id} . ' ' . $mobile_apps->{$goal->{app_id}} . ': ' .
                ($goal->{custom_name} ? $goal->{custom_name} :
                    ($goal->{event_type} eq 'CLIENT' ? $goal->{event_name} : $goal->{event_type} . '_' . $goal->{event_subtype})),
            status => 'Active'
        };
        push @result_goals, $result_goal;
    }

    return @result_goals;
}

=head2 mobile_content_orders_goals

    Возвращаем список целей метрики аналогично методу orders_goals, нодля РМП кампаний
    Для них у нас соступен фиксироваррый тут %MOBILE_APP_SPECIAL_GOALS список целей

=cut

sub mobile_content_orders_goals {

    my $goals;
    my @result_goals;
    for my $goal_id (keys %MOBILE_APP_SPECIAL_GOALS) {
        $goals->{$goal_id} = {
                name => iget($MOBILE_APP_SPECIAL_GOALS{$goal_id}->{name}),
                goal_id => $goal_id,
                status => 'Active',
        };  
    }
    my @goal_ids = sort {$a <=> $b} keys %MOBILE_APP_SPECIAL_GOALS;    
    for my $goal_id (@goal_ids) {
        if ($goal_id == APP_INSTALL()) {
            unshift @result_goals, $goals->{$goal_id};
        } else {
            push @result_goals, $goals->{$goal_id};
        } 
    } 
    return \@result_goals;
}

=head2 operator_has_mobile_app_special_goal_permission

Есть ли у оператора доступ (по роли или фиче) к специальной цели для РМП

=cut

sub operator_has_mobile_app_special_goal_permission {
    my ( $goal_id, %opts ) = @_;

    my $is_manager = $opts{operator_is_manager};
    my $is_internal_role = $opts{operator_has_internal_role};
    my $is_internal_ad_role = $opts{operator_has_internal_ad_role};
    my $client_id = $opts{operator_client_id};

    my $permissions = $MOBILE_APP_SPECIAL_GOALS{$goal_id}->{permissions};

    if ( $permissions->{everybody} ) {
        return 1;
    }

    if ( $permissions->{manager} && $is_manager ) {
        return 1;
    }

    if ( $permissions->{internal_roles} && $is_internal_role ) {
        return 1;
    }

    if ( $permissions->{internal_ad_roles} && $is_internal_ad_role ) {
        return 1;
    }

    if ( $permissions->{features} && @{ $permissions->{features} } ) {
        my $features = $opts{only_features} ? xisect($permissions->{features}, $opts{only_features}) : $permissions->{features};
        for my $feature (@$features) {
            if ( Client::ClientFeatures::has_access_to_new_feature_from_java( $client_id, $feature ) ) {
                return 1;
            }
        }
    }

    return 0;
}

=head2 get_valuable_goal_id

    Получить значимый для статистики ID цели. Для обычной цели - это ее собственный id, а для составной - id последнего шага.

=cut

sub get_valuable_goal_id {
    my ($goal_id) = @_;

    my $subgoals = CampaignTools::get_metrika_goals(where => {parent_goal_id => $goal_id});
    unless ($subgoals && %$subgoals) {
        return $goal_id;
    }

    my @subgoals_ordered = sort {$b->{subgoal_index} <=> $a->{subgoal_index}}
                           grep {($_->{goal_status} // '') eq 'Active'}
                           values %$subgoals;
    return $subgoals_ordered[0]->{goal_id} // $goal_id;
}

=head2 get_region_info

=cut

sub get_region_info {
    my $region = shift||0;
    my $region_field = shift || 'region';
    my $info = {};
    my $reg_defined = $region && exists $geo_regions::GEOREG{$region};
    my $name = $reg_defined
        ? $geo_regions::GEOREG{$region}->{GeoTools::get_geo_name_field()}
        : iget('не определен');
    $info->{$region_field} = $region;
    if ( $reg_defined && @{$geo_regions::GEOREG{$region}->{childs}} ) {
        $info->{"${region_field}_name"} = iget('%s (более точно регион не определён)', $name);
        $info->{"${region_field}_not_exactly_detected"} = 1;
        $info->{"original_${region_field}_name"} = $name;
    } else {
        $info->{"${region_field}_name"} = $name;
    }
    return $info;
}

=head2 get_ret_cond_ids_by_name
    возвращает ссылку на массив ret_cond_id по названию условия и OrderID
=cut

sub get_ret_cond_ids_by_name {
    my ($ret_cond_name, $oids) = @_;

    return get_one_column_sql(PPC(OrderID => $oids), [
            "select rc.ret_cond_id
             from campaigns c
               join users u on u.uid = c.uid
               join retargeting_conditions rc on rc.ClientID = u.ClientID
            ", where => {'c.OrderID' => $oids, 'rc.condition_name' => $ret_cond_name}
        ]);
}

=head2 get_plus_regions_by_geo 
    по строке geo (допускается массив строк) возвращает массив плюс-регионов (с учетом минусов)
=cut

sub get_plus_regions_by_geo {
    my ($geo_str_list, $translocal_params) = @_;

    my @plus_regions = ();
    for my $geo_str (xflatten $geo_str_list) {
        push @plus_regions, map {  
                                    m/^\!(\d+)$/ ? $1 : get_geo_children($_, $translocal_params)
                                } @{GeoTools::convert_mixed_geo_to_plus_reg_list($geo_str, $translocal_params)};
    }
    return [uniq @plus_regions];
}

=head2 get_geo_ids_by_geo_text
    по строке geo (названия и номера регионов через запятую)
=cut

sub get_geo_ids_by_geo_text {
    my $geo_text = join ',', split /\s*,\s*/, $_[0] // '';
    $geo_text =~ s/^\s+//;
    $geo_text =~ s/\s+$//;

    my @geo_ids = grep { is_valid_int($_) } split ',', $geo_text;
    push @geo_ids, grep {$_} split ',', GeoTools::get_geo_numbers($geo_text);

    return \@geo_ids;
}

=head2 get_page_id_by_name
    по массиву названий площадок возвращает массив соответствующих номеров площадок по вхождению строки
=cut

sub get_page_id_by_name {
    my ($page_names) = @_;
    return get_one_column_sql(PPCDICT, "SELECT PageID FROM pages WHERE ". 
                                      join(' OR ', '0', map { 'name LIKE ' . sql_quote('%' . sql_quote_like_pattern($_) . '%') } xflatten $page_names));
}

=head2 get_page_id_by_name_exact
    по массиву названий площадок возвращает массив соответствующих номеров площадок, по точному совпадению
=cut

sub get_page_id_by_name_exact {
    my ($page_names) = @_;
    return get_one_column_sql(PPCDICT, ["SELECT PageID FROM pages WHERE", [name => $page_names]]);
}

=head2 get_page_id_by_group_nick
    по массиву идентификаторов групп площадок возвращает массив соответствующих номеров площадок (точное соответствие)
=cut

sub get_page_id_by_group_nick {
    my ($group_nick) = @_;
    return get_one_column_sql(PPCDICT, ["SELECT PageID FROM pages ". where => {group_nick => $group_nick}]);
}

=head2 convert_page_name_filter_to_ids($values, $op, $lang)

    Превращение подстроки названия площадки в список `PageID` с учётом того,
    что мы локализуем магическую площадку "Яндекс" в "Yandex" для некоторых локалей.

=cut
sub convert_page_name_filter_to_ids {
    my ($values, $op, $lang) = @_;

    my $name_to_magic = get_page_name_to_old_magic_translations($lang);
    my %add;
    my %exclude;
    for my $page_name (keys %$name_to_magic) {
        my $magic = $name_to_magic->{$page_name};
        next if lc($magic) eq lc($page_name);

        for my $val (@$values) {
            # если пользователь с английской локалью запросил фильтр "dex", то добавляем к списку `PageID` также те,
            # которые относятся к площадке "Яндекс"
            if (index(lc($page_name), lc($val)) != -1) {
                $add{$magic} = 1;
            }
            # если пользователь с английской локалью хочет отфильтровать площадки, содержащие "декс", нужно
            # убрать из фильтра `PageID`, соответствующие площадке "Яндекс", т.к. иначе у пользователя пропадёт строка
            # "Yandex"
            if (index(lc($magic), lc($val)) != -1) {
                $exclude{$magic} = 1;
            }
        }
    }
    my @page_ids = @{get_page_id_by_name($values)};
    if (%add) {
        @page_ids = uniq(@page_ids, @{get_page_id_by_name_exact([keys %add])});
    }
    if (%exclude) {
        @page_ids = @{xminus(\@page_ids, get_page_id_by_name_exact([keys %exclude]))};
    }
    return \@page_ids;
}

{
    my %lang_map_for_magic_pages = (
        'en' => 'en',
        'tr' => 'en',
    );
    # МОЛ присылает строчку "yandex", нам нужно нарисовать её как "Яндекс"/"Yandex"
    my %magic_page_names = (
        'yandex' => {
            'ru' => 'Яндекс',
            'en' => 'Yandex',
        },
        'yandexmain' => {
            'ru' => 'yandex.ru (главная страница)',
            'en' => 'yandex.ru (main page)',
        },
    );
    # Когда берём название площадки из своего словарика, преобразуем "Яндекс" в "Yandex" для английских локалей
    my %old_magic_page_names = (
        'Яндекс' => {
            'ru' => 'yandex.ru (главная страница)',
            'en' => 'Yandex',
        },
    );

=head2 get_magic_page_name_translations($lang)

    Возвращает мапу "магическая площадка" => "локализованное название" для указанной локали

=cut
    sub get_magic_page_name_translations {
        my ($lang) = @_;

        my $norm_lang = $lang_map_for_magic_pages{$lang} // 'ru';
        return {map { $_ => $magic_page_names{$_}{$norm_lang} } keys %magic_page_names};
    }

=head2 get_page_name_to_magic_translations($lang)

    Возвращает мапу "локализованное название" => "магическая площадка" для указанной локали

=cut
    sub get_page_name_to_magic_translations {
        my ($lang) = @_;

        my $norm_lang = $lang_map_for_magic_pages{$lang} // 'ru';
        return { 'yandex.ru' => 'yandexmain',  map { $magic_page_names{$_}{$norm_lang} => $_ } keys %magic_page_names};
    }

=head2 get_old_magic_page_name_translations($lang)

    Возвращает мапу "магическая площадка" => "локализованное название" для указанной локали
    для случаев, когда название площадки берётся из своего словарика

=cut
    sub get_old_magic_page_name_translations {
        my ($lang) = @_;

        my $norm_lang = $lang_map_for_magic_pages{$lang} // 'ru';
        return {map { $_ => $old_magic_page_names{$_}{$norm_lang} } keys %old_magic_page_names};
    }

=head2 get_page_name_to_old_magic_translations($lang)

    Возвращает мапу "локализованное название" => "магическая площадка" для указанной локали
    для случаев, когда название площадки берётся из своего словарика

=cut
    sub get_page_name_to_old_magic_translations {
        my ($lang) = @_;

        my $norm_lang = $lang_map_for_magic_pages{$lang} // 'ru';
        return {map { $old_magic_page_names{$_}{$norm_lang} => $_ } keys %old_magic_page_names};
    }
}

=head2 get_adgroups_info

    Выбирает из базы информацию про группы по указанному условию
    $adgroups_info = Stat::Tools::get_adgroups_info(cid => [$cid1, $cid2, ...]);
    $adgroups_info = Stat::Tools::get_adgroups_info(pid => $pids);
    $adgroups_info = Stat::Tools::get_adgroups_info(cid => $cids, pid => $pids, bid => $bids);
    $adgroups_info = Stat::Tools::get_adgroups_info(cid => [$cid1, $cid2, ...], group_name => 'mygroup1');
    $adgroups_info => {
        $pid => {
            ...
            pid => $pid,
            group_name => $group_name,
            ...
        }
    }

=cut

sub get_adgroups_info {
    my (%adgroup_cond) = @_;

    my %filter_cond;
    for my $field (qw(cid bid pid group_name)) {
        next unless defined $adgroup_cond{$field};
        $filter_cond{$field} = delete $adgroup_cond{$field};
    }
    if (!(grep { $_ ne 'group_name' } keys %filter_cond)) {
        die "no supported keys in get_adgroups_info condition";
    } elsif (%adgroup_cond) {
        my $excess_cond_keys = join ', ', keys %adgroup_cond;
        die "excess keys in get_adgroups_info condition: $excess_cond_keys";
    }

    my %get_pure_groups_cond = (
        adgroup_types => [qw(base dynamic mobile_content performance mcbanner cpm_banner cpm_video content_promotion_video content_promotion cpm_outdoor cpm_indoor cpm_audio cpm_geoproduct internal cpm_yndx_frontpage)],
        %filter_cond,
    );
    my $adgroups_data = Models::AdGroup::get_pure_groups(\%get_pure_groups_cond, {only_pid => 1});
    Models::AdGroup::calc_banners_quantity($adgroups_data);
    return { map {$_->{adgroup_id} => $_} @$adgroups_data };
}

=head2 add_content_targetings_goal_ids_text

    Добавить названия жанров и категорий в статистику в человекочитаемом виде

=cut

sub add_content_targetings_goal_ids_text {
    my $row = shift;

    my $goal_id = $row->{content_targeting};
    return unless defined $goal_id;
    $goal_id = 0 if $goal_id eq "";
    state $goal_ids_hash //= _get_all_goal_ids_names();

    if (!$goal_ids_hash->{$goal_id}->{targeting_name}) {
        my $i_goal_id = $goal_id;
        my @names = ();
        do {
            my $goal = $goal_ids_hash->{$i_goal_id};
            push @names, iget($goal->{name});
            $i_goal_id = $goal->{parent_goal_id};
        } while($i_goal_id > 0);
        $goal_ids_hash->{$goal_id}->{targeting_name} = join " > ", reverse @names;
    }

    $row->{content_targeting_name} = $goal_ids_hash->{$goal_id}->{targeting_name};
}

sub _get_all_goal_ids_names() {
    my $goal_ids = get_hashes_hash_sql(PPCDICT, 
            ["SELECT goal_id, parent_goal_id, name FROM crypta_goals",
            WHERE => { crypta_goal_type => [qw/content_genre content_category/] }]);
    $goal_ids->{0} = {goal_id => 0, parent_goal_id => 0, name => "Не определено"};
    return $goal_ids;
}

=head2 simple_dict

    Возвращает простую (старую) структуру словаря для словарных полей статистики.
    входные параметры:
        $dict - словарь в таком же виде как в Stat::Const
    выходные параметры:
        $simple_dict - простой словарь в формате
            {desktop => [0,3],
             mobile => 2,
             ...}

=cut

sub simple_dict {
    my $dict = shift;

    my $simple_dict = {};
    for my $k (keys %$dict) {
        $simple_dict->{$k} = ref $dict->{$k} eq 'HASH' ? $dict->{$k}->{bs} : $dict->{$k};
    }

    return $simple_dict;
}

=head2 consumer_dict_by_name

    По нозванию словаря и потребителю, возвращает подготовленный к использованию в МОЛ словарь
    входящие параметры:
        $dict_name - название словаря (DEVICE_TYPES|DETAILED_DEVICE_TYPES)
        $consumer - (api|web_ui|default) - потребитель
    выходные значения:
        $prepared_dict - ссылка на "подготовленный" словарь
        %O - именованные параметры
            use_text_keys => 0|1, если 1 - то в возвращаемом словаре ключами являются текстовые константы "потребителя"
                                  если 0 - то ключами являются целые чила соответствующие лексикографической сортировке текстовых ключей
                                  по-умолчанию 0, чтоб не раздувать размер данных в CH БК (при группировке/сортировке)
        Подробней про форматы словарей - в POD к _prepare_dict

=cut

sub consumer_dict_by_name {
    my ($dict_name, $consumer, %O) = @_;
    $consumer //= 'default';

    state $dicts = {DEVICE_TYPES => {src => \%DEVICE_TYPES},
                    CLICK_PLACES => {src => \%CLICK_PLACES},
                    GENDERS      => {src => \%GENDERS},
                    AGES         => {src => \%AGES},
                    DETAILED_DEVICE_TYPES => {src => \%DETAILED_DEVICE_TYPES},
                    CONNECTION_TYPES => {src => \%CONNECTION_TYPES},
                    BS_PHRASE_STATUSES => {src => \%BS_PHRASE_STATUSES},
                    BROADMATCH_TYPES => {src => \%BROADMATCH_TYPES},
                    CAMP_TYPES   => {src => \%CAMP_TYPES},
                    BANNER_TYPES   => {src => \%BANNER_TYPES},
                    POSITION_TYPES => {src => \%POSITION_TYPES},
                    TARGET_TYPES => {src => \%TARGET_TYPES},
                    CONTEXT_TYPES => {src => \%CONTEXT_TYPES},
                    CRITERION_TYPES => {src => \%CRITERION_TYPES},
                    CONTEXT_COND_TYPES => {src => \%CONTEXT_COND_TYPES},
                    BANNER_IMAGE_TYPES => {src => \%BANNER_IMAGE_TYPES},
                    ATTRIBUTION_MODEL_TYPES => {src => \%ATTRIBUTION_MODEL_TYPES},
                    MATCH_TYPES => {src => \%MATCH_TYPES},
                    INVENTORY_TYPES => {src => \%INVENTORY_TYPES},
                    BS_TURBO_PAGE_TYPES => {src => \%BS_TURBO_PAGE_TYPES},
                    TARGETING_CATEGORIES => {src => \%TARGETING_CATEGORIES},
                    PRISMA_INCOME_GRADES => {src => \%PRISMA_INCOME_GRADES},
                    REGION_SOURCES => {src => \%REGION_SOURCES},
                };

    my $dict = $dicts->{$dict_name};
    die("Dict [$dict_name] is not supported") unless $dict;

    unless ($dict->{prepared} && $dict->{prepared}->{$consumer}) {
        $dict->{prepared} //= {};
        $dict->{prepared}->{$consumer} = _prepare_consumer_dict($dict->{src}, $consumer, %O);
    }
    return $dict->{prepared}->{$consumer};
}

=head2 _prepare_consumer_dict

    Подготавливаем словарь (словарного среза) к использованию
    входящие параметры:
        $dict - словарь, вида:
            {   
                undefined  => {bs => -1, api => 'CARRIER_TYPE_UNKNOWN', interface => {ord => 1, name => 'none'}},
                stationary => {bs => 0, api => 'STATIONARY', interface => {ord => 3}},
                mobile     => {bs => 1, api => 'CELLULAR', interface => {ord => 2}} 
            }
        $consumer - (api|web_ui|default) - потребитель
        Если потребитель default, или для конкретного значения не задано "потребительское" соответствие, берем основной ключ словаря
        Если для указанного потребителя задан ключ ord, он задан для всех(!) значений словаря - применяется сортировка по значениям ord;
            если ord разный для одинаковых значений словаря - падаем с ошибкой
            если ord не целое неотрицательное число - падаем с ошибкой
        %O - именованные параметры
            use_text_keys => 0|1
    выходные значения:
        $prepared_dict - ссылка на "подготовленный" словарь
        пример для $consumer eq 'api'
        {
            values => {
                CARRIER_TYPE_UNKNOWN => {bs => [-1], orig_names => [qw/undefined other/]},
                STATIONARY => {bs => [0], orig_names => ['stationary']},
                CELLULAR => {bs => [1], orig_names => ['mobile']}
            },
            need_reverse_translation => 0 # нужно ли полученные от БК значения еще раз преобразовывать для потребителя
        }
        пример для $consumer eq 'interface'
        {
            values => {
                1none => {bs => [-1], name => 'none', orig_names => [qw/undefined other/]},
                3stationary => {bs => [0], name => 'stationary'}, orig_names => ['stationary']}
                2mobile => {bs => [1], name => 'mobile', orig_names => ['mobile']}
            },
            need_reverse_translation => 1
        }

=cut

sub _prepare_consumer_dict {
    my ($dict, $consumer, %O) = @_;
    $consumer //= 'default';

    my %consumer_aliases = ('api4' => 'default');

    $dict = yclone($dict);

    my $use_ord = 1;
    my $max_digits = 0;
    my $have_any_ord = 0;
    for my $k (keys %$dict) {
        $dict->{$k} = {bs => $dict->{$k}} unless ref $dict->{$k} eq 'HASH';
        my $v = $dict->{$k};
        $v->{bs} //= [];

        my $consumer_alias = !defined $v->{$consumer} && $consumer_aliases{$consumer} ? $consumer_aliases{$consumer} : $consumer;
        if (defined $v->{$consumer_alias} && ref $v->{$consumer_alias} ne 'HASH') {
            $v->{$consumer_alias} = {name => $v->{$consumer_alias}};
        }
        if ($v->{$consumer_alias} && defined $v->{$consumer_alias}->{ord}) {
            $have_any_ord = 1;
            die("Dict error, field 'ord' should be non-negative integer: $v->{$consumer_alias}->{ord}") unless is_valid_int($v->{$consumer_alias}->{ord}, 0);
            $max_digits = max($max_digits, length($v->{$consumer_alias}->{ord}+0));
        } else {
            $use_ord = 0;
        }
    }
    die("Dict error, field 'ord' should be defined for all values or for none") if $have_any_ord && !$use_ord;

    my $pdict = {values => {},
                 need_reverse_translation => 0};
    for my $k (keys %$dict) {
        my $v = $dict->{$k};
        my $consumer_alias = !$v->{$consumer} && $consumer_aliases{$consumer} ? $consumer_aliases{$consumer} : $consumer;

        # имя значения, которое отдается потребителю
        my $result_name = $k;
        if ($v->{$consumer_alias} && defined $v->{$consumer_alias}->{name}) {
            $result_name = $v->{$consumer_alias}->{name};
        }

        # имя значения, на которое будем маппиться в БК, и которое будет участвовать в сортировке
        my $name = $result_name;
        if ($use_ord) {
            die("Dict error, different 'ord' with same consumer name: $name / $v->{$consumer_alias}->{ord}") 
                if any { m/^\d{$max_digits}$name$/ } keys %{$pdict->{values}};
            $name = sprintf("%0${max_digits}d%s", $v->{$consumer_alias}->{ord}, $name);
        }

        my $vals = $pdict->{values};
        if ($vals->{$name}) {
            @{$vals->{$name}->{bs}} = sort(uniq(@{$vals->{$name}->{bs}}, xflatten $v->{bs}));
            @{$vals->{$name}->{orig_names}} = sort(uniq(@{$vals->{$name}->{orig_names}}, $k));
        } else {
            $vals->{$name} = {bs => [sort(xflatten($v->{bs}))],
                              orig_names => [$k]};
            if ($use_ord) {
                $vals->{$name}->{name} = $result_name;
                $pdict->{need_reverse_translation} = 1;
            }
        }
    }

    unless ($O{use_text_keys}) {
        my $idx = 0;
        my %text2idx = ();
        for my $k (sort { $a cmp $b } keys %{$pdict->{values}}) {
            $idx++;
            $text2idx{$k} = $idx;
        }

        unless ($pdict->{need_reverse_translation}) {
            for my $k (keys %{$pdict->{values}}) {
                $pdict->{values}->{$k}->{name} = $k;
            }
            $pdict->{need_reverse_translation} = 1;
        }
        $pdict->{values} = hash_kmap { $text2idx{$_} } $pdict->{values};
    }
    return $pdict;
}

=head2 reverse_consumer_dict

    На основе подготовленного потребителю словаря (consumer_dict_by_name) возвращает простой хеш, ключи которого - идентификаторы, принятые в БК,
    а значения - наименование потребителя

=cut

sub reverse_consumer_dict {
    my $cdict = shift;

    my $rdict = {};
    for my $k (keys %{$cdict->{values}}) {
        my $v = $cdict->{values}->{$k};
        for my $bs_id (@{$v->{bs} // []}) {
            $rdict->{$bs_id} = $cdict->{need_reverse_translation} ? $v->{name} : $k;
        }
    }

    return $rdict;
}

=head2 reverse_by_orig_names_dict

    На основе подготовленного потребителю словаря (consumer_dict_by_name) возвращает простой хеш, ключи которого - идентификаторы по-умолчанию,
    а значения - наименование потребителя

=cut

sub reverse_by_orig_names_dict {
    my $cdict = shift;

    my $rdict = {};
    for my $k (keys %{$cdict->{values}}) {
        my $v = $cdict->{values}->{$k};
        for my $orig_name (@{$v->{orig_names} // []}) {
            $rdict->{$orig_name} = $cdict->{need_reverse_translation} ? $v->{name} : $k;
        }
    }

    return $rdict;
}

=head2 log_report_master_heavy_request 

    логгирует тяжелые (много строк, или считаем что будет долго строиться, или БК не ответило по timeout/memory out ...)
    На вход:
    $response_info - информация об ответе
        error => (BS_TOO_MUCH_STATISTICS|XLS_ROWS_LIMIT_EXCEEDED_ERROR|XLS_PROCESS_TIME_LIMIT_EXCEEDED_ERROR|...)
        rows_num_forecast
        process_time_forecast
        format => (default|xls|xlsx|csv)
    $request_info - информация о запросе
        как правило - $report_options, который формируется в Stat::ReportMaster

=cut

sub log_report_master_heavy_request ($$) {
    my ($response_info, $request_info) = @_;

    state $log //= Yandex::Log::Messages->new();
    $log->bulk_out(stat_report_master_heavy_request => [{request => $request_info, response => $response_info}]);
}

=head2 is_need_criterion_id

    проверяет по группировке можно ли для каждой группы определить № Условия показа
    $group_by - список колонок для группировки

=cut

sub is_need_criterion_id($){
    my $group_by = shift;
    if (!any {$_ eq 'contextcond_orig'} @$group_by) {
        return 0;
    }
    return any {$_ eq 'adgroup' || $_ eq 'banner'} @$group_by
}

=head2 get_metrika_counters_data

    Получить информации о счетчиках кампаний и их доступе

=cut

sub get_metrika_counters_data {
    my ($client_id, $uid, $order_ids) = @_;

    my $counters2cids = get_campaign_ids_by_metrika_counters($order_ids);
    if (!%$counters2cids) {
        return {};
    }
    
    my $counter_ids = [keys %$counters2cids];

    my $client_counters = MetrikaCounters::get_client_id_reps_counters_by_ids($client_id, $counter_ids, skip_errors => 1);
    my %client_counters_hash = map { $_ => 1 } @{$client_counters};

    # Получаем данные о запрашиваемом ранее доступе к счетчикам
    my $inaccessible_counter_ids = [grep {not $client_counters_hash{$_}} @$counter_ids];
    my $metrika_counters_grants = {};
    for my $counter_ids_chunk (chunks($inaccessible_counter_ids, 500)) {
        my $grants_chunk = MetrikaCounters::get_grant_access_requests_status_by_ids($uid, $counter_ids_chunk);
        for my $counter_id (keys %$grants_chunk) {
            $metrika_counters_grants->{$counter_id} = $grants_chunk->{$counter_id};
        }
    }

    my %result;
    for my $counter_id (keys %{$counters2cids}) {
        $result{$counter_id}->{cids} = $counters2cids->{$counter_id};
        $result{$counter_id}->{access} = exists $client_counters_hash{$counter_id} ? 1 : 0;
        $result{$counter_id}->{access_requested} = ( exists $client_counters_hash{$counter_id} || $metrika_counters_grants->{$counter_id} ) ? 1 : 0;
    }

    return \%result;
}

=head2 get_campaign_ids_by_metrika_counters

    Получить id кампаний по их счетчикам метрики

=cut

sub get_campaign_ids_by_metrika_counters {
    my ($order_ids) = @_;

    my $items = get_all_sql(PPC(OrderID => $order_ids), [
        'SELECT mc.cid, mc.metrika_counter
        FROM metrika_counters mc
        JOIN campaigns c ON c.cid = mc.cid',
        WHERE => { 'c.OrderID' => $order_ids },
    ]);

    my %counters2cids;
    for my $item (@$items) {
        push @{$counters2cids{$item->{metrika_counter}}}, $item->{cid};
    }
    return \%counters2cids;
}

=head2 get_all_counter_source_by_id

    Возвращает мапу "id счетчика" => "источник" для указанных счетчиков

=cut

sub get_all_counter_source_by_id {
    my ($counters) = @_;
    my $counters_response = eval { MetrikaIntapi->new()->get_existent_counters($counters) };

    my $counter_source_by_id = { 
                                 map { $_->{counter_id} => exists $_->{counter_source} ? $_->{counter_source} : 'unknown'}
                                 @{$counters_response->{response}} 
                               };
    return enrich_counter_source_by_id($counters, $counter_source_by_id);
}

=head2 get_allow_use_counter_without_access_by_id

    Возвращает мапу "id счетчика" => можно ли использовать счетчик без доступа

=cut

sub get_allow_use_counter_without_access_by_id {
    my ($counters) = @_;
    my $response = eval { MetrikaIntapi->new()->get_existent_counters($counters) };
    if (!$response) {
        state $log //= Yandex::Log::Messages->new();
        $log->out("Unable to get existing counters by ids: " . join(', ', @$counters));
        return {};
    }

    my %result;
    foreach my $line (@{$response->{response}}) {
        if (JSON::is_bool($line->{direct_allow_use_goals_without_access})) {
            $result{$line->{counter_id}} = ( $line->{direct_allow_use_goals_without_access} ? 1 : 0 );
        } else {
            $result{$line->{counter_id}} = 0;
        }
    }
    return \%result;
}

=head2 enrich_counter_source_by_id

    Заполняет источники счетчиков из полученной мапы "id счетчика" => "источник"

=cut

sub enrich_counter_source_by_id {
    my ($counters, $counter_source_by_id) = @_;
    for my $counter (@$counters) {
        if(!exists $counter_source_by_id->{$counter}){
            $counter_source_by_id->{$counter} = 'unknown';
        }
        if($counter_source_by_id->{$counter} eq 'turbodirect'){
            $counter_source_by_id->{$counter} = 'turbo';
        }
    }
    return $counter_source_by_id;
}

1;
