#######################################################################
#
#  Direct.Yandex.ru
#
#  StatXLS
#
#  $Id$
#
#######################################################################

=head1 NAME

Stat

=head1 DESCRIPTION

StatXLS

=cut

package StatXLS;

use Direct::Modern;

use base qw/Exporter/;
our @EXPORT = qw//;
our @EXPORT_OK = qw/stat_excel/;

require Spreadsheet::WriteExcel;

use Yandex::ReportsXLSWriter;
use Yandex::ReportsXLSXWriter;
use Yandex::ReportsXLS;
use Yandex::ReportsXLSX;
use Yandex::CSV;
use Yandex::CSVWriter;

use Encode;
use Date::Calc qw/Localtime/;
use Date::Parse;
use List::MoreUtils;
use List::Util qw/min/;
use IO::Scalar;
use Time::HiRes qw/gettimeofday tv_interval/;

use Campaign::Types;
use MirrorsTools;
use Yandex::I18n;
use Yandex::DateTime qw/date/;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xflatten nsort/;
use Yandex::Trace;

use Stat::Const qw/%SPECIAL_PHRASES $CONTEXT_TYPE_DYNAMIC $CONTEXT_TYPE_PERFORMANCE $CONTEXT_TYPE_RET $CONTEXT_TARGET_TYPE/;
use Stat::Fields;
use Stat::Tools qw/periods_suffix_list/;
use Yandex::ListUtils;
use Currency::Format qw/format_currency/;
use TextTools;
use URLDomain qw/clear_banner_href/;
use Carp;
use Yandex::DBShards;
use JavaIntapi::GetInternalAdInfo;
use JavaIntapi::GetTemplateResource;
use List::MoreUtils qw/uniq/;
use Primitives qw/get_internal_product_names_by_client_ids/;
use Settings;

our @ALLOWED_TYPES = qw/pages campaigns common custom date geo phrase_date by_agency_clients by_clients by_managers by_agencies/;

my %post_click_field_names = map {$_ => 1} qw/adepth agoalnum aconv agoalcost agoalroi agoalcrr agoalincome agoals_profit aprgoodmultigoal
                                              aprgoodmultigoal_cpa aprgoodmultigoal_conv_rate avg_time_to_conv/;

our %column_name = (
    #sort key => ['field','field caption', $opt]
     5 => ['shows',     iget_noop('Показы')],
    10 => ['eshows',    iget_noop('Взвешенные показы')],
    15 => ['clicks',    iget_noop('Клики')],
    20 => ['ctr',       iget_noop('CTR (%)')],
    25 => ['ectr',      iget_noop('wCTR (%)')],
    30 => ['sum',       iget_noop('%sРасход%s'), { currency => 1}],
    35 => ['avg_cpm',   iget_noop('%sСр. цена тыс. показов%s'), { currency => 1}],
    40 => ['av_sum',    iget_noop('%sСр. цена клика%s'), { currency => 1}],
    45 => ['avg_bid', iget_noop('%sСр. ставка за клик%s'), { currency => 1}],
    50 => ['avg_cpm_bid', iget_noop('%sСр. ставка за тыс. показов%s'), { currency => 1}],
    55 => ['av_day',    iget_noop('%sСр. расход за день%s'), { currency => 1}],
    60 => ['fp_shows_avg_pos', iget_noop('Ср. позиция показов')],
    65 => ['avg_x',     iget_noop('Ср. объём трафика')],
    70 => ['fp_clicks_avg_pos', iget_noop('Ср. позиция кликов')],
    75 => ['adepth',    iget_noop('Глубина (стр.)')],
    80 => ['agoalnum',  iget_noop('Конверсии')],
    85 => ['aconv',     iget_noop('Конверсия (%)')],
    90 => ['agoalcost', iget_noop('%sЦена цели%s'), { currency => 1 }],
    95 => ['agoalroi', iget_noop('Рентабельность')],
    98 => ['agoalcrr', iget_noop('Доля рекламных расходов')],
   100 => ['agoalincome', iget_noop('Доход%s'), {currency_only => 1}],
   105 => ['agoals_profit', iget_noop('Прибыль%s'), {currency_only => 1}],
   110 => ['uniq_viewers', iget_noop('Охват')],
   115 => ['avg_view_freq', iget_noop('Ср. частота показов')],
   120 => ['aprgoodmultigoal', iget_noop('Сумма ВC')],
   125 => ['aprgoodmultigoal_cpa', iget_noop('%sЦена ВC%s'), { currency => 1}],
   130 => ['aprgoodmultigoal_conv_rate', iget_noop('Доля ВC (%)')],
   135 => ['avg_time_to_conv', iget_noop('Ср. время до конверсии, час.')],
   140 => ['auction_hits', iget_noop('Участие в аукционах')],
   145 => ['auction_wins', iget_noop('Выигрыши в аукционах')],
   150 => ['imp_to_win_rate', iget_noop('Доля видимых показов (%)')],
   155 => ['auction_win_rate', iget_noop('Доля выигрышей (%)')],
   160 => ['imp_reach_rate', iget_noop('SOV (%)')],
   165 => ['served_impressions', iget_noop('Выходы')],
   170 => ['pv_adepth',    iget_noop('Post-view: Глубина (стр.)')],
   175 => ['pv_aconv',     iget_noop('Post-view: Конверсия (%)')],
   180 => ['pv_agoalnum',  iget_noop('Post-view: Конверсии')],
   182 => ['all_agoalnum',  iget_noop('Итоговая Конверсия')],
   185 => ['pv_agoalcost', iget_noop('Post-view: Цена цели%s'), { currency => 1 }],
   190 => ['pv_agoalroi',  iget_noop('Post-view: Рентабельность')],
   195 => ['pv_agoalincome',iget_noop('Post-view: Доход%s'), {currency_only => 1}],
   200 => ['pv_agoals_profit',iget_noop('Post-view: Прибыль%s'), {currency_only => 1}],
   205 => ['pv_aprgoodmultigoal', iget_noop('Post-view: Сумма ВC')],
   210 => ['pv_aprgoodmultigoal_cpa', iget_noop('Post-view: Цена ВC%s'), { currency => 1}],
   215 => ['pv_aprgoodmultigoal_conv_rate', iget_noop('Post-view: Доля ВC (%)')],
   220 => ['pv_avg_time_to_conv', iget_noop('Post-view: Ср. время до конверсии, час.')],
   225 => ['pv_agoalcrr', iget_noop('Post-view: Доля рекламных расходов')],
   Stat::Fields::get_columns_for_xls(start_num => 230),
   );

our %additional_column_name = (
    1 => ['xls_target_all', iget_noop('всего')],
    2 => ['xls_target_0',   iget_noop('поиск')],
    3 => ['xls_target_1',   iget_noop('сети')],
);

use constant MAX_XLS_ROWS => 2**16;

# отчеты какого размера (количество строк) нужно логгировать
my $ROWS_NUM_TO_LOG_REQUEST = 200_000;

# для условий нацеливания в качестве названия пишем:
my $performance_title_template = iget_noop('%s (фильтр)');
my $ret_title_template = iget_noop('%s (условие подбора аудитории)');
my $rm_title_template = iget_noop('%s (автотаргетинг)');
my $phrase_column_title = iget_noop('Условие показа');

my %money_fields = map {$_ => undef} Stat::Fields::get_all_money_fields();

=head2 statxls_common

Вывести общую статистику в excel

=cut

sub statxls_common {

    my ($vars, $skip) = @_;
    $skip ||= {};

    my $opts = get_common_xls_stat_options($vars, $skip);
    $opts->{report_type} = 'common';
    @{$opts->{group_by}} = grep { !$skip->{$_} } qw/adgroup banner banner_title banner_body banner_href phrase_state contextcond/;
    $opts->{totals} = get_separate_totals($vars);

    unless ($vars->{summary_only}) {
        for my $banner ( @{$vars->{banners} // []} ) {
            my $banner_info = get_min_banner_info($banner);
            for my $phrase_state ( qw/ active bad_ctr past / ) {
                for my $suffix ( "", "_categories" ) {
                    for my $phr (xsort { (exists($SPECIAL_PHRASES{$_->{PhraseID}}), $_->{text}) } @{ $banner->{"$phrase_state$suffix"} // [] }) {
                        hash_merge $phr, $banner_info;
                        $phr->{phrase_state} = $phrase_state;
                        push @{$opts->{data_array}}, $phr;
                    }
                }
            }
        }
    }
    
    return statxls_report_master({  %$vars, %$opts }, undef, sheetname => iget("Общая статистика"));
}

=head2 statxls_date

Вывести статистику по дням в excel

=cut

sub statxls_date {

    my ($vars, $skip) = @_;
    $skip ||= {};
    $vars->{banners} ||= [];

    my $is_media_order = is_media_camp(OrderID => $vars->{OrderID});

    my $opts = get_common_xls_stat_options($vars, $skip);
    $opts->{totals} = get_separate_totals($is_media_order ? $vars : $vars->{campstat});
    if (!$vars->{stat_summary_only} && $vars->{show_banners_stat}) {
        for my $banner (grep { $_ } @{$vars->{banners}}) {
            my $banner_info = get_min_banner_info($banner);
            for my $row (sort {$a->{sorting} <=> $b->{sorting}} @{ $banner->{dates} } ) {
                push @{$opts->{data_array}}, hash_merge($row, $banner_info);
            }
        }
                          ;
        @{$opts->{group_by}} = grep { !$skip->{$_} } $is_media_order
                        ? qw/mcb_banner mcb_banner_title mcb_banner_format banner_href date/
                        : qw/adgroup banner banner_title banner_body banner_href date/;
    } else {
        push @{$opts->{data_array}}, $is_media_order 
                                ? @{$vars->{media_groups}->[0]->{dates} || []}
                                : (sort {$a->{sorting} <=> $b->{sorting}} @{$vars->{campstat}->{dates} || []});
        @{$opts->{group_by}} = qw/date/;
    }

    return statxls_report_master({  %$vars, %$opts }, undef, sheetname => iget("Статистика по дням"));
}

=head2 get_suffixes

    Возвращает список суффиксов для разделения данных на поиск/сети/всего

=cut

sub get_suffixes
{
    my $vars = shift;
    my @suffixes = ();
    push @suffixes,"" if defined $vars->{xls_target_all} || ( !defined $vars->{xls_target_0} && !defined $vars->{xls_target_1});
    push @suffixes,"_0" if defined $vars->{xls_target_0};
    push @suffixes,"_1" if defined $vars->{xls_target_1};
    return @suffixes;
}

=head2 get_separate_totals

    Берет из $vars_list (хеш или массив хешей в порядке возрастания приоритета) поля статистики с префиксом "t" (если такого нет - берет без префикса), 
    и добавляет без префикса в новый хеш с тоталами

=cut

sub get_separate_totals
{
    my $vars_list = shift;
    my %totals = ();

    for my $vars (ref $vars_list eq 'ARRAY' ? @$vars_list : $vars_list) {
        for my $f (qw/av_day av_grouping av_day_shows/, map { $_->[0] } values %column_name) {
            for my $suf (Stat::Tools::suffix_list()) {
                my $val = $vars->{"t$f$suf"} // $vars->{"$f$suf"};
                next unless defined $val;
                $totals{"$f$suf"} = $val;
            }
        }
    }

    return \%totals;
}

=head2 get_min_banner_info

    Выделяет из входящего хеша минимально необходимую информацию о баннере, для сквозной статистики

=cut

sub get_min_banner_info {
    my $banner = hash_cut($_[0], qw/bid BannerID href domain real_banner_type adgroup_main_domain image_ad creative body title 
                              adgroup_id adgroup_name group_name
                              mbid width height name /);
    $banner->{adgroup_id} //= $_[0]->{pid};
    return $banner;
}

=head2 get_common_xls_stat_options

    Возвращает шаблон "типовых" опций для добавления к $vars при передаче в statxls_report_master

=cut

sub get_common_xls_stat_options {
    my ($vars, $skip) = @_;

    return {  group_by => [],
              columns => [grep { !$skip->{$_} } map { $column_name{$_}->[0] } nsort keys %column_name],
              data_array => [],
              totals => {},
              date_from => sprintf('20%d-%02d-%02d', @{$vars}{qw/y1 m1 d1/}),
              date_to => sprintf('20%d-%02d-%02d', @{$vars}{qw/y2 m2 d2/}),  };

}

=head2 statxls_phrase_date

Вывести статистику по фразам/дням в excel

=cut

sub statxls_phrase_date {

    my ($vars, $skip) = @_;
    $skip ||= {};

    my $is_media_order = is_media_camp(OrderID => $vars->{OrderID});

    my $opts = get_common_xls_stat_options($vars, $skip);
    @{$opts->{group_by}} = grep { !$skip->{$_} } $vars->{with_reach_stat}
        ? qw/adgroup banner banner_href contextcond criterion_type date/
        : ($is_media_order
                        ? qw/mcb_banner mcb_banner_title mcb_banner_format banner_href contextcond date/
                        : qw/adgroup banner banner_title banner_body banner_href contextcond date/);
    
    $opts->{totals} = get_separate_totals($vars);

    my $banners_info = {};
    for my $groups (@{($is_media_order ? $vars->{media_groups} : [$vars]) // []}) {
        for my $banner ( @{$groups->{banners} // []} ) {
            for my $phr (sort {($is_media_order ? $a->{BannerID} <=> $b->{BannerID} : 0) ||
                               $a->{ContextType} <=> $b->{ContextType} ||
                               $a->{phrase} cmp $b->{phrase} ||
                               $a->{sorting} cmp $b->{sorting}
                               } @{$banner->{all_stat}} ) {
                $banners_info->{$phr->{BannerID}} //= get_min_banner_info($is_media_order ? (grep { $_->{BannerID} == $phr->{BannerID} } @{$groups->{banners}} )[0] : $banner);
                $phr->{date} = $phr->{Date};
                push @{$opts->{data_array}}, hash_merge($phr, $banners_info->{$phr->{BannerID}});
            }
        }
    }

    return statxls_report_master({  %$vars, %$opts }, undef, sheetname => iget("Условия показа по дням"));
}

=head2 statxls_geo

Вывести статистику по географии в excel

=cut

sub statxls_geo {

    my ($vars, $skip) = @_;
    $skip ||= {};

    my $is_media_order = is_media_camp(OrderID => $vars->{OrderID});

    # статистика может быть в разбивке по баннерам или нет
    my $by_banners  = exists $vars->{banners};

    my $opts = get_common_xls_stat_options($vars, $skip);
    $opts->{totals} = get_separate_totals($vars);

    if ($by_banners) {
        @{$opts->{group_by}} = grep { !$skip->{$_} } $is_media_order 
                        ? qw/mcb_banner mcb_banner_title mcb_banner_format banner_href region/
                        : qw/adgroup banner banner_title banner_body banner_href region/;
        for my $banner ( @{$vars->{banners}} ) {
            for my $reg ( @{$banner->{regions}} ) {
                my $banner_info = get_min_banner_info($banner);
                push @{$opts->{data_array}}, hash_merge($reg, $banner_info);
            }
        }
    } else {
        @{$opts->{group_by}} = qw/region/;
        push @{$opts->{data_array}}, @{$vars->{regions}} if $vars->{regions};
    }

    return statxls_report_master({  %$vars, %$opts }, undef, sheetname => iget("Статистика по регионам"));
}

=head2 statxls_by_agency_clients

Вывести статистику по клиентам агентства в excel

=cut

sub statxls_by_agency_clients
{
    my $vars = shift;
    my $skip = shift() || {};

    my $xls_data;

    push @{$xls_data}, [{data => iget("Период %s - %s", "$vars->{d1}.$vars->{m1}.$vars->{y1}", "$vars->{d2}.$vars->{m2}.$vars->{y2}"), global_format => { size=>10, bold=>1 }}];

    my (@column_caption, @targets, @field_name);

    foreach my $key (sort {$a<=>$b} keys %column_name) {
        my ($f,$c,$col_opt) = @{$column_name{$key}};
        next if (exists $skip->{$f});

        push(@field_name, $f);
        push(@column_caption, iget($c, _apply_column_opt($vars, $col_opt)));
        foreach (sort {$a <=> $b} keys %additional_column_name) {
            push @targets, iget($additional_column_name{$_}->[1])
                if defined $vars->{$additional_column_name{$_}->[0]}
        }
    }

    my %stat_type_captions;
    if ($vars->{by_managers}) {
        %stat_type_captions = (one => iget("Менеджер"), many => iget("По всем менеджерам"));
    } elsif ($vars->{by_agencies}) {
        %stat_type_captions = (one => iget("Агентство"), many => iget("По всем агентствам"));
    } else {
        %stat_type_captions = (one => iget("Клиент"), many => iget("По всем клиентам"));
    }

    my @suffixes = get_suffixes($vars);

    my $quantity = @suffixes - 1;
    push @{$xls_data}, [
        {data => 'login', global_format => { size=>8, bold=>1 }}, $stat_type_captions{one},
        iget('Дата'),
        (map {
            ($_, ('') x $quantity)
        } @column_caption)
    ];
    push @{$xls_data},['', '', '', @targets] if $quantity > 0;

    #### выводим статистику по всем клиентам/менеджерам/агентствам ####

    my $first_line_flag = 1;
    foreach my $stat_by_date (sort {$a->{sorting} <=> $b->{sorting}} @{$vars->{dates}}){
        my $xls_data_row;
        if($first_line_flag){
            $first_line_flag = 0;
            push @{$xls_data_row}, ({data => "", global_format => { size=>8, bold=>0 }}, $stat_type_captions{many}, "". $stat_by_date->{date});
        } else {
            push @{$xls_data_row}, ("", "", "". $stat_by_date->{date});
        }

        for my $f (@field_name) {
            push @$xls_data_row, {
                data => _format_value($stat_by_date->{$f . $_}, field => $f),
                format => {align => 'right'}
            } foreach @suffixes;
        }
        push @{$xls_data}, $xls_data_row;

    }

    my $xls_data_row;

    push @{$xls_data_row}, ("", "", {data => iget("Итого:"), global_format => { size=>8, bold=>1 }});

    for my $f (@field_name) {
        push @$xls_data_row, {
            data => _format_value($vars->{$f . $_}, field => $f),
            format => {align => 'right'}
        } foreach @suffixes;
    }

    push @{$xls_data}, $xls_data_row;
    push @{$xls_data}, [{data => "", global_format => { size=>8, bold=>0 }}];

    #### выводим статистику по клиентам по отдельности ####

    foreach my $stat_by_order (@{$vars->{orders}}) {
        if ($stat_by_order->{shows}) {
            my $first_line_flag = 1;
            foreach my $stat_by_order_date (sort {$a->{sorting} <=> $b->{sorting}} @{$stat_by_order->{dates}}) {
                my $xls_data_row;
                if ($first_line_flag) {
                    $first_line_flag = 0;
                    if ($vars->{by_managers}) {
                        push @{$xls_data_row}, ($stat_by_order->{ManagerLogin}, $stat_by_order->{ManagerFio}, "". $stat_by_order_date->{date});
                    } elsif ($vars->{by_agencies}) {
                        push @{$xls_data_row}, ($stat_by_order->{AgencyLogin}, $stat_by_order->{AgencyName}, "". $stat_by_order_date->{date});
                    } else {
                        push @{$xls_data_row}, ($stat_by_order->{login}, $stat_by_order->{FIO}, "". $stat_by_order_date->{date});
                    }
                } else {
                    push @{$xls_data_row}, ("", "", "". $stat_by_order_date->{date});
                }

                for my $f (@field_name) {
                    push @$xls_data_row, {
                        data => _format_value($stat_by_order_date->{$f . $_}, field => $f),
                        format => {align => 'right'}
                    } foreach @suffixes;
                }
                push @{$xls_data}, $xls_data_row;
            }

            undef $xls_data_row;

            push @{$xls_data_row}, ("", "", {data => iget("Итого:"), global_format => { size=>8, bold=>1 }});

            for my $f (@field_name) {
                push @$xls_data_row, {
                    data => _format_value($stat_by_order->{$f . $_}, field => $f),
                    format => {align => 'right'}
                } foreach @suffixes;
            }
            push @{$xls_data}, $xls_data_row;
            push @{$xls_data}, [{data => "", global_format => { size=>8, bold=>0 }}];
        }
    }

    my $note = _get_discount_note($vars);
    if ($note) {
        push @$xls_data, [{data => $note}];
    }

    my $xls_format = {
        freeze_panes => [$quantity > 0 ? 3 : 2, 2],
        set_column => [
            {col1 => 0, col2 => 0, width => 15},
            {col1 => 1, col2 => 1, width => 20}
        ],
    };

    return $xls_data, $xls_format;
}

=head2 statxls_by_clients

    Статистика по клиентам
    
=cut

sub statxls_by_clients {
   statxls_by_agency_clients(@_);
}

=head2 statxls_by_managers

    Статистика по клиентам менеджера
    
=cut

sub statxls_by_managers {
    my ($vars, $skip) = @_;
    statxls_by_agency_clients({%$vars, by_managers => 1}, $skip);
}

=head2 statxls_by_agencies

    Статистика по клиентам агенства
    
=cut

sub statxls_by_agencies {
    my ($vars, $skip) = @_;
    statxls_by_agency_clients({%$vars, by_agencies => 1}, $skip);
}

=head2 statxls_common_smart_banners

    Общая статистика по смарт-баннерам

=cut

sub statxls_common_smart_banners {
    my ($vars, $skip) = @_;
    $skip ||= {};

    my $opts = get_common_xls_stat_options($vars, $skip);
    @{$opts->{group_by}} = qw/adgroup banner banner_title banner_body banner_href/;
    $opts->{totals} = get_separate_totals($vars);

    unless ($vars->{summary_only}) {
        for my $banner ( @{$vars->{banners} // []} ) {
            my $banner_info = get_min_banner_info($banner);
            hash_merge $banner, $banner_info;
            push @{$opts->{data_array}}, $banner;
        }
    }

    return statxls_report_master({  %$vars, %$opts }, undef, sheetname => iget("Общая статистика"));
}

=head2 stat_excel($type => $data, $skip_columns)

    Сформировать отчет статистики

    Параметры:
        $type - отчет который необходимо сформировать, доступные типы см. @ALLOWED_TYPES
                имя типа это statxls_$ALLOWED_TYPES
        $data - данные для формирования отчёта
        $skip_columns - имена колонок которые нужно пропустить в отчете

    Результат:
        $excel_type - тип сформированного excel файла (xls|xlsx)
        $excel_data - содержимое отчета

=cut

sub stat_excel {
    my ($type, @args) = @_;

    my $code;
    {
        no strict 'refs';
        my $_PACKAGE = __PACKAGE__ . '::';
        $code = $_PACKAGE->{$type};
    }
    croak sprintf 'can\'t find %s::%s', __PACKAGE__, $type unless $code;

    # типы отчетов, которые формируются тем же методом что и отчеты для Мастера отчетов
    # в данном случае мы получаем уже готовый бинарник, и его тип
    my %already_binary_report_types = (
        statxls_date => 1,
        statxls_common => 1,
        statxls_phrase_date => 1,
        statxls_geo => 1,
        statxls_pages => 1,
        statxls_campaigns => 1,
        statxls_common_smart_banners => 1,
    );

    if ($already_binary_report_types{$type}) {
        return $code->(@args);
    } else {
        my ($excel_data, $excel_format) = $code->(@args);
        if (@$excel_data > MAX_XLS_ROWS) {
            my $excel = Yandex::ReportsXLSX->new(compatibility_mode => 1);
            return xlsx => $excel->array2excel2scalar($excel_data, $excel_format);
        }

        my $excel = Yandex::ReportsXLS->new();
        return xls => $excel->array2excel2scalar($excel_data, $excel_format);
    }
}

=head2 statxls_pages

Вывести статистику по площадкам в excel

=cut

sub statxls_pages {

    my ($vars, $skip) = @_;
    $skip ||= {};

    my $opts = get_common_xls_stat_options($vars, $skip);
    @{$opts->{group_by}} = qw/page_group targettype date/;
    $opts->{totals} = get_separate_totals($vars);

    for my $page (@{$vars->{data}}) {
        my $page_info = {page_name => $page->{name} . ($vars->{use_page_id} ? " ($page->{page_domain} - $page->{page_id})" : '')};
        $page_info->{targettype} = $page->{TargetType} == $CONTEXT_TARGET_TYPE ? 'context' : 'search';
        for my $d (@{$page->{data}}) {
            push @{$opts->{data_array}}, hash_merge($d, $page_info);
        }
    }

    return statxls_report_master({  %$vars, %$opts }, undef, sheetname => iget("Статистика по площадкам"));
}

=head2 statxls_campaigns

Вывести общую статистику по кампаниям в excel

=cut

sub statxls_campaigns {

    my ($vars, $skip) = @_;

    my $opts = get_common_xls_stat_options($vars, $skip);
    @{$opts->{group_by}} = qw/campaign date/;
    $opts->{totals} = get_separate_totals($vars);

    for my $camp (@{$vars->{orders}}) {
        my $camp_info = {camp_name => $camp->{name},
                         cid => $camp->{cid}};
        for my $d ( sort {$a->{sorting} cmp $b->{sorting}} @{$camp->{dates}} ) {
            push @{$opts->{data_array}}, hash_merge($d, $camp_info);
        }
    }

    return statxls_report_master({  %$vars, %$opts }, undef, sheetname => iget("Статистика по кампаниям"));
}

{
=head2 dictionaries for statxls_report_master

    Словари для statxls_report_master

=cut
    
    my %banner_image_type_text = (text_only => iget_noop('текстовый'),
                                  text_image => iget_noop('с изображением'),
                                  text_video => iget_noop('с видео'),
                                  smart_creative => iget_noop('смарт-баннер'),
                                  smart_tgo => iget_noop('смарт-объявление'),
                                  adaptive_image => iget_noop('адаптивный креатив'),
                                  smart_tile => iget_noop('смарт-плитка'),
                                  smart_video => iget_noop('смарт-видео'),
                                  multicard => iget_noop('галерея'),
                              );
    my %image_size_text = (no_image => iget_noop('без изображения'),
                           with_image => iget_noop('с изображением'));

    my %targettype_text = (search => iget_noop('поиск'),
                           context => iget_noop('сети'),);

    my %contexttype_text = (phrases => iget_noop('фраза'),
                            retargeting => iget_noop('условие подбора аудитории'),
                            dynamic => iget_noop('условие нацеливания ДО'),
                            performance => iget_noop('фильтр смарт-баннеров'),
                            synonym => iget_noop('синоним'),
                            'relevance-match' => iget_noop('автотаргетинг'),
                            'auto-added-phrases' => iget_noop('ДРФ'), );

    my %device_type_text = (desktop => iget_noop('десктоп'),
                            mobile => iget_noop('мобильные'),
                            tablet => iget_noop('планшеты'));

    my %position_text = ($Stat::Const::PRIME_TYPE_ID => iget_noop('спецразмещение'),
                         $Stat::Const::NON_PRIME_TYPE_ID => iget_noop('прочее'),
                         $Stat::Const::ALONE_TYPE_ID => iget_noop('эксклюзивное размещение'),
                         $Stat::Const::SUGGEST_TYPE_ID => iget_noop('реклама в саджесте'),
                         $Stat::Const::ADV_GALLERY_TYPE_ID => iget_noop('товарная галерея'));

    my %click_place_text = (undefined => iget_noop('не определено'),
                            title => iget_noop('заголовок'), 
                            sitelink1 => iget_noop('быстрые ссылки (1)'), 
                            sitelink2 => iget_noop('быстрые ссылки (2)'), 
                            sitelink3 => iget_noop('быстрые ссылки (3)'),
                            sitelink4 => iget_noop('быстрые ссылки (4)'),
                            sitelink5 => iget_noop('быстрые ссылки (5)'),
                            sitelink6 => iget_noop('быстрые ссылки (6)'),
                            sitelink7 => iget_noop('быстрые ссылки (7)'),
                            sitelink8 => iget_noop('быстрые ссылки (8)'),
                            vcard => iget_noop('визитка'),
                            display_href => iget_noop('отображаемая ссылка'),
                            mobile_content_button => iget_noop('кнопка приложения'),
                            mobile_content_icon => iget_noop('иконка приложения'),
                            dialog => iget_noop('чат'),
                            turbo_gallery => iget_noop('турбо-галерея'),
                            phone => iget_noop('телефон'),
                            button => iget_noop('кнопка'),
                            adv_gallery => iget_noop('товарные дополнения'),
                            market => iget_noop('маркет'),
                            promo_extension => iget_noop('промоакция'),
    );

    my %gender_text = ( undefined => iget_noop('не определен'),
                        male => iget_noop('мужской'),
                        female => iget_noop('женский') );

    my %age_text = ( undefined => iget_noop('не определен'),
                      '0-17'  => iget_noop('младше 18'),
                      '18-24' => iget_noop('18-24'),
                      '25-34' => iget_noop('25-34'),
                      '35-44' => iget_noop('35-44'),
                      '45-54' => iget_noop('45-54'),
                      '55-'   => iget_noop('старше 55'),
                      '45-'   => iget_noop('старше 45') );
    
    my %detailed_device_type_text = (
        undefined => "-",
        other     => iget_noop("не определен"),
        android   => iget_noop("Android"),
        ios       => iget_noop("iOS"), 
    );

    my %connection_type_text = (
        undefined  => "-",
        mobile     => iget_noop("мобильная связь"),
        stationary => iget_noop("Wi-Fi"),
    );

    my %bm_type_text = ( 
        adf => iget_noop('авторасширение'),
        drf => iget_noop('дополнительная релевантная фраза'),
        experiment => iget_noop('эксперимент'),
        undefined => iget_noop('не определен'),
    );

    my %campaign_type_text = ('text' => iget_noop('Текстово-графические объявления'),
                              'dynamic' => iget_noop('Динамические объявления'),
                              'mobile_content' => iget_noop('Реклама мобильных приложений'),
                              'performance' => iget_noop('Смарт-баннеры'),
                              'cpm_banner' => iget_noop('Медийная кампания'),
                              'cpm_deals' => iget_noop('Медийная кампания со сделками'),
                              'cpm_yndx_frontpage' => iget_noop('Медийный продукт на главной'),
                              'mcbanner' => iget_noop('Медийно-контекстный баннер на поиске'),
                              'content_promotion' => iget_noop('Продвижение контента'),
                          );


    my %banner_type_text = ('text' => iget_noop('Текстово-графическое объявление'),
                            'image_ad' => iget_noop('Графическое объявление'),
                            'cpm_video' => iget_noop('Видео'),
                            'other' => '-',);

    my %phrase_state_text = (
        active  => iget_noop('активные'),
        past    => iget_noop('удалённые'),
        bad_ctr => iget_noop('отключенные')
    );

    my %cond_add_status_text = (
        added    => iget_noop('Добавлено'),
        excluded => iget_noop('Исключено'),
        none     => '-',
    );

    my %match_type_text = ( 
        rkw => iget_noop('ДРФ'),
        syn => iget_noop('семантическое'),
        experiment => iget_noop('эксперимент'),
        kwd => iget_noop('пословное'),
        none => iget_noop('-'),
    );

    my %criterion_type_text = (
        phrases              => iget_noop('фраза'),
        retargeting          => iget_noop('условие подбора аудитории'),
        'webpage-filter'     => iget_noop('фильтр по сайту'),
        'feed-filter'        => iget_noop('фильтр по фиду'),
        'mobile-content-interest' => iget_noop('интерес к мобильным приложениям'),
        'relevance-match'    => iget_noop('автотаргетинг'),
        'reach-audience'     => iget_noop('профиль пользователей'),
    );

    my %inventory_type_text = (
        undefined => iget_noop("Баннеры"),
        instream_web => iget_noop("Потоковое видео"),
        inpage => iget_noop("Текстовый контент"),
        interstitial => iget_noop("Приложения"),
        inapp => iget_noop("Приложения"),
        inbanner => iget_noop("Видео в баннере"),
        rewarded => iget_noop("Видео за вознаграждение"),
    );

    my %turbo_page_type_text = (
        turbo => iget_noop('турбо'),
        none => iget_noop('-'),
    );
    
    my %targeting_category_text = (
        undefined => iget_noop("не определена"),
        exact     => iget_noop("Целевые запросы"),
        alternative     => iget_noop("Альтернативные запросы"),
        competitor     => iget_noop("Запросы с упоминанием конкурентов"),
        broader     => iget_noop("Широкие запросы"),
        accessory     => iget_noop("Сопутствующие запросы"),
    );
    
    my %prisma_income_grade_text = (
        others        => iget_noop("Остальные"),
        above_average => iget_noop("6-10%"),
        high          => iget_noop("2-5%"),
        very_high     => iget_noop("1%"),
    );


=head2 statxls_report_master

    Выгрузить статистику из мастер отчётов на аккаунт

    Входные параметры:
        $vars
        $format - (xls|xlsx|csv), по-умолчанию xls
        %O - именованные параметры
            title - название отчета (первая строка в выгружаемом xls-файле)
            sheetname - название листа в xls-файле
            use_stat_iterator - основные данные статистики приходят в виде итератора Stat::CustomizedArray::Iterator, в поле $vars->{stat_iterator}
            rows_limit - лимит на количество выгружаемых строк статистики, при превышении - падаем с формулировкой $Stat::Const::XLS_ROWS_LIMIT_EXCEEDED_ERROR
            four_digits_precision => 0|1 – округлять денежные поля до 4 знаков
            goals_list => массив с информацией (главное, с названиями) о целях (для мультицелей)
            form_goals => строка с целями, если статистика разбита по конкретным целям (для мультицелей)
            attribution_model_bs_value => номер запрашиваемой модели атрибуции; число, которое приписывается ко внутренним названиям полей после номера цели (для мультицелей)
            process_time_limit - лимит на время формирования выгрузки (проверяется по "прогнозному" времени после обработки первого чанка),
                                 при превышении - падаем с формулировкой $Stat::Const::XLS_PROCESS_TIME_LIMIT_EXCEEDED_ERROR

=cut

sub statxls_report_master($$;%)
{
    my ($vars, $format, %O) = @_;
    $format ||= 'xls';

    my $is_media_order = $vars->{OrderID} ? is_media_camp(OrderID => $vars->{OrderID}) : 0;

    # статистика в разбивке по конкретным целям, если целей больше одной
    my @goal_ids = (defined $O{form_goals} && $O{form_goals} =~ /^[0-9]+(,[0-9]+)+$/) ? split /,/, $O{form_goals} : ();
    my %goal_names;
    my $attribution_model_bs_value;
    if (@goal_ids) {
        %goal_names = map {$_->{goal_id} => $_->{name}} @{$O{goals_list}};
        $attribution_model_bs_value = $O{attribution_model_bs_value};
    }

    my @data = ();
    my $title = '';
    if (defined $O{title}) {
        $title = $O{title};
    } else {
        if ($vars->{cid}) {
            my $cname = $vars->{campaign} ? $vars->{campaign}->{name} // $vars->{campaign}->{cname} : '';
            my $cname_extra = '';
            if ($is_media_order) {
                $cname_extra = iget(", БАЯН, ").
                                ( $vars->{media_groups}[0]{mcb_theme}
                                 ? $vars->{media_groups}[0]{mcb_theme}{parent_name}.'/'.$vars->{media_groups}[0]{mcb_theme}{name}
                                 : iget('Тема не выбрана'));
            }

            $title = $vars->{compare_periods}
                            ? iget(qq/Кампания "%s" (%s), сравнение периодов %s - %s и %s - %s/, $cname, $vars->{cid}
                                                               , date($vars->{date_from})->dmy('.'), date($vars->{date_to})->dmy('.')
                                                               , date($vars->{date_from_b})->dmy('.'), date($vars->{date_to_b})->dmy('.'))
                            : iget(qq/Кампания "%s" (%s)%s, период %s - %s/, $cname, $vars->{cid}, $cname_extra
                                                               , date($vars->{date_from})->dmy('.'), date($vars->{date_to})->dmy('.'));
        } elsif ( $vars->{multi_clients_mode} ) {
            $title = $vars->{compare_periods}
                            ? iget("Мультиклиентность, сравнение периодов %s - %s и %s - %s", date($vars->{date_from})->dmy('.'), date($vars->{date_to})->dmy('.')
                                                                , date($vars->{date_from_b})->dmy('.'), date($vars->{date_to_b})->dmy('.'))
                            : iget("Мультиклиентность, период %s - %s", date($vars->{date_from})->dmy('.'), date($vars->{date_to})->dmy('.'));            
        } else {
            $title = $vars->{compare_periods}
                            ? iget("Клиент %s (%s), сравнение периодов %s - %s и %s - %s", $vars->{user_fio}, $vars->{user_login}
                                                               , date($vars->{date_from})->dmy('.'), date($vars->{date_to})->dmy('.')
                                                               , date($vars->{date_from_b})->dmy('.'), date($vars->{date_to_b})->dmy('.'))
                            : iget("Клиент %s (%s), период %s - %s", $vars->{user_fio}, $vars->{user_login}
                                                               , date($vars->{date_from})->dmy('.'), date($vars->{date_to})->dmy('.'));
        }
    }

    push @data, [{data => $title, format => {size => 10, bold => 1}  }];

    my $f_b8 = {size => 8, bold => 1};

    my ($i_deflabel, $i_defgroup, $i_image, $i_notimage, $i_nodeal)
        = (iget('без метки'), iget('без группы'), iget('с изображением'), iget('без изображения'), iget('без сделки'));

    my $f_left = { align => 'left' };
    my $f_bold_right = { align => 'right', bold=>1 };
    my $f_gray  = { align => 'left', italic => 1, color => 'gray' };

    my $need_context_type = ((($vars->{FORM}->{phrasedate} // 'no') eq 'yes') || (($vars->{report_type} // '') eq 'common')) && !$vars->{with_reach_stat} ? 1 : 0;
    my %fields_prop = (banner      => {  title => iget("№ Объявления"), width => 10,
                                         data_sub => sub { 'M-' . $_[0]->{bid}, $f_left }},
                       banner_href   => {  title => iget("Ссылка"), width => 10,
                                           data_sub => sub {
                                                my ($href, $domain) = _get_href_and_domain($_[0], try_get_domain_from_href => 1);
                                                ($href ? {data => $domain, url => $href} : undef), $f_left;
                                           }},
                       banner_title  => {  title => iget("Заголовок"), width => 20,
                                           data_sub => sub {
                                                html2string(_get_banner_title($_[0])) || undef, $f_left;
                                            }},
                       banner_body   => {  title => iget("Текст"), width => 10,
                                           data_sub => sub {
                                                {data => html2string($_[0]->{body}) || undef, as_text => 1}, $f_left;
                                            }},
                       banner_type   => {  title => iget("Тип объявления"), width => 20,
                                           data_sub => sub {
                                                iget($banner_type_text{$_[0]->{banner_type} // ''} // ''), $f_left;
                                            }},
                       mcb_banner  =>   {  title => iget("№ Объявления"), width => 10,
                                           data_sub => sub { 'T-' . $_[0]->{mbid}, $f_left }},
                       mcb_banner_title  => {  
                                           title => iget("Название"), width => 10,
                                           data_sub => sub {
                                                {data => $_[0]->{name}, as_text => 1}, $f_left;
                                            }},
                       mcb_banner_format  => {  
                                           title => iget("Формат"), width => 10,
                                           data_sub => sub {
                                                sprintf("%dx%d", @{$_[0]}{qw/width height/}), $f_left;
                                            }},
                       campaign_type => {  
                                         title => iget("Тип кампании"), width => 20,
                                         data_sub => sub {
                                                iget($campaign_type_text{$_[0]->{campaign_type} // ''} // ''), $f_left;
                                            }
                                      },

                       campaign    => [{ title => iget("Кампания"), width => 17, 
                                         data_sub => sub {
                                                # LibreOffice даже строку, начинающуюся на равно, пытается интерпретировать как формулу
                                                my $camp_name = $_[0]->{camp_name} =~ s/^(=)/ $1/r;
                                                $camp_name, $f_left;
                                            }
                                       },
                                       { title => iget("№ Кампании"), width => 10, data_field => 'cid', data_format => $f_left} ],

                       tags        => {  title => iget("Метка"), width => 17,
                                         data_sub => sub {
                                                if (($_[0]->{tag_names} // '') ne '') {
                                                    return $_[0]->{tag_names}, $f_left;
                                                } else {
                                                    return $i_deflabel, $f_gray;
                                                }
                                            }
                                      },
                       ab_segment => { title => iget("Сегмент"), width => 30,
                                           data_sub => sub {
                                                 return $_[0]->{ab_segment} || '-', $f_left
                                             }
                                     },
                       adgroup     => [{ title => iget("Группа"), width => 17,
                                         data_sub => sub { 
                                                $_[0]->{adgroup_id} 
                                                    ? ($_[0]->{adgroup_name} // $_[0]->{group_name}, $f_left) 
                                                    : ($i_defgroup, $f_gray);
                                            }
                                       },
                                       { title => iget("№ Группы"), width => 10, 
                                         data_sub => sub { 
                                                $_[0]->{adgroup_id} 
                                                    ? ("$_[0]->{adgroup_id}", $f_left) 
                                                    : ('', $f_gray);
                                            }
                                       }],
                       targettype  => { title => iget("Тип площадки"), width => 12, 
                                        data_sub => sub {
                                                iget($targettype_text{$_[0]->{targettype} // ''} // ''), $f_left;
                                            }},
                       page_group  => [{ title => iget("Площадка"), width => 15, data_field => 'page_name', data_format => $f_left} ],
                       page        => 'page_group', # инициализируется ниже, как копия page_group
                       contextcond => {  title => iget("Условие показа"), width => 30, 
                                        data_sub => sub {
                                                my $phrase_text;
                                                if ($_[0]->{ContextType} && $_[0]->{ContextType} == $CONTEXT_TYPE_PERFORMANCE) {
                                                    if ($need_context_type) {
                                                        $phrase_text = iget($performance_title_template, $_[0]->{phrase});
                                                    } else {
                                                        $phrase_text = $_[0]->{phrase};
                                                    }
                                                } elsif ($_[0]->{ContextType} && $_[0]->{ContextType} == $CONTEXT_TYPE_RET) {
                                                    if ($need_context_type && !$_[0]->{is_interest}) {
                                                        $phrase_text = iget($ret_title_template, $_[0]->{phrase});
                                                    } else {
                                                        $phrase_text = $_[0]->{phrase};
                                                    }
                                                } elsif ($_[0]->{category_id}) {
                                                    $phrase_text = iget("Рубрика: %s", $_[0]->{phrase});
                                                } else {
                                                    if ($need_context_type && $_[0]->{phrase} eq '---autotargeting') {
                                                        $phrase_text = iget($rm_title_template, "'" . $_[0]->{phrase});
                                                    } elsif ($_[0]->{phrase} =~ /^---/) {
                                                        $phrase_text = "'" . $_[0]->{phrase};
                                                    } else {
                                                        $phrase_text = $_[0]->{phrase};
                                                    }
                                                }
                                                return $phrase_text, $f_left
                                            }
                                      },
                       contextcond_ext => 'contextcond',
                       contextcond_orig => 'contextcond',
                       bs_criterion_id => { title => iget("№ Условия показа"), width => 10,
                                           data_sub => sub {
                                                 return $_[0]->{bs_criterion_id} || '-', $f_left
                                             }
                                         },
                       matched_phrase => { title => iget("Подобранная фраза"), width => 30,
                                           data_sub => sub {
                                                 return $_[0]->{matched_phrase} || '-', $f_left
                                             }
                                         },
                       match_type  => { title => iget("Тип соответствия"), width => 12, 
                                        data_sub => sub {
                                                iget($match_type_text{$_[0]->{match_type} // ''} // ''), $f_left;
                                            }
                                      },
                       contexttype => { title => iget("Тип условия показа"), width => 12, 
                                        data_sub => sub {
                                                iget($contexttype_text{$_[0]->{contexttype} // ''} // ''), $f_left;
                                            }
                                      },
                       contexttype_orig => 'contexttype',
                       bm_type     => { title => iget("Тип дополнительной фразы"), width => 12, 
                                        data_sub => sub {
                                                iget($bm_type_text{$_[0]->{bm_type} // ''} // ''), $f_left;
                                            }
                                      },
                       sim_distance => { 
                                        title => iget("№ алгоритма подбора"), width => 10,
                                        data_sub => sub {
                                                return ($_[0]->{sim_distance} || '-'), $f_left;
                                            },
                                      },
                       phrase_state => { title => iget("Состояние"), width => 10,
                                         data_sub => sub {
                                                iget($phrase_state_text{$_[0]->{phrase_state} // ''} // ''), $f_left;
                                            }
                                       },
                       retargeting_coef => { title => iget("Условие подбора (корректировки)"), width => 20, 
                                             data_sub => sub {
                                                 return $_[0]->{coef_ret_cond_name} // '-', $f_left
                                             } },
                       search_query => { title => iget("Поисковый запрос"), width => 30, data_field => 'search_query', data_format => $f_left },
                       region      => {  title => iget("Регион таргетинга"), width => 15, data_field => 'region_name', data_format => $f_left},
                       physical_region => {  title => iget("Регион местонахождения"), width => 15, data_field => 'physical_region_name', data_format => $f_left},
                       position    => {  title => iget("Позиция"), width => 12,
                                         data_sub => sub {
                                                iget($position_text{$_[0]->{position} // 0} // ''), $f_left;
                                            }
                                      },
                       has_image   => {  title => iget("Изображение"), width => 10,
                                         data_sub => sub {
                                                ($_[0]->{has_image} ? $i_image : $i_notimage), $f_left;
                                            }
                                      },
                       image_size  => {  title => iget("Размер изображения"), width => 10,
                                         data_sub => sub {
                                                iget($image_size_text{$_[0]->{image_size}} // $_[0]->{image_size} // ''), $f_left;
                                            }
                                      },
                       banner_image_type => {
                                         title => iget("Формат"), width => 10,
                                         data_sub => sub {
                                                iget($banner_image_type_text{$_[0]->{banner_image_type}} // ''), $f_left;
                                            }
                                      },
                       device_type => {  title => iget("Тип устройства"), width => 12,
                                         data_sub => sub {
                                                iget($device_type_text{$_[0]->{device_type} // ''} // ''), $f_left;
                                            }
                                      },
                       click_place => {  title => iget("Место клика"), width => 12,
                                         data_sub => sub {
                                                iget($click_place_text{$_[0]->{click_place} // ''} // ''), $f_left;
                                            }
                                      },
                       gender      => {  title => iget("Пол"), width => 12,
                                         data_sub => sub {
                                                iget($gender_text{$_[0]->{gender} // ''} // ''), $f_left;
                                            }
                                      },
                       age         => {  title => iget("Возраст"), width => 12,
                                         data_sub => sub {
                                                iget($age_text{$_[0]->{age} // ''} // ''), $f_left;
                                            }
                                      },
                       detailed_device_type => {
                                         title => iget("Тип операционной системы"), width => 12,
                                         data_sub => sub {
                                                iget($detailed_device_type_text{$_[0]->{detailed_device_type} // ''} // ''), $f_left;
                                            }
                                      },
                       connection_type => {
                                         title => iget("Тип связи"), width => 12,
                                         data_sub => sub {
                                                iget($connection_type_text{$_[0]->{connection_type} // ''} // ''), $f_left;
                                            }
                                      },
                       ssp         => { title => iget("Внешние сети"), width => 15, data_field => 'ssp', data_format => $f_left},
                       date        => {  title => iget("Дата"), width => 17,
                                         data_sub => sub {
                                                my $date = $_[0]->{'date' . ($_[1] // '')};
                                                $date =~ s/^(\d{2}\.\d{2}\.)(\d{2})$/${1}20$2/g;
                                                $date = _format_value($date, field => 'date');
                                                $date =~ s!&ndash;!-!;
                                                return $date, $f_left;
                                            }
                                      },
                       search_query_status => { title => iget("Статус запроса"),
                                                data_sub => sub {
                                                    iget($cond_add_status_text{$_[0]->{search_query_status} // ''} // ''), $f_left;
                                                }
                                              },
                       ext_phrase_status => { title => iget("Статус условия"), width => 30, ,
                                                data_sub => sub {
                                                    iget($cond_add_status_text{$_[0]->{ext_phrase_status} // ''} // ''), $f_left;
                                                }
                                              },
                       criterion_type => {
                            title => iget("Тип условия показа"), width => 12, 
                            data_sub => sub {
                                                iget($criterion_type_text{$_[0]->{criterion_type} // ''} // ''), $f_left;
                                            }
                                      },

                       deal     => [{ title => iget("Сделка"), width => 17,
                                        data_sub => sub {
                                            $_[0]->{deal_export_id}
                                            ? ($_[0]->{deal_name}, $f_left)
                                            : ($i_nodeal, $f_gray);
                                        }
                                    },
                                    { title => iget("№ Сделки"), width => 10,
                                        data_sub => sub {

                                           $_[0]->{deal_export_id}
                                           ? ("$_[0]->{deal_export_id}", $f_left)
                                           : ('-', $f_left);
                                        }
                                    }],
                        inventory_type => {
                            title => iget("Тип инвентаря"), width => 10,
                            data_sub => sub {
                                iget($inventory_type_text{$_[0]->{inventory_type} // 'undefined'} // ''), $f_left;
                            }
                        },
                       turbo_page_type => {  title => iget("Турбо-площадка"), width => 12,
                                         data_sub => sub {
                                                iget($turbo_page_type_text{$_[0]->{turbo_page_type} // ''}) // '', $f_left;
                                            }
                                      },
                       targeting_category => {  title => iget("Категория таргетинга"), width => 12,
                                         data_sub => sub {
                                                iget($targeting_category_text{$_[0]->{targeting_category} // ''}) // '', $f_left;
                                            }
                                      },
                       prisma_income_grade => {  title => iget("Уровень платежеспособности"), width => 12,
                                         data_sub => sub {
                                                iget($prisma_income_grade_text{$_[0]->{prisma_income_grade} // ''}) // '', $f_left;
                                            }
                                      },
                        client_id         => {  title => $vars->{multi_clients_mode} && $vars->{enable_internal_campaigns}
                                                            ? iget("Название продукта")
                                                            : iget("Идентификатор клиента"),
                            width => 12,
                            data_sub => sub {
                                return $_[0]->{client_id} // '-', $f_left
                            }
                        },
                        client_login         => {  title => iget("Логин клиента"), width => 12,
                            data_sub => sub {
                                return $_[0]->{client_login} // '-', $f_left
                            }
                        },
                        place_id         => {  title => iget("Номер плейса"), width => 12,
                            data_sub => sub {
                                return $_[0]->{place_id} // '-', $f_left
                            }
                        },
                        place_description         => {  title => iget("Описание плейса"), width => 12,
                            data_sub => sub {
                                return $_[0]->{place_description} // '-', $f_left
                            }
                        },
                        operating_system         => {  title => iget("Операционная система"), width => 12,
                            data_sub => sub {
                                return $_[0]->{operating_system} // '-', $f_left
                            }
                        },
                        browser         => {  title => iget("Браузер"), width => 12,
                            data_sub => sub {
                                return $_[0]->{browser} // '-', $f_left
                            }
                        },
                        page_id         => {  title => iget("Номер пейджа"), width => 12,
                            data_sub => sub {
                                return $_[0]->{page_id} // '-', $f_left
                            }
                        },

                    );
    $fields_prop{content_targeting} = { title => iget("Жанры и тематики"),
                                        width => 17,
                                        data_sub => sub {
                                            return $_[0]->{content_targeting_name} // '-', $f_left;
                                        }
                                      };

    foreach my $field (keys %fields_prop) {
        # подставляем параметры для полей-копий
        $fields_prop{$field} = $fields_prop{$fields_prop{$field}} unless ref $fields_prop{$field};
        # добавляем общий формат для заголовка
        for my $fp (xflatten $fields_prop{$field}) {
            $fp->{format} = hash_merge $fp->{format} // {}, $f_b8;
        }
    }

    my %suffix_titles = ('_a'        => iget(' (период A)'),
                         '_b'        => iget(' (период B)'),
                         '_delta'    => iget(' (разница, %)'),
                         '_absdelta' => iget(' (разница)'),
                         '' => '/' . iget('всего'),
                         '_0' => '/' . iget('поиск'),
                         '_1' => '/' . iget('сети'),
                          );
    my %non_currency_suffix = map { $_ => 1 } qw/_delta/;

    my @suffix_list = $vars->{compare_periods} ? qw/_delta _absdelta _a _b/ : get_suffixes($vars);

    my %extra_column_title = (winrate => iget('Процент полученных показов (%)'),
                             bounce_ratio => iget('Отказы (%)'),
                             pv_bounce_ratio => iget('Post-view: Отказы (%)'));

    my $camp_type = $vars->{_extra}->{report_opts}->{camp_type} // '';
    if ($camp_type eq 'cpm_price') {
        $extra_column_title{ad_site_clicks} = iget('Открытие раскрывающегося блока');
    }

    if (camp_kind_in(type => $camp_type, 'cpm') 
        && $Stat::Const::ATTRIBUTION_MODEL_TYPES{last_yandex_direct_view_cross_device}->{bs} == $attribution_model_bs_value
        && $vars->{features_enabled_for_client}->{post_view_statistics}) {
        for my $column_value (values %column_name) {
            if ($post_click_field_names{$column_value->[0]}) {
                $column_value->[1] = 'Post-click: '.$column_value->[1];
            }
        }
    }

    my %countable_fields = map { $_->[0] => {title => $_->[1], opt => $_->[2] // {}} } values %column_name;
    for my $name (keys %extra_column_title) {
        $countable_fields{$name} = {
            opt => {},
            %{ $countable_fields{$name} || {} },
            title => $extra_column_title{$name},
        };
    }

    my @group_by_fields_list = (($vars->{group_by_date} && $vars->{group_by_date} ne 'none' ? 'date' : ()),
                                @{$vars->{group_by}} );
    if (($vars->{stat_type} // '') eq 'ext_phrases') {
        my $ind = List::MoreUtils::first_index {$_ eq 'banner'} @group_by_fields_list;
        my $ind_title = List::MoreUtils::first_index {$_ eq 'banner_title'} @group_by_fields_list;
        if (defined $ind && !defined $ind_title) {
            splice @group_by_fields_list, $ind + 1, 0, 'banner_title';
        }
    }

    # формируем заголовок таблицы со статистикой
    my @table_header = ();
    my @columns_width = ();
    for my $field (@group_by_fields_list) {
        for my $fp (xflatten $fields_prop{$field}) {
            for my $suf ($vars->{compare_periods} && $field eq 'date' ? periods_suffix_list() : '') {
                push @table_header, {data => $fp->{title} . ($suf ? $suffix_titles{$suf} // '' : ''), format => $fp->{format}};
                push @columns_width, $fp->{width};
            }
        }
    }

    my %multigoal_fields = map {$_ => undef} Stat::Fields::get_multigoal_field_names;
    delete @multigoal_fields{qw/agoalroi agoalincome pv_agoalroi pv_agoalincome/}; # https://st.yandex-team.ru/DIRECT-116896#5ecce321d63e5d3b7a087ab5
    my @countable_columns_caption = ();
    for my $field (@{$vars->{columns}}) {
        my $cf = $countable_fields{$field};
        if (@goal_ids && exists $multigoal_fields{$field}) {
            for my $goal_id (@goal_ids) {
                push @countable_columns_caption, iget($cf->{title}, _apply_column_opt($vars, $cf->{opt}, undef))
                                                . '/' . ($goal_names{$goal_id} // $goal_id);
                push @columns_width, 10;
            }
        } else {
            for my $suf (@suffix_list) {
                push @countable_columns_caption, iget($cf->{title}, _apply_column_opt($vars, $cf->{opt}, $non_currency_suffix{$suf}))
                                                . ($suf || scalar(@suffix_list) > 1 ? $suffix_titles{$suf} // '' : '');
                push @columns_width, 10;
            }
        }
    }

    # итоговые суммы по всей выгруженной статистике
    # для выгрузки в CSV разделителем целой и дробной частей числа используем запятую
    my %comma_sep = $format eq 'csv' ? (comma_sep => 1) : ();

    my $primary_totals_title = iget("Всего");
    my $primary_totals_data = $vars->{totals};

    my ($secondary_totals_title, $secondary_totals_data);
    if (($vars->{stat_type} // '') eq 'search_queries') {
        $primary_totals_title = iget("Всего по поисковым запросам");
        $secondary_totals_title = iget("Другие поисковые запросы");
        $secondary_totals_data = $vars->{other_search_queries_totals};
    }

    push @data, [map { {data => $_, format => $f_b8} } ((map { $primary_totals_title . ($_ ? $suffix_titles{$_} // '' : '') } ($vars->{compare_periods} ? periods_suffix_list() : '')), "",
        ($vars->{with_reach_stat} ? (iget("Охват"), iget("Ср. частота показов"))
                                                        : (map { ($is_media_order 
                                                                    ? iget("Ср. показы за день, тыс.") 
                                                                    : iget("%sСр.расход за день%s", _apply_column_opt($vars, {currency => 1}, $non_currency_suffix{$_})) ) . 
                                                                      ($_ || scalar(@suffix_list) > 1 ? $suffix_titles{$_} // '' : '') } @suffix_list)), 
                                                        @countable_columns_caption)];
    my @totals_row = map { iget("с %s по %s", date($vars->{"date_from$_"})->dmy('.'), date($vars->{"date_to$_"})->dmy('.')) } ($vars->{compare_periods} ? ('', '_b') : '');
    push @totals_row, "";
    my @totals_countable_columns = ($vars->{with_reach_stat} ? ('uniq_viewers', 'avg_view_freq') : $is_media_order ? 'av_day_shows' : 'av_day', @{$vars->{columns}});
    for my $f (@totals_countable_columns) {
        if (@goal_ids && exists $multigoal_fields{$f}) {
            for my $goal_id (@goal_ids) {
                push @totals_row, _format_value($primary_totals_data->{join('_', $f, $goal_id, $attribution_model_bs_value)},
                    field => $f, four_digits_precision => $O{four_digits_precision}, %comma_sep);
            }
        } else {
            for my $suf (@suffix_list) {
                push @totals_row, _format_value($primary_totals_data->{$f.$suf}, field => $f, four_digits_precision => $O{four_digits_precision}, %comma_sep);
            }
        }
    }
    push @data, [ map { {data => $_, format => $f_b8} } @totals_row ];

    if ($secondary_totals_data) {
        my @secondary_totals_row = ($secondary_totals_title, "");
        for my $f (@totals_countable_columns) {
            for my $suf (@suffix_list) {
                push @secondary_totals_row, _format_value($secondary_totals_data->{$f.$suf}, field => $f, four_digits_precision => $O{four_digits_precision}, %comma_sep);
            }
        }
        push @data, [ map { {data => $_, format => $f_b8} } @secondary_totals_row ];
    }

    push @data, [undef];

    my $output_data;
    my $data_writer;
    my $xls_worksheet;
    my $fh = new IO::Scalar \$output_data;

    # проверяем, не нужно ли зафоллбечиться xls => xlsx
    # инициализируем Writer
    my $rows_num = 0;
    my $chunk_size = $format eq 'xls' ? MAX_XLS_ROWS-scalar(@data)+1 : undef;
    my $data_source = ($O{use_stat_iterator} ? $vars->{stat_iterator} : $vars->{data_array});

    # аккумулируем время обработки первого чанка данных (вплоть до записи в xls), чтоб спрогнозировать, успеет ли отчет сформироваться
    # за лимит отведенного времени (сейчас - 10мин таймаут nginx-а)
    my $process_first_chunk_t0 = [gettimeofday];
    my ($chunk, $rows_num_forecast) = _next_data_chunk($data_source, $rows_num, $chunk_size);
    if ($rows_num_forecast && $rows_num_forecast > $ROWS_NUM_TO_LOG_REQUEST) {
        Stat::Tools::log_report_master_heavy_request({format => $format,
                                                      rows_num_forecast => $rows_num_forecast,
                                                      uid => $vars->{_extra}->{uid}, 
                                                      }, 
                                                      $vars->{_extra}->{report_opts});
    }
    my $process_first_chunk_time = tv_interval($process_first_chunk_t0);

    if ($format eq 'xls' && scalar(@data) + scalar(@{$chunk // []}) > MAX_XLS_ROWS) {
        $format = 'xlsx';
    }

    my @bids = ();
    if ($vars->{_extra}->{report_opts}->{with_resources} && $vars->{_extra}->{report_opts}->{with_resources} == 1) {
        for my $ch (@$chunk) {
            if (defined $ch->{bid} && $ch->{bid} > 0) {
                push @bids, $ch->{bid};
            }
        }
    }
    my @template_ids = ();
    my $internal_ad_infos = {};

    my $has_group_by_client_id = List::MoreUtils::any { $_ eq 'client_id' } @{$vars->{_extra}{report_opts}{group_by}};
    if ($vars->{multi_clients_mode} && $vars->{enable_internal_campaigns} && $has_group_by_client_id) {
        my @client_ids = ();
        for my $ch (@$chunk) {
            if (defined $ch->{client_id}) {
                push @client_ids, $ch->{client_id};
            }
        }

        @client_ids = uniq @client_ids;
        if ( @client_ids ) {
            my $product_name_by_client_id = get_internal_product_names_by_client_ids(\@client_ids);

            for my $ch (@$chunk) {
                if (defined $ch->{client_id}) {
                    $ch->{client_id} = $product_name_by_client_id->{$ch->{client_id}};
                }
            }
        }
    }

    if (@bids) {
        my @uniq_bids = uniq @bids;
        my $bid2shard = get_shard_multi('bid' => \@uniq_bids);

        for my $shard (uniq values %$bid2shard) {
            my @shardbids = grep {$bid2shard->{$_} eq $shard} keys $bid2shard;
            hash_merge $internal_ad_infos, JavaIntapi::GetInternalAdInfo->new(
                shard => $shard,
                bids => \@shardbids,
            )->call;
        }

        for my $bid (@uniq_bids){
            if (exists $internal_ad_infos->{$bid}->{template_id}){
                push @template_ids, $internal_ad_infos->{$bid}->{template_id};
            }
        }
    }
    my $template_resources = [];
    # добавляем заголовок таблицы для значений ресурсов внутренних объявлений
    # при условии что все объявления одного шаблона
    my $with_resources_data;
    if (scalar(uniq @template_ids) == 1 && $vars->{_extra}->{report_opts}->{with_resources} == 1) {
        $with_resources_data = 1;
    }

    if ($with_resources_data) {
        my @uniq_template_ids = uniq @template_ids;
        $template_resources = JavaIntapi::GetTemplateResource->new(
            template_ids => \@uniq_template_ids,
        )->call;
         for my $templ_res (@$template_resources) {
             push @table_header, {data => $templ_res->{description}, format => $f_b8};
             push @group_by_fields_list, $templ_res->{id};
         }
    }
    push @table_header, map { {data => $_, format => $f_b8} } @countable_columns_caption;
    # добавляем заголовок таблицы со статистикой
    my $table_header_idx;
    if ($chunk) {
        push @data, \@table_header;
        $table_header_idx = $#data;
    }

    my %sheetname_by_stat_type = ('search_queries' => iget("Поисковые запросы"), );
    my $xls_format = {sheetname => $O{sheetname} // $sheetname_by_stat_type{$vars->{stat_type} // ''} 
                                                 // iget("Мастер отчётов"), 
                      set_column => [],
                      defined $table_header_idx ? (freeze_panes => [$table_header_idx+1, 0]) : (),};
    for my $col (0..$#columns_width) {
        push @{$xls_format->{set_column}}, {col1 => $col, count => 1, width => $columns_width[$col]};
    }

    if (List::MoreUtils::any { $format eq $_ } qw/xls xlsx/) {
        if ($format eq 'xls') {
            $data_writer = new Yandex::ReportsXLSWriter($fh);
        } elsif ($format eq 'xlsx') {
            $data_writer = new Yandex::ReportsXLSXWriter($fh);
        }
        $xls_worksheet = $data_writer->add_worksheet($xls_format->{sheetname});
        $data_writer->add_data($xls_worksheet, $xls_format, \@data);
    } else {
        $data_writer = new Yandex::CSVWriter(\$output_data, bom_header => 1, sep_char => ';');
        _prepare_data_for_csv(\@data);
        $data_writer->add_data(\@data);
    }
    @data = ();
    $chunk_size = undef;

    # добавляем в таблицу собственно статистику
    my $process_time_forecast;
    while ($chunk) {
        $process_first_chunk_t0 = [gettimeofday];

        $rows_num += scalar(@$chunk);
        if ($O{rows_limit} && ($rows_num > $O{rows_limit} 
                                || (!defined $O{process_time_limit} || defined $process_time_forecast) 
                                    && $rows_num_forecast 
                                    && $rows_num_forecast > $O{rows_limit}) ) {
            Stat::Tools::log_report_master_heavy_request({error => 'XLS_ROWS_LIMIT_EXCEEDED_ERROR', 
                                                          format => $format,
                                                          rows_num_forecast => $rows_num_forecast,
                                                          process_time_forecast => $process_time_forecast,
                                                          uid => $vars->{_extra}->{uid}, 
                                                          }, 
                                                          $vars->{_extra}->{report_opts});
            die \$Stat::Const::XLS_ROWS_LIMIT_EXCEEDED_ERROR;
        }

        if ($with_resources_data) {
            for my $templ_res (@$template_resources) {
                $fields_prop{$templ_res->{id}} = {
                    title => iget($templ_res->{description}), width => 12, format => $f_b8,
                    data_sub => sub {
                        my $display_values_by_id = {map { $_->{template_resource_id} => $_->{display_value} }
                            @{$internal_ad_infos->{$_[0]->{bid}}->{template_variables}} };
                        my $display_value = $display_values_by_id->{$templ_res->{id}};
                        return $display_value // '-', $f_left;
                    }
                };
            }
        }
        for my $d (@$chunk) {
            my @row = ();
            for my $field (@group_by_fields_list) {
                for my $fp (xflatten $fields_prop{$field}) {
                    for my $suf ($vars->{compare_periods} && $field eq 'date' ? periods_suffix_list() : '') {
                        my $col = {};
                        if ($fp->{data_field}) {
                            ($col->{data}, $col->{format}) = ($d->{$fp->{data_field} . $suf}, $d->{data_format} // {});
                        } else {
                            ($col->{data}, $col->{format}) = $fp->{data_sub}->($d, $suf);

                            if (ref($col->{data}) eq 'HASH') {
                                my $data = delete $col->{data};
                                hash_merge($col, $data);
                            }
                        }
                        push @row, $col;
                    }
                }
            }
            for my $field (@{$vars->{columns}}) {
                if (@goal_ids && exists $multigoal_fields{$field}) {
                    for my $goal_id (@goal_ids) {
                        push @row, {data => _format_value($d->{join('_', $field, $goal_id, $attribution_model_bs_value)},
                        field => $field, four_digits_precision => $O{four_digits_precision}, %comma_sep), format => $f_b8};
                    }
                } else {
                    for my $suf (@suffix_list) {
                        push @row, {data => _format_value($d->{$field.$suf}, field => $field, four_digits_precision => $O{four_digits_precision}, %comma_sep),
                                    format => $f_b8};
                    }
                }
            }
            push @data, \@row;
        }

        my $profile = Yandex::Trace::new_profile('stat_xls:statxls_report_master', tags => 'add_data_to_xls');
        if (List::MoreUtils::any { $format eq $_ } qw/xls xlsx/) {
            $data_writer->add_data($xls_worksheet, {}, \@data);
        } else {
            _prepare_data_for_csv(\@data);
            $data_writer->add_data(\@data);
        }
        undef $profile;
        @data = ();
        # после первой порции данных считаем что все "общелистовые" форматы (цвета, фризы, ширина колонок) уже применены
        $xls_format = {};

        $process_first_chunk_time += tv_interval($process_first_chunk_t0);
        if (defined $O{process_time_limit}) {
            if (!defined $process_time_forecast && $rows_num_forecast && scalar(@$chunk)) {
                $process_time_forecast = $rows_num_forecast / scalar(@$chunk) * $process_first_chunk_time;

                if ($process_time_forecast > $O{process_time_limit}) {
                    Stat::Tools::log_report_master_heavy_request({error => 'XLS_PROCESS_TIME_LIMIT_EXCEEDED_ERROR', 
                                                                  format => $format,
                                                                  rows_num_forecast => $rows_num_forecast,
                                                                  process_time_forecast => $process_time_forecast,
                                                                  uid => $vars->{_extra}->{uid}, 
                                                                  }, 
                                                                  $vars->{_extra}->{report_opts}, );
                    die \$Stat::Const::XLS_PROCESS_TIME_LIMIT_EXCEEDED_ERROR;
                }
            } else {
                $process_time_forecast = 0;
            }
        }

        ($chunk, $rows_num_forecast) = _next_data_chunk($data_source, $rows_num, $chunk_size);
    }

    my $note = _get_discount_note($vars);
    if ($note) {
        push @data, ([undef], [{data => $note}]);
        if (List::MoreUtils::any { $format eq $_ } qw/xls xlsx/) {
            $data_writer->add_data($xls_worksheet, {}, \@data);
        } else {
            _prepare_data_for_csv(\@data);
            $data_writer->add_data(\@data);
        }
    }
    $data_writer->close();
    return $format => $output_data;
}

=head2 _next_data_chunk

    возвращет очередной чанк данных со статистикой, и прогнозное общее количество строк в источнике

    my ($chunk, $rows_num_forecast) = _next_data_chunk($data_source, $processed_rows_num, $chunk_size);

=cut

sub _next_data_chunk {
    my ($data_source, $processed_rows_num, $chunk_size) = @_;

    $chunk_size //= 50_000;
    return ref($data_source) eq 'ARRAY'
                            ? ( ($processed_rows_num < scalar(@$data_source) ? [@{$data_source}[ $processed_rows_num .. min($processed_rows_num + $chunk_size - 1, $#{$data_source}) ]] : undef),
                                scalar(@$data_source) )
                            : ( $data_source->next_chunk( chunk_size => $chunk_size ),
                                $data_source->headers->{recieved_rows_forecast} );
}

=head2 _prepare_data_for_csv

    отбрассываем из данных все кроме собственно наполнения ячеек (для выгрузки в csv)

=cut

sub _prepare_data_for_csv {
    my $data = shift;

    for my $row (@$data) {
        @$row = map { ref $_ ? $_->{data} : $_ } @$row;
    }
}

}

=head2 _format_value

    Округляет, заменяет undef и CTR >100% (вчерашние клики) на прочерки

=cut

sub _format_value {
    my ($val, %O) = @_;
    state $extended_precision_fields //= {map { $_ => 1 } Stat::Fields::get_countable_extended_precision_fields()};

    if ($O{field} eq 'ctr' && defined $val && $val > 100) {
        return '-';
    }
    my $sprintf_fmt = "%.2f";
    if ($extended_precision_fields->{$O{field}} || ($O{four_digits_precision} && exists $money_fields{$O{field}})) {
        $sprintf_fmt = "%.4f";
    }
    $val = defined($val) ? ($val =~ /^\d+\.\d+$/ ? sprintf($sprintf_fmt, $val) :  $val ) : '-';
    return $O{comma_sep} ? $val =~ s/\./,/gr : $val; 
}

=head2 statxls_mcb_theme_mutation

# Вывести статистику по географии в excel

=cut

sub statxls_mcb_theme_mutation {
    my $vars = shift;
    my $skip = shift() || {};
    my $out  = shift;

    my $work = Spreadsheet::WriteExcel->new($out || \*STDOUT);
    my $list = $work->addworksheet(iget("Изменения пакетов МКБ"));

    my $ccnt;

    $ccnt=0;
    $list->set_column($ccnt, $ccnt++, 10);
    $list->set_column($ccnt, $ccnt++, 12);
    $list->set_column($ccnt, $ccnt++, 20);
    $list->set_column($ccnt, $ccnt++, 30);
    $list->set_column($ccnt, $ccnt++, 3);
    $list->set_column($ccnt, $ccnt++, 10);
    $list->set_column($ccnt, $ccnt++, 10);
    $list->set_column($ccnt, $ccnt++, 10);
    $list->set_column($ccnt, $ccnt++, 6);
    $list->set_column($ccnt, $ccnt++, 6);
    $list->set_column($ccnt, $ccnt++, 12);
    $list->set_column($ccnt, $ccnt++, 12);
    $list->set_column($ccnt, $ccnt++, 12);
    $list->set_column($ccnt, $ccnt++, 30);

    my $f_right = $work->addformat( align => 'right' );
    my $f_left = $work->addformat( align => 'left' );
    my $f_bold_left = $work->addformat( align => 'left', bold=>1 );
    my $f_bold_right = $work->addformat( left=>1, align => 'right' );
    my $f_bold = $work->addformat( left=>1, align => 'right', bold=>1 );

    my $row = 0;

    $ccnt=0;
    $list->write($row, $ccnt++, iget("№ заказа"));
    $list->write($row, $ccnt++, iget("клиент"));
    $list->write($row, $ccnt++, iget("название пакета"));
    $list->write($row, $ccnt++, iget("фраза"));
    $list->write($row, $ccnt++, iget("доб/удал"));
    $list->write($row, $ccnt++, iget("прогноз по фразе"));
    $list->write($row, $ccnt++, iget("прогноз по итог. пакету"));
    $list->write($row, $ccnt++, iget("прогноз по исх. пакету"));

    $list->write($row, $ccnt++, iget("итог. кол-во фраз"));
    $list->write($row, $ccnt++, iget("исх. кол-во фраз"));

    $list->write($row, $ccnt++, iget("начало кампании"));
    $list->write($row, $ccnt++, iget("менеджер"));
    $list->write($row, $ccnt++, iget("агентство"));
    $list->write($row, $ccnt++, iget("ссылки на сайты"));

    $row++;

    for my $order ( sort {$a->{cid} <=> $b->{cid}} values %{$vars->{orders}} ) {
        for my $g ( sort {$a->{mgid} <=> $b->{mgid}} values %{$order->{groups}} ) {
            for my $sign(qw/plus_phrases minus_phrases/) {
                for my $ph (sort {$a->{phrase} cmp $b->{phrase}} values %{$g->{$sign}} ) {

                    my @Localtime = Localtime(str2time($order->{start_time}));
                    $ccnt=0;
                    $list->write($row,$ccnt++, $order->{cid}, $f_left);
                    $list->write_string($row,$ccnt++, $order->{login});
                    $list->write_string($row,$ccnt++, $g->{mcb_theme_name});
                    $list->write_string($row,$ccnt++, $ph->{phrase});
                    $list->write($row,$ccnt++, $sign =~ m/plus/ ? '+' : '-');
                    $list->write($row,$ccnt++, $ph->{showsForecast}||0);
                    $list->write($row,$ccnt++, $g->{FinalShowsForecast}||0);
                    $list->write($row,$ccnt++, $g->{OrigShowsForecast}||0);
                    $list->write_string($row,$ccnt++, scalar values %{$g->{final_phrases}});
                    $list->write_string($row,$ccnt++, scalar values %{$g->{orig_phrases}});
                    $list->write($row,$ccnt++, sprintf("%02d.%02d.%04d", reverse @Localtime[0..2] ));
                    $list->write_string($row,$ccnt++, $order->{mlogin});
                    $list->write_string($row,$ccnt++, $order->{alogin});
                    $list->write_string($row,$ccnt++, $g->{href_list});

                    $row++;
                }
            }
        }
    }

}

=head2 _get_stat_currency

    Возвращает валюту, в которой получилась статистика

    $currency = _get_stat_currency($vars);

=cut

sub _get_stat_currency {
    my ($vars) = @_;

    # если статистика по одной кампании - то есть campaign.currency, иначе берем валюту клиента (work_currency) или валюту статистики (currency)
    return ($vars->{campaign} // {})->{currency} || $vars->{currency} || $vars->{work_currency} || 'YND_FIXED'; # currency_defaults
}

=head2 _need_note

    Нужно ли показывать примечание про НДС и скидку

    $need_note = _need_note($vars);

=cut

sub _need_note {
    my ($vars) = @_;

    return (!$vars->{no_data_found} && _get_stat_currency($vars) ne 'YND_FIXED' && ($vars->{had_nds} || $vars->{had_discounts})) ? 1 : 0;
}

=head2 _apply_column_opt

    Функция возвращает дополнительные параметры для iget, если они нужны

=cut

sub _apply_column_opt
{
    my ($vars, $opt, $is_non_currency_suffix) = @_;
    my @res;
    if ($opt->{currency} || $opt->{currency_only}) {
        push @res, (_need_note($vars) ? '* ' : '') if $opt->{currency};
        if ($vars->{no_data_found}) {
            push @res, '';
        } else {
            my $currency = _get_stat_currency($vars);
            push @res, ' (' . format_currency($currency) . ')';
        }
    }
    if ($is_non_currency_suffix) {
        @res = map { '' } @res;
    }
    return @res;
}

=head2 _get_discount_note

    Возвращает строку о применении НДС и/или скидки

    $note = _get_discount_note($vars);

=cut

sub _get_discount_note {
    my ($vars) = @_;

    if (_need_note($vars)) {
        if ($vars->{had_nds} && $vars->{had_discounts}) {
            if ($vars->{with_nds} && $vars->{with_discount}) {
                return iget('* Данные приведены с учётом НДС и скидки');
            } elsif ($vars->{with_nds} && !$vars->{with_discount}) {
                return iget('* Данные приведены с учётом НДС до применения скидки');
            } elsif (!$vars->{with_nds} && $vars->{with_discount}) {
                return iget('* Данные приведены без учёта НДС после применения скидки');
            } else {
                return iget('* Данные приведены без учета НДС и до применения скидки');
            }
        } elsif ($vars->{had_nds}) {
            if ($vars->{with_nds}) {
                return iget('* Данные приведены с учётом НДС');
            } else {
                return iget('* Данные приведены без учёта НДС');
            }
        } elsif ($vars->{had_discounts}) {
            if ($vars->{with_discount}) {
                return iget('* Данные приведены с учётом скидки');
            } else {
                return iget('* Данные приведены без учета скидки');
            }
        }
    }

    return undef;
}

=head3 _get_href_and_domain($banner, %params)

    Получить ссылку и домен по данным баннера.

    my ($href, $domain) = _get_href_and_domain($banner);

    Параметры позиционные:
        $banner - ссылка на хеш с данными по баннеру (используемые ключи:
                    href, domain, real_banner_type, adgroup_main_domain)
    Параметры именованные:
        try_get_domain_from_href    - флаг, разрешающий вычисление $domain из $href,
                                      если домен не определен в баннере

    Результат:
        $href   - ссылка, может быть undef
        $domain - домен, может быть undef

=cut
sub _get_href_and_domain {
    my ($banner, %params) = @_;
    my ($href, $domain);

    if ($banner->{href}) {
        $href = Encode::encode('utf8', clear_banner_href($banner->{href}));
        $domain = $banner->{domain};
        if (!$domain && $params{try_get_domain_from_href}) {
            $domain = MirrorsTools::url2host({}, $banner->{href})
        }
    } elsif (($banner->{real_banner_type} // '') eq 'dynamic' && $banner->{adgroup_main_domain}) {
        $domain = $banner->{adgroup_main_domain};
        $href = Encode::encode('utf8', "http://$domain");
    }

    return ($href, $domain);
}

=head3 _get_banner_title($banner)

    Получить название баннера по данным баннера.

    my $banner_title = _get_banner_title($banner);

    Параметры позиционные:
        $banner - ссылка на хеш с данными по баннеру (используемые ключи:
                    real_banner_type, title, image_ad, creative)

    Результат:
        $banner_title - название баннера или строка в формате "ширина х высота" для ГО

=cut

sub _get_banner_title {
    my ($banner) = @_;
    my $banner_title;
    if (($banner->{real_banner_type} // '') ne 'image_ad') {
        $banner_title = $banner->{title};
    } else {
        if ($banner->{image_ad}) {
            $banner_title = "$banner->{image_ad}->{width} x $banner->{image_ad}->{height}";
        } else {
            $banner_title = "$banner->{creative}->{width} x $banner->{creative}->{height}";
        }
    }
    return $banner_title;
}

1;

