package StoredVars;

use warnings;
use strict;
use utf8;

use Settings;
use Yandex::DBTools;
use Yandex::HashUtils; 
use Yandex::Compress;
use HashingTools;
use YAML::Syck;
use Data::MessagePack;

our $MAX_YAML_DATA_SIZE = 12 * 1024 * 1024;
our $MAX_STORED_VARS_COUNT = 5;

=head2 new

    Создание объекта StoredVars для храненния сессионных данных

    Параметры:
        $hash => ссылка на хеш с параметрами, где
            data => данные, которые нужно сохранить
            name => ключ сессии (нужен при загрузке данных)
            uid => id пользователя
            cid => id кампании
            prefer_string => 0|1  использовать при сохранении/загрузке строкое представление для чисел, вместо перевода в число
                                  например: 000123 будет сохранен как "000123", а не 123

=cut

sub new($$) {
    my ($class, $hash) = @_;

    my $self  = {
        dbh => MONITOR,
        statusLocked => 'No'
    };

    hash_merge $self, $hash if $hash;

    bless $self, $class;

    return $self;
}

=head2 dbh

    Возвращяет хэндл текущего подключения к БД

=cut

sub dbh($) {
    my $self = shift;
    return get_dbh($self->{dbh});
}

=head2

    Устанавливает/возвращяет сохраненные сессионные данные

=cut

sub data($;$) {
    my $self = shift;

    $self->{data} = $_[0] if scalar @_;

    return $self->{data};
}

=head2

    Сохранить сессионные данные в БД

=cut

sub save($) {
    my $self = shift;

    # TODO: хорошо бы проверку на размер, иначе будем мучаться с битыми данными

    # deflate(Data::MessagePack) по сравнению с deflate(YAML) эффективнее по размеру данных в 5 раз
    my $mp = Data::MessagePack->new();
    $mp->canonical->utf8;
    $mp->prefer_integer unless $self->{prefer_string};
    my $packed = $mp->pack($self->{data});
    $self->{yaml_data} = deflate $packed;

    return undef if length($self->{yaml_data}) > $MAX_YAML_DATA_SIZE;
    $self->{name} = url_hash(join("#", map { $self->{$_} || 0 } qw/yaml_data uid cid/));

    my $ids = join(",", @{get_one_column_sql($self->{dbh}, qq{
        select id
        from stored_vars
        where uid = ?
        order by logdate desc
        limit $MAX_STORED_VARS_COUNT
    }, $self->{uid}) || []});

    do_sql($self->{dbh}, qq{
        delete from stored_vars
        where uid = ?
        and id not in ($ids)
    }, $self->{uid}) if $ids;

    do_sql($self->{dbh}, qq{
        replace into stored_vars (yaml_data, name, uid, cid, statusLocked)
        values (?,?,?,?,?)
    }, map { $self->{$_} || 0 } qw/yaml_data name uid cid statusLocked/) or return undef;

    $self->{id} = $self->dbh()->last_insert_id(undef, undef, undef, undef);
    return $self;
}

=head2

    Загрузить сессионные данные из БД

=cut

sub load($) {
    my $self = shift;

    my $vars = get_one_line_sql($self->{dbh}, qq{
        select id, cid, uid, yaml_data, name, statusLocked
        from stored_vars
        where name = ? and uid = ? and cid = ?
    }, map { $self->{$_} || 0 } qw/name uid cid/) || {};

    if ($vars and $vars->{id}) {
        eval {
            hash_merge $self, $vars; 
            my $packed = inflate $self->{yaml_data};
            # EXP 2013-12-01 удалить следующий eval и YAML::Syck::Load, когда в таблице не останется данных, закодированых в YAML
            eval {
                my $mp = Data::MessagePack->new();
                $mp->canonical->utf8;
                $mp->prefer_integer unless $self->{prefer_string};
                $self->{data} = $mp->unpack($packed);
                1;
            } or do {
                $self->{data} = YAML::Syck::Load $packed;
            };
            1;
        } or do {
            warn "FAIL: $@\n";
            return undef;
        };
        return $self;
    }

    return undef;
}

=head2

    Установить флаг блокировки сессионых данных в БД

=cut

sub lock($) {
    my ($self) = @_;
    
    my $res = do_sql($self->{dbh}, qq{
        update stored_vars set statusLocked = 'Yes' where name = ? and uid = ? and cid = ? and statusLocked != 'Yes'
    }, map { $self->{$_} || 0 } qw/name uid cid/);

    return $res;
}

=head2

    Удалить ранее сохраненные сессионные данные

=cut

sub delete($) {
    my ($self) = @_;

    do_sql($self->{dbh}, qq{delete from stored_vars where name = ? and uid = ? and cid = ?}, map { $self->{$_} || 0 } qw/name uid cid/);
}

1;
