package Surveys;

# $Id$

use 5.010;
use strict;
use warnings;
use utf8;

use Carp;
use Exception::Class ( 'SurveyError' );
use JSON;

use Yandex::TimeCommon;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::HashUtils;
use Yandex::I18n qw/iget/;

use Settings;

use Survey::Surveymonkey;
use Survey::TNS;



our @MUTABLE_FIELDS = qw/ title start_date finish_date /;
our @COMMON_FIELDS = (
    qw/ type /,
    @MUTABLE_FIELDS,
);

our %SURVEY_CLASS = (
    surveymonkey => 'Survey::Surveymonkey',
    tns =>          'Survey::TNS',
);


=head2 get_user_survey_url

Индивидуальный урл для опроса

=cut

sub get_user_survey_url
{
    my ($survey_data, %opt) = @_;

    my $ext_survey = get_ext_survey($survey_data);
    return $ext_survey->get_user_url(%opt);
}


=head2 get_ext_survey

Возвращает объект класса-адаптера опроса

=cut

sub get_ext_survey
{
    my ($survey_data) = @_;
    
    my $type = $survey_data->{type};
    SurveyError->throw(error => iget("Не задан тип опроса"))  if !$type;

    my $class = $SURVEY_CLASS{$type};
    SurveyError->throw(error => iget("Неизвестный тип опроса: %s", $type))  if !$class;

    my $ext_survey = $class->new(%$survey_data);
    return $ext_survey;
}




=head2 get_user_active_surveys

    my $surveys = get_user_active_surveys($uid)

Активные опросы пользователя

=cut

sub get_user_active_surveys
{
    my ($uid) = @_;

    my $user_ids = get_hash_sql(PPC(uid => $uid),
        'SELECT survey_id, ext_user_id FROM users_surveys WHERE uid=? AND status="New"', $uid
    );

    return []  if !%$user_ids;

    my $surveys = get_surveys( condition => {
            survey_id => [keys %$user_ids],
            _TEXT => 'now() BETWEEN start_date AND finish_date',
        });

    my @survey_forms = map {{
            survey_id => $_->{survey_id},
            title => $_->{title},
            url => get_user_survey_url($_, uid => $uid, ext_user_id => $user_ids->{$_->{survey_id}}),
        }} @$surveys;

    return \@survey_forms;
}



=head2 get_surveys

    my $surveys = get_surveys(%opt);

Получить список опросов

Опции:
    condition
    with_logins
    as_hash

=cut

sub get_surveys
{
    my (%opt) = @_;

    my $surveys = get_hashes_hash_sql(PPCDICT, [
            'SELECT',
            join(q{, }, 'survey_id', @COMMON_FIELDS, 'extra_json'),
            'FROM surveys',
            WHERE => $opt{condition} // 1,
        ]);

    for my $survey (values %$surveys) {
        hash_merge $survey, from_json(delete $survey->{extra_json} // '{}');
    }

    if (%$surveys && $opt{with_logins}) {
        my $survey_logins =
            overshard order => 'login',
            get_all_sql(PPC(shard=>'all'), [
                'SELECT survey_id, login
                FROM users_surveys
                JOIN users USING(uid)',
                WHERE => {
                    survey_id => [ keys %$surveys ],
                }, 
            ]);

        for my $survey_login (@$survey_logins) {
            my $survey_id = $survey_login->{survey_id};
            push @{$surveys->{$survey_id}->{users_list}}, $survey_login->{login};
        }
    }

    return $surveys  if $opt{as_hash};
    return [ values %$surveys ];
}


=head2 get_survey

    my $survey = get_survey($survey_id);

=cut

sub get_survey
{
    my ($survey_id, %opt) = @_;

    my $surveys = get_surveys(condition => {survey_id => $survey_id});
    return shift @$surveys;
}



=head2 delete_survey

    delete_survey($survey_id);

Удалить опрос

=cut

sub delete_survey
{
    my ($survey_id) = @_;

    do_delete_from_table(PPC(shard=>'all'), 'users_surveys', where => {survey_id => $survey_id});
    do_delete_from_table(PPCDICT, 'surveys', where => {survey_id => $survey_id});

    return;
}



=head2 update_survey

    update_survey(\%survey_data);

=cut

sub update_survey
{
    my ($survey_data) = @_;

    my $survey_id = $survey_data->{survey_id};
    SurveyError->throw(error => iget("Не задан id опроса"))  if !$survey_id;

    my $survey = get_survey($survey_id);
    SurveyError->throw(error => iget("Не найден опрос id=%s", $survey_id))  if !$survey;

    my $change = _adjust_fields(hash_cut $survey_data, @MUTABLE_FIELDS);
    for my $field (keys %$change) {
        delete $change->{$field} if $change->{$field} eq $survey->{$field};
    }

    return if !%$change;

    do_update_table(PPCDICT, 'surveys', $change, where => {survey_id => $survey_id});
    return;
}


=head2 create_survey

=cut

sub create_survey
{
    my ($survey_data) = @_;

    my @survey_form_fields = grep {!/^type$/} @COMMON_FIELDS;
    for my $field (@survey_form_fields) {
        SurveyError->throw(error => iget("Не заполнено поле %s", $field))  if !$survey_data->{$field}
    }

    my $ext_survey = get_ext_survey($survey_data);
    my $ext_data = $ext_survey->export();

    my $survey = hash_cut $survey_data, @COMMON_FIELDS;
    $survey->{extra_json} = to_json $ext_data;
    _adjust_fields($survey);

    my $survey_id = do_insert_into_table(PPCDICT, 'surveys', $survey);
    return $survey_id;
}



sub _adjust_fields
{
    my ($survey) = @_;

    # время приводим к суткам
    $survey->{start_date}  &&= unix2mysql([ts_get_day_borders(mysql2unix($survey->{start_date}))]->[0]);
    $survey->{finish_date} &&= unix2mysql([ts_get_day_borders(mysql2unix($survey->{finish_date}))]->[1]);

    return $survey;
}


=head2 add_survey_users

    add_survey_users($survey_id => \@uids);

=cut

sub add_survey_users
{
    my ($survey_id, $uids) = @_;

    my $survey = get_survey($survey_id);
    my $ext_survey = get_ext_survey($survey);

    foreach_shard uid => $uids, sub {
        my ($shard, $chunk) = @_;
        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT IGNORE INTO users_surveys (uid, survey_id, ext_user_id) VALUES %s',
            [ map {[$_, $survey_id, $ext_survey->get_user_id($_)]} @$chunk ],
        );
    };

    return;
}


=head2 complete_user_survey

    complete_user_survey($survey_id, %opt)

Options:
  * uid OR ext_user_id
  * status

=cut

sub complete_user_survey
{
    my ($survey_id, %opt) = @_;

    my $uid = $opt{uid};
    $uid ||= $opt{ext_user_id} && get_one_field_sql(PPC(shard => 'all'), [
            'SELECT uid FROM users_surveys',
            WHERE => {
                survey_id => $survey_id,
                ext_user_id => $opt{ext_user_id},
            },
        ]);
    croak 'User not defined'  if !$uid;

    my $values = hash_cut \%opt, qw/ status ext_respondent_id /;
    $values->{status} ||= 'Completed';
    
    do_update_table(PPC(uid=>$uid), 'users_surveys', $values,
        where => { survey_id => $survey_id, uid => $uid },
    );

    return;
}

1;
