#######################################################################
#
#  Direct.Yandex.ru
#
#  TTTools
#  Tools for use from templates (Template::Toolkit)
#
#  $Id$
#
#######################################################################

=head1 NAME

TTTools

=head1 DESCRIPTION

Tools for use from templates (Template::Toolkit)

=cut

package TTTools;

use warnings;
use strict;

use Settings;
use EnvTools;
use Yandex::I18n;
use Yandex::TimeCommon;
use Yandex::Shell qw/yash_qx/;
use Encode;
use URLDomain qw/clear_banner_href/;

use URI::Escape qw/uri_escape_utf8/;
use Data::Dumper;
use JSON;
use List::Util qw/max min/;
use Encode qw/encode_utf8 encode decode is_utf8/;
use POSIX qw/strftime ceil/;
use Currencies;
use BannerTemplates;
use TextTools qw/round2s/;

use locale;

use utf8;

our $VERSION = '0.01';

# global vars saved in DoCmd
my ($script);

my $json = JSON->new->allow_nonref;

=head1 FUNCTIONS

=cut

#======================================================================

=head2 _save_vars

save global vars

  TTTools::_save_vars($script);

=cut

sub _save_vars
{
    ($script) = @_;
}

#======================================================================

=head2 dumper

print dump of variable to STDERR from templates

use: [% dumper(var) %]

=cut

sub dumper
{
    print STDERR Dumper(@_);
    return;
}

#======================================================================

=head2 sort_table_data

  sort data in table

  use:
    [% camps = sort_table_data(ref_of_array_of_hashes, FORM, "default_column_name", [array_of_columns_whith_string_sorting], {options}) %]
    {options} as {opt1 => "value"} if needed

  options:
    phrasestat => 1     # for overwrite default sort process in campaignphrasedetal.html
    url_param  => '_my' # for overwrite default sort, reverse url param; now params is "sort_my", "reverse_my"
    sort_as_currency => 'field_name' # sort field with currency codes by corresponding currency texts

  example:
    [% camps = sort_table_data(camps, FORM, "cid", ["name", "fio"], {opt1 => "value"}) %]
    [% camps = sort_table_data(camps, FORM, "cid", ["name", "status.text"]) %] # $camp->{status}->{text} будут сортироваться как строки

=cut

sub sort_table_data
{
    my ($data, $FORM, $def_col, $cols_with_str_sort, $options, $first_sort) = @_;
    
    return $data if ref($data) ne 'ARRAY';
    
    # если хотя бы одно поле сортировки текстовое, либо полей для сортировки несколько - сортировка текстовая
    my ($a1, $b1, $a2, $b2); 
    
    my $string_sort = {};
    foreach my $cn (@$cols_with_str_sort) {
        $string_sort->{$cn} = 1;
    }
    
    my $url_param_sort = defined $options->{url_param} ? "sort$options->{url_param}" : "sort";
    my $url_param_reverse = defined $options->{url_param} ? "reverse$options->{url_param}" : "reverse";
    my $fsort_direct;

    my ($col, $coll, @fsort) = ([], {});
    if (defined $FORM->{$url_param_sort}) {
        foreach my $colsort (split(/\,\s*/,$FORM->{$url_param_sort})) {
            $coll->{$colsort} = 1;
            push @$col, $colsort;
        }
    } else {
        if (ref $def_col eq 'ARRAY') {
            foreach my $cn (@$def_col) {
                $coll->{$cn} = 1;
                push @$col, $cn;
            }
        } else {
            $coll->{$def_col} = 1;
            push @$col, $def_col;
        }
    }
    
    if (defined $first_sort) {
        foreach my $c (@$first_sort) {
            if ($c =~ s/^\!//) {
                $fsort_direct->{$c} = -1;
            }
            push @fsort, $c;
        }
    }
    
    return $data if !scalar @$col;
    
    ## здесь у Freenode::DollarAB ложное срабатывание
    ## no critic (Freenode::DollarAB)
    return [sort {
        
        # by manager or agency uid (in list campaigns only)
        if ($coll->{mauid}) {
           $a2 = $a->{ManagerUID} || $a->{AgencyUID} || 0;
           $b2 = $b->{ManagerUID} || $b->{AgencyUID} || 0;
           return $a2 <=> $b2;
        }
        
        my $cmp_a2b2 = _cmp_some_columns($a, $b, $col, $string_sort, $FORM->{$url_param_reverse}, $options);
        
        # for campaignphrasedetal.html only
        # first sort by phrase
        # all other columns with numerical sort
        # special phrases last
        if ($options->{phrasestat}) {
            my $special_flag_cmp = ($a->{special_flag} ? 1 : 0) <=> ($b->{special_flag} ? 1 : 0);
            my $a3 = defined $a->{phrase} ? lc($a->{phrase}) : '';
            my $b3 = defined $b->{phrase} ? lc($b->{phrase}) : '';
            return $special_flag_cmp || $a3 cmp $b3 || $cmp_a2b2;
        }

        # сортируем обязательные поля
        my $ab_cmp = _cmp_array_columns($a, $b, \@fsort, $fsort_direct);        
        
        return $ab_cmp || $cmp_a2b2

      } @$data
    ]; # / sort { ...
}

sub _cmp_array_columns
{
    ## эта функция вызывается только из sort, так что здесь можно $a и $b
    ## no critic (Freenode::DollarAB)
    my ($a, $b, $columns_sort, $direction_sort) = @_;
    
    if ($columns_sort && @$columns_sort) {
        my ($cname, @arr) = @$columns_sort;
        my ($i, $j) = ($a->{$cname}||'', $b->{$cname}||'');
        ($i, $j) = ($j, $i) if $direction_sort->{$cname};
        
        my $l_res = $i cmp $j;
        my $l_res2 = _cmp_array_columns($a, $b, \@arr, $direction_sort);
        
        return $l_res || $l_res2;
    }
    
    return 0;
}

sub _cmp_some_columns
{
    ## эта функция вызывается только из sort, так что здесь можно $a и $b
    ## no critic (Freenode::DollarAB)
    my ($a, $b, $cols, $string_sort, $reverse_sort, $options) = @_;
    
    foreach my $colname (@$cols) {
        # ходим по вложенным хешам
        my $accessor = sub {
            my ($structure, $default) = @_;

            my @parts = split /\./, $colname;
            for my $part (@parts) {
                return $default unless exists $structure->{$part};
                $structure = $structure->{$part};
            }
            return $structure;
        };

        if ($string_sort->{$colname}) {
            my $aa = $accessor->($a, '');
            my $bb = $accessor->($b, '');
            if ($options->{sort_as_currency} && $options->{sort_as_currency} eq $colname) {
                $aa = get_currency_constant($aa, 'name') if $aa;
                $bb = get_currency_constant($bb, 'name') if $bb;
            }
            if (! $options->{case_sensitive}) {
                ($aa, $bb) = (lc($aa), lc($bb));
            }
            
            ($aa, $bb) = ($bb, $aa) if $reverse_sort;            

            if (my $res = $aa cmp $bb) {
                return $res;
            }
        } else {
            my $aa = $accessor->($a, 0);
            $aa = 0 unless $aa && $aa =~ /^\-?\d+(\.\d+)?$/;
            my $bb = $accessor->($b, 0);
            $bb = 0 unless $bb && $bb =~ /^\-?\d+(\.\d+)?$/;
            
            ($aa, $bb) = ($bb, $aa) if $reverse_sort;

            if (my $res = $aa <=> $bb) {
                return $res;
            }
        }
    }
    
    return 0;
}

#======================================================================

=head2 get_sort_table_header

  generate column header for table with sorting data

  use:
      get_sort_table_header("column_name", "column caption", "default_column_name", FORM, {option => value, ...})

  options:
      anchorset: set on this column header <a name="...">
      anchor: name of anchor for go to <a name="anchor_name">
      url_param: '_postfix' for overwrite default sort, reverse url param; now params is "sort_postfix", "reverse_postfix"
      default_reverse: set default reverse sort
      dont_remove_page_from_url: remove 'page' param from url(1: no, 0 or undef: yes)

  example:
      <td>[% get_sort_table_header("login",    "Логин",    "login", FORM, {anchorset => "grp1", anchor => "grp1"}) %]</td>
      <td>[% get_sort_table_header("clientid", "ClientID", "login", FORM, {anchor => "grp1"}) %]</td>
      <td>[% get_sort_table_header("fio",      "ФИО",      "login", FORM, {anchor => "grp1"}) %]</td>
      <td>[% get_sort_table_header("phone",    "Телефон",  "login", FORM, {anchor => "grp1"}) %]</td>
      <td>[% get_sort_table_header("email",    "E-mail",   "login", FORM, {anchor => "grp1"}) %]</td>

=cut

sub get_sort_table_header
{
    my ($col, $caption, $def_col, $FORM, $options) = @_;
    
    my $cols = ref $col eq 'ARRAY' ? join(",", @$col) : $col;

    my $url_param_sort = defined $options->{url_param} ? "sort$options->{url_param}" : "sort";
    my $url_param_reverse = defined $options->{url_param} ? "reverse$options->{url_param}" : "reverse";
    my $remove_page_param = defined $options->{dont_remove_page_from_url} ? "" : "|page";

    my $sorturl = '';
    for my $key (keys %$FORM) {
        next if $key =~ /^($url_param_sort|$url_param_reverse|ncrnd|UID|ouid_url$remove_page_param)$/;
        $sorturl .= uri_escape_utf8($key)."=" . uri_escape_utf8($FORM->{$key}) . '&';
    }

    my $sort = $FORM->{$url_param_sort} || (ref $def_col ? join(",",@$def_col) : $def_col);
    my $lego_classes = $options->{lego_classes};
    my $classes = $options->{classes} // '';
    my $default_reverse = $options->{default_reverse} ? 1 : 0;
    my $draw_reverse = defined $FORM->{$url_param_reverse} ? $FORM->{$url_param_reverse} : $default_reverse;
    my $new_reverse = $cols ne $sort ? $default_reverse : ($draw_reverse ? 0 : 1);
    my $arrow_text = '';
    my $arrow_char = $draw_reverse ? '&darr;' : '&uarr;';
    $arrow_text = '<span class="b-table__sort__arrow">' . $arrow_char . '</span>' if ($cols eq $sort) && $lego_classes;
    $arrow_text = '&nbsp;<small style="color: #707070;">' . $arrow_char . '</small>' if ($cols eq $sort) && !$lego_classes;

    my $anchor = defined $options->{anchor} ? "#$options->{anchor}" : '';
    my $anchorset = (defined $options->{anchorset} && defined $options->{anchor} && $options->{anchorset}) ?
                    qq| name="$options->{anchor}"| : '';

    if ($lego_classes) {
        return qq|<a class="b-table__sort |.($classes // '').qq|" $anchorset href="$script?${sorturl}$url_param_sort=$cols&$url_param_reverse=$new_reverse$anchor"><span class="b-table__sort__word">$caption</span>$arrow_text</a>|;
    } else {
        return qq|<a$anchorset class="|.($classes // '').qq|" href="$script?${sorturl}$url_param_sort=$cols&$url_param_reverse=$new_reverse$anchor">$caption</a>$arrow_text|;
    }
}

#======================================================================

=head2 get_sort_table_all_headers

  generate all headers for table with sorting data
  DONT use with get_sort_table_header() on one page !!!

  use:
      get_sort_table_all_headers(array_of_arrays_of_names_and_captions_and_optional_template,
                                 "default_column_name", FORM, "template like for printf (%s), {option => value, ...}")

  options:
      anchor: name of anchor for go to <a name="anchor_name">
      not_anchorset: true - dont set <a name="anchor_name"> in first column header
      not_check_valid_col: true - for not check on valid sort column and set default sort column
      url_param: '_my' - for overwrite default sort, reverse url param; now params is "sort_my", "reverse_my"
      default_reverse: set default reverse sort

  example:
      [% get_sort_table_all_headers([["login",    "Логин"],
                                     ["clientid", "ClientID"],
                                     ["fio",      "ФИО"],
                                     ["total",    "осталось", '<td align="right">%s</td>'], # override default template
                                     ["phone",    "Телефон"],
                                     ["email",    "E-mail"]
                                    ], "login", FORM, "<td>%s</td>", {anchor => "grp1"}) %]

=cut

sub get_sort_table_all_headers
{
    my ($columns, $def_col, $FORM, $template, $options) = @_;
    my $result = '';

    my $url_param_sort = defined $options->{url_param} ? "sort$options->{url_param}" : "sort";
    my $url_param_reverse = defined $options->{url_param} ? "reverse$options->{url_param}" : "reverse";

    # check on valid sort column
    if (! $options->{not_check_valid_col} && (! defined $FORM->{$url_param_sort} || ! grep {$FORM->{$url_param_sort} eq $_->[0]} @$columns)) {
        # rewrite global FORM variable
        # this new FORM uses in get_sort_table_header() with new 'sort' and 'reverse' keys
        $FORM->{$url_param_sort} = $def_col;
        $FORM->{$url_param_reverse} = 0;
    }

    # generate by one
    my $i = 0;
    for my $row (@$columns) {
        delete $options->{anchorset}; # reset
        $options->{anchorset} = 1 if ! $options->{not_anchorset} && defined $options->{anchor} && ++$i == 1; # in first link only
        my $current_template = defined $row->[2] ? $row->[2] : $template;
        $result .= sprintf($current_template, get_sort_table_header($row->[0], $row->[1], $def_col, $FORM, $options));
    }

    return $result;
}

#======================================================================

=head2 _add_number_group_separators

    Добавляет разделитель разрядов между группами из 3-х символов
    Разделителем по умолчанию является неразрывный пробел

    _add_number_group_separators(1001.5637) # => 1<неразрывныйпробел>001.5673
    _add_number_group_separators(1001.5637, separator => '_') # => 1_001.5673

=cut

sub _add_number_group_separators {
    my ($number, %O) = @_;

    my $separator = (defined $O{separator}) ? $O{separator} : "\x{00A0}";
    $number =~ s/(^\d+?|\G\d{3})(?=(\d{3})+(\D|$))/$1$separator/g;
    return $number;
}

=head2 format_price

format number (add group separators, round2s)

use: [% format_price(1001.5637) %] => 1<неразрывныйпробел>001.56

    $price_formatted = format_price($price, $opts);

    $opts => {
        separator => '_', # разделитель, добавляемый между группами разрядов
    }

=cut

sub format_price {
    my ($price, $opts) = @_;
    return undef unless defined $price && length $price;
    $opts ||= {};

    $price = round2s($price) == 0 && $price >= $Currencies::EPSILON ? 0.01 : sprintf("%.2f", $price);

    # добавляем разделители групп разрядов
    if (!defined $opts->{separator} || $opts->{separator} ne '') {
        $price = _add_number_group_separators($price, separator => $opts->{separator});
    }

    return $price;
}

=head2 format_units_shows

    форматировать количество показов, зачисленных на медийную кампанию

    format_units_shows(1001.5637); # => 1<неразрывныйпробел>001.56

=cut

sub format_units_shows {
    my $units = shift;
    # форматируем нулями
    my $res = sprintf("%0.3f", ($units||0)/1000);
    return '0' if $res eq '0.000';
    # добавляем неразрывный пробелы - разделители разрядов
    $res = _add_number_group_separators($res);
    return $res;
}

# форматировать юниты (зачисленные средства на кампании)
sub format_units {
    my ($units, $name) = @_;
    return $units if ! defined $units || $units eq '-';
    if (!$name || $name eq 'money') {
        return format_price($units);
    } elsif ($name eq 'shows') {
        return format_units_shows($units);
    } else {
        return $units;
    }
}

=head2 format_int

    форматировать целое число

    format_int(123456.789); # 123<неразрывный пробел>456 
    format_int(123456.789, {separator => '_'}); # 123_456
=cut

sub format_int {
    my ($val, $opts) = @_;
    return undef unless defined $val;

    $val = sprintf("%d", $val);
    # Добавляем разделитель
    my %add_number_group_options;
    if ($opts && defined $opts->{separator}) {
        $add_number_group_options{separator} = $opts->{separator};
    }
    $val = _add_number_group_separators($val, %add_number_group_options);
    return $val;
}

# форматировать размер файла
sub format_file_size {
    my $size = shift;
    return ceil($size/1024)." КБ";
}

# добавить если нужно http:// и openstat
sub format_href {
    my ($href, $openstat) = @_;
    if (!defined $href || !$href || $href =~ /^#/) {
        return $href;
    }
    $href = clear_banner_href($href);

    $href =~ s/$TEMPLATE_METKA/$1/ig;
    if ($openstat && $openstat eq 'Yes') {
        # clear anchor
        my $pure_href = $href =~ s/\#.*$//r;
        my $sep = $pure_href =~ /\?/ ? '&' : '?';
        $href =~ s/($|\#)/${sep}_openstat=${Settings::OPENSTAT_LABEL}$1/;
    }
    return $href;
}

{
my @fd_mon = iget_noop('янв', 'фев', 'мар', 'апр', 'май', 'июн', 'июл', 'авг', 'сен', 'окт', 'ноя', 'дек');
my %months = (
    i => [iget_noop('январь', 'февраль', 'март', 'апрель', 'май', 'июнь', 'июль', 'август', 'сентябрь', 'октябрь', 'ноябрь', 'декабрь')],
    r => [iget_noop('января', 'февраля', 'марта', 'апреля', 'мая', 'июня', 'июля', 'августа', 'сентября', 'октября', 'ноября', 'декабря')],
    d => [iget_noop('январю', 'февралю', 'марту', 'апрелю', 'маю', 'июню', 'июлю', 'августу', 'сентябрю', 'октябрю', 'ноябрю', 'декабрю')],
    v => [iget_noop('январь', 'февраль', 'март', 'апрель', 'май', 'июнь', 'июль', 'август', 'сентябрь', 'октябрь', 'ноябрь', 'декабрь')],
    t => [iget_noop('январем', 'февралем', 'мартом', 'апрелем', 'маем', 'июнем', 'июлем', 'августом', 'сентябрем', 'октябрем', 'ноябрем', 'декабрем')],
    p => [iget_noop('январе', 'феврале', 'марте', 'апреле', 'мае', 'июне', 'июле', 'августе', 'сентябре', 'октябре', 'ноябре', 'декабре')],
);
my %wdays = (
    dow    => [iget_noop('понедельник', 'вторник', 'среда', 'четверг', 'пятница', 'суббота', 'воскресенье')],
    dow_short => [iget_noop('пн', 'вт', 'ср', 'чт', 'пт', 'сб', 'вс')],
    dow_at => [iget_noop('в понедельник', 'во вторник', 'в среду', 'в четверг', 'в пятницу', 'в субботу', 'в воскресенье')],
);

my %date_part = (
    mon => sub {my $aref=shift;$fd_mon[$aref->[4]]},
    Mon => sub {my $aref=shift;ucfirst($fd_mon[$aref->[4]])},
    MON => sub {my $aref=shift;uc($fd_mon[$aref->[4]])},
    
);
foreach my $case (keys %months) {
    $date_part{"month_$case"} = sub {my $aref=shift;return $months{$case}->[$aref->[4]]};
    $date_part{"Month_$case"} = sub {my $aref=shift;return ucfirst($months{$case}->[$aref->[4]])};
    $date_part{"MONTH_$case"} = sub {my $aref=shift;return uc($months{$case}->[$aref->[4]])};    
}
foreach my $part (keys %wdays) {
    $date_part{$part} = sub { my $aref=shift; return iget($wdays{$part}->[ ($aref->[6]||7) - 1]) };
}

# Форматировать дату
sub format_date {
    my ($date, $type, @params) = @_;
    return '' if !$date;
    $type ||= 'date';
    my @lt = localtime(mysql2unix($date));
    if ($type eq 'date') {
        # 31.12.1969
        # стандартная дата
        return strftime("%d.%m.%Y", @lt);
    } elsif ($type eq 'datetime') {
        # 31.12.1969 23:59
        # стандартная дата и время до минут
        return strftime("%d.%m.%Y %H:%M", @lt);
    } elsif ($type eq 'datetimesec') {
        # 31.12.1969 23:59:59
        # стандартная дата и время до секунд
        return strftime("%d.%m.%Y %H:%M:%S", @lt);
    } elsif ($type eq 'time') {
        # 23:59
        # только время до минут
        return strftime("%H:%M", @lt);
    } elsif ($type eq 'timesec') {
        # 23:59:59
        # только время до секунд
        return strftime("%H:%M:%S", @lt);
    } elsif ($type eq 'strftime') {
        # вызов strftime, принимает параметр
        
        my $res = Encode::decode_utf8(strftime(Encode::encode_utf8($params[0]), @lt));
        $res =~ s/\{(\w+)\}/$date_part{$1} ? $date_part{$1}->(\@lt) : ""/eg;
        return $res;
    } elsif ($type eq 'month') {
        # дек 1969 г.
        # стандартный месяц
        return sprintf "%s %d", iget($fd_mon[$lt[4]]), $lt[5]+1900;
    } elsif ($type eq 'year') {
        # 1969 г.
        # стандартный год
        return sprintf iget("%d г."), $lt[5]+1900;
    } else {
        warn "incorrect format '$type'\n";
        return strftime("%d.%m.%Y", @lt);
    }
}
}


#======================================================================

=head2 $SKIP_URL_PARAMS
    
    Параметры, исключаемые из url при формировании навигации

=cut

our $SKIP_URL_PARAMS = [qw/uid_url ncrnd UID/];


=head2 get_current_url_params

  get current url params
  like: "cmd=copyCamp&ulogin=login&cid=23473"

=cut

sub get_current_url_params
{
    my $FORM = shift;
    my $to_skip = shift || [];

    my $res = '';
    my %skip = map {$_ => 1} @$SKIP_URL_PARAMS, @$to_skip;
    if (ref $FORM eq 'HASH' && %$FORM) {
        for my $key (keys %$FORM) {
            next if exists $skip{$key};
            $res .= "&".uri_escape_utf8($key)."=" . uri_escape_utf8($FORM->{$key});
        }
    }
    $res =~ s/^&//;

    return $res;
}

#======================================================================

=head2 get_plot_link

  generate plot link

=cut

sub get_plot_link
{
    my ($FORM, $options, $return_vars) = @_;

    my %vars;

    my $params = {};
    for my $key (keys %$FORM) {
        $params->{$key} = $FORM->{$key} if defined $FORM->{$key};
    }
    $params->{cmd} = undef;
    for my $key (keys %$options) {
        if (defined $options->{$key}) {
            $params->{$key} = $options->{$key};
        } elsif (exists $params->{$key}) {
            delete $params->{$key};
        }
    }

    for my $key (keys %$params) {
        next if $key =~ /^(ncrnd|UID|uid_url)$/;

        if ($key =~ /^sort/) {
            next if defined $params->{plotcol} && $params->{plotcol} =~ m/^(clicks|shows|ctr|sum|av_sum|all)$/;
            if ($params->{$key} =~ m/(clicks|shows|ctr|sum|av_sum|all)(|_0|_1)/) {
                push @{$vars{plotcol}}, $1;
            } else {
                push @{$vars{plotcol}}, 'shows';
            }
        } elsif ($key =~ m/^([dmy][12]|ulogin|cid|target_\w*|group|plot_data|norefresh)$/) {
            push @{$vars{$key}}, $params->{$key};
        } elsif ($key =~ /^group_by/) {
            if (ref $params->{$key} eq 'ARRAY') {
                push @{$vars{group_by}}, $_ for @{$params->{$key}};
            } elsif (ref $params->{$key} eq '') {
                my @group_by = grep {/^(banner|page|geo|phrase|phraseid|date|position)$/} split /\s*,\s*/, $params->{$key};
                push @{$vars{group_by}}, $_ for @group_by;
            }
        } elsif ($key =~ /^filter_/) {
            if (ref $params->{$key} eq 'ARRAY') {
                push @{$vars{$key}}, $_ for @{$params->{$key}};
            } elsif (ref $params->{$key} eq '') {
                push @{$vars{$key}}, $params->{$key};
            }
        } elsif ($key eq 'stat_type' && $params->{$key} eq 'campdate') {
            push @{$vars{stat_type}}, 'campdate';
        }
    }

    push @{$vars{plotcol}}, $1 if defined $params->{plotcol} && $params->{plotcol} =~ m/^(clicks|shows|ctr|sum|av_sum|all)(|_0|_1)$/;
    push @{$vars{cmd}}, (defined $params->{cmd} && $params->{cmd} eq 'showCampStat' ? 'showCampStat' : 'showPlotPage');
    push @{$vars{plot}}, 1;
    push @{$vars{onpage}}, 10000;
    push @{$vars{show_favorites}}, 1 if $params->{show_favorites};
    push @{$vars{norefresh}}, 1 unless defined $params->{norefresh};

    my $url = join '&', (map { my $key = $_; (map { uri_escape_utf8($key).'='.uri_escape_utf8($_) } @{$vars{$key}}) } keys %vars);

    return $return_vars ? \%vars : qq|$script?$url|;
}

sub trim_float {
    ## no critic (Freenode::DollarAB)
    my ($a, $max_len, $tail_len) = @_;
    my $len = length(sprintf("%.0f", $a));
    my $tail = min(max($max_len - $len, 0), $tail_len);
    return sprintf "%.${tail}f", $a;
}

#========================================================================


=head2 typograph

    Простейшее форматирование текста в html - замена пробелов на &nbsp; после предлогов.

=cut
sub typograph {
    my $text = shift;
    # пока считаем, что 1-2 символа - предлог
    $text =~ s/(^|\s)(\w{1,2})\s+/$1$2&nbsp;/g;
    # замена пробелов перед временным смещением
    $text =~ s/\s+([\+\-]\d{2}:\d{2})/&nbsp;$1/g;
    # заменяем многоточия
    $text =~ s/\.{3,}/&hellip;/g;
    # добавляем пробелы к перечислениям
    $text =~ s/(\d+)\.([^\d\s])/$1. $2/g;
    # заменяем переводы строк
    $text =~ s/\n/<br \/>/g;
    return $text;
}

#======================================================================

=head2 get_url

  build direct link in templates:
  [% get_url(cmd, params_hash) %]

  <a href="[% get_url('showCamp', {cid => 1123245, tab => 'active'}) %]">show my camp</a>
  <a href="[% get_url('delBanner', {cid => 1123245, bid => 43242333}) %]">delete banner</a>
  <a href="[% get_url('showManagerMyClients') %]">my clients</a>

=cut

sub get_url
{
    my ($script, $csrf_token, $cmd, $params) = @_;

    my $direct_url = $script . '?cmd=' . uri_escape_utf8($cmd);

    # теперь есть Yandex::HTTP::make_url
    if ($params) {
        $direct_url .= '&' . join '&', map {uri_escape_utf8($_) . '=' . uri_escape_utf8($params->{$_})} keys %$params;
    }

    $direct_url .= '&csrf_token=' . uri_escape_utf8($csrf_token) if $csrf_token;

    return $direct_url;
}

sub json_dump {

    my $json_dump = $json->encode(@_);
    $json_dump =~ s[/][\\/]g;
    return $json_dump;
}


#======================================================================

=head2 console_log

  Dumps arbitary perl data to Firebug console. 

  [% console_dump(campaigns) %]
  
=cut

sub console_log
{
    my $json_dump = json_dump(@_);
    return '<script type="text/javascript">(window.opera ? window.opera.postError : console.log)(' . $json_dump . ')</script>';
}

=head2 server_warn

  Dumps arbitary perl data to webserver error log

  [% server_warn(campaigns) %]
  
=cut

sub server_warn
{
    my $json = JSON->new->pretty(1);
    print STDERR join "\n",
        strftime('%Y-%m-%d %H:%M:%S', localtime).' warn',
        ('=' x 70),
        (map {s/\n$//; $_} map {ref $_ ? $json->encode($_) : $_} @_),
        ('=' x 70),
        '';
    return '';
}

#======================================================================

=head2 generate_id

    Случайно созданный id для элемента

=cut
sub generate_id {
    return sprintf("id%0.8x",rand()*0xffffffff);
}

#======================================================================

=head2 is_string

    Проверяет явлеяется ли переданный аргумент строчкой

=cut
sub is_string {
    my $arg = shift;
    return !ref($arg) || ref($arg) eq 'SCALAR';
}

#======================================================================

=head2 get_svn_info

    Бранч и ревизия текущей рабочей копии,
    на продакшне и тестинге отдаёт undef

=cut

sub get_svn_info
{
    if (is_beta()){
        my $svn_info = eval {yash_qx("svn info $Settings::ROOT 2>/dev/null")} || '';
        return {} unless $svn_info;
        my ($branch) = $svn_info =~ m{URL: .*arcadia\.yandex\.ru/arc/(?:(trunk|branches/direct/perl/.*))};
        my ($revision) = $svn_info =~ m{Revision: (\d+)};
    
        return {
            branch => $branch,
            revision => $revision,
        };
    }
    else{
        return undef;
    }
}

=head2 human_device_type_targeting($device_type_targeting)

Возвращает человекочитаемое название таргетинга для вида устройства
    Параметры:
        $device_type_targeting - таргетинг в виде строки(разделитель - ",")
                                ("tablet,phone")

=cut

{
my %device_types = (phone => iget_noop('Смартфоны'), tablet => iget_noop('Планшеты'));
sub human_device_type_targeting {
    my $device_type_targeting = shift;
    my @types = (ref $device_type_targeting eq 'ARRAY') ? @$device_type_targeting : split /\s*,\s*/, $device_type_targeting;
    return iget('Все') if 2 == @types;
    return join '/', map { iget($device_types{$_}) } @types; 
}
}

=head2 human_network_targeting($network_targeting)

Возвращает человекочитаемое название таргетинга для типа сети
    Параметры:
        $network_targeting - таргетинг в виде строки(разделитель - ",")
                                ("wifi,cell")

=cut

sub human_network_targeting {
    my $network_targeting = shift;
    my @networks = (ref $network_targeting eq 'ARRAY') ? @$network_targeting : split /\s*,\s*/, $network_targeting;
    return @networks == 1 ? iget('Только Wi-Fi') : iget('Мобильная связь и Wi-Fi');
}

1;

