package Tag;

=head1 NAME

    Tag

=head1 DESCRIPTION

    Работа с метками.

=cut

use strict;
use warnings;

use List::MoreUtils qw/uniq/;

use Yandex::I18n;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::HashUtils qw/hash_cut hash_grep/;
use Yandex::ListUtils qw/xminus xuniq/;

use Settings;
use PrimitivesIds;
use ShardingTools;


use base qw/Exporter/;
our @EXPORT = qw/
    get_groups_tags
    get_tags_groups
    save_tags_groups
    get_all_campaign_tags
    mass_get_all_campaign_tags
    get_all_campaign_tags_count
    get_untagged_banners_num
    get_pids_by_tags
    get_pids_by_tags_text
    get_tag_ids_by_tags_text
    save_campaign_tags
    add_campaign_tags
    validate_adgroup_tags
    validate_tags 
    check_tag_text                 
    check_camp_count_limit
    check_banner_count_limit
    get_tags_by_tag_ids
    delete_campaign_tags
    delete_group_tags

    $MAX_TAG_LENGTH
    $MAX_TAGS_FOR_CAMPAIGN
    $MAX_TAGS_FOR_BANNER
    /;

use utf8; 

# максимальная длина описания на клиенте
our $MAX_TAG_LENGTH = 25;
our $MAX_TAGS_FOR_CAMPAIGN = 200;
our $MAX_TAGS_FOR_BANNER = 30;

=head2 get_all_campaign_tags (cid)

    Возвращает список всех меток для кампании.
    Параметры:
        cid - номер кампании

=cut 
sub get_all_campaign_tags {
    my ($cid) = @_;
    return {} unless $cid;
    return mass_get_all_campaign_tags([$cid])->{$cid};
}

=head2 mass_get_all_campaign_tags (cid)

    Возвращает хэш списков всех меток для кампаний
    Параметры:
        cids - ссылка на массив с номерами кампаний

=cut 

sub mass_get_all_campaign_tags {
    my ($cids) = @_;
    return {} unless $cids && @$cids;

    my $data = overshard order => 'value', get_all_sql(PPC(cid => $cids), ["SELECT
            tc.cid, tc.tag_id, tc.tag_name as value, count(tg.pid) as uses_count
        FROM
            tag_campaign_list tc
            LEFT JOIN tag_group tg USING (tag_id)",
        where => { 'tc.cid' => SHARD_IDS },  
        "GROUP BY tag_id"
    ]) || [];

    my $result = {};
    $result->{$_} = [] foreach @$cids;

    foreach my $tag (@$data) {
        push @{$result->{$tag->{cid}}}, hash_cut $tag, qw/tag_id value uses_count/;
    }

    return $result;
}

=head2 get_all_campaign_tags_count 

    Возвращает количество уже существующих меток на кампанию.
    Параметры:
        cid - номер кампании
=cut
sub get_all_campaign_tags_count {
    my ($cid) = @_;
    return get_one_field_sql(PPC(cid => $cid), "SELECT count(*) FROM tag_campaign_list WHERE cid=?", $cid) || 0;
}


=head2 get_untagged_groups_num

    Возвращает количество групп, которые не используют метки.
    Параметры:
        cid - номер кампании

=cut
sub get_untagged_groups_num {
    my $cid = shift;

    return mass_get_untagged_groups_num([$cid])->{$cid} || 0;
}


=head2 mass_get_untagged_groups_num

    Возвращает количество групп, которые не используют метки.
    Параметры:
        cids -  список id кампаний

=cut
sub mass_get_untagged_groups_num {
    my $cids = shift;

    return {}  if !@$cids;

    return get_hash_sql(PPC(cid => $cids), [
        'SELECT cid, COUNT(*)
        FROM phrases g
        LEFT JOIN tag_group tg USING(pid)',
        WHERE => {
            'g.cid' => $cids,
            'tg.pid__is_null' => 1
        },
        'GROUP BY cid'
    ]);
}



=head2 get_pids_by_tags (tags_ids)

    Выбирает все id групп(pid) с нужными метками.
    Параметры:
        tags_ids - список меток для фильтрации.

=cut

sub get_pids_by_tags {
    my $tags_ids = shift;
    return [] unless $tags_ids;
    return get_one_column_sql(PPC(tag_id => $tags_ids), ["SELECT pid FROM tag_group", where => { tag_id => SHARD_IDS }]) || [];
}

=head2 get_pids_by_tags_text (tags, cids, bids)

    Выбирает все id групп(pid) с нужными метками среди переданных кампаний или баннеров
    Параметры:
        tags - список меток для фильтрации.
        bids - список баннеров
        cids - список кампаний

=cut

sub get_pids_by_tags_text {
    
    my ($tags, $cids, $bids) = @_;
    if (!$cids && $bids) {
        $cids = get_cids(bid => $bids);
    }
    if (scalar @$cids) {
        my $tags_ids = get_one_column_sql(PPC(cid => $cids), [
            "select tag_id from tag_campaign_list",
            where => { cid => SHARD_IDS, tag_name => $tags }
        ]) || [];
        return get_pids_by_tags($tags_ids);
    }
    return [];
}

=head2 get_tag_ids_by_tags_text (tags, bids, cids)

    Выбирает все баннеры(bid) с нужными метками среди переданных кампаний или баннеров
    Параметры:
        tags - список меток для фильтрации.
        bids - список баннеров
        cids - список кампаний

=cut 
sub get_tag_ids_by_tags_text {
    my ($tags, $cids, $bids) = @_;
    if (!$cids && $bids) {
        $cids = get_cids(bid => $bids);
    }
    if (scalar @$cids) {
        return get_one_column_sql(PPC(cid => $cids), ["select tag_id from tag_campaign_list", where => {cid => SHARD_IDS, tag_name => $tags}]) || [];
    }
    return [];
}

=head2 save_campaign_tags (cid, tags, OPT)

    Сохраняет список меток на кампанию.
    Параметры:
        cid - номер кампании
        tags – список меток, которые должны быть установлены для кампании. Каждая метка имеет вид: {name=>"Имя метки", tag_id=>123}
            Если метка новая, то ее tag_id=0
        OPT:
            return_inserted_tag_ids - если надо вернуть идентификаторы свеже добавленных меток, то флаг должен быть установлен.
    Перед сохранением необходимо воспользоваться валидирующей функцией validate_tags(cid, tags)

=head3 INTERNAL

    Переименование меток реализовано черед удаление + вставку

=cut 
sub save_campaign_tags {
    my ($cid, $tags, %OPT) = @_;

    # уже существующие на кампании метки в хеше id => name
    my $camp_tags_id_name = get_hash_sql(PPC(cid => $cid), 
        'SELECT tag_id, tag_name FROM tag_campaign_list WHERE cid = ?', $cid 
    );
    my @old_tags_id = keys %$camp_tags_id_name;

    my @skip;           # метки не изменились, ничего с ними делать не будем
    my @changed_tags;   # изменившиеся метки (удалим и добавим заново)
    my %for_insert;     # новые метки (добавим)

    for my $tag (@$tags) {
        $tag->{tag_id} ||= 0;
        next unless $tag->{tag_id} =~ /^\d+$/;
        my $name = trim($tag->{name});

        if ($tag->{tag_id}) {
            # ничего не делаем с метками не из этой кампании (должна была сработать валидация)
            next unless exists $camp_tags_id_name->{ $tag->{tag_id} };
            if ($camp_tags_id_name->{ $tag->{tag_id} } eq $name) {
                # не изменившиеся метки запоминаем, чтобы не удалить и пропускаем
                push @skip, $tag->{tag_id};
                next;
            }

            # иначе - старую метку нужно удалить и создать новую с тем же id.
            push @changed_tags, [ $tag->{tag_id}, $cid, $name ];
        } else {
            # дополнительно прогоняем через хеш, чтобы не упасть из-за дубликатов
            $for_insert{ $tag->{name} } = undef;
        } 
    }

    my $inserted_tag_ids;
    do_in_transaction {
        # Удаляем все метки, кроме тех, что не изменились
        my $for_delete_only_tags = xminus(\@old_tags_id, \@skip);
        do_delete_from_table(PPC(cid => $cid), 'tag_campaign_list', where => { tag_id => $for_delete_only_tags });

        # Удаляем все привязки меток к группам, кроме тех меток, что были указаны
        my $for_delete_tags_groups = xminus(\@old_tags_id, [@skip, map { $_->[0] } @changed_tags ]);
        do_delete_from_table(PPC(cid => $cid), 'tag_group',  where => {tag_id => $for_delete_tags_groups });
        delete_shard(tag_id => $for_delete_tags_groups);

        if (@changed_tags) {
            do_mass_insert_sql(PPC(cid => $cid),
                'INSERT INTO tag_campaign_list (tag_id, cid, tag_name) values %s',
                \@changed_tags
            );
        }

        # Добавляем новые метки.
        $inserted_tag_ids = add_campaign_tags(
            $cid,
            [keys %for_insert],
            return_inserted_tag_ids => $OPT{return_inserted_tag_ids}
        );
    };
    return $inserted_tag_ids;
}

=head2 add_campaign_tags (cid, tags_names, OPT)

    Добавляет в список меток на кампанию новые метки.
    Параметры:
        cid - номер кампании
        tags – список меток, которые должны быть установлены для кампании.
        OPT:
            return_inserted_tag_ids - если надо вернуть идентификаторы свеже добавленных меток, то флаг должен быть установлен.

=cut 
sub add_campaign_tags {
    my ($cid, $tags_names, %OPT) = @_;

    # если пустой запрос на добавление - выходим сразу
    return [] unless ( scalar( @{ $tags_names } ) );

    my $camp_tags = get_hash_sql(PPC(cid => $cid), 
        'SELECT tag_name, tag_id FROM tag_campaign_list WHERE cid = ?', $cid 
    );
    # переводим имена в нижний регистр для уникализации (DIRECT-27347)
    $camp_tags = { map {lc $_ => $camp_tags->{$_} } keys %$camp_tags };

    # удаляем лишние пробелы
    # отсеиваем те теги, что уже есть (все равно не вставятся в базу из-за ограничения на уникальность cid+name) - проверяем в нижнем регистре (DIRECT-27347)
    # уникализируем новые теги
    my @new_tags = xuniq { lc $_ } grep { !exists $camp_tags->{lc ($_)} } map { trim($_) } @$tags_names;

    my $num = @new_tags;
    # если теперь ничего не осталось - тоже выходим (чтобы не брать в метабазе новые id)
    return [] unless $num;

    # получаем массив id для новых тегов
    my $new_tag_ids = get_new_id_multi('tag_id', $num, cid => $cid);

    # формируем и вставляем теги
    my @for_insert = map { [ shift @$new_tag_ids, $cid, $_ ] } @new_tags;
    do_mass_insert_sql(PPC(cid => $cid),
        'INSERT IGNORE INTO tag_campaign_list (tag_id, cid, tag_name) values %s',
        \@for_insert
    );

    # возвращем новые теги, если попросили
    if ($OPT{return_inserted_tag_ids}) {
        return get_all_sql(PPC(cid => $cid), ["SELECT tag_id, tag_name AS name FROM tag_campaign_list",
                            WHERE => {cid => $cid, tag_name => \@new_tags}]);
    }
}

=head2 delete_campaign_tags ($cid)

    Удалить метки для кампании.
    Принимает номер кампании или ссылку на массив номеров (в таком случае удалит метки со всех кампаний из списка)

=cut
sub delete_campaign_tags {
    my $cid = shift;

    return unless $cid;

    my $tag_ids = get_one_column_sql(PPC(cid => $cid), [
        "SELECT tag_id FROM tag_campaign_list",
        where => { cid => SHARD_IDS }
    ]) || [];

    if (scalar(@{$tag_ids})) {
        do_delete_from_table(
            PPC(cid => $cid),
            'tag_campaign_list',
            where => { cid => SHARD_IDS }
        );
        do_delete_from_table(
            PPC(tag_id => $tag_ids),
            'tag_group',
            where => { tag_id => SHARD_IDS }
        );
        delete_shard(tag_id => $tag_ids);
    }
}

=head2 delete_group_tags ($pids)

    Удалить метки для групп.

=cut
sub delete_group_tags {
    my $pids = shift;

    $pids = [$pids] unless ref $pids eq 'ARRAY';
    if (@$pids) {
        do_delete_from_table(
            PPC(pid => $pids),
            'tag_group',
            where => { pid => SHARD_IDS }
        );
    }
}

=head2 save_tags_groups (pids, new_tags_groups)

    Cохраняет список тегов для групп. 
    Параметры:
        pids - указательно массив идентификаторов групп (для которых надо сохранить метки)
        new_tags_groups – хеш, в котором ключами являются идентификаторы меток, а значениями - массив групп(pid), которые данную метку имеют.

=cut
sub save_tags_groups {
    my ($pids, $new_tags_groups) = @_;

    my $old_groups_tags = get_groups_tags(pid => $pids);
    my $new_groups_tags;

    for my $tag_id (keys(%{$new_tags_groups})) {
        for my $pid (@{$new_tags_groups->{$tag_id}}) {
            push @{$new_groups_tags->{$pid}}, $tag_id;
        }
    }

    for my $chunk ( sharded_chunks(pid => $pids) ) {
        my @to_delete_where = ();
        my @for_insert = ();

        foreach my $pid ( @{ $chunk->{pid} } ) {
            my $to_delete = xminus($old_groups_tags->{$pid}, $new_groups_tags->{$pid});
            if (@$to_delete) {
                push @to_delete_where, ( _AND => { pid => $pid, tag_id => $to_delete } );
            }

            my $to_insert = xminus($new_groups_tags->{$pid}, $old_groups_tags->{$pid});
            foreach my $tag_id (@$to_insert) {
                push @for_insert, [$pid, $tag_id];
            }
        }

        if (@to_delete_where) {
            do_delete_from_table(
                PPC(shard => $chunk->{shard}),
                'tag_group',
                # условие получится вида (pid = XX AND tag_id IN (YY, ZZ, ..)) OR (pid = XX AND tag_id IN (...)) ...
                where => { _OR => \@to_delete_where },
            );
        }

        if (@for_insert) {
            do_mass_insert_sql(
                PPC(shard => $chunk->{shard}),
                'INSERT INTO tag_group (pid, tag_id) VALUES %s',
                \@for_insert,
            );
        }
    }
}


=head2 validate_adgroup_tags (cid, tag_ids)

    Проверка валидности меток:
        - все теги из $tag_ids принадлежат нужной кампании

=cut
sub validate_adgroup_tags {
    my ($cid, $tag_ids) = @_;

    my $all_campaign_tags = [map {$_->{tag_id}} @{(get_all_campaign_tags($cid))}];

    # false  если в tag_ids есть элементы, которые не входят в список тегов для кампании
    return iget("Метки не входят в список меток кампании.") if scalar(@{(xminus($tag_ids, $all_campaign_tags))});

    return;
}

=head2 validate_tags (cid, tags)

    Проверяет несколько меток на валидность.
    Параметры:
        cid - номер кампании, которой принадлежат метки
        tags - список меток, в котором каждый элемент имеет вид: {tag_id => $tag_id, name => $name}

    Проверяется:
        текст меток
        количество меток на кампанию (сопоставляется число меток в tags с допустимым лимитом,
            т.к. предполагается, что после валидации метки на кампании будут заменены этим набором)
        то, что перечисленные метки (кроме новых) существуют в кампании
        то, что метки уникальны (все метки в списке - имеют уникальный текст без учета регистра)

=cut 
sub validate_tags {
    my ($cid, $tags) = @_;

    # уже существующие на кампании метки в хеше id => name
    my $camp_tags_id_name = get_hash_sql(PPC(cid => $cid), 
        'SELECT tag_id, tag_name FROM tag_campaign_list WHERE cid = ?', $cid 
    );
    my %tags_names;
    my $errors_by_tag_id = {};

    # проверяем сами метки
    for my $tag (@$tags) {
        my @errors_for_current_tag;

        # Так же, как и при сохранении, считаем что для новых меток tag_id => 0
        $tag->{tag_id} ||= 0;

        # так везде метки сохраняются после trim(name), то и перед проверкой на дубликаты удалим лишние пробелы:
        # кроме того - иначе не сработает валидация на "пустую" метку
        # normalize вместо trim - DIRECT-27347
        my $tag_text = normalize_name($tag->{name});

        my $error_text = check_tag_text($tag_text);
        push @errors_for_current_tag, $error_text if $error_text;

        if ($tag->{tag_id} && !exists $camp_tags_id_name->{ $tag->{tag_id} }) {
            push @errors_for_current_tag, iget('Метка не входит в список меток кампании.');
        }

        push @errors_for_current_tag, iget('Данная метка уже используется') if exists $tags_names{$tag_text};

        # запоминаем, что такая метка уже есть, для проверки на дубликаты.
        $tags_names{$tag_text} = undef;

        push @{ $errors_by_tag_id->{ $tag->{tag_id} } }, @errors_for_current_tag if @errors_for_current_tag;
    }

    # проверяем, не превышен ли лимит на количество меток в кампании
    my $count_limit_error = check_camp_count_limit(scalar(@$tags));
    push @{ $errors_by_tag_id->{0} }, $count_limit_error if ($count_limit_error);

    # уникализируем тексты ошибок (так как они не содержат информации о том, к какой метке относятся)
    # можно было бы для новых тегов в текст ошибки включать саму метку, но это нужно на верстке делать эскейпинг против XSS
    $errors_by_tag_id->{0} = [ uniq @{$errors_by_tag_id->{0}} ] if $errors_by_tag_id->{0};

    return $errors_by_tag_id;
}

=head2 check_tag_text (tag)

    Проверка валидности меток:
        - все теги не превышают 25 символов
        - все теги содержат только допустимые символы.
    Параметры:
        tag - текст метки

=cut
sub check_tag_text {
    my $tag = shift;
    return iget("Недопустимые символы") if ($tag =~ $Settings::DISALLOW_BANNER_LETTER_RE) || ($tag =~ /,/);
    return iget("Превышена длина метки") if (length($tag) > $MAX_TAG_LENGTH);
    return iget("Пустые метки недопустимы") if (length($tag) == 0);
    return ;
}

=head2 check_camp_count_limit (count)

    Проверка количества меток на кампанию.
    Параметры:
        count - количество меток, которые предполагается сохранить для кампании.

=cut
sub check_camp_count_limit {
    my $count = shift;
    return iget("Превышено допустимое количество меток.") if $count > $MAX_TAGS_FOR_CAMPAIGN;
    return;
}

=head2 check_banner_count_limit (count)

    Проверка количества меток на баннер.
    Параметры:
        count - количество меток, которые предполагается сохранить для баннера.

=cut
sub check_banner_count_limit {
    my $count = shift;
    return iget("Превышено допустимое количество меток на объявление.") if $count > $MAX_TAGS_FOR_BANNER;
    return;
}

=head2 get_tags_groups (pids)
    
    Взять все метки для группы. Используется в редактировании меток на баннер (группу?).
    Параметры:
        pids - указатель на список групп (pid)

    Возвращает:
        хеш, в котором ключами являются номера меток, а значениями - список групп, которые эти метки используют.

=cut
sub get_tags_groups {
    my $pids = shift;
    
    my $all_tags = get_all_sql(PPC(pid => $pids), [
        "SELECT * FROM tag_group",
        where => {pid => SHARD_IDS}
    ]);
    my $result = {};
    for my $tag (@{$all_tags}) {
        $result->{$tag->{tag_id}} = [] unless (defined($result->{$tag->{tag_id}}));
        push @{$result->{$tag->{tag_id}}}, $tag->{pid};
    }
    return $result;
}

=head2 get_groups_tags (%filter)
    
    Взять все метки для группы.
    Параметры:
        pid - список групп([pid,...])
        cid - список кампаний([cid,....])

    Возвращает:
        хеш, в котором ключом является pid группы, а значениями - список меток([tag_id, ....])

=cut

sub get_groups_tags {
    my %where = @_;

    return {} unless keys %where;

    $where{'tg.pid'}    = delete $where{pid}    if $where{pid};
    $where{'tg.tag_id'} = delete $where{tag_id} if $where{tag_id};
    $where{'g.cid'}     = delete $where{cid}    if $where{cid};
    my @shard = choose_shard_param(\%where, ['pid', 'cid', 'tag_id'], set_shard_ids => 1);

    my $tags = get_all_sql(PPC(@shard), [
                'SELECT tg.tag_id, g.pid FROM phrases g LEFT JOIN tag_group tg USING(pid)',
                WHERE => \%where]);
    my %groups;
    foreach (@$tags) {
        $groups{$$_{pid}} = [] unless exists $groups{$$_{pid}};
        push @{$groups{$$_{pid}}}, $_->{tag_id} if $_->{tag_id};
    }
    
    return \%groups;
}

=head2 get_tags_by_tag_ids

    Возвращает все имена меток по tag_id в виде хеша {<tag_id> => <tag_name>}

=cut
sub get_tags_by_tag_ids {
    my $tag_ids = shift;
    return {} unless $tag_ids && @$tag_ids;
    my $tags = get_all_sql(PPC(tag_id => $tag_ids), 
        ["SELECT tag_id, tag_name FROM tag_campaign_list", where => { tag_id => SHARD_IDS }]);
    return {map {$_->{tag_id}=>$_->{tag_name}} @$tags};
}

=head2 copy_campaign_tags (src_cid, dst_cid, pids)

    копирование тегов кампании в новую кампанию.
    Параметры:
          src_cid - cid старой кампании
          dst_cid - cid новой кампании
          pids - хеш вида {<старый pid> => <соответствующий ему новый pid>}

=cut
sub copy_campaign_tags {
    my ($src_cid, $dst_cid, $pids) = @_;

    my $src_tags = get_all_campaign_tags($src_cid);
    my %src_tag_ids = map {($_->{value} => $_->{tag_id})} @$src_tags;
    my $src_groups = get_tags_groups([keys %$pids]);
    
    my $dst_tags = add_campaign_tags($dst_cid, [map {$_->{value}} @$src_tags], return_inserted_tag_ids => 1);
    
    my @groups;
    foreach my $tag (@$dst_tags) {
       my $src_id = $src_tag_ids{$tag->{name}};
       next unless $src_id && $src_groups->{$src_id};
       
       my $id = $tag->{tag_id};
       push @groups, map {
           [$id, $pids->{$_}]
       } @{$src_groups->{$src_id}}
    }
    
    do_mass_insert_sql(PPC(cid => $dst_cid),
            'INSERT INTO tag_group (tag_id, pid) VALUES %s', 
            \@groups) if @groups
}

sub trim {
    my $str = shift;
    return unless defined $str;

    $str =~ s/\s+/ /g;
    $str =~ s/^\s*//g;
    $str =~ s/\s*$//g;
    return $str;
}

=head2 normalize_name

    Приводим один тег к виду для сравнения - без пробелов и в нижнем регистре.
    my $normalized_one_tag = normalize_name($one_tag);

=cut

sub normalize_name {
    return lc trim($_[0]);
}


1;
