package TestUsers;

=encoding utf8

=head1 NAME

TestUsers

=head1 DESCRIPTION

Управление суперпользователями на тестовых серверах: работа с базой.
В базе две таблицы: ppcdict.testusers в качестве таблицы прав и
ppclog.testusers_log в качестве журнала изменений.

=head1 SYNOPSIS

 TestUsers::create_or_replace(
     uid          => 225634549,
     domain_login => 'alice',
     role         => 'super',
     UID          => 225634549,
 );
 
 my $rows;
 $rows = TestUsers::get_all()
 # в результате: в $rows результат выборки из таблицы ppcdict.testusers
 
 TestUsers::remove( uid => 225634549, UID => 225634549 );

=cut

# $Id $

use strict;
use warnings;

use Settings;

use Yandex::DBTools;
use LogTools qw/log_messages/;

=head1 SUBROUTINES/METHODS

=head2 create_or_replace

Создать в базе запись для указанного uid; если в базе такой уже есть, он перезаписывается.

Параметры:

    * uid          - ID пользователя.
    * domain_login - Логин пользователя во внутренней сети (на yandex-team.ru).
    * role         - Роль пользователя (manager, media, placer, super, superreader, support)
    * UID          - ID пользователя, который создаёт запись.

=cut

sub create_or_replace {
    my (%args) = @_;

    my $uid          = $args{uid};
    my $domain_login = $args{domain_login};
    my $role         = $args{role};
    my $UID          = $args{UID};

    do_replace_into_table( PPCDICT,
        'testusers',
        {
            uid          => $uid,
            domain_login => $domain_login,
            role         => $role,
        },
    );

    log_messages('testusers_log', {
            adminuid     => $UID,
            action       => 'Add',
            uid          => $uid,
            domain_login => $domain_login,
            role         => $role,
            });

    return;
}

=head2 get_all

Получить данные из таблицы в базе в виде массива строк (каждая строка
представляет собой ассоциативный массив).

=cut

sub get_all {
    my $rows = get_all_sql( PPCDICT, ['select * from testusers'] );
    return $rows;
}

=head2 remove

Удаляет пользователя с указанным uid из таблицы прав. Возвращает 1,
если такой пользователь нашёлся в базе; в ином случае возвращает 0.

Параметры:

    * uid - ID пользователя.
    * UID - ID пользователя, который удаляет запись.

=cut

sub remove {
    my (%args) = @_;

    my $uid = $args{'uid'};
    my $UID = $args{'UID'};

    my $all_rows = get_all();
    my ($row_to_remove) = grep { $_->{'uid'} == $uid } @$all_rows;

    return unless $row_to_remove;

    my $result = int do_sql( PPCDICT, [ 'delete from testusers', where => { uid => $uid } ] );

    if ($result) {
        log_messages('testusers_log', {
                adminuid     => $UID,
                action       => 'Remove',
                uid          => $uid,
                domain_login => $row_to_remove->{domain_login},
                role         => $row_to_remove->{role},
                });
    }

    return $result;
}

1;
