package TextTools;

=head1 NAME

    TextTools -- функции про простые преобразования текста

=head1 DESCRIPTION

    Простые функции для преобразования текста, специфичные для Директа. 

    Не должен зависеть ни от чего в Директе (включая Settings), 
    из yandex-lib допустимы только низкоуровневые модули типа Yandex::HashUtils,
    из внешнего допустимы только небольшие несложные модули про текст.

=cut

use Direct::Modern;

use base qw/Exporter/;
use Yandex::I18n;

use Math::Round;

our @EXPORT = qw(
    normalize_login
    validate_login
    phone_domain

    round2s
    round4s
    round_low_2s
    nice_int

    string2html
    html2string
    string2xml
    js_quote
    url_quote
    
    space_cleaner
    smartstrip
    smartstrip2
    strip_phrases

    get_num_array_by_str
    get_signed_num_array_by_str
    array_to_compact_str

    lcfirst_na

    get_word_for_digit

    process_text_template
    iget_or
    front_bool2int
);

=head2 normalize_login(login)

    Нормализует логин - удаляет пробелы вначале и вконце
    Заменяет точку на тире, кроме случая с лайт-логинами

=cut

sub normalize_login
{
    my $login = shift;

    return undef unless defined $login;

    $login =~ s/\./-/g unless $login =~ /\@/;
    $login =~ s/^\s+//g;
    $login =~ s/\s+$//g; 

    return $login;
}


=head2 validate_login( string, OPT )

    Проверяет правильность написания логина.
    Логин должен начинаться с буквы, и заканчиваться буквой, либо цифрой.
    Допускается использование `-`(тире)
    
    Не допускается повторение подряд точки или тире.
    
    OPT{lite} - делает простую проверки (когда абсолютная точность проверки не требуется)
                может требоваться т.к. правила в Паспорте меняются и существуют логины, созданные по другим правилам
                проверяется только допустимые символы.
    
=cut
sub validate_login
{
    my ($login, %OPT) = @_;
    return 0 if !$login;
    
    if ($OPT{lite}) {
        return 1 if $login =~ /^[a-zA-Z0-9\-\.\@]+$/;
    } elsif ($login =~ /^[a-zA-Z][a-zA-Z0-9\-\.]*[a-zA-Z0-9]$/ && length($login) <= 30) {
        # если есть подряд идущие символы "-" или "."
        if ($login =~ /\-{2}|\.{2}/) {
            return 0;
        }
        return 1;
    }
    
    return 0;
}


# Преобразовать телефон в нашей нотации в домен
sub phone_domain($) {
    my $str = shift;
    return '' if !defined $str || $str eq '';
    $str =~ s/[^\d\#]//g;
    my ( $country_c, $city_c, $phone, $ext ) = split "#" , $str;
    return $country_c.$city_c.$phone.".phone";
}


=head2 round2s

  round float number: 1.239999 -> 1.24

=cut

sub round2s
{
    return 0 if !defined $_[0];
    return int(int($_[0] * 1_000_000) / 10_000 + ($_[0] > 0 ? 0.5 : -0.5)) / 100;

    # $_[0] || return 0; # alternative
    # my $a = $_[0];
    # $a += ($a > 0) ? 0.005 : -0.005;
    # return substr($a, 0, 3 + length(int($a))) + 0;

    # $_[0] || return 0; # old
    # if ($_[0] > 0) {
        # return int($_[0] * 100 + 0.5) / 100;
    # } else {
        # return int($_[0] * 100 - 0.5) / 100;
    # }
}

=head2 round4s

  round float number to 4 decimal points: 1.236666 → 1.2367

=cut

sub round4s
{
    return 0 if !defined $_[0];
    return int(int($_[0] * 100_000_000) / 10_000 + ($_[0] > 0 ? 0.5 : -0.5)) / 10_000;
}

=head2 round_low_2s

    Returns the next lower multiple of the number
    Uses for wallets, because it is the same as in java

=cut
sub round_low_2s
{
    return 0 if !defined $_[0];
    return Math::Round::nlowmult(.01, $_[0]);
}

=head2 string2html

    mark up HTML metacharacters as corresponding entities.
    modify parametr !!

=cut
sub string2html
{
    return unless defined $_[0];
    for ($_[0]) {
        s/&/&amp;/g;
        s/</&lt;/g;
        s/>/&gt;/g;
        s/\"/&quot;/g;
    }
    return $_[0];
}

sub html2string
{
    return unless defined $_[0];
    for ($_[0]) {
        s/&lt;/</g;
        s/&gt;/>/g;
        s/&quot;/\"/g;
        s/&amp;/&/g;
    }
    return $_[0];
}

=head2 string2xml

    mark up XML metacharacters as corresponding entities.
    modify parameter !!

=cut
sub string2xml
{
    return unless defined $_[0];
    for ($_[0]) {
        s/&/&amp;/g;
        s/</&lt;/g;
        s/>/&gt;/g;
        s/\"/&quot;/g;
        s/\'/&apos;/g;
    }
    return $_[0];
}

=head2 js_quote

    quote string for javascript

=cut
sub js_quote
{
    my $str = shift;
    return unless defined $str;

    for ($str) {
        s/([\'\"\\\/])/\\$1/g;
        s/\r//g;
        s/\n/\\n/g;
    }

    return $str;
}


=head2 url_quote

    classical function for TTTmplate url filter;

=cut
sub url_quote {
    my $encode = shift;
    return unless defined $encode;

    $encode =~ s/([^A-Za-z0-9\-_.!~*\'() ])/ uc sprintf( "%%%02x",ord $1) /eg;
    $encode =~ tr/ /+/;
    return $encode;
}

=head2 smartstrip2

    Функция аналогична smartstrip, только не модифицирует свой аргумент

=cut

sub smartstrip2
{
    my ($text, %O) = @_;
    return smartstrip($text, %O) // undef;
}

=head2 smartstrip

  замена \x0D \x0A на пробелы
  обрезка концевых и начальных пробелов
  замена хитрых кавычек на обычные

  ВНИМАНИЕ! Модифицирует свой аргумент! Но пользоваться этим не стоит. Правильно смотреть на возвращаемое значение.

  Принимает необязательные именованные дополнительные параметры:
    dont_replace_angle_quotes => 1, # если 1, то кавычки-ёлочки («») не будут заменяться на "обычные" кавычки

  $stripped_text = smartstrip($text);
  $stripped_text = smartstrip($text, dont_replace_angle_quotes => 1);

=cut
sub smartstrip
{
    my ($text, %O) = @_;
    return unless defined $text;
    for ($_[0]) { # Modify in-place.
        s/\s+/ /g;
        s/^\s+//g;
        s/\s+$//g;
        s/[\x{2010}\x{2011}\x{2012}\x{2015}\x96]/-/g;
        s/[\x{203a}\x{2039}\x{2033}\x{201e}\x{201d}\x{201c}\x{201a}\x{201b}\x{2018}]/\"/g;
        s/[\x{ab}\x{bb}]/\"/g unless $O{dont_replace_angle_quotes};
        s/[\x{2032}`]/\'/g;
    }

    return $_[0];
}


=head2 strip_phrases

    очистка фраз от лишних символов

=cut
sub strip_phrases
{    
    for(@_) {
        s/(,[,\s]*)/,/gs;
        s/[,\s]+$//g;
        s/\s*"\s*/"/g;
        s/!\s+/!/g;
        s/\+\s+/+/g;
    }
}


=head2 space_cleaner(string)

    1. Удаляет пробелы в начале/конце строки.
    2. Заменяет множественные пробелы одним.
    3. Удаляет пробелы между буквами и пунктуацией.

=cut
sub space_cleaner
{
    my $str = shift;
    return undef if !defined $str;
    return '' if $str eq '';

    # strip
    $str =~ s/^\s+//gsi;
    $str =~ s/\s+$//gsi;
    $str =~ s/\s+/ /gsi;

    # удаляем пробелы между словами и знаками
    $str =~ s/(?<=[\w])\s+(?=[\.,;!:\(\)\?])//gs;
    $str =~ s/(?<=[\.,;!:\(\)\?])\s+(?=[\w])//gs;
    return $str;
}


=head2 get_word_for_digit

    get_word_for_digit(5 ,'баннер' ,'баннера' ,'баннеров');

    return 'баннеров';

=cut
sub get_word_for_digit
{
    my $number = shift;
    my @word_list = @_;

    my $last_digit = $number % 10;
    my $prev_last_digit = 0;

    if ($number > 10) {
        $prev_last_digit = int(($number % 100) / 10);
    }

    if ($last_digit > 1 && $last_digit < 5 && $prev_last_digit != 1) {
        return $word_list[1];
    } elsif ($prev_last_digit == 1 || $last_digit > 4 || $last_digit == 0) {
        return $word_list[2];
    }

    return $word_list[0];
}


=pod

    округление целых чисел до одной значащей цифры

    1,2,3,4,5,6,7,8,9
    10,
    20,
    etc

=cut
sub nice_int($) {
    my $cnt = int(sprintf "%.00E", shift);

    return $cnt;
}


=head2 $str =  array_to_compact_str(20..30, 2..10, 1);

    1-10, 20-30

    get nice representation of the integers array

=cut
sub array_to_compact_str(@) {
    my %_uniq;
    my @arr = grep {defined $_} grep {!$_uniq{$_}++} sort { $a <=> $b } @_;
    
    my @ranges;
    my $last_n = undef;
    my $series_head = undef;

    for my $n (@arr, undef) {
        if( defined $series_head and defined $n and $last_n + 1 == $n ) {
                    
        } elsif ( defined $series_head and $series_head < $last_n ) {
            push @ranges, "$series_head-$last_n";
            $series_head = $n;
        } elsif ( defined $series_head and $series_head == $last_n ) {
            push @ranges, "$series_head";
            $series_head = $n;
        } else {
            $series_head = $n;
        }
        $last_n = $n;
    }
    return join(", ", @ranges);
}


=head2 lcfirst_na

    привести первую букву к нижнему регистру, если после неё не следует также заглавная. 
    
=cut
sub lcfirst_na {
    my $str = shift;
    if ($str =~ m/^(\w)(\w)(.*)$/ ) {
        if ( $2 eq uc($2) ) {
            return $str;
        } else {
            return lcfirst($str);
        }
    } else {
        return $str;
    }
}


=head2 get_num_array_by_str

    На вход получает строку вида "123,456,344,3456754", и делиметер (по умолчанию ',')
    На выходе ссылка на массив из этих чисел: [123, 456, 344, 3456754]
    Если в строке были не численные значения, они игнорируются.
    Используется для преобразования параметров страницы в массив элементов. 
    Пример использования: 
        my $bids = get_num_array_by_str($FORM{bid});
        my $cids = get_num_array_by_str($FORM{cid}, ',');

=cut
sub get_num_array_by_str {
    my ($str, $delimeter) = @_;
    $delimeter = qr/\s*,\s*/ unless defined $delimeter;

    my @result = (defined $str) ? grep {/^\d+$/} split($delimeter, $str =~ s/(^\s+)|(\s+$)//gr) : ();
    return \@result;
}

=head2 get_signed_num_array_by_str

    Функция аналогична get_num_array_by_str только поддерживает (не игнорирует) отрицательные числа

=cut
sub get_signed_num_array_by_str {
    my ($str, $delimeter) = @_;
    $delimeter = qr/\s*,\s*/ unless defined $delimeter;

    my @result = (defined $str) ? grep {/^-?\d+$/} split($delimeter, $str =~ s/(^\s+)|(\s+$)//gr) : ();
    return \@result;
}

=head2 process_text_template($text, %params)

    Замена шаблонов в тексте, вида: #field# на соответствующее значение параметра.

    Название параметра в тексте должно удовлетворять выражению /^[a-z_]+$/i

    Пример:
        process_text_template("hello, #who#", who => "world"): hello, world

=cut

sub process_text_template {
    my ($text, %params) = @_;

    return $text unless defined $text && $text =~ /\#/;

    for my $key (grep { /^[a-z_]+$/i } keys %params) {
        my $val = $params{$key};
        # Сложная логика: $text =~ s/\#$key\#/$val/ while $text =~ /(\#+)$key(\#+)/ && (length($1) % 2) && (length($2) % 2);
        $text =~ s/\#\Q$key\E\#/$val/g;
    }
    # Сложная логика: $text =~ s/##/#/g;

    return $text;
}


=head2 truncate_text

(moved from TTTools)

truncates text up to $length characters on the closest $separator available

    truncate_text('Long text text', 10, '...', ' ')

equals to

    "Long text..."

=cut

sub truncate_text
{
    my ($text, $length, $omission, $separator) = @_;
    $length ||= 30;
    $omission //= '...';
    $separator ||= ' ';

    if ($text) {
        if (length($text) <= $length) {
            return $text;
        }
        my $str = substr($text, 0, $length);
        if (!$str) {
            return $omission;
        }
        my $idx = rindex($str, $separator) || $length;
        return substr($str, 0, $idx) . $omission;
    }
}

=head3 extract_first_words($text, $limit)
    
    Выделяет из текста $text целые слова, из первых $limit символов.
    
    Возвращает список из двух строк:
        $first_words, - первые несколько слов текста, влезающие в первые $limit символов
        $rest         - оставшаяся часть текста

=cut
sub extract_first_words {
    my ($text, $limit) = @_;

    my @words = $text =~ /((?:\W+)?\w*)/g;

    my $first_words = '';
    $first_words .= shift @words while @words && length($first_words.$words[0]) <= $limit;

    my $rest = join('', @words);
    $rest =~ s/(^\s+|\s+$)//g;

    return ($first_words, $rest);
}

=head2 iget_or


=cut

sub iget_or
{
    my ($str, $or_else) = @_;
    $or_else //= '';
    return $or_else if !defined $str || $str eq '';
    return iget($str);
}

=head2 front_bool2int($bool_text)

    Переводит значение вкл/выкл переключателя из фронтового значения в 1/0

=cut

sub front_bool2int {
    my ($bool_text) = @_;

    return int($bool_text =~ /^(?:1|true|on|yes)$/i);
}

1;
