package TimeTarget;

# $Id$

=head1 NAME

    TimeTarget
    Работа с временным таргетингом кампаний

=head1 DESCRIPTION

    Виды кодирования временного таргетинга (все в виде строк):

    1. устаревший (встречается в БД, если параметры кампании не сохранялись после введения нового формата из п.2):
       длинна строки = 31
       можно включить сразу час во всех днях
       первые 7 цифр - включен день недели, или "-" - выключен
       следующие 24 буквы - включен час, "A" - работа 00:00 -- 00:59, "B" - работа 01:00 -- 01:59 ..., или "-" - выключен
       формат: [\-1-7]{7}[\-A-X]{24}

    2. новый без коэффициентов цен:
       длинна строки = 0 - 200
       можно включить каждый час в каждом дне отдельно + на праздниках диапазон часов

       каждый день недели начинается с числа (1 - пн, 2 - вт, ... 8 - праздники, 9 - учитывать рабочие выходные)
       потом строка работающий часов "A" - работа 00:00 -- 00:59, ...,
       не работающие часы пропускаються, если весь день не работает, то он тоже пропускается

       формат: ( [1-8] [A-X]{1,24} ){1,8}

    3. новый c коэффициентами цен на каждый час:
       длинна строки = 0 - 392
       так же как в п.2 но на каждый час можно указать коэффициент для корректировки цены на поиске,
       коэффициент указывается после включенного часа буквой в нижнем регистре "b" - 10%, "c" - 20%, ... "j" - 90%, 'l' - 110%, ... 'u' - 200%.

       формат: ( [1-8] ([A-X][b-jl-u]?){1,24} ){1,8}

    4. для БК:
       DIRECT-5728:
       В зависимости от того, задан таргетинг на праздники или нет, необходимы два формата временного таргетинга.
       Если не задан отдельный таргетинг на праздники, то
       - настройки приходят как и сейчас в переменной TargetTime (но в виде массива - если в разные дни
         заданы разные интервалы) - в виде запрещённых масок, как и прежде (8-го дня недели быть не должно).

       "Запрещенные маски" (TargetTime) интерпретируются так:
       массив строк вида "<номера дней><номера часов>", дни нумеруются с 1 (пн) по 7 (вс), часы -- с A (0) до X (23).
       Отдельная строка означает: во все дни, кроме перечисленных, реклама показывается во все часы, кроме перечисленных.
       Такой формат был выбран исторически для того, чтобы оптимизировать передачу таргетингов "почти всегда включено".
       Например, ["234567A", "167"]:
         в понедельник (все, кроме вт.-вс.) показывать рекламу во все часы, кроме первого;
         в вт.-пт. (все, кроме пн., сб., вс.) показывать во все часы без исключения.


       Если на праздники задан таргетинг (показы запрещены полностью или выбрано время, когда надо/не надо показываться), то - настройки
       приходят в новой переменной TargetTimeLike (также в виде массива), в виде разрещённых масок, для праздников отдельный 8-й день недели.

       Т.е. должен приходить либо TargetTime, либо TargetTimeLike, оба одновременно приходить не должны.

       описание кодирования коэффициентов для БК: YABS-16782, YABS-21456, DIRECT-14054:
       Для коэф-тов добавляем новое поле на уровне кампании - "TimeTargetCoef",
       представляющей из себя строку фиксированной длины 192 символа ((7+1)*24=192),
       каждый символ соответствует часу и дню недели (8-й день недели - праздники и выходные).
       Т.е. сначала идут часы понедельника, поток часы вторника и т.д.
       Коэф-ты кодируются латинскими символами: 'a' - 0%, 'b' - 10%, 'c' - 20%,..., 'j' - 90%, 'k' - 100%, 'l' - 110%, ... 'u' - 200%.
       Присылать коэф-ты нужно только когда они меняются (и хотя бы один час отличается от 100%) - если
       добавилась настройка, то мы добавим у себя; если всё вернулось к 100%,
       то присылать пустую строку, мы удалим у себя настройку.

       Если выбрана настройка без учета праздников, то ставим коэффициент 'z'

    DIRECT-24258:
    Введем настройки для данного формата. Настройки разделяются символом ';' и хранятся в виде key:value
    Варианты настроек:
      p - выбранный пользователем preset:
        a - all
        w - worktime
        o - other

    Пример: 1Ab2Ab3A8;p:w

=cut

use strict;
use warnings;
use 5.010;

use DateTime;
use List::MoreUtils qw/uniq none/;

use Yandex::I18n;
use Yandex::DateTime;
use Yandex::HashUtils;
use Yandex::Validate;

use Settings;
use Yandex::DBTools;
use Tools qw//;
use TTTools qw//;
use Yandex::Clone 'yclone';
use Holidays qw/is_great_holiday is_weekend_workday/;
use geo_regions;

use utf8;

# минимальное количество рабочих часов
our $MIN_WORK_HOURS = 40;
our $MIN_WORK_HOURS_NEW = 8;

our $CHANGE_DAY_WEIGHT_PROP_NAME = 'bs_transport_change_day_weight';
our $CHANGE_DAY_WEIGHT_PROP_TTL = 60;

# все часы
our $DEFAULT_TIMETARGET = join('', map {join '', $_, 'A'..'X'} 1..7);

# список полей формы
our @FORM_FIELDS = qw/
    timeTarget timezone_id time_target_holiday_from time_target_holiday_to
    time_target_holiday_dont_show time_target_holiday time_target_holiday_coef
    time_target_preset time_target_working_holiday/;

# умолчательная таймзона
our $MSK_TIMEZONE = 'Europe/Moscow';
our $MSK_GROUP_NICK = 'russia';
# интерфейсно тайзоны разбиты на группы
our @TZ_GROUPS = (
    ## sort_sub вызываются только из sort, так что в них можно $a и $b
    ## no critic (Freenode::DollarAB)
    {nick => 'russia', name => iget_noop('Россия'),
     sort_sub => sub {$a->{offset} <=> $b->{offset}},
     offset_str_sub => sub {$_[0]->{timezone} ne $MSK_TIMEZONE ? "(MSK $_[0]->{msk_offset})" : ''}
    },
    {nick => 'cis', name => iget_noop('СНГ'),
     sort_sub => sub {$a->{name} cmp $b->{name}},
     offset_str_sub => sub {"(MSK $_[0]->{msk_offset}, GMT $_[0]->{gmt_offset})"},
    },
    {nick => 'world', name => iget_noop('Страны мира'),
     sort_sub => sub {$a->{name} cmp $b->{name}},
     offset_str_sub => sub {"(GMT $_[0]->{gmt_offset})"},
    },
    );
our %TZ_GROUPS_HASH = map {$_->{nick} => $_} @TZ_GROUPS;

# получение название столбца, в котором хранится имя для установленного языка
{
my %aliases = (
    ru => 'name',
    ua => 'ua_name',
    en => 'ename',
    tr => 'tr_name'
);
sub _name_field {

    my $lang = Yandex::I18n::current_lang();
    return $aliases{$lang} || 'name';
}
}

# Расширить хэш с таймзоной - перевычислить имя, смещения, ...
sub _extend_timezone {
    my $tz = shift;
    my $msk_offset = tz_offset($MSK_TIMEZONE);
    $tz->{offset} = tz_offset($tz->{timezone});
    $tz->{gmt_offset} = tz_offset_sec_to_str($tz->{offset});
    $tz->{msk_offset} = tz_offset_sec_to_str($tz->{offset} - $msk_offset);

    $tz->{id} = $tz->{timezone_id};
    $tz->{offset_str} = $TZ_GROUPS_HASH{$tz->{group_nick}}->{offset_str_sub}->($tz);
    $tz->{name} .= " ".$tz->{offset_str} if $tz->{offset_str};
}

=head2 get_timezone_groups

    Получить список групп таймзон

=cut
sub get_timezone_groups {
    my $name_field = _name_field();
    my $all_timezones = get_all_sql(PPCDICT, "
                                    SELECT timezone_id, country_id, timezone, group_nick,
                                           $name_field as name
                                      FROM geo_timezones
                                ");
    my @ret;
    for my $tz_group (@TZ_GROUPS) {
        my $group = hash_merge {}, $tz_group;
        $group->{name} = iget($group->{name});
        $group->{timezones} = [grep {$_->{group_nick} eq $group->{nick}} @$all_timezones];
        for my $tz (@{$group->{timezones}}) {
            _extend_timezone($tz);
        }
        $group->{timezones} = [sort {$group->{sort_sub}->()} @{$group->{timezones}}];
        delete @$group{qw/sort_sub offset_str_sub/};
        push @ret, $group;
    }
    return \@ret;
}

=head2 get_timezone

    Получить структуру с таймзоной по id или по переданным параметрам
    По-умолчанию - возвражаем московскую

=cut

sub get_timezone {
    my @params = @_;
    my ($cond, $order) = ('', '');
    if (!@params || @params == 1 && !$params[0]) {
        $cond = {group_nick => $MSK_GROUP_NICK, timezone => $MSK_TIMEZONE};
    } elsif (@params == 1) {
        # по id
        $cond = {timezone_id => $params[0]};
    } else {
        $cond = {@params};
        $order = "ORDER BY IF(group_nick='russia',0,1), timezone_id";
    }
    my $name_field = _name_field();
    my $tz =  get_one_line_sql(PPCDICT, ["
                                    SELECT timezone_id, country_id, timezone, group_nick,
                                           $name_field as name
                                      FROM geo_timezones",
                                     WHERE => $cond,
                                     $order,
                                     LIMIT => 1,
                               ]);
    _extend_timezone($tz) if $tz;
    return $tz;
}

=head2 get_timezone_form_params

    По параметрам, аналогичным get_timezone, получить хэш из полей
    timezone_id, timezone_text

=cut
sub get_timezone_form_params {
    my $tz = get_timezone(@_);
    return {timezone_id => $tz->{timezone_id}, timezone_text => $tz->{name}};
}

=head2 tz_offset_sec_to_str

    Преобразование смещения из секунд в человекочитаемый вид

=cut
sub tz_offset_sec_to_str {
    my $sec = shift;
    my $h = int(1.0001 * $sec / 3600);
    my $m = abs($sec - $h * 3600)/60;
    return sprintf("%s%02d:%02d", ($sec >= 0 ? '+' : '-'), abs($h), $m);
}

=head2 parse_timetarget

    По строке '1ABC2DB8'
    Построить хэш:
      * time_target_preset - какой режим расписания показа выбран - all -
        круглосуточно, worktime - в рабочее время, other - другое
      * time_target_holiday - значение галки "учитывая праздничные дни"
      * time_target_holiday_dont_show - праздничные дни 0 - "показывать в
        праздник", 1 - "не показывать в праздники"
      * time_target_holiday_from, time_target_holiday_to - значения
        селекторов  времени в учете праздничных дней - от 0 до 24
      * timeTarget - строка без праздников
      * time_target_working_holiday - учитывать рабочие выходные

=cut
sub parse_timetarget {
    my ($timetarget) = @_;
    $timetarget ||= $DEFAULT_TIMETARGET;
    my %ret;

    state $hour2letter = { map { $_ => chr(ord('A') + $_) } 0 .. 23 };

    # Разбираем настройки
    # DIRECT-24258: Прочитаем из настроек значение time_target_preset
    if ($timetarget =~ /^([^;]*);(.*)$/) {
        $timetarget = $1 || $DEFAULT_TIMETARGET;
        my %settings = $2 ? (map { /^([^:]+):(.*)$/ } (split /;/, $2)) : ();
        for my $k (keys %settings) {
            my $v = $settings{$k};
            if ($k eq 'p') {
                # time_target_preset
                if    ($v eq 'a') { $ret{time_target_preset} = 'all'; }
                elsif ($v eq 'w') { $ret{time_target_preset} = 'worktime'; }
                elsif ($v eq 'o') { $ret{time_target_preset} = 'other'; }
            }
        }
    }

    my $is_extend;
    my $hours = hours_hash($timetarget);
    $ret{time_target_working_holiday} = delete($hours->{9}) ? 1 : 0;
    if (exists $hours->{8}) {
        $ret{time_target_holiday} = 1;
        my @holi_hours = sort {$a <=> $b} keys %{$hours->{8}};
        if (@holi_hours) {
            $ret{time_target_holiday_from} = $holi_hours[0];
            $ret{time_target_holiday_to} = $holi_hours[-1] + 1;

            # если выставлен коэффициент на праздники (он есть только один на все часы)
            my $time_target_holiday_coef = ( values %{$hours->{8}} )[0];
            if ($time_target_holiday_coef
                && ($time_target_holiday_coef > 0 && $time_target_holiday_coef < 100
                    ||
                    $time_target_holiday_coef > 100 && $time_target_holiday_coef <= 200
                   )
               ) {
                $ret{time_target_holiday_coef} = $time_target_holiday_coef;
                $is_extend = 1;
            }

        } else {
            $ret{time_target_holiday_dont_show} = 1;
        }
    }

    # DIRECT-24258: time_target_preset может быть уже получен из настроек
    if (!$ret{time_target_preset}) {
        if (!$ret{time_target_holiday} && $timetarget eq $DEFAULT_TIMETARGET) {
            $ret{time_target_preset} = 'all';
        } else {
            # получаем часы
            my @days = grep {$hours->{$_}} (1..7);
            my @tt = map { [ sort {$a <=> $b} keys %{$hours->{$_}} ] } grep {$hours->{$_}} (1..7);
            if (
                # во все дни часы одинаковые
                uniq(map {join ":", @$_} @tt) == 1
                # нет дырок в днях
                && $days[-1] - $days[0] + 1 == @days
                # нет дырок в часах
                && $tt[0]->[-1] - $tt[0]->[0] + 1 == @{$tt[0]}
            ) {
                $ret{time_target_preset} = 'worktime';
            } else {
                $ret{time_target_preset} = 'other';
            }
        }
    }

    # собираем timeTarget
    my @days_parts;
    for my $day_num (sort grep {/^[1-7]$/} keys %$hours) {
        my $hours_on_day = '';
        for my $hour_num (sort {$a <=> $b} keys %{ $hours->{$day_num} }) {
            next unless $hours->{$day_num}->{$hour_num};
            $hours_on_day .= $hour2letter->{$hour_num};
            if ($hours->{$day_num}->{$hour_num} > 0 && $hours->{$day_num}->{$hour_num} < 100
                ||
                $hours->{$day_num}->{$hour_num} > 100 && $hours->{$day_num}->{$hour_num} <= 200
               )
            {
                $hours_on_day .= coef2letter($hours->{$day_num}->{$hour_num});
                $is_extend = 1;
            }
        }

        push @days_parts, $day_num . $hours_on_day;
    }
    $ret{timeTarget} = join "", @days_parts;
    $ret{timeTargetMode} = $is_extend ? 'extend' : 'simple';
    $ret{show_on_weekend} = exists $hours->{6} && exists $hours->{7};

    return \%ret;
}

=head2 pack_timetarget

    Упаковка timeTarget, пришедшего из формы к виду для бд

=cut
sub pack_timetarget {
    my ($tt) = @_;

    # DIRECT-24258: Запишем time_target_preset в блок настроек
    my %settings;
    if (my $v = $tt->{time_target_preset}) {
        if    ($v eq 'all')      { $settings{p} = 'a'; }
        elsif ($v eq 'worktime') { $settings{p} = 'w'; }
        elsif ($v eq 'other')    { $settings{p} = 'o'; }
    }

    my $timetarget = $tt->{timeTarget};
    if ( !$tt->{time_target_holiday} && (!$timetarget || $timetarget eq $DEFAULT_TIMETARGET)
            && !$tt->{time_target_working_holiday}
    ) {
        return %settings ? ';'.join(';', map { $_.':'.$settings{$_} } keys %settings) : undef;
    }
    $timetarget ||= $DEFAULT_TIMETARGET;
    if ($tt->{time_target_holiday}) {
        if ($tt->{time_target_holiday_dont_show}) {
            $timetarget .= '8';
        } else {
            my $time_target_holiday_coef = '';
            if ($tt->{time_target_holiday_coef}
                && ($tt->{time_target_holiday_coef} > 0 && $tt->{time_target_holiday_coef} < 100
                    ||
                    $tt->{time_target_holiday_coef} > 100 && $tt->{time_target_holiday_coef} <= 200
                   )
               )
            {
                $time_target_holiday_coef = coef2letter($tt->{time_target_holiday_coef});
            }
            $timetarget .= '8' . join '', map {hour2letter($_) . $time_target_holiday_coef}
                                            ($tt->{time_target_holiday_from} .. $tt->{time_target_holiday_to}-1);
        }
    }
    $timetarget .= '9' if $tt->{time_target_working_holiday};

    $timetarget .= ';'.join(';', map { $_.':'.$settings{$_} } keys %settings) if %settings;

    return $timetarget;
}

=head2 validate_timetarget

    Проверка параметров временного таргетинга

=cut

sub validate_timetarget {
    my ($tt, $allow_extended_timetarget, $client_id, $has_new_min_days_limit_flag) = @_;
    my $timeTarget = $tt->{timeTarget} || '';

    return iget("Временной таргетинг задан неверно") if ! $allow_extended_timetarget && is_extended_timetarget($timeTarget);

    # Проверим флаги на соответствие пресету
    return iget('Временной таргетинг задан неверно') if ($tt->{time_target_preset} // '') eq 'all' && ($tt->{time_target_holiday} || $tt->{time_target_working_holiday});

    my @tt_days = split /(?=\d)/, $timeTarget;
    if (@tt_days > 7 || grep {!/^\d([A-X][b-jl-u]?){0,24}$/} @tt_days) {
        return iget("Временной таргетинг задан неверно");
    }
    my $hours = hours_hash($timeTarget);

    if ($timeTarget ne '' && !$has_new_min_days_limit_flag && scalar(map {keys %{$hours->{$_}}} grep {/[1-5]/} keys %$hours) < $MIN_WORK_HOURS) {
        return iget('Объявления должны быть включены не менее %s часов в неделю в рабочие дни. Измените расписание показа объявлений.', $MIN_WORK_HOURS);
    }

    if ($timeTarget ne '' && $has_new_min_days_limit_flag && scalar(map {keys %{$hours->{$_}}} grep {/[1-7]/} keys %$hours) < $MIN_WORK_HOURS_NEW) {
        return iget('Объявления должны быть включены не менее %s часов в неделю. Измените расписание показа объявлений.', $MIN_WORK_HOURS_NEW);
    }

    if ($tt->{timezone_id}) {
        return iget('Неверно указана временная зона') if !get_timezone($tt->{timezone_id});
    }

    if ($tt->{time_target_holiday} && !$tt->{time_target_holiday_dont_show}) {
        if (!is_valid_int($tt->{time_target_holiday_from}, 0, 23) || !is_valid_int($tt->{time_target_holiday_to}, $tt->{time_target_holiday_from}+1, 24)) {
            return iget("Временной таргетинг задан неверно");
        }
    }

    return ();
}

=head2 hours_hash

    Из строки timetarget сделать хэш с ключами деньНедели / час

=cut
sub hours_hash {
    my ($timetarget) = @_;

    state $letter2hour = { map { $_ => ord($_) - ord('A') } ( 'A' .. 'X' ) };
    state $letter2coef = { map { $_ => (ord($_) - ord('b') + 1) * 10 } ( 'b' .. 'u' ) };

    state %cache;

    return yclone( $cache{$timetarget} ) if exists $cache{$timetarget};

    %cache = () if keys %cache > 1000;

    my %hours;
    if (length($timetarget) == 31 && $timetarget =~ /^[1-7\-]{7}[A-X\-]{24}$/) {
        # старый формат
        # TODO - убрать после конвертации базы, expire_date - 01.04.2010
        my %day_hours = map {$_ => 100} grep {index($timetarget, hour2letter($_)) < 0} (0..23);
        %hours = map {$_ => {%day_hours}} grep {$timetarget !~ /$_/} (1..7);
    } else {
        # новый, нормальный формат
        $timetarget =~ s/^([^;]*);(.*)$/$1/;
        my $dow = '';
        my $hour;
        for my $char (grep {/[0-9A-Xb-jl-u]/} split //, $timetarget) {
            if ($char =~ /[0-9]/) {
                $dow = $char;
                $hours{$dow} = {};
            } elsif ($char =~ /[A-X]/) {
                $hour = $letter2hour->{$char};
                $hours{$dow}->{$hour} = 100;
            } elsif ($char =~ /[b-jl-u]/ && defined $hour) {
                $hours{$dow}->{$hour} = $letter2coef->{$char};
            } else {
                warn "$timetarget is not valid";
            }
        }
    }

    $cache{$timetarget} = \%hours;
    return yclone(\%hours);
}

=head2 timetarget_status

    По строке timetarget получить тесктовое состояние кампании на текущий момент

    $str_status = TimeTarget::timetarget_status($camp->{timeTarget}, $camp->{timezone_id}); # 'Идут показы'

=cut

sub timetarget_status {
    my ($timeTarget, $timezone_id) = @_;

    my $timetarget_status = timetarget_status_raw($timeTarget, $timezone_id);
    my $tz_offset_str = $timetarget_status->{tz_offset} ? " $timetarget_status->{tz_offset}" : "";

    if (! defined $timetarget_status->{status_id}) {
        return undef;
    } elsif ($timetarget_status->{status_id} eq 'ACTIVE') {
        return iget('Идут показы');
    } elsif ($timetarget_status->{status_id} eq 'TODAY') {
        return iget('Показы начнутся в %s', $timetarget_status->{time}) . $tz_offset_str;
    } elsif ($timetarget_status->{status_id} eq 'TOMORROW') {
        return iget('Показы начнутся завтра в %s', $timetarget_status->{time}) . $tz_offset_str;
    } elsif ($timetarget_status->{status_id} eq 'LATER') {
        return iget('Показы начнутся %s в %s', $timetarget_status->{date}, $timetarget_status->{time}) . $tz_offset_str;
    }
}

=head2 timetarget_current_coef

    По строке timetarget получить текущий действующий коэффициент

    $current_coef = TimeTarget::timetarget_current_coef($camp->{timeTarget}, $camp->{timezone_id}); # 90

=cut

sub timetarget_current_coef {
    my ($timeTarget, $timezone_id) = @_;

    my $timetarget_status = timetarget_status_raw($timeTarget, $timezone_id);
    return $timetarget_status->{coef};
}

=head2 timetarget_status_raw

    по $timeTarget, $timezone_id возвращает id статуса (идут ли показы или параметры когда начнутся) и с каким коэффициентом
    результаты кешируются

=cut

{
    my %timetarget_status_raw_cache = ();
    my $cache_hour_id = ''; # последний час на который сохранили вычисленные статусы, если он не совпадает с текущим то нужно очистить кеш и вычислить заново

sub timetarget_status_raw {
      my ($timeTarget, $timezone_id) = @_;

      # Блок настроек тут фигурировать не должен
      $timeTarget =~ s/^([^;]*);(.*)$/$1/ if defined $timeTarget;

      my $current_hour_id = join("-", (localtime(time()))[2 ... 5]); # в id часа учитываем текущий час, день, месяц и год
      my $lang = Yandex::I18n::current_lang();
      my $cache_id = $lang . $current_hour_id . ($timeTarget || '') . ($timezone_id || '');

      if ($cache_hour_id eq $current_hour_id && exists $timetarget_status_raw_cache{$cache_id}) {
          # в текущем часу такой timetarget уже вычисляли
          return $timetarget_status_raw_cache{$cache_id};
      }

      if ($cache_hour_id ne $current_hour_id) {
          # наступил следующий час, чистим кеш
          %timetarget_status_raw_cache = ();
          $cache_hour_id = $current_hour_id;
      }

      # по умолчанию - показываем всегда
      my $hours = hours_hash($timeTarget || $DEFAULT_TIMETARGET);

      my $tz = TimeTarget::cached_tz_by_id($timezone_id);
      my $region_id = $tz->{country_id} // $geo_regions::RUS;

      # Если регион таимтаргетинга не из страны присутствия (КУБР+Т) и для него включены
      # настройки показа по праздникам/переносом, то БК использует российские праздники и переносы.
      # И мы вслед за БК должны такое поддерживать.
      if (none {$region_id == $_} @Holidays::LOCAL_HOLIDAYS_SUPPORTED_REGIONS) {
          $region_id = $geo_regions::RUS;
      }

      my $now = DateTime->now(time_zone => $tz->{timezone}||$MSK_TIMEZONE)->set(minute => 0, second => 0);
      my $cur = $now->clone;
      # дальше, чем за месяц не заходим
      my $border_time = DateTime->now(time_zone => $MSK_TIMEZONE)->set(minute => 0, second => 0, hour => 0)
                            + duration(months => 1);

      my $coef = 0;

      # находим время, когда начнём показываться - оно окажется в $cur
      # cur_day двигается по дням, чтобы не пропустить день из-за перевода часов прибавляем по 23 часа,
      # если один и тот же день нам попадётся дважды - не страшно.

      # наверное, правильней всего было бы добавлять к now@UTC по часу, менять таймзону и проверять,
      # но это затратней
      my $cur_day = $cur->clone;
      my $carrying_day;
    DATES:
      while($cur_day < $border_time) {
          $cur = $cur_day->clone;
          # print STDERR "  $cur\n";
          # 8 - holidays 9 - working holidays
          my $dow = exists $hours->{9} && ($carrying_day = is_weekend_workday($cur->strftime("%Y-%m-%d"), $region_id))
              ? $carrying_day
              : (exists $hours->{8} && is_great_holiday($cur->strftime("%Y-%m-%d"), $region_id) ? 8 : $cur->dow);
          # print STDERR "  dow: $dow\n";
          my @good_hours = sort {$a <=> $b} keys %{$hours->{$dow}};
          # print STDERR "  gh: ".join(",", @good_hours)."\n";

          for my $h (@good_hours) {
              eval {
                  $cur->set_hour($h);
                  $coef = $hours->{$dow}->{$h};
              };
              last DATES if !$@ && $cur >= $now;
          }
          $cur_day->add(hours => 23);
      }
      #print STDERR " res: $cur\n";

      # красиво форматируем ответ
      #$cur->set_time_zone($MSK_TIMEZONE);
      my $tz_offset = $tz->{msk_offset} ? $tz->{offset_str} : undef;
      my $result;

      if ($cur >= $border_time) {
          $result = {status_id => undef, coef => 0};
      } elsif ($cur == $now) {
          $result = {status_id => 'ACTIVE', coef => $coef};
      } else {
          my $days_left = $now->delta_days($cur)->delta_days(); # вычисляем через сколько дней начнутся показы
          if ($days_left == 0) {
              # 'Показы начнутся в 12:00'
              $result = {status_id => 'TODAY', time => sprintf('%s:00', $cur->hour), tz_offset => $tz_offset, coef => 0};
          } elsif ($days_left == 1) {
              # 'Показы начнутся завтра в 14:00'
              $result = {status_id => 'TOMORROW', time => sprintf('%s:00', $cur->hour), tz_offset => $tz_offset, coef => 0};
          } elsif ($days_left < 7) {
              # 'Показы начнутся в среду в 15:00'
              $result = {status_id => 'LATER', date => TTTools::format_date($cur->ymd, strftime => "{dow_at}"), time => sprintf('%s:00', $cur->hour), tz_offset => $tz_offset, coef => 0};
          } else {
              # 'Показы начнутся 11.11 в 16:00'
              $result = {status_id => 'LATER', date => $cur->strftime("%d.%m"), time => sprintf('%s:00', $cur->hour), tz_offset => $tz_offset, coef => 0};
          }
      }

      $timetarget_status_raw_cache{$cache_id} = $result;
      return $result;
}}


=head2 letter2hour

    По букве A-X вернуть номер часа 0-23

=cut

sub letter2hour {
    my $letter = shift;
    return ord(uc($letter)) - ord('A');
}

=head2 letter2coef

    По букве b-j, l-u вернуть коэффициент 10 - 90, 110 - 200
    для "k" возвращаем 100, но сохранять такой не нужно, также как и "a"

=cut

sub letter2coef {
    my $letter = shift;
    return (ord(lc($letter)) - ord('b') + 1) * 10;
}

=head2 hour2letter

    По номеру часа 0-23 вернуть букву A-X

=cut

sub hour2letter {
    my $hour = shift;
    return chr(ord('A') + $hour);
}

=head2 coef2letter

    По коэффициенту 0 - 200 вернуть букву a-u

=cut

sub coef2letter {
    my $coef = shift;
    return chr(ord('b') + $coef / 10 - 1);
}

=head2 is_extended_timetarget

    По строке таргетинга определить является ли таргетинг расширенным (используются коэффициенты для цены)
      10%-90%
      110%-200%

=cut

sub is_extended_timetarget {
    my $timetarget = shift || '';
    my ($tt_wo_settings) = split(/;/, $timetarget, 2);
    return defined $tt_wo_settings && $tt_wo_settings =~ /[b-jl-u]/ ? 1 : 0;
}

# -----------------------------------------------------------------------------

=head2 clear_extended_timetarget

    Стираем коэффициенты временного таргетинга

=cut

sub clear_extended_timetarget($) {
    my $timetarget = shift;
    return undef if ! defined $timetarget || $timetarget eq '';
    my ($tt_wo_settings, $settings) = split(/;/, $timetarget, 2);
    $tt_wo_settings =~ s/[a-u]//g;
    return $tt_wo_settings . (defined $settings && $settings ne '' ? ";$settings" : "");
}

=head2 bs_timetarget

    По строке таргетинга получить хэш условия для транспорта в БК

=cut
sub bs_timetarget {
    my ($timetarget) = @_;
    return undef if !$timetarget || $timetarget =~ /^;/;
    my $hours = hours_hash($timetarget);

    my $working_holiday = delete($hours->{9}) ? 1 : 0;
    my $bs_timetarget;

    if (!exists $hours->{8}) {
        # запрещённые маски
        my %masks_hash;
        while(my ($day, $h) = each %$hours) {
            my $mask = join "", map {hour2letter($_)} grep {!exists $h->{$_}} (0..23);
            push @{$masks_hash{$mask}}, $day;
        }
        my @masks;
        while(my ($hour_mask, $days_arr) = each %masks_hash) {
            my %days_hash = map {$_ => 1} @$days_arr;
            my $days_mask = join '', grep {!$days_hash{$_}} 1..7;
            push @masks, "$days_mask$hour_mask";
        }

        $bs_timetarget = @masks == 1 && $masks[0] eq ''
            ? undef
            : {TargetTime => [sort @masks]};
    } else {
        # разрешённые маски
        my %masks_hash;
        while(my ($day, $h) = each %$hours) {
            my $mask = join "", map {hour2letter($_)} grep {exists $h->{$_}} (0..23);
            next if !$mask;
            push @{$masks_hash{$mask}}, $day;
        }
        my @masks;
        while(my ($hour_mask, $days_arr) = each %masks_hash) {
            push @masks, join('', sort {$a<=>$b} @$days_arr).$hour_mask;
        }

        $bs_timetarget = {TargetTimeLike => [sort @masks]};
    }

    $bs_timetarget->{TargetTimeWorking} = $working_holiday if $bs_timetarget;
    return $bs_timetarget;

}

=head2 get_day_weight
    По строке таргетинга считаем дни с разрешенными показами и в зависимости от этого считаем DayWeight
=cut
sub get_day_weight {
    my ($timetarget) = @_;

    state $change_day_weight_prop = Property->new($CHANGE_DAY_WEIGHT_PROP_NAME);
    # объяснение магических чисел https://st.yandex-team.ru/DIRECT-159807
    my $day_weight_without_time_target = $change_day_weight_prop->get($CHANGE_DAY_WEIGHT_PROP_TTL) ? 214285 : 333333;

    return $day_weight_without_time_target if !$timetarget || $timetarget =~ /^;/;
    my $hours = hours_hash($timetarget);
    my @days = grep {keys %{$hours->{$_}}} (1..7);
    my $days_count = scalar(@days);
    if ($days_count == 1) {
        return 1000000;
    } elsif ($days_count == 2) {
        return 500000;
    } else {
        return 333333;
    }
}

=head2 bs_timetarget_coef

    По строке таргетинга получить строку коэффициентов для транспорта в БК
    YABS-16782

=cut

sub bs_timetarget_coef {
    my $timetarget = shift;

    return '' if ! $timetarget || $timetarget =~ /^;/;
    return '' if ! is_extended_timetarget($timetarget);

    my $hours = hours_hash($timetarget);

    my $time_target_coef_str = '';
    for my $day (1 .. 7) {
        for my $hour (0 .. 23) {
            my $coef = exists $hours->{$day} && exists $hours->{$day}->{$hour} ? $hours->{$day}->{$hour} : 0;
            $time_target_coef_str .= coef2letter($coef || 0);
        }
    }

    # если без учета праздников - то коеф-ты 'z'
    if (exists $hours->{8}) {
        for my $hour (0 .. 23) {
            my $coef = exists $hours->{8}->{$hour} ? $hours->{8}->{$hour} : 0;
            $time_target_coef_str .= coef2letter($coef || 0);
        }
    } else {
        $time_target_coef_str .= 'z' x 24;
    }

    return $time_target_coef_str;
}

{
my %TZ_BY_ID;

=head2 cached_tz_by_id

    Кеширующее получение имени таймзоны по id

=cut

sub cached_tz_name_by_id {
    my $timezone_id = shift;

    my $result = cached_tz_by_id($timezone_id);

    return $result && $result->{timezone} ? $result->{timezone} : $MSK_TIMEZONE;
}

sub cached_tz_by_id {
    my $timezone_id = shift;

    my $cache_key = defined $timezone_id ? $timezone_id : 'undef';
    # от состояния локали может зависеть результат get_timezone
    my $lang = Yandex::I18n::current_lang();
    if (! exists $TZ_BY_ID{$lang}{$cache_key}) {
        $TZ_BY_ID{$lang}{$cache_key} = get_timezone($timezone_id);
    }

    return yclone($TZ_BY_ID{$lang}{$cache_key});
}

}

1;
