##################################################
#
#  Direct.Yandex.ru
#
#  Tools
#  собрание разнообразных функций
#
#  $Id$
#
##################################################

=head1 NAME

Tools - собрание разнообразных функций

=head1 DESCRIPTION

вспомогательные функции
взятые из старого Common.pm

=cut

package Tools;
## no critic (TestingAndDebugging::RequireUseStrict, TestingAndDebugging::RequireUseWarnings)

require Exporter;

our @ISA = qw(Exporter);
our @EXPORT = qw(
    log_cmd
    log_mediaplan

    validate_structure

    direct_frontends

    string2sms_time
    sms_time2string

    get_childpids_by_parentpid

    get_order_offset_limit
    get_order_offset_limit_sql


    encode_json_and_compress
    decode_json_and_uncompress
    
    parse_image_name_from_url

    make_copy_sql_strings

    get_mds_avatars_handler
    get_clickhouse_handler

    calc_percentile

    calc_min_pay_sum

    calc_payment_suggest_block
);

use warnings;
use strict;
use feature 'state';

# subs_placing: Tools не должен зависеть от Settings. Если есть функции, которым это почему-то точно надо -- отселить 
use Settings;
# subs_placing: Tools, зависящий от DBTools -- подозрительно и неправильно. Функции, работающие с базой -- в Primitives или более высокоуровневые модули
use Yandex::DBTools;
use Yandex::I18n;
use POSIX qw(strftime floor ceil);

use Yandex::Avatars;
use Yandex::HighlightWords;
use Yandex::TimeCommon;
use Yandex::DateTime;
use Yandex::Log;
use Yandex::Shell;
use Yandex::IDN qw(is_valid_domain);
use Yandex::URL qw/get_num_level_domain get_top_level_domain/;
use Yandex::Compress;
use Yandex::Runtime;
use Yandex::Trace qw/current_trace/;
use Yandex::ListUtils qw/xminus/;

use Carp qw/cluck croak/;
use utf8;
use Encode;
use Time::Local qw(timelocal);
use Path::Tiny;
use List::Util qw/max min sum first any/;
use Scalar::Util qw(looks_like_number);
use YAML::XS;
use JSON;
use Property;
use Currencies;

use EnvTools qw//;
use HashingTools qw/bs_md5int_utf8/;

use DirectContext;

my $UID;
my $client_id;

=head1 FUNCTIONS

=cut

#======================================================================

=head2

for execute from main.pl - save global $UID

=cut

sub _save_vars
{
    $UID = shift;
}

=head2

for execute from main.pl - save global $UID

=cut
sub _save_client_id
{
    $client_id = shift;
}

=head2

get operator uid

=cut
sub _get_operator_uid
{
    return $UID;
}

=head2

get client id

=cut
sub _get_client_id
{
    return $client_id;
}

#======================================================================

=head2 log_cmd

  log_cmd(\%FORM);
  log of all cmds

=cut

{
sub log_cmd
{
    my ($FORM) = @_;
    state $log_syslog = Yandex::Log->new(use_syslog => 1, no_log => 1, syslog_prefix => 'PPCLOG', log_file_name => 'ppclog_cmd.log');

    my $FORM_FULL = {%$FORM};
    # прячем пароли и токены
    $FORM_FULL->{$_} = '********' for grep {/(token|password)[0-9]*$/} keys %$FORM_FULL;

    # эти данные пишутся в собственные столбцы (uid) или не пишутся совсем (authredirlevel)
    my @params_remove = qw/sort reverse ncrnd uid UID cmd rcmd uid_url
        submitcmd submitprice ulogin authredirlevel
        _runtime _http_status _logtime _cpu_user_time _role _reqid
        ir_param_oauth_token/;
    my $params_remove_re = join '|', @params_remove;
    delete $FORM_FULL->{$_} for @params_remove;

    my ($uid, $cids_string, $bid, $cmd, $runtime, $logtime, $cpu_user_time, $http_status, $role) =
        @{$FORM}{qw/uid cid bid cmd _runtime _logtime _cpu_user_time _http_status _role/};
    if (!defined $role && defined $DirectContext::current_context) {
        $role = $DirectContext::current_context->login_rights->{role};
    }
    my $pid;
    if ($FORM->{pid} && $FORM->{pid} =~ m/^([0-9]+)(?:,[0-9]+)*$/) {
        $pid = $1
    } elsif ($FORM->{adgroup_ids} && $FORM->{adgroup_ids} =~ m/^([0-9]+)(?:,[0-9]+)*$/) {
        $pid = $1
    }
        
    $logtime = strftime("%Y%m%d%H%M%S", localtime(int($logtime || time())));
    $cids_string ||= $FORM->{campaign_old} if ($FORM->{add_to}||'') eq 'old'; # в saveNewCamp, addBanner может означать cid.

    if (!$bid && $FORM->{bids} && $FORM->{bids} =~ m/^([0-9]+)(?:,[0-9]+)*$/) {
        # после объединения редактирования и мультиредактирования номер баннера стал полем bids
        # сохраняем первый из номеров
        $bid = $1;
    }

    for ($UID, $uid, $cids_string, $bid, $pid, $runtime, $http_status, $cpu_user_time) {$_ = 0  unless defined $_}
    for ($cmd, $ENV{REMOTE_ADDR}) {$_ = '' unless defined $_}

    my $yandexuid = ($ENV{HTTP_COOKIE} || '') =~ /(?:^|\s|;)yandexuid=(\d+)/ ? $1 : 0;

    #параметр cid м.б. списком cid'ов, берем первый
    my @cids = split(/\D+/, $cids_string);

    my $log_data = {
        logtime => $logtime,
        cid => $cids[0],
        bid => $bid,
        pid => $pid,
        ip => $ENV{REMOTE_ADDR},
        service => 'direct.perl.web',
        cmd => $cmd,
        runtime => $runtime,
        http_status => $http_status,
        cluid => $uid,
        reqid => Yandex::Trace::trace_id() || 0,
        uid => $UID,
        host => $EnvTools::hostname,

        # web-interface only params
        cpu_user_time => $cpu_user_time,
        yandexuid => $yandexuid,
        role => $role,

        param => trim_log_data($FORM_FULL),
        proc_id => $$,
        log_type => 'cmd',
    };

    eval {
        $log_syslog->out($log_data);
    };
    if ($@) {
        print STDERR "failed to syslog: $@\n";
    }

    return;
}
}

#======================================================================

=head2 log_mediaplan
    action - текст, откуда вызывается это логирование
    cid    - номер кампании
    values - hash_href со значениями

=cut
sub log_mediaplan($$$) 
{
    my ($action, $cid, $values) = @_;
    state $log_syslog = Yandex::Log->new(use_syslog => 1, no_log => 1, syslog_prefix => 'MEDIAPLAN', log_file_name => 'mediaplan.log');
    my $log_data = {
        action => $action,
        cid => $cid,
        reqid => Yandex::Trace::trace_id() || 0,
        values => trim_log_data($values),
    };
    eval {
        $log_syslog->out($log_data);
    };
    if ($@) {
        print STDERR "failed to syslog: $@\n";
    }

}

#======================================================================

=head2 validate_structure($original, $example, %options)

    Сравнение структуры $original со структурой $example
    %options
    only_type => 1|0 только проверка совпадения типов $original и $example

=cut

# subs_placing: validate_structure + _is_same_struct -- в какой-нибудь модуль про валидацию

sub validate_structure {
    
    my ($original, $example, %options) = @_;
        
    my $is_only_type = 
        $options{only_type}
        || ref $example eq 'ARRAY' && @$example == 0
        || ref $example eq 'HASH' && keys(%$example) == 0;

    return 'type_mismatch' if ref $original ne ref $example;
    return '' if $is_only_type;
    return _is_same_struct($original, $example, {}) ? '' : 'not equal';
}

sub _is_same_struct {
    
    my ($original, $samples, $addresses) = @_;
    
    if (ref $samples) {
        die 'a circular reference defined' if exists $addresses->{$samples};
        $addresses->{$samples} = 1
    }
    
    if (ref $original xor ref $samples) {
        return 0;  
    } elsif (ref $original ne ref $samples) {
        return 0
    } elsif (ref $samples eq 'HASH') {
        return 0  if any {!exists $samples->{$_}} keys $original;
        foreach (keys %$samples) {
            next  if !defined $samples->{$_};
            return 0  if !exists $original->{$_};
            my $is_same;
            return $is_same  unless ($is_same = _is_same_struct($original->{$_}, $samples->{$_}, $addresses));
        }
        return 1;
    } elsif (ref $samples eq 'ARRAY') {
        if (@$original == @$samples) {
            foreach (0..@$original) {
                my $is_same;
                return $is_same 
                    unless ($is_same = _is_same_struct($original->[$_], $samples->[$_], $addresses));
            }
            return 1;
        }
        return 0;
    }
    return 1;
}


#======================================================================
# получить список фронтендов из конфигурации
sub direct_frontends {
    my $hosts = YAML::XS::Load(path("$Settings::ROOT/etc/hosts")->slurp);
    my @frontends = @{$hosts->{frontends}};
    for my $host (@frontends) {
        die "Incorrect frontend host: $host" if !is_valid_domain($host);
    }
    return @frontends;
}

=head2 string2sms_time

    преобразование строки вида "09:30:21:15" в 4 числa (рабочего времени)
    my ($hour_from, $min_from, $hour_to, $min_to) = string2sms_time("09:30:21:15");

=cut

sub string2sms_time
{
    my $sms_time = shift;

    return '09', '00', '21', '00' unless $sms_time; # default
    return split ':', $sms_time;
}

#-----------------------------------------------------------

=head2 sms_time2string

    преобразование 4 чисел (рабочего времени) в строку вида "09:30:21:15"
    my $sms_time = sms_time2string($hour_from, $min_from, $hour_to, $min_to);

=cut

sub sms_time2string
{
    my ($hour_from, $min_from, $hour_to, $min_to) = @_;

    my $sms_time = '09:00:21:00'; # default if data is incorrect

    # check correct
    my $is_correct = 1;

    for ($hour_from, $min_from, $hour_to, $min_to) {
        if (! defined $_ || ! m/^\d+$/ || $_ < 0) {
            $is_correct = 0;
            last;
        }
    }

    $is_correct = 0 if($is_correct && ($hour_from > 23 || $hour_to > 24));
    $is_correct = 0 if($is_correct && $hour_to == 24 && $min_to > 0);

    if ($is_correct) {
        for ($min_from, $min_to) {
            if (! m/^(?: 0?0 | 15 | 30 | 45)$/x) {
                $is_correct = 0;
                last;
            }
        }
    }

    if ($is_correct) {
        $sms_time = sprintf "%02i:%02i:%02i:%02i", $hour_from, $min_from, $hour_to, $min_to;
    }

    return $sms_time;
}


=head2 parse_logcmd_param

    На входе -- строка param из logcmd_* 
        (get_one_field_sql(PPCLOG, "select param from logcmd_$date where reqid = ?", ...);)

    На выходе -- ссылка на хеш параметров

=cut
sub parse_logcmd_param
{
    my ($str) = @_;

    my $form = {};

    # try json first
    eval {
        $form = from_json($str);
    };

    # if not json
    if ($@) {
        $form = {};
        my @params = split /\t/, $str;
        foreach my $l (@params) {
            my ($k, $v) = split /\=/, $l;
            $form->{$k} = $v;
        }
    }

    return $form;
}

=head2 get_childpids_by_parentpid

    Сбор pid-ов по родительскому

=cut

sub get_childpids_by_parentpid {
    my $pid = shift;

    my $cmd_text = sprintf( 'ps h -o%%p --ppid=%d', $pid);
    my (@results) = yash_qx( split /\s+/, $cmd_text );

    if ( my $_pids = shift @results ) {
        return grep {$_ && int($_)>0} split(/[\r\n\s]+/, $_pids);
    }
}

=head2 check_available_cmd_by_stop_file(type, cmd)

    Функция используется для блокировки доступа к отдельным контроллерам (type, cmd)
    Где type - тип интерфейса, например, 'api'.

    Проверяет наличие специального файла $Settings::CMD_STOP_FLAG, читает содержимое
        и если переданное имя метода содержится в файле - возвращает 'blocked'

    Иначе - undef

    Возможен специальный синтаксис файла:
       type0_cmd0,type1_cmd1 - блокирует контроллер cmd1 интерфейса type1, и контроллер cmd2 интерфейса type2
       type0_* - блокирует все контроллеры интерфейса указанного типа
       all_* - блокирует все интерфейсы, использующие данный тип проверки ограничений.

=cut

sub check_available_cmd_by_stop_file($$)
{
    my ($type, $cmd) = @_;
    
    if (-f $Settings::CMD_STOP_FLAG) {
        my $t = path($Settings::CMD_STOP_FLAG)->slurp;

        my @cmds = split /\s*\,\s*/, $t;

        my %r = ();
  
         map {/([^\_]+)\_([^\_\n\r]+)/; $r{$1}{$2} = 1}
        grep {/([^\_]+)\_([^\_]+)/} 
             @cmds;

        if ($r{all}{'*'}) {
            return 'blocked';
        } elsif ($r{$type}{'*'} || $r{$type}{$cmd}) {
            return 'blocked';
        }
    }

    return undef;
}


=head2 get_order_offset_limit

    Подставить в хеш с опциями по order/offset/limit настройки по-умолчанию, и привести к удобному к применению виду

    Входные параметры:

    $opts => {
        group_by      # группировака 
        default_order # сортировка по умолчанию
        order_by      # перечень полей для сортировки
        +order_by     # инкрементируемый перечень полей для сортировки
        limit         
        offset
    }

=cut

sub get_order_offset_limit {
    my $opts = shift;

    my $default_order  = $opts->{default_order} // [];
    my $default_limit;
    my $default_offset = 0;

    my $order_by = [@$default_order]; 
    if ( defined $opts->{'order_by'} && ref $opts->{'order_by'} eq 'ARRAY' ) {
        @$order_by = @{$opts->{'order_by'}} 
    };
    unshift @$order_by, @{ $opts->{'+order_by'}  } if ( @{ $opts->{'+order_by'} || [] } );

    @$order_by = map { /^(.+)\s+(asc|desc)\s*$/i ? {field => $1, dir => $2} : {field => $_, dir => 'asc'} } @$order_by;

    my $limit  = $opts->{limit}  || $default_limit;
    my $offset = $opts->{offset} || $default_offset;

    return {order_by => $order_by,
            group_by => $opts->{group_by},
            limit => $limit,
            offset => $offset};
}

=head2 get_order_offset_limit_sql

    Подготовить строку для sql с параметрами group by order by ... limit ... offset

    Входные параметры такие же как в get_order_offset_limit

=cut

sub get_order_offset_limit_sql {
    my $opts = get_order_offset_limit(shift);

    my $order_by = @{$opts->{order_by}} ? 'order by '.join(', ', map { $_->{field} . ' ' . $_->{dir} } @{$opts->{order_by}}) : '';

    my $limit  = $opts->{limit}  ? "limit $opts->{limit}"   : '';
    my $offset = $opts->{offset} ? "offset $opts->{offset}" : '';

    my $group_by = '';
    $group_by = "group by $opts->{group_by}" if defined $opts->{group_by};

    return "$group_by $order_by $limit $offset";
}


=head2 encode_json_and_compress

    Сначала данные приводит в json формат, а затем сжимает по аналогии сжатия данных в mysql.

=cut

sub encode_json_and_compress {
    
    my $data_json = JSON->new->allow_nonref->utf8(1)->encode(shift);
    return Yandex::Compress::mysql_compress($data_json);
}

=head2 decode_json_and_uncompress

    Сначала распаковывает данные по аналогии с uncompress в mysql, а затем из полученного json создает perl-объект.

=cut
sub decode_json_and_uncompress {

    my $data_json = Yandex::Compress::mysql_uncompress(shift) || "{}";
    return JSON->new->utf8(1)->decode($data_json);

}

=head2 parse_image_name_from_url

из внешнего url картинки получить имя файла, или url если имя файла не определяется

=cut

sub parse_image_name_from_url
{
    my $url = shift;
    (my $fname = $url) =~ s!^.*/+!!;
    $fname ||= $url;
    if (length($fname) > 255) {
        $fname = substr($fname,length($fname)-255); # обрезаем имя слева до 255 символов
    }
    return $fname;
}

=head2 make_copy_sql_strings

    Помощь при составлении sql запроса, который копирует данные между таблицами.
    На входе:
        fields_to_copy - поля, которые требуется скопировать
        fields_override - данные, которые надо не копировать, а поставить специальное значение
            Указатель на хеш вида {field_name => 'my_new_value'} или вида {field_name => {$old_id1 => $new_id1, ...}}
            Во втором варианте обязательно должен быть указан $O{by}
            Если имя поля будет заканчиваться на __sql, то в значении ожидается кусок SQLя, формирующего значения для поля после копирования
        %O – опции:
            by — имя ключа, по значению которого выбирать значения для переопределённых значений, переданных в виде хешей
                ВАЖНО: если передан пустой хеш для some_field_name (внутри fields_override) - будет подставлен NULL
            override_fields_to_copy_only - если флаг задан, то перезаписываться специальными значениями из fields_override будут только поля которые есть в fields_to_copy
                ВАЖНО: без этого флага все поля из fields_override будут добавлены к полям fields_to_copy
    Возвращает:
        Две строки: строка полей через запятую ("`id`, `phone`, `country`, `city`")
                    строка значений через запятую ("'10', `phone`, `country`, `city`")

    @fields_to_copy = ('id', 'name', status, ...);
    %fields_override = (
        status => 'New',
        id => {
            $old_id1 => $new_id1,
            ...
        },
        ...
    );
    my ($fields_str, $values_str) = make_copy_sql_strings(\@fields_to_copy, \%fields_override, by => 'id');
    $fields_str => '`id`, `name`, `status`, ...';
    $values_str => "(CASE WHEN $old_id1 THEN $new_id1 ...), `name`, `status`";
    do_sql(..., "INSERT INTO table ($fields_str) SELECT $values_str FROM table WHERE ...");

=cut

sub make_copy_sql_strings {
    need_list_context();
    my ($fields_to_copy, $fields_override, %O) = @_;

    my (@fields, @values_quoted);

    if ($fields_override && %$fields_override) {
        if ($O{override_fields_to_copy_only}) {
            # подставялем специальные значения только для тех полей, которые есть в fields_to_copy
            delete @{$fields_override}{@{xminus([keys %$fields_override], [@$fields_to_copy, map { $_.'__sql'} @$fields_to_copy])}};
        }
        # не копируем переопределённые поля
        $fields_to_copy = [grep { !exists $fields_override->{$_} && !exists $fields_override->{$_.'__sql'}} @$fields_to_copy];

        for my $field (sort keys %$fields_override) {
            my $value = $fields_override->{$field};
            my $value_quoted;
            if ($value && ref($value) eq 'HASH') {
                die "HASH given as field $field override value, but there is no 'by' parameter in options" unless $O{by};
                $value_quoted = sql_case($O{by}, $value, default => undef);
            } elsif ($field =~ s/__sql$//) {
                $value_quoted = $value;
            } else {
                $value_quoted = sql_quote($value);
            }
            push @fields, $field;
            my $field_quoted = sql_quote_identifier($field);
            push @values_quoted, "$value_quoted AS $field_quoted";
        }
    }

    push @fields, @$fields_to_copy;
    push @values_quoted, map { sql_quote_identifier($_) } @$fields_to_copy;

    my $fields_str = join ', ', map {sql_quote_identifier($_)} @fields;
    my $values_str = join ', ', @values_quoted;

    return ($fields_str, $values_str);
}



=head2 force_number_recursive

Принимает один скаляр (строку/число/ссылку), 
рекурсивно превращает все, похожее на число, в настоящее число (+=0)

Меняет переданные данные И возвращает результат 
(возвращаемым значением пользоваться не рекомендуется, 
потому что это создает иллюзию того, что конструируется новая структура данных, 
а это не так). 

Полезно перед превращением структуры в json, 
чтобы числа не оказывались в кавычках

Tools::force_number_recursive($vars->{arr});
respond_text($r, to_json($vars->{arr}));

=cut
sub force_number_recursive
{
    if (ref $_[0] eq ""){
        if ( looks_like_number($_[0]) ){
            $_[0] += 0;
        }
    } elsif ( ref $_[0] eq 'ARRAY' ){
        force_number_recursive($_) for @{$_[0]};
    } elsif ( ref $_[0] eq 'HASH' ) {
        $_[0]->{force_number_recursive($_)} = force_number_recursive(delete $_[0]->{$_}) for keys %{$_[0]};
    }

    return $_[0];
}


=head2 set_yt_environment(conf)

    Установка настроек относящихся к YT из Settings в %ENV
    Существующие значения переменных окружения не перетираются

=cut

sub set_yt_environment {
    return _internal_set_yt_environment(cluster => $_[0]);
}

=head2 force_set_yt_environment(conf)

    Установка настроек относящихся к YT из Settings в %ENV
    Существующие значения переменных YT_PREFIX и YT_PROXY - заменяются всегда,
    YT_TOKEN_PATH - не затирается на is_beta(), в остальных случаях - затирается

=cut

sub force_set_yt_environment {
    return _internal_set_yt_environment(cluster => $_[0], force => 1);
}

=head2 _internal_set_yt_environment(cluster => $name, force => 1)

    Общий код для установки настроек YT из Settings в %ENV

=cut

sub _internal_set_yt_environment {
    my (%options) = @_;
    $options{cluster} //= $ENV{YT_DIRECT_CLUSTER} // 'prod';

    my $conf = $Settings::YT_CONF->{ $options{cluster} } || croak "Undefined YT cluster $options{cluster}";

    for my $param (qw/prefix token_path proxy/) {
        my $env_param = "YT_" . uc($param);
        if ($options{force} && ($param ne 'token_path' || ! EnvTools::is_beta()) || ! defined $ENV{$env_param}) {
            $ENV{$env_param} = $conf->{$param};
        }
    }

    $ENV{PATH} = $ENV{PATH} ? "$ENV{PATH}:/usr/local/bin" : '/usr/local/bin';
}

{
# $configuration => { $namespace => Yandex::Avatars=HASH(0xDEADBEEF) }
my %MDS_AVATARS_HANDLERS;

=head2 get_mds_avatars_handler($namespace [, $get_host])

    Получить объект для работы с аватарницей MDS.

=cut

sub get_mds_avatars_handler {
    my ($namespace, $get_host) = @_;

    $get_host ||= $Yandex::Avatars::MDS::GET_HOST;

    die "Missing required parameter namespace" unless $namespace;

    unless ($MDS_AVATARS_HANDLERS{$get_host}->{$namespace}) {
        $MDS_AVATARS_HANDLERS{$get_host}->{$namespace} = Yandex::Avatars::MDS->new(
            namespace => $namespace,
            get_host  => $get_host,
        );
    }

    return $MDS_AVATARS_HANDLERS{$get_host}->{$namespace};
}
}

=head2 get_clickhouse_handler( 'logs' | 'ppc' )

TODO: сделать так, чтобы параметр ReadOnly в конфиге учитывался; сейчас там декларация, которая
никак не обрабатывается. Чтобы учитывать параметр, надо поменять Yandex::Clickhouse.

=cut

sub get_clickhouse_handler {
    my ($instance) = @_;

    my $instance_config = get_db_config("ppchouse:$instance");

    require Yandex::Clickhouse;
    return Yandex::Clickhouse->new(
        host => $instance_config->{host},
        use_https => $instance_config->{ssl},
        port => $instance_config->{port},
        user => $instance_config->{user},
        password => $instance_config->{pass} // '',
        timeout => $instance_config->{timeout},
        settings => {
            database => $instance_config->{db},
            max_block_size => 1024,
        },
    );
}

=head2 trim_log_data($data, $max_len = 4096)

Функция для усечения длинных строк внутри сложных структур до нужной длины. Дописывает длину усеченной строки в конец:
Исходный параметр не модифицирует.

perl -ME -MTools -e 'p Tools::trim_log_data({ a => "12345678901234567890", b => [ "1234567890", "12345678901234567890"  ], c => [ { a => "12345678901234567890" } ] }, 15)'
$VAR1 = [
          {
            'a' => '123456789012345 [20]',
            'b' => [
                     '1234567890',
                     '123456789012345 [20]'
                   ],
            'c' => [
                     {
                       'a' => '123456789012345 [20]'
                     }
                   ]
          }
        ];

=cut

sub trim_log_data
{
    my $params = shift;
    my $max_len = shift // 70_000;
    my %special_len = (
        # AdImageRawData->RawData
        RawData => 5,
    );
    if (!ref $params || ref $params eq 'Fh') {
        my $len = length($params//'');
        if ($len > $max_len) {
            return substr($params, 0, $max_len) . " [$len]";
        }
        else {
            return $params;
        }
    }
    elsif (ref $params eq 'ARRAY') {
        return [ map { trim_log_data($_, $max_len) } @$params ];
    }
    elsif (ref $params eq 'HASH') {
        return { map { $_ => trim_log_data($params->{$_}, $special_len{$_} // $max_len) } keys %$params };
    }
    elsif (ref $params eq 'JSON::XS::Boolean') { # Special case
        return trim_log_data($params."", $max_len);
    }
    else {
        Carp::cluck "dont know how to trim '".(ref $params).q/'/;
        return $params;
    }
}

=head2 stop_the_world_for_dst_transition_hour

    Умирает во время 25-го дополнительного часа, образующегося из-за перевода времени назад в октябре 2014
    Надо использовать во всех местах, работающих со временем как с часами и минутами (lastchange и т.п.)

=cut

sub stop_the_world_for_dst_transition_hour {
    my ($log) = @_;

    my $now = unix2mysql(time);
    if ($now ge '20141026005000' && $now le '20141026021000') {
        my $msg = qq(It's DST transition hour($now), stopping);
        if ($log) {
            $log->die($msg);
        } else {
            die $msg;
        }
    }
}

=head2 get_help_url

    относительная ссылка на топик помощи.

=cut
sub get_help_url
{
    my ($id, $lang) = @_;
    $lang ||= Yandex::I18n::current_lang();
    return "/tooltip.html?id=$id&lang=$lang";
}


=head2 get_help_url_absolute

    абсолютная ссылка на топик помощи.
    домен верхнего уровня в вычисляется по языку

=cut
sub get_help_url_absolute
{
    my ($id, $lang, $tld) = @_;
    $lang ||= Yandex::I18n::current_lang();
    # Если понадобятся разные домены для разных языков -- можно $tld = $Settings::LANG_TO_TLD{$lang} || 'ru'
    $tld ||= 'ru';
    return "https://direct.yandex.$tld/tooltip.html?id=$id&lang=$lang";
}

=head2 is_turkish_domain($domain)

    Является ли домен турецким
    
=cut

sub is_turkish_domain {
    my $domain = shift;
    return $domain =~ m/yandex\.(?:com\.tr)$/;
}

=head2 get_direct_domain($host)

    Возвращает полный домен директа(тестовый, нагрузочный и т.д.)

=cut

sub get_direct_domain {

    my $host = shift;
    return get_num_level_domain(
        $host,
        get_top_level_domain($host) eq 'tr' ? 4 : 3);
}


=head2 calc_percentile($values, $percentile)

    Возвращает перцентиль по списку чисел

    $percentile = Tools::calc_percentile([65,234,742,7,12,24], 0.5); // вернет медиану

=cut

sub calc_percentile {
    my ($values, $percentile) = @_;

    die 'values must be not empty array ref' unless ref($values) eq 'ARRAY' && scalar(@$values);
    die 'percentile must be 0 <= percentile <= 1' unless 0 <= $percentile && $percentile <= 1;

    my @sorted_values = sort {$a <=> $b} @$values;
    my $percentile_index = $#sorted_values * $percentile;
    my $low_index = floor($percentile_index);
    my $high_index = ceil($percentile_index);

    return $sorted_values[int($low_index)] if $low_index == $high_index;

    my $low_value = $sorted_values[int($low_index)] * ($high_index - $percentile_index);
    my $high_value = $sorted_values[int($high_index)] * ($percentile_index - $low_index);
    return $low_value + $high_value;
}

=head2 calc_min_pay_sum

    По заданному распределению процентов для а/б эксперимента с минимальным платежом
    рассчитать нужную сумму для рублевого клиента.
    Если распределение не задано, то возвращается MIN_PAY из валютных констант.
    Распределение храните в виде массива json с полями sum - сумма минимального платежа -
    и percent - на какой процент клиентов действует сумма

=cut

sub calc_min_pay_sum {
    my ($id, $currency) = @_;

    state $prop_value = Property->new('MIN_PAY_AB');
    my $value = $prop_value->get(60);

    my $default =  get_currency_constant($currency, 'MIN_PAY');

    if ($value && $currency eq 'RUB') {
        return calc_ab($value, $default, $id);
    }

    return $default;
}

=head2 calc_payment_suggest_block

    По заданному распределению получить предлагаемый блок для нового воркфлоу оплаты

=cut

sub calc_payment_suggest_block {
    my ($id, $autopay_active, $autooverdraft_enabled, $autooverdraft_active) = @_;

    state $prop_value = Property->new('PAYMENT_SUGGEST_BLOCK_AB');
    my $value = $prop_value->get(60);

    my $default = "none";

    if ($value) {
        my $res = calc_ab($value, $default, $id);

        if (((($res eq 'autopay') || ($res eq 'autopay_opt_out') || ($res eq 'autopay_save')) && !$autopay_active)
            || (($res eq 'autooverdraft') && $autooverdraft_enabled && !$autooverdraft_active)) {
            return $res;
        }
    }

    return $default;
}

=head2 calc_ab

    Базовая функция для подсчета а/б эксперимента по переданному распределению $dist
    $dist - json массив из объектов с ключами:
        - sum - значение из набора ожидаемых для эксперимента
        - percent - на какой процент пользователей действует заданное значение

=cut

sub calc_ab {
    my ($dist, $default, $id) = @_;

    my $cur_percent = 0;
    my $id_hash_mod = bs_md5int_utf8($id) % 100;

    foreach my $pair (@{from_json($dist)}) {
        $cur_percent += $pair->{percent};

        if ($id_hash_mod < $cur_percent) {
            return $pair->{sum};
        }
    }

    return $default;
}

=head2 get_clean_text

    Возвращает строку в которой нет символов входящих в Settings::DISALLOW_BANNER_LETTER_RE
    Позаимствовано из PSGI::Base.pm::remove_disallowed_letters

=cut

sub get_clean_text {
    my $s = shift;
    return $s unless $s;
    my $DISALLOW_BANNER_LETTER_RE = qr/[^\Q$Settings::ALLOW_BANNER_LETTERS_STR@\E\r\t\n]/i;
    return ($s =~ s/$DISALLOW_BANNER_LETTER_RE//gr);
}

1;

