package URLDomain;

# $Id$

=head1 NAME
    
    URLDomain

=head1 DESCRIPTION

    Функции для проверки и разнообразной обработки url'ов из объявлений. 

    Место в иерархии модулей: "внутренний механизм Директа", т.е. 
      используется всюду, где надо
      использует другие утилиты и сервисные модули

=cut

use Direct::Modern;

use LWP::UserAgent::Zora;
use GoZoraClient;
use List::MoreUtils qw/uniq firstidx any/;
use URI::Escape qw/uri_unescape/;

use Yandex::I18n;
use Yandex::IDN;
use Yandex::HashUtils;
use Yandex::MirrorsTools::Hostings qw/strip_domain/;
use Yandex::URL;

use Settings;
use LogTools qw/
    log_zora_exception
    log_gozora_exception
    log_messages
/;
use TextTools qw/space_cleaner smartstrip/;
use Property;

use Yandex::DBTools;
use HashingTools;
use MirrorsTools;
use LWPRedirect;
use BannerTemplates;
use KnownDomains;

use JavaIntapi::ValidateEmails;


use base qw/Exporter/;
our @EXPORT = qw/
    get_filter_domain
    update_filter_domain
    validate_banner_href
    clear_banner_href
    _spec_filter_domain
    can_edit_domain
    get_url_domain
    url_domain_checksign
    url_domain_getsign
    url_domain_sign
    is_known_redirect
    divide_href_protocol
/;



our $DEBUG;


=head2 $MAX_TRACKING_REDIRECTS

Максимальное количество редиректов для трекеров мобильных приложений

=cut

our $MAX_TRACKING_REDIRECTS ||= 9;


our %VALIDATE_HREF_ERROR = (
    banner_href => {
        wrong_format    => iget_noop('Неправильный формат ссылки'),
        length_exceeded => iget_noop('Максимальная длина (%s) ссылки превышена'),
    },
    sitelink => {
        wrong_format    => iget_noop('Неправильный формат быстрой ссылки: %s'),
        length_exceeded => iget_noop('Максимальная длина (%s) быстрой ссылки превышена'),
    },
);

=head2 $USER_AGENT

    Задаёт значение заголовка User-Agent при простукивании ссылок для определения домена из функции get_url_domain.
    Значение должно быть описано в вебмастере и соответствовать общеяндексовому формату:
      http://help.yandex.ru/webmaster/?id=995329
      http://yandex.com/bots

=cut

our $USER_AGENT ||= 'Mozilla/5.0 (compatible; YaDirectFetcher/1.0; Dyatel)';

=head2 $TIMEOUT

    Таймаут при простукивании ссылок для определения домена из функции get_url_domain.

=cut

our $TIMEOUT ||= 7;
our $ZORA_TIMEOUT_CODE = 1031;
our $GOZORA_TIMEOUT_CODE = 1006;

sub get_filter_domain {
    my $domain = shift;

    return '' if ! defined $domain || $domain eq '';
    $domain = Yandex::IDN::idn_to_ascii($domain);

    my $mirror = MirrorsTools->new(
        dont_load_file => 1,
        use_db => 1,
    );

    return $mirror->domain_filter($domain);
}

my $user_params_re = join "|", map {quotemeta} qw/CampaignID BannerID AdID AdGroupID param1 param2/;

# проверка ссылки на доступность, получение домена после редиректов
{
my $log;
sub get_url_domain {
    my $url = shift||"";
    my $opt = shift||{};

    if ($DEBUG) {
        require Yandex::Log;
        require JSON;
        $log ||= Yandex::Log->new(log_file_name => 'get_url_domain', msg_prefix => "[$$]");
        $log->out("Call: $url, " . JSON::to_json($opt, {allow_blessed=>1}));
    }
    

    $url = clear_banner_href($url);

    $url = Yandex::IDN::idn_to_ascii($url);

    my ($template_used, $params_used);

    # TEMPLATE_METKA
    $template_used += $url =~ s/$TEMPLATE_METKA/$1/ig  if !$opt->{no_template};

    # users's and url's params
    if (!$opt->{no_params}) {
        $params_used += $url =~ s/(?:({(?:source|source_type|position_type|position|keyword|addphrases|region_name)}))+/test/gi;
        $params_used += $url =~ s/(?:({(?:region_id)}))+/123/gi;
        $params_used += $url =~ s/{device_type}/desktop/i;

        $params_used += $url =~ s/\{(?:$user_params_re)\}//gi;
    }

    my $result;

    state $gozora_prop = Property->new('use_gozora_in_process_images_queue');
    my $use_gozora = $gozora_prop->get(300);

    my $ua_class = $use_gozora ? 'GoZoraClient' : 'LWP::UserAgent::Zora';
    my $ua = $opt->{ua} || $ua_class->new(
        env_proxy => 0, 
        agent => $USER_AGENT, 
        # Это таймаут "сколько Zora должна ждать сайта". Ниже по коду есть другой таймаут со смыслом "сколько мы ждем Зору"
        zora_timeout => $TIMEOUT,
    );
    $ua->protocols_allowed(\@Yandex::URL::VALID_PROTOCOLS);
    $ua->default_headers->push_header('Accept' => "*/*");
    $ua->default_headers->push_header('Accept-Encoding' => "gzip, deflate");
    $ua->default_headers->push_header('Connection' => "close");
    #$ua->max_size(10);
    $ua->cookie_jar({});

    my $is_tracking_href = !is_valid_tracking_href($url);
    my $is_known_redirect = is_known_redirect($url);
    my $max_redirect_num = $is_tracking_href || $is_known_redirect ? $MAX_TRACKING_REDIRECTS : $LWPRedirect::MAX_REDIRECTS_NUM;

    my $chain;
    eval {
        my $redirect_info = LWPRedirect::get_redirect_chain(
                        $url,
                        ua => $ua,
                        # timeout -- это таймаут "сколько мы ждем Зору"
                        # Должен быть чуть больше, чем zora_timeout выше по коду, чтобы Зора успевала ответить "я не дожидаюсь сайта"
                        # в таком случае мы можем различать торможение сайта и Зоры
                        # чтобы воспроизвести сценарий "Zora тормозит" для тестирования, можно передавать $TIMEOUT - 2 или любое другое значение меньше $TIMEOUT
                        timeout => $TIMEOUT + 2,
                        predefined_redirect => \&predefined_redirect,
                        is_known_redirect => $opt->{check_for_http_status} ? undef : \&is_known_redirect,
                        max_redirect_num => $max_redirect_num,
        );
        $chain = $redirect_info->{chain};
        1;
    }
    # если упали из-за того, что зора не работает - считаем урл валидным!
    or do {
        if ($use_gozora) {
            die $@  if !ref $@ || ref $@ ne 'GoZoraException';
            $log->out("Got GoZoraException: $@->{response}")  if $DEBUG;
            log_gozora_exception($@);
        } else {
            die $@  if !ref $@ || ref $@ ne 'ZoraException';
            $log->out("Got ZoraException: $@->{response}")  if $DEBUG;
            log_zora_exception($@);
        }
        $chain = [{ url => $url, request => undef, response => undef }];
    };

    my $resp = $chain->[-1]->{response};

    $result = {
        redirect_chain => $chain,
    };

    my $is_error;
    if ($use_gozora) {
        $is_error = $resp && $resp->is_error;
    } else {
        $is_error = $resp && !($resp->header("X-Yandex-Orig-Http-Code") && $resp->header("X-Yandex-Orig-Http-Code") == 200);
    }
    if ($resp && $resp->status_line =~ /read timeout/ && $resp->request->{proxy} =~ /\Q$Settings::SSRF_PROXY\E/){
        # Особый случай: запрос делали через Zora (proxy), но не дождались никакого ответа
        # считаем, что это проблема с Zora, а сайт в порядке
        # если тормозит сам сайт -- выполнение должно попадать в is_error с статусом 'timeout spider response'
        @$chain = ({ url => $url, request => undef, response => undef });
        $result->{warn} = iget("Не удалось проверить адрес");
        $is_error = 0;
    }

    if ( ($resp && $resp->header("X-Yandex-Gozora-Error-Code") && $resp->header("X-Yandex-Gozora-Error-Code") == $GOZORA_TIMEOUT_CODE)
        || ($resp && $resp->header("X-Yandex-Http-Code") && $resp->header("X-Yandex-Http-Code") == $ZORA_TIMEOUT_CODE)) {
        # Zora ответила, что не дождалась сайта
        hash_merge $result, {
            res => 0,
            msg => iget("Ваш сайт не ответил в течение семи секунд"),
        };
    } elsif ( $is_error ) {
        my $status_line = $resp->status_line;
        if ( $status_line =~ /^\d+ \s+ zora:/xms ) {
            if ( $resp->header("X-Yandex-Status") ) {
                $status_line = $resp->code . q{ } . $resp->header("X-Yandex-Status");
            } else {
                $status_line =~ s/zora:.*/internal spider error/xms;
            }
        }

        utf8::decode($status_line) if !utf8::is_utf8($status_line);

        if (!$opt->{no_stderr}) {
            print STDERR "get_url_domain: incorrect url '$url' - ".$status_line."\n";
        }
        if ( $status_line =~ /read timeout/ ) {
            # запрос был без Zora, и ответа не дождались
            hash_merge $result, {
                res => 0, 
                msg => iget("Ваш сайт не ответил в течение семи секунд"),
            };
        } elsif ($status_line =~ /Line too long/) {
            hash_merge $result, {
                res => 0, 
                msg => iget("Размер заголовка ответа сервера больше допустимого"),
            };
        } elsif ( $status_line =~ /limit/ ) {
            # https://st.yandex-team.ru/DIRECT-68227#1500392655000
            my $err_field = $is_known_redirect ? 'warn' : 'msg';
            hash_merge $result, {
                res => $is_known_redirect ? 1 : 0,
                $err_field => iget("В цепочке редиректов более %d URL", $max_redirect_num),
            };
        } else {
            my $subst_warning;
            # XXX Можно записать компактнее, но что тогда делать с iget?
            if ($params_used && $template_used) {
                $subst_warning = iget("Пожалуйста, проверьте корректность подстановки шаблона и параметров в ссылку на сайт");
            } elsif ($template_used) {
                $subst_warning = iget("Пожалуйста, проверьте корректность подстановки шаблона в ссылку на сайт");
            } elsif ($params_used) {
                $subst_warning = iget("Пожалуйста, проверьте корректность подстановки параметров в ссылку на сайт");
            }
            hash_merge $result, {
                res => 0,
                msg => iget("Ваш сервер вернул ошибку: ") . $status_line . ($subst_warning ? ". $subst_warning" : ""),
            };
        }
    } else {
        my $last_url = $chain->[-1]->{url};
        my @reasons = validate_banner_href($last_url, hash_merge $opt, {allow_long_url => 1});

        unless ( @reasons ) {
            my $domain_redir = get_host($last_url); # это пойдет в banners.domain_redir
            my $redirect_result_href = trusted_redirect_chain(map {$_->{url}} @$chain); 
            my $domain_redir_show = get_host($redirect_result_href); # что показывать в баннере

            # проверка для специальных доменов
            if (my $filter_domain = _spec_filter_domain($url)) {
                hash_merge $result, {
                    res => 1,
                    msg => Yandex::IDN::idn_to_unicode(lc($filter_domain)),
                    domain_redir => Yandex::IDN::idn_to_unicode(lc($filter_domain)),
                    redirect_result_href => Yandex::IDN::idn_to_unicode($url),
                };
            }
            else {
                hash_merge $result, {
                    res => 1,
                    msg => Yandex::IDN::idn_to_unicode(lc( $domain_redir_show )),
                    domain_redir => Yandex::IDN::idn_to_unicode(lc($domain_redir)),
                    redirect_result_href => Yandex::IDN::idn_to_unicode($redirect_result_href),
                };
            }

        } else {
            hash_merge $result, {
                res => 0, 
                msg => join("\n", @reasons),
            };
        }
    }

    if ($opt->{check_for_http_status}) {
        if ($opt->{strict_redirect}) {
            my @urls = map { $_->{url} } @$chain;
            for my $i (0 .. $#urls - 1) {
                if ( !is_trusted_redirect(get_host($urls[$i]), get_host($urls[$i+1]), allow_counters => 0) ) {
                    hash_merge $result, {
                        res => 0,
                        msg => iget("Ссылка баннера не может содержать редирект на другой ресурс"),
                    };
                }
            }
        }
        else {
            my $unknown_redirect_idx = firstidx { !is_known_redirect($_->{url}) } @$chain;
            if ( $unknown_redirect_idx >= 0 ) {
                my $item = $chain->[$unknown_redirect_idx];
                if ( $result->{res} == 1 ) {
                    hash_merge $result, {
                        domain_redir => Yandex::IDN::idn_to_unicode(get_host($item->{url})),
                        msg => Yandex::IDN::idn_to_unicode(get_host($item->{url})),
                        redirect_result_href => Yandex::IDN::idn_to_unicode($item->{url}),
                    };
                }
            }
        }
    }

    $log->out("Result: " . JSON::to_json($result, {allow_blessed=>1})) if $DEBUG;
    return $result;
}
}

=head2 predefined_redirect($url)

    Для некоторых счётчиков мы можем определить редирект просто по урлу, не делая запрос

=cut
sub predefined_redirect
{
    my ($url) = @_;
    return undef if !defined $url;
    my $redir;
    if ($url =~ m!^(?:https?://)?pixel.everesttech.net/!) {
        if ($url =~ /[;&?]url=\!(.*)/) {
            # если начинается с ! - урл незаэскейплен
            $redir = $1;
        } elsif ($url =~ /[;&?]url=([^&;#]+)/) {
            $redir = uri_unescape($1);
        }
    }

    return $redir && $redir =~ m!^(?:https?://)! && Yandex::IDN::is_valid_domain(get_host($redir)) ? $redir : undef;
}

=head2 is_trusted_redirect

Проверяем, является ли редирект "хорошим" или "плохим"
Хорошими считаем:
 * Редиректы внутри одного домена (sub1.example.com => sub2.example.com), сюда же попадают редиректы на/с www.
 ** Исключение составляют публичные домены (ya.ru, org.ru etc.), перечисленные в Yandex/MirrorsTools/Hostings.pm
 * Редиректы через известные счетчики/сокращалки: tinyurl.com, goo.gl, click0[12].begun.ru etc.

Все остальные считаем плохими

Именованые параметры:
    allow_counters => 0|1 - считать ли редирект на счетчик хорошим. По-умолчанию 1

=cut

sub is_trusted_redirect
{
    my ($domain, $href, %opt) = @_;
    $opt{allow_counters} //= 1;

    return 0 if !$domain || !$href;
    $href = get_host($href);
    return 0 unless $href;
    $domain = lc $domain;
    if ($opt{allow_counters}) {
        return 1 if is_known_redirect($href);
    }
    my $shost   = strip_domain($href);
    my $sdomain = strip_domain($domain);
    
    return 1 if $shost eq $sdomain;
    return 0;
}

=head2 is_known_redirect($url)

 Принадлежит ли ссылка к списку известных сокращалок/счетчиков
 Если на входе строка - возвращает boolean
 Если на входе ссылка на массив - возвращает хеш { domain => 1/0 }

=cut

sub is_known_redirect
{
    my ($href) = @_;

    my @hrefs = uniq grep { defined $_ && length $_ } (ref($href) eq 'ARRAY' ? @$href : $href);
    my $domains = [ map { get_host($_), get_second_level_domain($_); } @hrefs ];
    my $found_domains = KnownDomains::get_domains_by_type('counter', from_list => $domains);
    my %is_domain_found = map {($_ => 1)} @$found_domains;

    my $res;

    if (ref $href eq 'ARRAY') {
        $res = { map { get_host($_) => $is_domain_found{get_host($_)} || $is_domain_found{get_second_level_domain($_)} ? 1 : 0 } @hrefs };
    } else {
        $res = @$found_domains ? 1 : 0;
    }

    return $res;
}

=head2 trusted_redirect_chain(@urls)

Проверяет цепочку редиректов
Проверить редирект (goo.gl => ... => domain.com) еще недостаточно, т.к. это может быть цепочка,
одно из звеньев которой является зловредным.
Возвращает последний хороший домен, который и будет показан в объявлении

На вход хочет массив строк (url)

=cut

sub trusted_redirect_chain
{
    my @chain = @_;
    my @domain_chain = map {get_host($_)} @chain;
    return $chain[0] if @chain == 1;

    for my $i (0 .. $#chain - 1) {
        if (!is_trusted_redirect($domain_chain[$i+1], $domain_chain[$i])) {
            return $chain[$i];
        }
    }
    return $chain[-1];
}

=head2 _get_domain_re($allow_ip)

    Возвращаем регулярное выражение для проверки домена

=cut

sub _get_domain_re
{
    my $allow_ip = shift;
    my @domain_res = ();
    push @domain_res, qq/(?:[${Settings::ALLOWED_ALPHABET_LETTERS}\\d\\-\\_]+ \\.)+ [${Settings::ALLOWED_ALPHABET_LETTERS}][${Settings::ALLOWED_ALPHABET_LETTERS}\\d]{1,62}/;
    push @domain_res, q/(?:[12]?\d{1,2}\.){3}(?:[12]?\d{1,2})/ if $allow_ip;

    return join '|', @domain_res;
}

sub _get_protocol_re
{
    return "(?:".join('|', @Yandex::URL::VALID_PROTOCOLS).")\:\/\/";
}


=head2 validate_banner_href

    проверка ссылки
    пустые ссылки и undef допустимы (требуется при создании кампании, хотя и странно)

    my @errors = validate_banner_href('http://ya.ru/', {%options}, $context);
    %options:
      MAX_URL_LENGTH - свой предел на длину урла
      allow_ip       - допустим ip вместо хоста
      allow_long_url - может быть любой длины
      allow_ftp      - допустимы ftp:// ссылки

    $context - в каком месте проверяем (влияет только на текст ошибок):
      "banner_href" - в баннере (по умолчанию)
      "sitelink"    - в сайтлинках

=cut 

sub validate_banner_href
{
    my $url = shift;
    my $opt = shift||{};
    my $context = shift || 'banner_href';

    $url = defined $url ? $url : '';
    my $url_src = $url;
    my $max_url_length = $opt->{MAX_URL_LENGTH} || $MAX_URL_LENGTH;
    my @result;

    if (length($url) > 0) {
        $url = Yandex::IDN::idn_to_unicode($url);

        local @Yandex::URL::VALID_PROTOCOLS = @Yandex::URL::VALID_PROTOCOLS;
        push @Yandex::URL::VALID_PROTOCOLS, 'ftp' if $opt->{allow_ftp};
        my $protocol_re = _get_protocol_re();
        my $domain_re = _get_domain_re($opt->{allow_ip});

        push @result, {error => 'wrong_format', params => [$url_src]} if $url =~ m/^ (?:$domain_re) $TEMPLATE_METKA/x;
        push @result, {error => 'wrong_format', params => [$url_src]} if $url !~ m/^ (?: $protocol_re )
                                                                         (?:$domain_re)
                                                                         (?: : \d{1,5} )?
                                                                         (?: [\/?#] [\[\]\-\w\/=&%#?():;.,~+{}^\@\$\*\!\|\']* )? \z/x;

        push @result, {error => 'wrong_format', params => [$url_src]} if _check_for_domain_templates($url);
        push @result, {error => 'length_exceeded', params => [$max_url_length]} if length($url) > $max_url_length && !$opt->{allow_long_url};

        # проверка на длину поддомена
        push @result, {error => 'wrong_format', params => [$url_src]} if ! $opt->{allow_ip} && ! Yandex::IDN::is_valid_domain(get_host($url));
    }
    return uniq
           map { iget($VALIDATE_HREF_ERROR{$context}->{$_->{error}}, @{$_->{params} || []}) }
           @result;
}

=head3 _check_for_domain_templates

Проверить, нет ли подстановочных параметров в доменной части у ссылки.
    Параметры обозначаются либо парой {}, либо парой ##. Доменная часть должна заканчиваться '/' или '?', иногда '#'.
    Также замечает единичные '#', '{', '}' до '/' или '?'.

    Возвращает правду, если обнаружились подобные аномалии

=cut

sub _check_for_domain_templates
{
    my $href = shift;
    $href = strip_protocol($href);
    if ($href =~ /^([^\/\?]*)[\/\?]/) {
        $href = $1;
        return $href =~ /[#{}]/;
    }
    return $href =~ /[{}]/ || (($href =~ tr/#//) > 1);
}


=head3 _spec_filter_domain(href)
    
    Функция возвращает домен третьего уровня для определенных владельцев
    Например, googlepages.com
    Домены выше третьего уровня обрезает до 3-его.
    
    Пример: 
        http://example.googlepages.com -> example.googlepages.com
    
=cut

sub _spec_filter_domain
{
    my $href = lc(shift) || return;
    my $res = $href =~ /(?:[a-z]+\:\/\/)?([^\/^\?]+)[\/\?]?/ ? $1 : $href;
    
    my @part =  reverse split(/\./, $res);
    my $regexp = qr/googlepages.com/;
    
    if ($res =~ /\.($regexp)$/) {
        $#part = 2 if $#part > 1;
    } else {
        return;
    }
    
    return join ('.', reverse @part);
}

# подписать домен
sub url_domain_sign {
    my $vars = shift;
    my $href = clear_banner_href($vars->{href}, $vars->{url_protocol});
    return md5_hex_utf8(join(':', map { $_ // '' } $href, $vars->{domain}, $Settings::SECRET_PHRASE));
}


# проверить подпись домена
sub url_domain_checksign {
    my $vars = shift;
    return 0 if !defined $vars->{domain_sign};
    if ($vars->{domain_redir}) {
        return 0 if ($vars->{domain_redir_sign} // '') ne url_domain_sign({href => $vars->{href}, url_protocol => $vars->{url_protocol}, domain => $vars->{domain_redir}});
    }
    return $vars->{domain_sign} eq url_domain_sign($vars);
}

=head2 clear_banner_href

    Причесывает ссылку. Если она была без протокола, то добавляет его.

=cut
sub clear_banner_href {
    my ($href, $protocol) = @_;
    return '' unless $href;
    $href = smartstrip(space_cleaner($href));
    $protocol = get_protocol($href) unless $protocol;
    return clear_protocol($protocol).strip_protocol($href);
}

=head2 divide_href_protocol

    Раздеряет ссылку на протокол и ссылку без протокола. Используется в контроллерах для отобращения в формах редактирования.

=cut
sub divide_href_protocol($;$) {
    my ($href, $prefix) = @_;
    $prefix ||= '';
    return {"${prefix}url_protocol" => get_protocol($href), "${prefix}href" => strip_protocol($href)};


}

=head2 can_edit_domain(login_rights)

     Имеет ли пользователь права на редактирование поля domain у banner

=cut

sub can_edit_domain{
    my ( $login_rights ) = @_;
    return ( $login_rights->{super_control} || $login_rights->{support_control} || $login_rights->{placer_control} || $login_rights->{manager_control} )
}

# Получить домен и подписать его
sub url_domain_getsign {
    my ( $vars, $login_rights ) = @_;
    my $err;
    my $domain_re = _get_domain_re();
    if ( $vars->{domain} && $vars->{domain_redir} && url_domain_checksign($vars) ) {
        # all ok, no error
    } elsif (
        $vars->{domain} && $vars->{domain} =~ /^$domain_re$/ix 
        && can_edit_domain($login_rights)
    ) {
        $vars->{domain} = lc $vars->{domain};
        $vars->{domain_sign} = url_domain_sign( $vars );
        $vars->{domain_redir} = $vars->{domain};
        $vars->{domain_redir_sign} = $vars->{domain_sign};
        # no error
    } else {
        my $check_params = { check_for_http_status => 1 };
        $check_params->{no_template} = $check_params->{no_params} = 1  if $vars->{adgroup_type} eq 'mobile_content';
        my $result_check = get_url_domain($vars->{href}, $check_params);
        my ($res, $text, $redir) = @{$result_check} {qw/res msg domain_redir/};
        if ( !$res ) {
            $err = $text;
        } else {
            $vars->{domain} = $text;
            $vars->{domain_sign} = url_domain_sign( $vars );
            $vars->{domain_redir} = $redir;
            $vars->{domain_redir_sign} = url_domain_sign({href => $vars->{href}, url_protocol => $vars->{url_protocol}, domain => $redir});
            # no error
        }
    }
    if (!$err) {
        my $filter_domain = get_filter_domain($vars->{domain});
        if (my $act = get_one_field_sql(PPCDICT, "SELECT action 
                                                    FROM bad_domains 
                                                   WHERE filter_domain = ?", $filter_domain)
        ) {
            if ($act eq 'deny') {
                $err = iget("При размещении объявлений на домен %s было нарушено пользовательское соглашение (https://yandex.ru/legal/rules/)", $vars->{domain});
            }
            log_messages("bad_domain", ($act eq 'deny' ? "Попытка создать" : "Создали")." объявление, рекламирующее $filter_domain в кампании $vars->{cid}");
        }
    }
    return $err;
}

=head2 update_filter_domain

  update filter_domain table

  update_filter_domain(bid, "domain.org");
  Номер баннера нужен для определения шарда, в котором будет обновлена таблица

=cut

sub update_filter_domain
{
    my ($bid, $domain) = @_;
    
    return if ! defined $domain || $domain eq '';

    my $filter_domain = get_filter_domain($domain);
    die "Filter domain for '$domain' is not defined" if !defined $filter_domain;

    return 0 if $domain eq $filter_domain;

    return do_sql(PPC(bid => $bid),
                "insert into filter_domain (domain, filter_domain)
                         values (?, ?)
                         on duplicate key update
                         filter_domain = values(filter_domain)", $domain, $filter_domain);

}

# --------------------------------------------------------------------

=head2 ajax_check_url

Перенесено из контролера cmd_ajaxCheckUrl

На входе хеш с параметрами
    url
    url_protocol
    is_tracking_href

=cut

sub ajax_check_url {
    my ($url_data) = @_;
    my ($url, $url_protocol) = @{$url_data}{qw/url url_protocol/};

    # Запоминаем исходный URL для отправки клиенту (клиент сравнивает его с кэшированным значением)
    my $form_url = $url;
    $url = clear_banner_href($url, $url_protocol);

    my @errors = validate_banner_href($url);

    my $error_type;
    my $is_invalid_tracking_href;
    if (!@errors && $url_data->{is_tracking_href} && (my $defect = is_valid_tracking_href($url))) {
        $is_invalid_tracking_href = 1;
        $error_type = $defect->suffix;
        push @errors, $defect->description;
    } else {
        $error_type = 'IncorrectUrl';
    }

    my $res = @errors ? 0 : 1;
    my $text = join(', ', @errors);
    my $redir = '';
    my $warn;
    my $has_redirect;

    if (! @errors) {
        my $opt = { check_for_http_status => 1 };
        if ($url_data->{is_tracking_href}) {
            $opt->{no_params} = 1;
            $opt->{no_template} = 1;
        }
        my $result = get_url_domain($url, $opt);
        ($res, $text, $redir, $warn) = @{$result} {qw/res msg domain_redir warn/};

        my $redirect_chain = $result->{redirect_chain};
        if (@$redirect_chain > 1) {
            $has_redirect = 1;

            #
            # Попытаемся определить, что это редирект в пределах одного общего домена
            # Например:
            #   http://www.mail.ru -> http://mail.ru -> https://mail.ru
            #   http://domain.com -> http://www.domain.com
            #
            my $redirect_host = _get_idn_host($redirect_chain->[0]->{redirect});
            for (my $i = 1; $i < @$redirect_chain; $i++) {
                my $chain = $redirect_chain->[$i];

                if ($i < (@$redirect_chain - 1)) {
                    my $checking_host = _get_idn_host($chain->{redirect});
                    last unless $checking_host && $redirect_host =~ /\Q$checking_host\E$/ || $checking_host =~ /\Q$redirect_host\E$/;
                } else {
                    my $orig_host = _get_idn_host($url);
                    $has_redirect = 2 if !$chain->{redirect} && ($redirect_host =~ /\Q$orig_host\E$/ || $orig_host =~ /\Q$redirect_host\E$/);
                }
            }
        }
    }

    my $res_hash = {
        url  => $form_url,
        code => $res,
    };

    $res_hash->{is_valid_tracking_href} = !$is_invalid_tracking_href  if $url_data->{is_tracking_href};

    if ($res) {
        $res_hash->{domain} = $text;
        $res_hash->{domain_sign} = url_domain_sign({href => $url, domain => $text});
        $res_hash->{domain_redir} = $redir;
        $res_hash->{domain_redir_sign} = url_domain_sign({href => $url, domain => $redir});
        $res_hash->{has_redirect} = $has_redirect;
        $res_hash->{warn} = $warn if defined $warn;
    } else {
        $res_hash->{text} = $text;
        $res_hash->{error_type} = $error_type;
    }

    return $res_hash;
}

sub _get_idn_host {
    my ($url) = @_;
    return Yandex::IDN::idn_to_unicode(get_host($url));
}


=head2 is_valid_tracking_href

Проверяет трекинговую ссылку на валидность. 

=cut

sub is_valid_tracking_href {
    my ($href) = @_;

    my $result = KnownDomains::is_known_href(mobile_app_counter => $href);
    if ($result == 1) {
        $result = KnownDomains::is_known_href(mobile_app_impression_counter => $href);
    }
    # Если is_known_href вернул 1, то проверяем домен второго уровня
    return $KnownDomains::TRACKING_HREF_ERRORS{$result}->() if (!$result || $result == 2);

    my $second_level_domain = get_second_level_domain($href);
    $result = KnownDomains::is_known_href(mobile_app_counter => $second_level_domain);
    if ($result == 1) {
        $result = KnownDomains::is_known_href(mobile_app_impression_counter => $second_level_domain);
    }
    return $KnownDomains::TRACKING_HREF_ERRORS{$result}->();
}

=head2 is_valid_email_java

Проверяет валидность email-адреса по мнению org.apache.commons.validator.routines.EmailValidator и нашего самописного валидатора
Вход:
    строка с email
Выход:
    1, если валиден, 0, если нет

=cut

sub is_valid_email_java {
    my $email = shift;

    # мы не умеем писать на ЖКХ@yandex.ru, отсеим
    if (!Yandex::IDN::is_valid_email($email)) {
        return 0;
    }

    my $validation_result = JavaIntapi::ValidateEmails->new(items => [{email => $email}])->call();
    if (!exists $validation_result->{$email}) {
        die("Failed to validate a email $email, couldn't find it the response");
    }
    return $validation_result->{$email} ? 1 : 0;
}

1;
