package Units;

use strict;
use warnings;
use utf8;

=pod

=head1 NAME

    Units - класс предназначен для определения суммы потраченных и доступных для
    списания баллов, текущего лимита для клиента, списания баллов.

=head1 SYNOPSIS

    use Units;
    use Units::Storage::Memcached;
    use Yandex::Memcached;

    my $units = Units->new({
        storage => Units::Storage::Memcached->new(
            memcached => Yandex::Memcached->new(servers => 'localhost:11211')
        ),
        client_id => 12345,
    });

    if ( $units->is_available( 500 ) ) {
        ...
    }

    $units->withdraw( 500 );

=head1 DESCRIPTION

    Концепция баллов введена для ограничения количества и сложности запросов
    от клиентов. Каждой операции поставлено в соответствие определенное количество
    баллов (в зависимости от количества необходимых для ее выполнения ресурсов
    Direct-а), которое будет списано при ее выполнении. Каждому клиенту назначается
    определённое количество баллов, которые он может израсходовать в рамках
    скользящего окна, выполняя какие-либо операции. Скользящее окно - это набор
    последовательных временных интервалов равной длины. Интервал является
    минимальной единицей измерения времени, за которое учитываются израсходованные
    клиентом баллы. При достижении максимально разрешенного количества баллов в
    рамках такого скользящего окна клиенту разрешается делать только некоторые
    типы запросов.

=cut

use API::Settings;
use Mouse;

=head1 VARIABLES

=head2 DEFAULT_BASKET_NAME

    Название корзины, операции с баллами из которой будут производятся.
    По умолчанию 'API'.

=cut

our $DEFAULT_BASKET_NAME //= 'API';

=head2 DEFAULT_LIMIT

    Лимит баллов, которые клиент может потратить в рамках скользящего окна. По
    умолчанию равен 40000. (взято отсюда https://st.yandex-team.ru/DIRECT-39913)

=cut

our $DEFAULT_LIMIT //= $API::Settings::UNITS_BY_SPENT->[0]->{units};

=head2 DEFAULT_INTERVAL_SIZE

    Размер интервала в секундах. По умолчанию равен 1 часу.

=cut

our $DEFAULT_INTERVAL_SIZE //= 60 * 60;

=head2 DEFAULT_INTERVAL_SIZE

    Количество интервалов, составляющих скользящее окно. По умолчанию равно 24.

=cut

our $DEFAULT_INTERVALS_NUMBER //= 24;

=head1 METHOD

=head2 new

    Конструктор

    Параметры:
        storage       - ссылка на объект, реализующий интерфейс Units::Storage, обязательный
        client_id     - id клиента, число, обязательный
        basket_name   - название корзины, строка
        interval_size - количество интервалов в скользящем окне, число
        intervals     - длительность интервала в секундах, число
        limit         - лимит баллов в рамках скользящего окна, число

    Результат:
        Ссылка на объект класса Units

=head2 storage

    Текущей backend хранилища баллов

    Параметры:
        нет

    Результат:
        Ссылка на объект, реализующий интерфейс Units::Storage

=cut

has storage => ( is => 'ro', isa => 'Units::Storage', required => 1 );

=head2 client_id

    Id клиента, операции с баллами которого производятся

    Параметры:
        нет

    Результат:
        Число

=cut

has client_id => ( is => 'ro', isa => 'Int', required => 1 );

=head2 basket_name

    Название корзины, операции с баллами из которой производятся

    Параметры:
        нет

    Результат:
        Строка

=cut

has basket_name => ( is => 'ro', isa => 'Str', default => $DEFAULT_BASKET_NAME );

=head2 interval_size

    Размер интервала в секундах

    Параметры:
        нет

    Результат:
        Число

=cut

has interval_size => ( is => 'ro', isa => 'Int', default => $DEFAULT_INTERVAL_SIZE );

=head2 intervals

    Количество интервалов, составляющих скользящее окно

    Параметры:
        нет

    Результат:
        Число

=cut

has intervals => ( is => 'ro', isa => 'Int', default => $DEFAULT_INTERVALS_NUMBER );

=head2 limit

    Количество баллов, которое разрешено израсходовать в рамках скользящего окна.

    Параметры:
        нет

    Результат:
        Число

=cut

has limit => ( is => 'ro', isa => 'Int', lazy => 1, default => sub {
    $DEFAULT_LIMIT;
} );

=head2 ttl

    Время жизни в хранилище ключа, идентифицирующего количество потраченных
    определенным клиентом в определенном интервале баллов

    Параметры:
        нет

    Результат:
        Число

=cut

has ttl => ( is => 'ro', isa => 'Int', default => sub { $_[0]->intervals * $_[0]->interval_size }, lazy => 1 );


=head2 key_format

    sprintf-like формат для генерации имен ключей, аргументы такие:
    - ClientID, число
    - basket_name, строка
    - interval, число

=cut
has key_format => ( is => 'ro', isa => 'Str', required => 1 );


=head2 _interval_key

    Получение ключа для интервала

    Параметры:
        Число, интервал, обязательный

    Результат:
        Ключ, строка

=cut

sub _interval_key {
    my ( $self, $interval ) = @_;

    return sprintf $self->key_format => $self->client_id, $self->basket_name, $interval;
}

=head2 _intervals_to_keys

    Возвращает ключи для интервалов

    Параметры:
        $intervals - массив интервалов

    Результат:
        Ссылка на хеш вида { интервал => ключ, ... }

=cut

sub _intervals_to_keys {
    my ( $self, $intervals ) = @_;

    my %result;
    foreach my $interval ( @$intervals ) {
        $result{ $interval } = $self->_interval_key( $interval );
    }

    return \%result;
}

=head2 _get_interval

    Получение интервала по timestamp-у

    Параметры:
        Число, время в формате unix timestamp, обязательный

    Результат:
        Интервал, число

=cut

sub _get_interval {
    my ( $self, $ts ) = @_;

    return int( $ts / $self->interval_size ) * $self->interval_size; # round up current timestamp
}

=head2 _get_sliding_window_intervals

    Возвращает интервалы, составляющие скользящее окно

    Параметры:
        нет

    Результат:
        Ссылка на массив интервалов

=cut

sub _get_sliding_window_intervals {
    my $self = shift;

    my @intervals;
    my $interval = $self->current_interval;
    foreach ( 1 .. $self->intervals ) {
        push @intervals, $interval;
        $interval -= $self->interval_size;
    }

    return \@intervals;
}

=head2 _get_sliding_window_keys

    Возвращает ключи интервалов, составляющих скользящее окно

    Параметры:
        нет

    Результат:
        Ссылка на массив ключей

=cut

sub _get_sliding_window_keys {
    my $self = shift;

    my $intervals = $self->_get_sliding_window_intervals;

    my @keys;
    foreach my $interval ( @$intervals ) {
        push @keys, $self->_interval_key( $interval );
    }

    return \@keys;
}

=head2 _get_spent_by_intervals

    Возвращает израсходованные в рамках окна баллы в разбивке по интервалам

    Параметры:
        нет

    Результат:
        ссылка на хеш вида { интервал => сумма_израсходованных_баллов, ... }

=cut
sub _get_spent_by_intervals {
    my $self = shift;

    my $intervals = $self->_get_sliding_window_intervals();

    my @keys;
    my %interval_by_key;
    foreach my $interval ( @$intervals ) {
        my $key = $self->_interval_key( $interval );
        push @keys, $key;
        $interval_by_key{ $key } = $interval;
    }

    my $spent = $self->storage->get_multi( \@keys );

    return { map { $interval_by_key{ $_ } => $spent->{ $_ } } @keys };
}

=head2 _restore_balance

    Удаляет данные о использованных в рамках скользящего окна баллах, а также
    ключ со значение суточного лимита на пользователя, провоцируя его повторное
    вычисление

    Параметры:
        нет

    Результат:
        bool

=cut

sub _restore_balance {
    my $self = shift;

    my $keys = $self->_get_sliding_window_keys();

    return $self->storage->delete_multi( $keys );
}

=head2 current_interval

    Идентификатор текущего интервала

    Параметры:
        нет

    Результат:
        Число

=cut

sub current_interval {
    my $self = shift;

    return $self->_get_interval( time );
}

=head2 spent

    Количество баллов, израсходованное в рамках скользящего окна (т.е. за текущий
    и предыдущие intervals() - 1 интервалов)

    Параметры:
        нет

    Результат:
        Число

=cut

sub spent {
    my $self = shift;

    my $spent_by_interval = $self->_get_spent_by_intervals();

    my $sum = 0;
    foreach my $amount ( values %$spent_by_interval ) {
        $sum += $amount || 0;
    }

    return $sum;
}

=head2 balance

    Количество баллов, которые разрешено израсходовать в текущем интервале до
    достижения лимита

    Параметры:
        нет

    Результат:
        Число

=cut

sub balance {
    my $self = shift;

    my $spent_by_interval = $self->_get_spent_by_intervals();

    my $per_interval = $self->limit / ( $self->intervals || 1 );

    my $balance = 0;
    foreach my $interval ( sort keys %$spent_by_interval ) {
        my $spent = $spent_by_interval->{ $interval } || 0;
        $balance += $per_interval - $spent;
        $balance = ( $balance > 0 ) ? $balance : 0; # write off per interval debts
    }

    # round up if needed
    if ( int($balance) != $balance ) {
        $balance = int($balance) + 1;
    }

    return $balance;
}

=head2 is_available

    Проверяет - разрешено ли израсходовать указанное количество баллов в рамках скользящего окна

    Параметры:
        количество баллов

    Результат:
        1 | 0

=cut

sub is_available {
    my ( $self, $amount ) = @_;

    $amount ||= 0;

    return $self->balance >= $amount ? 1 : 0;
}

=head2 withdraw

    Увеличивает счетчик потраченных в текущем интервале баллов на указанную величину

    Параметры:
        количество баллов

    Результат:
        1 | 0

=cut

sub withdraw {
    my ( $self, $amount ) = @_;

    $amount ||= 0;

    my $key = $self->_interval_key( $self->current_interval );

    my $res = $self->storage->incr_expire( $key, $amount, $self->ttl );

    return $res ? 1 : 0;
}

__PACKAGE__->meta->make_immutable();

1;
