
package User;

# $Id$

=head1 NAME

    User

=head1 DESCRIPTION

    Работа с сущностью "пользователь"

=cut

use Direct::Modern;

use YAML;
use Carp qw/longmess/;
use Yandex::ListUtils qw/xisect/;
use Yandex::HashUtils;
use Yandex::ScalarUtils;
use Yandex::SendMail qw/send_alert/;
use Yandex::Passport;
use Yandex::I18n;
use Yandex::TVM2;

use Direct::Feature;
use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Balance;
use TextTools;
use ShardingTools;
use Tools;
use Primitives;
use PrimitivesIds;
use Client;

use Rbac qw/:const/;
use RBACDirect;

use geo_regions;

use List::Util qw/first/;
use List::MoreUtils qw/any notall uniq/;

use JavaIntapi::CheckPhoneVerified;

use base qw/Exporter/;
our @EXPORT = qw/
    is_user_exists
    create_update_user
    create_user_email
    get_user_data
    get_users_data
    get_one_user_field

    get_user_options
    mass_get_user_options
    set_user_options
    update_user_options

    get_user_typed_options
    mass_get_user_typed_options
    update_user_typed_options
    set_user_typed_options

    filter_geo_mcb_users
    get_valid_emails
    get_users_campaigns_emails
    sorting_campaigns

    is_bad_passport_karma
    delete_user_from_db
/;

use utf8;

=head2 %USER_TABLES

    Хэш с именами таблиц с пользовательскими данными и списком полей в каждой.

=cut

our %USER_TABLES = (

        users =>
                { order => 1,
                  fields => [ qw /
                        ClientID
                        rep_type
                        email
                        valid
                        LastChange
                        fio
                        phone
                        verified_phone_id
                        sendNews
                        sendWarn
                        createtime
                        login
                        hidden
                        sendAccNews
                        not_resident
                        statusArch
                        statusBlocked
                        description
                        lang
                        captcha_freq
                        allowed_ips
                        statusYandexAdv
                        showOnYandexOnly
                    /],
                 },

        users_options =>
                { order => 2,
                  fields => [ qw /
                        ya_counters
                        statusPostmoderate
                        manager_office_id
                        geo_id
                        sendAgencyMcbLetters
                        sendAgencyDirectLetters
                        options
                        show_fa_teaser
                        tags_allowed
                        sendClientLetters
                        sendClientSMS
                        use_camp_description
                        manager_use_crm
                        agency_email_to_chief
                        passport_karma
                        opts
                        recommendations_email
                    /],
                  set_fields => {
                      opts => [qw/
                        fold_infoblock
                        notify_about_new_domains
                    /],
                },
                },

        users_api_options =>
                { order => 3,
                  fields => [ qw /
                        advq_queries_lim
                        api_offer
                        api_units_daily
                        allow_create_subclients
                        api_allowed_ips
                        api_allow_finance_operations
                        api_send_mail_notifications
                        api_geo_allowed
                        api_allow_old_versions
                        api_developer_name
                        api_developer_email
                        upload_adgroup_xls
                        excel_rows_limit
                    /],
                },

        internal_users =>
                { order => 4,
                  fields => [ qw /
                        domain_login
                        manager_private_email
                        is_developer
                        is_super_manager
                    /],
                },
        users_agency =>
                { order => 5,
                  fields => [ qw /
                        show_agency_contacts
                        is_no_pay
                        disallow_money_transfer
                        lim_rep_type
                        group_id
                    /],
                },
        users_captcha =>
                { order => 6,
                  fields => [ qw /
                        captcha_expires
                        is_captcha_amnested
                    /],
                },
);
{
    # в каждой таблице создаём служебное поле _fields_hash
    # ключ - имя аттибута,
    # значение - undef, если поле самостоятельное, или имя set-поля
    for my $table (keys %USER_TABLES) {
        my $data = $USER_TABLES{$table};
        $data->{_fields_hash} = {
            map {$_ => undef} @{$data->{fields}}
        };
        if ($data->{set_fields}) {
            for my $sf (keys %{$data->{set_fields}}) {
                hash_merge $data->{_fields_hash}, {map {$_ => $sf} @{$data->{set_fields}->{$sf}}};
            }
        }
    }
}

# максимальный размер сохраняемых настроек пользователя в users_options.options
our $USER_OPTIONS_MAX_SIZE_BYTES = 2**16-1;
# максимальный размер сохраняемых типизированных настроек пользователя в users_typed_options.options_json
our $USER_TYPED_OPTIONS_MAX_SIZE_BYTES = 2**24-1;


=head2 SPECIAL_MANAGER_UID

Есть некие агентства, которые находятся на каком-то определенном статусе и называются "самоходными агентствами"
Эти агентства ведутся только небольшим списком менеджеров

=cut

our %SPECIAL_MANAGER_UID = (
    self_agency => {qw/
        41329487    artem-zateev
        89550209    dushkin-vit
        226541728   yndx-infedorova
        395174907   yndx-kleimenova-manager-1
        173179219   yndx-kozhanova-manager
        282597286   yndx-kozlikin
        242059493   yndx-mlugar
        386870987   yndx-m-salina-manager
        330055775   yndx-shtan15
        398821980   yndx-yazhevika
        222344052   yndx-tonymineev
        247053843   yndx-popovskikh
    /},
    self_agency_ua => {qw/
        176441802   yndx-andsv-manager
        312612842   yndx-perishkova-manager
    /},
    self_agency_tr => {qw/
        277371208   yndx-sedletskaya-manager
    /},
    self_agency_cis => {qw/
        166169808   yndx-kolesnicova
        321056438   yndx-vecher-manager
    /},
);


=head2 get_special_manager_type

Является ли uid менеджером "самоходного агентства".
Возвращаек код типа или undef

=cut

sub get_special_manager_type {
    my ($uid) = @_;

    state $special_manager_type = +{
        map { my $t = $_;  map {($_ => $t)} keys %{$SPECIAL_MANAGER_UID{$t}} }
        keys %SPECIAL_MANAGER_UID
    };

    return $special_manager_type->{$uid};
}


=head2

    Проверяет существование пользователя в БД

=cut

sub is_user_exists {
    my $uid = shift;
    return $uid && defined get_shard(uid => $uid) ? 1 : 0;
}

=head2 create_update_user

    Создает или апдейтит поля пользователя в таблицах users, users_options, internal_users, users_api_options
    Параметры позиционные:
        uid
        data - хэш опций

    Схема работы:
        1) получает логин по uid из паспорта и пытается добавить пользователя;
        2) обновляет users.LastChange, если нет users_data, но есть users_options_data и/или internal_users_options_data;
        3) пробегается по переданным параметрам и обновляет соответствующие таблицы.

    Замечание 1: Если хэш data не передан, будет создан пользователь на основе данных паспорта.
    Замечание 2: Обновляются только реально изменившиеся поля
    Замечание 3: Если создается новый клиент в Директе, неотзя передавать $data->{ClientID}, даже если он известен из Баланса
                 иначе мы не отправим в Баланс страну и валюту.

=cut


sub create_update_user($;$) {

    my ($uid, $data) = @_;

    $uid && $uid =~ m /^\d+$/ || die "create_update_user: uid is not specified or incorrect";

    if (! $data) {
        $data = {};
    }

    my $old_data = get_user_data($uid, [keys %$data, 'ClientID']);
    my $is_user_client_exists = $data->{ClientID} || $old_data->{ClientID};
    die "Client cannot be created without UID" if (! $is_user_client_exists && ! $data->{UID});

    # проверяем наличие пользователя в паспорте
    my $passport = get_info_by_uid_passport($uid);
    die "create_update_user: No such user" if !$passport;

    my $fio = $data->{fio} || $data->{FIO} || $passport->{fio};
    my $email = $data->{email} || $passport->{email} || "$passport->{login}\@yandex.ru";
    my $verified_phone_id = $data->{verified_phone_id};

    my $balance_new_client_data = hash_cut $data, qw/
        phone ClientID role subrole initial_country initial_currency initial_subregion agency is_touch tin tin_type
    /;
    $balance_new_client_data->{FIO} = $fio;
    $balance_new_client_data->{email} = $email;
    $balance_new_client_data->{gdpr_agreement_accepted_time} = $data->{gdpr_agreement_accepted_time} if $data->{gdpr_agreement_accepted_time};

    my $user_new = hash_cut($data, qw/phone ClientID rep_type/);

    # вытаскиваем из паспорта недостающие данные
    # заменяем в логине точку на минус
    my $login = first { defined $_ } ($data->{login}, $passport->{login}, $data->{_login});
    $login = normalize_login($login);

    $data->{domain_login} = normalize_login($data->{domain_login}) if $data->{domain_login};

    smartstrip($login);
    $data->{login}     = $login if $data->{login};

    $user_new->{uid} = $uid;
    $user_new->{login} = $login;
    $user_new->{fio} = $fio;
    $user_new->{email} = $email;
    $user_new->{verified_phone_id} = $verified_phone_id;
    $user_new->{createtime__dont_quote}='UNIX_TIMESTAMP(now())';

    smartstrip($user_new->{email});

    if ($verified_phone_id) {
        my $is_verified = JavaIntapi::CheckPhoneVerified
            ->new(uid => $uid, phoneId => $verified_phone_id)
            ->call();

        die iget('Пожалуйста, подтвердите телефон'), "\n" unless $is_verified;
    }

    if (! defined $login || $login eq '') {
        # если $login нет, значит, в $data->{login} его нет и в $data->{_login} его нет и в $passport его тоже нет
        # в $data его может не быть, потому что функцию так вызвали, а вот отсутствие в ответе чёрного ящика
        # скорее всего значит аномальную ситуацию, так что данные оттуда будут полезны в логе
        require JSON;
        my $passport_data = JSON::to_json($passport);
        die("create_update_user(): login for $uid does not exist; passport data: $passport_data");
    }

    if (! $is_user_client_exists || $data->{initial_country} || $data->{initial_currency} || $data->{initial_subregion}) {
        $user_new->{ClientID} = create_client_in_balance($data->{UID}, $uid, %$balance_new_client_data) or die "create_client_in_balance error for UID:$data->{UID}, uid:$uid";
    } elsif (!%{$old_data} && $data->{ClientID} && $data->{UID} && !$data->{not_create_client_id_association}) {
        create_client_id_association($uid, $data->{ClientID}, $data->{UID}) or die "create_client_id_association error for UID:$data->{UID}, uid:$uid";
    }

    my $client_id = $user_new->{ClientID} || $old_data->{ClientID};
    # Смена ClientID через обновление пользователя - запрещена и должна выпоняться отдельно
    if ($old_data->{ClientID} && $old_data->{ClientID} != $client_id) {
        my $msg = q/Changing User's ClientID via create_update_user is PROHIBITED/;
        send_alert(Carp::longmess($msg), 'attempting to change ClientID via create_update_user');
        die $msg;
    }

    # если нам не указали тип представителя - пытаемся его вычислить
    if (!$old_data->{ClientID} && !$user_new->{rep_type}) {
        my $reps = get_uids(ClientID => $client_id);
        $user_new->{rep_type} = @$reps ? 'main' : 'chief';
    }

    if ($old_data->{rep_type} && $user_new->{rep_type} && $old_data->{rep_type} ne $user_new->{rep_type}) { # меняем тип представителя
        if ($old_data->{rep_type} eq $REP_MAIN && $user_new->{rep_type} eq $REP_READONLY) {
            remove_client_id_association($data->{UID}, $uid, $old_data->{ClientID}) or die "remove_client_id_association($data->{UID}, $uid, $old_data->{ClientID}) failed";
        } elsif ($old_data->{rep_type} eq $REP_READONLY && $user_new->{rep_type} eq $REP_MAIN) {
            create_client_id_association($uid, $old_data->{ClientID}, $data->{UID}) or die "create_client_id_association($uid, $old_data->{ClientID}, $data->{UID}) failed";
        }
    }

    # те же данные в метабазу кладутся в Client::create_client_in_balance
    # по идее, здесь оно никогда не должно срабатывать
    my $shard = get_shard(ClientID => $client_id);
    if (! defined $shard ) {
        $shard = get_new_available_shard($client_id, $data->{UID});
        save_shard(ClientID => $client_id, shard => $shard);
    }

    # создается представитель уже существующему
    # (есть ClientID - по нему определился шард, но нет данных про пользователя - поэтому нет $old_data)
    if ( !$old_data || !$old_data->{ClientID} ) {
        # при слиянии клиентов обновлять метабазу следует отдельно
        save_shard(uid => $uid, ClientID => $client_id);
        save_shard(login => $login, uid => $uid);
    }

    # пытаемся создать нового пользователя, если не получается, игнорируем неудачу - значит, есть уже пользователь
    do_insert_into_table(PPC(uid => $uid), 'users'
                             , $user_new
                             , ignore => 1
                        );

    $data->{geo_id} = $data->{initial_subregion} if defined $data->{initial_subregion};

    # фильтруем не изменившиеся поля
    $data = hash_kgrep {
                defined $old_data->{$_} != defined $data->{$_}
                || str($old_data->{$_}) ne str($data->{$_})
            } $data;

    my $user = hash_copy({}, $data, @{$USER_TABLES{users}{fields}});

    smartstrip($user->{email}) if $user && $user->{email};

    # если есть users_data обновляем таблицу users, если нет, обновляем поле LastChange
    if(scalar keys %{$user}){
        do_update_table(PPC(uid => $uid), 'users', $user, where => {uid => $uid});
    } elsif (scalar keys %{$data}) {
        do_sql(PPC(uid => $uid), 'update users set LastChange=NOW() where uid = ?', $uid);
    }

    # обновляем таблицы
    foreach ( grep {$_ ne 'users'} keys %USER_TABLES ) {
        _update_user_table($uid, $_, $data);
    }

}

=head2 create_user_email($uid)

    Возвращает email пользователя из Паспорта.
    Если у пользователя нет почты, то создает ящик.

    $user_email = create_user_email($uid);
    # $user_email = 'aaa@bbb.ccc';

=cut

sub create_user_email($)
{
    my ($uid) = shift;
    my $ticket = eval { Yandex::TVM2::get_ticket($Settings::PASSPORT_TVM2_ID) } or die "Cannot get ticket for $Settings::PASSPORT_TVM2_ID: $@";
    Yandex::Passport::subscribe_service($uid, 'mail', tvm_ticket => $ticket);
    my $passport_info = get_info_by_uid_passport($uid);
    return $passport_info->{email};
}

=head2 get_user_data($uid, \@fields)

    Возвращает хэш информации по пользователю (поля перечисленные в @fields)

    $user_data = get_user_data($uid, [qw/ClientID not_resident/]);
    # $user_data => { ClientID => 12345, not_resident => 0 };

=cut

sub get_user_data($$) {
    my ($uid, $fields) = @_;

    return {} if ! scalar @$fields;
    $uid && $uid =~ m/^[0-9]+$/ or die "get_user_data: uid is not specified or incorrect";

    my $res = get_users_data([$uid], $fields);
    return $res->{$uid};
}

=head2 get_users_data(\@uids, \@fields)

    Возвращает хэш информации по пользователям (поля перечисленные в @fields)

    $user_data = get_users_data($uids, [qw/ClientID not_resident/]);
    # $user_data => { 123123 => {ClientID => 12345, not_resident => 0} };

=cut

sub get_users_data($$)
{
    my ($uids, $fields) = @_;

    return {} if ! scalar @$fields;

    my (@tables, @clean_fields, @sql_fields, %sets2fields, $uid_required);

    foreach my $table (keys %USER_TABLES) {
        my $_fh = $USER_TABLES{$table}->{_fields_hash};
        my @searched_fields;
        for my $f (@$fields) {
            if (exists $_fh->{$f}) {
                push @searched_fields, $f;
            } elsif ($f eq 'uid') {
                $uid_required = 1;
            }
        }
        if (@searched_fields) {
            push @tables, $table;
            push @clean_fields, @searched_fields;
            push @sql_fields, uniq map {sql_quote_identifier("$table.".($_fh->{$_} // $_))} @searched_fields;
            for my $f (grep {defined $_fh->{$_}} @searched_fields) {
                push @{$sets2fields{$_fh->{$f}}}, $f;
            }
        }
    }

    if (scalar @clean_fields) {
        # чтобы гарантировать, что выберутся все существующие поля, в случае если надо обработать более 1 таблицы,
        # джойним users и сортируем таблицы для join в порядке, определенном в USER_TABLES
        push @tables, 'users' if @tables > 1 && (! any {$_ eq 'users'} @tables);
        @tables = sort {$USER_TABLES{$a}{order} <=> $USER_TABLES{$b}{order}} @tables;
        my $tables_sql = join(' ', $tables[0], map {"left join $_ using (uid)"} @tables[1..$#tables]);

        my $fields_sql = join(',', @sql_fields);
        my $result = get_hashes_hash_sql(PPC(uid => $uids),
            [ "select uid, $fields_sql from $tables_sql", where => {uid =>  SHARD_IDS} ]);

        my %clean_fields_set = map {$_ => 1} @clean_fields;
        foreach my $uid (@$uids) {
            # если пользователя нет - текущее поведение хэш с undef-ами. нехорошо.
            my $data = $result->{$uid} //= {map {$_ => undef} @clean_fields};
            # превращаем set-поля в реальные поля
            for my $f (keys %$data) {
                if (exists $sets2fields{$f}) {
                    my %set_data = map {$_ => 1} split /,/, $data->{$f}//'';
                    for my $sf (@{$sets2fields{$f}}) {
                        $data->{$sf} = $set_data{$sf};
                    }
                }
                delete $data->{$f} if !$clean_fields_set{$f} && !($uid_required && $f eq 'uid');
            }
        }
        return $result;
    } else {
        warn 'get_user_data: unknown fields requested ('.join(', ', @$fields).')';
    }

    return {};
}

=head2 get_one_user_field($uid, $field)

    Возвращает один параметр пользователя

=cut

sub get_one_user_field($$)
{
    my ($uid, $field) = @_;
    my $data = get_user_data($uid, [$field]);
    return $data->{$field}
}

=head2 sorting_campaigns($uid, \%FORM)

    Параметры сортировки кампаний пользователя.
    Если параметров сотрировки нет вернутся ранее сохраненные параметры.

=cut

sub sorting_campaigns {

    my ($uid, $request) = @_;

    my %sort;
    if (exists $request->{sort}) {
        %sort = map {$_ => $request->{$_}} qw/sort reverse settings_sorted/;
        update_user_options($uid, {campaigns_sort => \%sort})
    } elsif ((my $option = get_user_options($uid))) {
        my $ref = ref $option->{campaigns_sort};
        %sort = %{$option->{campaigns_sort}} if $ref && $ref eq 'HASH';
    }

    return \%sort;
}

=head2 mass_get_user_options

Для указаных uid вернуть хеш { $uid => { различные пользовательские настройки } }

=cut

sub mass_get_user_options
{
    my $uids = shift;
    my $raw_options = get_hash_sql(PPC(uid => $uids), ["select uid, options from users_options", where => { uid => SHARD_IDS }]);
    my $options = hash_map { parse_user_options($_) } $raw_options;
    return $options;
}


=head2 parse_user_options($str)

    парсинг и переформатирование сериализованных user_options

=cut
sub parse_user_options {
    my ($opt) = @_;
    if ($opt) {
        $opt = YAML::Load($opt);
        # тизер про переход в реальную валюту прячем не навсегда
        # если время истекло, то возвращаем его в умолчальное состояние (открыт)
        if ($opt->{reset_multicurrency_teaser_at} && $opt->{reset_multicurrency_teaser_at} < time()) {
            delete $opt->{multicurrency_teaser};
        }
        delete $opt->{reset_multicurrency_teaser_at};
    } else {
        $opt = {};
    }
    return $opt;
}


=head2 get_user_options

    Возвращает ссылку на хеш с дополнительными параметрами пользователя
    (в основном, различные настройки для интерфейса)

    $user_options = get_user_options($uid);
    $user_options => {
        campaigns_sort => {
            reverse => 1,
            settings_sorted => undef,
            sort => 'clicks',
        },
        phones_teaser => 'link',
        show_favorite_campaigns_only => 'false',
        show_my_campaigns_only => 'false',
        show_wide_money_transfer => 'false',
        stat_periods => '2012-07-01:2012-07-31,2010-12-01:2012-08-01,2011-01-01:2012-07-15',
        multicurrency_teaser => 'link'|'div',
    };

=cut

sub get_user_options {
    my ($uid) = @_;
    return mass_get_user_options($uid)->{$uid} || {};
}

sub set_user_options
{
    my ($uid, $option) = @_;
    $option = defined $option ? YAML::Dump($option) : undef;

    # предотвращаем сохранение в БД битого YAML (если он не помещается в БД)
    use bytes;
    if (length($option) > $USER_OPTIONS_MAX_SIZE_BYTES) {
        die "length of user options exceeded limits: uid=$uid";
    }
    no bytes;

    create_update_user( $uid,  { options => $option } );
}

=head2 update_user_options

  обновляем только заданные поля в users.user_options
  my $new_user_options = update_user_options($uid, {stat_periods => "2010-10-11:2010-11-11;2010-11-11:2010-12-11"});

=cut

sub update_user_options
{
    my ($uid, $new_users_options) = @_;

    my $users_options = get_user_options($uid);
    $users_options = hash_merge $users_options, $new_users_options;
    set_user_options($uid, $users_options);

    return $users_options;
}

=head2 get_user_typed_options

    Возвращает ссылку на хеш с дополнительными параметрами пользователя определенного типа

    $user_typed_options = get_user_typed_options($uid, 'stat_report_master');

=cut

sub get_user_typed_options {
    my ($uid, $type) = @_;

    return mass_get_user_typed_options($uid, $type)->{$uid} || {};
}

=head2 mass_get_user_typed_options

Для указаных uid вернуть хеш { $uid => { параметры пользователя соответствующего типа } }

=cut

sub mass_get_user_typed_options
{
    my ($uids, $type) = @_;
    my $raw_options = get_hash_sql(PPC(uid => $uids), ["select uid, options_json from users_typed_options",
                                                         where => { uid => SHARD_IDS, type => $type }]);
    my $options = hash_map { $_ ? Tools::decode_json_and_uncompress($_) : {} } $raw_options;
    return $options;
}

=head2 set_user_typed_options

Сохраняет опции пользователя заданного типа

=cut

sub set_user_typed_options
{
    my ($uid, $type, $option) = @_;
    die "Options type is not defined" unless $type;

    if (defined $option) {
        my $options_json_compressed = Tools::encode_json_and_compress($option);
        # предотвращаем сохранение в БД битых данных (если они не помещаются в БД)
        use bytes;
        if (length($options_json_compressed) > $USER_TYPED_OPTIONS_MAX_SIZE_BYTES) {
            die "length of user typed options exceeded limits: uid=$uid, type=$type";
        }
        no bytes;

        do_insert_into_table(PPC(uid => $uid), 'users_typed_options'
                             , {uid => $uid,
                                type => $type,
                                options_json => $options_json_compressed}
                             , on_duplicate_key_update => 1
                             , key => [qw/uid type/]);
    } else {
        do_delete_from_table(PPC(uid => $uid), 'users_typed_options', where => {uid => $uid, type => $type});
    }
}

=head2 update_user_typed_options

  обновляем только заданные опции по типу, в users_typed_options.options
  my $new_user_typed_options = update_user_typed_options($uid, 'stat_report_master', {stat_periods => "2010-10-11:2010-11-11;2010-11-11:2010-12-11"});

=cut

sub update_user_typed_options
{
    my ($uid, $type, $new_user_typed_options) = @_;
    die "Options type is not defined" unless $type;

    my $user_typed_options = get_user_typed_options($uid, $type);
    $user_typed_options = hash_merge $user_typed_options, $new_user_typed_options;
    set_user_typed_options($uid, $type, $user_typed_options);

    return $user_typed_options;
}

=head2 _update_user_table

    Обновляет или добавляет запись в одну из таблиц с пользовательскими данными,
    если есть что обновлять

=cut

sub _update_user_table
{
    my ($uid, $table, $data) = @_;

    my $table_data = {};
    my $_fh = $USER_TABLES{$table}{_fields_hash};
    for my $f (grep {exists $_fh->{$_}} keys %$data) {
        if (defined $_fh->{$f}) {
            # поле типа set
            $table_data->{"$_fh->{$f}__smod"}->{$f} = $data->{$f};
        } else {
            $table_data->{$f} = $data->{$f};
        }
    }
    $table_data->{uid} = $uid;
    return if scalar(keys(%$table_data)) <= 1; # не вставляем запись если есть только uid

    do_insert_into_table(PPC(uid => $uid), $table
                             , $table_data
                             , on_duplicate_key_update => 1
                             , key => 'uid');
}

=head2 filter_geo_mcb_users

    Фильтрует пользователей Директа от Геоконтекста и Баяна

    На вход - ссылка на массив uid'ов
    На выход - ссылка на массив uid без гео- и мкб-пользователей

=cut

sub filter_geo_mcb_users
{
    my $uids = shift;

    return [] unless $uids && scalar @$uids;

    my $geo_users = get_hashes_hash_sql(PPC(uid => $uids),
                    [ "select uid,
                              IFNULL(SUM( IF(type in ('geo', 'mcb'), 1, 0) ), 0) as geo_mcb_campcount,
                              count(cid) as campcount
                       from campaigns",
                       where => {uid => SHARD_IDS},
                       "group by uid"
                    ] );

    return [ grep { !$geo_users->{$_} || !$geo_users->{$_}{geo_mcb_campcount} || $geo_users->{$_}{campcount} > $geo_users->{$_}{geo_mcb_campcount} } @$uids ];
}

=head2 get_valid_emails

    Получает email ы пользователя из кампаний и базы

=cut

sub get_valid_emails
{
    my ( $uid, $email ) = @_;

    $email = '' unless defined $email;

    my (@arr, %hs);

    # predefined
    $hs{$email} = 1 if $email;

    my $user_shard = get_shard(uid => $uid);
    if ($user_shard) {
        # Пользователь существует
        my $emails = get_users_campaigns_emails([$uid])->{$uid};

        if (ref($emails) eq 'ARRAY' && @$emails) {
            foreach (@$emails) {
                $hs{$_} = 1;
            }
        }

        my $user_email = get_one_field_sql(PPC(shard => $user_shard), [ "select email from users", where => { 'uid' => $uid, 'valid' => 2}]);

        if (defined $user_email) {
            $hs{$user_email} = 1;
        }
    }

    for (keys %hs) {
        push(@arr, {email => $_, select => $email eq $_ ? 'SELECTED' : ''});
    }

    return \@arr;
}

=head2 get_users_campaigns_emails

    Возвращает email' ы пользователей из их кампаний
    Входные параметры: ссылка на массив uid
    Выходные параметры:
        {
            uid1 => ['email1@ya.ru', 'email2@ya.ru'],
            uid2 => ['email3@ya.ru', 'email4@ya.ru'],
        }

=cut

sub get_users_campaigns_emails
{
    my $uids = shift;

    return {} unless $uids && @$uids;

    my $result = get_all_sql(PPC(uid => $uids), [ "select c.uid as uid, email from camp_options o
                                       join campaigns c on c.cid = o.cid",
                                      where => { 'c.uid' => SHARD_IDS, 'valid' => 2, 'o.email__ne' => ''} ]);

    my $emails = {};

    foreach my $row (@$result) {
        smartstrip($row->{email});
        $emails->{$row->{uid}}{$row->{email}} = 1;
    }

    return {map {$_ => [keys %{$emails->{$_}}]} keys %$emails};
}

=head2 is_bad_passport_karma

    Возвращает спам карму пользователя из паспорта, если она сохранена в базе,
    в противном случае ходит за ней в паспорт и сохраняет в базе значение

=cut

sub is_bad_passport_karma
{
    my ($uid, $u_karma) = @_;

    my $karma = defined $u_karma ? $u_karma : get_one_user_field($uid, 'passport_karma');

    if (! defined $karma) {
        # если сохраненной кармы еще нет - то запрашиваем ее
        $karma = get_info_by_uid_passport($uid)->{karma} || 0;
        create_update_user($uid, {passport_karma => $karma});
    }

    return $karma >= $Settings::KARMA_API_AUTOBLOCK;
}

# --------------------------------------------------------------------

=head2 delete_user_from_db

Удаление логина из всех таблиц, важно: предпологаем, что все права проверены, и что логин "чистый" (удален из rbac и биллинга)

    my $deleted_rows = delete_user_from_db($uid);
    my $deleted_rows = delete_user_from_db([$uid1, $uid2, ...]);

=cut

our @TABLES_TO_CLEAN_BY_CLIENTID = qw(
    account_score
    api_special_user_options
    client_brands
    client_currency_changes
    client_discounts
    client_domains
    client_domains_stripped
    client_firm_country_currency
    client_limits
    client_nds
    clients_api_options
    clients_autoban
    clients_custom_options
    clients_geo_ip
    clients_options
    clients_stat
    clients_to_fetch_nds
    clients_to_force_multicurrency_teaser
    client_reminder_letters
    currency_convert_money_correspondence
    currency_convert_queue
    force_currency_convert
    infoblock_teasers_factors
    minus_words
    mobile_content
    retargeting_conditions
    clients
    client_teaser_data_lastupdate
);

sub delete_user_from_db($) {
    my $uid = shift;

    die "delete_user_from_db() require uid" unless $uid;

    my $client_id = get_clientid(uid => $uid);
    my $login = get_login(uid => $uid);

    my $rows_count = 0;
    do_in_transaction {
        for my $users_table_name (keys %USER_TABLES) {
            $rows_count += do_delete_from_table(PPC(uid => $uid), $users_table_name, where => {uid => $uid});
        }

        if (!get_one_field_sql(PPC(ClientID => $client_id), "SELECT 1 FROM users WHERE ClientID = ? LIMIT 1",$client_id)) {
            for my $table (@TABLES_TO_CLEAN_BY_CLIENTID) {
                $rows_count += do_delete_from_table(PPC(ClientID => $client_id), $table, where => {ClientID => SHARD_IDS});
            }
        }

        delete_shard(uid => $uid);
        if ($login && get_shard(login => $login, 'uid') == $uid) {
            delete_shard(login => $login);
        }
    };

    return $rows_count;
}

sub get_actual_for_show_teaser {
    my $teasers = shift;

    return 'multicurrency' if $teasers->{multicurrency};
    return 'android' if $teasers->{android};
    return 'recommendations_email' if $teasers->{recommendations_email};

#    Пока не используется, но просили оставить, в любой момент может тизер вернуться
#    return 'personal-manager' if $teasers->{personal-manager};

    # Из cpa_optimization и first_aid выбираем случайный.
    my @r = keys %{hash_cut $teasers, qw/cpa-optimization first-aid/};
    return $r[int(rand(scalar(@r)))];
}

=head2 client_has_discount

    my $has_discount = client_has_discount(ClientID => 123);

Проверяет есть ли у клиента скидка.

=cut
sub client_has_discount {
    my ($key, $id) = @_;
    die "'$key' search not supported" unless $key eq 'ClientID';
    my $client_id = $id;
    my $client_discount = get_client_discount($client_id);
    return ($client_discount && $client_discount > 0);
}

=head2 is_turkish_client($client_id, %options)

    Является ли клиент турецким пользователем (турецким плательщиком)

    Параметры:
        $client_id - id плательщика
        %options - domain => '' - если страна плательщика не известна,
                                определяем принадлежность пользователя по текущему домену
                                (не обязательный параметр)

=cut

sub is_turkish_client {

    my ($client_id, %options) = @_;

    my $client = $client_id ? get_client_data($client_id, [qw/country_region_id work_currency/]) : undef;
    # для ряда пользователей $country_region_id не определен (== 0)
    # и информации о некоторых клиентах может не быть
    return $client && $client->{country_region_id}
        ? $client->{country_region_id} == $geo_regions::TR
        : $client && $client->{work_currency}
            ? $client->{work_currency} eq 'TRY'
            : $options{domain}
                ? Tools::is_turkish_domain($options{domain})
                : 0;
}

=head2 get_agency_primary_manager($AgencyUID)

Поиск менеджера который ведет агенство в директе (текстовые кампании)

Параметры:
    $AgencyUID - uid агенства

Результат:
    Информация о менеджере агенства
    {login => '', email => '', FIO => ''}
    или undef если в качестве $AgencyUID передали не агенство

=cut

sub get_agency_primary_manager {

    my $AgencyUID = shift;

    my $primary_manager_uid = get_one_field_sql(
        PPC(uid => $AgencyUID),
        "SELECT c.primary_manager_uid FROM clients c JOIN users u using(ClientID) WHERE u.uid = ?", $AgencyUID
    );
    my $manager;
    if ($primary_manager_uid) {
        $manager = get_user_data($primary_manager_uid, [qw/login email FIO/]);
        $manager->{special_type} = get_special_manager_type($primary_manager_uid);
    }
    return $manager;
}


state $_FCH_COUNTRIES = [$geo_regions::RUS, $geo_regions::BY, $geo_regions::KAZ, $geo_regions::UZB];

=head2 is_fc_help_allowed

    Возвращает флаг доступен/недоступен пользователю показ тизеров помощи в настройке первой кампании

=cut

sub is_fc_help_allowed {
    my ($rbac, $uid, $country) = @_;

    return
        (any {$country == $_} @$_FCH_COUNTRIES)
        && !RBACDirect::rbac_has_agency($rbac, $uid);
}

=head2 set_suggest_service_flags

    Выставляет флаги показа тизеров "помощь яндекса"/"помощь не-яндекса",
    в зависимости от наличия соответствующих фич у клиента

=cut

sub set_suggest_service_flags {
    my ($client_id, $dataset) = @_;

    $dataset->{suggest_yandex_service} = Client::ClientFeatures::has_feature($client_id => 'suggest_yandex_service');
    $dataset->{suggest_non_yandex_service} = Client::ClientFeatures::has_feature($client_id => 'suggest_non_yandex_service');

    return $dataset;
}

=head2 _does_user_have_feature($uid, $feature_name)

    Спрашивает у java-intapi, есть ли у $uid доступ к фиче $feature_name.
    Нужна для случаев, когда ClientID пока не создан.

    ВАЖНО! Функция не кеширует результат похода к java-intapi, как это сделано в
    Client::ClientFeatures::allowed_features
    Если у функции появятся ещё пользователи, нужно ещё раз подумать, нужно ли кеширование.

=cut
sub _does_user_have_feature {
    my ($uid, $feature_name) = @_;

    my $features_resp = Direct::Feature::get_clients_uids_features(undef, [$uid]);

    if ($features_resp->{error}) {
        utf8::decode($features_resp->{error}) unless utf8::is_utf8($features_resp->{error});

        die sprintf('java intapi error: uid %s, %s', $uid, $features_resp->{error});
    }
    my $features = $features_resp->{per_uid_features}->{$uid} || [];
    my $found = first { $_ eq $feature_name } @$features;

    return !!$found;
}

=head2 has_touch_direct_feature

    Есть ли у пользователя доступ к фиче touch_direct_enabled.

    Пока у пользователя нет ClientID в Директе, можно получить фичу,
    только если она включена на процент (и процент вычисляется от хеша от uid)
    А когда у пользователя уже есть ClientID в Директе, вычисляется доступ к фиче
    по этому ClientID (такова логика в джаве)

=cut
sub has_touch_direct_feature {
    my ($uid) = @_;

    return _does_user_have_feature($uid, 'touch_direct_enabled');
}

=head2 has_webvisor_feature

    Есть ли у пользователя доступ к фиче touch_direct_enabled.

    Пока у пользователя нет ClientID в Директе, можно получить фичу,
    только если она включена на процент (и процент вычисляется от хеша от uid)
    А когда у пользователя уже есть ClientID в Директе, вычисляется доступ к фиче
    по этому ClientID (такова логика в джаве)

=cut
sub has_webvisor_feature {
    my ($uid) = @_;

    return _does_user_have_feature($uid, 'webvisor_for_new_users_enabled_for_dna');
}

=head2 has_collecting_verified_phones_feature

    Включена ли у пользователя фича collecting_verified_phones_for_new_users.

    Пока у пользователя нет ClientID в Директе, можно получить фичу,
    только если она включена на процент (и процент вычисляется от хеша от uid)
    А когда у пользователя уже есть ClientID в Директе, вычисляется доступ к фиче
    по этому ClientID (такова логика в джаве)

=cut
sub has_collecting_verified_phones_feature {
    my ($uid) = @_;

    return _does_user_have_feature($uid, 'collecting_verified_phones_for_new_users');
}

=head2 has_collecting_verified_phones_required_feature

    Включена ли у пользователя фича collecting_verified_phones_required_for_new_users.

=cut
sub has_collecting_verified_phones_required_feature {
    my ($uid) = @_;

    return _does_user_have_feature($uid, 'collecting_verified_phones_required_for_new_users');
}

=head2 uac_desktop_welcome_enabled

    Есть ли у пользователя доступ к фиче uac_desktop_welcome_enabled.

    Пока у пользователя нет ClientID в Директе, можно получить фичу,
    только если она включена на процент (и процент вычисляется от хеша от uid)
    А когда у пользователя уже есть ClientID в Директе, вычисляется доступ к фиче
    по этому ClientID (такова логика в джаве)

=cut
sub uac_desktop_welcome_enabled {
    my ($uid) = @_;

    return _does_user_have_feature($uid, 'uac_desktop_welcome_enabled');
}

=head2 enable_uc_dna_user_choice

    Есть ли у пользователя доступ к фиче enable_uc_dna_user_choice.

    Пока у пользователя нет ClientID в Директе, можно получить фичу,
    только если она включена на процент (и процент вычисляется от хеша от uid)
    А когда у пользователя уже есть ClientID в Директе, вычисляется доступ к фиче
    по этому ClientID (такова логика в джаве)

=cut

sub enable_uc_dna_user_choice {
    my ($uid) = @_;

    return _does_user_have_feature($uid, 'enable_uc_dna_user_choice');
}

=head2 save_data_for_new_user($uid, %data)

    save_data_for_new_user(12345, welcome_email => "abc@def.gh");

    Сохранить данные для нового клиента (у которого еще нет записи в users/clients).
    При получении неизвестных именованных параметров - умирает.

    Параметры позиционные:
        $uid - паспортный UID
    Параметры именованные:
        welcome_email   - email для получения рекоммендаций и полезных писем
        gdpr_agreement_accepted_time - время принятия GDPR соглашения

    Допускаются параметры с суффиксом __dont_quote, при наличии ключа в обоих
    вариантах береться тот, что без суффикса

=cut

{
my @new_data_keys = qw/
    welcome_email
    gdpr_agreement_accepted_time
/;

sub save_data_for_new_user {
    my ($uid, %data) = @_;
    my %values = (uid => $uid);

    for my $key (@new_data_keys) {
        my $key_dont_quote = $key."__dont_quote";
        if (exists $data{$key}) {
            $values{$key} = delete $data{$key};
        } elsif (exists $data{$key_dont_quote}) {
            $values{$key_dont_quote} = delete $data{$key_dont_quote};
        }
    }

    if (%data) {
        croak 'Unsupported values: ' . join(', ', keys %data);
    }

    do_insert_into_table(PPCDICT, 'new_users_data', \%values, on_duplicate_key_update => 1, key => 'uid');
}

=head2 get_new_user_data($uid)

    Получить сохраненные для нового клиента данные.

    Параметры:
        $uid - (обязательный) паспортный UID клиента, для которого нужно получить данные
        $fields - (опционально) ссылка на массив название полей, которые нужно получить.
                  если не указан, будут выбраны все доступные поля (@new_data_keys)
    Результат:
        hashref с ключами из @new_data_keys
        или undef, если данных нет

=cut

sub get_new_user_data {
    my ($uid, $fields) = @_;

    my $db_fields;
    if ($fields && ref $fields eq 'ARRAY') {
        $db_fields = xisect(\@new_data_keys, $fields)
    } else {
        $db_fields = \@new_data_keys;
    }

    my $select_fields_str = join(',', map { sql_quote_identifier($_) } @$db_fields);

    return get_one_line_sql(PPCDICT, ["SELECT $select_fields_str FROM new_users_data", WHERE => {uid => $uid}]);
}
}

1;

