package User::Actions;

use Direct::Modern;

use Client;
use Client::CustomOptions;
use Direct::Validation::Client;
use Direct::ResponseHelper;
use GeoTools;
use PrimitivesIds;
use RBACDirect;
use RBACElementary;
use User;
use Yandex::Balance;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::I18n;
use Yandex::Validate;
use Wallet;


=head2 set_user_init_values($c, $uid, $client_data, %O)

    Создание клиента в Директе и, если нужно, в Балансе.
    Клиенту прописывается роль 'client'

    $uid - паспортный uid создаваемого клиента

    $client_data = {
        login   => ...,
        email   => ...,
        phone   => ...,
        fio     => ...,
        # город
        subregion_id    => ...,
        currency    => ...,
        # страна
        country_region_id   => ...,
    }
    %O = (
        # UID представителя агентства, которому добавляется субклиент
        agency_uid => ...,
        # создавать клиента без общего счета
        without_wallet => ...,
    )
=cut

sub set_user_init_values {
    my ($c, $uid, $client_data, %O) = @_;

    my $rbac = $c->rbac();
    my $agency_uid = $O{agency_uid};
    my $without_wallet = $O{without_wallet};

    my $client_id = get_clientid(uid => $uid);
    if (!$client_id) {
        # смотрим, есть ли этот представитель уже в Балансе
        $client_id = get_clientid_by_uid($uid);
    }
    my $client_role = rbac_who_is($rbac, $uid);

    my $user_data = hash_cut($client_data, qw/login email phone verified_phone_id fio lang tin tin_type/);
    if ($agency_uid) {
        $user_data->{agency} = 'yes';
        # при создании субклиента, подменяем UID на представителя агентства, т.к. он используется далее
        # для привязки субклиента к агентству
        $user_data->{UID} = $agency_uid;
    } else {
        $user_data->{UID} = $c->UID;
    }
    if ($client_data->{subregion_id}) {
        $user_data->{initial_subregion} = $client_data->{subregion_id};
    }
    $user_data->{role} = 'client';

    my $needs = Client::need_country_currency(
        client_id => $client_id,
        client_role => $client_role,
        is_direct => $c->is_direct,
        is_agency_client => ($agency_uid ? 1 : 0),
    );

    if ($needs->{need_country_currency}) {
        if (!$needs->{need_only_currency}) {
            my $country_region_id = $client_data->{country_region_id};
            die iget('Пожалуйста, укажите страну'), "\n" unless $country_region_id && is_valid_int($country_region_id, 0);

            my $region_data = GeoTools::get_exact_geo_region($country_region_id);
            die "invalid country region_id: $country_region_id" unless Client::is_valid_client_country($country_region_id), "\n";
            $user_data->{initial_country} = $country_region_id;
        }
        $user_data->{initial_currency} = $client_data->{currency};

        if ($c->is_direct) {
            #Проверка валюты и страны только в Директе, Баян использует только YND_FIXED.
            my $agency_client_id = get_clientid(uid => $agency_uid);
            my $vr = Direct::Validation::Client::check_country_and_currency({ClientID => $client_id, uid => $uid, %$user_data}, $agency_client_id, is_direct => $c->is_direct);
            if (!$vr->is_valid()) {
                die sprintf('invalid country or currency: %s/%s (%s)', @$user_data{qw/initial_country initial_currency/}, $vr->get_first_error_description()), "\n";
            }
        }
    }

    if (my $stashed_user_data = User::get_new_user_data($uid)) {
        if ($stashed_user_data->{welcome_email}) {
            $user_data->{recommendations_email} = $stashed_user_data->{welcome_email};
        }
    }
    # вытащили раньше, т.к. нужно было отдать ошибку в контроллере
    $user_data->{gdpr_agreement_accepted_time} = $client_data->{gdpr_agreement_accepted_time} if $client_data->{gdpr_agreement_accepted_time};

    create_update_user($uid, $user_data);
    $client_id = get_clientid(uid => $uid);

    # создаем клиента в rbac
    my $rbac_error;
    if ($agency_uid) {
        # прописываем клиенту агентство
        Client::on_campaign_creation($client_id, $agency_uid);
        $rbac_error = rbac_create_agency_subclient($rbac, $agency_uid, $uid, 'commit');
    } elsif ($c->login_rights->{manager_control}) {
        $rbac_error = rbac_create_manager_client($rbac, $c->UID, $uid, 'commit');
    }
    else {
        $rbac_error =  rbac_create_client($rbac, $uid, 'commit');
    }

    if ($rbac_error) {
        if ($rbac_error eq 'NOT_ENOUGH_FREEDOM') {
            error(iget("нет разрешения на обслуживание этого клиента"));
        }
        error(iget("Сохранить клиента не удалось"), undef, "rbac_create_*_client error: $rbac_error (in set_user_init_values)");
    }

    my $is_touch = $client_data->{is_touch};
    if ($without_wallet) {
        create_update_client({client_data => {ClientID => $client_id, create_without_wallet => 1, is_touch => $is_touch}});
    }
    set_user_custom_options($c->client_client_id, {assessor_offer_accepted => 1}) if $client_data->{assessor_offer_accepted};

    if ( # Если
        !$without_wallet # нет явного запрета на общий счет
        && Client::ClientFeatures::has_payment_before_moderation_feature($client_id) # и клиент попал в эксперимент
    ) { # то создаем ему сразу кошелек и прописываем опцию позволяющию оплату до модерации
        $c->client_client_id($client_id);
        my $result = Wallet::enable_wallet($c, $client_data->{currency}, $agency_uid, allow_wallet_before_first_camp => 1);
        if ($result->{wallet_cid}) { # Ошибка создания общего счета не повод полностью ломать создание клиента, просто игнорим
            create_update_client({client_data => {ClientID => $client_id, feature_payment_before_moderation => 1, is_touch => $is_touch}});
        }
    }
}

1;
