package UserOperations;
## no critic (TestingAndDebugging::RequireUseStrict, TestingAndDebugging::RequireUseWarnings)

use base qw/Exporter/;
our @EXPORT = qw//;
our @EXPORT_OK = qw/
    block_user
    block_user_error_description
/;

use Direct::Modern;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Settings;

use RBACElementary qw/rbac_who_is/;
use EnvTools qw/is_beta/;
use User qw//;
use Campaign qw/stop_camp/;
use ServicedClient qw//;
use PrimitivesIds;
use LogTools qw//;
use Tools qw/log_cmd/;
use Direct::Model::Wallet::Manager;
use Direct::Wallets;
use Notification;

=head2 _find_cids_for_blocking

Хелпер для block_user, находит cid'ы всех активных кампаний пользователя.

=cut
sub _find_cids_for_blocking {
    my ($uid) = @_;
    return get_one_column_sql(
        PPC(uid => $uid),
        [
            "select cid from campaigns",
            where => {
                statusShow => "Yes",
                archived => "No",
                uid => $uid,
                type__ne => "wallet"
	     }
        ]
    ) || [];
}

=head2 block_user

    my $error_or_undef = block_user($uid, %options);

Останавливает все кампании пользователя и блокирует его. Перед
блокировкой выполняется проверка на допустимость этой
блокировки. Правила это проверки можно отключать и включать через %options.

Опции:
    rbac - экземпляр RBAC для проверки роли, обязательный параметр.
    discount_check - по умолчанию 1. Если передать 0, то можно будет заблокировать клиента даже с установленной скидкой.
    camp_count_check - по умолчанию 1. Если передать 0, то можно будет заблокировать клиента даже если у него больше 200 кампаний.
    serviced_check - по умолчанию 0. Если передать 1, то нельзя будет заблокировать клиента с сервисируемыми кампаниями.
    UID - опционально, для логгирования.
Возвращаемое значение: undef в случае, если пользователь был успешно заблокировано, или одна из строк:
    admin - нельзя блокировать админских пользователей нигде, кроме бет
    has_discount - нельзя блокировать пользователей с установленной скидкой
    too_many_campaigns - нельзя блокировать пользователей с более чем 200-ми кампаниями.
    serviced - нельзя блокировать пользователей с сервисируемыми кампаниями.

=cut
sub block_user {
    my $uid = shift;
    my %options = (
        discount_check => 1,
        camp_count_check => 1,
        serviced_check => 0,
        @_
    );

    my $role = rbac_who_is($options{rbac}, $uid);
    return 'admin' if !is_beta() and $role !~ /client$/ and $role ne 'agency'; # DIRECT-30326, DIRECT-31059, r58990

    my $client_id = get_clientid(uid => $uid);
    return 'has_discount' if $options{discount_check} and User::client_has_discount(ClientID => $client_id);

    return 'serviced' if $options{serviced_check} and ServicedClient::is_client_serviced($client_id);

    my $camps = _find_cids_for_blocking($uid);
    return 'too_many_campaigns' if $options{camp_count_check} and scalar @$camps > 200;

    LogTools::log_messages('block_spam_users', {client_id => $client_id, camps => $camps, uid => $uid});
    log_cmd({cmd => '_block_user', UID => $options{UID}, uid => $uid, stopped_camps => join(',', @$camps)}) if $options{UID};

    if ($options{block_autopay_by_context}){
        # Если есть контекст - выключаем автопополнение
        my $context = $options{block_autopay_by_context};
        my ($wallet) = Direct::Wallets->get_actual_client_wallet($context);
        if (defined $wallet) {
            $wallet->autopay_mode('none');
            Direct::Model::Wallet::Manager->new(items => [$wallet])->update();
        }
    }

    foreach my $cid (@$camps) {
        stop_camp($uid, $cid, has_operator_super_control => 1, has_operator_manager_control => 0);
        log_cmd({ cmd => '_stopCamp', UID => $options{UID}, uid => $uid, cid => $cid });
    }

    do_update_table(PPC(ClientID => $client_id), "users", {statusBlocked => "Yes"}, where => {ClientID => SHARD_IDS});
    if ($role eq 'client') {
        add_notification($options{rbac}, 'blocked_client', {ClientID => $client_id}, {init_manager_uid => 1, init_ag_manager_uid => 1});
    }
    return;
}

=head2 block_user_error_description

Преобразует возвращаемый block_user код ошибки в человекочитаемую строку.

=cut
my %BLOCK_USER_ERRORS = (
    admin => iget_noop('Запрещена (раз)блокировка админ-ролей'),
    has_discount => iget_noop('Запрещена блокировка за спам крупных клиентов (со скидкой) - обратитесь на account@'),
    too_many_campaigns => iget_noop('Операция запрещена для клиентов с более 200 кампаниями. Для блокировки доступа воспользуйтесь страницей параметров клиента. Активные кампании нужно остановить вручную.'),
    serviced => iget_noop('Запрещена блокировка клиентов с сервисируемыми кампаниями'),
    __default__ => iget_noop('Неизвестная ошибка при блокировке пользователя'),
);
sub block_user_error_description {
    my ($error_code) = @_;
    return unless $error_code;
    return $BLOCK_USER_ERRORS{$error_code} // $BLOCK_USER_ERRORS{__default__};
}

1;
