package VCards;

# $Id$

=head1 NAME
    
    VCards 

=head1 DESCRIPTION

    модуль для функций, относящихся к "виртуальным визиткам" (=контактная информация в объявлениях)

=cut

use strict;
use warnings;

use Carp;
use YAML;

use List::MoreUtils qw/all none uniq any/;
use Yandex::HashUtils;
use Yandex::ListUtils;
use Yandex::ScalarUtils;
use Yandex::I18n;
use Digest::MD5 qw/md5 md5_hex/;
use Storable qw//;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use MailNotification;
use HashingTools;
use ShardingTools;
use GeoTools qw/get_geoid_by_cityname get_cityname_by_geoid get_geo_numbers/;
use Primitives;
use PrimitivesIds;
use OrgDetails;
use TextTools;
use ModerateChecks;
use JavaIntapi::GenerateObjectIds;
use JavaIntapi::ValidateVcards qw//;

use Direct::Model::VCard;

use Direct::Validation::VCards qw//;
use Direct::Validation::Errors;

use base qw/Exporter/;
our @EXPORT = qw/
        $VCARD_FIELDS
        $VCARD_FIELDS_DB
        $VCARD_FIELDS_FORM
        $VCARD_FIELDS_UNIQUE

        @PHONE_FIELDS

        get_vcards
        get_one_vcard
        add_vcard_to_banner
        copy_vcard
        update_vcard
        assign_vcard_to_banners
        create_vcards
        dissociate_vcards

        get_vcard_uses_count

        delete_vcard_from_db

        parse_phone
        compile_phone

        get_phone
        get_worktime
        get_worktimes_array
        make_org_details_from_vcard

        validate_contactinfo
        validate_contactinfo_api

        get_common_contactinfo_for_camp
        set_common_contactinfo
        user_has_accepted_vcard

        vcard_hash
        get_contacts_string
        compare_vcards
        separate_vcard
/;

use utf8; 

=head1 Нормализация данных визиток и их дедубликация

Есть несколько типов полей, которые должны по разному
обрабатываться. Причём как во входящих значениях, так и в хранящихся в
БД есть неоднозначности, когда разные данные означают одно и то же.

На стороне Perl в процессе нормализации мы полностью от этих
неоднозначностей избавляемся, для каждого типа поля по разному.

Для массового поиска визиток мы используем хэширование, как в Perl,
так и в БД. Для формирования части хэша в обоих этих случаях также
используются специфичные для типа поля преобразования. При вычислении
в БД надо не забывать про неоднозначности.

Также при необходимости можно найти в БД запись по нормализованным на
стороне Perl данным, или сгруппировать записи в БД для поиска
дубликатов. Но по хорошему этим не стоит пользоваться в дальней
перспективе.

- INTEGER NOT NULL (integer_not_null)
  При нормализации данных в Perl '' и undef превращается в 0.
  При вычислении хэша на стороне Perl нормализованное значение используется как есть.
  При вычислении хэша на стороне БД хранящееся там значение используется как есть.
  Для сравнения данных в БД между собой, а также с переданными из Perl нормализованными, можно просто использовать =
  Для группировки в БД - поле как есть.

- INTEGER NULL; 0 и NULL по смыслу отличаются (integer_with_zero_and_null)
  При нормализации данных в Perl '' превращается в undef.
  Хэш в Perl - undef превращается в ''
  Хэш в БД - NULL превращается в ''
  Для сравнения данных в БД - ifnull(column, -1) = ifnull(?, -1)
    В целях упрощения работы предполагаем, что валидные только положительные значения. Это позволяет всегда сравнивать приведённым выше
    условием, и нет необходимости несколько раз упоминать одно и то же значение (а с плэйсхолдерами это будет немного путать).
  Для группировки в БД - поле как есть.

- INTEGER NULL; 0 и NULL - одно и то же. (integer_null)
  При нормализации в Perl '' и 0 превращаются в undef.
  Хэш в Perl - undef превращается в 0
  Хэш в БД - NULL превращаются в 0
  Для сравнения данных в БД - ifnull(x, 0) = ifnull(?, 0)
  Для группировки в БД - ifnull(x, 0)

- VARCHAR/TEXT NULL (text)
  При нормализации в Perl '' превращается в undef.
  Хэш в Perl - undef превращается в ''
  Хэш в БД - undef превращается в ''
  Для сравнения данных в БД - ifnull(x, '') = ifnull(?, '')
  Для группировки в БД - ifnull(x, '')

  Так как таких полей больше всего, делаем это поведением по умолчанию.

- (identity)
  Специальный случай для идентификаторов uid/cid, для которых не нужно никаких преобразований, вынес отдельным типом
  чтобы не испытывать терпение оптимизатора mysql всякими ifnull и чтобы работать с полями как можно более единообразно.

=cut
my %special_field_handlers = (
    integer_not_null => {
        'normalize' => sub { my $val = shift;
                             defined($val) ? (!length($val) ? 0 : $val ) : 0 },
        'hash_db' => sub { my $field_exp = shift;
                           $field_exp },
        'hash_perl' => sub { my $val = shift;
                             $val },
        'compare_db' => sub { my $field_exp = shift;
                              "$field_exp = ?" },
    },
    integer_with_zero_and_null => {
        'normalize' => sub { my $val = shift;
                             defined($val) ? (!length($val) ? undef : $val ) : undef},
        'hash_db' => sub { my $field_exp = shift;
                           "ifnull($field_exp, '')" },
        'hash_perl' => sub { my $val = shift;
                             defined($val) ? $val : '' },
        'compare_db' => sub { my $field_exp = shift;
                              "ifnull($field_exp, -1) = ifnull(?, -1)" },
    },
    integer_null => {
        'normalize' => sub { my $val = shift;
                             defined($val) ? ((!length($val) or $val == 0) ? undef : $val ) : undef },
        'hash_db' => sub { my $field_exp = shift;
                           "ifnull($field_exp, '0')" },
        'hash_perl' => sub { my $val = shift;
                             defined($val) ? $val : '0' },
        'compare_db' => sub { my $field_exp = shift;
                              "ifnull($field_exp, 0) = ifnull(?, 0)"},
    },
    text => {
        'normalize' => sub { my $val = shift;
                             defined($val) ? (!length($val) ? undef : $val ) : undef },
        'hash_db' => sub { my $field_exp = shift;
                           "ifnull($field_exp, '')" },
        'hash_perl' => sub { my $val = shift;
                             defined($val) ? $val : ''},
        'compare_db' => sub { my $field_exp = shift;
                              "ifnull($field_exp, '') = ifnull(?, '')" },
    },
    identity => {
        normalize => sub { shift },
        hash_db => sub { shift },
        hash_perl => sub { shift // '' },
        compare_db => sub {
            my $field_exp = shift;
            "$field_exp = ?";
        },
    },
);

=head2 $VCARD_FIELDS, $VCARD_FIELDS_DB, $VCARD_FIELDS_FORM

    Список содержательных полей визитки 

    $VCARD_FIELDS
        поля, которые есть у визитки везде
        по-хорошему, почти нигде не должна использоваться (а получилось по-плохому)

    $VCARD_FIELDS_DB 
        список полей визитки в ppc.vcards

    $VCARD_FIELDS_UNIQUE
        список полей визитки по которым они проверяются на дубликаты

    $VCARD_FIELDS_FORM
        список полей визитки в формах

    Странное поле -- phone. Его надо вовремя преобразовывать в "собранный" или "разобранный" формат. 
    В таблице в качестве phone хранится "собранный" телефон ("+7#812#2128506#340"), 
    а в формах используются отдельные значения country_code, city_code, phone, ext, т.е. phone -- просто номер безо всяких кодов. 

=cut

our @PHONE_FIELDS = qw/country_code city_code phone ext/;

our $VCARD_FIELDS = [qw/
    apart
    build
    city
    contact_email
    contactperson
    country
    extra_message
    house
    im_client
    im_login 
    metro
    name
    org_details_id
    phone
    street
    worktime
/];

our $VCARD_FIELDS_DB = [ @$VCARD_FIELDS, qw/
    geo_id
    address_id
    cid
    uid
/];

# Поля из $VCARD_FIELDS_DB, которые обрабатываются отличным от обычного (text из %special_field_handlers) образом.
my %special_fields = (
    geo_id => 'integer_not_null',
    address_id => 'integer_null',
    org_details_id => 'integer_null',
    metro => 'integer_with_zero_and_null',

    # Эти 2 добавляем чтобы в коде не было для них отдельных if'ов. А так они всегда должны быть на месте и специальной обработки не требуется.
    cid => 'identity',
    uid => 'identity',
);

our $VCARD_FIELDS_UNIQUE = [
    @$VCARD_FIELDS, qw/
    geo_id
    address_id
    cid
    uid
/];

# country_code, city_code и ext - это часть phone в БД.
# ogrn отображается в org_details_id через отдельную табличку.
our $VCARD_FIELDS_FORM = [ @$VCARD_FIELDS, qw/
    country_code 
    city_code 
    ext

    auto_precision
    manual_point
    manual_bounds
    auto_point
    auto_bounds

    ogrn
/];

# страны, в которых переформатируем телефоны
our %PHONE_BEAUTIFY_ZONE = map {( "+$_" => 1 )} (
    7,      # Россия, Казахстан
    375,    # Беларусь
    380,    # Украина
);

our $USE_JAVA_VCARDS_IN_SMART_PROPERTY_NAME = 'use_java_vcards_in_smart';
our $USE_JAVA_VCARDS_FOR_MANAGE_VCARDS_PROPERTY_NAME = 'use_java_vcards_for_mv';
our $OTHER_BANNERS_IN_MANAGE_VCARDS_PROPERTY_NAME = 'other_banners_in_mv';

# Описание структуры визитки, используется для валидации
my %VCARD_STRUCT = (
    name => { length => 255, iname => iget_noop('Название компании/ФИО'), iname_api => 'CompanyName' },
    country => { length => 50, iname => iget_noop('Страна'), iname_api => 'Country' },
    city => { length => 55, iname => iget_noop('Город'), iname_api => 'City' },
    work_time => { length => 255, iname => iget_noop('Время работы'), iname_api => 'WorkTime' },
    phone => {
        length => 25, iname => iget_noop('Телефон'), iname_api => 'CountryCode CityCode Phone PhoneExt',
        subfields_iname => {
            country_code => iget_noop('Телефон/Код страны'),
            city_code => iget_noop('Телефон/Код города'),
            phone_short => iget_noop('Телефон'),
            phone_ext => iget_noop('Телефон/Добавочный'),
        },
        subfields_iname_api => {
            country_code => 'CountryCode',
            city_code => 'CityCode',
            phone_short => 'Phone',
            phone_ext => 'PhoneExt',
        },
    },
    contact_person => { length => 155, iname => iget_noop('Контактное лицо'), iname_api => 'ContactPerson' },
    street => { length => 55, iname => iget_noop('Улица'), iname_api => 'Street' },
    house => { length => 30, iname => iget_noop('Дом'), iname_api => 'House' },
    building => { length => 10, iname => iget_noop('Корпус'), iname_api => 'Build' },
    apartment => { length => 100, iname => iget_noop('Офис'), iname_api => 'Apart' },
    extra_message => { iname => iget_noop('Подробнее о товаре/услуге'), iname_api => 'ExtraMessage' },
    im_client => { length => 25, iname => iget_noop('Интернет-пейджер'), iname_api => 'IMClient' },
    im_login => { length => 255, iname => iget_noop('Интернет-пейджер'), iname_api => 'IMLogin' },
    contact_email => { length => 255, iname => iget_noop('E-mail'), iname_api => 'ContactEmail' },
    metro => { isa => 'Int', iname => iget_noop('Метро') },
    ogrn => { length => 255, iname => iget_noop('ОГРН/ОГРНИП'), iname_api => 'OGRN' },
);

=head2 normalize_vcard($vcard, %options)

Нормализация значений полей визитки перед отправкой этой информации в БД (для вставки или сравнения).

    my $normalized_vcard = normalize_vcard($vcard);

Возвращяет новую визитку (без изменения исходной) с нормализованными
данными. Поведение можно изменить на обновление исходной, добавив
descructive к опциям:

    normalize_vcard($vcard, destructive => 1);

=cut
sub normalize_vcard {
    my ($vcard, %options) = @_;

    $vcard = Storable::dclone($vcard) unless $options{destructive};

    for my $field (@$VCARD_FIELDS_UNIQUE) {
        $vcard->{$field} = $special_field_handlers{$special_fields{$field} // 'text'}{normalize}->($vcard->{$field});
    }

    return $vcard;
}

=head2 Представления сущности vcard

    1) Плоский хеш, считанный из БД, uid/cid присутствуют
    2) хеш с развёрнутым телефоном и массивом времени работы, uid/cid присутствуют
    3) сериализованное представление (что сериализуется пока не понял)

    Наружуу хорошо возвращать только 1о из 2х представлений (1, 2). Магия преобразования между ними должна быть скрыта в модуле.
    Пр.3 должно получаться из возвращаемого наружу представления (1 или 2)

=head2 get_vcards($where, $options)

    Выбирает из БД визитки по uid, cid или vcard_id
    Принимает две ссылки на хеши:
        $where      - условие выборки. Должен содержать cid, uid или vcard_id
        $options    - дополнительные параметры
            from_banners_only       - выбирать только используемые в баннерах визитки
                is_mediaplan            - флажок, что проверку использования визиток нужно проводить по баннерам медиаплана
                only_non_arch           - флажок, что проверку использования визиток нужно проводить только по неархивным баннерам
                skip_empty              - параметр, используемый в get_common_contactinfo_for_camp
                                            1 - будут выбраны только визитки с номером телефона и используемые в баннерах
                                            0 - если у пользователя/кампании есть баннеры БЕЗ визитки,
                                                то при определении "единости" КИ - будет считаться что есть 1 пустая визитка
            worktime_type_result    - параметр для форматирования времени работы, см. get_worktime
            take_24hour             - аналогично, см. get_worktime
            only_if_common          - флажок, означающий что необходимо вернуть визитку, только если она является "единой"
                                        * если визитка - ЕКИ - будет возвращена ссылка на массив из одной визитки
                                        * если нет - undef
    Результат:
        * ссылка на массив визиток (может быть пустым)
        * undef в особом случае (только с флажками from_banners_only и only_if_common) - если визитка "не единая"

    get_vcards({uid => 12345});
    get_vcards({cid => 987654});
    get_vcards({vcard_id => 743 });
    get_vcards({uid => 12345, cid => 987654});

=cut
sub get_vcards($;$) {
    my ($where, $options) = @_;
    my $vc_condition = hash_grep { defined $_ } hash_cut $where, [qw/uid cid vcard_id/];
    my $banners_condition = { %$vc_condition }; # копия
    $banners_condition->{banner_type} = ['text', 'dynamic'];
    $options ||= {};

    return [] unless keys %$vc_condition;

    my $fields = sql_fields(qw/uid cid vcard_id geo_id/, map { "vc.$_" } @$VCARD_FIELDS);


    $vc_condition->{vcard_id} = [$vc_condition->{vcard_id}] if $vc_condition->{vcard_id} && !ref $vc_condition->{vcard_id};

    if ( $options->{from_banners_only} ) {
        my @shard = choose_shard_param($banners_condition, [qw/uid cid vcard_id/], set_shard_ids => 1);
        my $table = $options->{is_mediaplan} ? 'mediaplan_banners' : 'banners';
        $banners_condition->{statusArch} = 'No' if !$options->{is_mediaplan} && $options->{only_non_arch};
        $banners_condition = {map {(/^(uid)$/ ? "c.$_" : "b.$_") => $banners_condition->{$_}} keys %$banners_condition};
        # $banners_condition->{vcard_id__is_not_null} = 1 if $options->{skip_empty};

        my $vcards_from_banners = get_one_column_sql(PPC(@shard), [
            "SELECT DISTINCT b.vcard_id FROM $table b join campaigns c using (cid)",
            where => $banners_condition,
        ]) || [];

        # если баннеров нет совсем - то и выбирать никакие визитки не нужно
        return [] unless @$vcards_from_banners;

        if ($options->{skip_empty}) {
            @$vcards_from_banners =  grep { defined $_ } @$vcards_from_banners;
        }

        if ($options->{only_if_common}) {
            return undef if ( @$vcards_from_banners > 1 || !defined $vcards_from_banners->[0]);
        }

        if ($vc_condition->{vcard_id} && @{ $vc_condition->{vcard_id} }) {
            $vc_condition->{vcard_id} = xisect($vc_condition->{vcard_id}, $vcards_from_banners);
        } else {
            $vc_condition->{vcard_id} = $vcards_from_banners;
        }
    }

    # если от нас требуют ЕКИ, а баннеры ссылаются на несколько разных визиток (или на визитку и undef) - то КИ - НЕ единая

    my @shard = choose_shard_param($vc_condition, [qw/uid cid vcard_id/], set_shard_ids => 1);

    $vc_condition->{_TEXT} = q/IFNULL(vc.phone, '') != ''/ if $options->{skip_empty};
    my $vcards = get_all_sql(PPC(@shard), [
            "SELECT $fields
                  , addresses.precision AS auto_precision
                  , CONCAT_WS(',', maps.x, maps.y) AS manual_point
                  , CONCAT_WS(',', maps.x1, maps.y1, maps.x2, maps.y2) AS manual_bounds
                  , CONCAT_WS(',', maps_auto.x, maps_auto.y) AS auto_point
                  , CONCAT_WS(',', maps_auto.x1, maps_auto.y1, maps_auto.x2, maps_auto.y2) AS auto_bounds
                  , addresses.map_id, addresses.map_id_auto
              FROM vcards vc
                   LEFT JOIN addresses ON vc.address_id = addresses.aid
                   LEFT JOIN maps ON addresses.map_id = maps.mid
                   LEFT JOIN maps maps_auto ON addresses.map_id_auto = maps_auto.mid
        ", where => $vc_condition
    ]);

    for my $v (@$vcards){
        $v->{worktimes} = get_worktime( $v->{worktime}, $options->{worktime_type_result} || '', $options->{take_24hour} || '');
        $v->{compiled_phone} = $v->{phone};
        hash_merge($v, parse_phone($v->{phone}));
        $v->{metro_name} = get_cityname_by_geoid($v->{metro}) if $v->{metro};
        $v->{is_auto_point} = $v->{map_id} && $v->{map_id_auto} && $v->{map_id_auto} == $v->{map_id} ? 1 : 0;  
        $v->{country_geo_id} = get_geo_numbers($v->{country});

        if( defined $v->{org_details_id} ) {
            my $org_details = get_org_details($v->{org_details_id});
            $v->{ogrn} = $org_details->{ogrn};
        }
    }

    return $vcards;
}

=head2 get_one_vcard ($vcard_id, $options)
    
    Выбирает из БД одну визитку по vcard_id ($options транслируется в get_vcards)

=cut
sub get_one_vcard($;$) {
    my ($vcard_id, $options) = @_;

    my $vcards = $vcard_id ? get_vcards({vcard_id => $vcard_id}, $options) : [];
    my $vcard = @$vcards ? $vcards->[0] : {};

    return $vcard;
}

=head2 add_vcard_to_banner($bid, $values)

    "Умное" добавление визитки с указанными полями к указанному объявлению 

    Если в кампании уже есть подходящая визитка -- будет использована она, если нет подходящей -- будет создана новая,
    если в $values нет поля phone -- то от баннера будет отвязана старая визитка.

    Если не указано обратное, будет сделан апдейт в таблице banners -- будет записан id визитки и обновлены статусы модерации (statusModerate, phoneflag)

    Параметры:
      $bids -- номер баннера, которому добавляется визитка
      $values -- хеш с визиточными полями. Может содержать лишние поля, их проигнорируем
      $options -- дополнительные (необязательные) параметры
        $options->{contacts}
        $options->{old_contacts} -- если хотя бы одно из полей присутствует, то они будут использованы для определения, была ли раньше в объявлении КИ 
          Если contacts и old_contacts нет -- будем извлекать старую визитку из базы
        $options->{dont_update_banners} -- флаг. Если выставлен, то таблицу banners обновлять не будем.
          По умолчанию 0, т.е. в banners обновится vcard_id и статусы модерации

    Результат:
      {
          vcard_id          => 123456,            # идентификатор присвоенной визитки (или undef, если визитка была пустой)
          created_vcard_id  => 123456             # id новосозданной визитки
      }

    Применение:
      $b{vcard_id} = add_vcard_to_banner($b{bid}, \%b)->{vcard_id};

=cut
sub add_vcard_to_banner {
    my ($bid, $values, $options) = @_;
    my $vcard_id = undef;

    my $res = {};
    my ($uid, $cid, $old_vcard_id, $phoneflag) = get_one_line_array_sql(PPC(bid => $bid), "SELECT c.uid, b.cid, b.vcard_id, b.phoneflag FROM banners b join campaigns c using(cid) WHERE b.bid = ?", $bid);

    # порядок важен - сначала визитка, потом uid и cid
    my $vcard_values = hash_cut(hash_merge({}, $values, {uid => $uid, cid => $cid}), @$VCARD_FIELDS_DB);
    $vcard_values->{metro} = undef if !defined $vcard_values->{metro} or $vcard_values->{metro} eq '';

    # Пытаемся вытащить address_id из структурированной визитки, если есть
    # Функция check_address_map сохраняет address_id внутрь хеша map.
    # В плоском хеше баннера: address_id может быть старым
    if (ref($values->{map}) eq 'HASH') {
        $vcard_values->{address_id} = $values->{map}{aid};
    }

   if (!$vcard_values->{phone}) {
        $vcard_id = undef;
    } else {
        create_vcards($uid, [$vcard_values]);
        $vcard_id = $vcard_values->{vcard_id};
        $res->{created_vcard_id} = $vcard_id unless $vcard_values->{duplicate};
    }

    if ($old_vcard_id and ($vcard_id // -1) != $old_vcard_id) {
        dissociate_vcards($old_vcard_id);
    }

    unless($options->{dont_update_banners}){
        my $old_vcard = hash_merge {phoneflag=>$phoneflag}, get_one_vcard($old_vcard_id);
        $old_vcard->{phone} = compile_phone($old_vcard);
        my $need_moderate = ($old_vcard_id) ? ModerateChecks::check_moderate_contactinfo($vcard_values, $old_vcard) : 1;

        my $new_values = {vcard_id => $vcard_id};
        if ($vcard_id) {
            if ($need_moderate) {
                $new_values->{phoneflag} = "IF( statusModerate = 'New', 'New', 'Ready')";
            }
        } else {
            $new_values->{phoneflag} = q{'New'};
        }
        do_update_table(PPC(bid => $bid), 'banners', $new_values , where => {bid => $bid}, dont_quote => ['phoneflag']);
    }

    $res->{vcard_id} = $vcard_id;

    return $res;
}

=head2 create_vcards($uid, $vcards)

Создание тех визиток из пачки, которые еще не существуют в БД.
Создаётся минимально необходимое количество записей в БД.
Результаты сохраняет в массиве визиток $vcards и его же возвращает.

    Параметры:
        $uid
        $vcards - массив визиток [{}, {}]
           - должны содержать cid.
           - должны содержать uid совпадающий с параметром функции или ничего (в этом случае будет использован параметр).

    Результат:
        [{vcard_id => }, {vcard_id => }] - массив визиток с выставленым vcard_id
        если в результате поиска дубликатов такие были найдены, у дублирующих визиток появится ключ duplicate = 1

=cut
sub create_vcards {
    my ($uid, $vcards, %options) = @_;

    return [] unless @$vcards;

    my %hash_groups;
    foreach my $vcard (@$vcards) {
        # Доводим визитку до вида, в котором она должна храниться в БД
        check_add_vcard_geo_id($vcard);
        normalize_vcard($vcard, destructive => 1);
        if (not $vcard->{uid}) {
            $vcard->{uid} = $uid;
        } elsif ($vcard->{uid} != $uid) {
            die "vcard uid mismatch: '$uid' != '$vcard->{uid}'";
        }
        $vcard->{md5_hash} = vcard_hash($vcard) unless exists $vcard->{md5_hash};
        # Если в пачке больше чем одна визитка с таким хэшем, сразу помечаем её как дубликат.
        $vcard->{duplicate} = 1 if exists $hash_groups{$vcard->{md5_hash}};
        push @{$hash_groups{$vcard->{md5_hash}}}, $vcard;
    }

    # uid используется и как часть проверки на уникальнось, а в запросе он явно указан только чтобы индекс использовался.
    my $sql_hash_placeholders = join ',', ('?') x scalar keys %hash_groups;
    my $sql_hash_exp = vcard_hash_sql_exp('vc');
    my $sql = "SELECT $sql_hash_exp, vc.vcard_id
                   FROM vcards vc
                   WHERE uid = ? AND $sql_hash_exp IN ($sql_hash_placeholders)";

    my $exists_vcards = get_hash_sql(PPC(uid => $uid), $sql, $uid, keys %hash_groups);

    my @new_vcards;
    while (my($hash, $vcards_with_same_hash) = each %hash_groups) {
        if (exists $exists_vcards->{$hash}) {
            for (@$vcards_with_same_hash) {
                $_->{duplicate} = 1;
                $_->{vcard_id} = $exists_vcards->{$hash};
            }
        } else {
            push @new_vcards, $vcards_with_same_hash->[0];
        }
    }

    if (@new_vcards) {
        my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'vcard',
                count => scalar @new_vcards, client_id => get_clientid(uid => $uid))->call();
        my @to_insert;
        foreach my $vcard (@new_vcards) {
            my $vcard_id = shift @$ids;
            $_->{vcard_id} = $vcard_id for @{$hash_groups{$vcard->{md5_hash}}};

            $vcard->{metro} = undef if !defined $vcard->{metro} or $vcard->{metro} eq '';
            $vcard->{uid} = $uid;
            warn "vcard: cid missing" unless $vcard->{cid};
            $vcard = normalize_vcard($vcard);
            push @to_insert, [@$vcard{@$VCARD_FIELDS_DB, 'vcard_id'}];
        }

        do_mass_insert_sql(PPC(uid => $uid),
            sprintf('INSERT INTO vcards(%s) VALUES %%s', join ',', map {sql_quote_identifier($_)} @$VCARD_FIELDS_DB, 'vcard_id'),
            \@to_insert);
    }

    return $vcards;
}

=head2 copy_vcard

    Копирование визитки по заданному vcard_id.
    Просто добавляет новую визитку а таблицу vcards c полями как у оригинальной визитки.
    Если очень хочется - поля визитки-копии можно переопределить, указав желаемые значения в хеше overrides

    Возвращает новый vcard_id

=cut
sub copy_vcard($;$) {
    my ($old_vcard_id, $overrides) = @_;
    $overrides ||= {};

    return unless $old_vcard_id;
    die "can't override vcard_id" if exists $overrides->{vcard_id};

    my @ex = @{ xminus([keys %$overrides], $VCARD_FIELDS_DB) };
    die "Fields @ex doesn't exists in vcard!" if @ex;

    # поля LastChange и vcard_id не выбираем за ненадобностью
    my $vcard = get_one_line_sql(PPC(vcard_id => $old_vcard_id),
        ['SELECT', sql_fields(@$VCARD_FIELDS_DB), 'FROM vcards', where => { vcard_id => $old_vcard_id }]
    );
    return unless $vcard && $vcard->{cid};

    hash_merge($vcard, $overrides);

    return create_vcards($vcard->{uid}, [$vcard])->[0]{vcard_id};
}

=head2 get_vcard_uses_count

    сколько раз используется визитка?

=cut
sub get_vcard_uses_count {
    my $vcard_id = shift; 

    return 0 unless $vcard_id;

    # визитка может использоваться в обычном объявлении или в медиапланном
    my $b_uses = get_one_field_sql(PPC(vcard_id => $vcard_id),
        "SELECT COUNT(*) 
           FROM vcards vc 
                JOIN banners b ON b.vcard_id = vc.vcard_id 
          WHERE vc.vcard_id = ?", $vcard_id
    ) || 0;

    my $mb_uses = get_one_field_sql(PPC(vcard_id => $vcard_id),
        "SELECT COUNT(*) 
           FROM vcards vc 
           JOIN mediaplan_banners mb ON mb.vcard_id = vc.vcard_id 
          WHERE vc.vcard_id = ?", $vcard_id
    ) || 0;

    return $b_uses + $mb_uses;
}

=head2 assign_vcard_to_banners ($bids, $vcard_id)

    Параметры
        $bids 
        $vcard_id 

=cut
sub assign_vcard_to_banners {
    my ($bids, $vcard_id) = @_;
    my $res = {};
    return $res unless @$bids; # NOOK

    my $cids = get_cids(bid => $bids);
    return $res unless @$cids;
    
    my $vcard = get_one_line_sql(PPC(vcard_id => $vcard_id),
        ['SELECT', sql_fields('vcard_id', @$VCARD_FIELDS_DB), 'FROM vcards', where => { vcard_id => $vcard_id }]
    );

    if (@$cids == 1 && $cids->[0] == $vcard->{cid}) {
        my $old_vcard_ids = get_one_column_sql(
            PPC(vcard_id => $vcard_id), [
                "select distinct vcard_id from banners",
                where => { bid => $bids, vcard_id__ne => $vcard_id }
            ]
        );
        dissociate_vcards($old_vcard_ids);

        do_update_table(PPC(vcard_id => $vcard_id), 'banners',
            {
                vcard_id => $vcard_id,
                phoneflag => "IF( statusModerate = 'New', 'New', 'Ready')"
            },
            where => { bid => $bids, cid => $vcard->{cid} },
            dont_quote => ['phoneflag']
        );
        do_sql(PPC(bid => $bids), [
            "delete bmg from banners_minus_geo bmg join banners b on b.bid = bmg.bid",
            where => {
                'bmg.bid' => SHARD_IDS,
                'b.statusModerate__in' => [qw/Ready Sending Sent/],
                'bmg.type' => 'current',
            }
        ]);
    } else {
        for my $bid (@$bids){
            my $b_res = add_vcard_to_banner($bid, $vcard);
            hash_merge $res, hash_cut($b_res, qw/vcard_id created_vcard_id/);
        }
    }

    return $res;
}

=head2 update_vcard($vcard_id, $values);

Попытка обновления визитки для всех баннеров, которые ссылаются на указанный $vcard_id.
Если переданные $values уже соответствуют хранящимся в БД данным, то ничего выполнено не будет.

Если же для указанных данных выбрана (или создана) с отличным от старого vcard_id, то обновляет все
баннеры для использование новой визитки, а также обновляет их статусы модерации.

Параметры:
    $vcard_id -- id визитки, подлежащей обновлению
    $values -- хеш с визиточными значениями. Может содержать лишние поля, их проигнорируем.
               Должен обязательно содержать поля cid и uid.

=cut
sub update_vcard($$) {
    my ($vcard_id, $vcard_values) = @_;

    for (qw/uid cid/) {
        confess "No $_ for update_vcard('$vcard_id', ...)" unless $vcard_values->{$_};
    }

    my $old_vcard_data = get_one_line_sql(PPC(vcard_id => $vcard_id), "SELECT cid, phoneflag FROM banners b WHERE vcard_id = ?", $vcard_id);

    $vcard_values = {%$vcard_values}; # собственная копия, для create_vcards.
    my $new_vcard_id = create_vcards($vcard_values->{uid}, [$vcard_values])->[0]{vcard_id};
    if ($new_vcard_id != $vcard_id) {
        dissociate_vcards($vcard_id);
    
        my $old_vcard = get_one_vcard($vcard_id);
        $old_vcard->{phoneflag} = (defined $old_vcard_data) ? $old_vcard_data->{phoneflag} : 'New';
        $old_vcard->{phone} = compile_phone($old_vcard);
        my $need_moderate = ($old_vcard_data) ? ModerateChecks::check_moderate_contactinfo($vcard_values, $old_vcard) : 1;

        my $new_values = {vcard_id => int($new_vcard_id),
                          statusBsSynced => 'No', 
                         };
        if ($need_moderate) {
            $new_values->{phoneflag__dont_quote} = "IF( statusModerate = 'New', 'New', 'Ready')";
        }

        do_update_table( PPC(vcard_id => $vcard_id), 'banners', $new_values, where => { vcard_id => $vcard_id});
    }
    return $new_vcard_id;
}

=head2 parse_phone

    Разбирает "собранный" телефонный номер (country_code#city_code#phone#ext) в хеш

    На входе -- строка вида "+7#812#2128506#340"
    На выходе -- ссылка на хеш {country_code => "+7", city_code => "812", phone => "2128506", ext => "340" )

    Типичное использование: 
        hash_merge($v, parse_phone($v->{phone}));

=cut
sub parse_phone($) {
    my $phone = shift;
    return {} unless defined $phone;
    my $parsed_phone = {country_code => '', city_code => '', phone => '', ext => ''};
    @$parsed_phone{qw/country_code city_code phone ext/} = split('#', $phone, 4) if defined $phone;
    return $parsed_phone;
}

=head2 compile_phone
    
    Собирает телефон в строку так, что его можно распарсить функцией parse_phone 
    
    На входе -- ссылка на хеш, содержащий поля {country_code => "+7", city_code => "812", phone => "2128506", ext => "340" )
    На выходе -- строка вида "+7#812#2128506#340"
        Если все 4 поля пустые - вернётся undef

    Типичное использование: 
       
        our @PHONE_FIELDS = qw/country_code city_code phone ext/;
        
        ...

        my $phone_hash = hash_cut \%FORM, @PHONE_FIELDS;
       
        delete @$vcard_flat{@PHONE_FIELDS};
        $vcard_flat->{phone} = compile_phone($phone_hash);

=cut
sub compile_phone($) {
    my ($phone_hash) = @_;

    my %phone = map {( $_ => str $phone_hash->{$_} )} @PHONE_FIELDS;
    return undef  if none { length $_ } values %phone;

    # если в нужной стране, переформатируем номер
    if ( $PHONE_BEAUTIFY_ZONE{$phone{country_code}} ) {
        $phone{phone} = _reformat_phone($phone{phone});
    }

    return join '#', map {$phone{$_}} @PHONE_FIELDS;
}


=head2 _reformat_phone

    Переформатируем телефонный номер в стандартизованный формат

=cut
sub _reformat_phone {
    my ($phone) = @_;

    $phone =~ s/\D//gxms;
    # режем на группы по 2 цифры
    $phone =~ s/ (?<= \d) (?= (?: \d\d)+ $ ) /-/gxms;
    # объединяем первые группы, если в первой 1 цифра, а всего групп больше трёх
    if ( length $phone > 9 ) {
        $phone =~ s/(?<= ^\d) -//xms;
    }

    return $phone;
}

=head2 serialize($vcard) 

    Сериализуем хеш с контактной информацией

=cut
sub serialize($) {
    my ($vcard) = @_;

    my $vcard_slice = hash_cut $vcard, @$VCARD_FIELDS, qw/ogrn/;

    return YAML::Dump($vcard_slice);
}

=head2 deserialize($string)

    Востанавливаем объект из строкового представления
    
TODO: написать проверку валидности востановленного хеша 

=cut
sub deserialize($) {
    my ($vcard_string) = @_;

    my $vcard = YAML::Load($vcard_string);

    return $vcard;
}

=head2 vcard_hash_sql_exp

Формирует фрагмент sql-кода для подсчёта хэша визитки в БД. Алгоритм
соответствует используемому в vcard_hash.

    my $sql_expr = vcard_hash_sql_exp($table_name);

где C<$table_name> - алиас для таблицы визиток в том выражении, куда
результат этой функции подставляться будет.

=cut
sub vcard_hash_sql_exp {
    my ($table_name) = @_;

    my $table = sql_quote_identifier($table_name);

    my $fields = join ',', map {
        $special_field_handlers{$special_fields{$_} // 'text'}{hash_db}->(
            "$table.@{[sql_quote_identifier($_)]}"
        )
    } @$VCARD_FIELDS_UNIQUE;

    return "MD5(CONCAT_WS('~', $fields))";
}

=head2 vcard_hash

    Вычисляет (какую-то) хеш-сумму от визитки.
    Существует способ вычислить такую-же хэш-сумму в БД - vcard_hash_sql_exp
    Способ сравнить, одинаковы ли две визитки: vcrad_hash($vc_1) eq vcard_hash($vc_1)

    %options:
        ignore_fields => ссылка на массив, какие поля из VCARD_FIELDS_UNIQUE игнорировать при вычислении хеша
                         несовместимо с дедубликацией/иммутабельностью
                         единственное назначение: сравнение при загрузке из xls (где невозможно задать метро)

=cut
sub vcard_hash {
    my ($vcard, %options) = @_;

    $vcard = normalize_vcard($vcard);

    my %ignore_fields;
    if ($options{ignore_fields}) {
        $ignore_fields{$_} = 1 for @{$options{ignore_fields}};
    }

    my $vcard_str = join '~', map {
        $special_field_handlers{$special_fields{$_} // 'text'}{hash_perl}->($vcard->{$_})
    } grep { !$ignore_fields{$_} } @$VCARD_FIELDS_UNIQUE;

    return md5_hex_utf8($vcard_str);
}

=head2 compare_vcards($vcard1, $vcard2)
    
    Сравнить 2 визитки в представлении  flat_hash.

    Возвращает 
        0 Визитки одинаковы
        1 визитки разные

    В данный момент точки на карте сравниваются по geo_id и address_id, (должны сравниваться по развёрнутой информации о точках).
    Рабочее время сравнивается как строка (worktine)
    Детальная информация сравнивается при помощи вызова соответствующей функции из модуля OrgDetails. Мне кажется это единственный правильный подход.

    

=cut
sub compare_vcards($$) {
    my ($vcard1, $vcard2) = @_;
    
    return 0 if !defined $vcard1 && !defined $vcard2;
    return 1 if (defined $vcard1 && !defined $vcard2) || (!defined $vcard1 && defined $vcard2);
        
    # После нормализации сравнение через str(..) eq str(..) имеет больше смысла.
    $vcard1 = normalize_vcard($vcard1);
    $vcard2 = normalize_vcard($vcard2);

    # Плоские поля визитки и worktime (не worktimes !!)
    my $and_result = all {str($vcard1->{$_}) eq str($vcard2->{$_})} @$VCARD_FIELDS;

    # точки на карте
    $and_result = $and_result && all {str($vcard1->{$_}) eq str($vcard2->{$_})} 'address_id', 'geo_id';

    # NB Вообще конечно для сравнения было бы хорошо использовать vcards_hash(), но эта функция используется
    # для несохранённых в БД визиток, поэтому оставляем всё как есть ради следующего куска кода.
    # Детальная информация
    $and_result = $and_result && !compare_org_details(make_org_details_from_vcard($vcard1), make_org_details_from_vcard($vcard2));

    return !$and_result;
}

=head2 Историческое примечание

    get_phone
    get_worktime
    get_worktimes_array   
    get_contacts_string
    validate_contactinfo
    
    Были раньше в Common.pm
    Перенесены сюда, чтобы уменьшить Common и не создавать лишних циклических зависимостей

=cut

=head2 get_phone

    Делает из "собранного" телефонного номера human-readable строчку

    На входе -- hashref контактной информации с полем phone="+7#812#2128506#340" или полями country_code city_code phone ext
    На выходе -- строка вида "+7 (812) 2128506 доб. 340"

    Использование: 
        подозрительное какое-то, в основном для get_contacts_string
        
    Противоречие со здравым смыслом: 
        особенности отображения данных внесены в perl-код. Нехорошо. 

=cut
sub get_phone {
    my $contact_info = shift || {};
    return '' if !defined $contact_info->{phone} || $contact_info->{phone} eq '';
    my ($country_code, $city_code, $phone, $ext) = ( "", "", "", "" );
    if (index($contact_info->{phone}, '#')) {
        ($country_code, $city_code, $phone, $ext) = split "#", $contact_info->{phone};
    } else {
        ($country_code, $city_code, $phone, $ext) = map { $contact_info->{$_} || '' } @PHONE_FIELDS;
    }
    
    return ($country_code||'').( (defined $city_code && $city_code ne '') ? ' ('.$city_code.') ':'' ).($phone||'').( $ext ? ' доб. '.$ext : '' );
}

my %WEEK = (
    0 => iget_noop('пн'),
    1 => iget_noop('вт'),
    2 => iget_noop('ср'), 
    3 => iget_noop('чт'), 
    4 => iget_noop('пт'), 
    5 => iget_noop('сб'),
    6 => iget_noop('вс')
);

=head2 get_worktime($worktime, $type_result, $take_24hour, $days_of_week_words)

    Разбирает время работы из "собранной" строки в массив хешей либо красивых строк 

    Параметры 
      $worktime -- строка вида "1#3#10#15#18#30;4#6#10#30#20#25"
      $type_result -- необязательно. 
        "string" -- будет возвращена ссылка на массив красивых строк 
        другое   -- будет возвращена ссылка на массив удобных хешей
      $take_24hour -- необязательно. Отслеживать ли круглосуточную работу.
      $days_of_week_words -- не приводить дни недели к человекочетаемым названиям
    
    Результат
      При запуске с параметрами по-умолчанию, возвращает ссылку на массив хешей (по одному на каждый временной интервал) вида:
        {
            d1   "вт",
            d2   "чт",
            h1   10,
            h2   18,
            m1   15,
            m2   30
        }
      При этом соседние даты с одинаковыми временными интервалами "склеиваются" в одну, напрмиер "1#1#10#15#18#30;2#2#10#15#18#30" (2 интервала) будет представлено одним интервалом "вт-ср 10:15-18:30".
      Если интервал состоял из одного дня - d2 в хеше равняется пустой строке "".
      
      При указании вторым аргументов "string" - функция возвращает скаляр, содержащий строки (по одной на интервал) следующего вида:
        "пн-вт 22:00-02:00
        ср-сб 07:00-08:00
        "
      При этом скаляр оканчивается пустой строкой (\n)! (ФИЧА-БАГ?)

      При указании третьим аргументом '24hour', диапозон времени 0:0-0:0 будет записан как "круглосуточно". В хеше это значение будет у ключа h1, для остальных ключи времени (m1, h2, m2) значениями будут ''
        Если результирующий интервал (после склейки) будет состоять из всех дней недели и иметь время вида 06:00-06:00 - он тоже будет представлен как "круглосуточно"

      Четвертый аргумент функции отвечает за то, чтобы оставлять дни недели цифрами (если аргумент истенен в булевом контексте)
=cut
sub get_worktime {
    my $worktime = shift || '';
    my $type_result = shift || '';
    my $take_24hour = shift || '';
    my $days_of_week_words = shift;

    my @worktimes;
    my ($prev_d1, $prev_d2, $prev_h1, $prev_m1, $prev_h2, $prev_m2);
    $prev_d2 = 7;    # чтобы при первом проходе условие не выполнилось и не было предупреждения о неинициализированной переменной

    foreach my $str (sort grep {$_} split ';', $worktime) {
        my ($d1, $d2, $h1, $m1, $h2, $m2) = split '#', $str;

        my $to_glue;
        # условие следующего IF означает, что:
        # начальный день текущего интервала - "следующий" за конечным предыдущего
        # полностью совпадают начальные и конечные часы и минуты
        if ($d1 == $prev_d2 + 1 && $h1 == $prev_h1 && $m1 == $prev_m1 && $h2 == $prev_h2 && $m2 == $prev_m2) {
            # если истина - то будем "склеивать" текущий интервал с предыдущим,
            # т.е. менять последний элемент результирующего массива @worktimes
            $to_glue = 1;
            # берем начальный день для текущего интервала - из предыдущего.
            $d1 = $prev_d1;
        } else {
            # если ложь, то текущий интервал сам по себе, и будем его добавлять в @worktimes
            $to_glue = 0;
        }

        # Сохраняем предыдущие значения здесь и сейчас, потому что дальше по коду проверяемые переменные меняются
        ($prev_d1, $prev_d2, $prev_h1, $prev_m1, $prev_h2, $prev_m2) = ($d1, $d2, $h1, $m1, $h2, $m2);

        my $day1 = $days_of_week_words ? $d1 : iget($WEEK{$d1});
        my $day2 = $days_of_week_words ? $d2 : ($d1 != $d2 ? iget($WEEK{$d2}) : '');

        if ($take_24hour eq '24hour' 
                && ( ("$h1:$m1-$h2:$m2" =~ /^0{1,2}:0{1,2}-0{1,2}:0{1,2}$/)
                        || ($d1 == 0 && $d2 == 6 && $h1 == $h2 && $m1 == $m2) )
        ) {
            $h1 = iget('круглосуточно');
            $m1 = $h2 = $m2 = '';
        }
        
        # для часов и минут добавляем ведущий 0 -- так красивее и единообразнее
        s/^(\d)$/0$1/ for ($h1, $m1, $h2, $m2);

        # Делаем хеш для интервала
        my $interval = { 'd1' => $day1, 'd2' => $day2, 'h1' => $h1, 'm1' => $m1, 'h2' => $h2, 'm2' => $m2 };

        if ($to_glue) { # меняем в массиве предыдущий на объединенный (предыдущий+текущий)
            $worktimes[-1] = $interval;
        } else {        # добавляем новый интервал в список результатов
            push @worktimes, $interval;
        }
    }

    if( $type_result eq 'string' ) {
        return join('', map {
            $_->{d1}.( $_->{d1} ne $_->{d2} ? "-" : '' ).$_->{d2}." ".$_->{h1}.":".$_->{m1}."-".$_->{h2}.":".$_->{m2} . "\n"
        } @worktimes);
    }
    return \@worktimes;
}

=head2 get_worktimes_array

    Время работы организации в виде списка

=cut
sub get_worktimes_array {
    my ($worktime, $days_of_week_words) = @_;
    return get_worktime($worktime, undef, undef, !$days_of_week_words);
}

# значения для "круглосуточно", которые ожидает от нас БК
# не используем iget, т.к. перевод слова может измениться, а БК ожидает предопределённые значения
# для неописанных языков будет использоваться русская версия
my %texts4roundclock = (
    ru => 'круглосуточно',
    tr => '24 saat',
    en => 'Round the clock',
    uk => 'Цілодобово',
    de => 'Round the clock',
    uz => '24 soat',
);

=head2 get_worktime_for_bs

    Возвращает время работы из визитки в формате, пригодном для отправки в БК в $banner->{ContactInfo}->{WorkTime}
    $lang — язык баннера (украинский — uk)

    $banner->{ContactInfo}->{WorkTime} = get_worktime_for_bs($worktime_from_db, $lang);

=cut

sub get_worktime_for_bs {
    my ($worktime_from_db, $lang) = @_;

    my @newWorkTimes;
    my $worktimes = get_worktime($worktime_from_db, undef, '24hour', 1);
    
    for my $worktime (@$worktimes) {
        my @parts;
        
        my $is_round_clock = $worktime->{h1} && $worktime->{m1} eq '' && $worktime->{h2} eq '' && $worktime->{m2} eq '';
        my $is_all_days = $worktime->{d1} == 0 && $worktime->{d2} && $worktime->{d2} == 6;
        if (($is_all_days && !$is_round_clock) || !$is_all_days) {
            push @parts, iget($WEEK{$worktime->{d1}}) . ($worktime->{d2} && $worktime->{d1} != $worktime->{d2} ? '-' . iget($WEEK{$worktime->{d2}}) : '');
        }
        if ($is_round_clock) {
            my $round_clock_str;
            if ($lang && exists $texts4roundclock{$lang}) {
                $round_clock_str = $texts4roundclock{$lang};
            } else {
                $round_clock_str = $texts4roundclock{ru};
            }
            push @parts, $round_clock_str;
        } else {
            push @parts, sprintf("%d:%02d-%d:%02d", $worktime->{h1}, $worktime->{m1}, $worktime->{h2} == 0 && $worktime->{m2} == 0 ? 24 : $worktime->{h2} , $worktime->{m2});
        }
        push @newWorkTimes, join ' ', @parts;
    }

    return join ", ", @newWorkTimes;
}

=head2 get_contacts_string

    Собирает из визиточных полей строку с "человеческим" описанием контактной информации

    Используется: 
      для сравнения старой и новой КИ при сохранении объявления 
      для текста уведомлений (mail_notification)
        
    Неформально:
      подозрительная функция, и использование тоже подозрительное
      * передавать бы удобнее хеш, а не список скаляров
      * для сравнения визиток считать бы какую-нибудь хеш-функцию
      * для уведомлений хорошо бы убрать текст из кода

=cut
sub get_contacts_string {
    my ($contact_info, $sep, $options) = @_;
    $options ||= {};

    $contact_info = {%$contact_info};
    $contact_info->{worktime} = str(get_worktime($contact_info->{worktime}, 'string'));
    $contact_info->{phone} = str(get_phone($contact_info));
    $contact_info->{metro} = get_cityname_by_geoid($contact_info->{metro}) if $contact_info->{metro};
    my %names = (   phone => 'Телефон', name => 'Название организации', contactperson => 'Контактное лицо', 
                    worktime => 'Время работы', country => 'Страна', city => 'Город', street => 'Улица', 
                    house => 'Дом', build => 'Корпус', apart => 'Офис', metro => 'Метро', 
                    im_client => 'Интернет-пейджер', im_login => 'Логин', contact_email => 'Контактный E-mail', extra_message => 'Подробное описание товара/услуги',
                    ogrn => 'ОГРН',
                );
    my @result = ();
    $sep = "\n" if !defined $sep;

    for my $field (qw/phone name contactperson worktime country city street house build apart metro/) {
        my $delimiter = $field eq 'worktime' ? ":\n" : ': ';
        push @result, $names{$field}.$delimiter.$contact_info->{$field} if defined $contact_info->{$field} && $contact_info->{$field} ne '';
    }

    my @other_fields = qw/im_client im_login contact_email extra_message/;
    push @other_fields, qw/address_id org_details_id/ if ! $options->{dont_include_ids};

    for my $field (@other_fields) {
        my $field_name = $names{$field} || '';
        push @result, $field_name . ': ' . $contact_info->{$field} if $contact_info->{$field};
    }

    my $ogrn = $contact_info->{ogrn};
    if ($contact_info->{org_details_id} && ! defined($ogrn)) {
        my $org_details = get_org_details($contact_info->{org_details_id});
        $ogrn = $org_details->{ogrn} if ref($org_details) eq 'HASH' && $org_details->{ogrn};
    }
    push @result, $names{ogrn} . ': ' . $ogrn if defined($ogrn);


    return join $sep, @result;
}

# внутренняя функция преобразования хешей визиток в другой формат
# (вынесено из _base_validate_contactinfo,
# для переданных визиток будет вызван smartstrip)

sub _convert_vcards_for_validation {
    my ($banners) = shift;

    my @vcards;
    foreach my $banner (@$banners) {
        for my $field (@$VCARD_FIELDS_FORM) { smartstrip($banner->{$field}) };
        foreach( qw /name street house build apart metro contactperson city country extra_message/) {
            next unless defined $banner->{$_} && length $banner->{$_};
            $banner->{$_} =~ s/[\n\r]+/ /gs;
        }
 
        # Данные на валидацию
        my %vcard = (
            name => $banner->{name},
            country => $banner->{country},
            city => $banner->{city},
            work_time => $banner->{worktime},
            phone => (any { defined($banner->{$_}) } qw(country_code city_code ext)) ? join('#', map { $banner->{$_} // '' } qw(country_code city_code phone ext)) : $banner->{phone},
            contact_person => $banner->{contactperson},
            street => $banner->{street},
            house => $banner->{house},
            building => $banner->{build},
            apartment => $banner->{apart},
            extra_message => $banner->{extra_message},
            im_client => $banner->{im_client},
            im_login => $banner->{im_login},
            contact_email => $banner->{contact_email},
            metro => $banner->{metro},
 
            (exists $banner->{ogrn} ? (ogrn => $banner->{ogrn}) : (ogrn => undef)),
        );
 
        # Пустые строки приравниваем к undef
        $vcard{$_} = undef for grep { defined $vcard{$_} && !length $vcard{$_} } keys %vcard;
        push @vcards, \%vcard;
    }
    return \@vcards;
}

#
# Внутренняя функция валидации визитки
#

sub _base_validate_contactinfo {
    my $banner = shift;
    my %options = @_;

    my $ipostfix = $options{api} ? '_api' : '';

    my %vcard = %{ _convert_vcards_for_validation([$banner])->[0] // {} };

    # Список ошибок
    my @result;

    # Валидация структуры
    for my $field (keys %VCARD_STRUCT) {
        if (exists $vcard{$field}) {
            my $p = $VCARD_STRUCT{$field};
            my $val = $vcard{$field};
            if (defined $val) {
                my $iname = $p->{"iname$ipostfix"};
                my $field_name = (defined $iname ? iget($iname) : $field);
                push @result, iget('Поле "%s" должно быть числом', defined $iname ? iget($iname) : $field)
                    if $p->{isa} && $p->{isa} eq 'Int' && $val !~ /^\d+$/;
                push @result, error_MaxLength(undef, field => '"'.$field_name.'"', length => $p->{length})->description
                    if $p->{length} && length($val) > $p->{length};
            }
        }
    }

    # Для валидации данных нужна валидная структура
    return @result if @result;

    # Валидация данных
    my $vcard = Direct::Model::VCard->new(%vcard);

    my $validation = Direct::Validation::VCards::validate_vcards([$vcard]);
    if (!$validation->is_valid) {
        my $vr0 = $validation->get_objects_results->[0];
        for my $field ($vr0->get_fields) {
            my $iname = $VCARD_STRUCT{$field}->{"iname$ipostfix"};
            if ($VCARD_STRUCT{$field}->{"subfields_iname$ipostfix"}) {
                for my $subfield ($vr0->get_field_result($field)->get_fields) {
                    my $iname = $VCARD_STRUCT{$field}->{"subfields_iname$ipostfix"}->{$subfield} // $VCARD_STRUCT{$field}->{"iname$ipostfix"};
                    push @result, process_text_template(
                        $vr0->get_field_result($field)->get_field_result($subfield)->get_first_error_description,
                        field => '"'.(defined $iname ? iget($iname) : $field).'"'
                    );
                }
                # Generic ошибки
                push @result, (map { process_text_template($_->description, field => '"'.(defined $iname ? iget($iname) : $field).'"') } @{$vr0->get_field_result($field)->get_generic_errors});
            } else {
                push @result, process_text_template($vr0->get_field_result($field)->get_first_error_description, field => '"'.(defined $iname ? iget($iname) : $field).'"');
            }
        }
        push @result, (map { $_->description } @{$vr0->get_generic_errors});
    }

    return @result;
}

=head2 validate_contactinfo

    Проверка, правильно ли заполнена контактная информация
    Принимает ссылку на хеш с баннером, возвращает массив ошибок валидации

=cut

sub validate_contactinfo {
    my $banner = shift;
    return _base_validate_contactinfo($banner);
}

=head2 validate_contactinfo_api

    Функция аналогична validate_contactinfo, только для API
    (по сравнению с оригинальной функцией изменены названия полей)

=cut

sub validate_contactinfo_api {
    my $banner = shift;
    return _base_validate_contactinfo($banner, api => 1);
}

=head2 get_common_contactinfo_for_camp(cid, options)

    Получает единую контактную информацию для кампании
    
    options:
        is_mediaplan - если нужно следует брать объявления из медиаплана
        skip_empty - НЕ учитывать учитывать объявления с пустой контактной инофрмацией
        dont_skip_arch - учитывать архивные баннеры
        org_details_separated -- флаг, если взведен -- вместо $vcard->{ogrn} будет хеш: $vcard->{org_details} = {...}

=cut
sub get_common_contactinfo_for_camp {
    my ($cid, $options) = @_;
    
    my $all_banners_vcards = get_vcards({cid => $cid}, {
        from_banners_only   => 1,
        only_if_common      => 1,
        only_non_arch       => ($options->{dont_skip_arch} ? 0 : 1),
        is_mediaplan        => $options->{is_mediaplan},
        skip_empty          => $options->{skip_empty},
    });

    my $contact_info;
    if ($all_banners_vcards) {
        if (@$all_banners_vcards) {
            # единая визитка
            $contact_info = $all_banners_vcards->[0];
        } else {
            # на кампании нет баннеров
            my $camp_ci_string = get_one_field_sql(PPC(cid => $cid), "SELECT contactinfo FROM camp_options WHERE cid = ?", $cid );
            $contact_info = deserialize($camp_ci_string);
        }
    }
    # else - нет ЕКИ: или разные визитки, или на каких-то баннерах они есть, а на каких-то нет

    # Преобразуем контактную информацию к формату, обрабатываемому вне модуля VCards
    if ($contact_info) {
            $contact_info->{worktimes} = get_worktimes_array( $contact_info->{worktime} ) if defined $contact_info->{worktime};

            if( $options->{org_details_separated} ){
                if ($contact_info->{ogrn}) {
                    # случай кампании без объявлений, ogrn взят из camp_options.contactinfo
                    separate_org_details($contact_info);
                }
                if ($contact_info->{org_details_id}) {
                    # случай обыкновенной визитки со ссылкой на org_details
                    expand_org_details($contact_info);
                }
            } elsif ( defined $contact_info->{org_details_id} ){
                my $org_details = get_org_details($contact_info->{org_details_id});
                $contact_info->{ogrn} = $org_details->{ogrn};
            } 
    }
    return $contact_info;
}

=head2 set_common_contactinfo

    Установить единую (общую, одинаковую) контактную информацию в кампании. 

    Берем объявления (все -- и те, у которых уже была КИ, и те, у которых не было) и записываем к ним новую КИ

    Если в $ci пустой хеш, то удаляем КИ из всех объявлений, у которых есть ссылка на сайт (=из тех объявлений, которые могут обойтись без КИ)

=cut
sub set_common_contactinfo {
    my ( $cid, $ci, $uid ) = @_;
    return 0 unless $cid && $cid =~ /^\d+$/;
    my ( $new_worktime, $new_phone, $new_contacts, $SQL_ADD ) = ( '', '', '', '' );
    my $new_contacts_for_mail;

    my $nonempty_ci = keys %{$ci}; 

    # К сожалению, нельзя использовать VCards::serialize в таком виде, как он есть, так как он фильтрует поля по @VCARD_FIELDS
    do_update_table(PPC(cid => $cid), 'camp_options', {contactinfo => ($nonempty_ci) ? YAML::Dump(hash_cut $ci, @$VCARD_FIELDS_FORM) : ''}, where => {cid => $cid});

    if( $nonempty_ci ) {
        $new_contacts = get_contacts_string( $ci, "\n" );
        $new_contacts_for_mail = get_contacts_string($ci, "\n", {dont_include_ids => 1});

        $ci->{org_details_id} = add_org_details(make_org_details_from_vcard($ci, { uid => $uid}));
        $ci->{phone} = compile_phone($ci);
    } else {
        $SQL_ADD = qq|(b.banner_type = 'dynamic' OR COALESCE(b.href, '') != '') and|;
    }

    if (get_one_field_sql(PPC(cid => $cid), qq{select 1 from banners where cid = ? and statusArch='No' and banner_type IN ('text', 'dynamic') limit 1}, $cid)) {
        # Выясняем, есть ли уже в кампании визитка с подходящими данными
        # если есть -- используем ее, нет -- записываем новую
        # кр. того, если есть -- проверяем, не поменялся ли geo_id или address_id
        # если поменялся -- все объявления с этой визиткой отправляем в БК
        #
        # при необходимости можно будет здесь же склеивать одинаковые визитки
        my $vcard_id = undef;

        if($nonempty_ci){
            $vcard_id = create_vcards($uid, [hash_merge({}, $ci, {uid => $uid, cid => $cid})])->[0]{vcard_id};
        }

        my $sql_bid = qq/   SELECT b.bid, b.title, b.title_extension, b.body, b.href, b.domain, b.phoneflag,
                                   vc.phone, vc.name, vc.contactperson, vc.worktime, vc.country, vc.city, vc.street, vc.house, vc.build, vc.apart, vc.metro, vc.geo_id, 
                                   vc.im_client, vc.im_login, vc.extra_message, vc.contact_email, vc.org_details_id
                              FROM banners b
                                   join phrases p using(pid)
                                   left join vcards vc on vc.vcard_id = b.vcard_id
                                   
                             WHERE $SQL_ADD p.cid = ? and b.statusArch = 'No' and (IFNULL(b.vcard_id, 0) != IFNULL(?, 0))
                                AND b.banner_type IN ('text', 'dynamic')/; 

        my %bids = ();
        my $bids_map_only = [];
        my $bids_geo_id_only = [];

        if($vcard_id){
            my $opt = get_one_line_sql(PPC(cid => $cid), "SELECT vc.geo_id, vc.address_id,
                                                     CONCAT_WS(',', maps.x, maps.y) as manual_point,
                                                     CONCAT_WS(',', maps.x1, maps.y1, maps.x2, maps.y2) as manual_bounds
                                                FROM vcards vc
                                           left join addresses adr on adr.aid = vc.address_id
                                           left join maps on maps.mid = adr.map_id
                                           left join maps maps_auto on maps_auto.mid = adr.map_id_auto 
                                               WHERE vc.vcard_id = ?", $vcard_id);
            
            $ci->{$_} ||= 0 foreach qw/geo_id address_id/;

            if ($opt->{geo_id} != $ci->{geo_id}
                || $opt->{address_id} != $ci->{address_id}
            ) {
                $bids_geo_id_only = get_one_column_sql(PPC(cid => $cid), "select bid from banners where vcard_id = ?", $vcard_id) || [];
            } else {
                my @map_fields = qw/manual_point manual_bounds/;
                foreach (@map_fields) {
                    $ci->{$_} = '' if !defined $ci->{$_};
                }
                if (grep { $ci->{$_} ne $opt->{$_} } @map_fields) {
                    $bids_map_only = get_one_column_sql(PPC(cid => $cid), "select bid from banners where vcard_id = ?", $vcard_id) || [];
                }
            }
        }

        my @bids_changed_no_moderate;
        my $sth = exec_sql(PPC(cid => $cid), $sql_bid, $cid, $vcard_id );
        while( my $row = $sth->fetchrow_hashref ) {
            foreach ( 'title', 'title_extension', 'body' ) {
                $row->{$_} = '' if !defined $row->{$_};
                html2string($row->{$_});
            }
            my $ci_copy = {%$ci};
            $ci_copy->{phoneflag} = $row->{phoneflag};
            
            my $is_vcard_changed = ModerateChecks::check_moderate_contactinfo($ci_copy, $row);

            my $old_contacts = get_contacts_string( $row, "\n" );
            my $old_contacts_for_mail = get_contacts_string($row, "\n", {dont_include_ids => 1});

            if ($is_vcard_changed) {
                my $title_extension_for_mail = (defined $row->{title_extension}) ? $row->{title_extension}.'\n' : '';
                $bids{ $row->{ bid } }{ old } = $row;
                $bids{ $row->{ bid } }{ old_text } = qq!$row->{title}\n$title_extension_for_mail$row->{body}\n$row->{href}\n$row->{domain}\n$old_contacts_for_mail!;
                $bids{ $row->{ bid } }{ new_text } = qq!$row->{title}\n$title_extension_for_mail$row->{body}\n$row->{href}\n$row->{domain}\n$new_contacts_for_mail!;
            } elsif ($old_contacts ne $new_contacts) {
                # визитка поменялась незначительно - нужно только переотправить в БК, но не на модерацию
                push @bids_changed_no_moderate, $row->{bid};
            }
        }

        #Если общая визитка была удалена, добавляем команду в очередь модерации
        if ( !$nonempty_ci && scalar keys %bids ) {
            delete_vcards_mod_versions([keys %bids]);
        }
        my $bids_str = join ',' , keys %bids;

        if (@bids_changed_no_moderate) {
            do_update_table(PPC(cid => $cid), 'banners', 
                { vcard_id => $vcard_id, statusBsSynced => 'No' },
                where => { cid => $cid, bid => \@bids_changed_no_moderate, statusArch => 'No' }
            );
        }

        if( $bids_str ne '' ) {
            my $ready_phone = $ci && $ci->{phone} ? 'Ready' : 'New'; 

            my $SQL = qq/   UPDATE banners  SET statusBsSynced = 'No', phoneflag = IF(statusModerate = 'New', 'New', "$ready_phone"), vcard_id = ?
                             WHERE cid = ? and bid in ( $bids_str ) and statusArch = 'No'/;
            do_sql(PPC(cid => $cid), $SQL, $vcard_id, $cid);

            clear_banners_moderate_flags([keys %bids]);

            foreach( keys %bids ) {
                mail_notification('banner', 'b_text', $_, $bids{ $_ }{ old_text } , $bids{ $_ }{ new_text }, $uid ) if $bids{ $_ }{ new_text } ne $bids{ $_ }{ old_text };
            }
        }

        # Если изменились только координаты на карте - отдельно переотправляем баннеры в БК
        my @bids_resend_only = grep { !$bids{ $_ } } @$bids_map_only;
        if (@bids_resend_only) {
            do_update_table(PPC(cid => $cid), 'banners', {statusBsSynced => 'No'}, 
                                    where => {bid => \@bids_resend_only,
                                              cid => $cid} );
        }

        if( @$bids_geo_id_only ) {
            do_update_table(PPC(cid => $cid), 'vcards', {geo_id => $ci->{geo_id}, address_id => $ci->{address_id}}, where => {cid => $cid , vcard_id => $vcard_id} );
            do_update_table(PPC(cid => $cid), 'banners', {statusBsSynced => 'No'}, where => {cid => $cid , bid => $bids_geo_id_only} );
            do_delete_from_table(PPC(cid => $cid), 'auto_moderate', where => {bid => $bids_geo_id_only});
        }
    } elsif (my $count_mediabanners = get_one_field_sql(PPC(cid => $cid), qq{select count(*) from mediaplan_banners where cid = ?}, $cid)) {
        my $vcard_id;
        if($nonempty_ci){
            # порядок важен - сначала визитка, потом uid и cid
            my $vcard_values = hash_cut(hash_merge({}, $ci, {uid => $uid, cid => $cid}), @$VCARD_FIELDS_DB);
            create_vcards($uid, [$vcard_values]);
            $vcard_id = $vcard_values->{vcard_id};
        }

        my $SQL = qq{   UPDATE mediaplan_banners 
                           SET vcard_id = ?
                         WHERE cid = ?};
        do_sql(PPC(cid => $cid), $SQL, $vcard_id, $cid);
    }
    
    # Чистим список ОГРН для пользователя
    clean_org_details(uid => $uid);

    return 1;
}

=head2 delete_vcard_from_db($vcard_ids, %options)

    удаляем записи из vcards,
      + удаляем привязаные записи из addresses
      + вызываем clean_org_details()

    $vcard_ids - или один vcard_id или ссылка на массив для нескольких визиток
    %options
        skip_org_details => 1 - не вызывать clean_org_details()

    Возвращает
        [vcard_id1, vcard_id2, vcard_id3 ... ] массив vcard_id удаленных визиток

=cut
sub delete_vcard_from_db {
    my ($vcard_ids, %options) = @_;
    $vcard_ids = [$vcard_ids] if ref($vcard_ids) ne 'ARRAY';

    my @deleted_ids;
    foreach_shard vcard_id => $vcard_ids, sub {
        my ($shard, $vcard_ids_chunk) = @_;

        # uid главного представителя
        my $vcards = get_hashes_hash_sql(PPC(shard => $shard), ["SELECT vcard_id, uid, address_id FROM vcards", WHERE => {vcard_id => $vcard_ids_chunk}]);
        do_sql(PPC(shard => $shard), [
            'DELETE vc
               FROM vcards vc
                    LEFT JOIN banners b ON vc.vcard_id = b.vcard_id
                    LEFT JOIN mediaplan_banners mb ON vc.vcard_id = mb.vcard_id',
              WHERE => {'vc.vcard_id' => $vcard_ids_chunk, 'b.bid__is_null' => 1, 'mb.mbid__is_null' => 1}
        ]);

        my $rest_vcards = get_one_column_sql(PPC(shard => $shard), ['SELECT vcard_id FROM vcards', WHERE => {vcard_id => $vcard_ids_chunk}]) || [];
        delete @{$vcards}{@$rest_vcards};
        
        my @deleted_vcards = values %$vcards;
        if (@deleted_vcards) {
            my (%addresses, %users);
            foreach (@deleted_vcards) {
                $addresses{$_->{address_id}} = 1 if $_->{address_id};
                $users{$_->{uid}} = 1;
            }
            
            # оставляем только те address_id, на которые не ссылаются (после удаления) никакие визитки
            do_sql(PPC(shard => $shard), [
                'DELETE addr
                   FROM addresses addr
                        LEFT JOIN vcards vc ON vc.address_id = addr.aid',
                  WHERE => {'vc.vcard_id__is_null' => 1, 'addr.aid' => [keys %addresses]}]
            );
            unless ($options{skip_org_details}) {
                clean_org_details(uid => $_) for keys %users;
            }
        }
        # чистим мета-базу
        push @deleted_ids, (map {$_->{vcard_id}} @deleted_vcards); 
    };

    delete_shard(vcard_id => \@deleted_ids); 
    return \@deleted_ids;
}

=head2 Функции преобразования между разлиными представлениями визитки

   make_org_details_from_vcard($vcard [ , $extra_data ] ) - создать flat_hash Детальной информации из представления визитки типа flat_hash 

=cut

=head2 make_org_details_from_vcard($vcard [ , $extra_data ] )

    Создать flat_hash детальной информации из flat_hash объёекта типа vcard

    Значимые поля подтягиваются из $vcard
    uid и org_details_id берутся оттуда же, или выбираются из хеша с дополнительной информацией

    Вопрос к ревьюерам:
    * Где лучше располагать эту фуфнкцию? В VCards или в OrgDetails ? 
    * Может быть правильнее унести OrgDetails в VCards::OrgDetails ?

=cut
sub make_org_details_from_vcard($;$) {
    my ($vcard, $extra_data) = @_;

    my $details = {
        ogrn  => $vcard->{ogrn},
    };
    hash_merge $details, (hash_cut $extra_data, qw/uid org_details_id/), (hash_cut $vcard, qw/uid org_details_id/);

    return $details;
}

=head2 separate_vcard

Перенести в хеше (баннере) все поля, которые относятся к визитке, в под-хеш vcard

=cut
sub separate_vcard {
    my ($banner) = @_;
    if (defined $banner->{phone}) {
        $banner->{vcard} = hash_cut $banner, @$VCARD_FIELDS_FORM;
        hash_merge $banner->{vcard}, parse_phone($banner->{vcard}->{phone});
        $banner->{vcard}->{worktimes} = get_worktimes_array( $banner->{vcard}->{worktime} ) if defined $banner->{vcard}->{worktime};
        delete $banner->{$_} for @$VCARD_FIELDS_FORM;
        expand_org_details($banner->{vcard});
    }
}

=head2 separate_vcard_multi

    Перенести в хеше (баннере) все поля, которые относятся к визитке, в под-хеш vcard

=cut

sub separate_vcard_multi {
    
    my $banners = shift;
    
    my @org_details_ids = uniq grep {$_} map {$_->{org_details_id}} @$banners;
    my %org_details = map {
        ($_->{org_details_id} => $_)
    } @org_details_ids ? @{OrgDetails::get_org_details_multi(\@org_details_ids)} : ();
    
    foreach my $banner (@$banners) {
        next unless $banner->{phone};
        my $org_details_id = delete $banner->{org_details_id};
        separate_vcard($banner);
        $banner->{vcard}->{org_details} = $org_details{$org_details_id} if $org_details_id; 
    }
    
    return $banners;
}


=head2 check_add_vcard_geo_id

    Проверяет/восстаналивает значение поля geo_id в визитке.
    Принимает ссылку на хеш с визиткой/плоским_баннером И МОДИИФИЦИРУЕТ его.

    Если geo_id отсутствует или равно 0, и задан город - пытаемся определить geo_id по названию города с помощью GeoTools
    Если город не определен - записывает geo_id = 0

=cut
sub check_add_vcard_geo_id {
    my $vcard = shift;

    if ( !defined $vcard->{geo_id} || !$vcard->{geo_id} ) {
        if ( defined $vcard->{city} && $vcard->{city} ne '' ) {
            $vcard->{geo_id} = get_geoid_by_cityname( $vcard->{city} );
        } else {
            $vcard->{geo_id} = 0;
        }
    }
}

=head2 user_has_accepted_vcard 

    Есть ли у пользователя хотя бы одна промодерированная визитка?

=cut
sub user_has_accepted_vcard($) {
    my $uid = shift;
    return get_one_field_sql(PPC(uid => $uid), [
          "SELECT STRAIGHT_JOIN 1
             FROM vcards vc
                  JOIN banners b ON b.vcard_id = vc.vcard_id",
            where => {
              'vc.uid'      => $uid,
              'b.phoneflag' => 'Yes',
            },
            "LIMIT 1"
        ]
    ) || 0;
}

=head2 delete_vcards_mod_versions 

    Удаляет по заданным bid-ам версии модерации визиток
    Используется при отвязке визитки от баннера

=cut

sub delete_vcards_mod_versions {
    my $bids = shift;

    do_delete_from_table(PPC(bid => $bids), 'mod_object_version', where => { obj_id => SHARD_IDS, obj_type => 'contactinfo'});
}

=head2 dissociate_vcards

Обновляет у визиткок временную метку о факте их крайней отвязки от баннера.

    dissociate_vcard($vcard_id);

или

    dissociate_vcard([$vcard_id_1, $vcard_id_2, ...]);

=cut
sub dissociate_vcards {
    my ($vcard_id) = @_;

    do_update_table(PPC(vcard_id => $vcard_id), 'vcards',
                    {
                        LastChange__dont_quote => 'LastChange',
                        last_dissociation__dont_quote => 'NOW()',
                    },
                    where => {vcard_id => SHARD_IDS});
}

=head2 set_vcards_validation_result($vcards, $opts)

    обертка над JavaIntapi::ValidateVcards
    принимает массив (хешей) визиток (вида как в validate_contactinfo), 
    для каждого элемента этого массива складывает массив текстов ошибок валидации в _vcard_validation_result

=cut

sub set_vcards_validation_result {
    my ($vcards_orig, $opts) = @_;
    unless ( Property->new($USE_JAVA_VCARDS_IN_SMART_PROPERTY_NAME)->get() 
            && $vcards_orig && @$vcards_orig) {
        return;
    }

    my $vcards = _convert_vcards_for_validation($vcards_orig);
    $_->{campaign_id} = $opts->{campaign_id} for @$vcards;

    my $vcards_vr = JavaIntapi::ValidateVcards->new(items => $vcards, 
            operator_uid => $opts->{operator_uid}, client_id => $opts->{client_id})->call();

    my @generic_error_descriptions = map { $_->description } @{ $vcards_vr->get_generic_errors() };
    foreach my $i (0..$#$vcards_orig) {
        my $vr = $vcards_vr->get_nested_vr_by_index($i) // $vcards_vr->next();
        $vcards_orig->[$i]->{_vcard_validation_result} = [@generic_error_descriptions, @{ $vr->get_error_descriptions }];
    }
}

1;
