use strict;
use warnings;
use utf8;

package VIPMediaplanXLS;

=head1 NAME
    
    VIPMediaplanXLS

=head1 DESCRIPTION

    Выгрузка медиаплана для кампаний с большим количеством фраз, в делении по отраслям

=cut

use Yandex::I18n;
use Yandex::ReportsXLS;

use Settings;
use GeoTools;
use Currencies;
use Currency::Pseudo;
use URLDomain qw/clear_banner_href/;

use List::MoreUtils qw(uniq);
use POSIX qw/strftime/;

=head2 new(%param)

    Конструктор

    Параметры именованные:
        draft - ссылка на хеш с данными по фразам/категориям
            geo - регионы показа
            pseudo_currency_id - валюта
            campaign_name - имя кампании
            arr - ссылка на массив хешей баннеров, содержит
                title - заголовок баннера
                Phrases - ссылка на массив хешей поисковых фраз/категорий, содержит:
                    category_name - название категории
                    phrase - поисковая фраза
                    place - рекомендуемая позиция
                    p_clicks/p_ctr/pmin - клики/ctr/cpc в спецразмещении
                    fp_clicks/fp_ctr/max - клики/ctr/cpc на 1-ом месте
                    clicks/ctr/min - клики/ctr/cpc в гарантии
                    shows - число показов

=cut

sub new {
    my ($class, %option) = @_;

    $option{format} = {
        head => {
            size => 8, bold => 1, valign => 'vcenter',
            align => 'center', border => 1, fg_color => 22,
            text_wrap => 1, pattern => 1
        },
        head_link => {
            size => 8, bold => 1, valign => 'vcenter',
            align => 'center', border => 1, fg_color => 22,
            text_wrap => 1, pattern => 1, underline => 1
        },
        simple_link => {
            size => 8, align => 'center', valign => 'vcenter',
            border => 1, num_format => '0', text_wrap => 1, underline => 1
        },
        head_blue => {
            size => 8, bold => 1, valign => 'vcenter', bg_color => 61,
            align => 'center', border => 1,
            text_wrap => 1, pattern => 1
        },
        head_blue_left => {
            size => 8, bold => 1, valign => 'vcenter', bg_color => 61,
            align => 'left', border => 1,
            text_wrap => 1, pattern => 1
        },
        price_blue => {
            size => 8, bold => 1, valign => 'vcenter', bg_color => 61,
            align => 'center', border => 1, num_format => '0.00',
            text_wrap => 1, pattern => 1
        },
        top_head => {
            size => 8, bold => 1, valign => 'vcenter',
            align => 'center', fg_color => 22, text_wrap => 1, pattern => 1,
            top => 2, left => 1, right => 1, bottom => 1
        },
        right_head => { 
            size => 8, bold => 1, valign => 'vcenter',
            align => 'center', border => 1, fg_color => 22,
            text_wrap => 1, pattern => 1, top => 2, right => 2
        },
        sheet_name => {size => 12, bold => 1, align => 'left', valign => 'vcenter', left => 0, bottom => 0, top => 2, right => 2},
        sheet_name_wb => {size => 12, bold => 1, align => 'left', valign => 'vcenter', border => 0},
        right_border => {right => 2, top => 0, left => 0, bottom => 0},
        sheet_params => {size => 8, bold => 1, align => 'left', valign => 'vcenter', left => 0, top => 0, bottom => 0, right => 2},
        phrase_group => {size => 9, bold => 1, align => 'left', valign => 'vcenter', top => 2, left => 2, right => 2, bottom => 1},
        total_group => {size => 8, bold => 1, align => 'left', valign => 'vcenter', border => 2},
        simple => {
            size => 8, align => 'center', valign => 'vcenter',
            border => 1, num_format => '0', text_wrap => 1
        },
        simple_right => {
            size => 8, align => 'center', valign => 'vcenter',
            num_format => '0', text_wrap => 1, left => 1, top => 1, bottom => 1, right => 2
        },
        simple_bottom => {
            size => 8, align => 'center', valign => 'vcenter',
            num_format => '0', text_wrap => 1, bottom => 2, left => 1, right => 1, top => 1
        },
        price_bottom => {
            size => 8, align => 'center', num_format => '0.00', 
            text_wrap => 1, valign => 'vcenter', bottom => 2, left => 1, right => 1, top => 1,
        },
        price_right => {
            size => 8, align => 'center', num_format => '0.00',
            text_wrap => 1, valign => 'vcenter', bottom => 2, right => 2, left => 1, top => 1,
        },
        price => {size => 8, align => 'center', border => 1, num_format => '0.00', text_wrap => 1, valign => 'vcenter'},
        simple_silver => {
            size => 8, align => 'center', valign => 'vcenter',
            border => 1, num_format => '0', text_wrap => 1, color => 'silver'
        },
        price_silver => {
            size => 8, align => 'center', border => 1,
            num_format => '0.00', text_wrap => 1, valign => 'vcenter', color => 'silver'
        },
        simple_left => {size => 8, align => 'left', border => 1, text_wrap => 1, valign => 'vcenter'},
        phrase => {size => 8, align => 'left', border => 1, text_wrap => 0, valign => 'vcenter'},
#        without border
        simple_wb => {size => 8, align => 'left', border => 0, border_color => 'white', text_wrap => 1, valign => 'vcenter'},
        simple_bold_wb => {size => 8, align => 'left', border => 0, border_color => 'white', text_wrap => 1, valign => 'vcenter', bold => 1},
        selected_simple => {
            size => 8, align => 'center', border => 1, num_format => '0', bg_color => 59, valign => 'vcenter'
        }, 
        selected_price => {
            size => 8, align => 'center', border => 1, num_format => '0.00', bg_color => 59, valign => 'vcenter'
        },
        ($option{format} ? %{$option{format}} : ())
    };
    $option{query} ||= {};
    $option{plan} ||= {};
    $option{use_nds} = ($option{draft}->{currency} eq "YND_FIXED") ? iget("с учетом НДС") : iget("без НДС");
    bless {%option}, ref $class || $class;
}

use constant MONTHS => [iget_noop('январь'), iget_noop('февраль'), iget_noop('март'), iget_noop('апрель'),
    iget_noop('май'), iget_noop('июнь'), iget_noop('июль'), iget_noop('август'), iget_noop('сентябрь'), iget_noop('октябрь'),
    iget_noop('ноябрь'), iget_noop('декабрь')];
    
use constant DECLINE_MONTHS => [iget_noop('января'), iget_noop('февраля'), iget_noop('марта'), iget_noop('апреля'),
    iget_noop('мая'), iget_noop('июня'), iget_noop('июля'), iget_noop('августа'), iget_noop('сентября'), iget_noop('октября'),
    iget_noop('ноября'), iget_noop('декабря')];
    
use constant POSITION => {
    1 => iget_noop('спецразмещение'),
    2 => iget_noop('1-ое место'),
    3 => iget_noop('гарантия'),
    99 => iget_noop('Сети'),
};    

sub _regions {
    
    my $self = shift;
        
    unless ($self->{_regions}) {
        my $draft = $self->{draft};
        $self->{_regions} = iget('Регионы показа: ') . join '; ', uniq map {get_geo_names($$_{geo}, ', ')} $draft->{arr} ? @{$draft->{arr}} : ($draft) 
    }
    return $self->{_regions};
}

sub _exchange {
    
    my $self = shift;
    my $pseudo_currency = $self->{draft}->{pseudo_currency} || get_pseudo_currency(id => $self->{draft}->{pseudo_currency_id});
    if ($self->{draft}->{currency} eq "YND_FIXED") {
        return $pseudo_currency->{rate}, $pseudo_currency->{name};
    }
    return 1, Currencies::get_currency_constant($self->{draft}->{currency}, 'name');
}

sub _banners_with_pos {
    
    my $self = shift;
    
    my $draft = $self->{draft};
    my (@banners, %position);
    foreach my $row (@{$draft->{arr}}) {
        my @phrases;
        foreach (@{$row->{Phrases}}) {
            $_->{pos} = 4 - (int($_->{place}) || 2)  unless $_->{pos};
            next unless exists POSITION->{$$_{pos}};
            push @phrases, $_;
            $position{$$_{pos}} = 1;
        }

        next unless @phrases;
        push @banners, {
            (map {$_ => $row->{$_}} qw/href title vcard_id body/),
            phrases => \@phrases
        }
    }
    
    return wantarray ? (\@banners, \%position) : \@banners;
}


=head2 queries()

    Страница "Запросы". 
    Поисковые запросы/категории в делении по отраслям.

=cut

sub queries {
    
    my $self = shift;
    my ($draft, %format) = ($self->{draft}, %{$self->{format}});
    
    foreach (qw/simple price simple_left/) {
        $format{"total_$_"} = {%{$format{$_}}, size => 10, bold => 1, italic => 1, border => 2};
        $format{"interim_$_"} = {%{$format{$_}}, bold => 1, border => 2};
    }
    foreach (qw/simple price selected_simple selected_price head head_blue/) {
        $format{"right_$_"} = {%{$format{$_}}, right => 2};
        $format{"left_$_"} = {%{$format{$_}}, left => 2};
    }
    
    my @exchange = $self->_exchange;
    my (@merge, @total);
    my $cnt = 0;
        
    my @statics = (
        iget('Средние цены за клик указаны по состоянию на %s и могут быть изменены без предварительного уведомления.', strftime('%d.%m.%Y', localtime)),
        iget('*      Число показов и переходов указано по данным за предыдущий месяц.'),
        iget('**    Реальный бюджет может отличаться от прогнозируемого, т.к. он подсчитан на основе анализа ставок конкурентов и CTR их кампаний, а эти параметры могут изменяться в процессе работы вашей рекламной кампании.')
             . ' ' . iget('Кроме этого, в прогнозе бюджета не учитываются показы объявлений в сети (Рекламная сеть Яндекса и внешние сети) и настройки временного таргетинга.'),
        iget('В случае, если цена за клик окажется недостаточной для входа в гарантированные показы, реальное количество показов и бюджет по соответствующим словам (словосочетаниям) может оказаться существенно меньше прогнозируемого.'),
        iget('Медиаплан состоит из смысловых групп запросов. В рамках подготовки и проведения рекламной кампании наборы слов, входящих в смысловые группы, могут изменяться для повышения эффективности и актуальности запросов.')
    );   
    
    my $position;
    ($self->{_banners}, $position)  = $self->_banners_with_pos;
    @$position{qw/1 3/} = (1, 1);

    my @available_pos = sort {$a <=> $b} keys %$position;
    my @available_cols = map {my $n = ($_-1) * 4; ($n..$n+3)} @available_pos;
    my $merge_col = 3 + @available_pos * 4 - 1;
    my $end_group;
    
    $format{rt_head_blue} = {%{$format{head_blue}}, right => 2, top => 2, bottom => 1};
    delete $format{head_blue}->{border};
    
    $format{tb_head_blue} = {%{$format{rt_head_blue}}, left => 2, bottom => 1}; 
    
    $format{position} = {
        size => 8, bold => 1, valign => 'vcenter', bg_color => 61,
        align => 'center', text_wrap => 1, pattern => 1,
        top => 2, right => 1, left => 2, bottom => 1
    };
    $format{clicks} = {
        size => 8, bold => 1, valign => 'vcenter',
        align => 'center', fg_color => 22,
        text_wrap => 1, pattern => 1,
        left => 2, top => 0, right => 1,
        bottom => 2,
    };
    
    my ($premium, $guarantee, $first_place)
        = (iget('спецразмещение'), iget('гарантия'), iget('1-ое место'));
    my $data = [
        [
            {image => {filename => "$Settings::ROOT/data/t/logo-big-txt.bmp"}}
        ],
        [
            {data => iget('Предложение смысловых групп слов для рекламной кампании ') . ($draft->{campaign_name} ? qq["$$draft{campaign_name}"] : iget('на Яндекс.Директе')), format => $format{sheet_name}},
            ({data => undef, format => $format{sheet_name}}) x $merge_col
        ],
        [
            {data => iget('Срок кампании: 1месяц'), global_format => $format{sheet_params}},
            ({data => undef, format => $format{right}}) x $merge_col
        ],
        [$self->_regions, ({data => undef, format => $format{right}}) x $merge_col],
        [
            {data => iget('Рекомендуемые поисковые запросы'), global_format => undef, format => $format{top_head}},
            {data => iget('Рекомендуемая позиция'), format => $format{top_head}},
            {data => iget('Количество показов в месяц* (прогноз)'), format => $format{top_head}},
            (exists $position->{1} ? (
                {data => iget('Премиум-позиция (Спецразмещение)'), format => $format{position}},
                ({data => undef, format => {top => 2, left => 1, right => 1, bottom => 1}}) x 2,
                {data => undef, format => {right => 2, left => 1, top => 2, bottom => 1}}
            ) : ()),
            (exists $position->{2} ? (
                {data => iget('1-ое место'), format => $format{position}},
                ({data => undef, format => $format{tb_head_blue}}) x 2,
                {data => undef, format => $format{rt_head_blue}}
            ) : ()),
            (exists $position->{3} ? (
                {data => iget('Позиция под результатами выдачи на первой странице (Гарантия)'), format => $format{position}},
                ({data => undef, format => $format{tb_head_blue}}) x 2,
                {data => undef, format => $format{rt_head_blue}}
            ) : ()),
        ],
        [({data => undef, format => $format{top_head}}) x 3, ({data => iget('Примерное количество переходов*'), format => $format{clicks}},
            {data => iget('Прогноз CTR'), format => $format{head}},
            {data => iget('Примерная цена клика, %s', $exchange[1]), format => $format{head}},
            {data => iget('Примерный бюджет**, %s', $exchange[1]), format => $format{right_head}}) x scalar(@available_pos)
        ],
        (map { ## no critic (ProhibitComplexMappings)
            my $row = $_;
            my @banner;
            ++$cnt;
            push @merge, {col1 => 0, row1 => "qr$cnt", col_count => $merge_col, row_count => 0};
            push @banner, [
                {data => $row->{title}, save_cell => "banner$cnt", save_row => "qr$cnt", format => $format{phrase_group}},
                ({data => undef, format => $format{phrase_group}}) x $merge_col
            ];
            $row->{xls_link} = 'A' . ($cnt + 6);
            my $start_group = $cnt;

            push @banner, map { ## no critic (ProhibitComplexMappings)
                $cnt++;
                my $word = $_;
                my %phrase = map {$_ => {
                    (/min|max/ ? (formula => sprintf '=%f*%f', $word->{$_}, $exchange[0]) : (data => $word->{$_})),
                    format => /price|min|max|ctr/ ? 'price' : 'simple',
                    save_cell => "${_}$cnt",
                }} qw/p_clicks fp_clicks clicks p_ctr fp_ctr ctr pmin max min/;
          
                $phrase{pos} = iget($word->{pos} == 1 ? 'спецразмещение' : ($word->{pos} == 2
                    ? '1-ое место' : 'гарантия'));
                
                [
                    {data => $_->{category_url} ? iget('Рубрика: ') . $_->{category_name} : $_->{phrase}, format => $format{phrase}},
                    {data => $phrase{pos}, format => $format{simple}, save_cell => "pos$cnt"},
                    {data => $word->{shows}, save_cell => "shows$cnt", format => $format{right_simple}},
                    ((map {                        
                        my $key = $_ eq 'p_clicks' ? 'left_' : '';
                        $key .= ($word->{pos} == 1 ? 'selected_' : '');
                        $phrase{$_}->{format} = $format{$key . $phrase{$_}->{format}};
                        $phrase{$_};
                    } qw/p_clicks p_ctr pmin/),
                    {formula => "{p_clicks$cnt}*{pmin$cnt}", format => $format{$word->{pos} == 1 ? 'right_selected_price' : 'right_price'}, save_cell => "p_budget$cnt"},
                    (map {
                        $phrase{$_}->{format} = $format{($word->{pos} == 2 ? 'selected_' : '') . $phrase{$_}->{format}};
                        $phrase{$_};
                    }   qw/fp_clicks fp_ctr max/),
                    {formula => "{fp_clicks$cnt}*{max$cnt}", format => $format{$word->{pos} == 2 ? 'right_selected_price' : 'right_price'}, save_cell => "fp_budget$cnt"},
                    (map {
                        $phrase{$_}->{format} = $format{($word->{pos} == 3 ? 'selected_' : '') . $phrase{$_}->{format}};
                        $phrase{$_};
                    } qw/clicks ctr min/),
                    {formula => "{clicks$cnt}*{min$cnt}", format => $format{$word->{pos} == 3 ? 'right_selected_price' : 'right_price'}, save_cell => "budget$cnt"},
                    )[@available_cols]                    
                ]
            } sort {
                ($a->{category_name} ? 1 : 0) <=> ($b->{category_name} ? 1 : 0)
                    || lc($a->{phrase} || $a->{category_name} || '') cmp lc($b->{phrase} || $b->{category_name} || '')
            } @{$row->{phrases}};
            $start_group++;
            $end_group = $cnt;

            push @total, $cnt += 2;
            (@banner, [
                {data => iget('Итого по группе для рекомендованных позиций**'), format => $format{total_group}, save_row => "group_total_$cnt"},
                {data => undef, format => $format{interim_simple}},
                {formula => "=SUM({shows$start_group}:{shows$end_group})", save_cell => "gs$cnt", format => $format{interim_simple}},
                ({
                    formula => qq[=SUMIF({pos$start_group}:{pos$end_group},"=$premium",{p_clicks$start_group}:{p_clicks$end_group})],
                    save_cell => "gpc$cnt", format => $format{interim_simple}
                },                
                {
                    formula => qq[=IF(SUMIF({pos$start_group}:{pos$end_group},"=$premium",{shows$start_group}:{shows$end_group})<>0,{gpc$cnt}/SUMIF({pos$start_group}:{pos$end_group},"=$premium",{shows$start_group}:{shows$end_group})*100,"-")],
                    format => $format{interim_price}
                }, 
                {formula => qq[=IF({gpc$cnt}<>0,{gpb$cnt}/{gpc$cnt},"-")], format => $format{interim_price}}, 
                {
                    formula => qq[=SUMIF({pos$start_group}:{pos$end_group},"=$premium",{p_budget$start_group}:{p_budget$end_group})],
                    save_cell => "gpb$cnt", format => $format{interim_simple}
                },
                {
                    formula => qq[=SUMIF({pos$start_group}:{pos$end_group},"=$first_place",{fp_clicks$start_group}:{fp_clicks$end_group})],
                    save_cell => "gfpc$cnt", format => $format{interim_simple}
                },
                {
                    formula => qq[=IF(SUMIF({pos$start_group}:{pos$end_group},"=$first_place",{shows$start_group}:{shows$end_group})<>0,{gfpc$cnt}/SUMIF({pos$start_group}:{pos$end_group},"=$first_place",{shows$start_group}:{shows$end_group})*100,"-")],
                    format => $format{interim_price}
                },                 
                {formula => qq[=IF({gfpc$cnt}<>0,{gfpb$cnt}/{gfpc$cnt},"-")], format => $format{interim_price}}, 
                {
                    formula => qq[=SUMIF({pos$start_group}:{pos$end_group},"=$first_place",{fp_budget$start_group}:{fp_budget$end_group})],
                    save_cell => "gfpb$cnt", format => $format{interim_simple}
                },                
                {
                    formula => qq[=SUMIF({pos$start_group}:{pos$end_group},"=$guarantee",{clicks$start_group}:{clicks$end_group})],
                    save_cell => "gminc$cnt", format => $format{interim_simple}
                },                
                {
                    formula => qq[=IF(SUMIF({pos$start_group}:{pos$end_group},"=$guarantee",{shows$start_group}:{shows$end_group})<>0,{gminc$cnt}/SUMIF({pos$start_group}:{pos$end_group},"=$guarantee",{shows$start_group}:{shows$end_group})*100,"-")],
                    format => $format{interim_price}
                }, 
                {formula => qq[=IF({gminc$cnt}<>0,{gminb$cnt}/{gminc$cnt},"-")], format => $format{interim_price}}, 
                {
                    formula => qq[=SUMIF({pos$start_group}:{pos$end_group},"=$guarantee",{budget$start_group}:{budget$end_group})],
                    save_cell => "gminb$cnt", format => $format{interim_simple}
                })[@available_cols]                
            ], [undef])
        } @{$self->{_banners}}),
        (@{$self->{_banners}} ? [
            {data => iget('Итого по кампании**:'), format => $format{total_simple_left}}, {data => undef, format => $format{total_simple}},
            {formula => '=SUM(' . join(',', map {"{gs$_}"} @total) . ')', save_cell => 'gs', format => $format{total_simple}},            
            ({formula => '=SUM(' . join(',', map {"{gpc$_}"} @total) . ')', save_cell => 'gpc', format => $format{total_simple}},
            {
                formula => qq[=IF(SUMIF({pos2}:{pos$end_group},"=$premium",{shows2}:{shows$end_group})<>0,{gpc}/SUMIF({pos2}:{pos$end_group},"=$premium",{shows2}:{shows$end_group})*100,"-")], 
                format => $format{total_price}
            },
            {formula => '=IF({gpc}<>0,{gpb}/{gpc},"-")', format => $format{total_price}},
            {formula => '=SUM(' . join(',', map {"{gpb$_}"} @total) . ')', save_cell => 'gpb', format => $format{total_simple}},            
            {formula => '=SUM(' . join(',', map {"{gfpc$_}"} @total) . ')', save_cell => 'gfpc', format => $format{total_simple}},
            {
                formula => qq[=IF(SUMIF({pos2}:{shows$end_group},"=$first_place",{shows2}:{shows$end_group})<>0,{gfpc}/SUMIF({pos2}:{shows$end_group},"=$first_place",{shows2}:{shows$end_group})*100,"-")], 
                format => $format{total_price}
            }, 
            {formula => '=IF({gfpc}<>0,{gfpb}/{gfpc},"-")', format => $format{total_price}},
            {formula => '=SUM(' . join(',', map {"{gfpb$_}"} @total) . ')', save_cell => 'gfpb', format => $format{total_simple}},            
            {formula => '=SUM(' . join(',', map {"{gminc$_}"} @total) . ')', save_cell => 'gminc', format => $format{total_simple}},
            {
                formula => qq[=IF(SUMIF({pos2}:{pos$end_group},"=$guarantee",{shows2}:{shows$end_group})<>0,{gminc}/SUMIF({pos2}:{pos$end_group},"=$guarantee",{shows2}:{shows$end_group})*100,"-")], 
                format => $format{total_price}
            },
            {formula => '=IF({gminc}<>0,{gminb}/{gminc},"-")', format => $format{total_price}},
            {formula => '=SUM(' . join(',', map {"{gminb$_}"} @total) . ')', save_cell => 'gminb', format => $format{total_simple}}
            )[@available_cols]
        ] : ()),
        [undef],
        [{data => $statics[0], format => $format{simple_bold_wb}}],
        (map {[{data => $_, format => $format{simple_wb}}]} @statics[1..@statics-1]),
    ];

    $cnt += @{$self->{_banners}} ? 8 : 7;
    my $l = 3;
    $self->{query} = {
        last => $cnt-1,
        banners_quantity => scalar(@{$self->{_banners}}),
        position => $position,
        available => \@available_pos,        
        map { ## no critic (ProhibitComplexMappings)            
            my @col;
            if (exists $position->{$_}) {                
                @col = ($_ => [$l, $l+3]);
                $l += 4;
            }
            @col
        } (1..3)
    };

    return ($data, {
        sheetname => iget('Запросы'),
        hide_gridlines => 2,
        set_color => [
            [62, 225, 225, 225],
            [61, 185, 205, 229],
            [59, 230, 224, 236],
        ],
        margins => [0.2, 0, 0, 0],
        freeze_panes => [6,0],
        sheetcolor => 'blue',
        merge_cells => [ 
            (map {{row1 => "group_total_$_", col1 => 0, row_count => 0, col_count => 1}} @total),
            (map {{row1 => $_, col1 => 0, row_count => 0, col_count => $merge_col}} (0..3)),
            (map {{row1 => 4, col1 => 3 + 4 * $_, row_count => 0, col_count => 3}} (0..$#available_pos)),
            {row1 => 4, col1 => 0, row_count => 1, col_count => 0},
            {row1 => 4, col1 => 1, row_count => 1, col_count => 0},
            {row1 => 4, col1 => 2, row_count => 1, col_count => 0},
            @merge,
            (map {{row1 => $cnt + $_, col1 => 0, row_count => 0, col_count => $merge_col}} (0..4))
        ],
        set_row => [
            {row => 0, height => 65},
            {row => 1, height => 20},
            {row => 2, height => 20},
            {row => 3, height => 20},
            {row => 4, height => 24},
            {row => 5, height => 37},
            map {
                {row => $cnt + $_, height => length $statics[$_] > 140 ? 10 + 15 * int(length($statics[$_])/140) : 17}
            } (0 .. @statics-1)
        ],
        set_column => [
            {col1 => 0, count => 0, width => 25},
            {col1 => 1, count => 0, width => 13},
            {col1 => 2, col2 => $merge_col, width => 11},
        ]
    })
}


=head2 annual_planning()

    Страница "Годовое планирование". 
    Рассчет бюджета на год исходя из бюджета на один месяц с учетом сезоных коэффициентов. 

=cut

sub annual_planning {
    
    my $self = shift;
    my ($draft, %format) = ($self->{draft}, %{$self->{format}});
    
    my @exchange = $self->_exchange;
    
    my $data = [
        [
            {image => {filename => "$Settings::ROOT/data/t/logo-big-txt.bmp"}}
        ],
        [
            {data => iget('Итоговый прогноз с учетом сезонных коэффициентов*'), format => $format{sheet_name}},
            ({data => undef, format => $format{sheet_name}}) x 14
        ],
        [
            {data => iget('Название рекламной кампании'), global_format => $format{top_head}, save_row => 'season', save_col => 'season_col'},
            {data => iget('Базовый бюджет на 1 месяц (%s)', $exchange[1]), save_col => 'base_budget'}, 
            (localtime)[5] + 1900, (undef) x 11,
            {data => iget('Суммарный бюджет на кампанию, %s (%s)', $exchange[1], $self->{use_nds}), save_col => 'sbudget', global_format => undef, format => $format{right_head}}
        ],
        [ 
            ({data => undef, format => $format{head}}) x 2,
            (map {{data => iget($_), format => $format{head}}} @{MONTHS()}),
            {data => undef, format => $format{right_head}},
        ],
        [
            {data  => $$draft{campaign_name}, global_format => undef, save_row => 'coeff', format => $format{simple_bottom}},
            {
                formula => sprintf('=%s!%s', iget('План'), $self->{plan}->{total_budget}), format => $format{price_bottom}},
                {data => undef, save_cell => 'm1', global_format => $format{price_bottom}
            },
            (undef) x 10, {data => undef, save_cell => 'm12'},
            {
                formula => sprintf('=%s!%s*SUM({m1}:{m12})', iget('План'), $self->{plan}->{total_budget}),
                global_format => undef, format => $format{price_right}
            }
        ],
        [{data => undef, global_format => undef, save_row => 'blank1'}],        
        [{
            data => iget('* Сезонные коэффициенты показывают динамику интереса аудитории к выбранной теме, в зависимости от времени года. Данные взяты в результате анализа статистики за последний год'),
            save_row => 'note_1',
            format => $format{simple_wb}
        }]
    ];
    
    return $data, {
        sheetname => iget('Годовое планирование'),
        hide_gridlines => 2,
        set_column => [
            {col1 => 'sbudget', count => 0, width => 38},
            {col1 => 'season_col', count => 0, width => 15},
            {col1 => 'base_budget', count => 0, width => 20},
        ],
        set_row => [
            {row => 0, height => 65},
            {row => 1, height => 20},
            {row => 'season', height => 38},
            {row => 'note_1', height => 28},
            {row => 'coeff', height => 18},
        ],
        merge_cells => [
            {row1 => 0, col1 => 0, col_count => 14, row_count => 0},
            {row1 => 1, col1 => 0, col_count => 14, row_count => 0},
            {row1 => 'note_1', col1 => 0, col_count => 14, row_count => 0},
            {row1 => 'blank1', col1 => 0, col_count => 14, row_count => 0},
            {row1 => 'season', col1 => 0, col_count => 0, row_count => 1},
            {row1 => 'season', col1 => 1, col_count => 0, row_count => 1},
            {row1 => 'season', col1 => 2, col_count => 11, row_count => 0},
            {row1 => 'season', col1 => 'sbudget', col_count => 0, row_count => 1},
        ]
    }    
}

=head2 plan()

    Страница "План". 
    Рассчет сум бюджета по позициям. Рассчет cpc, показов и кликов по каждой из выбранной позиции.

=cut

sub plan {
    
    my $self = shift;
    my ($draft, %format) = ($self->{draft}, %{$self->{format}});
    
    my %position = %{$self->{query}->{position}};
    my @available_pos = @{$self->{query}->{available}};
    push @available_pos, 99 if exists $position{3};
    my $is_first = 1;
    my @exchange = $self->_exchange;

    my @statics = (
        iget('Средние цены за клик указаны по состоянию на %s и могут быть изменены без предварительного уведомления.', strftime('%d.%m.%Y', localtime)),
        iget('*  Объявления, размещенные через Яндекс.Директ, могут быть показаны на всех страницах выдачи результатов поиска Яндекса по выбранному вами запросу, а также на страницах сайтов-участников сети, на страницах результатов поиска по Каталогу Яндекса, на страницах результатов поиска по Яндекс.Адресам и по Блогам, на страницах просмотра всех объявлений Яндекс.Директа по ключевым словам.'),
        iget('**   Подробное описание позиций представлено на листе "Общая информация"'),
        iget('***  Количество показов, кликов и средняя цена клика - прогнозируемые величины, и могут меняться в зависимости от поведения пользователей в части поисковых запросов, действий конкурентов и других факторов. Например, проведение рекламных акций в других медиа может повлиять на число запросов пользователей по рекламируемым товарам и услугам.'),
        iget('****  Реальный бюджет может отличаться от прогнозируемого, т.к. он подсчитан на основе анализа ставок конкурентов и CTR их кампаний, а эти параметры могут изменяться в процессе работы вашей рекламной кампании.')
    );    
    
    $format{loud_head} = {%{$format{head}}, top => 2, left => 1, right => 1, bottom => 1};
    delete $format{loud_head}->{border};
    $format{rloud_head} = {%{$format{loud_head}}, right => 2};
    $format{head_blue}->{bottom} = $format{head_blue}->{right} = 2; $format{head_blue}->{bold} = 1;
    $format{price_blue}->{border} = 2; $format{price_blue}->{bold} = 1;
    $format{loud_price} = {%{$format{price}}, top => 1, left => 1, right => 2, bottom => 1};
    delete $format{loud_price}->{border};
    
    my $columns = 7;
    my ($las_row, $query, $query_last) = (4, iget('Запросы'), $self->{query}->{last});
    
    my @localtime = localtime;
    $localtime[5] += 1900;
    
    my $banners_quantity = $self->{query}->{banners_quantity};
    my $data = [
        [
            {image => {filename => "$Settings::ROOT/data/t/logo-big-txt.bmp"}}, 
            (undef) x 5,
            {data => iget("Дата подготовки предложения\n%i %s %i года", $localtime[3], iget(DECLINE_MONTHS->[$localtime[4]]), $localtime[5]), format => {
                    size => 8, bold => 1, italic => 1, align => 'left', valign => 'top',
                    border => 0, num_format => '0', text_wrap => 1
            }}
        ],
        [
            {data => iget('План рекламной кампании ') . ($draft->{campaign_name} ? qq["$$draft{campaign_name}"] : iget('на Яндекс.Директе')), format => $format{sheet_name}},
            ({data => undef, format => $format{sheet_name}}) x $columns
        ],
        [
            {data => iget('Прогноз месячной статистики с учетом размещения на рекомендуемых позициях'), format => $format{sheet_name_wb}},
            ({data => undef, format => {top => 0, left => 0, bottom => 0, right => 2} }) x 7,
        ],
        [
            {data => iget('Срок кампании: 1месяц'), global_format => $format{sheet_params}},
            ({data => undef, format => $format{right_border}}) x 7,            
        ],
        [$self->_regions, ({data => undef, format => $format{right_border}}) x 7],
        [(map {{data => $_, global_format => undef, format => $format{loud_head}}}
            iget('Места размещения рекламных материалов'), 
            iget('Название рекламной кампании'),
            iget('Срок размещения'), 
            iget('Позиция**'),
            iget('Показы рекламных материалов***'), 
            iget('Переходы (клики)***'), 
            iget('Средняя цена за клик, %s (%s)***', $exchange[1], $self->{use_nds})),
            {data => iget('Месячный бюджет по позиции, %s ****', $exchange[1]), save_col => 'budget_by_pos', format => $format{rloud_head}},
            ({data => undef, format => $format{simple_wb}}) x 6
        ],
        (map { ## no critic (ProhibitComplexMappings)
            $las_row++;
            my $current = iget(POSITION->{$_});
            
            my @pos = (
                ($is_first ? (
                    {data => iget('Яндекс.Директ: Страницы результатов поиска + сети*'), global_format => $format{simple}}, $$draft{campaign_name}
                ) : ({data => undef, global_format => $format{simple}}, undef)),
                iget('4 недели'),
                $current,
            );
            $is_first = 0;
            # РСЯ рассчитываем как 20% от гарантии
            if ($_ == 99) {
                push @pos, (
                    {formula => '=ROUND({clicks99}/0.002, 0)', save_cell => 'show99', format => $format{simple}},
                    {formula => '=IF({price99}<>"-",ROUND({budget99}/{price99}, 0),0)', save_cell => 'clicks99', format => $format{simple}},
                    {formula => '={price3}', save_cell => 'price99', global_format => undef, format => $format{price}},
                    {formula => sprintf('=SUM({budget%i}:{budget%i})*0.2', $available_pos[0], $available_pos[-2]), save_cell => 'budget99', format => $format{loud_price}}
                );
            }
            else {
                # [shows_column, clicks_column, budget_column]
                my @diapason = @{$self->{query}->{$_}};
                my ($clicks, $budget) = (Yandex::ReportsXLS::_colnum2letter($diapason[0] || 1),
                    Yandex::ReportsXLS::_colnum2letter($diapason[1] || 1)); 
                push @pos, (                    
                    {
                        formula => $banners_quantity > 0
                            ? sprintf('=SUMIF(%s!B8:B%i, %s, %s!C8:C%i)', $query, $query_last, qq["=$current"], $query, $query_last)
                            : '=0',
                        save_cell => "show$_",
                        global_format => undef,
                        format => $format{simple},
                    },
                    {
                        formula => $banners_quantity > 0 ? "=$query!$clicks$query_last" : '=0',
                        save_cell => "clicks$_", format => $format{simple}
                    },
                    {formula => qq[=IF({clicks$_} <> 0, {budget$_}/{clicks$_}, "-")], save_cell => "price$_", format => $format{price}},
                    {
                        formula => $banners_quantity > 0 ? "$query!$budget$query_last]" : '=0',
                        save_cell => "budget$_", format => $format{loud_price}
                    },
                );
            }
            \@pos
        } @available_pos),
        [
            {data => undef, global_format => $format{simple_bottom}}, undef,
            {data => iget('4 недели'), global_format => $format{head_blue}}, iget('Всего по выбранным позициям'),            
            {data => "=SUM({show$available_pos[0]}:{show$available_pos[-1]})", save_cell => 'ptshow'},
            {data => "=SUM({clicks$available_pos[0]}:{clicks$available_pos[-1]})", save_cell => 'ptclicks'},
            {formula => '=IF({ptclicks} <> 0, {ptbudget}/{ptclicks}, "-")', global_format => undef, format => $format{price_blue}},
            {formula => "=SUM({budget$available_pos[0]}:{budget$available_pos[-1]})", save_cell => 'ptbudget', format => $format{price_blue}},
        ],
        [{data => undef, global_format => undef, save_row => 'blank1'}],
        [{data => $statics[0], format => $format{simple_bold_wb}}],
        (map {$las_row++; [{data => $_, global_format => undef, format => $format{simple_wb}}]} @statics[1..@statics-1])
    ];
    
    $self->{plan} = {begin => 7, end => 7 + $#available_pos, total_budget => 'H' . (8 + $#available_pos)};
    my $pos = scalar @available_pos;
    return ($data, {
        sheetname => iget('План'),
        hide_gridlines => 2,
        merge_cells => [
            {col1 => 0, row1 => 0, col_count => 5, row_count => 0},
            {col1 => 6, row1 => 0, col_count => 1, row_count => 0},
            (map {{col1 => 0, row1 => $_, col_count => 7, row_count => 0}} (1..4)),
            {row1 => 'blank1', col1 => 0, col_count => 7, row_count => 0},
            {row1 => 6, col1 => 0, col_count => 0, row_count => $pos},
            {row1 => 6, col1 => 1, col_count => 0, row_count => $pos},
            (map {{row1 => $las_row + $_, col1 => 0, row_count => 0, col_count => 7}} (0..$#statics)),
        ],
        set_color => [
            [62, 225, 225, 225]
        ],
        set_column => [
            {col1 => 0, count => 0, width => 24},
            {col1 => 1, col2 => 7, width => 12},
            {col1 => 'budget_by_pos', count => 0, width => 16}
        ],
        set_row => [
            {row => 0, height => 65},
            {row => 1, height => 20},
            {row => 2, height => 20},
            {row => 3, height => 20},
            {row => 4, height => 20},
            {row => 5, height => 37},
            (map {
                {row => $las_row + $_, count => 0, height => length $statics[$_] > 120 ? 10 + 15 * int(length($statics[$_])/120) : 17}
            } (0 .. @statics-1)),
            map {{row => 6 + $_, height => 35}} (0..$pos)
        ]
    })
}

=head2 strategy()

    Страница "Стратегия". 
    Распределение бюджета/кликов по доступным позициям.
    Распределение в виде круговой диаграммы.

=cut

sub strategy {
    
    my $self = shift;
    my ($draft, %format) = ($self->{draft}, %{$self->{format}});
    
    my @statics = (
        iget('    В качестве дополнительного средства оценки эффективности рекламных кампаний, а также для анализа посещаемости сайтов рекламодателей на Директе существует специальный инструмент — Метрика. Это бесплатный невидимый счетчик, который позволяет ответить на вопрос, что происходит на сайте рекламодателя после клика на рекламное объявление.'),
        iget('    Метрика работает по традиционному принципу онлайн-счетчиков: код, установленный на страницах вашего сайта, регистрирует каждое посещение, собирая данные о нем.')
             . " " . iget('Обладая информацией о показах вашей рекламы и кликах по ней, а также о посещаемости вашего сайта, Директ предоставляет статистику о привлеченных посетителях.')
             . " " . iget('Это дает возможность проводить анализ привлеченной аудитории в разрезах рекламных кампаний, объявлений, ключевых слов, регионов, площадок для выявления наиболее эффективных средств рекламы.')
    );
    
    my ($cnt, $query) = (0, iget('План'));
    my $name = $draft->{campaign_name} ? qq["$$draft{campaign_name}" ] : ''; 
    my @distribution = ($self->{plan}->{begin}, $self->{plan}->{end});

    my $data = [
        [{image => {filename => "$Settings::ROOT/data/t/logo-big-txt.bmp"}}],
        [{
            data => iget('Предлагаемая стратегия распределения бюджета РК %sна Яндекс.Директе', $name),
            format => $format{sheet_name_wb}
        }],
        [{
           data => iget('    Для кампании %sпредлагается следующее распределение бюджета, которое должно будет помочь достичь максимального эффекта с оптимальными затратами:', $name),
           format => $format{simple_wb},
        }],
        [{chart => {
            type => 'pie', 
            series => {
                categories => sprintf('=%s!$D$%i:$D$%i', $query, @distribution),
                'values' => sprintf('=%s!$H$%i:$H$%i', $query, @distribution),
                name => iget('Распределение бюджета по ключевым секторам'),
            },
        }}],
        ([undef]) x 19,
        [{chart => {
            type => 'pie', 
            series => {
                categories => sprintf('=%s!$D$%i:$D$%i', $query, @distribution),
                'values' => sprintf('=%s!$F$%i:$F$%i', $query, @distribution),
                name => iget('Распределение переходов по ключевым секторам'),
            },
        }}],
        ([undef]) x 19,
        (map {$cnt++; [{data => $_, save_row => "st$cnt", format => $format{simple_wb}}]} @statics),
    ];
    return $data, {
        sheetname => iget('Стратегия'),
        hide_gridlines => 2,
        sheetcolor => 'red',
        merge_cells => [
            {row1 => 0, col1 => 0, row_count => 0, col_count => 13},
            {row1 => 1, col1 => 0, row_count => 0, col_count => 13},
            {row1 => 2, col1 => 0, row_count => 0, col_count => 13},
            {row1 => 'st1', col1 => 0, row_count => 0, col_count => 13},
            {row1 => 'st2', col1 => 0, row_count => 0, col_count => 13}
        ],
        set_row => [
            {row => 0, height => 65},
            {row => 2, height => 40},
            {row => 'st1', height => 60},
            {row => 'st2', height => 60},
        ]
    };
}

=head2 texts()

    Страница "Тексты". 
    Заголовки и тексты баннеров медиаплана.

=cut

sub texts {
    
    my $self = shift;
    my ($draft, %format) = ($self->{draft}, %{$self->{format}});
    
    $format{text_bold} = {
        size => 8, align => 'left', valign => 'vcenter',
        border => 1, num_format => '0', text_wrap => 1, bold => 1
    };
    $format{text_italic} = {
        size => 8, align => 'center', valign => 'vcenter',
        border => 1, num_format => '0', text_wrap => 1, italic => 1
    };
    $format{stitalic} = {
        size => 8, align => 'left', valign => 'vcenter',
        num_format => '0', text_wrap => 1, italic => 1
    };
    $format{lower_bottom} = {bottom => 1, right => 1};
    
    my @statics = (
        iget('Объявление состоит из:'),
        iget('• заголовка (до 33 символов, включая пробелы и знаки препинания);'),
        iget('• текста (до 75 символов, включая пробелы и знаки препинания);'),
        iget('• ссылки на сайт (URL страницы на вашем сайте, содержащей рекламное предложение) и/или контактных данных.'),
        iget('*    Независимо от того, есть ли у вас сайт, можно заполнить форму с контактными данными. Тогда либо с заголовках объявления (если вы не указали адрес сайта), либо с отдельной дополнительной ссылки «Адрес и телефон» клик будет вести на виртуальную визитку – страницу с контактной информацией.'),
        '',
        iget('Пример поисковой выдачи с рекламными объявлениями'),
    );
    
    $format{head_right} = {%{$format{head}}, right => 2, left => 1, top => 1, bottom => 1};
    delete $format{head_right}->{border};
    $format{right_border}->{top} = 1;
    
    my ($cnt, $queries) = (0, iget('Запросы'));
    my $data = [
        [
            {image => {filename => "$Settings::ROOT/data/t/logo-big-txt.bmp"}}
        ],
        [
            {data => iget('Предложение текстовых блоков для рекламной кампании ') . ($draft->{campaign_name} ? qq["$$draft{campaign_name}"] : iget('на Яндекс.Директе')), format => $format{sheet_name}},
            ({data => undef, format => $format{sheet_name}}) x 3
        ],
        [{data => undef, save_row => 'blank1', format => $format{head_blue_left}}, ({data => undef, format => $format{right_border}}) x 3],
        [{data => iget('Цель:'), save_row => 'blank2', format => $format{head_blue_left}}, ({data => undef, format => $format{right_border}}) x 3],
        (map {
            my $row = $_;
            ++$cnt;
            ([
                {url => "internal:$queries!$$row{xls_link}", data => $row->{title}, global_format => $format{head_link}, save_row => "thead$cnt"},
                iget('Текст'), iget('Ссылка'), {data => iget('Использование визитки*'), global_format => $format{head_right}},
            ], [
                {data => iget('Заголовок'), global_format => undef, format => $format{text_italic}},
                {data => $row->{title}, format => $format{text_bold}},
                {url => clear_banner_href($$row{href}), data => clear_banner_href($$row{href}), format => $format{simple_link}, save_row => "href$cnt"},
                {data => iget($row->{vcard_id} ? 'да' : 'нет'), format => $format{simple_right}}
            ], [
                {data => iget('Текст'), format => $format{text_italic}}, {data => $row->{body}, format => $format{text_bold}}, 
                {data => undef, format => $format{lower_bottom}}, {data => undef, format => $format{simple_right}}
            ])
        } @{$self->{_banners}}),
        [({data => undef, format => {top => 2}}) x 4],
        map {[{data => $_, format => $format{stitalic}}]} @statics
    ];
    
    my $r = 7;
    my $last_row = $cnt*3 + 5;
    return $data, {
        sheetname => iget('Тексты'),
        hide_gridlines => 2,
        sheetcolor => 'green',
        merge_cells => [
            {row1 => 0, col1 => 0, row_count => 0, col_count => 3},
            {row1 => 1, col1 => 0, row_count => 0, col_count => 3},
            {row1 => 2, col1 => 0, row_count => 0, col_count => 3},
            {row1 => 3, col1 => 0, row_count => 0, col_count => 3},
            {row1 => 'blank1', col1 => 0, col_count => 3, row_count => 0},
            {row1 => 'blank2', col1 => 0, col_count => 3, row_count => 0},
            (map {
                {row1 => "href$_", col1 => 2, row_count => 1, col_count => 0},
                {row1 => "href$_", col1 => 3, row_count => 1, col_count => 0}
            } (1..$cnt)),
            map {{row1 => $last_row + $_, col1 => 0, row_count => 0, col_count => 3}} (0..$#statics)
        ],
        set_row => [
            {row => 0, height => 65}, {row => 1, height => 23},
            (map {{row => "thead$_", height => 5*$r}} (1..$cnt)), 
            map {
                {row => $last_row + $_, height => length $statics[$_] > 140 ? 10 + 15 * int(length($statics[$_])/140) : 17}
            } (0..$#statics)
        ],
        set_column => [
            {col1 => 0, count => 0, width => 2.61*$r},
            {col1 => 1, count => 0, width => 10*$r},
            {col1 => 2, count => 0, width => 4.37*$r},
            {col1 => 3, count => 0, width => 1.9*$r},
        ]
    };
}

=head2 info()

    Страница "Общая информация"(статический текст)

=cut

sub info {
    
    my $self = shift;
    my ($draft, %format) = ($self->{draft}, %{$self->{format}});
    
    
    $format{simple_wb}->{size} = 10;
    $format{simple_wb_small} = {
        size => 8, align => 'left', border => 0, border_color => 'white',
        text_wrap => 1, valign => 'vcenter'
    };
    
    my @statics = (
        iget('   Реклама по слову (или словосочетанию) будет показываться во всех и только в тех случаях, когда в запросе присутствует это слово (или словосочетание), а также в сети.')
             . " " . iget('Так, реклама по слову "мебель" будет показана и тому, кто спросил "мебель", и тому, кто спросил "каталог мебели".'),
        iget('   На страницах с результатами поиска Яндекса, поиска по Каталогу Яндекса и в рубриках каталога показывается до трех объявлений в Спецразмещении.')
             . " " . iget('Спецразмещение – это выигрышное рекламное место с высокой кликабельностью, расположенное над результатами поиска или над основным содержанием страницы в Яндекс.Каталоге.')
             . " " . iget('В Спецразмещении объявления показываются гарантированно при каждом запросе, полностью совпадающе см указанной рекламодателем ключевой фразой'),
        iget('   Гарантированные показы, "гарантия" — места для объявлений, расположенные под результатами поиска на 1-ой странице, где объявления показываются гарантированно каждый раз, когда поступает запрос по указанным рекламодателем ключевым словам или словосочетаниям.'),
        iget('   Минус-фразы — это фразы, по запросам с которыми рекламное объявление показываться не будет.')
             . " " . iget('Так, реклама по условию "шкаф -жарочный" будет показана по запросам "продажа шкафов", "шкаф-купе", но не будет показана по запросу "жарочный шкаф".'),
        iget('   Сеть — это группа сайтов, на которых показываются объявления Яндекс.Директа.')
             . " " . iget('В сеть принимаются только посещаемые и качественные сайты.')
             . " " . iget('После заявки на вступление в рекламную сеть каждый сайт проходит процедуру многоступенчатой проверки на предмет соответствия условиям участия.')
             . " " . iget('Яндекс не принимает в сеть сайты с некачественным контентом и сайты, созданные специально для заработка на размещении рекламы, поэтому вы можете быть уверены в качестве их аудитории.')
    );
    my $cnt = 0;
    my $data = [
        [{image => {filename => "$Settings::ROOT/data/t/logo-big-txt.bmp"}, global_format => $format{simple_wb}, save_row => "info$cnt"}],
        (map {$cnt++; [{data => $_, save_row => "info$cnt"}]} @statics),
        [undef], [undef],
        [{data => iget('Положение рекламного блока на спецразмещении:'), global_format => undef, format => $format{simple_wb_small}, save_row => 'info' . ($cnt + 1)}],
        [undef, {image => {filename => "$Settings::ROOT/data/t/special-banner.bmp"}}],
        ([undef]) x 16,
        [{data => iget('Положение рекламного блока в гарантии:'), global_format => undef, format => $format{simple_wb_small}, save_row => 'info' . ($cnt + 2)}],
        [undef, {image => {filename => "$Settings::ROOT/data/t/banner.bmp"}}],
        ([undef]) x 15,
        [{
            data => iget('Более подробную информацию о рекламе на Яндексе можно узнать на сайте http://direct.yandex.ru/help/'), 
            global_format => undef, format => $format{simple_wb_small},
            save_row => 'info' . ($cnt + 3)
        }]
    ];
    
    return $data, {
        sheetname => iget('Общая информация'),
        hide_gridlines => 2,
        merge_cells => [
            map {{row1 => "info$_", col1 => 0, row_count => 0, col_count => 12}} (0..$cnt+3) 
        ],
        set_row => [
            {row => 0, height => 65},
            map {
                {row => 1 + $_, height => length $statics[$_] > 110 ? 10 + 15 * int(length($statics[$_])/110) : 17}
            } (0..$#statics)
        ]
    };
}

=head2 as_xls()

    Выгрузка медиаплана в Excel. 
    Выходной поток напрявляется в STDOUT.

=cut

sub as_xls {
    
    my $self = shift;
  
    my (@formats, @sheets);
    ($sheets[3], $formats[3]) = $self->queries;
    ($sheets[1], $formats[1]) = $self->plan;
    ($sheets[0], $formats[0]) = $self->strategy;
    ($sheets[4], $formats[4]) = $self->texts;
    ($sheets[5], $formats[5]) = $self->info;    
    ($sheets[2], $formats[2]) = $self->annual_planning;
    $_->{print_orient} = 'landscape' foreach @formats;
    
    return Yandex::ReportsXLS->new()->array2excel2scalar(\@sheets, \@formats);
}

1;
