package VarsSchema;

=head1 NAME

    VarsSchema

=head1 DESCRIPTION

    Работа со схемой данных, передаваемых в шаблоны

=cut

use strict;
use warnings;
use utf8;

use English qw( -no_match_vars );
use File::Slurp;
use Data::Leaf::Walker;
use JSON;
use List::MoreUtils qw/part uniq/;
use Scalar::Util qw/looks_like_number/;
use Settings;
use Yandex::HashUtils qw/hash_merge/;


our $BLOCK_DIR ||= "$Settings::ROOT/data3/desktop.blocks";
our $BUNDLE_DIR ||= "$Settings::ROOT/data3/desktop.bundles";

our $NUMBER_KEY = 'NNNNN';


=head2 get_schema($value)

    Рекурсивно вычисляем схему, описывающую значение
    Допущения:
      - все массивы считаем однородными
      - все числовые ключи хеша дают доступ к однородным данным

    При описании хешей все числовые ключи описываем как свойство $NUMBER_KEY

=cut

sub get_schema
{
    my $value = shift;
    my $_path = shift || [];

    my $schema = {description => ''};
    if (ref $value eq 'HASH') {
        hash_merge $schema, {
            type => 'object',
            properties => {},
        };
        my ($numeric, $other) = part {looks_like_number $_ ? 0 : 1} keys %$value;
        if ($numeric) {
            $schema->{properties}->{$NUMBER_KEY} = generalize([map {get_schema($value->{$_}, [@$_path, $_])} @$numeric], [@$_path, $NUMBER_KEY]);
        }
        if ($other) {
            for my $key (@$other) {
                $schema->{properties}->{$key} = get_schema($value->{$key}, [@$_path, $key]);
            }
        }
    } elsif (ref $value eq 'ARRAY') {
        hash_merge $schema, {
            type => 'array',
        };
        if (@$value) {
            hash_merge $schema, {items => generalize([map {get_schema($value->[$_], [@$_path, $_])} 0 .. $#$value], [@$_path, 'i'])};
        }
    } elsif (looks_like_number $value) {
        hash_merge $schema, {
            type => 'number',
        };
    } else {
        hash_merge $schema, {
            type => 'string',
        };
    }
    return $schema;
}


=head2 generalize(\@schemas)

    Вычисляем обобщающую схему по массиву схем

    Строим пересечение и объединение properties,
    чтобы определить возможные и обязательные параметры

=cut

sub generalize
{
    my $schemas = shift;
    my $_path = shift || [];

    my @types = sort(uniq(map {$_->{type}} @$schemas));
    my $type = $types[0];
    if (@types == 2 && $types[0] eq 'number' && $types[1] eq 'string') {
        $type = 'string';
    } elsif (@types > 1) {
        die 'mixed types: ' . join(', ', map {"<$_>"} @types).' at '.join('.', @$_path);
    }

    my $result = {description => ''};

    if ($type eq 'string') {
        hash_merge $result, {type => 'string'};
    } elsif ($type eq 'number') {
        hash_merge $result, {type => 'number'};
    } elsif ($type eq 'object') {
        my $properties_slice = {};
        my $seen = {};
        my $required_intersection = {};
        for my $schema (@$schemas) {
            for my $key (keys %{$schema->{properties}}) {
                $seen->{$key} += 1;
                $properties_slice->{$key} ||= [];
                push @{$properties_slice->{$key}}, $schema->{properties}->{$key};
            }
            for my $key (@{$schema->{required}}) {
                $required_intersection->{$key} += 1;
            }
        }
        my $properties = {};
        for my $key (keys %$properties_slice) {
            $properties->{$key} = generalize($properties_slice->{$key}, [@$_path, $key]);
        }
        my $required = [
            sort
            grep {$_ ne $NUMBER_KEY}
            grep {$seen->{$_} == @$schemas}
            grep {!%$required_intersection || exists $required_intersection->{$_} && $required_intersection->{$_} == @$schemas}
            keys %$seen
        ];
        hash_merge $result, {type => 'object', properties => $properties, required => [sort @$required]};
    } elsif ($type eq 'array') {
        hash_merge $result, {type => 'array'};
        my $items_slice = [grep {$_} map {$_->{items}} @$schemas];
        if (@$items_slice) {
            hash_merge $result, {items => generalize($items_slice, [@$_path, 'i'])};
        }
    } else {
        die "unknown type <$type> at ".join('.', @$_path);
    }

    return $result;
}


=head2 merge_descriptions($dst, $src)

    Пополняем схему $dst заполненными описаниями из схемы $src

=cut

sub merge_descriptions
{
    my $dst = shift;
    my $src = shift;

    my $dst_walker = Data::Leaf::Walker->new($dst);
    my $src_walker = Data::Leaf::Walker->new($src);

    while (my ($k, $v) = $src_walker->each) {
        if ($k->[-1] eq 'description' && length $v) {
            if ($dst_walker->exists($k)) {
                $dst_walker->store($k, $v);
            }
        }
    }
}


=head2 logvars($name, $vars)

    Вычисляем тип $vars и логгируем в файл

=cut

sub logvars
{
    my ($name, $vars) = @_;

    my $schema = get_schema($vars);
    my $logdir = get_logdir();
    mkdir $logdir unless -d $logdir;
    write_file("$logdir/$name.$$", {append => 1, binmode => ':utf8'}, to_json($schema)."\n");
}


=head2 get_logdir

    Возвращаем путь к директории, в которой хранятся логи

=cut

sub get_logdir
{
    return "$Settings::LOG_ROOT/logvars";
}


=head2 validate_vars($block, $vars)

    Проверяем $vars на соответствие схеме блока $block
    Возвращаем ссылку на список ошибок

=cut

sub validate_vars
{
    my $block = shift;
    my $vars = shift;

    my $schema = read_schema($block);

    my $errors = [];
    validate_value($vars, $schema, [], $errors);
    return $errors;
}


=head2 validate_value($value, $schema, \@path, \@errors)

    Рекурсивно проверяем $value на соответствие схеме $schema

    Заполняем массив @errors
    @path содержит "абсолютный путь", чтобы ссылаться на $value в сообщениях об ошибках

=cut

{
my $rules = [
    {
        check => sub {ref $_[0] eq 'HASH'},
        type => 'object',
        nested => sub {
            my ($value, $schema, $path, $errors) = @_;

            for my $key (sort keys %$value) {
                my $subschema = looks_like_number $key
                                ? $schema->{properties}->{$NUMBER_KEY}
                                : $schema->{properties}->{$key};
                if ($subschema) {
                    validate_value($value->{$key}, $subschema, [@$path, $key], $errors);
                } else {
                    push @$errors, {path => $path, unexpected_property => $key};
                }
            }
            if (my $required = $schema->{required}) {
                if (my @absent = sort grep {not exists $value->{$_}} @$required) {
                    push @$errors, {path => $path, absent_properties => \@absent};
                }
            }
        },
    },
    {
        check => sub {ref $_[0] eq 'ARRAY'},
        type => 'array',
        nested => sub {
            my ($value, $schema, $path, $errors) = @_;

            if (my $subschema = $schema->{items}) {
                for my $i (0 .. $#$value) {
                    validate_value($value->[$i], $subschema, [@$path, $i], $errors);
                }
            }
        },
    },
    {
        check => sub {looks_like_number $_[0]},
        type => 'number',
    },
    {
        check => sub {1},
        type => 'string',
    },
];

sub validate_value
{
    my ($value, $schema, $path, $errors) = @_;

    for my $rule (@$rules) {
        next unless $rule->{check}->($value);
        if ($schema->{type} eq $rule->{type}) {
            if ($rule->{nested}) {
                $rule->{nested}->($value, $schema, $path, $errors);
            }
        } else {
            return if $schema->{type} eq 'string' && $rule->{type} eq 'number';
            push @$errors, {path => $path, expected => $schema->{type}, got => $rule->{type}};
        }
        return;
    }
}
}


=head2 read_schema($block)

    Читаем схему с диска

=cut

sub read_schema
{
    my $block = shift;

    my $filename = _get_filename($block);
    if (-f $filename) {
        my $schema_data = eval { from_json(scalar read_file($filename, binmode => ':utf8')) };
        die "invalid json in $filename: $EVAL_ERROR" if $EVAL_ERROR;
        return $schema_data;
    } else {
        return undef;
    }
}

=head2 read_schema_bundle()

    Чтение полной схемы с диска

=cut

sub read_schema_bundle
{

    my $filename = $BUNDLE_DIR . '/direct/direct.schema.js';
    if (-f $filename) {
        my $schema_data = eval { from_json(scalar read_file($filename, binmode => ':utf8')) };
        die "invalid json in $filename: $EVAL_ERROR" if $EVAL_ERROR;
        return $schema_data;
    } else {
        die "read_schema_bundle: file doesn't exist (${filename})";
    }
}

=head2 write_schema($block, $schema)

    Записываем схему на диск

=cut

sub write_schema
{
    my $block = shift;
    my $schema = shift;

    my $filename = _get_filename($block);
    write_file($filename, {atomic => 1, binmode => ':utf8'}, to_json($schema, {canonical => 1, pretty => 1}));
}


sub _get_filename
{
    my $block = shift;

    return "$BLOCK_DIR/$block/$block.schema.json";
}

1;
