package Wallet;

=head1 NAME

Wallet - работа с "единым счетом"
https://jira.yandex-team.ru/browse/DIRECT-19914

=cut

# $Id$

use Direct::Modern;

use Settings;
use Agency;
use Client;
use User;
use Campaign;
use Campaign::Types;
use Currencies;
use RBACElementary;
use RBACDirect;
use MoneyTransfer;
use Primitives;
use PrimitivesIds;
use Property;
use Common qw/:globals :subs/;
use BalanceQueue;
use TextTools;
use Stat::OrderStatDay;

use Direct::BillingAggregates;
use Direct::Model::Wallet;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::ScalarUtils;
use Yandex::DateTime qw/now/;
use Yandex::SendSMS;
use Yandex::TimeCommon qw/today yesterday mysql_round_day/;

use List::Util qw(sum min);
use List::MoreUtils qw(any uniq firstval none);

=head2 $DAILY_BUDGET_STOP_WARNING_TIME_PROP

    Название Property где хранится время, до которого нужно показывать предупреждение о том, что кампанию остановили слишком рано

=cut
our $DAILY_BUDGET_STOP_WARNING_TIME_PROP = 'DAILY_BUDGET_STOP_WARNING_TIME';

=head2 $DAILY_BUDGET_STOP_WARNING_TIME_PROP

    Умолчальное значение, если Property с именем $DAILY_BUDGET_STOP_WARNING_TIME_PROP не выставлена

=cut
our $DAILY_BUDGET_STOP_WARNING_TIME_DEFAULT = '16:00:00';

# --------------------------------------------------------------------

=head2 get_wallets_by_uids

Возвращаем кампании общие счета (+текстовые кампании) для клиента по типу сервисирования
    my $result = Wallet::get_wallets_by_uids($array_params, additional_wallet_fields => [qw/AgencyID currency/]);
    $array_params - массив из хешей с элементами
        c -- DirectContext по каждому клиенту
             $c - должен соответствовать одному UID (тот кто вызывает: менеджер, агентство или клиент)
        agency_client_id -- ClientID агентства (необязательный параметр)
        all_campaigns -- результат get_user_camps, если уже есть на момент доставания кошельков (необязательный параметр)
        client_currency -- валюта клиента [get_client_currencies($clientid)->{work_currency}]

    именнованные, не обязательные параметры:
        additional_wallet_fields -- ссылка на массив с именами ключей которые нужно дополнительно взять из кампании-счета из get_user_camps()
        get_sum_available -- нужно ли добавлять суммы доступные к переносу, используется в АПИ (1|0)
        with_nds — не вычитать НДС из сумм, используется в АПИ (1|0)
        with_bonus — добавлять скидочный бонус в суммы, используется в АПИ (1|0)
        with_outlay - добавлять информацию по расходу по кампаниям по счету (используется только на странице счета)
        with_emails_sms - добавлять email-ы клиента для уведомлений
        with_camp_stop_daily_budget_stats - добавлять в wallet_camp массив "camp_stop_daily_budget_stats" с временем отключения ОС по дневному бюджету за последние 14 дней
        force_pay_before_moderation - позволять оплату до модерации независимо от состояния опции feature_payment_before_moderation

    результат - массив хешей:
        [{
            wallet_cid -- # кампании счета
            uid -- uid главного представителя клиента
            agency_client_id -- ClientID агентства
            text_camps => [{}, {}, ...] -- список текстовых кампаний клиента по типу сервисирования
            wallet_camp => {} -- кампания счет, с дополнительными полями (время последнего включения, можно ли включить)
        }, ...]

        содержимое wallet_camp:
            enabled => 1
            , offer_sum_to_pay
            , outlay => {day => $sum_spents->{day}, week => $sum_spents->{week}, month => $sum_spents->{month}}
            , allow_pay
            , bonus
            , validEmails
            , sms_phone
            , cid
            , total
            , money_warning_value
            , sms_flags
            , sms_hour_from
            , sms_hour_to
            , sms_min_from
            , sms_min_to
            , email
            , day_budget => {daily_change_count => 0, recommended_sum => undef, show_mode => 'stretched', stop_time => '2017-04-14 13:45:00', sum => '12346.00'}
          если включена опция with_camp_stop_daily_budget_stats:
            , camp_stop_daily_budget_stats => ['2017-05-17 12:34:56', '2017-05-12 21:44:14']
            , camp_stop_daily_budget_warnings => ['2017-05-17 12:34:56']

        при отключенном счете
            enabled => 0
            , time_to_enable
            , allow_enable_wallet

=cut

sub get_wallets_by_uids($;%) {
    my $params = shift;
    my %OPT = @_;

    my $result = [];
    my $all_campaigns_cache = {}; # если запрашивается несколько кошельков одного клиента, то кампании достаточно загружать один раз
    my @agencies_client_id = uniq grep {$_} map { $_->{agency_client_id} } @$params;

    my $chief_reps_of_agencies = scalar @agencies_client_id
        ? rbac_get_chief_reps_of_agencies(\@agencies_client_id)
        : {};

    my $clients_nds_and_discount = {}; # кеш ндс и скидки по клиентам

    for my $param_row (@$params) {
        my ($c, $agency_client_id, $all_campaigns, $client_currency)
         = ($param_row->{c}, $param_row->{agency_client_id}, $param_row->{all_campaigns}, $param_row->{client_currency});

        my ($client_nds, $client_discount);

        my $client_data = get_client_data($c->client_client_id, ['feature_payment_before_moderation', 'debt', 'cashback_bonus', 'cashback_awaiting_bonus', 'country_region_id', 'is_brand', 'is_using_quasi_currency', 'auto_overdraft_lim', 'overdraft_lim', 'nextPayDate']);
        my $can_pay_before_moderation = $client_data->{feature_payment_before_moderation} || $OPT{force_pay_before_moderation};

        if (! $OPT{with_nds}) {
            if (defined $clients_nds_and_discount->{$c->client_client_id}->{nds}) {
                $client_nds = $clients_nds_and_discount->{$c->client_client_id}->{nds};
            } else {
                $client_nds = $clients_nds_and_discount->{$c->client_client_id}->{nds} = get_client_NDS($c->client_client_id, fetch_missing_from_balance => $can_pay_before_moderation);
            }
        }

        if ($OPT{with_bonus}) {
            if (defined $clients_nds_and_discount->{$c->client_client_id}->{discount}) {
                $client_discount = $clients_nds_and_discount->{$c->client_client_id}->{discount};
            } else {
                $client_discount = $clients_nds_and_discount->{$c->client_client_id}->{discount} = get_client_discount($c->client_client_id);
            }
        }

        if (! defined $all_campaigns) {
            if (exists $all_campaigns_cache->{ $c->client_chief_uid }) {
                $all_campaigns = $all_campaigns_cache->{ $c->client_chief_uid };
            } else {
                $all_campaigns
                = $all_campaigns_cache->{ $c->client_chief_uid }
                = get_user_camps(
                    $c->client_chief_uid
                    , mediaType => get_camp_kind_types('under_wallet'),
                    , client_nds => $client_nds
                    , client_discount => $client_discount
                    , can_pay_before_moderation => $can_pay_before_moderation
                    , without_spent_today => $OPT{without_spent_today}
                );
            }
        }

        my $wallet_campaigns = $all_campaigns->{wallet_campaigns};
        my ($wallet_camp, $campaigns, $wallet_var);

        if (! $agency_client_id) {

            # самоходный или сервисируемый общий счет
            $wallet_camp = firstval {! $_->{AgencyID}} @$wallet_campaigns;

            $campaigns = [
                grep {camp_kind_in(type => $_->{mediaType}, 'under_wallet')
                      && ! $_->{AgencyID}
                     }
                @{$all_campaigns->{campaigns}}
            ];

        } else {

            # счет для одного агентства (по FORM.AgencyID)
            $wallet_camp = firstval {$_->{AgencyID} && $_->{AgencyID} == $agency_client_id} @$wallet_campaigns;

            $campaigns = [
                grep {camp_kind_in(type => $_->{mediaType}, 'under_wallet')
                      && $_->{AgencyID}
                      && $_->{AgencyID} == $agency_client_id
                     }
                @{$all_campaigns->{campaigns}}
            ];

        }

        my $not_archived_campaigns = [grep {$_->{archived} eq 'No'} @$campaigns];

        my $subclient_info = {};
        if ($agency_client_id) {

            my %uids2clientids = ( $c->client_chief_uid => $c->client_client_id );

            my $all_subclients_info = Agency::get_client_agency_options(
                client_uids2_client_ids => \%uids2clientids,
                operator_uid => $chief_reps_of_agencies->{$agency_client_id}, # оптимизирует вызов на получение и фильтрацию по списку агентств
                not_filter_by_perm => 1,    # оптимизирует вызов на проверку прав оператора на агентство
                for_cur_agency => 1,        # оптимизируем вызов на получение списка агентств
            );

	        $subclient_info->{ $c->client_chief_uid } = $all_subclients_info->{ $c->client_client_id }->{ $agency_client_id };
        }

        if (ref($wallet_camp) eq 'HASH' && $wallet_camp->{cid} && (scalar(@$campaigns) > 0 || $can_pay_before_moderation)) {
            # общий счет есть и включен

            my $order_ids = [map {$_->{OrderID}} @$campaigns];

            my $date_to = now()->ymd("");

            # кампании, среди неархивных, которым не запретили оплату
            my @camps_not_forbidden_to_pay = grep {
                    $_->{statusNoPay} ne 'Yes'
                } @$not_archived_campaigns;

            # может ли логин класть деньги на ОС, если нет промодерированных кампаний
            my $login_can_pay_unmoderated_camps =
                $c->login_rights->{manager_control} ||
                $c->login_rights->{agency_control} ||
                $c->login_rights->{super_control};

            my $has_moderated_camps = (scalar(grep {
                    $_->{statusModerate} eq 'Yes'
                } @camps_not_forbidden_to_pay) ? 1 : 0);

            my $has_serviced_or_agency_camps = (scalar(grep {
                    $_->{AgencyID} ||
                    $_->{ManagerUID}
                } @camps_not_forbidden_to_pay) ? 1 : 0);

            my $allow_pay = $can_pay_before_moderation || (
                scalar(@camps_not_forbidden_to_pay) > 0 &&
                    (
                        $has_serviced_or_agency_camps ||
                        $has_moderated_camps ||
                        $login_can_pay_unmoderated_camps
                    )
            );

            # флажок, который говорит, что оплату можно будет сделать сразу после того, как какую-нибудь кампанию
            # промодерируют
            my $need_camp_moderation_to_pay = scalar(@camps_not_forbidden_to_pay) > 0 && !$allow_pay;

            if ($agency_client_id && $c->login_rights->{is_any_client}) {
                my $subclient_can_request_pay = $subclient_info->{$c->client_chief_uid}->{isSuperSubClient} ||
                    $subclient_info->{$c->client_chief_uid}->{allowTransferMoney};
                if (!$subclient_can_request_pay) {
                    $allow_pay = 0;
                    $need_camp_moderation_to_pay = 0;
                }
            }

            my $login_can_edit = rbac_is_owner_of_camps(undef, $c->UID, [$wallet_camp->{cid}]) &&
                rbac_user_allow_edit_camp(undef, $c->UID, $wallet_camp->{cid});


            my ($debt_without_nds, $overdraft_lim_without_nds, $auto_overdraft_lim_without_nds, $cashback_bonus_without_nds, $awaiting_cashback_bonus_without_nds) = (0, 0, 0, 0, 0);
            # добавляем/удаляем NDS в поля про деньги: долг, овердрафт, автовердрафт
            if (str($wallet_camp->{currency}) ne 'YND_FIXED') {
                $debt_without_nds = Currencies::remove_nds($client_data->{debt} // 0, $client_nds) // 0;
                $overdraft_lim_without_nds = Currencies::remove_nds($client_data->{overdraft_lim} // 0, $client_nds) // 0;
                $auto_overdraft_lim_without_nds = Currencies::remove_nds($client_data->{auto_overdraft_lim} // 0, $client_nds) // 0;
                $cashback_bonus_without_nds = Currencies::remove_nds($client_data->{cashback_bonus} // 0, $client_nds) // 0;
                $awaiting_cashback_bonus_without_nds = Currencies::remove_nds($client_data->{cashback_awaiting_bonus} // 0, $client_nds) // 0;
            }

            my $is_fake_nds = 0;
            # Клиенты с квазивалютами DIRECT-92263
            # Клиентам в квазивалюте НДС неприменим.
            if ((any { str($wallet_camp->{currency}) eq $_ } qw/BYN USD EUR TRY YND_FIXED/) ||   
                (str($wallet_camp->{currency}) eq 'KZT' && $client_data->{is_using_quasi_currency} == 1 && $client_data->{country_region_id} == $geo_regions::KAZ)) {
                $is_fake_nds = 1;
            }

            my $nontransferrable_sum = 0;
            if ($wallet_camp->{total} > $wallet_camp->{sum_balance}) {
                $nontransferrable_sum = $wallet_camp->{total} - $wallet_camp->{sum_balance};
            }

            $wallet_var = {
                enabled => 1
                , clientID => $c->client_client_id
                , wallet_cid => $wallet_camp->{cid}
                , wallet_order_id => $wallet_camp->{OrderID}
                , offer_sum_to_pay => get_currency_constant($wallet_camp->{currency}, 'RECOMMENDATION_SUM_MID'), # временно, потом будем считать через грубый прогноз
                , allow_pay => int($allow_pay // 0)
                , login_can_edit => $login_can_edit
                , bonus => ($wallet_camp->{sums_uni}->{bonus} // 0)
                , is_super_subclient => $subclient_info->{$c->client_chief_uid}->{isSuperSubClient}
                , debt => $debt_without_nds
                , debt_with_nds => 1.0 * $client_data->{debt} // 0
                , overdraft_lim => 1.0 * $overdraft_lim_without_nds
                , overdraft_lim_with_nds => 1.0 * $client_data->{overdraft_lim} // 0
                , auto_overdraft_lim => 1.0 * $client_data->{auto_overdraft_lim} // 0
                , auto_overdraft_lim_without_nds => $auto_overdraft_lim_without_nds
                , nextPayDate => $client_data->{nextPayDate}
                , is_fake_nds => $is_fake_nds
                , is_brand => $client_data->{is_brand}
                , allow_transfer_money => $agency_client_id ? $subclient_info->{$c->client_chief_uid}->{allowTransferMoney} : 1
                , allow_disable_wallet => 0
                , need_camp_moderation_to_pay => int($need_camp_moderation_to_pay)
                , nontransferrable_sum => $nontransferrable_sum
                , total => round_low_2s($wallet_camp->{total})
                , map {
                    $_ => $wallet_camp->{$_}
                } qw /
                      money_warning_value
                      sms_flags
                      sms_hour_from
                      sms_hour_to
                      sms_min_from
                      sms_min_to
                      email
                      email_notifications
                      money_type
                      day_budget
                      day_budget_daily_change_count
                      day_budget_show_mode
                      day_budget_stop_time
                      day_budget_recommended_sum
                      currency
                     /
            };

            # кэшбек отдаем только если он есть
            # Если правишь здесь, то нужно поправить аналогичный код для нового интерфейса в
            # https://a.yandex-team.ru/arc/trunk/arcadia/direct/libs-internal/grid-processing/src/main/java/ru/yandex/direct/grid/processing/service/campaign/CampaignInfoService.java?rev=7439568#L711
            if (    defined $client_data->{cashback_awaiting_bonus}
                 && Client::ClientFeatures::show_cashback_bonus($wallet_var->{clientID}))
            {
                # Сперва посчитаем начисления за последний месяц
                my ($last_month_bonus, $last_month_bonus_wo_nds) = get_one_line_array_sql(PPC(ClientID => $wallet_var->{clientID}), "
                    SELECT sum(reward) AS reward_sum, sum(reward_wo_nds) AS reward_wo_nds_sum
                    FROM clients_cashback_details
                    WHERE reward_date BETWEEN DATE_FORMAT(NOW() - INTERVAL 1 MONTH, '%Y-%m-01 00:00:00')
                          AND DATE_FORMAT(LAST_DAY(NOW() - INTERVAL 1 MONTH), '%Y-%m-%d 23:59:59')
                          AND ClientId = ?", $wallet_var->{clientID});
                $wallet_var->{cashback_bonus} = {
                    sum                 => 1.0 * $cashback_bonus_without_nds,
                    sumWithNds          => 1.0 * $client_data->{cashback_bonus} // 0,
                    awaitingSum         => 1.0 * $awaiting_cashback_bonus_without_nds,
                    awaitingSumWithNds  => 1.0 * $client_data->{cashback_awaiting_bonus} // 0,
                    lastMonthSum        => $last_month_bonus_wo_nds // 0,
                    lastMonthSumWithNds => $last_month_bonus // 0,
                };
            }

            Campaign::separate_day_budget($wallet_var);

            if (ref($OPT{additional_wallet_fields}) eq 'ARRAY') {
                $wallet_var->{$_} = $wallet_camp->{$_} for @{$OPT{additional_wallet_fields}};
            }

            if ($OPT{get_sum_available}) {
                $wallet_var->{sum_available} = get_camp_sum_available($wallet_camp, sums_without_nds => ! $OPT{with_nds}, wallet_group => $not_archived_campaigns);

                # Если были запрошены суммы со скидочным бонусом -- добавим его
                if ($client_discount) {
                    $wallet_var->{sum_available} = Currencies::add_bonus($wallet_var->{sum_available}, $client_discount);
                }
            }

            if ($OPT{with_outlay}) {
                my $sum_spents = Stat::OrderStatDay::get_orders_sum_spent([grep { $_ != 0 } @$order_ids], {
                        'day' => {from => now()->add(days => -1)->ymd(), to => now()->add(days => -1)->ymd()},
                        'week' => {from => now()->add(days => -7)->ymd(), to => now()->ymd()},
                        'month' => {from => now()->add(days => -30)->ymd(), to => now()->ymd()},
                }, $wallet_camp->{currency});
                if (str($wallet_camp->{currency}) ne 'YND_FIXED') {
                    $sum_spents->{$_} = Currencies::remove_nds($sum_spents->{$_}, $client_nds) for keys %$sum_spents;
                }

                $wallet_var->{outlay} = {day => $sum_spents->{day}, week => $sum_spents->{week}, month => $sum_spents->{month}};
            }

            if ($OPT{with_emails_sms}) {
                $wallet_var->{validEmails} = get_valid_emails($c->uid, $wallet_camp->{email});
                $wallet_var->{sms_phone} = sms_check_user($c->uid, $c->user_ip);
            }

            if ($OPT{with_camp_stop_daily_budget_stats}) {
                $wallet_var->{camp_stop_daily_budget_stats} = Campaign::get_day_budget_stop_history($wallet_camp->{cid});
                if ($wallet_var->{day_budget}{sum} > 0) {
                    $wallet_var->{camp_stop_daily_budget_warning} =
                            get_stop_daily_budget_warning($wallet_var->{camp_stop_daily_budget_stats}, $wallet_var->{day_budget}{stop_time});
                } else {
                    $wallet_var->{camp_stop_daily_budget_warning} = undef;
                }
            }

        } else {

            # счет выключен, отдаем время до включения
            my $off_wallet_camp = get_wallet_camp($c->client_chief_uid, $agency_client_id, $client_currency);

            my $time_to_enable;
            if ($off_wallet_camp->{time_to_enable}) {
                my ($hours, $minutes) = $off_wallet_camp->{time_to_enable} =~ /^(\d+):(\d\d)/;
                if (defined $hours && defined $minutes) {
                    $time_to_enable = {
                        hours => $hours,
                        minutes => $minutes,
                    };
                }
            }

            $wallet_var = {
                enabled => 0
                , wallet_cid => $off_wallet_camp->{wallet_cid}
                , wallet_order_id => $wallet_camp->{OrderID}
                , time_to_enable => $time_to_enable
                , allow_enable_wallet => (scalar(@$campaigns) > 0 ? 1 : 0)
                , allow_transfer_money => $agency_client_id ? $subclient_info->{$c->client_chief_uid}->{allowTransferMoney} : 1
                , total => (defined $off_wallet_camp->{total}) ? round_low_2s($off_wallet_camp->{total}) : undef
                , sum => $off_wallet_camp->{sum}
                , sum_spent => $off_wallet_camp->{sum_spent}
                , currency => $off_wallet_camp->{currency}
            };
            campaign_remove_nds_and_add_bonus($wallet_var, client_nds => $client_nds, client_discount => $client_discount);
        }

        $wallet_var->{total_with_nds} = $wallet_var->{currency} eq 'YND_FIXED' ?
                $wallet_var->{total}
                : Currencies::add_nds($wallet_var->{total} // 0, $client_nds);

        push @$result, {
            wallet_cid => $wallet_var->{wallet_cid}
            , uid => $c->client_chief_uid
            , agency_client_id => $agency_client_id ? $agency_client_id : 0
            , text_camps => $not_archived_campaigns
            , wallet_camp => $wallet_var
            , campaigns_count => scalar(@$campaigns)
        };
    }

    return $result;
}

# --------------------------------------------------------------------

=head2 get_wallets_by_cids

Возвращаем кампании общие счета (+текстовые кампании) для клиента по типу сервисирования
    my $result = Wallet::get_wallets_by_cids($wallet_cids_arrayref, $c, именнованные параметры);
    $c - должен соответствовать одному UID (тот кто вызывает: менеджер, агентство или клиент)

    именнованные, не обязательные параметры (такие же как в get_wallets_by_uids):
        additional_wallet_fields
        get_sum_available
        with_nds
        with_bonus

    результат - хеш хешей:
        {wallet_cid =>  {
            wallet_cid -- # кампании счета
            uid -- uid главного представителя клиента
            agency_client_id -- ClientID агентства
            text_camps -- [{}, {}, ...] -- список текстовых кампаний клиента по типу сервисирования
            wallet_camp -- {} -- кампания счет, с дополнительными полями (время последнего включения, можно ли включить)
        }, ...}

        содержимое wallet_camp:
            enabled => 1
            , offer_sum_to_pay
            , outlay => {day => $sum_spents->{day}, week => $sum_spents->{week}, month => $sum_spents->{month}}
            , allow_pay
            , bonus
            , validEmails
            , sms_phone
            , cid
            , total
            , money_warning_value
            , sms_flags
            , sms_hour_from
            , sms_hour_to
            , sms_min_from
            , sms_min_to
            , email

        при отключенном счете
            enabled => 0
            , time_to_enable
            , allow_enable_wallet

=cut

sub get_wallets_by_cids($$;%) {
    my ($wallet_cids, $c, %OPT) = @_;

    my $cid2uid_info = get_hashes_hash_sql(PPC(cid => $wallet_cids), [
        "select c.cid
              , c.uid
              , u.ClientID as client_client_id
              , c.AgencyID as agency_client_id
              , IFNULL(c.currency, 'YND_FIXED') AS currency
         from campaigns c
           join users u using(uid)
        ", where => {
            'c.cid' => SHARD_IDS
        }
    ]);

    my $params_for_get_wallets_by_uids = [];

    for my $wallet_cid (@$wallet_cids) {
        next unless exists $cid2uid_info->{$wallet_cid};

        my $context_for_uid = DirectContext->new({
            is_direct        => 1,
            uid              => $cid2uid_info->{$wallet_cid}->{uid},
            UID              => $c->UID,
            client_chief_uid => $cid2uid_info->{$wallet_cid}->{uid},
            client_client_id => $cid2uid_info->{$wallet_cid}->{client_client_id},
            rbac             => $c->rbac,
            login_rights     => $c->login_rights,
        });

        push @$params_for_get_wallets_by_uids, {
            c => $context_for_uid
            , agency_client_id => $cid2uid_info->{$wallet_cid}->{agency_client_id}
            , client_currency => $cid2uid_info->{$wallet_cid}->{currency}
        };
    }

    my $result = {
        map {$_->{wallet_cid} => $_}
        grep {$_->{wallet_cid} > 0} # не созданные счета пропускаем
        @{Wallet::get_wallets_by_uids($params_for_get_wallets_by_uids, %OPT)}
    };

    return $result;
}

# --------------------------------------------------------------------

=head2 enable_wallet

Включение общего счета на клиенте

    my $result = Wallet::enable_wallet($c, $client_currencies->{work_currency}, $agency_uid);
        $c -- DirectContext
        $client_currencies->{work_currency} -- валюта клиента
        $agency_uid -- uid агентства (не обязательный параметр)

    именованные параметры:
        allow_wallet_before_first_camp -- позвоялем подключать счет до того как кампания станет statusEmpty='No'
                                          используется при создании первой кампании из интерфейса
        first_camp_cid -- id первой создаваемой кампани клиента. Передаем для случаев, когда кампания создается со statusEmpty=Yes (через API)
        dont_check_onoff_time: (0|1) -- не смотрим на дату последнего включения счета, при подключении счета при создании первой кампании

    результат:
        {wallet_cid => 1234} -- при успехе
        {error => "Error message"} -- при ошибке

=cut

sub enable_wallet($$$;%) {
    my ($c, $currency, $agency_uid, %OPT) = @_;

    my ($wallet_cid, $wallet_info, $agency_client_id, $text_campaigns);
    $text_campaigns //= {};

    $agency_uid = $c->UID if ! $agency_uid && $c->login_rights->{agency_control};
    if ($agency_uid) {
        $agency_client_id = rbac_get_agency_clientid_by_uid($agency_uid);
        unless ($agency_client_id) {
            return {error => iget("Агентство не найдено")};
        }
    }
    my $user_info = get_user_info($c->client_chief_uid);
    my $client_id = $c->client_client_id;

    # берем лок, чтоб одновременно не подключать клиенту два разных кошелька
    my $db_lock_name = join('_', "WALLET_ENABLE_LOCK", $client_id, $agency_client_id // 0, $currency // 'undef');
    # Передаем таймаут, для ожидания освобождения sql lock (например, для кейса, когда в то же время кошелек создался в Java и еще залочен)
    my $db_lock_guard = sql_lock_guard(PPC(ClientID => $client_id), $db_lock_name, 10);

    my %camp_options = (
        client_chief_uid   => $c->client_chief_uid,
        ClientID           => $client_id,
        name               => ($agency_uid ? iget('Общий счет (агентский)') : iget('Общий счет')),
        type               => 'wallet',
        currency           => $currency,
        statusModerate     => 'Yes',
        statusPostModerate => 'Accepted',
        client_fio         => '',
        client_email       => '',
        statusEmpty        => 'No',
        client_fio         => $user_info->{fio},
        client_email       => $user_info->{email},
    );

    my $camp_types_under_wallet = get_camp_kind_types('under_wallet');

    if ($agency_uid) {
        # выбираем все кампании, включая statusEmpty='Yes' - нужно при включении счета при создании первой кампании
        $text_campaigns = get_hashes_hash_sql(PPC(ClientID => $client_id),
                                                   ["select c.cid, c.sum - c.sum_spent as total, c.statusEmpty
                                                         , IFNULL(c.currency, 'YND_FIXED') as currency
                                                         , c.type
                                                         , c.strategy_id
                                                    from campaigns c", where => {
                                                        'c.type' => $camp_types_under_wallet,
                                                        'c.uid' => $c->client_chief_uid,
                                                        'c.AgencyID' => $agency_client_id,
                                                    }, "
                                                      and IFNULL(c.ManagerUID, 0) = 0
                                                      and IFNULL(c.currency, 'YND_FIXED') = ?
                                                   "], $currency
                                              );

        $wallet_info = get_wallet_camp($c->client_chief_uid, $agency_client_id, $currency);

        $camp_options{agency_uid} = Rbac::get_client_agency_uid($client_id, $agency_uid);

    } else {
        $text_campaigns = get_hashes_hash_sql(PPC(ClientID => $client_id),
            ["select c.cid, c.sum - c.sum_spent as total, c.ManagerUID, c.statusEmpty, c.archived
                , IFNULL(c.currency, 'YND_FIXED') as currency
                , c.type
                , c.strategy_id
             from campaigns c", where => {
                    'c.type' => $camp_types_under_wallet,
                    'c.uid' => $c->client_chief_uid,
            }, "
                and IFNULL(c.AgencyUID, 0) = 0
                and IFNULL(c.currency, 'YND_FIXED') = ?
                "], $currency
        );

        if ($c->login_rights->{client_primary_manager_set_by_idm}) {
            $camp_options{manager_uid} = $c->login_rights->{client_primary_manager_uid};
        } 
        elsif ($c->login_rights->{manager_control} && !RBACDirect::has_idm_access_to_client($c->UID, $client_id)) {
            # Если общий счет включает менеджер имиеющий доступ к клиенту не через IDM, то берем его на обслуживание под этого менеджера
            $camp_options{manager_uid} = $c->UID;
        }
        else {
            # Если у клиента есть менеджерские кампании, то берем на обслуживание и кампанию-кошелек (DIRECT-28708)
            if (my @serviced_camps = grep { ($_->{statusEmpty} eq 'No' || ($OPT{first_camp_cid} && $_->{cid} == $OPT{first_camp_cid}))
                                             && $_->{archived} eq 'No' 
                                             && $_->{ManagerUID} } values %$text_campaigns) {
                $camp_options{manager_uid} = $serviced_camps[0]->{ManagerUID};
            }
        }
        $wallet_info = get_wallet_camp($c->client_chief_uid, 0, $currency);
    }

    return {error => iget("Для того чтобы подключить общий счет, необходимо создать хотя бы одну кампанию."), code => 520}
        if ! $OPT{allow_wallet_before_first_camp} && scalar(grep {$_->{statusEmpty} eq 'No'} values %$text_campaigns) == 0;

    my $error_hash;
    my $new_wallet_cid;
    if ($wallet_info->{wallet_cid}) {
        $wallet_cid = $wallet_info->{wallet_cid};

        if ($wallet_info->{is_enabled}) {
            $error_hash = {error => iget("Невозможно завершить операцию: общий счет уже подключен"), code => 519 };
        } elsif (! $wallet_info->{allow_enable_wallet}
                 && ! $c->login_rights->{super_control}
                 && ! $OPT{dont_check_onoff_time}
        ) {
            $error_hash = {error => iget("Повторное подключение общего счета возможно не ранее, чем через сутки после предыдущего"), code => 519 };
        }
    } else {
        $new_wallet_cid = get_new_id('cid', ClientID => $camp_options{ClientID});
    }

    return $error_hash if $error_hash;

    # можно ли включить у этого ОС новую схему учета зачислений (wallet_campaigns.is_sum_aggregated = 'Yes')
    my $need_to_migrate_to_new_sums = _can_migrate_to_new_sums($client_id, $currency, $wallet_cid, $text_campaigns);

    my $transaction_ok = eval {
        do_in_transaction sub {
            if ($new_wallet_cid) {
                my $error = Campaign::can_create_camp( %camp_options );
                if ($error) {
                    $error_hash = {error => iget("Сервис временно недоступен")};
                    warn "error in enable_wallet: $error - break transaction";
                    die "error in enable_wallet: $error - break transaction";
                }

                $wallet_cid = Campaign::create_empty_camp(%camp_options, cid => $new_wallet_cid);
            }

            do_insert_into_table(PPC(ClientID => $client_id), 'wallet_campaigns',
                                    {
                                        wallet_cid => $wallet_cid,
                                        onoff_date__dont_quote => 'NOW()',
                                    },
                                    on_duplicate_key_update => 1,
                                    key => ['wallet_cid'],
                                );

            # если никакая кампания не будет подключена к ОС, включаем новую схему зачислений здесь.
            if ($need_to_migrate_to_new_sums && !%$text_campaigns) {
                _migrate_to_new_sums($client_id, $wallet_cid);
            }
        };
        return 1;
    };

    return $error_hash if $error_hash;
    # если произошла необработанная (т.е. без присвоения $error_hash) ошибка
    die $@ if !$transaction_ok;

    # перенос средств на кошелёк
    if (%$text_campaigns) {

        my $nds = get_client_NDS($client_id);
        my $params = {from => {map {$_->{cid} => $_->{currency} ne 'YND_FIXED' ? remove_nds($_->{total}, $nds) : $_->{total}}
                               grep {$_->{total} && $_->{total} > 0}
                               grep {$_->{statusEmpty} eq 'No'}
                               values %$text_campaigns
                              }
                      , to => {$wallet_cid => sum
                                              map {$_->{currency} ne 'YND_FIXED' ? remove_nds($_->{total}, $nds) : $_->{total}}
                                              grep {$_->{total} && $_->{total} > 0}
                                              grep {$_->{statusEmpty} eq 'No'}
                                              values %$text_campaigns
                              }
                     };

        if ($params->{to}->{$wallet_cid} && $params->{to}->{$wallet_cid} > 0) {
            $params = prepare_transfer_money(undef, $c->uid, $params);
            # ошибку игнорируем, биллинг в любом случае попытается перенести деньги (DIRECT-29299)
            eval {process_transfer_money(undef, $c->uid, $c->UID, $params, move_all_qty => 1, is_enable_wallet => 1)};
        }

        my @text_campaigns_cids = keys %$text_campaigns;
        my @strategy_ids = map { $_->{strategy_id} } values %$text_campaigns;
        do_in_transaction sub {

            do_update_table(PPC(ClientID => $client_id), 'campaigns'
                               , {wallet_cid => $wallet_cid
                                  , statusBsSynced => 'No'
                                 }
                               , where => {
                                   uid => $c->client_chief_uid,
                                   type => $camp_types_under_wallet,
                                   cid => \@text_campaigns_cids
                               }
                           );

            do_update_table(PPC(ClientID => $client_id), 'strategies'
                , {wallet_cid => $wallet_cid}
                , where => {
                    strategy_id => \@strategy_ids
                }
            );

            if ($need_to_migrate_to_new_sums) {
                _migrate_to_new_sums($client_id, $wallet_cid);
            }

            if (!Direct::BillingAggregates->is_autocreate_disabled($client_id, $c->client_chief_uid, $currency)) {
                my @existing_camp_types = uniq map { $_->{type} } grep {$_->{statusEmpty} eq 'No'} values %$text_campaigns;
                my $missing_prod_types = Direct::BillingAggregates->get_missing_product_types($client_id, $wallet_cid, \@existing_camp_types);
                my $wallet_model = _make_wallet_model($wallet_cid, \%camp_options, $agency_client_id);
                if (@$missing_prod_types) {
                    Direct::BillingAggregates->make_new_aggregates_for_client(
                        $client_id,
                        $missing_prod_types,
                        $wallet_model
                    )->create($c->UID);
                }
            }
        };
        if ($c->is_beta) {
            # для бет отвязываем кампании от предыдущего счета в биллинге, такое может происходить после обновления базы (а в биллинге остается старый счет)
            # делаем это после обновления wallet_cid в кампаниях, иначе возникают гонки вроде DIRECT-86906
            create_campaigns_balance(undef, $c->UID, [@text_campaigns_cids], force_group_order_transfer => 1, beta_unlink_wallet => 1);
        }

        # перепосылаем в биллинг кампании, чтобы проставить признак общий счет
        my $camp_balance_response = eval { create_campaigns_balance(undef, $c->UID, [@text_campaigns_cids], force_group_order_transfer => 1) };
        if (! $camp_balance_response || $camp_balance_response->{error} || ! $camp_balance_response->{balance_res}) {
            # при ошибке биллинга при обновлении кампании, пытаемся еще раз оправить через оффлайновую очередь
            # счет не выключаем, на нем уже есть деньги (DIRECT-26311)
            BalanceQueue::add_to_balance_info_queue($c->UID, cid => \@text_campaigns_cids, BalanceQueue::PRIORITY_CAMPS_ON_ENABLE_WALLET);
        }
    }

    # сохраняем порог предупреждения об окончании денег (DIRECT-22528)
    my $max_money_warning_value = get_one_field_sql(PPC(ClientID => $c->client_client_id), [
        "select MAX(co.money_warning_value)
         from campaigns c
         join camp_options co using(cid)
        ", where => {
            'c.uid' => $c->client_chief_uid,
            'c.wallet_cid' => $wallet_cid,
            'co.money_warning_value__gt' => 0
        }
    ]) || $Settings::DEFAULT_MONEY_WARNING_VALUE;

    Wallet::update_wallet_sms_email_settings($wallet_cid,
            {money_warning_value => $max_money_warning_value, sms_time => "09:00:21:00"},
            paused_by_day_budget_sms => 1);

    return {wallet_cid => $wallet_cid};
}

sub _make_wallet_model {
    my ($wallet_cid, $camp_options, $agency_client_id) = @_;

    return Direct::Model::Wallet->new(
        id => $wallet_cid,
        client_id => $camp_options->{ClientID},
        user_id => $camp_options->{client_chief_uid},
        manager_user_id => $camp_options->{manager_uid},
        agency_id => $agency_client_id // 0,
        agency_user_id => $camp_options->{agency_uid},
        currency => $camp_options->{currency},
        email => $camp_options->{client_email},
        client_fio => $camp_options->{client_fio},
    );
}

=head2 _can_migrate_to_new_sums($client_id, $currency, $wallet_cid, $child_camps)

    Можно ли мигрировать ОС в новую схему зачислений.

=cut
sub _can_migrate_to_new_sums {
    my ($client_id, $currency, $wallet_cid, $child_camps) = @_;

    if ($currency eq 'YND_FIXED') {
        return 0;
    }

    my $no_child_camps = none { $_->{statusEmpty} eq 'No' } values %$child_camps;
    my $has_child_camps = !$no_child_camps;

    if ($no_child_camps && Client::ClientFeatures::has_new_wallet_with_aggregated_sums($client_id)) {
        return 1;
    }
    if ($has_child_camps && Client::ClientFeatures::has_aggregated_sums_for_old_clients($client_id)) {
        return 1;
    }
    return 0;
}

=head2 _migrate_to_new_sums($client_id, $wallet_cid)

    Миграция ОС в новую схему зачислений.
    При этом все зачисления переносятся с дочерних кампаний на ОС.
    Эта процедура должна вызываться в рамках открытой транзакции БД.

    Если на дочерних кампаниях не было зачислений, у ОС сразу выставляется признак
    перехода на новую схему (wallet_campaigns.is_sum_aggregated = 'Yes').
    Иначе, выполняется миграция общего счета и дочерних кампаний в новую схему

=cut
sub _migrate_to_new_sums {
    my ($client_id, $wallet_cid) = @_;

    my $current_status = get_one_field_sql(PPC(ClientID => $client_id), ["
        SELECT is_sum_aggregated
        FROM wallet_campaigns",
        WHERE => [
            wallet_cid => $wallet_cid,
        ]
    ]);

    # если под будущим ОС есть кампании в очень специальных очередях,
    # то не осмеливаемся мигрировать такой ОС
    my $camps_in_special_queues = get_one_column_sql(PPC(ClientID => $client_id), ["
        SELECT bes.cid
        FROM bs_export_specials bes
            JOIN campaigns c ON c.cid = bes.cid ",
        WHERE => [
            'c.wallet_cid' => $wallet_cid,
            'bes.par_type__not_in' => [qw/heavy buggy camps_only/],
        ]
    ]);
    if (@$camps_in_special_queues && $current_status eq 'No') {
        return;
    }

    do_sql(PPC(ClientID => $client_id), ["
        UPDATE campaigns wc
            JOIN wallet_campaigns wwc ON wwc.wallet_cid = wc.cid
        SET wc.`sum_balance` = wc.`sum`",
        WHERE => [
            'wc.cid' => $wallet_cid,
            # если ОС уже был в объединённой схеме зачислений во время включения,
            # не меняем его зачисления
            'wwc.is_sum_aggregated' => 'No',
        ]
    ]);
    do_sql(PPC(ClientID => $client_id), ["
        UPDATE campaigns SET `sum_balance` = `sum`",
        WHERE => [
            wallet_cid => $wallet_cid,
            # защищаемся от ситуаций, когда процедуру включения ОС запускают при уже включенном ОС
            # тогда у дочерних кампаний в sum записано 0
            # если у дочерних кампаний до включения ОС было 0 зачислений, то и sum_balance у них тоже 0,
            # и этот UPDATE всё равно не нужно делать.
            sum__gt => 0,
        ]
    ]);

    # всё что ниже можно безопасно выполнять даже если ОС уже включен и находится в новой схеме.

    my $sums = get_one_line_sql(PPC(ClientID => $client_id), ["
        SELECT SUM(c.sum_balance) AS sum_balance_sum, SUM(IFNULL(cms.chips_cost, 0)) AS chips_cost_sum
        FROM campaigns c
            LEFT JOIN campaigns_multicurrency_sums cms ON cms.cid = c.cid",
        WHERE => [
            _OR => [
                'c.cid' => $wallet_cid,
                'c.wallet_cid' => $wallet_cid,
            ]
        ]
    ]);

    do_update_table(PPC(ClientID => $client_id), 'campaigns c JOIN wallet_campaigns wc ON wc.wallet_cid = c.cid',
        {
            'c.sum' => $sums->{sum_balance_sum},
            'c.statusBsSynced' => 'No',
            'wc.total_chips_cost' => $sums->{chips_cost_sum},
            'wc.is_sum_aggregated' => 'Yes',
        },
        where => {
            'c.cid' => $wallet_cid,
        }
    );

    # Если на кампании были зачисления, их нужно переотправить в БК вместе с общим счетом
    # в спец. очереди camps_only - там высок шанс, что они уедут относительно быстро
    # и все вместе в одном запросе.
    my $cids_to_resend = get_one_column_sql(PPC(ClientID => $client_id), ["
        SELECT c.cid
        FROM campaigns c",
        WHERE => [
            'c.wallet_cid' => $wallet_cid,
            'c.sum__gt' => 0,
        ]
    ]);
    if (@$cids_to_resend) {
        push @$cids_to_resend, $wallet_cid;
        do_mass_insert_sql(PPC(ClientID => $client_id), 'INSERT INTO bs_export_specials (cid, par_type)
                                  VALUES %s
                                  ON DUPLICATE KEY UPDATE par_type = IF(par_type NOT IN ("nosend", "dev1", "dev2", "preprod"),
                                                                        VALUES(par_type),
                                                                        par_type
                                                                        )',
                           [ map { [ $_, 'camps_only' ] } @$cids_to_resend ]);

        # кампаниям, уже находящимся в очереди на отправку в БК, сбрасываем время queue_time,
        # чтобы мониторинг возраста очереди camps_only не давал ложных срабатываний.
        do_update_table(PPC(ClientID => $client_id), 'bs_export_queue',
            {
                'queue_time__dont_quote' => 'NOW()',
            },
            where => {
                cid => $cids_to_resend,
            }
        );
    }

    do_update_table(PPC(ClientID => $client_id), 'campaigns',
        {
            sum => 0,
            statusBsSynced => 'No',
        },
        where => {
            wallet_cid => $wallet_cid,
        }
    );
}

# --------------------------------------------------------------------

=head2 arc_camp

    Аналог Common::_arc_camp()
    Раньше ещё компенсировал минусы, сейчас полагаемся исключительно на ночные компенсации в Балансе

    my $result = Wallet::arc_camp(undef, $UID, $uid, [cid1, cid2, ...], options);

    Именованные параметры:
        - force, логический
        - archived_is_error, логический -- ошибка если кампания уже заархивирована
        - archive_non_stopped, логический -- архивировать кампании, на которых statusShow = 'Yes'
        - dont_stop_on_error - bool, при возникновении ошибки не завершать
            выполнение функции, а переходить к обработке следующей кампании

    возвращает:
        {result => 0} -- при успехе
        {result => 1, cid => 123, error => 'error message'} -- при ошибках без dont_stop_on_error
        {result => 1, errors => {123 => 'error message', ...}} — при ошибках с dont_stop_on_error

=cut

sub arc_camp($$$$;%) {
    my (undef, $UID, $uid, $cids, %OPT) = @_;

    my $dont_stop_on_error = delete $OPT{dont_stop_on_error};

    my %error_by_camp;
    for my $cid (@$cids) {
        my $arc_camp_opts = hash_cut \%OPT, qw(force archived_is_error archive_non_stopped);
        $arc_camp_opts->{UID} = $UID;
        my ($result, $error) = Common::_arc_camp($uid, $cid, %$arc_camp_opts);
        # arc_camp возвращает 0 при ошибке
        if ( ! $result && $error ) {
            return {result => 1, cid => $cid, error => $error} unless $dont_stop_on_error;
            $error_by_camp{ $cid } = $error;
        }
    }

    if ( keys %error_by_camp ) {
        return { result => 1, errors => \%error_by_camp };
    }

    return {result => 0};
}

# --------------------------------------------------------------------

=head2 update_wallet_sms_email_settings

    Сохраняем параметры уведомлений для общего счета:
        camp_options.
            email
            FIO
            sms_flags
            sms_time
            money_warning_value

    Wallet::update_wallet_sms_email_settings($wallet_cid, {email => "mail@ya.ru", FIO => "Pupkin"});
    Wallet::update_wallet_sms_email_settings($wallet_cid, {sms_flags => "active_orders_money_out_sms,notify_order_money_in_sms", sms_time => "09:00:21:00"});
    Wallet::update_wallet_sms_email_settings($wallet_cid, {money_warning_value => 35});

    Именованные параметры
        paused_by_day_budget_sms => 1/0     - вкл/выкл уведомления об остановке общего счета по смс. Если undef, настройка не меняется.
        paused_by_day_budget_email => 1/0   - вкл/выкл уведомления об остановке общего счета по email. Если undef, настройка не меняется.

    Если одна и та же настройка указана и в хешике во втором параметре, и в именованном параметре, именованный имеет приоритет

=cut

sub update_wallet_sms_email_settings {
    my ($wallet_cid, $settings, %OPT) = @_;

    my $fields = {};

    for my $name (qw/email FIO sms_flags sms_time money_warning_value email_notifications/) {
        $fields->{$name} = $settings->{$name} if defined $settings->{$name};
    }

    my %sms_flags;
    my %email_flags;

    if (defined($OPT{paused_by_day_budget_sms})) {
        $sms_flags{paused_by_day_budget_sms} = $OPT{paused_by_day_budget_sms};
    }
    if (defined($OPT{paused_by_day_budget_email})) {
        $email_flags{paused_by_day_budget} = $OPT{paused_by_day_budget_email};
    }

    _apply_new_set_flags($fields, 'sms_flags', \%sms_flags);
    _apply_new_set_flags($fields, 'email_notifications', \%email_flags);

    if (keys %$fields > 0) {
        return do_update_table(PPC(cid => $wallet_cid), 'camp_options'
                                  , $fields
                                  , where => {cid => $wallet_cid}
                              );
    }

    return undef;
}

# проверяет, есть ли строка со значением SETа под названием $set_name уже в хешике в $fields
# если есть, то руками применяет флаги $new_flags к этой строчке
# если нет, добавляет в хешик $fields выражение, чтобы mysql применил флаги уже на своей стороне
sub _apply_new_set_flags {
    my ($fields, $set_name, $new_flags) = @_;

    return unless %$new_flags;

    my $old_set_string = delete($fields->{$set_name});

    if (defined $old_set_string) {
        my $new_set_string = ",$old_set_string,";
        for my $key (keys %$new_flags) {
            if ($new_flags->{$key}) {
                $new_set_string .= "$key,";
            } else {
                $new_set_string =~ s/,$key,/,/g;
            }
        }

        $fields->{$set_name} = $new_set_string =~ s/^,|,$//gr;
    } else {
        $fields->{$set_name.'__smod'} = $new_flags;
    }
}

# --------------------------------------------------------------------

=head2 need_enable_wallet

    Будет ли создан кошелек при сохранении редактируемой кампании

    Именованные параметры, обязательные
        cid                 id редактируемой кампании
        client_id           uid главного представителя клиента, чья кампания редактируется


=cut

sub need_enable_wallet {
    my ( %in ) = @_;

    ( exists $in{$_} || croak "$_ required" ) foreach qw/cid client_id/;
    # сколько кампаний уже есть у клиента, кроме новой
    my $is_first_campaign = Primitives::is_first_camp_under_wallet($in{cid}, $in{client_id});
    my $create_without_wallet = get_one_field_sql(PPC(ClientID => $in{client_id}), [
            "SELECT FIND_IN_SET('create_without_wallet', client_flags)
            FROM clients_options",
            WHERE => {
                ClientID => $in{client_id},
            }
        ]);

    return $is_first_campaign && !$create_without_wallet;
}

=head2 get_stop_daily_budget_warning($camp_stop_daily_budget_stats, $camp_daily_budget_stop_time)

    С учетом статистики остановок за последние 14 дней вычисляет, нужно ли показать предупреждение
    о том, что дневной бюджет слишком маленький.

    Если предупреждение не нужно показывать, возвращает undef
    Если нужно показать - возвращает объект с датой и временем остановки, которая нам не понравилась,
    готовый к передаче на фронт.

=cut

sub get_stop_daily_budget_warning {
    my ($camp_stop_daily_budget_stats, $camp_daily_budget_stop_time) = @_;

    return undef if $camp_daily_budget_stop_time =~ /^0000/;

    my $warning_hour_prop = Property->new($DAILY_BUDGET_STOP_WARNING_TIME_PROP);
    my $warning_hour = $warning_hour_prop->get(60) // $DAILY_BUDGET_STOP_WARNING_TIME_DEFAULT;
    $warning_hour =~ s/\D+//g;

    my $today = today();
    my $yesterday = yesterday();
    my $warning = {stop_time => undef, msk_today => mysql_round_day($today, delim => '-')};
    for my $row (@$camp_stop_daily_budget_stats) {
        my $stop_datetime = $row;
        $stop_datetime =~ s/\D+//g;
        last if $stop_datetime lt $yesterday;

        my ($stop_date, $stop_time) = $stop_datetime =~ /(\d{8})(\d{6})/;
        if ((any {$stop_date eq $_} ($today, $yesterday)) && ($stop_time lt $warning_hour)) {
            $warning->{stop_time} = $row;
            return $warning;
        }
    }
    return undef;
}

1;
