package WalletUtils;

=head1 NAME

WalletUtils - полезные функции для работы с "общим счетом" (DIRECT-19914)

=cut

# $Id$

use warnings;
use strict;
use utf8;
use feature 'state';

use Carp;
use List::Util qw/sum0 min max any/;
use List::MoreUtils qw/uniq/;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::Balance qw/balance_get_client_persons/;
use Yandex::HashUtils;

use Campaign::Types qw/get_camp_kind_types/;
use Client;
use Stat::OrderStatDay;
use Property;
use Tools;
use User qw/get_user_data/;
use geo_regions;

use PrimitivesIds;
use Yandex::Balance qw/ balance_get_client_persons balance_create_person /;

use constant AUTOPAY_DEFAULT_PAYMENT_SUM   =>  1000;
use constant AUTOPAY_DEFAULT_REMAINING_SUM =>  100;
use constant AUTOOVERDRAFT_DEFAULT_SUGGEST_COEF =>  0.8;

use constant DEFAULT_NAME => "-";
use constant DEFAULT_LAST_NAME => "Фамилия";

=head2 get_sum_debt_for_wallets_by_uids($uids)

    Посчитать сумму задолженности для кампаний под ОС для uid'а
    Для валютных кошельков сумма возвращяется с НДС и без скидочного бонуса (ровно как храним в БД)

=cut

sub get_sum_debt_for_wallets_by_uids
{
    my ($uids) = @_;

    confess "invalid params" unless defined $uids && ref($uids) eq 'ARRAY';
    return {} unless @$uids;

    # Рассчитаем общую сумму задолженности для всех кампаний под одним общим счетом
    # Сумму задолженности считаем только для кампаний имеющих таковую (отрицательный баланс)
    my $sum_debt_all = get_hash_sql(PPC(uid => $uids), [q{
        SELECT
            c.wallet_cid, sum(IF(c.sum - c.sum_spent < 0, c.sum - c.sum_spent, 0)) AS sum_debt
        FROM campaigns c
        },
        where => {
            'c.uid' => SHARD_IDS,
            'c.statusEmpty' => 'No',
            #'c.archived' => 'No',
            'c.type' => get_camp_kind_types('under_wallet', 'billing_aggregate'),
            'c.wallet_cid__int__gt' => 0,
        },
        'GROUP BY c.wallet_cid',
    ]);

    return $sum_debt_all;
}

=head2 get_sum_debts_for_wallets($cids)

    Посчитать сумму задолженности для кампаний под ОС для cid кошелька.
    Для валютных кошельков сумма возвращяется с НДС и без скидочного бонуса (ровно как храним в БД)

=cut

sub get_sum_debts_for_wallets
{
    my ($cids) = @_;

    confess "invalid params" unless defined $cids;

    # Рассчитаем общую сумму задолженности для всех кампаний под указанным общим счетом
    # Сумму задолженности считаем только для кампаний имеющих таковую (отрицательный баланс)
    my $sum_debts = get_hash_sql(PPC(cid => $cids), [q{
        SELECT
            c.wallet_cid, sum(IF(c.sum - c.sum_spent < 0, c.sum - c.sum_spent, 0)) AS sum_debt
        FROM campaigns c
        },
        where => {
            'c.wallet_cid' => SHARD_IDS,
            'c.statusEmpty' => 'No',
            'c.type' => get_camp_kind_types('under_wallet')
        },
        'GROUP BY c.wallet_cid',
    ]);

    return $sum_debts;
}

=head2 is_auto_overdraft_applicable($currency, $auto_overdraft_lim, $clientID);

    Возвращает 1, если к указанной кампании-кошельку применим автоовердрафт
    Это зависит от значения валюты кампании, значения порога отключения (auto_overdraft_lim) и ClientID
    $auto_overdraft_lim может быть передан как undef
    Валюта должна быть RUB, порог отключения должен быть больше нуля, а клиент не должен быть одним из кривых клиентов
    (у которых есть более одного нефишечного активного кошелька -- см. DIRECT-87190)

    В противном случае возвращает 0.

=cut

sub is_auto_overdraft_applicable {
    my ($currency, $auto_overdraft_lim, $clientID) = @_;

    return !exists $Client::BAD_AUTO_OVERDRAFT_CLIENTS{ $clientID }
        # функционал открыт только для России, Белоруссии и Казахстана (см. DIRECT-72426 и связанных)
        && (grep {$_ eq $currency} @Client::AUTOOVERDRAFT_CURRENCY_AVAILABLE)
        && defined $auto_overdraft_lim && $auto_overdraft_lim > 0
        ? 1 : 0;
}

=head2 is_client_banned_in_balance($client_info)

    Возвращает 1, если клиент был забанен Балансом
    или если Баланс сбросил ему overdraft_lim при неотрицательном auto_overdraft_lim.

    В противном случае возвращает 0.

=cut
sub is_client_banned_in_balance {
    my ($client_info) = @_;

    return $client_info->{statusBalanceBanned} eq 'Yes'
        || ($client_info->{auto_overdraft_lim} > 0 && $client_info->{overdraft_lim} == 0);
}

=head2 get_auto_overdraft_addition($wallet_info, $client_info)

    Вычисляет и возвращает денежную "добавку" по автоовердрафту для кошелька клиента -- те деньги,
    которые он может потенциально использовать (или уже использовал) по автоовердрафтам.

    Возвращает 0, если автоовердрафт неприменим к указанному клиенту или кошельку.

    Входные данные:

    wallet_info = {
        type,                # должен быть 'wallet'
        currency,            # валюта кошелька
        sum,                 # величина campaigns.sum на кошельке
        wallet_sum_debt      # значение для кошелька, полученное из функции get_sum_debts_for_wallets или подобной ей
    }

    $client_info = {
        clientID,            # идентификатор клиента
        debt,                # текущий долг клиента (размер выставленного, но ещё не оплаченного счёта)
        overdraft_lim,       # 0 или лимит овердрафта
        auto_overdraft_lim,  # 0 или порог отключения, заданный пользователем
        statusBalanceBanned  # "забанен" ли клиент в Балансе ('Yes' или 'No')
    }

=cut

sub get_auto_overdraft_addition {
    my ($wallet_info, $client_info) = @_;

    # Если отправляемая кампания -- не кошелёк, или к нему неприменимы автоовердрафты
    if (!Campaign::Types::is_wallet_camp(type => $wallet_info->{type})
        || !is_auto_overdraft_applicable($wallet_info->{currency}, $client_info->{auto_overdraft_lim}, $client_info->{clientID}))
    {
        return 0;
    }

    # Учитываем текущий долг клиента - мы не можем давать ему "двойной" кредит
    my $available_auto_overdraft_lim = max(0, $client_info->{auto_overdraft_lim} - $client_info->{debt});

    # Отдельный кейс для забаненных в Балансе клиентов -- см DIRECT-88599
    if (is_client_banned_in_balance($client_info)) {
        # Получим сумму открутов для общего счета по всем входящим в него кампаниям, $debt <= 0
        my $debt = $wallet_info->{wallet_sum_debt};
        # Нам интересна сумма зачислений в валюте, поэтому по идее нужно вычесть из sum стоимость потраченных фишек (chips_cost)
        # Но мы этого делать не будем, т.к. они также учтены и в $debt (стоимость фишек включается в sum_spent)
        # Поэтому если вычитать, то нужно вычитать и из debt, а из неравенств ниже следует, что этого можно не делать
        my $wallet_sum_cur = $wallet_info->{sum};
        # Так как выполняем max($wallet_sum_cur, ...), то результат будет не меньше, чем $wallet_sum_cur
        my $limited_sum_cur = max($wallet_sum_cur, min(-$debt, $wallet_sum_cur + $available_auto_overdraft_lim));
        # Поэтому добавка будет неотрицательной
        return $limited_sum_cur - $wallet_sum_cur;
    } else {
        return $available_auto_overdraft_lim;
    }
}

=head2 calc_camp_uni_sums_with_wallet($camp, $sum_debt_all)

    Рассчитать остаток средств на кампании + кошельке с учетом общего счета.
    В результате создается (для обратной совместимости) новая структура sums_uni, с таким-же названием полей как и в $camp

    В подсчете участвуют следующие поля из $camp: sum, sum_spent, wallet_sum, wallet_sum_spent, ClientID

    А также $client_info следующей структуры (для рассчёта "добавки" по автоовердрафтам):

    $client_info = {
        clientID,            # идентификатор клиента
        debt,                # текущий долг клиента (размер выставленного, но ещё не оплаченного счёта)
        overdraft_lim,       # 0 или лимит овердрафта
        auto_overdraft_lim,  # 0 или порог отключения, заданный пользователем
        statusBalanceBanned  # "забанен" ли клиент в Балансе ('Yes' или 'No')
    }

    $client_info может быть не указан, тогда функция сходит за ним в базу сама.

    Формат создаваемой структуры:
    sums_uni => {
        currency => валюта,

        sum => всего оплачено на кампании, == $camp.sum
        sum_spent => всего потрачено на кампании, == $camp.sum_spent

        wallet_sum => всего оплачено на ОС, == $camp.wallet_sum
        wallet_sum_spent => потрачено на ОС помимо кампаний (всегда ноль),

        total => фактический остаток на кампании (доступный для откруток), с учетом задолженностей на кампаниях под ОС
        wallet_total => фактический остаток на ОС (доступный для откруток), с учетом задолженностей на кампаниях под ОС
        auto_overdraft_addition => деньги, которые клиент может потенциально использовать (или уже использовал) по автоовердрафтам
    }

    (!!) Функция должна вызываться ДО вычета НДС и добавления скидочного бонуса (Currencies::campaign_remove_nds_and_add_bonus)

=cut

sub calc_camp_uni_sums_with_wallet
{
    my ($camp, $sum_debt_all, $client_info) = @_;

    confess "invalid param: \$camp" unless defined $camp && ref($camp) eq 'HASH';
    confess "invalid param: \$sum_debt_all" if defined $sum_debt_all && ref($sum_debt_all) ne 'HASH';
    confess "invalid param: \$client_info" if defined $client_info && ref($client_info) ne 'HASH';

    $camp->{original_sum} = $camp->{sum};
    $camp->{original_sum_spent} = $camp->{sum_spent};

    my $total = $camp->{sum} - $camp->{sum_spent};

    my $sums_uni = $camp->{sums_uni} = {
        currency => $camp->{currency},

        sum => $camp->{sum},
        sum_spent => $camp->{sum_spent},
        total => $total,

        # Суммы для кошельков инициализируем нулями
        wallet_sum => 0,
        wallet_sum_spent => 0,
        wallet_total => 0,

        # Доступная добавка по автоовердрафтам (если она применима)
        auto_overdraft_addition => 0
    };

    if (!defined $client_info) {
        my $clients_info = Client::get_clients_auto_overdraft_info([$camp->{ClientID}]);
        $client_info = $clients_info->{$camp->{ClientID}};
    }

    # Проведем расчет с использованием ОС (для обычных кампаний)
    # (?) Возможно, тут нужно "ne 'wallet'" заменить на "eq 'text'"
    if (($camp->{mediaType} || $camp->{type}) ne 'wallet') {
        if ($camp->{wallet_cid}) {
            $sum_debt_all = get_sum_debt_for_wallets_by_uids([$camp->{uid}]) unless defined $sum_debt_all && defined $sum_debt_all->{$camp->{wallet_cid}};

            my $sum_debt_other = ($sum_debt_all->{$camp->{wallet_cid}} // 0) - ($total < 0 ? $total : 0);

            $sums_uni->{auto_overdraft_addition} = get_auto_overdraft_addition({
                type => 'wallet',
                currency => $camp->{currency},
                sum => ($camp->{wallet_sum} // 0),
                wallet_sum_debt => ($sum_debt_all->{$camp->{wallet_cid}} // 0)
            }, $client_info);

            $total += (($camp->{wallet_sum} // 0) - ($camp->{wallet_sum_spent} // 0)) + $sum_debt_other;
            my $wallet_total = (($camp->{wallet_sum} // 0) - ($camp->{wallet_sum_spent} // 0)) + ($sum_debt_all->{$camp->{wallet_cid}} // 0);

            $sums_uni->{wallet_sum} = $camp->{wallet_sum};
            $sums_uni->{wallet_sum_spent} = $camp->{wallet_sum_spent};
            $sums_uni->{total} = $total;
            $sums_uni->{wallet_total} = $wallet_total;
        }
    }
    # Для кампаний-кошельков проведем корректировку сумм отдельно
    else {
        $sum_debt_all = get_sum_debt_for_wallets_by_uids([$camp->{uid}]) unless defined $sum_debt_all && defined $sum_debt_all->{$camp->{cid}};

        $sums_uni->{total} += $sum_debt_all->{$camp->{cid}} // 0;

        $sums_uni->{auto_overdraft_addition} = get_auto_overdraft_addition({
            type => 'wallet',
            currency => $camp->{currency},
            sum => ($camp->{sum} // 0),
            wallet_sum_debt => ($sum_debt_all->{$camp->{cid}} // 0)
        }, $client_info);
    }

    return $camp;
}

=head2 get_wallet_spent_today($wallet_cid)

    Посчитать сегодняшние открутки по всем кампаниям под кошельком $wallet_cid

=cut

sub get_wallet_spent_today {
    my ($wallet_cid) = @_;

    my $orders_under_wallet = get_orders_under_wallet($wallet_cid);
    my $spent_today = Stat::OrderStatDay::get_order_spent_today_multi($orders_under_wallet);

    return sum0(map { $_ // 0 } values %$spent_today);
}

=head2 get_orders_under_wallet($wallet_cid)

    Возвращает массив OrderID кампаний под кошельком $wallet_cid

=cut

sub get_orders_under_wallet {
    my ($wallet_cid) = @_;

    return mass_get_orders_under_wallet([$wallet_cid])->{$wallet_cid};
}

=head2 mass_get_orders_under_wallet($wallet_cids, %O)

    Возвращает хеш с массивами OrderID кампаний под кошельками из массива $wallet_cids

    Именованные параметры
        wallets_client_ids - хеш wallet_cid => ClientID, который можно будет использовать для ускорения запроса.
            Если этот параметр не передать, функция сама сходит за ClientID всех указанных кошельков.

    Формат выходного хеша
    {
        wallet_cid => [OrderID, OrderID, OrderID],
        wallet_cid => [OrderID, OrderID, OrderID],
    }
    Если у какого-то кошелька нет кампаний с OrderID, для его wallet_cid возвращается пустой массив

=cut

sub mass_get_orders_under_wallet {
    my ($wallet_cids, %O) = @_;

    my $wallets_client_ids = $O{wallets_client_ids};
    unless ($wallets_client_ids) {
        $wallets_client_ids = get_hash_sql(PPC(cid => $wallet_cids), [
                "SELECT cid, ClientID FROM campaigns", WHERE => {cid => SHARD_IDS}
            ]);
    }

    my $result = {};
    foreach_shard cid => $wallet_cids, chunk_size => 1_000, sub {
        my ($shard, $wallet_cids) = @_;
        my $client_ids = [ uniq map { $wallets_client_ids->{$_} } @$wallet_cids ];

        my $rows = get_all_sql(PPC(shard => $shard), [
                "SELECT wallet_cid, OrderID FROM campaigns",
                    WHERE => {
                        ClientID__int => $client_ids,
                        wallet_cid => $wallet_cids,
                        OrderID__int__gt => 0,
                    }
            ]);

        for my $row (@$rows) {
            push @{$result->{$row->{wallet_cid}}}, $row->{OrderID};
        }
        for my $wallet_cid (@$wallet_cids) {
            $result->{$wallet_cid} //= [];
        }
    };

    return $result;
}

=head2 is_new_payment_workflow_enabled

    Получить признак доступность нового воркфлоу оплаты
    1) Включена фича
    2) Резидент в России в рублях
    3) Не агентский

=cut

sub is_new_payment_workflow_enabled {
    my ($client_id, $client_data) = @_;

    return Client::ClientFeatures::has_new_payment_workflow_enabled_feature($client_id)
        && ($client_data->{non_resident} eq '0') && ($client_data->{country_region_id} eq $geo_regions::RUS) && ($client_data->{work_currency} eq 'RUB')
        && !(defined $client_data->{agency_client_id});
}

=head2 get_new_payment_workflow_values

    Получить данные для нового воркфлоу оплаты

=cut

sub get_new_payment_workflow_values {
    my ($uid, $client_id, $wallet_cid, $remote_ip, $autooverdraft_enabled, $autooverdraft_active) = @_;

    my $profile = Yandex::Trace::new_profile('payment:get_values');

    my $vars = {};

    my $autopay_active = get_one_field_sql(PPC(ClientID => $client_id),
        ["select autopay_mode = 'min_balance' from wallet_campaigns",
          where => { wallet_cid => $wallet_cid }]);
    $vars->{payment_suggest_block} = calc_payment_suggest_block($client_id, $autopay_active, $autooverdraft_enabled, $autooverdraft_active);

    hash_merge $vars, get_autopay_suggest_sums();

    state $prop_autooverdraft_coef //= Property->new("AUTOOVERDRAFT_SUGGEST_COEF");

    $vars->{autooverdraft_suggest_coef} = $prop_autooverdraft_coef->get(60) // AUTOOVERDRAFT_DEFAULT_SUGGEST_COEF;

    state $autopay_save_period //= Property->new("NEW_PAYMENT_WORKFLOW_AUTOPAY_SAVE_PERIOD");
    $vars->{autopay_save_period} = $autopay_save_period->get(60);

    return $vars;
}

=head2 get_autopay_suggest_sums

    Получить суммы, которые нужно предлагать для настройки автопополнения

=cut

sub get_autopay_suggest_sums {
    state $prop_payment_sum //= Property->new("AUTOPAY_PAYMENT_SUM");
    state $prop_remaining_sum //= Property->new("AUTOPAY_REMAINING_SUM");

    return {
        autopay_suggest => {
            payment_sum => ($prop_payment_sum->get(60) // AUTOPAY_DEFAULT_PAYMENT_SUM),
            remaining_sum => ($prop_remaining_sum->get(60) // AUTOPAY_DEFAULT_REMAINING_SUM)
        }
    };
}

=head2 get_auto_overdraft_params

    Получить данные о пороге отключения

=cut

sub get_auto_overdraft_params {
    my ($client_client_id, $wallet, $work_currency, $client_nds)  = @_;

    my $vars = {};

    my $autooverdraft_params = Client::_get_autooverdraft_params($client_client_id);
    if (defined $autooverdraft_params && defined $autooverdraft_params->{overdraft_lim}) {
        my $overdraft_limit = delete $autooverdraft_params->{overdraft_lim};
        my $debt = delete $autooverdraft_params->{debt};
        my $auto_overdraft_limit = delete $autooverdraft_params->{auto_overdraft_lim};
        my $is_brand = delete $autooverdraft_params->{is_brand};
        $vars->{autooverdraft_enabled} = (any {$_ eq $work_currency} @Client::AUTOOVERDRAFT_CURRENCY_AVAILABLE)
            && defined $wallet
            && $wallet->{enabled}
            && ($overdraft_limit > 0 || $auto_overdraft_limit > 0)
            && !$is_brand
            && !(any {$_ eq $client_client_id} @Client::BAD_AUTO_OVERDRAFT_CLIENTS_LIST)
            ? 1 : 0;
        if ($vars->{autooverdraft_enabled}) {
            $vars->{autooverdraft_params} = $autooverdraft_params;

            my $overspending = -$wallet->{total_with_nds};
            my $auto_overdraft_upper_limit_with_nds = max($debt, $overspending + $debt, $overdraft_limit);
            $vars->{autooverdraft_params}->{auto_overdraft_upper_limit} = sprintf("%.2f", $auto_overdraft_upper_limit_with_nds);
            my $auto_overdraft_upper_limit_without_nds = Currencies::remove_nds($auto_overdraft_upper_limit_with_nds, $client_nds // 0);
            $vars->{autooverdraft_params}->{auto_overdraft_upper_limit_without_nds} = sprintf("%.2f", $auto_overdraft_upper_limit_without_nds);
            my $auto_overdraft_limit_default_min_value =
                delete $vars->{autooverdraft_params}->{auto_overdraft_limit_default_min_value};
            # округление до копеек, чтобы избежать редкой ошибки валидации,
            # когда округленный порог больше округленной верхней границы
            $vars->{autooverdraft_params}->{auto_overdraft_lower_limit} = sprintf("%.2f", max($debt, $overspending + $debt, $auto_overdraft_limit_default_min_value));
            $vars->{autooverdraft_params}->{auto_overdraft_lim} = $auto_overdraft_limit;
            $vars->{autooverdraft_reset_enabled} = $auto_overdraft_limit > 0 && $overspending <= 0 ? 1 : 0;
        }
    }

    return $vars;
}

=head2 get_or_create_ph_person

    Отдает неархивного плательщика-физика (любого) из Баланса или создает такого

=cut

sub get_or_create_ph_person {
    my $uid = shift;
    my $wallet_cid = shift;

    my $client_id = get_clientid(cid => $wallet_cid);
    my $persons = balance_get_client_persons($client_id);

    my $result_person_id = -1;
    foreach my $person (@$persons) {
        if (($person->{type} eq 'ph') && !$person->{hidden}) {
            $result_person_id = $person->{id};
            last;
        }
    }

    if ($result_person_id == -1) {
        my $user_data = get_user_data($uid, [qw/fio email phone/]);

        my @fio_splitted = split /\s+/, $user_data->{fio}, 2;

        my ($lname, $fname);

        if ($user_data->{fio} =~ /^\s*$/) {
            $fname = DEFAULT_NAME;
            $lname = DEFAULT_LAST_NAME;
        } elsif (scalar @fio_splitted == 1) {
            $fname = DEFAULT_NAME;
            $lname = $user_data->{fio};
        } else {
            $fname = $fio_splitted[0];
            $lname = $fio_splitted[1];
        }

        $result_person_id = (balance_create_person($uid, {
            client_id => $client_id,
            type => 'ph',
            lname => $lname, fname => $fname, mname => DEFAULT_NAME,
            phone => $user_data->{phone} // '+0',
            email => $user_data->{email}
        }))[0];
    }

    return $result_person_id;
}

1;
