#######################################################################
#
#  Direct.Yandex.ru
#
#  XLSCampExport
#   Functions for XLS campaign export/import
#
#  $Id$
#
#######################################################################

=head1 NAME

XLSCampExport

=head1 DESCRIPTION

Functions for XLS campaign export/import

=cut

package XLSCampImport;

use Direct::Modern;

use base qw/Exporter/;

our @EXPORT = qw/
        xls2camp_snapshot
        get_snapshot_warnings_and_errors_for_exists_camp
        csv2camp_snapshot
        camp_snapshot2excel
        get_camp_snapshot
        merge_camp_snapshot
        
        get_sample_camp_snapshot
        
        validate_tags_snapshot
        fill_empty_price_for_snapshot
        fill_empty_price_for_phrase
        
        check_common_contact_info_for_snapshot
        check_common_geo_for_snapshot

        has_oversized_banners
        split_oversized_groups
        check_domain_availability
        compare_currencies
        add_xls_image_processing_info
        get_adgroups_store_content_hrefs
        validate_rmp_store_hrefs
    /;

use Carp qw/croak/;
use List::Util qw/max min sum first/;
use List::MoreUtils qw/any none uniq all part firstidx zip/;
use List::UtilsBy qw/count_by/;
use Yandex::ListUtils;
use Yandex::Clone qw/yclone/;

use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xuniq/;
use Yandex::ScalarUtils;
use Yandex::TimeCommon;
use Yandex::ReportsXLS;
use Yandex::ReportsXLSX;
use Yandex::URL qw/get_host/;
use Yandex::CheckMobileRedirect qw/parse_store_url/;

use Settings;
use Yandex::MyGoodWords;
use URLDomain;
use Primitives;
use PrimitivesIds;
use Currencies;
use Common qw/:subs :globals/;
use VCards;
use Direct::Model::BidRelevanceMatch::Helper qw/$CONDITION_XLS_PREFIX_RELEVANCE_MATCH $CONDITION_XLS_RELEVANCE_MATCH/;
use Direct::Validation::VCards qw/vcard_im_client_options/;
use Direct::Validation::Banners qw//;
use Direct::Validation::SitelinksSets qw//;

use Sitelinks;
use Tools;
use TextTools;
use TTTools;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use GeoTools;
use geo_regions;
use MailNotification;
use RedirectCheckQueue;
use Campaign;
use PhraseText;
use Phrase;
use BannersCommon;
use BannerFlags;
use Tag;
use XLSParse;
use XLSVocabulary;
use Client;
use Rbac;
use RBACElementary;
use RBACDirect;
use XLSCampaign;
use BannerImages;
use Notification;
use MinusWords;
use MinusWordsTools;
use Retargeting;
use User;
use MobileContent;
use Wallet;

use Models::AdGroup;
use Models::Banner;
use Models::Phrase;

use Direct::AdGroups2::Smart;

# ограничение на количество строк в excel файле по умолчанию
our $DEFAULT_EXCEL_ROWS_LIMIT = 60_000;

sub _get_full_childs_list;

# --------------------------------------------------------------------

sub _merge_phrases_snapshot {
    
    my ($xls_phrases, $curr_phrases, $options) = @_;
    $curr_phrases ||= {};

    my $default_price = get_currency_constant($options->{currency}, 'DEFAULT_PRICE');
    my $fill_price_options = hash_cut $options, qw/strategy search_strategy context_strategy/;
    $fill_price_options->{default_price} = $default_price;

    # В некоторых случаях одна фраза может размножиться на несколько. И у этих нескольких фраз будет один айдишник.
    # Нужно оставить айдишник на той фразе, которая совпадает с оригинальной или (если таковой нет) на любой из дублей.
    # Тогда старая фраза будет помечена как измененная, а все оставшиеся дубли добавлены как новые.
    my %id2phrase;
    for my $x (grep { $_->{id} } @$xls_phrases) {
        push @{$id2phrase{$x->{id}}}, hash_merge {orig => $x}, get_phrase_props($x->{phr});
    }
    for my $id (grep { @{$id2phrase{$_}} > 1 && $curr_phrases->{$_} } keys %id2phrase) {
        my $old_phrase = $curr_phrases->{$id};
        my $keep_idx = firstidx { $_->{md5} eq $old_phrase->{md5} } @{$id2phrase{$id}};
           $keep_idx = 0 if $keep_idx == -1;
        $id2phrase{$id}->[$_]->{orig}->{id} = undef for grep { $_ != $keep_idx } 0..$#{$id2phrase{$id}};
    }

    my %added_phrases;
    foreach my $new_phrase (@$xls_phrases) {
        my $old_phrase;
        $old_phrase = $curr_phrases->{$new_phrase->{id}} if $new_phrase->{id} && $curr_phrases->{$new_phrase->{id}}; 
        fill_empty_price_for_phrase($new_phrase, %$fill_price_options, old_phrase => $old_phrase);

        unless ($old_phrase) {
            $new_phrase->{id} = 0 if exists $new_phrase->{id};
            next;
        };

        $added_phrases{$new_phrase->{id}} = 1;
        # изменились только минус-слова
        if (! $options->{apply_changes_minus_words}) {
            my ($old_plus_phrase, $old_minus_phrase) = _split_by_minus_words($old_phrase->{phr});
            my ($new_plus_phrase, $new_minus_phrase) = _split_by_minus_words($new_phrase->{phr});
            if ($old_plus_phrase eq $new_plus_phrase && $old_minus_phrase ne $new_minus_phrase) {
                # оставляем исходную фразу если пользователь захотел и изменились только минус-слова
                $new_phrase->{phr} = $old_phrase->{phr};
            }
        }

        # стояла опция - "Не изменять ставки у существующих фраз"
        if ($options->{dont_change_prices}) {
            $new_phrase->{price} = $old_phrase->{price};
            $new_phrase->{price_context} = $old_phrase->{price_context};
        }

        # игнорируем изменения цен в автобюджетных кампаниях
        if ($options->{strategy} && none {$options->{strategy} eq $_} @Campaign::MANUAL_PRICE_STRATEGIES) {
            $new_phrase->{price} = $old_phrase->{price};
            $new_phrase->{price_context} = $old_phrase->{price_context} if defined $old_phrase->{price_context};
        }

        # игнорируем изменения цен price_context в кампаниях не с отдельным размещением
        if ($old_phrase->{price_context}
            && $options->{strategy} && $options->{strategy} ne 'different_places'
            && !(defined $old_phrase->{rank} && $old_phrase->{rank} == 0)) {
                
            $new_phrase->{price_context} = $old_phrase->{price_context}
        }
                
        # кампания с раздельным размещением
        if ($options->{strategy} && $options->{strategy} eq 'different_places'
            && $options->{search_strategy} && $options->{search_strategy} eq 'stop') {
                
            $new_phrase->{price} = $old_phrase->{price};
            if ($options->{context_strategy} && none {$options->{context_strategy} eq $_} qw/default maximum_coverage/) {
                $new_phrase->{price_context} = $old_phrase->{price_context};
            }
        }
        $new_phrase->{autobudgetPriority} = $old_phrase->{autobudgetPriority} || 3;
    }
    
    return [
        @$xls_phrases,
        $options->{remove_lost_phrases}
            ? ()
            : grep {!$added_phrases{$_->{id}}} values %$curr_phrases
    ]
}

# --------------------------------------------------------------------

sub _merge_retargetings_snapshot {
    my ($new_retargetings, $old_retargetings, $options) = @_;

    my $for_delete = 0;
    my $to_add = 0;
    my $merged_retargetings = yclone($old_retargetings);
    my $price_changed;
    for my $ret (@$new_retargetings){
        if(not exists $merged_retargetings->{$ret->{ret_cond_id}}){
            $to_add++;
            $merged_retargetings->{$ret->{ret_cond_id}} = {};
        }

        # игнорируем изменения цен в автобюджетных кампаниях
        if ($options->{strategy} && none {$options->{strategy} eq $_} @Campaign::MANUAL_PRICE_STRATEGIES) {
            $ret->{price_context} = $merged_retargetings->{$ret->{ret_cond_id}}->{price_context};
            $price_changed = 1;
        }

        hash_merge $merged_retargetings->{$ret->{ret_cond_id}}, $ret;
    }

    if($options->{remove_lost_retargetings}){
        for my $ret_cond_id (keys %$merged_retargetings){
            if(none { $ret_cond_id == $_ } map {$_->{ret_cond_id}} @{$new_retargetings}){
                delete $merged_retargetings->{$ret_cond_id};
                $for_delete++;
            }
        }
    }

    my $result;
    if($for_delete || $to_add || $price_changed){
        $result = {retargetings => [values %$merged_retargetings]};
        $result->{to_add} =  $to_add if $to_add;
        $result->{for_delete} =  $for_delete if $for_delete;
    }

    return  $result;
}

# --------------------------------------------------------------------

sub _merge_relevance_match_snapshot {
    my ($new_relevance_matches, $old_relevance_matches, $options) = @_;
    $old_relevance_matches //= {};

    my $for_delete = 0;
    my $to_add = 0;
    my $changed;

    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($options->{campType}, $options->{ClientID});
    my %added_relevance_match;
    foreach my $new_relevance_match (@$new_relevance_matches) {
        my $old_relevance_match;

        $old_relevance_match = $old_relevance_matches->{$new_relevance_match->{bid_id}} if $new_relevance_match->{bid_id} && $old_relevance_matches->{$new_relevance_match->{bid_id}};
        unless ($old_relevance_match) {
            $to_add++;
            $new_relevance_match->{bid_id} = 0;
            next;
        }

        $new_relevance_match->{autobudgetPriority} = $old_relevance_match->{autobudgetPriority} || 3;
        delete $new_relevance_match->{price_context} unless $has_extended_relevance_match;

        $added_relevance_match{$new_relevance_match->{bid_id}} = 1;

        # стояла опция - "Не изменять ставки у существующих фраз"
        if ($options->{dont_change_prices}) {
            $new_relevance_match->{price} = $old_relevance_match->{price};
            $new_relevance_match->{price_context} = $old_relevance_match->{price_context};
        }

        # игнорируем изменения цен в автобюджетных кампаниях
        if ($options->{strategy} && none {$options->{strategy} eq $_} @Campaign::MANUAL_PRICE_STRATEGIES) {
            $new_relevance_match->{price} = $old_relevance_match->{price};
            $new_relevance_match->{price_context} = $old_relevance_match->{price_context} if $has_extended_relevance_match;
        }

        # кампания с раздельным размещением
        if ($options->{strategy} && $options->{strategy} eq 'different_places'
            && $options->{search_strategy}) {
            $new_relevance_match->{price} = $old_relevance_match->{price} if $options->{search_strategy} eq 'stop';
            $new_relevance_match->{price_context} = $old_relevance_match->{price_context}
                if  $has_extended_relevance_match &&
                    $options->{context_strategy} && none {$options->{context_strategy} eq $_} qw/default maximum_coverage/;
        }

        $changed = 1;
    }

    if ($options->{remove_lost_phrases}) {
        for my $bid_id (keys %$old_relevance_matches){
            if(none { $bid_id == $_ } map {$_->{bid_id}} @{$new_relevance_matches}){
                $for_delete++;
            }
        }
    }

    my $result;
    if ($for_delete || $to_add || $changed) {
        $result->{relevance_matches} = [
            @$new_relevance_matches,
            ($options->{remove_lost_phrases} ? () : grep {!$added_relevance_match{$_->{bid_id}}} values %$old_relevance_matches)
        ];
        $result->{to_add} = $to_add if $to_add;
        $result->{for_delete} = $for_delete if $for_delete;
    }

    return $result;
}

# --------------------------------------------------------------------

sub _merge_target_interests_snapshot {
    my ($new_target_interests, $old_target_interests, $options) = @_;

    my $for_delete = 0;
    my $to_add = 0;
    my $merged_target_interests = yclone($old_target_interests);
    my $price_changed;
    for my $target_interest (@$new_target_interests){
        if (not exists $merged_target_interests->{$target_interest->{target_category_id}}) {
            $to_add++;
            $target_interest->{ret_id} = 0;
            $merged_target_interests->{$target_interest->{target_category_id}} = {};
        }

        # игнорируем изменения цен в автобюджетных кампаниях
        if ($options->{strategy} && none {$options->{strategy} eq $_} @Campaign::MANUAL_PRICE_STRATEGIES) {
            $target_interest->{price_context} = $merged_target_interests->{$target_interest->{target_category_id}}->{price_context};
            $price_changed = 1;
        }

        hash_merge $merged_target_interests->{$target_interest->{target_category_id}}, $target_interest;
    }

    if ($options->{remove_lost_retargetings}) {
        for my $target_category_id (keys %$merged_target_interests){
            if(none { $target_category_id == $_ } map {$_->{target_category_id}} @{$new_target_interests}){
                delete $merged_target_interests->{$target_category_id};
                $for_delete++;
            }
        }
    }

    my $result;
    if($for_delete || $to_add || $price_changed){
        $result = {target_interests => [values %$merged_target_interests]};
        $result->{to_add} =  $to_add if $to_add;
        $result->{for_delete} =  $for_delete if $for_delete;
    }

    return  $result;
}

sub _validate_merged_campaign {
    
    my ($groups, $old_groups) = @_;
    
    my @errors;
    # проверяем ошибки
    for my $group (@$groups) {
        
        if ($group->{pid} && !$old_groups->{$group->{pid}}) {
            push @errors,
                $group->{group_name}
                    ? iget('Вы изменили значение поля ID группы в строке "%s". При внесении изменений в существующую кампанию не допускается изменение поля ID группы. При создании новой кампании или группы объявлений поле ID группы должно оставаться пустым', $group->{line_number}) 
                    : iget('Группа объявлений №-%d не существует.', $group->{pid});
            next;
        }
        my $old_group = $old_groups->{$group->{pid}} || {};
        my $old_banners = $old_group->{banners} || {}; 

        if ($old_groups->{$group->{pid}} && $group->{adgroup_type} eq 'mobile_content') {
            if ($group->{store_content_href} ne $old_group->{store_content_href}) {
                push @errors, iget("Строка %s: невозможно изменить ссылку на рекламируемое приложение.", $group->{line_number});
            }
        }

        for my $banner (@{$group->{banners}}) {
            if (($group->{adgroup_type} // 'base') eq 'base'
                && length(str($banner->{href})) == 0
                && $banner->{contact_info} eq '-'
                && !$banner->{turbolanding_id}
                && !$banner->{permalink}
            ) {
                push @errors, iget("Строка %s: объявление не может быть сохранено без контактной информации и без ссылки на сайт.", $banner->{line_number});
            }
            if ($banner->{bid}) {
                if (my $old_banner = $old_banners->{$banner->{bid}}) {
                    if ($old_banner->{banner_type} ne $banner->{banner_type}) {
                        push @errors, $banner->{banner_type} eq 'mobile'
                            ? iget('Строка %s: невозможно изменить тип десктопного объявления на мобильный', $banner->{line_number})
                            : iget('Строка %s: невозможно изменить тип мобильного объявления на десктопный', $banner->{line_number});
                    }
                } else {
                    push @errors, iget("Объявление M-%d не существует.", $banner->{bid});
                }
            }
        }
        
        my $old_phrases = $old_group->{phrases} || {};
        for my $phrase (@{$group->{phrases}}) {
            if ($phrase->{id} && !$old_phrases->{$phrase->{id}}) {
                # TODO adgroup: удалить про баннеры
                if (@{$group->{banners}} > 1) {
                    push @errors,
                        $group->{group_name}
                            ? iget('В группе объявлений "%s" не существует фразы № %d.', $group->{group_name}, $phrase->{id}) 
                            : iget('В группе объявлений не существует фразы № %d.', $phrase->{id});
                } else {
                    push @errors, iget("В объявлении M-%d не существует фразы № %d.", $group->{banners}->[0]->{bid}, $phrase->{id});
                }
            }
        }
    }
    
    return @errors;
}

=head2 get_max_rows_limit($uid)

    Получить максимально допустимое количество строк в экспортируемом excel файле

=cut

sub get_max_rows_limit {
    my $uid = shift;
    return get_one_user_field($uid, 'excel_rows_limit') || $DEFAULT_EXCEL_ROWS_LIMIT;
}


=head2 merge_camp_snapshot

    создать новую кампанию на основе xls и текущей кампании
    ($vars->{new_camp}, $vars->{merge_errors}, $vars->{merge_warnings}, $vars->{changes}, $vars->{stop_groups})
        = merge_camp_snapshot_with_one_diff2($curr_camp, $xls_camp, $options);

    $curr_camp - текущая кампания из БД
    $xls_camp - загружаемая кампания
    $options:
        strategy => $vars->{curr_camp}->{strategy}
        search_strategy => $vars->{curr_camp}->{search_strategy}
        context_strategy => $vars->{curr_camp}->{context_strategy}
        dont_change_prices => не менять цены на фразах
        remove_lost_banners => удалять баннеры при отсутствии их в xls
        remove_lost_phrases => удалять фразы при отсутствии их в xls
        remove_lost_retargetings => удалять условия ретаргетинга при отсутствии их в xls
        remove_lost_groups => удалять группы при отсутствии их в xls
        apply_changes_minus_words => если поменялись только минус-слова во фразе, то можно отказаться от такого изменения - 0, применить как в xls - 1
        host => текущий хост пользователя, для определения url изображения
        currency => валюта
        send_new_banners_to_moderation => отправлять ли новые баннеры на модерацию при этом.
        is_internal_user => 0|1 - при подсчете баллов учитывать ли уточнения (временно для клиентов без уточнений - нет)
        ignore_image => 0|1 - не учитывать картинки из xls

    $vars->{new_camp} - кампания которая получилась в результате
    $vars->{merge_errors} - ошибки
    $vars->{merge_warnings} - предупреждения

    изменения (кол-ва удаленных/соданных/измененных баннеров/фраз)
    $vars->{changes}: {
        create_banner => ...
        edit_banner => ...
        remove_banner => ...

        create_phrase => ...
        edit_phrase => ...
        remove_phrase => ...
    }

    $vars->{stop_groups} - группы баннеров которые необходимо остановить { pid => [bid, bid, bid], pid => [bid, bid, bid] }

=cut

sub merge_camp_snapshot($$;$) {
    my ($curr_camp, $xls_camp, $options) = @_;
    my $new_camp = {};
    $new_camp->{cid} = $xls_camp->{cid} if $xls_camp->{cid};
    my $is_contact_info_changed = _serialize_contact_info($curr_camp->{contact_info}) ne _serialize_contact_info($xls_camp->{contact_info}) ? 1 : 0;
    # контактная информация из xls если она там есть
    if (ref($xls_camp->{contact_info}) eq 'HASH' && defined $xls_camp->{contact_info}->{phone}) {
        $new_camp->{contact_info} = $xls_camp->{contact_info};
    } else {
        $new_camp->{contact_info} = $curr_camp->{contact_info};
    }

    $options->{campType} = $curr_camp->{type} // $curr_camp->{mediaType};
    # Единые минус-слова на кампанию
    $new_camp->{campaign_minus_words} = exists $xls_camp->{campaign_minus_words} ? $xls_camp->{campaign_minus_words} : $curr_camp->{campaign_minus_words};

    $new_camp->{opts} = $curr_camp->{opts};
    
    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($options->{campType}, $options->{ClientID});
    
    my %curr_groups = map {
        my $group = $_;
        $group->{pid} => hash_merge
            {
                phrases => {map {($_->{id} => yclone($_))} @{$group->{phrases}}},
                relevance_match => {map {($_->{bid_id} => yclone($_))} @{$group->{relevance_match}}},
                retargetings => {map {($_->{ret_cond_id} => yclone($_))} @{$group->{retargetings}}},
                target_interests => {map {($_->{target_category_id} => yclone($_))} @{$group->{target_interests} // []}},
                banners => {map {($_->{bid} => yclone($_))} @{$group->{banners}}},
                sum => $curr_camp->{sum}
            },
            hash_cut $group, qw/
                pid geo group_name adgroup_type tags minus_words statusModerate statusPostModerate
                store_content_href device_type_targeting network_targeting min_os_version
            /
    } @{$curr_camp->{groups}};
    
    die 'currency not given' unless $options->{currency};
    
    # Значение по умолчанию для группы, которая только появилась.
    my $default_new_statusModerate = $options->{send_new_banners_to_moderation} ? 'Ready' : 'New';
    my (@groups, @stop_banners);
    foreach my $group (@{$xls_camp->{groups}}) {
        my $new_group = yclone($group);
        my $old_group = ($group->{pid} ? $curr_groups{$group->{pid}} : undef) || {};
        # Для новой группы этот статус никак не используется дальше. Так как в Models::AdGroup::save_group для новой группы он принудительно проставляется
        # в New, и только если есть баннеры, отправленные на модерацию, тогда и группа отправляется.
        # Тут это это остается только для того, чтобы в случае перевода с Models::AdGroup на модели, когда данная неявная фича может испариться,
        # статус был все же проставлен правильно. Ниже проставляются статусы и для новых баннеров.

        $new_group->{statusModerate} = $old_group->{statusModerate} || $default_new_statusModerate;
        $new_group->{save_as_draft} = 1 if (!$options->{send_new_banners_to_moderation} && !$old_group->{statusModerate});
        $new_group->{phrases} = _merge_phrases_snapshot($group->{phrases}, $old_group->{phrases}, $options);
        my $merged_relevance_matches = _merge_relevance_match_snapshot($group->{relevance_match}, $old_group->{relevance_match}, $options);
        if ($merged_relevance_matches && %$merged_relevance_matches) {
            $new_group->{relevance_match} = $merged_relevance_matches->{relevance_matches};
            $new_group->{relevance_match_to_add} = $merged_relevance_matches->{to_add};
            $new_group->{relevance_match_for_delete} = $merged_relevance_matches->{for_delete};
        }

        my $merged_retargetings = _merge_retargetings_snapshot($group->{retargetings}, $old_group->{retargetings}, $options);
        if($merged_retargetings && %$merged_retargetings) {
            $new_group->{retargetings} = $merged_retargetings->{retargetings};
            $new_group->{retargetings_to_add} = $merged_retargetings->{to_add};
            $new_group->{retargetings_for_delete} = $merged_retargetings->{for_delete};
        }

        my $merged_target_interests = _merge_target_interests_snapshot($group->{target_interests} // [], $old_group->{target_interests}, $options);
        if($merged_target_interests && %$merged_target_interests) {
            $new_group->{target_interests} = $merged_target_interests->{target_interests};
            ($new_group->{retargetings_to_add} //= 0 ) += $merged_target_interests->{to_add} // 0;
            ($new_group->{retargetings_for_delete} //= 0 ) += $merged_target_interests->{for_delete} // 0;
        }

        ensure_phrase_have_props($_) foreach @{$new_group->{phrases}};
        $new_group->{phrases} = [xuniq {$_->{norm_phrase}} @{$new_group->{phrases}}];

        # Скопируем старые минус-слова, если в новой группе такого поля не существует
        $new_group->{minus_words} = $old_group->{minus_words} unless exists $new_group->{minus_words};

        # переносим сохраняемые поля со старых баннеров
        for my $new_banner (@{$new_group->{banners}}) {
            my $bid = $new_banner->{bid}  or next;
            my $old_banner = $old_group->{banners}->{$bid}  or next;
            
            unless ($xls_camp->{header_keys}->{turbolanding_id}) {
                if (exists $old_banner->{turbolanding}){
                    $new_banner->{turbolanding} = $old_banner->{turbolanding};
                }
    
                if (exists $old_banner->{sitelinks} && @{$old_banner->{sitelinks} // []}){
                    # Т.к. мы не можем однозначно привязать турболендинги существующего баннера к импортируемым сайтлинкам
                    # приклеим их в сайтлинк с аналогичным href - сайтлинки с одинаковыми href  внутри не имеют смысла,
                    # при изменении href логично отвязать турболендинг.
                    my %tl_by_href;
                    foreach (@{$old_banner->{sitelinks}}){
                        $tl_by_href{$_->{href}} //=  $_->{turbolanding} if exists $_->{turbolanding};
                    }
    
                    foreach my $sl (@{$new_banner->{sitelinks} // []}){
                        next unless exists $tl_by_href{$sl->{href}};
                        $sl->{turbolanding} = $tl_by_href{$sl->{href}};
                    }
                }
            }
            
            if ($options->{ignore_image}) {
                $new_banner->{image_hash} = $new_banner->{image} = $old_banner->{image};
                $new_banner->{image_url} = BannerImages::get_image_url($old_banner, {host => $options->{host}});
            }
            if ($options->{ignore_display_href}) {
                $new_banner->{display_href} = $old_banner->{display_href};
            }
            if ($old_banner->{video_resources} && !$new_banner->{video_resources}) {
                if ($xls_camp->{header_keys}->{creative_id}) {
                    # если это новый xls файл со столбцом Креатив - можно удалить
                    $new_banner->{video_resources} = {};
                }
                else {
                    # загрузили старый файл - не удаляем видео-дополнение
                    $new_banner->{video_resources} = $old_banner->{video_resources};
                }
            }
        }

        my %new_banners = map {$_->{bid} => 1} @{$new_group->{banners}};
        if ($options->{remove_lost_banners}) {
            foreach my $banner (values %{$old_group->{banners} || {}}) {
                next if $new_banners{$banner->{bid}};
                if (has_delete_banner_problem({bid=>$banner->{bid},
                                              BannerID=>$banner->{BannerID},
                                              banner_statusModerate => $banner->{statusModerate},
                                              banner_statusPostModerate => $banner->{statusPostModerate},
                                              group_statusModerate => $old_group->{statusModerate},
                                              group_statusPostModerate => $old_group->{statusPostModerate},
                                              sum => $curr_camp->{sum} })) {
                    push @stop_banners, $banner->{bid};
                }

            }
        } else {
            my %skipped_bids_hash = map {$_ => 1} @{$curr_camp->{skipped_bids} || []};
            my @rest_banners = grep {!$new_banners{$_->{bid}}} values %{$old_group->{banners} || {}};

            push @{$new_group->{banners}}, map {
                my $banner = $_;
                $banner->{image_url} = BannerImages::get_image_url($banner, {host => $options->{host}});
                $banner;
            } @rest_banners; 
            $new_group->{banners} = [grep {!$skipped_bids_hash{$_->{bid}}} @{$new_group->{banners}}];
        }

        $old_group->{is_processed} = 1;
        
        push @groups, $new_group;
    }

    if ($options->{remove_lost_groups}) {
        foreach my $group (values %curr_groups) {
            next if $group->{is_processed};
            foreach my $banner (values %{$group->{banners}}) {
                if (has_delete_banner_problem({bid=>$banner->{bid},
                                              BannerID=>$banner->{BannerID},
                                              banner_statusModerate => $banner->{statusModerate},
                                              banner_statusPostModerate => $banner->{statusPostModerate},
                                              group_statusModerate => $group->{statusModerate},
                                              group_statusPostModerate => $group->{statusPostModerate},
                                              sum => $curr_camp->{sum} })) {
                    push @stop_banners, $banner->{bid};
                }
            }
        }
    } else {
        push @groups, map {
            my $g = $_;
            hash_merge
                hash_cut($g, qw/
                    pid geo group_name adgroup_type minus_words statusModerate tags
                    store_content_href device_type_targeting network_targeting min_os_version
                /),
                {
                    banners => [values %{$g->{banners}}],
                    phrases => [values %{$g->{phrases}}],
                    relevance_match => [values %{$g->{relevance_match}}],
                    retargetings => [values %{$g->{retargetings}}],
                    target_interests => [values %{$g->{target_interests}}],
                } 
        } grep {!$_->{is_processed}} values %curr_groups
    }
    
    my @errors = _validate_merged_campaign(\@groups, \%curr_groups);
    $new_camp->{groups} = \@groups;

    # Валидация единых минус-слов
    my @warnings = @{validate_group_camp_minus_words($new_camp, use_line_numbers => 1)->{warnings}};
    if (@{$curr_camp->{skipped_bids} || []}) {
        push @warnings, iget("Изменения в следующих объявлениях будут проигнорированы, так как они заархивированы: %s", 
                             join ', ', map {"M-$_"} @{$curr_camp->{skipped_bids} || []});
    }
    # Посмотрим что изменилось после объединения кампаний
    my %changes;
    for my $group (@groups) {
        
        my $old_group = ($group->{pid} ? delete $curr_groups{$group->{pid}} : undef) || {};
        my $old_banners = $old_group->{banners} || {};
        
        unless ($group->{pid}) {
            $group->{_added} = 1;
            $changes{create_group}++;
        } else {
            if (_serialize_group($group) ne _serialize_group($old_group)) {
                $changes{edit_group}++;
                $group->{_changed} = 1;
            } 
        }
        
        for my $banner (@{$group->{banners}}) {
            unless ($banner->{bid}) {
                $banner->{_added} = 1;
                $group->{_has_changes_in_banners} = 1;
                $changes{create_banner}++;
                # Проставляем новому баннеру статус (зависит от галки "отправлять новые объявления на модерацию"),
                # Далее, в Models::AdGroup::save_group при наличии хотябы одного баннера в Ready, сама группа тоже отправляется на модерацию (statusModerate=Ready)
                $banner->{statusModerate} = $default_new_statusModerate;
                $banner->{save_as_draft} = 1 if !$options->{send_new_banners_to_moderation};

            } else {
                my $use_callouts_for_calc_units = $options->{is_internal_user} || ref($banner->{callouts}) eq 'ARRAY';
                my $old_banners_as_text = _serialize_banner($old_banners->{$banner->{bid}}, skip_contact_info => $banner->{contact_info} eq '', adgroup_type => $group->{adgroup_type}, use_callouts_for_calc_units => $use_callouts_for_calc_units);
                my $new_banners_as_text = _serialize_banner($banner, skip_contact_info => $banner->{contact_info} eq '', adgroup_type => $group->{adgroup_type}, use_callouts_for_calc_units => $use_callouts_for_calc_units);

                # если баннер поменялся
                # или если контактная информация измениласть и есть хотя бы один баннер в котором добавили КИ - это одно изменение баннера
                # или если удалили КИ
                if ($old_banners_as_text ne $new_banners_as_text
                    || $old_banners->{$banner->{bid}}->{vcard_id} && $banner->{contact_info} eq '-'
                    || $is_contact_info_changed && $banner->{contact_info} eq '+')
                {
                    $changes{edit_banner}++;
                    $banner->{_changed} = 1;
                    $banner->{save_as_draft} = 1 if !$options->{send_new_banners_to_moderation} && ($old_banners->{$banner->{bid}}->{statusModerate} || 'New') eq 'New';
                    $group->{_has_changes_in_banners} = 1;
                } elsif ($group->{_changed} && !$xls_camp->{is_group_format}) {
                    $changes{edit_banner}++;
                    $group->{_has_changes_in_banners} = 1;
                }

                delete $old_banners->{$banner->{bid}}; 
            }
        }

        my $old_phrases = $old_group->{phrases} || {};
        for my $phrase (@{$group->{phrases}}) {
            unless ($phrase->{id}) {
                $changes{create_phrase}++;
                $group->{_has_changes_in_phrases} = 1;
                $phrase->{_added} = 1;
            } else {

                my $old_phrase_as_text = _serialize_phrase($old_phrases->{$phrase->{id}}, $options);
                my $new_phrase_as_text = _serialize_phrase($phrase, $options);
                if ($old_phrase_as_text ne $new_phrase_as_text) {
                    $changes{edit_phrase}++;
                    $group->{_has_changes_in_phrases} = 1;
                    $phrase->{_changed} = 1;
                }
                delete $old_phrases->{$phrase->{id}};
            }
        }

        for my $relevance_match (@{$group->{relevance_match} // []}) {
            if ($relevance_match->{bid_id} && (
                $relevance_match->{price} != $old_group->{relevance_match}->{$relevance_match->{bid_id}}->{price}
                || ($has_extended_relevance_match &&
                    $relevance_match->{price_context} != $old_group->{relevance_match}->{$relevance_match->{bid_id}}->{price_context})
                || $relevance_match->{phrase_status} ne $old_group->{relevance_match}->{$relevance_match->{bid_id}}->{phrase_status}
                )
            ){
                $changes{edit_phrase}++;
                $group->{_has_changes_in_phrases} = 1;
            }
        }

        for my $retargeting (@{$group->{retargetings}}){
            # условие не в новом баннере и отличается цена
            if($retargeting->{bid} && $retargeting->{price_context} != $old_group->{retargetings}->{$retargeting->{ret_cond_id}}->{price_context}){
                $changes{edit_phrase}++;
                $group->{_has_changes_in_retargetings} = 1;
            }
        }

        for my $target_interests (@{$group->{target_interests} // []}) {
            # интерес не новый и отличается цена
            if($target_interests->{ret_id} && $target_interests->{price_context} != $old_group->{target_interests}->{$target_interests->{target_category_id}}->{price_context}){
                $changes{edit_phrase}++;
                $group->{_has_changes_in_retargetings} = 1;
            }
        }

        if ($group->{relevance_match_to_add}) {
            $changes{create_phrase} += $group->{relevance_match_to_add};
            $group->{_has_changes_in_phrases} = 1;
        }

        if ($group->{retargetings_to_add}){
            $changes{create_phrase} += $group->{retargetings_to_add};
            $group->{_has_changes_in_retargetings} = 1;
        }
        
        if (keys %$old_phrases) {
            $changes{remove_phrase} += keys %$old_phrases;
            $group->{_has_changes_in_phrases} = 1;
        }

        if ($group->{relevance_match_for_delete}) {
            $changes{remove_phrase} += $group->{relevance_match_for_delete};
            $group->{_has_changes_in_phrases} = 1;
        }

        if ($group->{retargetings_for_delete}){
            $changes{remove_phrase} += $group->{retargetings_for_delete};
            $group->{_has_changes_in_retargetings} = 1;
        }

        if (keys %$old_banners) {
            $changes{remove_banner} += keys %$old_banners;
            $group->{_has_changes_in_banners} = 1;
        }

    }
    
    if (my @rest_groups = keys %curr_groups) {
        $changes{remove_group} = scalar @rest_groups;
        my ($rem_banners, $rem_phrases) = (0, 0);
        foreach (values %curr_groups) {
            $rem_banners += scalar keys %{$_->{banners}};
            $rem_phrases += scalar keys %{$_->{phrases}};
        }
        $changes{remove_banner} += $rem_banners;
        $changes{remove_phrase} += $rem_phrases;
    }
    
    return ($new_camp, \@errors, \@warnings, \%changes, \@stop_banners);
}

=head2 fill_empty_price_for_snapshot

    заполняет пустые цены для фраз, минимальной ценой в соответствующей валюте
    используется при создании новой кампании

    $options: 
        необязательные опции
            strategy => $camp_snapshot->{strategy}
            search_strategy => $camp_snapshot->{search_strategy}
            context_strategy => $camp_snapshot->{context_strategy}
        обязательные опции
            currency => валюта

=cut
sub fill_empty_price_for_snapshot {
    my ($xls_camp, %O) = @_;

    my $currency = $O{currency};
    die 'no currency given' unless $currency;
    my $default_price = get_currency_constant($currency, 'DEFAULT_PRICE');
    $O{default_price} = $default_price;

    for my $group (@{ $xls_camp->{groups} }) {
        for my $phrase (@{ $group->{phrases} }) {
            fill_empty_price_for_phrase($phrase, %O);
        }
    }

    return $xls_camp;
}


=head2 fill_empty_price_for_phrase

    заполняет пустые цены для фразы, текущей сохраненной ценой или минимальной ценой в соответствующей валюте

    $options: 
        необязательные опции
            old_phrase => $camp_snapshot->banners[]->phrases[]
            strategy => $camp_snapshot->{strategy}
            search_strategy => $camp_snapshot->{search_strategy}
            context_strategy => $camp_snapshot->{context_strategy}
        обязательные опции
            default_price => дефолтная ставка согласно валюте

=cut
sub fill_empty_price_for_phrase {
    my ($phrase, %O) = @_;

    my $default_price = $O{default_price};
    die 'no default_price given' unless $default_price;

    if ($O{strategy} && $O{strategy} eq 'different_places') {
        if ($O{search_strategy} eq 'stop') {
            $phrase->{price_context} = ($O{old_phrase} ? $O{old_phrase}->{price_context} : $default_price) if !defined $phrase->{price_context} || $phrase->{price_context} eq '';
        } else {
            $phrase->{price} = ($O{old_phrase} ? $O{old_phrase}->{price} : $default_price) if !defined $phrase->{price} || $phrase->{price} eq '';
            $phrase->{price_context} = $phrase->{price} if !defined $phrase->{price_context} || $phrase->{price_context} eq '';
        }
    } else {
        $phrase->{price} = ($O{old_phrase} ? $O{old_phrase}->{price} : $default_price) if !defined $phrase->{price} || $phrase->{price} eq '';
        $phrase->{price_context} = $O{old_phrase}->{price_context} if $O{old_phrase} && (!defined $phrase->{price_context} || $phrase->{price_context} eq '');
    }

    return $phrase;
}

sub check_common_geo_for_snapshot($) {
    my ($xls_camp) = @_;

    return scalar(uniq map {$_->{geo}}  @{$xls_camp->{banners}}) <= 1 ? 1 : 0;
}

sub check_common_contact_info_for_snapshot($$$) {
    my ($uid, $cid, $xls_camp) = @_; 

    my $cinfo = $xls_camp->{contact_info};
    my $vcard_for_snapshot = hash_merge
        {   
            cid => $cid,
            phone=> length($cinfo->{phone}) ? join("#", map {$cinfo->{$_}||""} qw/country_code city_code phone ext/) : undef 
        },
        hash_cut $cinfo, qw/country city name contactperson worktime street house build apart im_client im_login extra_message contact_email org_details_id ogrn/;

    my $camp_ci_key = vcard_hash($vcard_for_snapshot, ignore_fields => [qw/metro/]);
    my $empty_ci_key = vcard_hash({}, ignore_fields => [qw/metro/]);
    my @bids_to_check;
    my $ci_hash={};
    
    for my $group (@{$xls_camp->{groups}}) {
        for my $banner (@{$group->{banners}}) {
            if($banner->{contact_info} && $banner->{contact_info} eq '+') {
                $ci_hash->{$camp_ci_key}++;
            } elsif ($banner->{contact_info} && $banner->{contact_info} eq '-') {
                $ci_hash->{$empty_ci_key}++;
            } elsif( $banner->{bid} ) {
                push @bids_to_check, $banner->{bid};
            }
        }
    } 
    
    my $vcard_ids;
    if (@bids_to_check) {
        $vcard_ids = get_one_column_sql(PPC(bid => \@bids_to_check), [
            "select distinct(vcard_id)
              from banners",
            WHERE => {bid => SHARD_IDS}]) || [];
    }

    my $vcards = get_vcards({vcard_id => $vcard_ids});
    for my $vcard (@$vcards) {
        $ci_hash->{vcard_hash($vcard, ignore_fields => [qw/metro/])}++;
    }
    
    return (scalar(keys %$ci_hash) == 1) ? 1 : 0;
}

sub get_camp_snapshot($$;$) {
    my ($uid, $cid, $options) = @_;
    $options ||= {};

    my $campaign = get_camp_info($cid, $uid);
    my $camp = {
        cid => $cid,
        strategy => detect_strategy($campaign),
        search_strategy => detect_search_strategy($campaign),
        platform  => $campaign->{platform},
        context_strategy => detect_context_strategy($campaign),
        currency => $campaign->{currency},
        campaign_minus_words => $campaign->{campaign_minus_words},
        sum => $campaign->{sum},
        all_retargeting_conditions => Retargeting::get_retargeting_conditions(uid => $uid),
        type => $campaign->{type},
    };
    
    $camp->{groups} = get_group_snapshot({
        uid => $uid, cid => $cid,
        skip_arch => $options->{skip_arch},
        host => $options->{host}, 
        pass_empty_groups => $options->{pass_empty_groups},
        platform => $camp->{platform},
        campType => $campaign->{type},
        ClientID => $campaign->{ClientID},
    });

    $camp->{skipped_bids} = get_one_column_sql(PPC(cid => $cid), 'select bid from banners where statusArch="Yes" and cid=?', $cid) || [] if $options->{skip_arch};

    $camp->{contact_info} = get_common_contactinfo_for_camp(
        $cid,
        {
            dont_skip_arch => (defined $options->{skip_arch} ? !$options->{skip_arch} : 1),
            skip_empty => 1
        }
    )||{};
    # при импорте по наличию поля ogrn мы определяем выгружен ли файл до появления в XLS ОГРН или после
    $camp->{contact_info}{ogrn} ||= '';
    foreach ($camp->{contact_info}{phone} ? () : @{$camp->{groups}}) {
        foreach (@{$_->{banners}}) {
            $_->{contact_info} = '' if $_->{contact_info} eq '+';
        }
    }

    return $camp;
}

sub get_group_snapshot {

    my $options = shift;

    my $cid = $options->{cid};
    my ($groups) = Models::AdGroup::get_groups_gr({
        cid => $cid,
        uid => $options->{uid},
        adgroup_types => [qw/base mobile_content/],
        ($options->{skip_arch} ? (arch_banners => 0) : ()),
    }, {
        get_tags => 1,
        get_phrases_statuses => 1,
        pass_empty_groups => $options->{pass_empty_groups} // 0,
    });
    
    my %tag_names;
    if (any { $_->{tags} && ref($_->{tags}) eq 'HASH' && %{$_->{tags}} } @$groups) {
       my $all_campaign_tags = get_all_campaign_tags($cid);
       %tag_names = map {
          ($_->{tag_id} => $_->{value}) 
       } @$all_campaign_tags
    }

    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($options->{campType}, $options->{ClientID});
    
    foreach my $group (@$groups) {
        my $tags = [];
        if ($group->{tags} && ref($group->{tags}) eq 'HASH' && %{$group->{tags}}) {
            $tags = [map {{tag_id => $_, tag_name => $tag_names{$_}}} keys %{$group->{tags}}];
        }
        my $is_mobile_content = ($group->{adgroup_type} eq 'mobile_content');

        $group->{tags} = $tags; 
        foreach my $banner (@{$group->{banners}}) {
            $banner->{sitelinks} = $banner->{sitelinks} ? [map {hash_cut $_, qw/title description href turbolanding/} @{$banner->{sitelinks}}] : undef;  
            $banner->{contact_info} = defined $banner->{phone} && length $banner->{phone} ? '+' : '-';
            if ($banner->{image}) {
                $banner->{image_url} = BannerImages::get_image_url($banner, {host => $options->{host}});
            } elsif ($banner->{image_ad} && $banner->{image_ad}->{hash}) {
                my $image_data = {
                    image_hash => $banner->{image_ad}->{hash},
                    image_type => 'image_ad',
                    mds_group_id => $banner->{image_ad}->{mds_group_id},
                    namespace => 'direct-picture',
                    image_type => 'image_ad',
                };
                $banner->{image_url} = BannerImages::get_image_url($image_data, {host => $options->{host}});
            } elsif($banner->{creative}){
                if ($banner->{creative}->{creative_type} eq 'html5_creative' || $banner->{creative}->{creative_type} eq 'bannerstorage') {
                    $banner->{image_url} = $banner->{creative}->{live_preview_url};
                } else {
                    $banner->{image_url} = sprintf '%s/%d/preview',$Settings::CANVAS_URL, $banner->{creative}->{creative_id};
                }
            }
            $banner->{short_status} =
                $banner->{statusArch} eq 'Yes' ? iget('Заархивировано') :
                $banner->{statusModerate} eq 'New' ? iget('Черновик') :
                $banner->{statusShow} eq 'No' ? iget('Остановлено') :
                $banner->{statusActive} eq 'Yes' ? iget('Активно') : '' ;

            if ($is_mobile_content) {
                    $banner->{xls_href} = $banner->{href};
                    my %reflected_attrs = map {$_ => 1} @{$banner->{reflected_attrs}};
                    foreach (qw/rating price icon rating_votes/) {
                        $banner->{"xls_".$_} = ($reflected_attrs{$_}) ? "+" : "-";
                    }
            }

            if ($banner->{ad_type} eq 'image_ad') {
                $banner->{title} = _get_image_ad_title($banner);
                $banner->{body} = q//;
            }
        }

        if ($is_mobile_content) {
            hash_merge $group, {
                xls_store_content_href => $group->{store_content_href},
                xls_min_os_version  => sprintf ("%s %.1f", $group->{mobile_content}->{os_type},
                                                       $group->{min_os_version}),
                xls_device_type_targeting => TTTools::human_device_type_targeting($group->{device_type_targeting}),
                xls_network_targeting => TTTools::human_network_targeting($group->{network_targeting}),
            };
        }
        
        foreach my $phrase (@{$group->{phrases}}) {
            my $status = iget("Работает везде");

            if ($phrase->{declined}) {
                $status = iget("Отклонена модератором");
            } elsif ($phrase->{is_suspended}) {
                $status = iget("Приостановлена");
            } elsif ($phrase->{rank}) {
                $status = iget("Работает везде");
            } elsif ($phrase->{context_stop_flag}) {
                $status = iget("Отключена везде");
            } else {
                $status = iget("Отключена на поиске");
            }
            $phrase->{phrase_status} = $status;
        }

        foreach my $relevance_match (@{$group->{relevance_match} // []}) {
            if ($relevance_match->{is_suspended}) {
                $relevance_match->{phrase_status} = iget("Приостановлена");
            } else {
                $relevance_match->{phrase_status} = iget("Активная");
            }
        }

        my $retargetings = Retargeting::get_group_retargeting(pid => [map {$_->{pid}} @$groups]);
        $group->{retargetings} = exists $retargetings->{$group->{pid}} ? $retargetings->{$group->{pid}} : [];
        foreach my $ret (@{$group->{retargetings}}) {
            $ret->{phrase_status} = $ret->{is_suspended} ? iget("Приостановлена") : iget("Активная");
        }
    }

    return $groups;
}

=head2 _get_image_ad_title

=cut

sub _get_image_ad_title {
    my ($banner) = @_;
    
    my $img_container = $banner->{image_ad} // $banner->{creative};

    return unless $img_container;
    return sprintf '%sx%s', @$img_container{qw/width height/};
}

=head2 _get_banner_image_url

=cut

sub _get_banner_image_url {
    my ($banner, $opt) = @_;

    if ($banner->{ad_type} eq 'image_ad') {
        return $banner->{image_url} if $banner->{creative};
        return BannerImages::get_image_url( { %{ $banner->{image_ad} }, image_type => 'image_ad' }, $opt);
    }
    else {
        $banner->{image_hash} = $banner->{image};
        return BannerImages::get_image_url($banner, $opt);
    }
}

sub _is_xls_supported_banner {
    my $banner = shift;
    return any(sub { $banner->{real_banner_type} eq $_ }, qw/text mobile_content image_ad/);
}

sub get_text_sheet {
    my ($camp, %options) = @_;

    my $format = $options{format} || 'xls'; 
    my $translocal_options = $options{ClientID} ? {ClientID => $options{ClientID}} : {host => $options{host}};

    BannersCommon::modify_groups_geo_for_translocal_before_show($camp->{groups}, $translocal_options);

    my $is_search_stop = $camp->{platform} eq 'context';
    my $is_mobile_content = ($camp->{type} || '' ) eq 'mobile_content';
    my $is_text_campaign  = ($camp->{type} || '' ) eq 'text';
    my $is_turbolandings_allowed = !$options{ClientID} || Client::ClientFeatures::get_is_featureTurboLandingEnabled(client_id => $options{ClientID});
    my $is_tycoon_organizations_feature_enabled = !$options{ClientID} || Client::ClientFeatures::has_tycoon_organizations_feature($options{ClientID});
    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($camp->{type}, $options{ClientID});

    # vspan - vertical span - merge N columns down
    # hspan - horizontal span - merge N columns right
    my $header = [
        { field => 'is_banner', width => 15, vspan => 1, global_format => { border => 2 } },
        { field => 'ad_type', width => 15, vspan => 1 },
        { field => 'banner_type', width => 15, vspan => 1, },
        { field => 'pid', width => 15, vspan => 1, },
        { field => 'group_name', width => 15, vspan => 1, },
        { field => 'number', width => 15, vspan => 1, },
        { field => 'serving_status', width => 15, vspan => 1, },
        { field => 'id', width => 15, vspan => 1, },
        { field => 'phr', width => 15, vspan => 1, },
        { field => 'bid', width => 15, vspan => 1, },
        { field => 'title', vspan => 1, },
        ($is_mobile_content ? () : ( { field => 'title_extension', vspan => 1, } ) ),
        { field => 'body', vspan => 1, },
        ($is_mobile_content ?
            ({ field => 'l1', hspan => 1, subrow => [
                    { field => 'l_title' },
                    { field => 'l_body' }
            ] })
            : ({ field => 'l1', hspan => 2, subrow => [
                    { field => 'l_title' },
                    { field => 'l_title_extension' },
                    { field => 'l_body' }
            ] })),
        { field => 'href', vspan => 1, },
        { field => 'display_href', vspan => 1, },
        { field => 'region', width => 15, vspan => 1, },
        ($is_tycoon_organizations_feature_enabled ?
            ({ field => 'permalink', width => 15, vspan => 1, })
            : ()),
        { field => 'price', width => 15, vspan => 1, },
        { field => 'price_context', vspan => 1, },
        { field => 'contact_info', vspan => 1, },
        { field => 'banner_status', vspan => 1, },
        { field => 'phrase_status', vspan => 1, },
        { field => 'sitelink_titles', vspan => 1, },
        { field => 'sitelink_descriptions', vspan => 1, },
        { field => 'sitelink_hrefs', vspan => 1, },
        ($is_text_campaign && $is_turbolandings_allowed ?
            ({ field => 'sitelink_turbolanding_ids', vspan => 1, })
            : ()
        ),
        { field => 'param1', vspan => 1, },
        { field => 'param2', vspan => 1, },
        { field => 'tags', width => 20, vspan => 1, },
        { field => 'image_url', vspan => 1, },
        ($is_mobile_content ? () : (
                { field => 'creative_id', vspan => 1 },
                { field => 'creative_status_moderate', vspan => 1 },
        )),
        ($is_text_campaign && $is_turbolandings_allowed ? 
            ({ field => 'turbolanding_id', vspan => 1, })
            : ()
        ),
        ($options{show_callouts} ? ({ field => 'callouts', vspan => 1 }) : ()),
        { field => 'minus_words', vspan => 1, },
        { field => 'age', vspan => 1, },
        { field => 'mobile_columns_adgroup_header', hspan => 3, subrow => [
                { field => 'mobile_store_content_href' },
                { field => 'mobile_device_type_targeting' },
                { field => 'mobile_network_targeting' },
                { field => 'mobile_os' },
            
        ] },
        { field => 'mobile_columns_banner_header', hspan => 4, subrow => [
                { field => 'mobile_href' },
                { field => 'mobile_icon' },
                { field => 'mobile_rating' },
                { field => 'mobile_rating_votes' },
                { field => 'mobile_price' },
        ]},
    ];

    my (@set_column, @merge_cells, @data_header, @data_header2);
    my $row_num = 9;
    my $col_num = 0;
    for my $field (@$header) {
        my $data_item = undef;
        if (defined $field) {
            my $row = $row_num;
            if ($field->{vspan}) {
                push @merge_cells, { row1 => $row, row2 => $row + $field->{vspan}, col1 => $col_num, col2 => $col_num  };
            }
            if ($field->{hspan}) {
                push @merge_cells, { row1 => $row, row2 => $row, col1 => $col_num, col2 => $col_num + $field->{hspan} };
            }
            if ($field->{width}) {
                push @set_column, { col1 => $col_num, col2 => $col_num, width => $field->{width} };
            }
            $data_item = hash_copy {}, $field, qw/global_format/;
            if ($field->{field} eq 'title' && $is_mobile_content) {
                $field->{field} .= '_mobile_content';
            }
            $data_item->{data} = XLSVocabulary::get_field_name($field->{field});
        }
        push @data_header, $data_item;
        if ($field->{subrow}) {
            if (scalar @{$field->{subrow}} != $field->{hspan}+1) {
                die "invalid hspan or subrows for field $field->{field}";
            }
            for my $field2 (@{$field->{subrow}}) {
                # для второго уровня пока не поддерживаем v/span
                if ($field2->{field} eq 'l_title' && $is_mobile_content) {
                    $field2->{field} .= '_mobile_content';
                }
                push @data_header2, { data => XLSVocabulary::get_field_name($field2->{field}) };
            }
            for (1 .. $field->{hspan}) {
                push @data_header, undef;
                $col_num++;
            }
        }
        else {
            push @data_header2, undef;
        }
        $col_num++;
    }

    my $xls_format = {
        sheetname=> iget("Тексты"),
        set_row => [
            {row => 5, height => 20},
            {row => 9, height => 20}, # table
            {row => 10, height => 20}, #   head
        ],
        set_column => \@set_column,
        merge_cells => \@merge_cells,
        freeze_panes => [11, 0],
    };
    
    my $data=[];
    # Multicurrency: В случае мультивалютности в XLS должно быть поле валюты. В остальном все должно выглядеть также как и раньше.
    push @$data, (
            ( map {[undef]} 1..5),  
            [{data=>iget('Предложение текстовых блоков для рекламной кампании'), format=>{bold=>1, size => 14}}],
            [undef, undef, undef, XLSVocabulary::get_field_name('campaign_type'), XLSVocabulary::get_field_name( $is_mobile_content ? 'camp_type_mobile_content' : 'camp_type_text')],
            [undef, undef, undef, XLSVocabulary::get_field_name('cid'), $camp->{cid}, undef,
             (($camp->{currency} || 'YND_FIXED') ne 'YND_FIXED') ? (XLSVocabulary::get_field_name('currency'), $camp->{currency}) : (undef, undef),
            ],
            [undef, undef, undef, XLSVocabulary::get_field_name('campaign_minus_words'), {data => str(MinusWordsTools::minus_words_array2interface_show_format($camp->{campaign_minus_words})), as_text => 1, format => {num_format => '@'}}],
        );
        
    push @$data, \@data_header;
    push @$data, \@data_header2;
    
    my $adgroup_number = 0;
    my $row_number = @$data + 1;
    my ($title_col, $title_extension_col, $text_col);
    if ($is_mobile_content) {
        ($title_col, $text_col) = $is_search_stop ? (qw/J K/) : (qw/K L/);
    } else {
        ($title_col, $title_extension_col, $text_col) = $is_search_stop ? (qw/J K L/) : (qw/K L M/);
    }
    
    for my $adgroup (@{$camp->{groups}}) {
        # TODO adgroup: Облегчаем жизнь пользователю на время переходного периода, чтобы руками не проставлять имена групп.
        local $adgroup->{group_name} = $adgroup->{group_name} || Models::AdGroup::get_auto_adgroup_name($adgroup, uid => $options{uid});

        $adgroup_number++;
        
        my $banner = first { _is_xls_supported_banner($_) } @{$adgroup->{banners}};
        next if !$banner;

        my $format_callouts = sub {
            my $callouts = shift;
            return '' if !$callouts;
            return {
                data => join('||', map {str($_->{callout_text})} @$callouts),
                as_text => 1,
                format => {num_format => '@'},
            };
        };

        my $status_moderate_translation = {
            'New'     => iget('Черновик'),
            'Ready'   => iget('Ожидает модерации'),
            'Sending' => iget('Ожидает модерации'),
            'Sent'    => iget('Ожидает модерации'),
            'Yes'     => iget('Принято'),
            'No'      => iget('Отклонено'),
            'Error'   => iget('Ожидает модерации'),
        };

        my $creative_status_moderate = $status_moderate_translation->{$banner->{video_resources}->{status_moderate}//''} // '';

        my $universal_fields = 
          { is_banner => '-',
            ad_type => XLSVocabulary::get_field_name("ad_type_$banner->{ad_type}"),
            banner_type => $banner->{banner_type} eq 'mobile' ? '+' : '-',
            pid => $adgroup->{pid},
            group_name => $adgroup->{group_name} || '',
            adgroup_number => $adgroup_number,
            serving_status => $adgroup->{is_bs_rarely_loaded} == 1 ? iget('мало показов') : '-',
            bid => $banner->{bid},
            title => {data => str(html2string($banner->{title})), as_text => 1, format => {num_format => '@'}},
            title_extension => !$is_mobile_content ? {data => str(html2string($banner->{title_extension})), as_text => 1, format => {num_format => '@'}} : '',
            body => {data => str(html2string($banner->{body})), as_text => 1, format => {num_format => '@'}},
            title_col           => _get_length_formula($title_col, $row_number),
            title_extension_col => _get_length_formula($title_extension_col, $row_number),
            text_col            => _get_length_formula($text_col, $row_number),
            href => !$is_mobile_content ? {data => str($banner->{href}), as_text => 1, format => {num_format => '@'}} : '',
            display_href => !$is_mobile_content ? {data => str($banner->{display_href}), as_text => 1, format => {num_format => '@'}} : '',
            permalink => $banner->{permalink} // '',
            geo => get_geo_names_minus($adgroup->{geo}, lang => Yandex::I18n::current_lang()),
            contact_info => !$is_mobile_content ? $banner->{contact_info} : '',
            short_status => $banner->{short_status},
            sitelinks => ($banner->{sitelinks} && !$is_mobile_content
                    ? [ {data => join('||', map {str($_->{title})} @{$banner->{sitelinks}}), as_text => 1, format => {num_format => '@'}},
                        {data => (any {defined $_->{description}} @{$banner->{sitelinks}})
                                 ? join('||', map {str($_->{description})} @{$banner->{sitelinks}})
                                 : "",
                         as_text => 1,
                         format => {num_format => '@'}
                        },
                        {data => join('||', map {str($_->{href})} @{$banner->{sitelinks}}), as_text => 1, format => {num_format => '@'}},
                        ($is_text_campaign && $is_turbolandings_allowed ? (
                            {data => (any {defined $_->{turbolanding}} @{$banner->{sitelinks}})
                                     ? join('||', map {$_->{turbolanding} ? str($_->{turbolanding}->{id}) : ''} @{$banner->{sitelinks}})
                                     : '',
                             as_text => 1,
                             format => {num_format => '@'}
                            } ) : ()
                         ),
                      ]
                    : [undef, undef, undef, ($is_text_campaign && $is_turbolandings_allowed ? undef : ())]),
            tags => {data => join(',', sort map {$_->{tag_name}} @{$adgroup->{tags}}), as_text => 1, format => {num_format => '@'}},
            images => [ _get_banner_image_url($banner, {host => $options{host}}) ],
            ($is_text_campaign && $is_turbolandings_allowed ?
                (turbolanding_id => {
                        data => (exists $banner->{turbolanding} ? $banner->{turbolanding}->{id} : ''),
                        as_text => 1,
                        format => { num_format => '@' }
                        }
                )
                : ()
            ),
            callouts => $format_callouts->($banner->{callouts}),
            minus_words => {data => str(MinusWordsTools::minus_words_array2interface_show_format($adgroup->{minus_words})), as_text => 1, format => {num_format => '@'}},
            age => BannerFlags::get_banner_flags_as_hash($banner->{flags})->{age},
            ($is_mobile_content ? () : (
                creative_status_moderate => $creative_status_moderate,
                creative_id => { data => $banner->{video_resources}->{id}, as_text => 1, format => { num_format => '@' }},
            )),
        };

        for my $retargeting (@{$adgroup->{retargetings}}) {
            my $retargeting_condition_name = $camp->{all_retargeting_conditions}->{$retargeting->{ret_cond_id}}->{condition_name};
            my @row = (
                (map { $universal_fields->{$_} } qw/is_banner ad_type banner_type pid group_name adgroup_number serving_status/),
                $retargeting->{ret_cond_id},
                {data => "$Retargeting::CONDITION_XLS_PREFIX $retargeting_condition_name", as_text => 1, format => {num_format => '@'}},
                (map { $universal_fields->{$_} } qw/bid title/),
                ($is_mobile_content ? () : (
                    $universal_fields->{title_extension},
                )),
                $universal_fields->{body},
                _get_length_formula($title_col, $row_number),
                ($is_mobile_content ? () : (
                    _get_length_formula($title_extension_col, $row_number),
                )),
                _get_length_formula($text_col, $row_number),
                (map { $universal_fields->{$_} } qw/href display_href geo/),
                ($is_tycoon_organizations_feature_enabled ? $universal_fields->{permalink} : ()),
                undef,
                ($retargeting->{price_context} > 0 ? $retargeting->{price_context} : ''),
                (map { $universal_fields->{$_} } qw/contact_info short_status/),
                $retargeting->{phrase_status},
                @{$universal_fields->{sitelinks}},
                undef,
                undef,
                $universal_fields->{tags},
                @{$universal_fields->{images}},
                ($is_mobile_content ? () : (
                    $universal_fields->{creative_id},
                    $universal_fields->{creative_status_moderate},
                )),
                ($is_text_campaign && $is_turbolandings_allowed ? $universal_fields->{turbolanding_id} : ()),
                ($options{show_callouts} ? $universal_fields->{callouts} : ()),
                $universal_fields->{minus_words},
                $universal_fields->{age},
            );

            push @row, _get_mobile_columns_to_row($adgroup, $banner);

            push @$data, \@row;

            ++$row_number;
        }

        for my $target_interest (@{$adgroup->{target_interests}}) {
            my @row = (
                (map { $universal_fields->{$_} } qw/is_banner ad_type banner_type pid group_name adgroup_number serving_status/),
                $target_interest->{ret_id},
                {data => "$Retargeting::CONDITION_XLS_PREFIX_TARGET_INTEREST ${ \iget($target_interest->{category_name}) }($target_interest->{target_category_id})", as_text => 1, format => {num_format => '@'}},
                (map { $universal_fields->{$_} } qw/bid title/),
                ($is_mobile_content ? () : (
                    $universal_fields->{title_extension},
                )),
                $universal_fields->{body},
                _get_length_formula($title_col, $row_number),
                ($is_mobile_content ? () : (
                    _get_length_formula($title_extension_col, $row_number),
                )),
                _get_length_formula($text_col, $row_number),
                (map { $universal_fields->{$_} } qw/href display_href geo/),
                ($is_tycoon_organizations_feature_enabled ? $universal_fields->{permalink} : ()),
                undef,
                ($target_interest->{price_context} > 0 ? $target_interest->{price_context} : ''),
                (map { $universal_fields->{$_} } qw/contact_info short_status/),
                $target_interest->{is_suspended},
                @{$universal_fields->{sitelinks}},
                undef,
                undef,
                $universal_fields->{tags},
                @{$universal_fields->{images}},
                ($is_mobile_content ? () : (
                    $universal_fields->{creative_id},
                    $universal_fields->{creative_status_moderate},
                )),
                ($is_text_campaign && $is_turbolandings_allowed ? $universal_fields->{turbolanding_id} : ()),
                ($options{show_callouts} ? $universal_fields->{callouts} : ()),
                $universal_fields->{minus_words},
                $universal_fields->{age},
            );

            push @row, _get_mobile_columns_to_row($adgroup, $banner);

            push @$data, \@row;

            ++$row_number;
        }

        for my $phrase (@{$adgroup->{phrases}} ? @{$adgroup->{phrases}} : ()) {
            
            my @row = (
                (map { $universal_fields->{$_} } qw/is_banner ad_type banner_type pid group_name adgroup_number serving_status/),
                $phrase->{id},
                {data => str($phrase->{phrase}), as_text => 1, format => {num_format => '@'}},
                (map { $universal_fields->{$_} } qw/bid title/),
                ($is_mobile_content ? () : (
                    $universal_fields->{title_extension},
                )),
                $universal_fields->{body},
                _get_length_formula($title_col, $row_number),
                ($is_mobile_content ? () : (
                    _get_length_formula($title_extension_col, $row_number),
                )),
                _get_length_formula($text_col, $row_number),
                (map { $universal_fields->{$_} } qw/href display_href geo/),
                ($is_tycoon_organizations_feature_enabled ? $universal_fields->{permalink} : ()),
                ($phrase->{price} > 0 ? {data => $phrase->{price}, format => {num_format => '0.00'}} : ''),
                ($phrase->{price_context} > 0 ? {data => $phrase->{price_context}, format => {num_format => '0.00'}} : ''),
                (map { $universal_fields->{$_} } qw/contact_info short_status/),
                $phrase->{phrase_status},
                @{$universal_fields->{sitelinks}},
                {data => str($phrase->{param1}), as_text => 1, format => {num_format => '@'}},
                {data => str($phrase->{param2}), as_text => 1, format => {num_format => '@'}},
                $universal_fields->{tags},
                @{$universal_fields->{images}},
                ($is_mobile_content ? () : (
                    $universal_fields->{creative_id},
                    $universal_fields->{creative_status_moderate},
                )),
                ($is_text_campaign && $is_turbolandings_allowed ? $universal_fields->{turbolanding_id} : ()),
                ($options{show_callouts} ? $universal_fields->{callouts} : ()),
                $universal_fields->{minus_words},
                $universal_fields->{age},
            );

            push @row, _get_mobile_columns_to_row($adgroup, $banner);

            push @$data, \@row;

            ++$row_number;
        }

        for my $relevance_match (@{$adgroup->{relevance_match}}) {

            my @row = (
                (map { $universal_fields->{$_} } qw/is_banner ad_type banner_type pid group_name adgroup_number serving_status/),
                $relevance_match->{bid_id},
                {data => "$CONDITION_XLS_PREFIX_RELEVANCE_MATCH", as_text => 1, format => {num_format => '@'}},
                (map { $universal_fields->{$_} } qw/bid title/),
                ($is_mobile_content ? () : (
                    $universal_fields->{title_extension},
                )),
                $universal_fields->{body},
                _get_length_formula($title_col, $row_number),
                ($is_mobile_content ? () : (
                    _get_length_formula($title_extension_col, $row_number),
                )),
                _get_length_formula($text_col, $row_number),
                (map { $universal_fields->{$_} } qw/href display_href geo/),
                ($is_tycoon_organizations_feature_enabled ? $universal_fields->{permalink} : ()),
                ($relevance_match->{price} > 0 ? {data => $relevance_match->{price}, format => {num_format => '0.00'}} : ''),
                ($relevance_match->{price_context} > 0 && $has_extended_relevance_match ? {
                        data => $relevance_match->{price_context}, format => {num_format => '0.00'}
                    } : ''),

                (map { $universal_fields->{$_} } qw/contact_info short_status/),
                $relevance_match->{phrase_status},
                @{$universal_fields->{sitelinks}},
                {data => str($relevance_match->{href_param1}), as_text => 1, format => {num_format => '@'}},
                {data => str($relevance_match->{href_param2}), as_text => 1, format => {num_format => '@'}},
                $universal_fields->{tags},
                @{$universal_fields->{images}},
                ($is_mobile_content ? () : (
                    $universal_fields->{creative_id},
                    $universal_fields->{creative_status_moderate},
                )),
                ($is_text_campaign && $is_turbolandings_allowed ? $universal_fields->{turbolanding_id} : ()),
                ($options{show_callouts} ? $universal_fields->{callouts} : ()),
                $universal_fields->{minus_words},
                $universal_fields->{age},
            );

            push @row, _get_mobile_columns_to_row($adgroup, $banner);

            push @$data, \@row;

            ++$row_number;
        }
        
        for my $banner (@{$adgroup->{banners}}[1..@{$adgroup->{banners}}-1]) {

            next unless _is_xls_supported_banner($banner);

            my @row = (
                '+',
                XLSVocabulary::get_field_name("ad_type_$banner->{ad_type}"),
                ($banner->{banner_type} eq 'mobile' ? '+' : '-'),
                $adgroup->{pid}, $adgroup->{group_name} || '', $adgroup_number, $adgroup->{is_bs_rarely_loaded} == 1 ? iget('мало показов') : '-',
                '', '',
                $banner->{bid}, 
                {data => str(html2string($banner->{title})), as_text => 1, format => {num_format => '@'}},
                ($is_mobile_content ? () : (
                    {data => str(html2string($banner->{title_extension})), as_text => 1, format => {num_format => '@'}},
                )),
                {data => str(html2string($banner->{body})), as_text => 1, format => {num_format => '@'}},
                _get_length_formula($title_col, $row_number),
                ($is_mobile_content ? () : (
                    _get_length_formula($title_extension_col, $row_number),
                )),
                _get_length_formula($text_col, $row_number),
                (!$is_mobile_content) ? {data => str($banner->{href}), as_text => 1, format => {num_format => '@'}} : '',
                (!$is_mobile_content) ? {data => str($banner->{display_href}), as_text => 1, format => {num_format => '@'}} : '',
                '',
                ($is_tycoon_organizations_feature_enabled ? $banner->{permalink} // '' : ()),
                '', '',
                !$is_mobile_content ? $banner->{contact_info} : '',
                $banner->{short_status},
                '',
                
                ($banner->{sitelinks} && !$is_mobile_content
                    ? (
                        {data => join('||', map {str($_->{title})} @{$banner->{sitelinks}}), as_text => 1, format => {num_format => '@'}},
                        {data => (any {defined $_->{description}} @{$banner->{sitelinks}})
                                 ? join('||', map {str($_->{description})} @{$banner->{sitelinks}})
                                 : "",
                         as_text => 1,
                         format => {num_format => '@'}
                        },
                        {data => join('||', map {str($_->{href})} @{$banner->{sitelinks}}), as_text => 1, format => {num_format => '@'}},
                        
                        ( $is_text_campaign && $is_turbolandings_allowed ? {
                                data => (any {defined $_->{turbolanding}} @{$banner->{sitelinks}})
                                    ? join('||', map { $_->{turbolanding} ? str($_->{turbolanding}->{id}) : undef} @{$banner->{sitelinks}})
                                    : "",
                                as_text => 1,
                                format => {num_format => '@'}
                            } : ()
                        ),
                    )
                    : (undef, undef, undef, ($is_text_campaign && $is_turbolandings_allowed ? undef : ()))
                ),
                    
                '', '',
                
                undef,
                _get_banner_image_url($banner, {host => $options{host}}),
                ($is_mobile_content ? () : (
                    { data => $banner->{video_resources}->{id}, as_text => 1, format => { num_format => '@' }},
                    $status_moderate_translation->{$banner->{video_resources}->{status_moderate}//''},
                )),
                ($is_text_campaign && $is_turbolandings_allowed ?
                    {
                        data => (exists $banner->{turbolanding} ? $banner->{turbolanding}->{id} : ''),
                        as_text => 1,
                        format => {num_format => '@'}
                    } : ()
                ),   
                ($options{show_callouts} ? $format_callouts->($banner->{callouts}) : ()),

                undef,
                BannerFlags::get_banner_flags_as_hash($banner->{flags})->{age},

            );

            push @row, _get_mobile_columns_to_row($adgroup, $banner, skip_adgroup => 1);
            
            push @$data, \@row;

            ++$row_number;
        }
    }

    return {data => $data, format => $xls_format};
}

sub _get_length_formula {
    my ($value, $row_number) = @_;
    return qq[=IF($value${row_number}="","",LEN(SUBSTITUTE(SUBSTITUTE(SUBSTITUTE(SUBSTITUTE(SUBSTITUTE(SUBSTITUTE($value${row_number},"!",""),",",""),".",""),";",""),":",""),"""","")))];
}

sub _get_mobile_columns_to_row {
    my ($adgroup, $banner, %options) = @_;
    my @row;
    if ($adgroup->{adgroup_type} ne 'mobile_content') {
        push @row, ({data => ""}) x 9;
    } else {
        if ($options{skip_adgroup}) {
            push @row, ({data => ""}) x 4;
        } else {
            foreach (qw/store_content_href device_type_targeting network_targeting min_os_version/) {
                push @row, {data => $adgroup->{"xls_".$_}};
            }
        }
        foreach (qw/href icon rating rating_votes price/) {
            push @row, {data => $banner->{"xls_".$_}};
        }
    }
    return @row;    
}

sub get_contact_sheet {
    my ($camp) = @_;

    my $contact_format = {
        sheetname=> iget("Контактная информация"),
        set_row => [
        ],

        set_column => [
            {col1 => 0, count => 0, width => 3},
            {col1 => 1, count => 0, width =>12},
            {col1 => 2, count => 0, width => 8},
            {col1 => 3, count => 0, width => 8},
            {col1 => 4, count => 0, width => 8},
            {col1 => 5, count => 0, width => 8},
            {col1 => 6, count => 0, width => 3},
            {col1 => 7, count => 0, width => 8},
            {col1 => 8, count => 0, width =>10},
            {col1 => 9, count => 0, width =>10},
            {col1 =>10, count => 0, width =>30},
        ],

        merge_cells => [
            {row1 => 8, row2 => 8, col1 => 2, col2 => 5},
            {row1 => 10, row2 => 10, col1 => 2, col2 => 5},
            {row1 => 12, row2 => 12, col1 => 8, col2 => 10},
            {row1 => 16, row2 => 16, col1 => 1, col2 => 5}, {row1 => 16, row2 => 16, col1 => 8, col2 => 10},
            {row1 => 19, row2 => 19, col1 => 1, col2 => 5},
            {row1 => 22, row2 => 22, col1 => 1, col2 => 2}, {row1 => 22, row2 => 26, col1 => 8, col2 => 10},
        ],
    };

    my @week_days = (iget('пн'),
                     iget('вт'),
                     iget('ср'),
                     iget('чт'),
                     iget('пт'),
                     iget('сб'),
                     iget('вс'));

    my $im_client_comma_list = join(', ', vcard_im_client_options);

    my $contact_data = [
        (map {[undef]} 1..5),
        [iget('Контактная информация')],
        [undef],
        [undef],
        [undef, iget('Страна*:'),map {{data=>$_, format=>{border=>2}}} ($camp->{contact_info}{country}, map {undef} 1..3)],
        [undef],
        [undef, iget('Город*:'), map {{data=>$_, format=>{border=>2}}} ($camp->{contact_info}{city}, map {undef} 1..3)],
        [(map {undef} ('a'..'h')),iget('ОГРН / ОРГНИП:')],
        [undef, iget('Телефон*:'),(map {{data=>$_, format=>{border=>2, num_format => '@'}, as_text => 1}} ( map { $camp->{contact_info}{$_} } qw/country_code city_code phone ext/)),undef,undef,({data=>$camp->{contact_info}{ogrn}, format=>{border=>2,num_format => '@'}, as_text=>1})], 
        [undef,undef,iget('код страны'),iget('код города'),iget('телефон'),iget('добавочный')],
        [undef],
        [undef,iget('Название компании / ФИО*:'), (map {undef} 1..6), iget('Контактный e-mail:')],
        [undef, (map {{data=>$_, format=>{border=>2}}} $camp->{contact_info}{name}, map {undef} 2..5), (map {undef} 1..2), (map {{data=>$_, format=>{border=>2}}} $camp->{contact_info}{contact_email}, map {undef} 2..3)],
        [undef],
        [undef, iget('Контактное лицо:'), (map {undef} 1..6), iget('Интернет-пейджер:')],
        [undef, (map {{data=>$_, format=>{border=>2}}} $camp->{contact_info}{contactperson}, map {undef} 2..5), (map {undef} 1..2), {data=>$camp->{contact_info}{im_client},format=>{border=>2}}, undef, {data=>$camp->{contact_info}{im_login},format=>{border=>2}}],
        [undef, undef, (map {undef} 1..6),"IM ($im_client_comma_list)", undef, iget('IM Логин')],
        [undef, iget('Почтовый адрес:'), (map {undef} 1..6), iget('Подробнее о товаре/услуге:')],
        [undef, (map {{data=>$_, format=>{border=>2, num_format=>'@'}, as_text => 1}} ($camp->{contact_info}{street}, undef, $camp->{contact_info}{house}, $camp->{contact_info}{build}, $camp->{contact_info}{apart})), (map {undef} 1..2), (map {{data=>$_,format=>{border=>2}}} $camp->{contact_info}{extra_message}, map {undef} 2..3)],
        [undef, iget('улица'), undef, iget('дом'),iget('корпус'),iget('офис'), undef, undef, (map {{format=>{border=>2}}} 1..3)],
        [undef, (map {undef} 1..7) ,(map {{format=>{border=>2}}} 1..3)],
        [undef,iget('Время работы*:'), (map {undef} 1..6), (map {{format=>{border=>2}}} 1..3)],
        
        (map {
            ## no critic (Freenode::DollarAB)
            my $a=$camp->{contact_info}{worktimes}[$_];
            [undef, {data=>$week_days[$a->{d1}], format=>{border=>2}}, {data=>$week_days[$a->{d2}], format=>{border=>2}}, undef, {data=>"$a->{h1}:$a->{m1}", format=>{border=>2}},{data=>"$a->{h2}:$a->{m2}", format=>{border=>2}}, undef, undef, $_ < 1 ? (map {{format=>{border=>2}}} 1..3) : ()]
        } 0..$#{$camp->{contact_info}{worktimes}} ),

        (map { [undef, (map {{format=>{border=>2}}} 1..2), undef, (map {{format=>{border=>2}}} 1..2), undef, undef, $_ < 1 ? (map {{format=>{border=>2}}} 1..3) : ()] } scalar(@{$camp->{contact_info}{worktimes}}) .. 2),
        [undef],
        [undef, iget('* - поля, обязательные для заполнения')],
    ];

    return {data => $contact_data, format => $contact_format};
}

sub get_regions_sheet {
        my %options = @_;

    my $translocal_options = $options{ClientID} ? {ClientID => $options{ClientID}} : {host => $options{host}};

    my @regions_list = make_regions_list($translocal_options);

    my $regions_format = {
        sheetname=> iget("Регионы"),
        set_row => [
        ],

        set_column => [
            {col1 => 0, count => 0, width => 8},
            {col1 => 1, count => 0, width => 3},
            {col1 => 2, count => 0, width => 3},
            {col1 => 3, count => 0, width => 3},
            {col1 => 4, count => 0, width => 3},
            {col1 => 5, count => 0, width => 3},
            {col1 => 6, count => 0, width =>40},
        ],

        merge_cells => [
            {row1 => 2, row2 => 2, col1 => 1, col2 => 6},

        ],
    };

    my $regions_data = [
        [undef],
        [undef],
        [undef, {data=>iget("Регионы"), format=>{border=>2}}, (map {{format=>{border=>2}}} 2..6) ],
    ];


    for my $i (0..$#regions_list) {
        my $region = $regions_list[$i];

        my ( $last_region, $next_region );
        $last_region = $regions_list[$i-1] if $i > 0;
        $next_region = $regions_list[$i+1] if $i < $#regions_list;

        next if ! defined $region->{level};

        if ($region->{level}==1) {
            push @{$regions_data}, [undef];
        }

        my $row = [undef, map {{
            format=>{
                ($region->{level} == 1 ? (top    => ( min($last_region->{level}||0, $region->{level})<3 ? 2 : 1))  :()),
                bottom => ( min($next_region->{level}||0, $region->{level})<3 ? 2 : 1),
                ($_==1 ? (left => 2) : ()),
                $_==6 || $_ == $region->{level} ? (right => 2) : (),
            }
            }} 1..6];
        $row->[$region->{level}]{data} = $region->{local_name};

        push @{$regions_data}, $row;

        push @{$regions_format->{merge_cells}}, {row1=>$#{$regions_data}, row2=>$#{$regions_data}, col1 => 1, col2 => $region->{level}-1} if $region->{level} > 1; 
        push @{$regions_format->{merge_cells}}, {row1=>$#{$regions_data}, row2=>$#{$regions_data}, col1 => $region->{level}, col2=>6}; 
    }
    return {data => $regions_data,  format => $regions_format};
}

sub get_vocabulary_sheet {
    my ($camp, %options) = @_;

    my $vocabulary_format = {
        sheetname=> iget("Словарь значений полей"),
        set_row => [
        ],

        set_column => [
            {col1 => 0, count => 0, width => 8},
            {col1 => 1, count => 0, width => 30},
        ],

        merge_cells => [
            {row1 => 1, row2 => 1, col1 => 3, col2 => 4},
        ],
    };

    my $camp_type_data = { header => 'campaign_type_header',
                           values => [qw/camp_type_text camp_type_mobile_content/]};
    my $ad_type_data = {
        header => 'ad_type',
        values => [qw/ad_type_text ad_type_image_ad/],
    };
    my $device_type_targeting_data = { header => 'mobile_device_type_targeting',
                                  values => [qw/mobile_device_type_targeting_all mobile_device_type_targeting_tablet mobile_device_type_targeting_phone/]};
    my $network_targeting_data = { header => 'mobile_network_targeting',
                                   values => [qw/mobile_network_targeting_all mobile_network_targeting_wifi/]};
    my $android_versions = {header => 'mobile_os_android',
                            values => $MobileContent::OS_VERSIONS{Android}};
    my $ios_versions = {header => 'mobile_os_ios',
                        values => $MobileContent::OS_VERSIONS{iOS}};

    my $left_part = [];
    foreach my $data ($camp_type_data, $ad_type_data, $device_type_targeting_data, $network_targeting_data) {
        push @$left_part, @{_make_vocabulary_list_of_values($data, get_name=>1)};
        push @$left_part, undef;
    }

    my $right_part_android = [{data=>XLSVocabulary::get_field_name('mobile_os')}];
    push @$right_part_android, @{_make_vocabulary_list_of_values($android_versions, get_name=>0)};
    my $right_part_ios = [undef];
    push @$right_part_ios, @{_make_vocabulary_list_of_values($ios_versions, get_name=>0)};

    my $length = max(scalar(@$left_part), scalar(@$right_part_android), scalar(@$right_part_ios));
    my $vocabulary_data = [ [(undef) x 5] ]; # Первая строка пустая для красоты
    for (my $i = 0; $i<$length; $i++) {
        push @$vocabulary_data, [undef,
                      $left_part->[$i],
                      undef,
                      $right_part_android->[$i],
                      $right_part_ios->[$i],
                     ];
    }
    return {data => $vocabulary_data, format => $vocabulary_format};
}

sub _make_vocabulary_list_of_values {
    my ($data, %options) = @_;
    my $cells = [{data=>XLSVocabulary::get_field_name($data->{header}), format=>{}},
                 map { {data   => $options{get_name} ? XLSVocabulary::get_field_name($_) : $_, 
                        as_text => 1,
                        format =>{left=>2, right=>2}} } @{$data->{values}},
                ];
    $cells->[1]->{format}->{top} = 2; # рисуем рамку сверху
    $cells->[-1]->{format}->{bottom} = 2; # рисуем рамку снизу
    return $cells;
}
sub camp_snapshot2excel($;%){
    my ($camp, %options) = @_;

    my $text = get_text_sheet($camp, %options);
    my $contact = get_contact_sheet($camp);
    my $regions = get_regions_sheet(%options);
    my $vocabulary = get_vocabulary_sheet($camp);

    my $format = $options{format} || 'xls';
    my $xls = $format eq 'xlsx' ? Yandex::ReportsXLSX->new(no_optimization => 1) : Yandex::ReportsXLS->new(compatibility_mode => 1);
    return $xls->array2excel2scalar(
        [$text->{data}, $contact->{data}, $regions->{data}, $vocabulary->{data}],
        [$text->{format}, $contact->{format}, $regions->{format}, $vocabulary->{format}],
    );  
}


=head2 make_regions_list

get the complete regions' list arranged in the right order 

    opt:
        ClientID -- для определения транслокального дерева регионов

=cut

sub make_regions_list($) {
    my $opt = shift;
    my @geo_list = _get_full_childs_list(0, $opt);

    my $lang = Yandex::I18n::current_lang();

    my %lang_field = (
        'ua' => 'ua_name',
        'en' => 'ename',
        'tr' => 'tr_name',
    );
    my $field = $lang_field{$lang} || 'name';

    my @result_list;

    # собираем данные из GEOREG + GEOREG_UNIQ_ALIAS
    for my $geo_id (@geo_list) {
        my $geo_data = { id => $geo_id };
        my $alias_data = exists $geo_regions::GEOREG_UNIQ_ALIAS{$geo_id} ? $geo_regions::GEOREG_UNIQ_ALIAS{$geo_id} : {};
        hash_merge $geo_data, GeoTools::get_translocal_region($geo_id, $opt), $alias_data;
        $geo_data->{local_name} = $geo_data->{$field};
        push @result_list, $geo_data;
    }

    return @result_list;
}

#
#   One recursion step for   sub make_regions_list() 
#
sub _get_full_childs_list{
    my $geo = shift;
    my $opt = shift;

    return map {$_, _get_full_childs_list($_, $opt)}
           @{GeoTools::get_translocal_region($geo, $opt)->{childs}};
}

sub xls2camp_snapshot {
    
    my ($format, $xls_file, $options) = @_;
    
    my @worksheets;
    eval {    
       @worksheets = XLSParse::read_excel($format => $xls_file);
    };
    if ($@) {
        warn "FAIL: $@\n";
        return {
            parse_errors => [iget("Не удалось разобрать формат excel файла")]
        }
    }
    
    return XLSParse::groupxls2camp_snapshot(\@worksheets, $options);
}

# ------------------------------------------------------------------------------

=head2 get_snapshot_warnings_and_errors_for_exists_camp

Получить предупреждение и ошибки при импорте в существующую кампанию

Предупреждаем при импорте в существующую кампанию:
 - при отсутствии в xls баннеров, фраз из кампании.
 - при отличиях только в минус-словах (например в результате работы автоуточнения)

При отсутствии некоторых объявлений
При отсутствии в файле строк с ID уже существующих в кампании баннеров пишем предупреждение:
В импортируемом файле отсутствуют объявления XXXX1, XXXX2, XXXX3.

Что вы хотите с ними сделать?
и радиобаттоны:

 - Оставить в кампании без изменений
 - При возможности удалить или остановить

При отсутствии некоторых фраз
При отсутствии в файле строк с уже существующими в кампании фразами пишем предупреждение:
В импортируемом файле отсутствуют следующие фразы:

    слон
    слон синий
    слон лесной

Что вы хотите с ними сделать?
и радиобаттоны:

 - Оставить в кампании без изменений
 - Удалить эти фразы из кампании

У фраз изменились минус-слова
Если хотя бы у одной фразы минус-слова в файле отличаются от её минус-слов на сервере.
Пишем на второй странице загрузки предупреждение:
У некоторых загружаемых ключевых фраз изменены минус-слова. Это могло быть следствием автоматического уточнения фраз.

Что вы хотите с ними сделать?
и радиобаттоны:

 - Оставить минус-слова без изменений
 - Применить изменения из загружаемого файла

Валюта файла не совпадает с валютой кампании
Пишем на второй странице загрузки предупреждение:
Файл был выгружен для рекламной кампании в %s, вы загружаете его в кампанию в %s. При загрузке указанные цены за клик будут иметь указанные в файле значения в валюте аккаунта.

Что вы хотите с ними сделать?
Чекбокс: Продолжить?

Сообщаем об ошибки в случае попытки редактировать баннер с креативом

На вход: cid, снапшот кампании.
На выходе: список (хеш c предупреждениями, список ошибкок)
    Хеш c предупреждениями:
        {
            lost_banners => '.....',
            lost_phrases => '.....',
            changes_minus_words => '.....',
            different_currency => '.....',
        }

    Список ошибок:
        (
            ...
        )

=cut

sub get_snapshot_warnings_and_errors_for_exists_camp($$$) {
    my ($uid, $cid, $snapshot) = @_;
    
    my (%warnings, @errors);
    
    my $exists_camp = get_camp_snapshot($uid, $cid, {pass_empty_groups => 1});
    my $all_retargeting_conditions =   Retargeting::get_retargeting_conditions(uid => $uid);
    
    my %lost_groups = map {
        my $g = $_;
        ($g->{pid} => {
            banners => {map {$_->{bid} => $_->{banner_type}} @{$g->{banners}}},
            phrases => {map {$_->{id} => $_->{phrase}} @{$g->{phrases}}},
            relevance_matches => {map {$_->{bid_id} => 1} @{$g->{relevance_match}}},
            retargetings => {map {$_->{ret_cond_id} => $all_retargeting_conditions->{$_->{ret_cond_id}}->{condition_name}} @{$g->{retargetings}}},
        })
    } @{$exists_camp->{groups}};

    my $changed_minus_words;
    my (@changed_mobile_banners_type, @changed_desktop_banners_type);
    foreach my $group (@{$snapshot->{groups}}) {
        
        next unless $group->{pid} && $lost_groups{$group->{pid}};
        my $item = $lost_groups{$group->{pid}}; 
        $item->{found} = 1;
        foreach (@{$group->{banners}}) {
            next unless $_->{bid};
            if ($item->{banners}->{$_->{bid}} ne $_->{banner_type}) {
                if ($item->{banners}->{$_->{bid}} eq 'mobile') { 
                    push @changed_mobile_banners_type, $_->{bid};
                } else {
                    push @changed_desktop_banners_type, $_->{bid};
                }
            }

            delete $item->{banners}->{$_->{bid}} if $_->{bid};
        }
        foreach my $phrase (@{$group->{phrases}}) {
            if ($phrase->{id} && $item->{phrases}->{$phrase->{id}}) {
                my $exists_phrase = delete $item->{phrases}->{$phrase->{id}};
                unless ($changed_minus_words) {
                    my ($old_phrase, $old_minus_words) = _split_by_minus_words($exists_phrase);
                    my ($new_phrase, $new_minus_words) = _split_by_minus_words($phrase->{phr});
                    
                    $changed_minus_words = $old_phrase eq $new_phrase && $old_minus_words ne $new_minus_words;
                }
            } 
        }

        foreach my $relevance_match (@{$group->{relevance_match}}) {
            if ($relevance_match->{bid_id} && $item->{relevance_matches}->{$relevance_match->{bid_id}}) {
                delete $item->{relevance_matches}->{$relevance_match->{bid_id}};
            }
        }

        foreach my $retargeting (@{$group->{retargetings}}) {
            if ($retargeting->{ret_cond_id} && $item->{retargetings}->{$retargeting->{ret_cond_id}}) {
                delete $item->{retargetings}->{$retargeting->{ret_cond_id}};
            }
        }
    }
    
    my (@lost_groups, @lost_banners, @lost_banners_with_creatives, @lost_phrases);
    while (my ($pid, $group) = each %lost_groups) {
        if ($group->{found}) {
            push @lost_banners, keys %{$group->{banners}};
            push @lost_phrases, values %{$group->{phrases}};
            push @lost_phrases, map {$Retargeting::CONDITION_XLS_PREFIX . $_} values %{$group->{retargetings}};
            push @lost_phrases, map {$CONDITION_XLS_RELEVANCE_MATCH . $_} keys %{$group->{relevance_matches}};
        } else {
            if ($snapshot->{is_group_format}) {
                push @lost_groups, $pid
            } else {
                push @lost_banners, keys %{$group->{banners}};
            }
        }
    }
    
    %warnings = (
        (@lost_groups
            ? (lost_groups => iget('В импортируемом файле отсутствуют группы: %s.', join ', ', nsort @lost_groups))
            : ()),
        ((@lost_banners || @lost_banners_with_creatives)
            ? (lost_banners => iget('В импортируемом файле отсутствуют объявления: %s.', join ', ', nsort (@lost_banners)))
            : ()),
        (@lost_phrases
            ? (lost_phrases => { title => iget('В импортируемом файле отсутствуют следующие фразы'), items => [sort @lost_phrases] })
            : ()),
        ($changed_minus_words
            ? (changes_minus_words => iget("У некоторых загружаемых ключевых фраз изменены минус-слова. Это могло быть следствием автоматического уточнения фраз."))
            : ()),
        (@changed_desktop_banners_type
            ? (changed_desktop_banners_type => { title => iget('Невозможно изменить тип десктопного объявления на мобильный'), items => [nsort @changed_desktop_banners_type] })
            : ()),
        (@changed_mobile_banners_type
            ? (changed_mobile_banners_type => { title => iget('Невозможно изменить тип мобильного объявления на десктопный'), items => [nsort @changed_mobile_banners_type] })
            : ())                            
    );
    
    hash_merge \%warnings, compare_currencies($snapshot->{currency}, $exists_camp->{currency});

    return \%warnings, \@errors;
}

sub compare_currencies {
    my ($new_currency, $exists_currency) = @_;
    
    return {error => iget("Валюта не найдена")} unless $new_currency && $exists_currency;

    if ($new_currency ne $exists_currency) {
        return {different_currency => iget("Файл был выгружен для рекламной кампании в %s, вы загружаете его в кампанию в %s. При загрузке указанные цены за клик будут иметь указанные в файле значения в валюте аккаунта.",
                                           Currencies::get_currency_constant($new_currency, 'name'), 
                                           Currencies::get_currency_constant($exists_currency, 'name'))};
    }
    return {};

}
# ------------------------------------------------------------------------------

=head2 csv2camp_snapshot

  импорт csv экспорта из adwords
  csv - на самом деле tsv

  колонки в файле:
    Campaign Type - тип кампании:
        "Display Network only - Mobile app installs" - мобильная
        остальные - текстовая
    Campaign - имя кампании, пока импортируем только одну, потом предоставим выбор, какую именно импортировать
    Keyword - фраза
    Keyword Type (в новом формате - Criterion Type) - тип фразы:
        Broad: аналог - просто ключевой запрос, знаки "+" игнорируем.
        Exact: аналог - фраза в кавычках, восклицательный знак перед каждым словом.
        Phrase: аналог - фраза без кавычек, восклицательный знак перед каждым словом.

        2. Добавление минус-слов к фразам
        Negative Broad: добавляем это минус-слово ко всем словам в группе
        Negative Exact: нет аналога, пропускаем ключевые слова этого типа.
        Negative Phrase: добавляем восклицательный знак к минус-слову, добавляем это минус-слово ко всем словам в группе.
        Campaign Negative Broad: минус фразы на всю кампанию
        Если минус-слово состоит из более чем одного слова - у нас нет аналога, пропускаем такие минус-слова.
    Tracking template, Custom parameters - трекинговая ссылка с дополнительными параметрами
    Final URL - url (также поддерживается 'Destination URL' для старого формата)

    (для стандартных объявлений)
    Headline - заголовок баннера
    Description Line 1 + Description Line 2 - тело баннера
    Display URL - отображаемый урл

    (для развёрнутых)
    Headline 1 - заголовок баннера
    Headline 2 + Description - тело баннера
    Path 1 + Path 2 - отображаемый урл

    (для сайтлинков)
    Link Text - текст
    Final URL или Destination URL - линк
    Description Line 1 + Description Line 2 - описание

    Ad Group - id баннера

    Для мобильных кампаний:
    Final URL - ссылка на стор
    Device Preference - таргетинг по устройству

    возвращается хеш: {csv_camp_array_ref, header_keys, parse_errors, geo_errors, parse_warnings}

    возвращаем ссылку на массив со всеми найдеными кампаниями в csv + стандартные ошибки как в xls2camp_snapshot()
    дополнительно в каждой кампании добавились ключи:
      campaign_minus_words: общие минус-слова на кампанию
      parse_warnings: ворнинги по баннерам которые не будут загружены
      camp_number: порядковый номер кампании для выбора в интерфейсе

    $options: опции
        csv_file_format -- формат csv-файла ("xls" | "csv")
        currency -- валюта
        ClientID -- обязателен, для валидации гео по транслокальному региону клиента

    DIRECT-9866, DIRECT-10143, DIRECT-10876, DIRECT-11045, DIRECT-10878, DIRECT-12700, DIRECT-17216
    DIRECT-17991

=cut

sub csv2camp_snapshot {
    my ($format, $csv_file, $options) = @_;

    my @csv;

    my $currency = $options->{currency};
    die 'no currency given' unless $currency;

    if (($options->{csv_file_format} || 'csv') =~ /xls/) {
        # если это csv сохраненный как xls
        my @sheets = XLSParse::read_excel($options->{csv_file_format} => $csv_file); 
        @csv = @{$sheets[0]};
        # undef -> ""
        for my $row (@csv) {
            for my $cell (@$row) {
                $cell = "" if ! defined $cell;
            }
        }

    } else {
        $csv_file =~ s/^\x{FEFF}//; # remove BOM
        for my $csv_file_line (split /\r?\n/, $csv_file) {
            $csv_file_line .= "\t"; # для последнего поля в строке
            my $row = [];
            # последовательно выбираем поля между \t, либо поля в двойных кавычках которые могут модержать в себе \t
            while ($csv_file_line =~ m/( " [^"]+ " | [^\t]+ )? \t/gx) {
                my $field = defined $1 ? $1 : '';
                push @$row, $field;
            }
            push @csv, $row;
        }
    }
    my $lang = Yandex::I18n::current_lang();

    my @header = map {s/\s+$//rxms} @{shift @csv};
    my %ind = map {($header[$_] => $_)} (0 .. $#header);

    unless (exists $ind{Campaign}
            && $ind{Keyword}
            && ($ind{'Keyword Type'} || $ind{'Criterion Type'})
            && ($ind{'Tracking template'} || $ind{'Final URL'} || $ind{'Destination URL'})
            && ($ind{'Headline'} || $ind{'Headline 1'})
            && ($ind{'Description Line 1'} || $ind{'Headline 2'} || $ind{'Description'})
            && $ind{'Ad Group'}
           )
    {
        return {
            csv_camps => [],
            header_keys => {},
            parse_errors => [iget('Неверный формат файла')],
            geo_errors => [],
        };
    }

    my %camp_info;
    my %groups;
    my %campaign_minus_words;
    my %campaign_sitelinks;
    my ($line_number, $group_number) = (1, 0);

    my $template_map = {
        lpurl => sub { shift()->{'Final URL'} },
        campaignid => 'campaign_id',
        adgroupid => 'gbid',
        device => 'device_type',
        creative => 'banner_id',
    };
    my $is_tycoon_organizations_feature_enabled = Client::ClientFeatures::has_tycoon_organizations_feature($options->{ClientID});
    my $template_re = join q{|}, map {quotemeta} keys %$template_map;
    for my $row (@csv) {
        $line_number++;
        my %rowh = zip @header, @$row;

        my $camp_name = $rowh{Campaign} || '';
        next if !$camp_name;

        $camp_info{$camp_name} = {mediaType => 'text'} unless exists $camp_info{$camp_name};
        # campaign params
        if ($rowh{'Campaign Type'} && $rowh{'Campaign Type'} =~ /Mobile app installs/) {
            $camp_info{$camp_name}->{mediaType} = 'mobile_content';
        }
        my $camp_type = $camp_info{$camp_name}->{mediaType};

        my $phrase_type = $rowh{'Keyword Type'} || $rowh{'Criterion Type'} || '';
        # Если имя группы пустое, значит текущая строка относится к параметрам кампании,
        # поэтому пропускаем обработку ниже
        my $group_name = $rowh{'Ad Group'} // '';
        my $group;
        if ($group_name =~ /\S/) {
            $group = $groups{$camp_name}->{$group_name} //= {
                banners => [],
                phrases => [],
                group_name => $group_name,
                group_number => ++$group_number,
            };
        }
        
        if ($rowh{Location}) {
            my $geo = get_geo_numbers($rowh{Location}, lang => $lang);
            if ($group) {
                $group->{geo} //= $geo;
            }
            else {
                $camp_info{$camp_name}->{geo} //= $geo;
            }
        }

        my $phrase = $rowh{Keyword} || '';
        for ($phrase) {
            s/[^$ALLOW_LETTERS \-]/ /g; # недопустимые символы убираем
            s/(?<!\S)-(\S)/ $1/g; # "-" до фразы убираем
            s/(\S)-(?!\S)/$1 /g;  # "-" после фразы убираем
            s/ - / /g;  # "-" вместо фразы убираем
            s/^\+//;
            s/\s+\+/ /;
            s/^\s+//;
            s/\s+$//;
        }

        if ($phrase_type =~ /Exact$/) {
            $phrase =~ s/\s+(\S)/ !$1/g;
            $phrase = qq/"!$phrase"/;
        } elsif ($phrase_type  =~ /Phrase$/) {
            $phrase =~ s/\s+(\S)/ !$1/g;
            $phrase = qq/!$phrase/;
        }

        if ($phrase_type =~ /^Campaign Negative/) {
            push @{$campaign_minus_words{$camp_name}}, $phrase;
            next;
        }

        my $tracking_url = $rowh{'Tracking template'} || '';
        my $url = $rowh{'Final URL'} || $rowh{'Destination URL'} || '';
        $url = $tracking_url || $url unless $camp_type eq 'mobile_content';
        s/\s+//g for ($url, $tracking_url);

        my $display_url =
            do { my ($du) = ($rowh{'Display URL'} || '') =~ m# ^ [^/]* / (.+) $ #xms; $du }
            || join('/' => grep {$_} ($rowh{'Path 1'}, $rowh{'Path 2'}))
            || undef;
        $display_url = undef  if Direct::Validation::Banners::validate_banner_display_href($display_url);

        my $banner_title = $camp_type eq 'mobile_content' ? $rowh{'Headline'} : $rowh{'Headline 1'};
        my $banner_title_extension = $rowh{'Headline 2'};
        my $sitelink_title = $rowh{'Link Text'};
        my $sitelink_body_tiny = $rowh{'Description Line 1'};
        my $banner_body = join ' ' => grep {$_} (
            $rowh{'Description Line 1'},
            $rowh{'Description Line 2'},
            $rowh{'Description'},
        );
        my $banner_type = $rowh{'Device Preference'};
        $banner_type = $banner_type && lc($banner_type) eq 'mobile' ? 'mobile' : 'desktop';
        if ($camp_type eq 'mobile_content') {
            # В кампании "Реклама мобильных приложений" - нет свойства is_mobile
            $banner_type = 'desktop';
        }

        for ($banner_body) {
            s/^\s+//;
            s/(?<!")"(?!")//g;
        }

        # убираем самые внешние и двойные кавычки из заголовока и текста баннера
        for ($banner_title, $banner_title_extension, $banner_body, $sitelink_title, $sitelink_body_tiny) {
            next unless defined $_;
            s/^"(.+)"\r?$/$1/;
            s/"{2,}/"/g;
        }

        # шаблоны
        for my $field_for_template ($banner_title, $banner_title_extension, $banner_body, $url, $tracking_url, $display_url, $sitelink_title, $sitelink_body_tiny) {
            next if !$field_for_template;
            $field_for_template =~ s/{($template_re)}/my $m = $template_map->{$1}; ref $m eq 'CODE' ? $m->(\%rowh) : "{$m}" /egxms;

            $field_for_template =~ s/{(keyword):([^}]+)}/\#$2\#/gi;
            # запоминаем нужный регистр для всех фраз в баннере
            if ($group && !$group->{need_change_capitalization_of_phrases} && $1 && $2) {
                $group->{need_change_capitalization_of_phrases} = $1;
            }
        }

        if ($sitelink_title && $url) {
            my $sitelink = {
                title => $sitelink_title,
                href => clear_banner_href($url),
                description => $banner_body || undef,
                description_small => $sitelink_body_tiny || undef,
            };
            if ($group) {
                push @{$group->{sitelinks}}, $sitelink;
            }
            else {
                push @{$campaign_sitelinks{$camp_name}}, $sitelink;
            }
        }
        next if !$group;

        if ($group && $banner_title && $banner_body) {
            $banner_body =~ s/ ,(\S)/, $1/;
            my $banner = {
                title           => $banner_title,
                title_extension => $banner_title_extension,
                body            => $banner_body,
                href            => $url,
                has_href        => !!$url,
                banner_type     => $banner_type,
                is_mobile       => $banner_type eq 'mobile',
                line_number     => $line_number,
                ad_type         => $camp_type,
            };

            if ($is_tycoon_organizations_feature_enabled) {
                $banner->{permalink} = $rowh{'Business Id'};
            }

            if ($camp_type eq 'text') {
                 $banner->{display_href} = $display_url;
            }
            elsif ($camp_type eq 'mobile_content') {
                $banner->{href} = $tracking_url;
                $banner->{has_href} = !!$tracking_url;
                $banner->{store_content_href} = $url;
                $banner->{device} = $rowh{'Device Preference'} || 'All';
            }
            push @{$group->{banners}}, $banner;
        }

        if ($group && $phrase_type =~ /^Negative/) {
            push @{$group->{minus_words}}, $phrase;
            next;
        }

        if ($group && $phrase && $phrase_type !~ /Negative/) {
            push @{$group->{phrases}}, {
                phr => $phrase,
                id => 0,
                price => get_currency_constant($currency, 'DEFAULT_PRICE'),
                line_number => $line_number
            }
        }
    }

    my $campaigns = [];
    my $camp_number = 0;

    # Специально для `mobile_content` кампании: храним результаты проверенных `store_content_href`
    my %store_content_href_to_content_info;

    for my $camp_name (keys %groups) {
        my $campaign = {
            camp_name => $camp_name,
            currency => $currency,
            groups => [],
            %{ $camp_info{$camp_name} },
            parse_warnings => [],
            parse_errors => [],
        };
        my $camp_type = $campaign->{mediaType} || 'text';

        my @valid_campaign_sitelinks;
        my $sum_sitelink_length = 0;
        for my $sitelink (@{$campaign_sitelinks{$camp_name}}) {
            if (@valid_campaign_sitelinks >= $Direct::Validation::SitelinksSets::SITELINKS_NUMBER) {
                my $msg = iget('Часть быстрых ссылок в кампании "%s" были пропущены из-за превышения допустимого количества', $camp_name);
                $campaign->{parse_warnings_hash}->{$msg} = 1;
                last;
            }

            my $title_length = length $sitelink->{title};
            if ($title_length > $Direct::Validation::SitelinksSets::ONE_SITELINK_MAX_LENGTH) {
                my $msg = iget('Часть быстрых ссылок в кампании "%s" были пропущены из-за превышения допустимого размера текста', $camp_name);
                $campaign->{parse_warnings_hash}->{$msg} = 1;
                next;
            }
            if ($sum_sitelink_length + $title_length > $Direct::Validation::SitelinksSets::SITELINKS_MAX_LENGTH) {
                my $msg = iget('Тексты быстрых ссылок в кампании "%s" превышают допустимый лимит, ссылки будут загружены неполностью', $camp_name);
                $campaign->{parse_warnings_hash}->{$msg} = 1;
                next;
            }
            if (length($sitelink->{href}) > 1024) {
                my $msg = iget('URL быстрой ссылки кампании "%s" превышает лимит символов, ссылка не будет добавлена', $camp_name);
                $campaign->{parse_warnings_hash}->{$msg} = 1;
                next;
            }
            if ($sitelink->{description} && length($sitelink->{description}) > $Direct::Validation::SitelinksSets::ONE_SITELINK_DESC_MAX_LENGTH) {
                my $msg = iget('Описания быстрых ссылок в кампании "%s" превышают допустимый лимит, описания будут загружены неполностью', $camp_name);
                $campaign->{parse_warnings_hash}->{$msg} = 1;

                $sitelink->{description} = length($sitelink->{description_small}) <= $Direct::Validation::SitelinksSets::ONE_SITELINK_DESC_MAX_LENGTH
                    ? $sitelink->{description_small}
                    : undef;
            }

            $sum_sitelink_length += $title_length;
            push @valid_campaign_sitelinks, $sitelink;
        }

        for my $group (sort {$a->{group_number} <=> $b->{group_number}} values %{$groups{$camp_name}}) {

            # add campaign warning about title or body
            unless (@{$group->{banners}}) {
                my $msg = iget(($camp_type ne 'mobile_content')
                    ? 'Не будет загружена группа "%s" из кампании "%s": не заданы Заголовок 1 и/или текст объявления'
                    : 'Не будет загружена группа "%s" из кампании "%s": не заданы Заголовок и/или текст объявления',
                        $group->{group_name}, $campaign->{camp_name});
                $campaign->{parse_warnings_hash}->{$msg} = 1;
            }

            # add campaign warning about phrases
            unless (@{$group->{phrases}}) {
                my $msg = iget('Не будет загружена группа "%s" из кампании "%s": не заданы фразы объявления',
                        $group->{group_name}, $campaign->{camp_name});
                $campaign->{parse_warnings_hash}->{$msg} = 1;
            }

            next if !@{$group->{banners}} || !@{$group->{phrases}};

            my @valid_sitelinks;
            $sum_sitelink_length = 0;
            for my $sitelink (@{$group->{sitelinks}}) {
                if (@valid_sitelinks >= $Direct::Validation::SitelinksSets::SITELINKS_NUMBER) {
                    my $msg = iget('Часть быстрых ссылок в группе "%s" были пропущены из-за превышения допустимого количества', $group->{group_name});
                    $campaign->{parse_warnings_hash}->{$msg} = 1;
                    last;
                }
                
                my $title_length = length $sitelink->{title};
                if ($title_length > $Direct::Validation::SitelinksSets::ONE_SITELINK_MAX_LENGTH) {
                    my $msg = iget('Часть быстрых ссылок в группе "%s" были пропущены из-за превышения допустимого размера текста', $group->{group_name});
                    $campaign->{parse_warnings_hash}->{$msg} = 1;
                    next;
                }
                if ($sum_sitelink_length + $title_length > $Direct::Validation::SitelinksSets::SITELINKS_MAX_LENGTH) {
                    my $msg = iget('Тексты быстрых ссылок в группе "%s" превышают допустимый лимит, ссылки будут загружены неполностью', $group->{group_name});
                    $campaign->{parse_warnings_hash}->{$msg} = 1;
                    next;
                }
                if (length($sitelink->{href}) > 1024) {
                    my $msg = iget('Часть быстрых ссылок в группе "%s" были пропущены из-за превышения допустимой длины ссылки', $group->{group_name});
                    $campaign->{parse_warnings_hash}->{$msg} = 1;
                    next;
                }
                if ($sitelink->{description} && length($sitelink->{description}) > $Direct::Validation::SitelinksSets::ONE_SITELINK_DESC_MAX_LENGTH) {
                    my $msg = iget('Описания быстрых ссылок в группе "%s" превышают допустимый лимит, описания будут загружены неполностью', $group->{group_name});
                    $campaign->{parse_warnings_hash}->{$msg} = 1;
                    $sitelink->{description} = length($sitelink->{description_small}) <= $Direct::Validation::SitelinksSets::ONE_SITELINK_DESC_MAX_LENGTH
                        ? $sitelink->{description_small}
                        : undef;
                }

                $sum_sitelink_length += $title_length;
                push @valid_sitelinks, $sitelink;
            }

            my @valid_banners;
            for my $banner (@{$group->{banners}}) {

                state $increase_ad_text_limits_prop = Property->new('increase_ad_text_limits');
                my $use_new_ad_text_limits = $increase_ad_text_limits_prop->get(120);
                my $max_title_length = $use_new_ad_text_limits ? $NEW_MAX_TITLE_LENGTH : $MAX_TITLE_LENGTH;

                if ($camp_type eq 'mobile_content') {
                    my $body_error = Direct::Validation::Banners::validate_banner_body_length_mobile($banner->{body});
                    my $title_error = Direct::Validation::Banners::validate_banner_title_length_mobile($banner->{title}, $max_title_length);
                    if ($title_error || $body_error) {
                        my $msg = iget('Часть объявлений в группе "%s" были пропущены из-за превышения допустимого количества символов в заголовке или тексте', $group->{group_name});
                        $campaign->{parse_warnings_hash}->{$msg} = 1;
                        next;
                    }
                } else {
                    my $body_error = Direct::Validation::Banners::validate_banner_body_length($banner->{body});
                    my $title_error = $use_new_ad_text_limits ?
                        Direct::Validation::Banners::validate_banner_title_length_mobile($banner->{title}, $max_title_length) :
                        Direct::Validation::Banners::validate_banner_title_length($banner->{title}, $max_title_length);
                    my $title_extension_error = Direct::Validation::Banners::validate_banner_title_length($banner->{title_extension}, $MAX_TITLE_EXTENSION_LENGTH);
                    if ($title_error || $title_extension_error || $body_error) {
                        my $msg = iget('Часть объявлений в группе "%s" были пропущены из-за превышения допустимого количества символов или превышения количества точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков в заголовках или тексте', $group->{group_name});
                        $campaign->{parse_warnings_hash}->{$msg} = 1;
                        next;
                    }
                }
                $banner->{sitelinks} = \@valid_sitelinks  if @valid_sitelinks;
                $banner->{sitelinks} ||= \@valid_campaign_sitelinks  if @valid_campaign_sitelinks;
                push @valid_banners, $banner;
            }
            $group->{banners} = \@valid_banners;

            # мобильные группы пилим на части по store_content_href+device_type_targeting
            my @banner_chunks;
            if ($camp_type eq 'mobile_content') {
                push @banner_chunks,
                    map {my $k = $_; [grep {_get_mobile_key($_) eq $k} @{$group->{banners}}]}
                    uniq map {_get_mobile_key($_)}
                    @{$group->{banners}};
            }
            else {
                push @banner_chunks, $group->{banners};
            }

            my $phrases = _extract_csv_group_phrases($group);
            my $idx = 0;
            for my $banner_chunk (@banner_chunks) {
                for my $keywords (_cut_phrases($phrases, $options->{ClientID})) {
                    my $direct_group = {
                        adgroup_type => ($camp_type eq 'mobile_content' ? 'mobile_content' : 'base'),
                        phrases => $keywords,
                        geo => $options->{geo} // $group->{geo} // $campaign->{geo} // '',
                        region => '', 
                        group_name => $group->{group_name},
                        minus_words => $group->{minus_words},
                    };
                    my $extra_banner_fields = { contact_info => '' };

                    my @group_errors;
                    my $errors = {};

                    if ($camp_type eq 'mobile_content') {
                        $direct_group->{store_content_href} = $banner_chunk->[0]->{store_content_href};
                        $direct_group->{device_type_targeting} = $banner_chunk->[0]->{device} eq 'Mobile' ? ['phone'] : ['phone', 'tablet'];
                        $direct_group->{network_targeting} = ['cell', 'wifi'];
                        $direct_group->{min_os_version} ||= '0';
                        $extra_banner_fields->{primary_action} ||= 'download';
                        $extra_banner_fields->{reflected_attrs} ||= ['rating','price','icon','rating_votes'];

                        # Простукиваем ссылку на приложение
                        my $content_info =
                            $store_content_href_to_content_info{$direct_group->{store_content_href}} //=
                                MobileContent::ajax_mobile_content_info($direct_group->{store_content_href}, $options->{ClientID});
                        if (my $content = $content_info->{response}) {
                            hash_merge $direct_group, $content;
                            if (!$direct_group->{min_os_version} || none { $_ eq $direct_group->{min_os_version} } @{$MobileContent::OS_VERSIONS{$direct_group->{os_type}}}) {
                                $direct_group->{min_os_version} = $MobileContent::OS_VERSIONS{$direct_group->{os_type}}->[0] || '0';
                            }
                            $extra_banner_fields->{primary_action} = $content->{available_actions}->[0];
                        } else {
                            push @group_errors, iget('Не удалось получить информацию о приложении (%s)', $content_info->{error}->{message});
                        }
                    };

                    $direct_group->{banners} = [map {hash_merge {}, $extra_banner_fields, $_} @$banner_chunk];

                    my $user_adgroup = yclone($direct_group);
                    for (@{$user_adgroup->{phrases} // []}) {
                        $_->{phrase} = $_->{phr};
                        $_->{is_suspended} = 0;
                    }

                    my $smart = Direct::AdGroups2::Smart->from_user_data($options->{uid}, 0, [$user_adgroup], campaign => $campaign);
                    unless ($smart->is_valid) {
                        push @group_errors, @{$smart->errors}, @{$smart->validation_result->get_error_descriptions};
                    }

                    push @group_errors, map {@{$errors->{$_}}} grep {$errors->{$_}} qw/common phrases/;
                    push @group_errors, $errors->{group_name} if $errors->{group_name};
                    push @group_errors, map {map {@$_} values %$_} values %{$errors->{banners}}; 
                    if (@group_errors) {
                        my $msg =
                            iget('Не будет загружена группа "%s": ', $group->{group_name})
                            . join '; ', uniq map {lcfirst($_)} @group_errors;
                        $msg .= '.' if $msg !~ /\.$/;
                        $campaign->{parse_warnings_hash}->{$msg} = 1;
                    } else {
                        $direct_group->{group_name} .= '-' . $idx if $idx > 0; 
                        push @{$campaign->{groups}}, $direct_group;
                        $idx++;
                    }
                }
            }
        }
                
        $campaign->{campaign_minus_words} = $campaign_minus_words{$camp_name} || [];
        push @{$campaign->{parse_warnings}}, keys %{ $campaign->{parse_warnings_hash} } if $campaign->{parse_warnings_hash};
        $campaign->{camp_number} = $camp_number++;
        unless (@{$campaign->{groups}}) {
            push @{$campaign->{parse_errors}},
                 iget('Не будет загружена кампания "%s": отсутствуют корректные группы для загрузки', $campaign->{camp_name});
        }

        # В валидации текст ключевой фразы в группах находятся в поле phrase, в нас же лежат в поле phr
        my $for_validate_campaign = yclone($campaign);
        foreach my $g (@{$for_validate_campaign->{groups}}) {
            foreach (@{$g->{phrases} // []}) {
                $_->{phrase} = $_->{phr};
            }
        }

        my $mw_validation_result = validate_group_camp_minus_words($for_validate_campaign, use_line_numbers => 1);

        if (@{$mw_validation_result->{errors}}) {
            push @{$campaign->{parse_errors}}, iget('Не будет загружена кампания "%s": ', $for_validate_campaign->{camp_name})
                                               . join '; ', uniq map {lcfirst($_)} @{$mw_validation_result->{errors}};
        } elsif (@{$mw_validation_result->{warnings}}) {
            push @{$campaign->{parse_warnings}}, iget('Кампания "%s": ', $for_validate_campaign->{camp_name})
                                                 . join '; ', uniq map {lcfirst($_)} @{$mw_validation_result->{warnings}};
        }
        $campaign->{campaign_minus_words} = MinusWords::polish_minus_words_array($campaign->{campaign_minus_words});
        $_->{minus_words} = MinusWords::polish_minus_words_array($_->{minus_words}) foreach  @{$campaign->{groups}};
        push @$campaigns, $campaign;
    }
    
    return {
        csv_camps => $campaigns,
        header_keys => {},
        parse_errors => @$campaigns ? [] : [iget('Невозможно загрузить пустую кампанию.')],
        geo_errors => [],
    };
}

sub _get_mobile_key {
    my ($banner) = $_;
    return join q{}, map {$banner->{$_} || q{-}} qw/ device store_content_href /;
}

sub _extract_csv_group_phrases {
    
    my $group = shift;
    
    my @phrases;
    for my $phrase (@{$group->{phrases}}) {
        
        # нужно ли менять регистр слов во фразе (для использования в шаблонах)
        if ($group->{need_change_capitalization_of_phrases}) {
            $phrase->{phr} = _change_capitalization_of_phrases($phrase->{phr}, $group->{need_change_capitalization_of_phrases});
        }

        # Для всех частиц, предлогов и др. принудительно проставляем символ "+".
        $phrase->{phr} = join(" ", map {Yandex::MyGoodWords::is_stopword($_) ? "+$_" : $_} split /\s+/, $phrase->{phr});

        # Не загружаем ключевые фразы длиннее 7 слов (загружаем объявление без этих фраз).
        next  if scalar(split_phrase_with_normalize($phrase->{phr})) > $Settings::MAX_WORDS_IN_KEYPHRASE;
        next  if length $phrase->{phr} > $Settings::MAX_PHRASE_LENGTH; 

        push @phrases, $phrase;
    }
    return \@phrases;
}

sub _cut_phrases {
    
    my ($phrases, $client_id) = @_;
    
    my @chunks;
    my $idx = Models::AdGroup::get_first_excess_phrase_idx({phrases => $phrases}, client_id => $client_id);
    while ($idx > 0) {
        push @chunks, [splice @$phrases, 0, $idx];
        $idx = Models::AdGroup::get_first_excess_phrase_idx({phrases => $phrases}, client_id => $client_id);
    }
    push @chunks, $phrases if @$phrases;

    return @chunks;
}

#-----------------------------------------------------------

=head2 _change_capitalization_of_phrases($phrase, $capitalization_type)

    для использования фраз в шаблоне требуется поменять регистр слов во фразе

    $capitalization_type:
        keyword ключевые слова не меняем. (туры в Египет)
        Keyword во всех ключевых фразах объявления первое слово должно быть с большой буквы (Туры в Египет)
        KeyWord все слова в ключевых фразах с большой буквы (Туры В Египет)
        KEYWord первое слово в ключевой фразе капсом, остальные слова с большой буквы (ТУРЫ В Египет)
        KeyWORD первое слово ключевой фразе с большой буквы, остальные слова капсом. (Туры В ЕГИПЕТ)
        KEYWORD = KeyWord (Туры В Египет)

=cut

sub _change_capitalization_of_phrases($$) {
    my ($phrase, $capitalization_type) = @_;

    return $phrase if $capitalization_type eq 'keyword';

    my @phrases = split /\s+/, $phrase;

    if ($capitalization_type eq 'Keyword') {
        $phrases[0] =~ s/([$ALLOW_LETTERS])/uc($1)/e;
    } elsif ($capitalization_type eq 'KeyWord' || $capitalization_type eq 'KEYWORD') {
        @phrases = map {s/([$ALLOW_LETTERS])/uc($1)/e; $_} @phrases;
    } elsif ($capitalization_type eq 'KEYWord') {
        @phrases = map {s/([$ALLOW_LETTERS])/uc($1)/e; $_} @phrases;
        $phrases[0] = uc($phrases[0]);
    } elsif ($capitalization_type eq 'KeyWORD') {
        my $first_phrase = $phrases[0];
        $first_phrase =~ s/([$ALLOW_LETTERS])/uc($1)/e;
        @phrases = map {uc($_)} @phrases;
        $phrases[0] = $first_phrase;
    } else {
        return $phrase;
    }

    return join(" ", @phrases);
}

sub validate_tags_snapshot {
    my ($cid, $banners) = @_;
    my $campaign_tags;    

    ## no critic (Freenode::DollarAB)
    foreach my $b (@$banners) {
        $campaign_tags = hash_merge $campaign_tags, {map { _normalize_tag_name($_->{tag_name}) => 1 } @{$b->{tags}}};
    }

    # подтягиваем уже существующие метки на кампанию, т.к. могут быть метки, не привязанные к объявлениям
    if ($cid) {
        $campaign_tags = hash_merge $campaign_tags, {map { _normalize_tag_name($_->{value}) => 1 } @{Tag::get_all_campaign_tags($cid)}};
    }
    return Tag::check_camp_count_limit(scalar keys %$campaign_tags);
}

=head2 validate_group_camp_minus_words ($camp, %options)

    Валидирует единые минус-слова на кампанию/группу для снапшота кампании в формате групп
    В валидацию входит:
        1) Проверка на длину в соответствии с настройками для кампании/группы
        2) Проверка на пересечение ключевых слов с минус-словами

    Входные параметры (массив):
        $camp    => ссылка на хеш, снапшот кампании
        %options => дополнительные опции

    Возвращяемое значение (ссылка на хеш):
        errors   => ссылка на массив со списком найденных ошибок (пока только превышение допустимой длины минус-слов)
        warnings => ссылка на массив со списком предупреждений (о пересечениях ключевых слов с минус-словами)

    Дополнительные опции:
        cid => дополнительно загрузить данные из кампании
        use_line_numbers => использовать номера строк для сообщений об ошибках/предупреждениях или нет
        extended => возвращать ошибки в errors/warnings в виде структуры с указанием номеров строк

=cut

# Процедура проверки результатов валидации минус-слов, используется только внутри validate_group_camp_minus_words
sub _process_intersection_result {
    my ($res, $adgroup, %options) = @_;

    my @warnings;
    # Пересечения ключевых слов с минус-словами на кампанию/группу
    for my $prefix ('campaign_', '') {
        if (@{$res->{"${prefix}key_words"}}) {
            my (@line_numbers, @line_numbers_prefix);
            if ($options{use_line_numbers} && $adgroup) {
                for my $w (@{$res->{"${prefix}key_words"}}) {
                    push @line_numbers, map { $_->{line_number } || () } (grep { ($_->{phr} || $_->{phrase}) =~ /\b\Q$w\E\b/ } @{$adgroup->{phrases}});
                }
                if (@line_numbers == 1) {
                    push @line_numbers_prefix, iget('Строка %s:', array_to_compact_str(@line_numbers));
                } elsif (@line_numbers > 1) {
                    push @line_numbers_prefix, iget('Строки %s:', array_to_compact_str(@line_numbers))
                }
            }

            my $text = $prefix eq 'campaign_'
                ? iget("Ключевые фразы (%s) пересекаются с едиными минус-фразами на кампанию. ",   join(', ', @{$res->{"${prefix}key_words"}}))
                : iget("Ключевые фразы (%s) пересекаются с минус-фразами на объявление. ", join(', ', @{$res->{"${prefix}key_words"}}))
            ;
            $text .= iget("Данные минус-фразы не будут учитываться при показах по перечисленным ключевым фразам.");

            $text = join(' ', @line_numbers_prefix, $text);
            push @warnings, !$options{extended} ? $text : +{line_number => min(@line_numbers) || 0, warning => $text};
        }
    }

    return @warnings;
}

sub validate_group_camp_minus_words {
    my $camp = shift;
    my %options = @_;

    my (@errors, @warnings);

    # Валидация единых минус-слов на кампанию
    my @campaign_minus_words_arg;
    if (exists $camp->{campaign_minus_words}) {
        if (my @x = @{MinusWords::check_minus_words($camp->{campaign_minus_words}, type => 'campaign')}) {
            my $text = join(" ", iget('Единые минус-фразы на кампанию:'), @x);
            push @errors, !$options{extended} ? $text : +{line_number => 0, error => $text};
        } else {
            push @campaign_minus_words_arg, campaign_minus_words => $camp->{campaign_minus_words};
            $camp->{campaign_minus_words} = MinusWords::polish_minus_words_array($camp->{campaign_minus_words});
        }
    } elsif ($options{cid}) {
        push @campaign_minus_words_arg, cid => $options{cid};
    }

    # Валидация минус-слов для групп
    for my $adgroup (@{$camp->{groups}}) {
        my @adgroup_prefix;
        push @adgroup_prefix, iget('Группа объявлений "%s":', $adgroup->{group_name}) if $adgroup->{group_name};
        push @adgroup_prefix, iget('Строка %s:', $adgroup->{line_number}) if $adgroup->{line_number} && $options{use_line_numbers};

        # Проверим длину
        if (my @x = @{MinusWords::check_minus_words($adgroup->{minus_words}, type => 'group')}) {
            my $text = join(" ", @adgroup_prefix, @x);
            push @errors, !$options{extended} ? $text : +{line_number => $adgroup->{line_number}, error => $text};
            next;
        }

        my $x = MinusWords::key_words_with_minus_words_intersection(
            @campaign_minus_words_arg,
            minus_words => $adgroup->{minus_words},
            key_words => [map { $_->{phrase} } @{$adgroup->{phrases}}],
            skip_errors => 1,
        );
        $adgroup->{minus_words} = MinusWords::polish_minus_words_array($adgroup->{minus_words});

        push @warnings, _process_intersection_result($x, $adgroup, %options);
    }

    # Дополнительная валидация минус-фраз кампании, заданной в cid
    if ($options{cid}) {
        my $x = MinusWords::key_words_with_minus_words_intersection(
            @campaign_minus_words_arg,
            cid => $options{cid},
            skip_errors => 1,
        );
        push @warnings, _process_intersection_result($x, undef, %options);
    }

    return {
        errors   => \@errors,
        warnings => \@warnings,
    };
}

=head2 _validate_group_camp_snapshot($xls, $client_id, %options)

Опции:
    ignore_existing_camp_tags - если выполняется валидация снапшота для уже существующей кампании, игнорировать проверки на лимит тэгов,
                                т.к. мы ещё не знаем, куда мы будем грузить эту кампании - сюда же, в другую или в новую.

=cut
sub _validate_group_camp_snapshot {

    my ($xls, $client_id, %options) = @_;
    my (@errors, @warnings);

    unless (scalar @{$xls->{groups}}) {
        push @errors, {priority => 0, error => iget("Неправильный формат файла загрузки.")};
    }
    if (any {$_->{contact_info} eq "+"} map {@{$_->{banners}}}  @{$xls->{groups}}) {
        push @errors, map {
            {priority => 1, error => $_}
        } map {iget('Контактная информация: ') . lcfirst_na($_)}
            validate_contactinfo($xls->{contact_info});
    }

    my $all_retargeting_conditions = $xls->{all_retargeting_conditions};
    my $cid = $xls->{cid};
    foreach my $group (@{$xls->{groups}}) {
        $cid = get_cid(pid => $group->{pid}) if $group->{pid} && !$cid;
        $cid = $options{cid} unless $cid;
        $group->{cid} = $cid if defined $cid;
        my $prefix = $group->{group_name} ? iget('Группа объявлений "%s":', $group->{group_name}) : '';
        my @group_lines = map {$_->{line_number}} @{$group->{banners}};
        my $line_numbers = @group_lines > 1
            ? iget('Строки %s:', array_to_compact_str(@group_lines))
            : iget('Строка %s:', array_to_compact_str(@group_lines));

        if ($group->{tags}) {
            my $count_error = Tag::check_banner_count_limit(scalar @{$group->{tags}});
            push @errors, {
                priority => 15,
                line_number => _get_group_line_number($group),
                pid => $group->{pid},
                error => $count_error
            } if $count_error;

            foreach my $t (@{$group->{tags}}) {
                my $tag_err = Tag::check_tag_text($t->{tag_name});
                push @errors, {
                    priority => 20,
                    line_number => _get_group_line_number($group),
                    pid => $group->{pid},
                    error => join " ", $prefix, $line_numbers, lcfirst_na($tag_err),
                } if $tag_err;
            }
        }

        foreach my $banner (@{$group->{banners}}) {
            $banner->{has_href} = 1 if $banner->{href};
            $banner->{has_vcard} = 1 if $banner->{contact_info} eq "+";
        }
        my $group_errors = Models::AdGroup::validate_group($group, {
            ClientID                                => $client_id,
            for_xls                                 => 1,
            skip_changing_banner_type               => 1,
            skip_price_check                        => 1,
            exists_ret_conds                        => $all_retargeting_conditions,
            exists_group_retargetings               => {}, # в excel нельзя задать ret_id, только вычисление ret_cond_id по имени условия
            is_tycoon_organizations_feature_enabled => $options{is_tycoon_organizations_feature_enabled},
        });

        if ($group_errors->{common}) {
            push @errors, map {{
                priority => 10,
                pid => $group->{pid},
                error => join "\n", $prefix, $line_numbers, $_
            }} @{$group_errors->{common}};
        }
        if ($group_errors->{group_name}) {
            push @errors, {
                priority => 10,
                pid => $group->{pid},
                error => join "\n", $prefix, $line_numbers, @{$group_errors->{group_name}}
            };
        }

        if ($group_errors->{banners}) {
            foreach my $banner (@{$group->{banners}}) {
                my $prefix = $banner->{bid} ? iget("Объявление M-%s:", $banner->{bid}) : "";
                my @banner_errors = map {@$_} values %{$banner->{errors}};
                my $line_numbers = iget('Строка %s:', $banner->{line_number});
                push @errors, map { {
                    priority => 10,
                    bid => $banner->{bid},
                    error => join " ", $prefix, $line_numbers, $_
                } } @banner_errors;
            }
        }

        foreach ($group_errors->{phrases} ? @{$group_errors->{phrases}} : ()) {
            push @errors, {
                priority => 10,
                pid => $group->{pid},
                error => join " ", $prefix, $_
            }
        }

        foreach ($group_errors->{relevance_match} ? @{$group_errors->{relevance_match}} : ()) {
            push @errors, {
                    priority => 10,
                    pid => $group->{pid},
                    error => join " ", $prefix, $line_numbers, $_
                }
        }

        foreach ($group_errors->{retargetings} ? @{$group_errors->{retargetings}} : ()) {
            push @errors, {
                priority => 10,
                pid => $group->{pid},
                error => join " ", $prefix, $_
            }
        }
    }

    my $tag_count_error = validate_tags_snapshot($options{ignore_existing_camp_tags} ? undef : $cid, $xls->{groups});
    push @errors, {priority => 15, error => $tag_count_error} if $tag_count_error;

    # Валидация единых минус-слов
    my $mw_validation_result = validate_group_camp_minus_words($xls, use_line_numbers => 1);
    push @errors, {priority => 20, error => $_} for @{$mw_validation_result->{errors}};
    push @warnings, @{$mw_validation_result->{warnings}};

    return {
        errors => [map {$_->{error}} sort { ($a->{priority}//-1) <=> ($b->{priority}//-1 ) or ($a->{line_number}//0) <=> ($b->{line_number}//0) } @errors],
        warnings => \@warnings,
    };
}

=head2 _validate_and_fix_xls_campaign_snapshot($xls_campaign, $chief_uid, $client_id)

=cut

sub _validate_and_fix_xls_campaign_snapshot {
    my ($xls_campaign, $chief_uid, $client_id) = @_;

    # Пока поддерживаем только `mobile_content` кампанию
    croak "Unsupported campaign" unless $xls_campaign->{campaign_type} eq 'mobile_content';

    return {errors => [iget('Неправильный формат файла загрузки.')], warnings => []} if !@{$xls_campaign->{groups}};

    my (@errors, @warnings);

    # Важно! Ошибки на визитку/сайтлинки пока не рассматриваем, т.к. `mobile_content` не поддерживает эти сущности.

    #
    # Предобработка данных:
    #   Удаляем неподдерживаемые параметры, выставляем значения по умолчанию, выдаём предупреждения.
    #
    for my $adgroup (@{$xls_campaign->{groups}}) {
        for my $banner (@{$adgroup->{banners}}) {
            my @banner_prefix;
            push @banner_prefix, iget('Объявление M-%s:', $banner->{bid}) if $banner->{bid};
            push @banner_prefix, iget('Строка %s:', $banner->{line_number});

            if ($adgroup->{adgroup_type} eq 'mobile_content') {
                # Визитка/картинка/сайтлинки не поддерживаются
                if (str($banner->{contact_info}) eq '+') {
                    push @warnings, +{
                        line_number => $banner->{line_number},
                        warning => join(" ", @banner_prefix, iget('Мобильные объявления не поддерживают контактную информацию.')),
                    };
                    $banner->{contact_info} = '-';
                }
                if (@{$banner->{sitelinks} // []}) {
                    push @warnings, +{
                        line_number => $banner->{line_number},
                        warning => join(" ", @banner_prefix, iget('Мобильные объявления не поддерживают быстрые ссылки.')),
                    };
                    $banner->{sitelinks} = undef;
                }
                if (ref($banner->{callouts}) eq 'ARRAY' && @{$banner->{callouts}}) {
                    push @warnings, +{
                        line_number => $banner->{line_number},
                        warning => join(" ", @banner_prefix, iget('Мобильные объявления не поддерживают уточнения баннеров.')),
                    };
                    $banner->{callouts} = undef;
                }
                if ($banner->{display_href}) {
                    push @warnings, +{
                        line_number => $banner->{line_number},
                        warning => join(" ", @banner_prefix, iget('Мобильные объявления не поддерживают отображаемую ссылку.')),
                    };
                    $banner->{display_href} = undef;
                }

                # Выставляем значения по умолчанию для незаполненных полей
                push @{$banner->{reflected_attrs}}, 'icon' if !$banner->{mobile_icon};
                push @{$banner->{reflected_attrs}}, 'price' if !$banner->{mobile_price};
                push @{$banner->{reflected_attrs}}, 'rating' if !$banner->{mobile_rating};
                push @{$banner->{reflected_attrs}}, 'rating_votes' if !$banner->{mobile_rating_votes};
            }
        }
    }

    #
    # Валидация
    #
    for (1) {
        my $user_campaign = {type => $xls_campaign->{campaign_type}, currency => $xls_campaign->{currency}};

        my $smart = Direct::AdGroups2::Smart->from_user_data($chief_uid, 0, $xls_campaign->{groups}, campaign => $user_campaign, mk_fake_canvas_creatives => 1);
        # Критические ошибки (например при обработке данных, до валидации
        push @errors, {priority => 0, error => $_} for @{$smart->errors};
        last if @errors;

        if (!$smart->is_valid) {
            my $vr = $smart->validation_result;

            # Общие ошибки на все группы
            push @errors, map { +{priority => 0, error => $_->description} } @{$vr->get_generic_errors};

            # Ошибки по каждой группе
            for (my $i = 0; $i < @{$xls_campaign->{groups}}; $i++) {
                my $adgroup = $xls_campaign->{groups}->[$i];
                my $vr_adgroup = $vr->nested_objects->[$i];
                my ($vr_banners, $vr_keywords, $vr_retargetings) = delete @{$vr_adgroup->defects_by_field}{qw/banners keywords retargetings/};

                my (@adgroup_prefix_name, @adgroup_prefix_line);
                push @adgroup_prefix_name, iget('Группа объявлений "%s":', $adgroup->{group_name}) if ($adgroup->{group_name} // '') =~ /\S/;
                push @adgroup_prefix_line, iget('Строка %s:', $adgroup->{line_number});

                my @adgroup_prefix = (@adgroup_prefix_name, @adgroup_prefix_line);

                #
                # Логика для `mobile_content`
                #
                if ($adgroup->{adgroup_type} eq 'mobile_content') {
                    my ($content_os_type, $content_min_os_version);

                    # Простукиваем ссылку на приложение
                    my $vr_store_content_href = $vr_adgroup->defects_by_field->{store_content_href} // Direct::ValidationResult->new();
                    if ($vr_store_content_href->is_valid) {
                        my $content_info = $adgroup->{mobile_content_info}
                                            || croak "Cannot find `mobile_content_info` in adgroup snapshot";
                        if (my $content = $content_info->{response}) {
                            $content_os_type = $content->{os_type};
                            $content_min_os_version = $content->{min_os_version};
                            if (!$content_min_os_version || none { $_ eq $content_min_os_version } @{$MobileContent::OS_VERSIONS{$content_os_type}}) {
                                $content_min_os_version = $MobileContent::OS_VERSIONS{$content_os_type}->[0] || '0';
                            }
                        } else {
                            my $store_data = parse_store_url($adgroup->{store_content_href});
                            ($content_os_type, $content_min_os_version) = @$store_data{qw/os_type min_os_version/};
                            push @errors, +{
                                priority => 10,
                                line_number => $adgroup->{line_number},
                                error => join(" ", @adgroup_prefix,
                                    iget('Не удалось получить информацию о приложении (%s)', $content_info->{error}->{message})
                                ),
                            };
                        }
                    }

                    # Часть ошибок переделываем в предупреждения

                    if (my $vr_network_targeting = delete $vr_adgroup->defects_by_field->{network_targeting}) {
                        if (!$vr_network_targeting->is_valid) {
                            push @warnings, +{
                                line_number => $adgroup->{line_number},
                                warning => join(" ", @adgroup_prefix, iget('Указано неверное значение поля "Тип связи". Будет сохранено значение "Мобильная связь и Wi-Fi".')),
                            };
                            $adgroup->{network_targeting} = [qw/wifi cell/];
                        }
                    }

                    if (my $vr_min_os_version = delete $vr_adgroup->defects_by_field->{min_os_version}) {
                        if (!$vr_min_os_version->is_valid && $vr_store_content_href->is_valid) {
                            # Выводим предупреждение только если ссылка на store корректная
                            $adgroup->{min_os_version} = $content_min_os_version // '0';
                            push @warnings, +{
                                line_number => $adgroup->{line_number},
                                warning => join(" ", @adgroup_prefix,
                                    iget('Указано неверное значение поля "Версия ОС". Будет сохранено значение "%s".', ($content_os_type ? $content_os_type.' ' : '').$adgroup->{min_os_version})
                                ),
                            };
                        }
                    }

                    if (my $vr_device_type_targeting = delete $vr_adgroup->defects_by_field->{device_type_targeting}) {
                        if (!$vr_device_type_targeting->is_valid) {
                            push @warnings, +{
                                line_number => $adgroup->{line_number},
                                warning => join(" ", @adgroup_prefix, iget('Указано неверное значение поля "Тип устройства". Будет сохранено значение "Все".')),
                            };
                            $adgroup->{device_type_targeting} = [qw/phone tablet/];
                        }
                    }
                }

                # Ошибки на группу
                push @errors, map { +{
                    priority => 10,
                    line_number => $adgroup->{line_number},
                    error => join(" ", @adgroup_prefix_name, @adgroup_prefix_line, $_),
                } } @{$vr_adgroup->get_error_descriptions};

                # Ошибки на баннеры
                if (!$vr_banners->is_valid) {
                    my @banner_lines = map { $_->{line_number} } @{$adgroup->{banners}};
                    my $banner_line_numbers = @banner_lines > 1
                        ? iget('Строки %s:', array_to_compact_str(@banner_lines))
                        : iget('Строка %s:', array_to_compact_str(@banner_lines));

                    # Общие на все баннеры
                    push @errors, map { +{
                        priority => 10,
                        line_number => $adgroup->{line_number},
                        error => join(" ", @adgroup_prefix_name, $banner_line_numbers, $_->description),
                    } } @{$vr_banners->get_generic_errors};

                    # По каждому баннеру
                    for (my $j = 0; $j < @{$adgroup->{banners}}; $j++) {
                        my $banner = $adgroup->{banners}->[$j];
                        my $vr_banner = $vr_banners->nested_objects->[$j];

                        my @banner_prefix;
                        push @banner_prefix, iget('Объявление M-%s:', $banner->{bid}) if $banner->{bid};
                        push @banner_prefix, iget('Строка %s:', $banner->{line_number});

                        #
                        # Логика для `mobile_content`
                        #
                        if ($adgroup->{adgroup_type} eq 'mobile_content') {
                            # Часть ошибок переделываем в предупреждения
                            if (my $vr_href = delete $vr_banner->defects_by_field->{href}) {
                                if (!$vr_href->is_valid) {
                                    push @warnings, +{
                                        line_number => $banner->{line_number},
                                        warning => join(" ", @banner_prefix, iget('Трекинговая ссылка не будет сохранена. Неверный формат ссылки.')),
                                    };
                                    $banner->{href} = undef;
                                }
                            }
                        }

                        push @errors, map { +{
                            priority => 10,
                            line_number => $banner->{line_number},
                            error => join(" ", @banner_prefix, $_),
                        } } @{$vr_banner->get_error_descriptions};
                    }
                }

                # Ошибки на ключевые фразы
                if (!$vr_keywords->is_valid) {
                    # Общие на все фразы
                    # Пропустим ошибку про превышение длины фраз
                    push @errors, map { +{
                        priority => 10,
                        line_number => $adgroup->{line_number},
                        error => join(" ", @adgroup_prefix_name, $_->description),
                    } } grep { $_->name ne 'ReachLimit' } @{$vr_keywords->get_generic_errors};

                    for (my $j = 0; $j < @{$adgroup->{phrases}}; $j++) {
                        my $keyword = $adgroup->{phrases}->[$j];
                        my $vr_keyword = $vr_keywords->nested_objects->[$j];

                        push @errors, map { +{
                            priority => 10,
                            line_number => $keyword->{line_number},
                            error => join(" ", @adgroup_prefix_name, iget('Строка %s:', $keyword->{line_number}), $_),
                        } } @{$vr_keyword->get_error_descriptions};
                    }
                }

                # Ошибки на ретаргетинг
                if (!$vr_retargetings->is_valid) {
                    # Общие на весь ретаргетинг
                    push @errors, map { +{
                        priority => 10,
                        line_number => $adgroup->{line_number},
                        error => join(" ", @adgroup_prefix_name, $_->description),
                    } } @{$vr_retargetings->get_generic_errors};

                    for (my $j = 0; $j < @{$adgroup->{retargetings}}; $j++) {
                        my $retargeting = $adgroup->{retargetings}->[$j];
                        my $vr_retargeting = $vr_retargetings->nested_objects->[$j];

                        push @errors, map { +{
                            priority => 10,
                            line_number => $retargeting->{line_number},
                            error => join(" ", @adgroup_prefix_name, iget('Строка %s:', $retargeting->{line_number}), $_),
                        } } @{$vr_retargeting->get_error_descriptions};
                    }
                }
            }
        } else {
            if ($xls_campaign->{campaign_type} eq 'mobile_content') {
                # Это альтернативная ветка, на случай если валидация прошла

                # Простукиваем ссылки на приложения
                for my $adgroup (@{$xls_campaign->{groups}}) {

                    my @adgroup_prefix;
                    push @adgroup_prefix, iget('Группа объявлений "%s":', $adgroup->{group_name}) if ($adgroup->{group_name} // '') =~ /\S/;
                    push @adgroup_prefix, iget('Строка %s:', $adgroup->{line_number});

                    my $content_info = $adgroup->{mobile_content_info}
                                       || croak "Cannot find `mobile_content_info` in adgroup snapshot";
                    if (!$content_info->{response}) {
                        push @errors, +{
                            priority => 10,
                            line_number => $adgroup->{line_number},
                            error => join(" ", @adgroup_prefix,
                                iget('Не удалось получить информацию о приложении (%s)', $content_info->{error}->{message})
                            ),
                        };
                    }
                }
            }
        }
    }

    # Валидация единых минус-слов (пересечения)
    my $mw_validation_result = validate_group_camp_minus_words($xls_campaign, use_line_numbers => 1, extended => 1);
    push @errors, +{priority => 20, %$_} for @{$mw_validation_result->{errors}};
    push @warnings, @{$mw_validation_result->{warnings}};

    return {
        errors => [
            map { $_->{error} . ($_->{error} !~ /\.$/ ? '.' : '') }
            sort { $a->{priority} <=> $b->{priority} or $a->{line_number} <=> $b->{line_number} } @errors
        ],
        warnings => [
            map { $_->{warning} . ($_->{warning} !~ /\.$/ ? '.' : '') }
            sort { $a->{line_number} <=> $b->{line_number} } @warnings
        ],
    };
}

=head2 get_adgroups_store_content_hrefs($groups)
    Получает список store_content_href всех групп кампании

=cut

sub get_adgroups_store_content_hrefs {
    my ($groups, $group_filter) = @_;

    die 'Missing required parameter: group_filter' unless defined $group_filter;
    die "Invalid paramater group_filter=$group_filter" unless $group_filter eq 'all' || $group_filter eq 'without_pid';

    my @store_content_hrefs;
    foreach my $group (@$groups) {
        if ( $group_filter eq 'without_pid' && $group->{pid} ) {
            next;
        }

        if ($group->{adgroup_type} eq 'mobile_content' &&  defined $group->{store_content_href}) {
            push @store_content_hrefs, $group->{store_content_href};
        }
    }

    return \@store_content_hrefs;
}

=head2 validate_rmp_store_hrefs($adgroups_store_hrefs, $campaign_store_href)
    Проверяет store_content_href, возвращает undef или ошибку

=cut

sub validate_rmp_store_hrefs {
    my ($adgroups_store_hrefs, $campaign_store_href) = @_;

    return unless @$adgroups_store_hrefs;

    my $uniq_store_hrefs_count = scalar uniq @$adgroups_store_hrefs;
    if ($uniq_store_hrefs_count > 1) {
        return iget("В кампании для рекламы мобильных приложений во всех группах объявлений должна быть ссылка на одно и то же приложение. Для рекламы нескольких приложений создайте отдельные кампании");
    } elsif ($campaign_store_href && $adgroups_store_hrefs->[0] ne $campaign_store_href) {
        return iget("Ссылка на приложение в группе объявлений не совпадает со ссылкой в кампании");
    }
    
    return;
}

# --------------------------------------------------------------------

=head2 get_sample_camp_snapshot($host)

Sample campaign snapshot for sample xls_file
Parameters:    
    host:       direct.yandex.ru|direct.yandex.com.tr...

=cut

sub get_sample_camp_snapshot {
    my ($host) = @_;
    return {
        cid => 123456,
        contact_info => {
            country_code    => '+7',
            city_code       => '495',
            phone           => '12376554',
            ext             => '1234',

            country         => iget('Россия'),
            city            => iget('Москва'),
            name            => iget('ООО Организация'),
            contactperson   => '',
            worktimes       => get_worktimes_array('0#1#10#00#11#00'),
            street          => '',
            house           => '', 
            build           => '',
            apart           => '', 
            im_client       => '', 
            im_login        => '', 
            extra_message   => '', 
            contact_email   => 'login@domain.ru',
            currency        => 'YND_FIXED',
        },
        campaign_minus_words => [iget('минус фразы'), iget('на кампанию')],
        groups => [{
            geo     => "3",
            group_name => iget('Группа объявлений 1'),            
            banners => [
                {
                ad_type => 'text',
                title           => iget("Текст заголовка1. Макс. $MAX_TITLE_LENGTH символов, пробелы и знаки препинания не учитываются"),
                title_extension => iget("Текст заголовка2. Макс. $MAX_TITLE_EXTENSION_LENGTH символов, пробелы и знаки препинания не учитываются"),
                body            => iget("Текст объявления. Максимум $MAX_BODY_LENGTH символов, пробелы и знаки препинания не учитываются."),
                href            => "yandex.ru",
                display_href => iget('отображаемая-ссылка'),
                banner_type => 'desktop',
                contact_info => '+',
                sitelinks   => [
                    {
                        title   => iget("Директ"),
                        href    => "direct.yandex.ru",
                    },
                    {
                        title   => iget("Маркет"),
                        href    => "market.yandex.ru",
                    },
                    {
                        title   => iget("Карты"),
                        href    => "maps.yandex.ru",
                    },
                ],
                image => 'AAAxXxXxXxXxXxXxXxXZZZ',
                callouts => [{callout_text => iget("Уточнение")}, {callout_text => iget("Уточнение2")}],
                },
                {
                    ad_type => 'image_ad',
                    title => _get_image_ad_title({image_ad => {width => 400, height => 300}}),
                    body => q//,
                    banner_type => 'desktop',
                    href  => "yandex.ru",
                    image_ad => {
                        hash => 'BBBxXxXxXxXxXxXxXxXYYY',
                        mds_group_id => 4517,
                    },
                },
            ],
            phrases => [
                {
                    phrase  => iget('Фраза'),
                    price   => 0.3,
                    param1  => '555666',
                    param2  => iget('Значение второго параметра'),
                },
                {
                    phrase  => iget('Фраза с -минус -словами'),
                    price   => 0.3,
                }
            ],
            tags => [
                {
                    tag_name => iget('первая тестовая метка'),
                },
                {
                    tag_name => iget('вторая тестовая метка'),
                },
            ],
            minus_words => [iget('минус фразы'), iget('на группу')], 
        }, {
            geo     => "225,-3",
            group_name => iget('Группа объявлений 2'),
            banners => [{
                real_banner_type => 'text',
                title           => iget("Первый баннер группы(заголовок 1)."),
                title_extension => iget("Первый баннер группы(заголовок 2)."),
                body            => iget("Первый баннер группы(текст)."),
                banner_type     => 'desktop',
                href            => "direct.yandex.ru",
                contact_info    => '-',
            }, {
                real_banner_type => 'text',
                title           => iget("Второй баннер группы(заголовок 1)."),
                title_extension => iget("Второй баннер группы(заголовок 2)."),
                body            => iget("Второй баннер группы(текст)."),
                href            => "direct.yandex.ru",
                banner_type     => 'mobile',
                contact_info    => '-',
            }],
            phrases => [
                {
                    phrase  => iget('Фраза без минус слов'),
                    price   => 0.3,
                },
                {
                    phrase  => iget('"Точная словоформа"'),
                    price   => 0.3,
                },
            ],
            tags => [
                {
                    tag_name => iget('первая тестовая метка'),
                },
            ],
            minus_words => [],
        }]
    };
}

=head2 has_oversized_banners

    Возвращает истину (в perl'овом смысле), если хотя бы в одном баннере суммарная длина фраз

    Опции:

    limit
    client_id

=cut

sub has_oversized_banners {
    my ($banners, %O) = @_;

    die unless $banners && ref($banners) eq 'ARRAY';
    return any { Models::AdGroup::is_group_oversized($_, %O) } @$banners;
}

=head2 shrink_oversized_banner

    Оставляет в переданном баннере только фразы, не выходящие за пределы ограничения на количество фраз.
    Возвращает список "лишних" фраз.
    Принимает ссылку на хеш с данными о баннере и его фразах и именованный параметр:
        splitter_length  — длина разделителя между фразами. если не указана, полагается равной 1 символу.
        client_id
    Если передано что-то, не похожее на баннер, умирает.

    $banner = {
        [...]
        phrases => [
            [...]
            {
                [...]
                phr => 'phrase text 1',
                [...]
            }
            [...]
        ],
        [...]
    };
    @excess_phrases = shrink_oversized_banner($banner, client_id => $client_id);

=cut

sub shrink_oversized_banner {
    my ($banner, %O) = @_;

    die 'bad banner' unless $banner && ref($banner) eq 'HASH' && exists $banner->{phrases} && ref($banner->{phrases}) eq 'ARRAY';

    $banner->{phrases} = [ sort {($b->{id}||0) <=> ($a->{id}||0) } @{$banner->{phrases}} ]; # phrases with id go first

    my $first_excess_idx = Models::AdGroup::get_first_excess_phrase_idx($banner, client_id => $O{client_id});
    die 'bad banner' unless defined $first_excess_idx;
    if ( $first_excess_idx != -1 ) {
        # есть невлезающие фразы
        my @left_phrases = @{$banner->{phrases}}[ 0 .. $first_excess_idx-1 ];
        my @excess_phrases = @{$banner->{phrases}}[ $first_excess_idx .. $#{$banner->{phrases}} ];
        $banner->{phrases} = \@left_phrases;
        return \@excess_phrases;
    } else {
        return;
    }
}

=head2 split_oversized_groups

    Переразбивает фразы в группе переданного снэпшота в случае превышения ими максимального количества.
    Принимает ссылку на хеш с данными о снэпшоте и именованный параметр:
        split_strategy — стратегия разбиения слишком больших баннеров. может принимать значения:
            always_new_banner — невлезающие фразы из каждой группы выделяются в отдельную группу. используется по умолчанию.
            shared_new_banner — невлезающие фразы собираются в общую группу
    
    split_oversized_groups($split_strategy, $groups);

=cut

sub _clone_group {
    my $new_group = yclone(shift);
    delete $new_group->{pid};
    delete $_->{bid} foreach @{$new_group->{banners}};
    delete $new_group->{phrases};

    return $new_group;
}

sub split_oversized_groups {
    
    my ($strategy, $groups, %O) = @_;
    
    my $groups_count = $O{groups_count} || 0;
    my $banners_count = $O{banners_count} || 0;
    
    $strategy ||= 'always_new_banner';
    die "unknow split strategy: $strategy" unless $strategy =~ /^(always_new_banner|shared_new_banner)$/;

    my %shared_groups;
    foreach my $group (@$groups) {
        my $excess_phrases = shrink_oversized_banner($group, client_id => $O{client_id});
        next unless $excess_phrases && @$excess_phrases;
        # часть фраз не влезла
        if ($strategy eq 'always_new_banner') {
            # При такой стратегии выделяем не влезающие фразы из каждого баннера в отдельный новый баннер,
            # который добавляем в конец списка. Если фраз больше, чем помещается в один баннер, он будет
            # в дальнейшем разбит на несколько общим алгоритмом
            my $new_group = _clone_group($group);
            $banners_count += scalar @{$new_group->{banners}};
            $groups_count++;
            $new_group->{phrases} = $excess_phrases;
            push @$groups, $new_group;
        } else {
            # А при такой стратегии пихаем все не влезающие фразы из всех баннеров в один новый баннер, 
            # который добавляем в конец списка. Если и в этом баннере будет переполнение, он будет разбит
            # на новые баннеры по аналогии со стратегией always_new_banner
            my $key = XLSParse::_get_group_key($group);
            unless (exists $shared_groups{$key}) {
                my $new_group = _clone_group($group);
                $new_group->{phrases} = [];
                $shared_groups{$key} = $new_group;
            }
            push @{$shared_groups{$key}->{phrases}}, @$excess_phrases;
        }
    }
    
    if (my @shared = values %shared_groups) {
        my ($shared_groups_cnt, $shared_banners_cnt) = split_oversized_groups(always_new_banner => \@shared, client_id => $O{client_id});
        $groups_count += $shared_groups_cnt;
        $banners_count += $shared_banners_cnt;
        push @$groups, @shared;
    }
    
    return ($groups_count, $banners_count);
}

=head2 check_domain_availability

    Проверка доменов на доступность
    Параметры: 
        $new_groups - список новых групп, те, которые хочется добавить.
        $old_groups - список старых групп, те, которые ранее находились в кампании 

=cut

sub check_domain_availability {
    
    my ($new_groups, $old_groups) = @_;
    
    my %extra_fields = (map {$_ => 1} qw/line_number/);
    my $new_urls = Models::AdGroup::get_urls($new_groups, with_extra_fields => [keys %extra_fields]);
    my $old_urls = Models::AdGroup::get_urls($old_groups, with_extra_fields => [keys %extra_fields]);

    # Удаляем ссылки, которые уже есть в этой кампании в БД или были на момент выгрузки файла
    # удаляем только если совпадают параметры (Param1, Param2) 1-й фразы баннера
    my @params_names = map {lc} @Models::Phrase::BIDS_HREF_PARAMS;
    my @to_delete = grep {
        my $href = $_;
        $old_urls->{$href}
            && all {str($old_urls->{$href}->{$_}) eq $new_urls->{$href}->{$_}} @params_names 
    } keys %$new_urls;
    delete @{$new_urls}{@to_delete} if @to_delete;
    
    my (@errors, %checked);
    my $domain_error_cnt = 0;
    while (my ($href, $params) = each %$new_urls) {
        
        my $domain = get_host($href);
        next if exists $checked{$domain};
        
        # Подставляем в ссылку параметры из первой фразы баннера
        while (my ($name, $value) = each %$params) {
            next if $extra_fields{name};
            $href =~ s/{\Q$name\E}/$value/gi if defined $value;
        }
        
        my ($res, $tmp);
        my ($redirect, $domain_redir) = RedirectCheckQueue::check_domain($domain);
        if ( $redirect ) {
            $res = 1;
            $tmp = $redirect;
            $checked{$domain} = [0, $href, $tmp, $domain_redir, undef]; # dont feed_dict
        } else {
            my $result_check = get_url_domain($href, {check_for_http_status => 1});
            ($res, $tmp, $domain_redir) = @{$result_check} {qw/res msg domain_redir/};
            $checked{$domain} = [$res, $href, $tmp, $domain_redir, $result_check->{redirect_result_href}];
        }
        unless ($res) {
            my $error = iget("Адрес недоступен: %s", $href);
            $error = iget('Строка %s: ', $params->{line_number}).$error if defined $params->{line_number};

            push @errors, $error;
            last if ++$domain_error_cnt > 5;
        }
    }
    
    RedirectCheckQueue::feed_dict( map { [ $_->[1], $_->[2], $_->[3], $_->[4] ] } grep {$_->[0]} values %checked );
    
    return \@errors;
}

=head2 validate_xls_geo

    Параметры:
        options - хеш с различными именованными параметрами:
                new_common_geo - регион, который надо присвоить всем.
                has_empty_geo  - флаг, был ли найден баннер без региона
                geo_errors     - флаг, были ли найдены ошибки в регионах при парсинге экселя.
                geo            - регион из экселя

=cut

sub validate_xls_geo {
    
    my %options = @_;
    
    my $geo = $options{geo};
    if (defined $options{new_common_geo}
        && $options{new_common_geo} =~ /^(\-?\d+)(\s*,\s*\-?\d+)*$/) {
        $geo = $options{new_common_geo};       
    }
    return if defined $geo; 
    
    if ($options{has_empty_geo}) {
        return iget("Укажите регион для всех объявлений в файле или выберите ЕДИНЫЙ регион");
    } elsif (@{$options{geo_errors}}) {
        return iget("Указанные в файле регионы содержат ошибки, укажите правильные значения или выберите ЕДИНЫЙ регион");
    }
}

sub set_geo_to_groups {
    
    my ($campaign, $geo, $common_geo) = @_;
    
    $geo = $common_geo if defined $common_geo && $common_geo =~ /^(\-?\d+)(\s*,\s*\-?\d+)*$/;
    $_->{geo} = $geo // ''  foreach @{$campaign->{groups}};
    
    return;
}


=head2 _normalize_tag_name

Приводим один тег к виду для сравнения - без пробелов и в нижнем регистре.

    my $normalized_one_tag = _normalize_tag_name($one_tag);

=cut

sub _normalize_tag_name {
    return lc(Tag::trim($_[0]));
}


sub _get_group_line_number {
    return (min map {$_->{line_number}} @{$_[0]->{banners}});
}


# --------------------------------------------------------------------

=head2 _split_by_minus_words

    разбиваем фразу на саму фразу и минус-слова (слова внутри сортируем)
    используем для сравнения по минус-словам

    my ($plus_phrase, $minus_phrase) = _split_by_minus_words($phrase);

=cut

sub _split_by_minus_words($) {
    my $phrase = shift;

    return (
        join(",", sort grep {! m/^-/} split(/\s+/, $phrase)),
        join(",", sort grep {m/^-/}   split(/\s+/, $phrase))
    );
}

# --------------------------------------------------------------------

=head2 _serialize_banner

    сериализуем баннер для сравнения
    my $banner_str = _serialize_banner($banner);

    опции:
        skip_contact_info - не учитывать контактную информацию
        adgroup_type - тип группы
        use_callouts_for_calc_units => 0|1 - при подсчете баллов учитывать ли уточнения (временно для клиентов - нет)

=cut

sub _serialize_banner {
    my ($banner, %options) = @_;

    my @fields = qw/bid banner_type title title_extension body href display_href permalink/;
    unless ($options{skip_contact_info}) {
        push @fields, qw/contact_info/;
    }
    if ($banner->{ad_type} eq 'image_ad') {
        @fields = qw/bid banner_type href/;
    }

    my @banner_freeze;
    for my $field (@fields) {
        push @banner_freeze, "$field: " . str($banner->{$field})
    }

    # чтобы не считать разными баннеры, выгруженные на одной бете и загружаемые на другой
    my $image_url = $banner->{image_url} // '';
    $image_url =~ s!(https?://)\d{4,5}\.beta\d\.!$1!;
    push @banner_freeze, "image_url: $image_url";

    my $turbolanding_id = ref $banner->{turbolanding} ? $banner->{turbolanding}->{id} : $banner->{turbolanding_id};
    push @banner_freeze, "turbolanding_id: $turbolanding_id" if $turbolanding_id;

    if ($banner->{ad_type} ne 'image_ad') {
        # Сайтлинки
        push @banner_freeze, "sitelinks: " . Sitelinks::serialize($banner->{sitelinks});

        if ($options{use_callouts_for_calc_units}) {
            push @banner_freeze, _serialize_callouts($banner->{callouts});
        }

        # Возврастные метки
        $banner->{flags} = BannerFlags::get_banner_flags_as_hash($banner->{flags}) if (ref $banner->{flags} ne 'HASH');
        push @banner_freeze, "age: " . BannerFlags::normalize_flag_value('age', $banner->{flags}->{age});

        if ($options{adgroup_type} eq 'mobile_content') {
            push @banner_freeze, "reflected_attrs: " . join(',', sort @{$banner->{reflected_attrs} || []});
            # `primary_action` пока нет
            # push @banner_freeze, "primary_action: " . str($banner->{primary_action});
        }
        else {
            push @banner_freeze, "creative_id: " . ($banner->{video_resources}->{id}//'');
        }
    }

    return join("/", @banner_freeze);
}

=head2 _serialize_group

    Сериализация группы для сравнения(без banners/phrases)

=cut

sub _serialize_group {
    
    my $group = shift;
    
    my @params;
    for my $field (qw/group_name geo tags minus_words/) {
        next unless exists $group->{$field};
        if ($field eq 'tags') {
            push @params, "tags: " . join ',', sort map {_normalize_tag_name($_->{tag_name})} @{$group->{tags}};
        } elsif ($field eq 'minus_words') {
            push @params, "minus_words: ". (MinusWordsTools::minus_words_array2str($group->{$field}) || '');
        } else {
            push @params, "$field: " . str($group->{$field});
        }
    }

    if ($group->{adgroup_type} eq 'mobile_content') {
        for my $field (qw/store_content_href device_type_targeting network_targeting min_os_version/) {
            next unless exists $group->{$field};
            if (ref($group->{$field}) eq 'ARRAY') {
                push @params, "$field: " . join(',', sort @{$group->{$field}});
            } else {
                push @params, "$field: " . str($group->{$field});
            }
        }
    }

    return join '/', @params;
}

# --------------------------------------------------------------------

=head2 _serialize_phrase

    сериализуем фразу для сравнения
    my $phrase_str = _serialize_phrase($phrase);

=cut

sub _serialize_phrase {
    my $phrase = shift;
    my $options = shift;

    my @phrase_freeze;
    for my $field (qw/id phr price price_context param1 param2/) {
        # если цена не задана - в xls это пустая ячейка
        my $value = '';
        
        if ($field eq 'price_context'
            && $options->{strategy} && $options->{strategy} ne 'different_places'
            && !(defined $phrase->{rank} && $phrase->{rank} == 0)) {
            
            $value = '(price_context_for_non_different_places)';
        } elsif ($field =~ /^(price|price_context)$/) {
            $value = str($phrase->{$field}) eq '' ? '0.00' : sprintf("%0.2f", $phrase->{$field});
        } else {
            # Поддерживаем старый формат выгрузки, где ретаргетинг побозначался в виде retargeting:
            # Не считаем баллы за это, так как это не вина клиента.2
            if ($field =~ /^phr$/ && $value =~ /^$Retargeting::CONDITION_XLS_PREFIX_OLD/) {
                $value =~ s/^($Retargeting::CONDITION_XLS_PREFIX_OLD)/$Retargeting::CONDITION_XLS_PREFIX/;
            }
            $value = str($phrase->{$field});
        }

        push @phrase_freeze, "$field: $value";
    }

    return join("/", @phrase_freeze);
}

# --------------------------------------------------------------------

=head2 _serialize_contact_info

    сериализуем контактную информацию для сравнения
    my $contact_info_str = _serialize_contact_info($contact_info);

=cut

sub _serialize_contact_info {
    my $contact_info = shift;

    my @contact_info_freeze;
    for my $field (qw/apart build city city_code contact_email contactperson country country_code ext extra_message house im_client im_login name ogrn phone street worktime/) {
        push @contact_info_freeze, "$field: " . str($contact_info->{$field});
    }

    return join("/", @contact_info_freeze);
}

# --------------------------------------------------------------------

=head2 _serialize_callouts

    сериализуем уточнения для сравнения
    my $callouts_str = _serialize_callouts($callouts);

=cut

sub _serialize_callouts($) {
    my ($callouts) = @_;

    my $callouts_str = "callouts: " . (ref($callouts) eq 'ARRAY' ? join(",", sort map {$_->{callout_text}} @$callouts) : "");
    return $callouts_str;
}

sub preimport_xls {

    my ($format, $xls, $options, $client_id) = @_;

    $options->{ClientID} ||= $client_id;

    $options->{is_tycoon_organizations_feature_enabled} = Client::ClientFeatures::has_tycoon_organizations_feature($options->{ClientID});

    my $snapshot;
    if ($format eq 'csv') {
        $snapshot = csv2camp_snapshot($format => $xls, $options);
    } else {
        # $format == 'xls' || 'xlsx'
        $snapshot = xls2camp_snapshot($format => $xls, $options);
    }

    unless (@{$snapshot->{parse_errors}}) {
        
        if ($format ne 'csv') {
            my $groups = $snapshot->{xls_camp}->{groups};
            if ($groups) {
                $snapshot->{has_empty_geo} = scalar grep {$_->{region} =~ m/^\s*$/} @$groups;
                $snapshot->{has_oversized_banners} = has_oversized_banners($groups, client_id => $client_id) ? 1 : 0;
            } else {
                $snapshot->{has_empty_geo} = 0;
                $snapshot->{has_oversized_banners} = 0;
            }
            $snapshot->{split_strategy} = $options->{split_strategy} || 'always_new_banner';
        }
        
        if (defined $options->{geo} && $options->{geo} =~ /^(\-?\d+)(\s*,\s*\-?\d+)*$/) {
            $snapshot->{geo} = $options->{geo};
        }

        my @errors;
        if ($format ne 'csv') {
            _preprocess_snapshot(snapshot => $snapshot, client_id => $client_id);
            if ($snapshot->{errors} && @{$snapshot->{errors}}) {
                push @errors, @{$snapshot->{errors}};
            }
            my $x;
            if ($snapshot->{xls_camp}->{mediaType} eq 'mobile_content') {
                # Новая валидация на основе Smart
                $x = _validate_and_fix_xls_campaign_snapshot($snapshot->{xls_camp}, $options->{uid}, $client_id);
            } else {
                # Старая валидация
                $x = _validate_group_camp_snapshot($snapshot->{xls_camp}, $client_id,
                    skip_max_phrases_length                 => 1,
                    ignore_existing_camp_tags               => 1,
                    is_tycoon_organizations_feature_enabled => $options->{is_tycoon_organizations_feature_enabled}
                );
            }
            push @errors, @{$x->{errors}};
            push @{$snapshot->{parse_warnings}}, @{$x->{warnings}};
        }
        
        # текст в этих iget'ах должен совпадать с таковым в validate_banner. и это криво.
        # TODO: возвращать из validate_banner ошибки не только в виде текстов, но и в виде хеша
        my $href_error_text = iget('Не введена ссылка на сайт');
        ($snapshot->{href_errors}, $snapshot->{new_errors}) = part { m/$href_error_text/i ? 0 : 1 } @errors;
        $snapshot->{href_errors} ||= [];
        $snapshot->{new_errors} ||= [];        
    }
    return $snapshot;
}

sub _preprocess_snapshot
{
    my %params = @_;
    if( !$params{client_id} || !$params{snapshot} ){
        croak "not enouth parameters";
    }

    my $snapshot = $params{snapshot};
    # Поддержка транслокальности
    for my $adgroup (@{$snapshot->{xls_camp}->{groups}}) {
        if ($adgroup->{geo}) {
            $adgroup->{geo} = GeoTools::modify_translocal_region_before_save($adgroup->{geo}, {ClientID => $params{client_id}});
            $adgroup->{geo_is_modified_before_save} = 1;
        }
    }

    $snapshot->{xls_camp}->{all_retargeting_conditions} = Retargeting::get_retargeting_conditions(ClientID => $params{client_id});
    _process_retargetings(retargetings => $snapshot->{xls_camp}->{all_retargeting_conditions}, groups => $snapshot->{xls_camp}->{groups});
}

sub _process_retargetings
{
    my %params = @_;
    if( !$params{retargetings} || !$params{groups} ){
        croak "not enouth parameters";
    }

    my $retargetings = $params{retargetings};

    for my $group (@{$params{groups}}){
        for my $r (@{$group->{retargetings}}){
            if(!$r->{ret_cond_id} || !$retargetings->{$r->{ret_cond_id}}){
                my $ret_cond_id = Retargeting::guess_condition_by_name( all_retargeting_conditions => $retargetings, condition_name => $r->{condition_name} ); 
                $r->{ret_cond_id} = $ret_cond_id;
            }
        }


        my %exist;
        my $uniq_retargetings;
        for my $r (@{$group->{retargetings}}){
            if(!$exist{$r->{ret_cond_id}}){
                push(@$uniq_retargetings, $r);
                $exist{$r->{ret_cond_id}} = 1;
            }
        }
        $group->{retargetings} = $uniq_retargetings;
    }
}

sub _clear_ids_in_groups {
    
    my ($groups, %options) = @_;
    
    foreach my $g (@$groups) {
        delete $_->{bid} foreach @{$g->{banners}};
        delete $_->{id} foreach @{$g->{phrases}};
        delete $_->{bid_id} foreach @{$g->{relevance_match}};
        delete $_->{ret_id} foreach @{$g->{target_interests} // []};
        delete $g->{pid};
        # В случае, если файл сохраняется в новую кампанию или добавляется в старую, то в случе
        # автогенерации имени (на этапе превью), их удаляем, чтобы сгенерировать более правильные,
        # с обновленными id
        delete $g->{group_name} if $g->{is_name_auto_generated} && !$options{dont_clear_group_name};
    }
}

sub load_xls_to_new_camp {
    
    my %options = @_;
    
    my ($new_camp, $user, $login_rights, $UID, $uid, $agency_uid, $context, $rbac, $auto_price_params, $xls_id, $cid, $strategy_id) = @options{
        qw/new_camp user   login_rights   UID   uid   agency_uid   context   rbac   auto_price_params   xls_id   cid strategy_id/
    }; 
    my $client_chief_uid = $user->{uid};
    my $client_id = $user->{ClientID};

    $new_camp->{xls_id} = $xls_id;
    
    # первую кампанию через XLS штатными средствами создать нельзя, поэтому считаем, что и клиент в Балансе в этот момент уже существует
    my $client_currencies = get_client_currencies($client_id);
    
    my %camp_options = (
        client_chief_uid => $client_chief_uid,
        ClientID         => $client_id,
        type             => $options{camp_mediaType},
        client_fio       => $user->{fio},
        client_email     => $user->{email},
        currency         => $client_currencies->{work_currency},
        domain           => $context->site_host,
        cid              => $cid,
        strategy_id      => $strategy_id,
    );

    my $client_perminfo = Rbac::get_perminfo(uid => $client_chief_uid);

    # Если клиент агентский - по умолчанию создаем кампанию под тем же агенством
    if ($client_perminfo->{agency_client_id}) {
        $camp_options{agency_id} = $client_perminfo->{agency_client_id}; 
        $camp_options{agency_uid} = $client_perminfo->{agency_uid} if $client_perminfo->{agency_uid};
    }

    if ($login_rights->{agency_control} || defined $agency_uid) {
        $camp_options{agency_uid} = Rbac::get_client_agency_uid($client_id, $agency_uid);
    }
    elsif ($client_perminfo->{primary_manager_set_by_idm}){
        #Если клиенту назначен основной менеджер - кампания должна завестись под ним
        $camp_options{manager_uid} = $client_perminfo->{primary_manager_uid};
    }
    # этот блок встречается трижды: здесь, в DoCmd, в API. Везде аккуратно переделать вызовы create_empty_camp
    elsif ($login_rights->{manager_control} && ! defined $agency_uid) {
        #Проставляем manager_uid только если к создаваемой кампании менеджер имеет доступ не по тирной схеме
        $camp_options{manager_uid} = $UID if (!RBACDirect::has_idm_access_to_client($UID, $client_id));
    }

    my $error = can_create_camp( %camp_options );
    if ($error ) {
        if ($error eq 'NOT_ENOUGH_FREEDOM') {
            $error = [iget("нет разрешения на обслуживание этого клиента")];
        }
        else {
            $error = [iget("Сохранить кампанию не удалось"), undef, "can_create_camp error: $error (in cmd_saveNewCamp)"];
        }
    }
    return undef, undef, $error if $error;

    create_empty_camp(%camp_options);

    #Если мы проставили кампании manager_uid - вызовем campaign_manager_changed
    if ($camp_options{manager_uid}) {
        campaign_manager_changed($rbac, $UID, $cid, $UID);
    } 

    my $camp_values = {
        cid                 => $cid,
        mediaType           => $new_camp->{mediaType} || 'text',
        uid                 => $client_chief_uid,
        name                => $new_camp->{camp_name} || iget("Новая"),
        start_time          => unix2mysql(ts_round_day(time)),
        finish_time         => undef,
        sendWarn            => 'Yes',
        offlineStatNotice   => 'Yes',
        statusModerate      => $new_camp->{statusModerate} || 'New',
        email_notifications => {
            paused_by_day_budget => 1,
            feed_status_change   => 1,
        },
        ($new_camp->{campaign_minus_words}
            ? (campaign_minus_words => $new_camp->{campaign_minus_words})
            : ()),
        (map {$_ => $user->{$_}} qw/fio email/),
        opts                => { enable_cpc_hold => 1 },
        attribution_model   => $new_camp->{attribution_model} // get_attribution_model_default(),
        mobile_app_id       => $new_camp->{mobile_app_id},
        strategy            => '',
        strategy_id         => $strategy_id,
    };

    if ($camp_values->{mediaType} eq 'text') {
        hash_merge $camp_values, { broad_match_flag => 1, broad_match_limit => Campaign::get_broad_match_default(), };
    }

    my $old_campaign = save_camp($context, $camp_values, $client_chief_uid, is_new_camp => 1);
    update_camp_auto_optimization($cid, $client_chief_uid, 'Yes', dont_send_notification => 1);

    my $client_data = get_client_data($client_id, [qw/common_metrika_counters/]);
    if ($client_data->{common_metrika_counters} && $camp_values->{mediaType} !~ /^(mcb|mobile_content)$/) {
        camp_save_metrika_counters($cid, $old_campaign->{strategy_id}, $client_data->{common_metrika_counters});
    }
    
    # создаём новую кампанию - игнорируем id баннеров и фраз
    _clear_ids_in_groups($new_camp->{groups});
    # dont_clean => 1, так как в новой кампании нечего чистить
    my ($campaign, $errors) = XLSCampaign::apply_camp_group_snapshot(
        $new_camp, $cid, $client_chief_uid,
        UID => $UID, ClientID => $client_id,
        dont_clean => 1,
        ignore_image => $options{ignore_image},
        i_know_href_params => 1,
        login_rights => $options{login_rights},
    );
    return (undef, undef, $errors) unless $campaign;

    my $strategy = Campaign::get_default_strategy_by_mediaType(
        $campaign->{mediaType},
        is_default_avg_cpa_feature_enabled => Client::ClientFeatures::has_default_avg_cpa_feature($client_id)
    );
    Campaign::mark_strategy_change ($cid, $client_chief_uid, $strategy, $strategy, $UID);

    # подключаем общий счет по умолчанию
    if (Wallet::need_enable_wallet(
            cid => $cid,
            client_id => $client_id,
    )) {
        my $wallet_result = Wallet::enable_wallet($context, $client_currencies->{work_currency}, $agency_uid
            , allow_wallet_before_first_camp => 1
            , dont_check_onoff_time => 1
        );

        if ($wallet_result->{error} && ! $wallet_result->{agency_dont_allow_wallet} && $wallet_result->{code} != 519) {
            # выдаем ошибку, только если не по причине запрета счета для клиента агентства
            # и не по дате включения счета
            my $error = [iget("Не удалось подключить общий счет")];
            return undef, undef, $error;
        }
    }

    # кампания готова и её можно показывать, скидываем statusEmpty
    do_update_table(PPC(cid => $cid), 'campaigns', {statusEmpty => 'No', statusBsSynced => 'No'}, where => {cid => SHARD_IDS});

    # добавляем в очередь проставление цены
    Common::add_camp_auto_price_queue($cid, $UID, $auto_price_params) if $auto_price_params;
    # отправляем уведомления всем менеджерам клиента
    my $this_client_have_managers = rbac_get_managers_of_client($rbac, $client_chief_uid);
    if ($UID == $uid         # Отправлять уведомления только когда клиент сам создаёт кампанию
        && $login_rights->{role} eq 'client'
        && ref($this_client_have_managers) eq 'ARRAY'
        && @$this_client_have_managers) {
        
        my $mailvars_ptn = {
            cid           => $cid,
            campaign_id   => $cid,
            camp_name     => $camp_values->{name},
            campaign_name => $camp_values->{name},
            campaign_type => $options{camp_mediaType},
            client_id     => $client_id,
            is_serviced   => 0,
            (map {("client_${_}" => $user->{$_})} qw/login fio email phone uid/),
        };
        
        my $managers = get_all_sql(PPC(uid => $this_client_have_managers), ["select uid, email, max(FIO) as fio
                                          from users",
                                          where => {uid => SHARD_IDS},
                                          "group by email"]);
        $managers = overshard(group => 'email', $managers);

        my $sql = "select count(*) from campaigns where uid = ? and ManagerUID = ? and statusEmpty = 'No'";
        foreach my $manager (@$managers) {
            my $qauntity = get_one_field_sql(PPC(uid => $client_chief_uid), $sql, $client_chief_uid, $manager->{uid});
            next unless $qauntity && rbac_who_is($rbac, $manager->{uid}) eq 'manager';
            
            add_notification($rbac, 'new_camp_info', {
                manager_fio => $manager->{fio},
                manager_uid => $manager->{uid},
                %$mailvars_ptn
            });
        }
    }
    
    return $campaign, $cid;
}


sub load_xls_to_other_camp {

    my %options = @_;
    my ($new_camp, $user, $UID, $xls_id) = @options{qw/new_camp user UID xls_id/};

    $new_camp->{xls_id} = $xls_id;

    _clear_ids_in_groups($new_camp->{groups});
    return XLSCampaign::apply_camp_group_snapshot(
        $new_camp, $options{to_campaign}, $user->{uid},
        UID => $UID, ClientID => $user->{ClientID},
        dont_clean => 1,
        ignore_image => $options{ignore_image},
        login_rights => $options{login_rights},
    );
}

sub load_xls_to_old_camp {
    
    my %options = @_;
    my ($new_camp, $user, $UID, $stop_banners, $xls_id) = @options{qw/new_camp user UID stop_banners xls_id/};

    $new_camp->{xls_id} = $xls_id;

    Models::AdGroup::stop_groups(
        get_pids(bid => $stop_banners),
        {
            uid => $user->{uid},
            bid => $stop_banners
        }
    ) if @$stop_banners;
    
    return XLSCampaign::apply_camp_group_snapshot(
        $new_camp, $options{to_campaign}, $user->{uid},
        UID => $UID, ClientID => $user->{ClientID},
        merge_mode => 1,
        ignore_image => $options{ignore_image},
        login_rights => $options{login_rights},
    );
}

=head2 add_xls_image_processing_info

    добавляем в список загруженных xls-файлов информацию о статусе загрузки изображений

=cut

sub add_xls_image_processing_info {
    my $imported_xls_list = shift;
    my $client_id = shift;
    my $queue = Yandex::DBQueue->new(PPC(ClientID => $client_id), 'banner_images');
    # старая схема - bid и xls_id связаны через таблицу banner_images_process_queue_bid
    my $job_ids = get_hash_sql(PPC(ClientID => $client_id), [
        "select job_id, xls_id from banner_images_process_queue_bid",
        where => { xls_id => [ map { $_->{id} } @$imported_xls_list ] }
    ]);
    my $xls_img_data = {};
    my $jobs = keys %$job_ids ? $queue->find_jobs(job_id => [ keys $job_ids ]) : [];
    my %jobs_by_id = map { $_->job_id => $_ } @$jobs;
    # вытаскиваем задания, где нет bid (image_ad)
    my $all_jobs = $queue->find_jobs(ClientID => $client_id, status__not_in => [qw/Finished Revoked/]);
    my @imagead_jobs = grep { !$jobs_by_id{$_->job_id} && $_->args->{xls_id} && $_->args->{ad_type} eq 'image_ad' } @$all_jobs;
    for my $job (@$jobs, @imagead_jobs) {
        my $jid = $job->job_id;
        unless ($job_ids->{$jid}) {
            $job_ids->{$jid} = $job->args->{xls_id};
        }
        if ($job->status =~ /New|Grabbed/) {
            $xls_img_data->{$job_ids->{$jid}}->{process_qty}++;
        } elsif ($job->status eq 'Failed') {
            $xls_img_data->{$job_ids->{$jid}}->{failed_qty}++;
        }
    }
    foreach my $xls (@$imported_xls_list) {
        my $img_data = $xls_img_data->{$xls->{id}} || {};
        if ($img_data->{process_qty}) {
            $xls->{status} = 'process';
        } elsif ($img_data->{failed_qty}) {
            $xls->{status} = 'fail';
        } else {
            $xls->{status} = 'success';
        }
    }
}

=head2 are_uniform_banners($camp)

    Содержатся ли в группах объявлений кампании только мобильные и/или только десктопные баннеры.

    ($only_mobiles, $only_desktops) = are_uniform_banners($camp)
    
    $only_mobiles == 1 - есть группы состоящие только из мобильных баннеров
    $only_desktops == 1 - есть группы состоящие только из десктопных баннеров

=cut

sub are_uniform_banners {

    my $camp = shift;
    
    my ($only_mobiles, $only_desktops) = (0, 0);
    foreach my $group (@{$camp->{groups}}) {

        my %types = count_by { $_->{banner_type} } @{$group->{banners}};
        next unless keys %types == 1;
        
        $only_mobiles ||= exists $types{mobile}; 
        $only_desktops ||= exists $types{desktop};
        last if $only_mobiles && $only_desktops;
    } 
    return ($only_mobiles, $only_desktops);
}

1;
