use strict;
use warnings;

package XLSCampaign;

use List::MoreUtils qw/any pairwise uniq part/;
use List::Util qw/first/; 
use Yandex::HashUtils;
use Yandex::MyGoodWords;
use Yandex::ScalarUtils;
use Yandex::DBTools;

use Settings;
use Models::AdGroup;
use Direct::AdGroups2::Smart::XLS;
use Direct::XLS::Validation;
use Direct::Bids::BidRelevanceMatch qw//;
use Direct::VideoAdditions;
use HashingTools;
use Client qw/get_client_data/;
use Campaign;
use Campaign::Types;
use Lang::Unglue;
use Common;
use Phrase;
use PhrasePrice;
use BannerImages;
use BannerImages::Queue;
use MinusWords;
use VCards;
use Retargeting;

sub apply_camp_group_snapshot {
    
    my ($xls_camp, $cid, $uid, %options) = @_;

    my $campaign = get_camp_info($cid, $uid, with_strategy => 1);
    my $currency = $campaign->{currency};
    $campaign->{contacts} = $xls_camp->{contact_info}; 

    # Сохраним единые минус-слова на кампанию
    save_campaign_minus_words($cid, $xls_camp->{campaign_minus_words}) if exists $xls_camp->{campaign_minus_words};

    my $groups_to_delete;
    unless ($options{dont_clean}) {
        # условие из Common::delete_banners()
        my $old_banners = get_all_sql(PPC(uid => $uid), [
            "select b.bid, b.pid
             from banners b
               join phrases ph ON b.pid = ph.pid
               join campaigns c ON c.cid = ph.cid
             where
                not (
                   BannerID != 0
                   or (b.statusModerate = 'Yes' and ph.statusModerate = 'Yes'
                       or b.statusPostModerate in ('Yes', 'Rejected')
                       or ph.statusPostModerate in ('Yes', 'Rejected') and b.statusPostModerate != 'No'
                      ) and c.sum > 0
               ) AND ",  {
                   'c.cid' => $campaign->{cid},
                   'c.uid' => $campaign->{uid},
                   'b.bid__not_in' => [grep {$_} map {map {$_->{bid}} @{$_->{banners}}} @{$xls_camp->{groups}}]  
               }
        ]);
        
        if (@$old_banners) {
            $groups_to_delete = {};
            foreach (@$old_banners) {
                $groups_to_delete->{$_->{pid}} = {banners => []} unless exists $groups_to_delete->{$_->{pid}};
                push @{$groups_to_delete->{$_->{pid}}->{banners}}, {bid => $_->{bid}};    
            }
        }
    }
    
    # сначала сохраним phraseIdHistory для всех фраз, т.к. в дальнейшем фразы могут удалиться
    my %old_norm_phrase2bid_ids = map {$_->{norm_phrase} => $_->{id}} grep {$_->{id}} map {@{$_->{phrases}}} @{$xls_camp->{groups}};

    my @prepared_groups;
    my %params = (%options, skipped_bids => $options{skipped_bids}, old_norm_phrase2bid_ids => \%old_norm_phrase2bid_ids);
    foreach my $group (@{$xls_camp->{groups}}) {
        my $need_apply_group = !$options{merge_mode} || $group->{_added} || $group->{_has_changes_in_phrases} || $group->{_has_changes_in_retargetings} || $group->{_has_changes_in_callouts};

        if ($need_apply_group) {
            foreach my $phrase (@{$group->{phrases}}) {
                $phrase->{phrase} = $phrase->{phr};
                $phrase->{md5} = md5_hex_utf8(Yandex::MyGoodWords::norm_words($phrase->{phr}));
            }
            # unglue_phrases require key "Phrases"
            $group->{Phrases} = $group->{phrases};
            unglue_phrases([$group]);
            delete $group->{Phrases};
            foreach my $phrase (@{$group->{phrases}}) {
                next if str($phrase->{phrase_unglued_suffix}) eq '';
                $phrase->{phr} .= $phrase->{phrase_unglued_suffix};
                $phrase->{_unglued} = 1;
                $group->{_has_unglued_phrases} = 1;
            }
        }

        if ($need_apply_group || $group->{_has_unglued_phrases} || $group->{_changed}
            || any {$_->{_changed} || $_->{_added}} @{$group->{banners}}) {
            push @prepared_groups, prepare_group_snapshot($campaign, $group, %params);
        }
    }
    my $groups = [];
    my @image_urls;
    my (@regular_groups, @deffered_groups);
    if (@prepared_groups) {
        my $need_image_download = 0;
        foreach my $group (@prepared_groups){
            # графические объявления, для которых нужно скачать картинку, создадим после скачивания картинки (в ppcProcessImageQueue.pl)
            my ($regular_banners, $deffered_banners) = part { ($_->{ad_type} ne 'image_ad' || $_->{image_skip_download} ) ? 0 : 1 } @{$group->{banners}};
            $_ //= [] foreach ($regular_banners, $deffered_banners);
            $need_image_download ||= 1 if @$deffered_banners;

            push @$_, {%$group} foreach (\@deffered_groups, \@regular_groups);
            $regular_groups[-1]->{banners}  = $regular_banners;
            $deffered_groups[-1]->{banners} = $deffered_banners;
        }

        my $smart = Direct::AdGroups2::Smart::XLS->from_user_data(
            $campaign->{uid},
            $campaign->{cid},
            \@regular_groups,
            (
                operator_uid => $options{UID},
            )
        );
        if ($smart->is_valid) {
            #Если у нас есть объявления, создание которых мы отложили до загрузки картинки - прогоним через смарт и их, чтобы выполнились проверки
            if ($need_image_download) {
                my $fake_smart =  Direct::AdGroups2::Smart::XLS->from_user_data(
                    $campaign->{uid},
                    $campaign->{cid},
                    \@deffered_groups,
                    (
                        operator_uid => $options{UID},
                        mk_fake_canvas_creatives => 1,
                    ),
                );
                
                return (undef, _extract_errors_from_smart($fake_smart, \@deffered_groups)) unless $fake_smart->is_valid;
            }
            
            $groups = $smart->apply();
 
            pairwise {
                my ($prepare_group, $applied_group) = ($a, $b);
                my $idx = 0;
                for my $banner (@{$prepare_group->{banners}}) {
                    my $applied_banner;
                    if ($banner->{ad_type} ne 'image_ad' || $banner->{image_skip_download}) {
                        $applied_banner = $applied_group->{banners}->[$idx];
                        ++$idx;
                    }
                    if ($banner->{image_url} && !$banner->{image_skip_download}) {
                        push @image_urls, {
                                ClientID => $options{ClientID},
                                url      => $banner->{has_missed_creative} ?
                                                'creative://'.$banner->{creative}->{creative_id}
                                                : $banner->{image_url},
                                bid      => ($applied_banner && $applied_banner->{id} || $banner->{bid} || undef),
                                xls_id   => $xls_camp->{xls_id},
                                ( $banner->{ad_type} eq 'image_ad'
                                    ? (
                                        pid => $applied_group->{adgroup}->id,
                                        ad_type => 'image_ad',
                                        href => $banner->{href},
                                    )
                                    : ()
                                ),
                            };
                    }
                }
            } @prepared_groups, @{$groups};

            my $client_id = $options{ClientID};
            
            my $client_options = get_client_data($client_id, [qw/auto_video/]);

            if ($client_options->{auto_video}) {
                Direct::VideoAdditions->enqueue_auto_resources($client_id,$cid, 'set', (
                    UID => $smart->operator_uid,
                    bid => [ map { $_->id } @{$smart->new_banners} ],
                ));
            }

        } else {
            my $errors = _extract_errors_from_smart($smart, \@regular_groups);
            return (undef, $errors) if $errors && @$errors;
        }
    }

    my $imq_jobs = BannerImages::Queue::add_items(\@image_urls, UID => $options{UID});

    if ($groups_to_delete) {
        Models::AdGroup::delete_groups(
            [map {{
                cid => $campaign->{cid}, pid => $_,
                banners => $groups_to_delete->{$_}->{banners}
            }} keys %$groups_to_delete]);
    }

    return { groups => $groups, imq_jobs => $imq_jobs };
}

sub _extract_errors_from_smart {
    my ($smart, $data) = @_;
    
    return $smart->errors if @{$smart->errors};
    my $errors_and_warnings = Direct::XLS::Validation->new(validation_result => $smart->validation_result)->get_errors_and_warnings_for_xls($data);
    
    return $errors_and_warnings->{errors};    
}


sub prepare_group_snapshot {
    my ($campaign, $group, %options) = @_;

    # first available bid
    # TODO adgroup: перейти на Models::get_group()
    my %old_phrases; 
    my $is_mobile_content_camp = ($campaign->{mediaType} // '') eq 'mobile_content';
    if (my $banner = first {$_->{bid}} @{$group->{banners}}) {
        
        my $old_banner = Common::get_user_banner($campaign->{uid}, $banner->{bid}, {
            skip_bs_prices => 1, 
            ctx_from_bs => $campaign->{strategy}->{name} eq 'different_places',
            get_phrases_statuses => 1,
            adgroup_types => get_camp_supported_adgroup_types(cid => $campaign->{cid}),
        });
        %old_phrases = map {$_->{id} => $_} @{$old_banner->{phr}};
    }

    my @phrases = map { ## no critic (ProhibitComplexMappings)
        my $phrase = $_;
        my $new_phrase = hash_merge
                {phrase => $phrase->{phr}, Param1 => $_->{param1}, Param2 => $_->{param2}},
                hash_cut($phrase, qw/id md5 norm_hash norm_phrase numword is_suspended/),
                {autobudgetPriority => ($phrase->{id} &&
                                        $old_phrases{$phrase->{id}} &&
                                        $old_phrases{$phrase->{id}}->{autobudgetPriority})
                                      ? $old_phrases{$phrase->{id}}->{autobudgetPriority}
                                      : 3};

        if (!$campaign->{strategy}->{is_search_stop}
            && (!$new_phrase->{id}
                || any { ($campaign->{strategy}->{name} || $campaign->{strategy}->{search}->{name} ) eq $_ } @Campaign::MANUAL_PRICE_STRATEGIES)) {
            $new_phrase->{price} = $phrase->{price}
        }
        
        if (($campaign->{strategy}->{name} // '') eq 'different_places'
            || ($phrase->{id} && $old_phrases{$phrase->{id}}
                && defined $old_phrases{$phrase->{id}}->{rank}
                && $old_phrases{$phrase->{id}}->{rank} == 0)) {
                    
            $new_phrase->{price_context} = $phrase->{price_context}
                || phrase_price_context($new_phrase->{price}, $campaign->{ContextPriceCoef}, $campaign->{currency});
        }


        unless ($group->{pid}) {
            my $old_bid_id = defined( $new_phrase->{norm_phrase} ) ? $options{old_norm_phrase2bid_ids}->{$new_phrase->{norm_phrase}} : undef;
            $new_phrase->{ctr_source_id} = $old_bid_id if $old_bid_id;
        }

        $new_phrase;
    } @{$group->{phrases}};

    # помечаем фразы на удаление
    my %remain_id = map {$_ => 1} map {$_->{id} || ()} @phrases;
    push @phrases, map {+{id => $_, _delete => 1}} grep {!$remain_id{$_}} keys %old_phrases;

    $group->{phrases} = \@phrases;

    my $client_id = $campaign->{ClientID};
    my $contacts  = $campaign->{contacts} || {};

    foreach my $banner (@{$group->{banners}}) {

        my $old_banner = {};
        my %banner_flags_params = (
            age_label => $banner->{flags}->{age},
            old_flags => "",
            bid       => $banner->{bid},
            role      => $options{login_rights}->{role},
            camp_type => $campaign->{mediaType},
            not_empty => 1,
        );
        if ($is_mobile_content_camp && defined $banner->{flags}) {
            if (BannerFlags::validate_ad_age_label(%banner_flags_params)) {
                $banner->{hash_flags} = {age => $banner->{flags}->{age}};
            } else {
                $banner->{hash_flags} = {age => $group->{mobile_content_info}->{response}->{age_label}};
            }
        } elsif ($banner->{bid}) {
            $old_banner = Common::get_user_banner($campaign->{uid}, $banner->{bid}, {
                skip_bs_prices => 1, 
                ctx_from_bs => $campaign->{strategy}->{name} eq 'different_places',
                adgroup_types => get_camp_supported_adgroup_types(cid => $campaign->{cid}),
            });
            hash_merge($old_banner, parse_phone($old_banner->{phone}));
            # вычисляем для каких баннеров мы можем сохранять/обновлять возрастные метки.
            # Так для РМП мы можем сохранять метки для всех баннеров, тогда как для остальных кампаний только если в объявлении ранее была выставлена метка.
            $banner_flags_params{old_flags} = $old_banner->{flags};
            if (BannerFlags::validate_ad_age_label(%banner_flags_params)) {
                $banner->{hash_flags} = {age => $banner->{flags}->{age}};
            }  
        }
        $banner->{hash_flags}->{age} =~ s/\D// if defined $banner->{hash_flags}->{age};

        # метро в XLS пока не поддерживается; не затираем, если оно уже указано
        my $card = $banner->{contact_info} && $banner->{contact_info} eq '-'
                    ? {} : {%$contacts, $banner->{bid} ? (metro => $old_banner->{metro}) : ()}; 
        my @vcard_xls_fields = uniq @$VCards::VCARD_FIELDS, @VCards::PHONE_FIELDS, 'ogrn';

        if ($banner->{contact_info} && $banner->{contact_info} ne '') {
            $card = hash_cut $card, @vcard_xls_fields;
        } else {
            $card = hash_cut $old_banner, @vcard_xls_fields;
        }
        $banner->{vcard} = $card if %$card;
        
        if (!$options{ignore_images}) {
            if ($banner->{image_url}) {
                if ($banner->{ad_type} eq 'image_ad' && $banner->{creative} && $banner->{creative}->{creative_id}){
                    $banner->{$banner->{has_existing_creative} ? 'image_skip_download' : 'has_missed_creative'} = 1;
                } else {
                    my (undef, $mds_group_id, $image_hash) = get_image_hash_by_url($banner->{image_url});
                    if ($image_hash) {
                        my $image_type = Models::Banner::does_client_have_image($client_id, $image_hash); 
                        if ($image_type) {
                            if ($banner->{ad_type} eq 'image_ad' && $mds_group_id) {
                                $banner->{image_skip_download} = 1;
                                $banner->{image_ad} = {
                                    hash => $image_hash
                                };
                            } elsif ($banner->{ad_type} ne 'image_ad') {
                                $banner->{image_skip_download} = 1;
                                $banner->{image} = $image_hash;
                            }
                        }
                    }
                }
            } else {
                # принудительно удалить картинку
                $banner->{image} = undef;
            }
        }
    }

    $group->{currency} = $campaign->{currency};
    if ($options{merge_mode} && $group->{pid}) {
        my $group_retargeting = Retargeting::get_group_retargeting(pid => $group->{pid});
        my %ret_cond_id2ret_id;
        if ($group_retargeting->{$group->{pid}}) {
            %ret_cond_id2ret_id = map {$_->{ret_cond_id} => $_->{ret_id}} @{$group_retargeting->{$group->{pid}}};
        }
        for my $retargeting (@{$group->{retargetings}}) {
            next unless ($ret_cond_id2ret_id{$retargeting->{ret_cond_id}});
            $retargeting->{ret_id} = $ret_cond_id2ret_id{$retargeting->{ret_cond_id}};
        }

        # помечаем ретаргетинги на удаление
        my %ret_remain_id = map {$_ => 1} map {$_->{ret_id} || ()} @{$group->{retargetings}};
        push @{$group->{retargetings}},
            map {+{ret_id => $ret_cond_id2ret_id{$_}, ret_cond_id => $_, _delete => 1}}
            grep {!$ret_remain_id{$ret_cond_id2ret_id{$_}}}
            keys %ret_cond_id2ret_id;

        # помечаем автотаргетинг на удаление
        my $bid_id2relevance_match = Direct::Bids::BidRelevanceMatch->get_by(adgroup_id => $group->{pid})->items_by('bid_id');
        my %relevance_match_remain_id = map { $_->{bid} => 1 } grep { $_->{bid} } @{$group->{relevance_match}};
        push @{$group->{relevance_match}}, map { +{bid_id => $_, _delete => 1} } grep { !$relevance_match_remain_id{$_} } keys %$bid_id2relevance_match;
    }

    return $group;
}

1;
