use strict;
use warnings;
use utf8;

package XLSParse;

use Spreadsheet::ParseExcel;
use Yandex::Spreadsheet::ParseExcel::FmtUTF8;
use Excel::Reader::XLSX;
use List::MoreUtils qw/all any part/;
use List::Util qw/min max first/;
use Yandex::ScalarUtils;
use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xminus xsort/;
use Yandex::DBTools;
use Yandex::Validate qw/is_valid_id/;

use Campaign::Types qw/get_camp_supported_adgroup_types/;
use Phrase;
use PrimitivesIds;
use Retargeting;
use PhraseText;
use PhrasePrice;
use Models::Phrase;
use Tools;
use TextTools;
use Settings;
use GeoTools;
use Currencies;
use BannerImages;
use MinusWords;
use MinusWordsTools;
use URLDomain;
use XLSVocabulary qw/get_field_name get_field_names/;
use MobileContent;
use Client qw/get_client_data/;
use Direct::CanvasCreatives qw//;
use Direct::Creatives::Tools;
use Direct::Model::BidRelevanceMatch::Helper qw/is_relevance_match/;

my @ALL_XLS_FIELDS = XLSVocabulary::get_all_fields();
my @REQUIRED_XLS_FIELDS = qw/is_banner pid id phr bid title body href region sitelink_titles sitelink_hrefs param1 param2 tags/;
my @PHRASE_XLS_FIELDS = qw/id phr price price_context param1 param2/;
my @RELEVANCE_MATCH_XLS_FIELDS = qw/id price price_context phrase_status param1 param2/;
my @RETARGETING_CONDITION_XLS_FIELDS = qw/id pid bid phr price_context phrase_status/;
my @TARGET_INTEREST_XLS_FIELDS = qw/price_context/;
my @BANNER_XLS_FIELDS = qw/bid banner_type ad_type title title_extension body href display_href contact_info age sitelink_titles
    sitelink_descriptions sitelink_hrefs image_url callouts mobile_href mobile_icon mobile_price mobile_rating
    mobile_rating_votes creative_id permalink/;
my @TURBOLANDING_XLS_FIELDS = qw/turbolanding_id sitelink_turbolanding_ids/;
my %HEADER_TURBOLANDING_FIELDS = map {$_ => 1} @TURBOLANDING_XLS_FIELDS;

my @GROUP_XLS_FIELDS = qw/pid region group_name number minus_words adgroup_type mobile_store_content_href mobile_network_targeting mobile_device_type_targeting mobile_os/;

# Значимые поля, которые мы считываем и сохраняем
my %HEADER_VALUABLE_FIELDS = map {$_ => 1} @GROUP_XLS_FIELDS, @BANNER_XLS_FIELDS, @RETARGETING_CONDITION_XLS_FIELDS, @PHRASE_XLS_FIELDS, @TURBOLANDING_XLS_FIELDS,
    qw/is_banner tags/;                                      
# Незначимые для считывания поля
# quality_score больше не показывается, здесь остался для обратной совместимости при загрузке кампании в xls в старом формате с продуктивностью
my %HEADER_NON_VALUABLE_FIELDS = map {$_ => 1} qw/l1 l_title l_body mobile_columns_adgroup_header mobile_columns_banner_header
    quality_score banner_status creative_status_moderate/;
my %device_type_targeting_fields_to_keys = (mobile_device_type_targeting_all => [qw/tablet phone/],
                                            mobile_device_type_targeting_tablet => ['tablet'],
                                            mobile_device_type_targeting_phone => ['phone']);
my %network_targeting_fields_to_keys = (mobile_network_targeting_all => [qw/cell wifi/], mobile_network_targeting_wifi => ['wifi']);

sub groupxls2camp_snapshot {
    
    my ($worksheets, $options) = @_;

    my $currency = $options->{currency};
    die 'no currency given' unless $currency;
    
    my ($raw_groups, $raw_contact) = @{$worksheets}[0,1];
    my $contact = _transform_raw_contact($raw_contact);
    my %xls = _transform_raw_groups($raw_groups, currency => $currency, allow_empty_geo => $options->{allow_empty_geo}, ClientID => $options->{ClientID});

    if ($xls{xls}->{campaign_type} eq 'mobile_content') {
        _set_store_content_info($xls{xls}, hash_cut $options, qw/ClientID/);
    }

    $xls{xls}->{contact_info} = $contact;
    $xls{xls}->{currency} ||= $currency;
    $xls{xls}->{is_group_format} = 1;
    push @{$xls{parse_warnings}}, XLSParse::validate_sheet_count(scalar(@$worksheets));

    return {
        xls_camp => $xls{xls}, 
        (header_keys => $xls{xls} && $xls{xls}->{header_keys} ? $xls{xls}->{header_keys} : {}),
        parse_errors => $xls{parse_errors},
        geo_errors => $xls{geo_errors},
        parse_warnings => $xls{parse_warnings},
        parse_warnings_for_exists_camp => $xls{parse_warnings_for_exists_camp}
    } 
}

sub _set_store_content_info {
    my ($xls, $options) = @_;
    my %store_content_href_to_content_info;
    foreach my $adgroup (@{$xls->{groups}}) {

        # Простукиваем ссылки на приложения
        my $content_info =
            $store_content_href_to_content_info{$adgroup->{store_content_href}} //=
                MobileContent::ajax_mobile_content_info($adgroup->{store_content_href}, $options->{ClientID});
        $adgroup->{mobile_content_info} = $content_info;
    }
}

{
my %HEADER_DICT;
sub make_header_dict {
    my $lang = Yandex::I18n::current_lang();
    for my $key (@ALL_XLS_FIELDS) {
        my $field_names = XLSVocabulary::get_field_names($key);

        foreach my $field_name (@$field_names) {
            $HEADER_DICT{$lang}->{$field_name} = $key;
        }
    } 
}

=head2 _get_header_data_from_line

    Из строки на входе (строки таблицы) возвращает массив преобразованных в короткое название полей которые могут являться заголовком. 
    Если ни одной такой ячейки не нашлось, то - undef.
    На входе:
        line - arrayref, массив строк из считываемого файла, на языке пользователя.
    На выходе:
        массив аналогичный входящему, только вместо строк пользователя исползуются краткие обозначения из XLSVocabulary.
        или undef, если входные данные неправильные или если в строке не был найден ни один заголовок.

=cut    

sub _get_header_data_from_line {
    my ($line, $client_id) = @_;    
    return undef if !defined $line || ref $line ne 'ARRAY';
    my $lang = Yandex::I18n::current_lang();
    make_header_dict() unless exists $HEADER_DICT{$lang};

    my $is_header_line = any {$HEADER_VALUABLE_FIELDS{$HEADER_DICT{$lang}->{$_} || ''} || $HEADER_NON_VALUABLE_FIELDS{$HEADER_DICT{$lang}->{$_} || ''}} grep {$_} @$line;
    return undef unless $is_header_line;
    my @header;
    for my $i (0..$#$line) {
        my $field = $HEADER_DICT{$lang}->{$line->[$i]} || '';
        $field =~ s/_mobile_content//;
        next if !$HEADER_VALUABLE_FIELDS{$field} || $HEADER_NON_VALUABLE_FIELDS{$field};
        next if $HEADER_TURBOLANDING_FIELDS{$field} && !Client::ClientFeatures::get_is_featureTurboLandingEnabled(client_id => $client_id);
        
        $header[$i] = $field if $field;
    }
    return \@header;
}


=head2 get_header_dict

    Возвращает объект "словарь" для заголовков колонок в файле.

=cut
sub get_header_dict {
    my $lang = shift;
    make_header_dict() unless exists $HEADER_DICT{$lang};
    return $HEADER_DICT{$lang};
}
}

sub _parse_raw_groups {
    my ($xls, $currency, $client_id) = @_;
    die 'no currency given' unless $currency;
    
    my @rows;
    my %xls = (
        data_header => undef,
        cid => undef,
        parse_errors => [],
        parse_warnings => [],
    );
    my $line_number = 0;
    my $whole_header = [];
    my $row;
    while (@$xls) {
        my $row = shift @$xls; 
        ++$line_number;
        my @fields = map {str $_} @$row;

        next if !@fields || all {/^\s*$/} @fields;

        # в строках с данными как минимум 3 колонки должно быть заполненно
        if ($xls{data_header} && scalar(grep {$_ =~ /\S/} @fields) < 3) {
            next;
        }

        # Если заголовки еще не все найдены.
        if (!$xls{data_header}) {
            # Заголовки колонок
            my $heads = _get_header_data_from_line(\@fields, $client_id);
            if (defined $heads) {
                for my $i (0..$#$heads) {
                    if ($heads->[$i]) {
                        $whole_header->[$i] = $heads->[$i];
                    } elsif (! $whole_header->[$i]) {
                        $whole_header->[$i] = '';
                    }
                }
            }
            if (@$whole_header) {
                my $absent_fields = xminus(\@REQUIRED_XLS_FIELDS, $whole_header);
                unless (@$absent_fields) {
                    my $second_head = shift @$xls;
                    ++$line_number;
                    my $heads = _get_header_data_from_line($second_head, $client_id);
                    if (defined $heads) {
                        for my $i (0..$#$heads) {
                            if ($heads->[$i]) {
                                $whole_header->[$i] = $heads->[$i];
                            } elsif (! $whole_header->[$i]) {
                                $whole_header->[$i] = '';
                            }
                        }
                    }
                    $xls{data_header} = $whole_header;
                    next;
                }
            }

            # тип кампании
            if (!$xls{campaign_type} && $fields[3] && lc($fields[3]) eq lc(get_field_name('campaign_type'))) {

                my $detected_camp_type = XLSVocabulary::get_campaign_type_by_parse_string(
                                            get_header_dict(Yandex::I18n::current_lang())->{$fields[4]});

                if (!defined $detected_camp_type) {
                    push @{$xls{parse_errors}}, 
                         iget("Строка %s: %s", $line_number, iget('Указано неверное значение типа кампании. Допустимые значения указаны на дополнительном листе "Словарь значений".'));
                } else {
                    $xls{campaign_type} = $detected_camp_type;
                }
            }
            # номер кампании
            if (!$xls{cid} && $fields[3] && length($fields[3]) > 0 && $fields[4] && $fields[4] =~ m/^(\d+)$/) {
                $xls{cid} = $1;
            }
            # валюта кампании, указанная в xls файле
            if (!$xls{currency} && $fields[7] && Currencies::is_valid_currency($fields[7])) {
                $xls{currency} = $fields[7];
            }

            # Единые минус-слова на кампанию
            if (!$xls{campaign_minus_words} && $fields[3] && lc($fields[3]) eq lc(get_field_name('campaign_minus_words')) ) {
                $xls{campaign_minus_words} = MinusWordsTools::minus_words_interface_show_format2array(smartstrip($fields[4]));
            }

        }
        # Так как в предыдущем условии (!$xls{data_header}) может заполиться это самое поле, то тут снова проверяем.
        if ($xls{data_header}) {
            push @rows, {line_number => $line_number, data => \@fields};
        }

    }
    my $absent_fields = xminus(\@REQUIRED_XLS_FIELDS,$whole_header);
    if (@$absent_fields) {
        push @{$xls{parse_errors}}, map {iget("Не удалось найти колонку: '%s'", XLSVocabulary::get_field_name($_))} @$absent_fields; 
    }

    return \%xls if (@{$xls{parse_errors}});

    if ($xls{data_header}) {
        for my $row (@rows) {
            my $record = parse_data_line($row->{data}, $xls{data_header}, $row->{line_number});
            push @{$xls{records}}, $record;
            my $validation_result = validate_data_record($record, $currency, campaign_type => $xls{campaign_type});
            foreach (qw/errors warnings/) {
                push @{$xls{"parse_$_"}}, map {
                    iget("Строка %s: %s", $record->{line_number}, $_ . (!/\.$/ ? '.' : ''))
                } @{$validation_result->{$_} || []};
            }
        }
    } else {
        $xls{parse_errors} = [iget("Не удалось распознать названия колонок")]
    }

    return \%xls;
}

=head2 parse_data_line

=cut

sub parse_data_line {
    my ($row, $data_header, $line_number) = @_;

    # make a hash for row data
    my %no_strip_fields = map {$_ => 1} qw/param1 param2/;
    my %no_replace_angle_quotes_fields = map {$_ => 1} qw/body title title_extension group_name/;
    my $record =  hash_merge { map {$data_header->[$_], $row->[$_]} grep {
        smartstrip($row->[$_], dont_replace_angle_quotes => $no_replace_angle_quotes_fields{$data_header->[$_]}) if $data_header->[$_] && !$no_strip_fields{$data_header->[$_]};
        $data_header->[$_] && '' ne str $row->[$_]
    } 0..$#$data_header}, {line_number => $line_number};

    $record->{price} =~ s/\,/./ if defined $record->{price};

    foreach (qw/href mobile_href/) {
        if (exists $record->{$_} && length $record->{$_}) {
            $record->{$_} = clear_banner_href($record->{$_});
        }
    }

    return $record;
}

=head2 validate_data_record

=cut

sub validate_data_record {
    my ($record, $currency, %options) = @_;

    die 'no currency given' unless $currency;

    my (@parse_errors, @parse_warnings);
    my $validate_phrase = 1;
    push @parse_errors, iget("Номер группы должен быть числом или пустой строкой '%s'.", $record->{number}) unless str($record->{number}) =~ m/^(|[1-9]\d*)$/;
    push @parse_errors, iget("ID группы должен быть числом или пустой строкой '%s'.", $record->{pid}) unless str($record->{pid}) =~ m/^(|[1-9]\d*)$/;
    push @parse_errors, iget('Доп. объявление группы должно иметь значение "+" или "-" вместо "%s".', $record->{is_banner}) unless str($record->{is_banner}) =~ m/^(\+|\-)$/;
    if (exists $record->{banner_type}) {
        push @parse_errors,
            iget('Колонка "Мобильное объявление" дожна иметь значение "+" или "-" вместо "%s".', $record->{banner_type}) unless str($record->{banner_type}) =~ m/^(\+|\-)$/;
    }
    if (exists $record->{ad_type}) {
        my @types = (
            get_field_name('ad_type_text'), get_field_name('ad_type_image_ad'),
        );
        my $types = join "|", map { "\Q$_\E" } @types;
        push @parse_errors, iget_noop('Указано неверное значение поля "Тип объявления"') unless $record->{ad_type} =~ /^$types$/;
    }
    $validate_phrase = $record->{is_banner} && $record->{is_banner} eq '-';

    push @parse_errors, iget("ID объявления должен быть числом или пустой строкой '%s'.", $record->{bid}) unless str($record->{bid}) =~ m/^(|[1-9]\d*)$/;

    push @parse_errors, iget("ID Турбо-страницы объявления должен быть числом или пустой строкой '%s'.", $record->{turbolanding_id})
            unless str($record->{turbolanding_id}) =~ m/^(|[1-9][0-9]*)$/;
    push @parse_errors, iget("ID Турбо-страницы сайтлинка должен быть числом или пустой строкой '%s'.", $record->{sitelink_turbolanding_ids})
            if any {str($_) !~ m/^(|[1-9][0-9]*)$/} split(/\|\|/, $record->{sitelink_turbolanding_ids} // '');
    my $is_camp_mobile = (($options{campaign_type} || '') eq 'mobile_content') ? 1 : 0;
    if ($is_camp_mobile && length(str($record->{href})) > 0 && length(str($record->{mobile_href})) == 0) {
        $record->{mobile_href} = $record->{href};
    }
    if (length(str($record->{href})) == 0 && str($record->{contact_info}) eq '-' && !$is_camp_mobile) {
        if (!$record->{permalink} && !$record->{turbolanding_id}) {
            push @parse_errors, iget("Не задана ссылка для объявления");
        }
    }

    if ($validate_phrase) {
        push @parse_errors, iget("ID фразы должен быть числом или пустой строкой '%s'.", $record->{id}) unless str($record->{id}) =~ m/^(|[1-9]\d*)$/;
    
        for my $param_name(qw/param1 param2/) {
            my $href_param_error = validate_phrase_one_href_param($record->{$param_name});
            push @parse_errors, $href_param_error if $href_param_error;
        }
    
        if (defined $record->{price} && $record->{price} ne '') {
            push @parse_errors, _validate_price($record->{price}, $currency, iget_noop('Ошибка в ставке: %s'));
        }
        if (defined $record->{price_context} && $record->{price_context} ne '') {
            push @parse_errors, _validate_price($record->{price_context}, $currency, iget_noop('Ошибка в поле «Ставка в сетях»: %s'));
        }
    }
    # Проверяем допустимость значений возрастных ограничений
    if (!BannerFlags::validate_flag_value("age", $record->{age})) {
        push @parse_warnings, iget_noop('Указано неверное значение поля "Возрастные ограничения". Значение возрастного ограничения будет определено после сохранения кампании автоматически.');
    }

    if (defined $record->{creative_id} && ! is_valid_id($record->{creative_id})) {
        push @parse_errors, iget("ID видеодополнения должен быть числом или пустой строкой")
    }

    return {errors=>\@parse_errors, warnings=>\@parse_warnings};
}


sub _validate_price {
    my ($price, $currency, $message_template) = @_;

    if (length(str($price)) > 0) {
        my $error = validate_phrase_price(str($price), $currency, dont_support_comma => 1);
        if ($error && $message_template) {
            $error = iget($message_template, $error);
        }
        return ($error) ? ($error) : ();
    }
    return ();
}

sub _get_group_key {
    
    my $row = shift;

    return $row->{pid} if $row->{pid};
    return join '/', map {str $row->{$_}} qw/group_name number/;
}

=head2 validate_sheet_count

    Проверяет, что количество вкладок в файле в рамках разумного
    На входе:
        count - количество страниц в файле
    На выходе:
        текст с ошибкой

=cut
sub validate_sheet_count {
    my $count = shift;
    return iget('Файл содержит слишком много вкладок. Возможно использование неправильных данных.') if $count > 4;
    return iget('Файл содержит слишком мало вкладок. Данные в файле могли быть утеряны.') if $count < 2;
    return;
}

=head2 get_banner_key

    Возвращает идентификатор (ключ), уникально идентифицирующий баннер. Все фразы (строки файла) с одинаковым ключём должны
    принадлежать одному баннеру.
    Принимает на вход ссылку на хеш с данными по баннеру и необязательный именованный параметр:
        ignore_banner_num — игнорировать поле "Номер объявления"

    $key = get_banner_key($banner);

=cut

sub get_banner_key {
    my ($banner, %O) = @_;

    if ($banner->{bid}) {
        return $banner->{bid};
    } else {
        my @key_fields = qw/ad_type title title_extension body banner_type real_banner_type href display_href region contact_info turbolanding_id
        sitelink_titles sitelink_descriptions sitelink_hrefs  sitelink_turbolanding_ids/;
        push @key_fields, 'callouts' if $O{header_keys} && $O{header_keys}->{callouts};
        unshift @key_fields, 'number' if !$O{ignore_banner_num};

        my $banner_key = join('/', map { str $banner->{$_} } @key_fields);

        if ($O{header_keys} && $O{header_keys}->{image_url}) {
            my $image_key = $banner->{image_hash} || $banner->{image_url} || q{};
            $banner_key .= "/$image_key";
        }

        return $banner_key;
    }
}

sub _extract_banner {
    
    my ($record, %options) = @_;

    my $is_camp_text = $options{campaign_type} eq 'text';
    my $is_camp_mobile = $options{campaign_type} eq 'mobile_content' ? 1 : 0;

    my %banner = (line_number => $record->{line_number});

    $banner{ad_type} = $record->{ad_type} eq get_field_name('ad_type_image_ad')
        ? 'image_ad'
        : 'text';

    if (length $record->{sitelink_titles} || length $record->{sitelink_hrefs}) {
        my @hrefs = map {clear_banner_href($_)} split /\|\|/, $record->{sitelink_hrefs};
        my @titles = map {smartstrip($_); $_} split /\|\|/, $record->{sitelink_titles};

        my @descriptions;
        if ($record->{sitelink_descriptions}) {
            @descriptions = map {smartstrip($_); $_} split /\|\|/, $record->{sitelink_descriptions};
        }

        my @turbolandings;
        if ($record->{sitelink_turbolanding_ids}) {
            @turbolandings = map {smartstrip($_); $_} split /\|\|/, $record->{sitelink_turbolanding_ids};
        }
        
        $banner{sitelinks} = [map {{
                                    href => $hrefs[$_],
                                    title => $titles[$_],
                                    description => $descriptions[$_] || undef,
                                    ($turbolandings[$_] ? (turbolanding => {id => $turbolandings[$_]} ) : ()),
                                }}
                              grep {(defined $hrefs[$_] || defined $turbolandings[$_]) && defined $titles[$_]}
                              0 .. $#titles
                             ];
    } else {
        $banner{sitelinks} = [];
    }

    if (exists $record->{callouts} && length $record->{callouts}) {
        $banner{callouts} = [map {smartstrip($_); {callout_text => $_}} split /\|\|/, $record->{callouts}];
    }

    $banner{banner_type} = ($record->{banner_type} || '-') eq '+' && $is_camp_text ? 'mobile' : 'desktop';
    $banner{is_mobile} = $banner{banner_type} eq 'mobile' ? 1 : 0;

    if ($is_camp_mobile) {
        $banner{reflected_attrs} = [];
        foreach (qw/icon rating rating_votes price/) {
            $record->{"mobile_".$_} = $record->{"mobile_".$_} =~ m/([\+\-])/ ? $1 : '';
            push @{$banner{reflected_attrs}}, $_ if $record->{"mobile_".$_} eq '+';
        }
    }

    $record->{contact_info} = $record->{contact_info} =~ m/([\+\-])/ ? $1 : '';
    $banner{flags} = {age => BannerFlags::normalize_flag_value(age => $record->{age})};
    $banner{permalink} = $record->{permalink};

    foreach my $field (
        @BANNER_XLS_FIELDS,
        ($options{client_can_do_turbolandings} ? @TURBOLANDING_XLS_FIELDS : ()),
    ){
        my $val = $record->{$field};
        $banner{$field} = $record->{$field} unless exists $banner{$field};
        push @{$banner{lines_by_field}{$field}{$val}||=[]}, $record->{line_number};
    }
    push @{$banner{lines_by_field}{is_banner}{$record->{is_banner}}||=[]}, $record->{line_number};

    $banner{href} = ($is_camp_mobile ? $banner{mobile_href} : $banner{href}) || $banner{href};
    $banner{has_href} = 1 if $banner{href};
    $banner{has_vcard} = 1 if $banner{contact_info} eq "+" && $is_camp_mobile;
    (undef, undef, $banner{image_hash}) = BannerImages::get_image_hash_by_url($banner{image_url});

    if ($record->{image_url}) {
        my $creative_id = Direct::CanvasCreatives->extract_creative_id($record->{image_url});
        $banner{creative}->{creative_id} = $creative_id if $creative_id;
    }

    $banner{turbolanding} = {id => $record->{turbolanding_id}} if $record->{turbolanding_id};
    
    # для XLS пустая строка валидное значение, т.к. в экселе это аналог пустого значения (null-а нет)
    # https://st.yandex-team.ru/DIRECT-67963
    $banner{title_extension} = undef if $banner{title_extension} eq '';

    if ($record->{creative_id}) {
        $banner{video_resources}->{id} = $record->{creative_id};
        $banner{video_resources}->{resource_type} = 'creative';
    }
    return get_banner_key(\%banner, header_keys => {image_url => 1}), \%banner;
}

# --------------------------------------------------------------------

=head2 _guess_bid_by_texts

    Угадываем номера объявлений по совпадению заголовка, текста, ссылки и текста сайтлинков

    my $bid = _guess_bid_by_texts($banner, $exists_banners);

=cut

sub _guess_bid_by_texts {
    my ($banner, $exists_banners) = @_;

    my $sitelink_titles = ref($banner->{sitelinks}) eq 'ARRAY'
        ? join('||', sort map {$_->{title}} @{$banner->{sitelinks}})
        : '';

    for my $exists_bid (keys %$exists_banners) {
        my $exists_banner = $exists_banners->{$exists_bid};
        next unless all {$banner->{$_} eq $exists_banner->{$_}} qw/banner_type title title_extension body href/;

        if (!defined $exists_banner->{sitelink_titles}) {
            my $existst_sitelinks = [];
            if ($exists_banner->{sitelinks_set_id}) {
                $existst_sitelinks = Sitelinks::get_sitelinks_by_set_id($exists_banner->{sitelinks_set_id});
            } elsif ($exists_banner->{sitelinks}) {
                $existst_sitelinks = $exists_banner->{sitelinks};
            }

            $exists_banner->{sitelink_titles} = join('||',
                map {$_->{title}}
                sort {$a->{title} cmp $b->{title}}
                @$existst_sitelinks
            );
        }

        # тут была ещё нерабочая проверка image_url, пока убрал

        return $exists_bid  if $sitelink_titles eq $exists_banner->{sitelink_titles};
    }

    return $banner->{bid}; # ничего не нашли, возвращаем старый bid (похоже там "")
}

=head2 _guess_cid_by_pids

    угадываем cid по списку групп, если все группы принадлежат одной кампании - отдаем эту кампанию
    my $cid = _guess_cid_by_pids(pid, pid, ...);

=cut

sub _guess_cid_by_pids {
    
    return undef unless @_;
    my @pids = @_;

    my $cids = get_cids(pid => \@pids);
    return scalar(@$cids) == 1 ? $cids->[0] : undef;
}

sub _extract_groups {
    
    my ($groups, $rows, %options) = @_;
    
    my $lang = Yandex::I18n::current_lang();
    my (@warnings, @errors);
    
    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($options{campaign_type}, $options{ClientID});
    
    foreach my $record (@$rows) {
            
        $record->{$_} = str $record->{$_} foreach @ALL_XLS_FIELDS;
        
        my $is_banner = $record->{is_banner} eq '+';
        
        my $key = _get_group_key($record);
        unless (exists $groups->{$key}) {
            $groups->{$key} = {
                banners => {},
                phrases => [],
                retargetings => [],
                phrases_ids => {},
                lines_without_pids => []
            };
        }
        
        my $group = $groups->{$key};
        if (!exists($group->{geo}) && !$is_banner) {
            hash_merge $group, hash_cut $record, 'line_number', grep { $_ ne 'minus_words' } @GROUP_XLS_FIELDS;
            $group->{region} =~ s/(^|(?<=\,))\s+|\s+((?=\,)|$)//g;
            my $geo = get_geo_numbers($group->{region}, lang => $lang);
            $group->{geo} = $geo;
            $group->{store_content_href} = $record->{mobile_store_content_href};
            $group->{$_} = _get_internal_value_by_text($record->{"mobile_".$_}) foreach (qw/device_type_targeting network_targeting/);
            my $reg = join "|", map {XLSVocabulary::get_field_name($_)} qw/mobile_os_android mobile_os_ios/;
            if ($record->{mobile_os} =~ /(?:$reg)\s*(\d+(?:\.\d+))?/i) {
                $group->{min_os_version} = $1;
            } else {
                $group->{min_os_version} = '';
            }
        }

        # используем метки из первой строки основного объявления
        if (!exists($group->{tags}) && !$is_banner) {
            $group->{tags} = $record->{tags}
                ? [map {smartstrip($_); +{tag_name => $_}} split ',', $record->{tags}]
                : [];
        }
        push @{$group->{lines_without_pids}}, $record->{line_number} unless $record->{pid};

        # Минус-слова на группу
        push @{$group->{minus_words}}, @{MinusWordsTools::minus_words_interface_show_format2array($record->{minus_words})};

        foreach my $field ($is_banner ? qw/pid group_name minus_words/ : (@GROUP_XLS_FIELDS, 'tags')) {
            my $val = $record->{$field};
            push @{$group->{lines_by_field}{$field}{$val}}, $record->{line_number} if defined $val;
        }

        my ($banner_id, $banner) = _extract_banner($record, campaign_type => $options{campaign_type}, ClientID => $options{ClientID},
            client_can_do_turbolandings => $options{client_can_do_turbolandings});

        if (my $existed_banner = $group->{banners}->{$banner_id}) {
            while (my ($field, $values) = each %{$banner->{lines_by_field}}) {
                foreach my $val (keys %$values) {
                    push @{$existed_banner->{lines_by_field}{$field}{$val}||=[]}, @{$values->{$val}};
                } 
            }
        } else {
            # главный баннер - баннер который будет первым создан в кампании (будет иметь минимальный bid)
            # и как следствие отобразится первым в группе (больше никакого смысла признак не несёт)
            $banner->{is_main_banner} = !$is_banner; 
            $group->{banners}->{$banner_id} = $banner;
        }

        if ($banner->{ad_type} eq 'image_ad' && $banner->{banner_type} eq 'mobile') {
            push @warnings, iget('Строка %s: ', $record->{line_number}).iget('Графическое объявление не может быть мобильным. Будет создано универсальное графическое объявление.');
            $banner->{banner_type} = 'desktop';
        }
        
        unless ($is_banner) {
            
            $group->{is_filled} = 1;
            if($record->{phr} =~ s/^($Retargeting::CONDITION_XLS_PREFIX|$Retargeting::CONDITION_XLS_PREFIX_OLD)\s*//){
                # это ретаргетинг
                my $retargeting = hash_cut $record, @RETARGETING_CONDITION_XLS_FIELDS, qw/line_number/;
                ($retargeting->{ret_cond_id}, $retargeting->{condition_name}, $retargeting->{is_suspended}) = delete @{$retargeting}{qw(id phr phrase_status)};
                $retargeting->{is_suspended} = $retargeting->{is_suspended} eq iget('Приостановлена') ? 1 : 0;  
                push @{$group->{retargetings}}, $retargeting;
            } elsif ($record->{phr} =~ s/^$Retargeting::CONDITION_XLS_PREFIX_TARGET_INTEREST.+\((\d+)\)\s*//) {
                # это таргетинг по интересам
                my $target_interest = hash_cut $record, @TARGET_INTEREST_XLS_FIELDS, qw/line_number/;
                $target_interest->{ret_id} = delete $record->{id};
                $target_interest->{target_category_id} = $1;
                push @{$group->{target_interests}}, $target_interest;
            } elsif (is_relevance_match($record->{phr})) {
                # это беcфразноый таргетинг
                my $relevance_match = hash_cut $record, @RELEVANCE_MATCH_XLS_FIELDS, qw/line_number/;
                delete $relevance_match->{price_context} unless $has_extended_relevance_match;
                my $phrase_status;
                ($relevance_match->{bid_id}) = delete $relevance_match->{id};
                $phrase_status = $relevance_match->{phrase_status};

                $relevance_match->{is_suspended} = $phrase_status eq iget('Приостановлена') ? 1 : 0;
                
                push @{$group->{relevance_match}}, $relevance_match;
            } else {
                # иначе это фраза
                my $texts = [$record->{phr}];
                my $br_result = process_phrase_brackets($texts);
                push @errors, iget("Строка %s:", $record->{line_number}) . ' ' . $br_result if $br_result;
                
                for my $keywords (@$texts) {
                    my $phrase = hash_cut $record, @PHRASE_XLS_FIELDS, qw/line_number/;
                    $phrase->{phrase} = $phrase->{phr} = $keywords;
                    push @{$group->{phrases}}, $phrase;
                }
                
                my $phrase = $group->{phrases}->[-1];
                if ($phrase->{id}
                    && $group->{phrases_ids}->{$phrase->{id}}++) {
                    push @errors, iget("Повторное вхождение фразы № %s в строке %s.", $phrase->{id}, $record->{line_number});
                }
            }
            # В XLS не передается autobudgetPriority, но требуется для валидации. Проставляем значение по умолчанию.                
            foreach (@{$group->{phrases} || []}, @{$group->{retargetings} || []}, @{$group->{target_interests} || []}, @{$group->{relevance_match} || []}) {
                    $_->{autobudgetPriority} = 3;
            }
        } else {
            # empty GroupID and group name
            if (any {$record->{$_}} 'region', 'tags', @PHRASE_XLS_FIELDS) {
                push @warnings,
                    iget('Фраза, регион, ставка и метки не указываются в дополнительных объявлениях группы. Эти поля в строке %d будут проигнорированы', $record->{line_number})
            }
            unless ($key) {
                push @errors,
                    iget('Неправильный формат дополнительного объявления группы в строке %d. Необходимо указать группу объявлений в поле "Название группы".', $record->{line_number})
            }
        }
    }

    my @canvas_banners = xsort { $_->{line_number} } grep {$_->{ad_type} eq 'image_ad' && $_->{creative}} map { values %{$_->{banners}} } values %$groups;
    my $existing_creatives = Direct::Creatives::Tools->get_existing($options{ClientID}, [ map {$_->{creative}->{creative_id}} @canvas_banners ], ['canvas','html5_creative','bannerstorage']);
    foreach my $banner (@canvas_banners){
        if ($existing_creatives->{$banner->{creative}->{creative_id}}) {
            $banner->{has_existing_creative} = 1;
        }
        else {
            # DIRECT-73488: отключаем офлайн-загрузку креативов, она работает некорректно
            push @errors, iget(
                    'Объявление M-%s: Строка: %s: canvas-креатив %s не найден.',
                    $banner->{bid}, $banner->{line_number}, $banner->{image_url}
            );
        }
    }

    if ($options{campaign_type} eq 'text') {
        my $client_options = get_client_data($options{ClientID}, [qw/auto_video/]);
            
        if ($client_options->{auto_video}) {
            my @video_banners =
                xsort { $_->{line_number} }
                grep { $_->{ad_type} eq 'text' && ! $_->{bid} }
                map { values %{$_->{banners}} }
                values %$groups;
            my @video_creatives = grep { is_valid_id($_) } map { $_->{video_resources}->{id}||() } @video_banners;
            my $existing_creatives = {};
            if (@video_creatives) {
                $existing_creatives = Direct::Creatives::Tools->get_existing($options{ClientID}, \@video_creatives, 'video_addition');
            }
            for my $banner (@video_banners) {
                next if $existing_creatives->{$banner->{video_resources}->{id}//''};
                push @warnings,
                    ($banner->{bid} ? iget('Объявление M-%s: ', $banner->{bid}) : '')
                    . iget('Строка: %s: Креатив для видеодополнения будет автоматически создан после завершения импорта файла', $banner->{line_number}) 
            }
        }
    }

    return \@errors, \@warnings;
}

sub validate_xls_groups {
    
    my ($groups, %options) = @_;

    my (@warnings, @errors, @geo_errors);
    foreach my $group (@$groups) {
        my $group_prefix = $group->{group_name} ? iget('Группа объявлений "%s":', $group->{group_name}) : '';
        foreach my $field (@GROUP_XLS_FIELDS) {
            if ($field eq 'minus_words') {
                if (scalar(grep { length $_ } keys %{$group->{lines_by_field}{$field}}) > 1) {
                    my @rows = sort {scalar @$a <=> scalar @$b} map { $group->{lines_by_field}{$field}->{$_} } grep { length $_ } keys %{$group->{lines_by_field}{$field}};
                    my @lines = reverse map {$rows[$_]->[0]} 0..min(10, $#rows);
                    push @warnings, $group_prefix . " " . iget("Значения в поле '%s' отличаются в строках %s. Данные значения будут объединены.", get_field_name($field), join(', ', @lines));
                }
            } else {
                if (scalar keys %{$group->{lines_by_field}{$field}} > 1) {
                    my @rows = sort {scalar @$a <=> scalar @$b} values %{$group->{lines_by_field}{$field}};
                    my @lines = map {$rows[$_]->[0]} 0..min(10, $#rows);
                    push @errors, $group_prefix
                            . " " . iget("Значения в поле '%s' отличаются в строках %s. Указанные поля должны быть одинаковыми для одной и той же группы.",
                            get_field_name($field), join ', ', @lines);
                }
            }
        }
        
        my @name_lines = map {@$_} values %{$group->{lines_by_field}{group_name}};
        my $lines_range = array_to_compact_str(@name_lines);
        if (scalar keys %{$group->{lines_by_field}{tags}} > 1) {
            push @warnings, $group_prefix
                . iget('В строках %s различаются значения меток. Для объявления с этими фразами будут созданы метки, указанные в строке %d',
                    $lines_range, min(@name_lines));
        }
        
        my $turbolanding_ids = _get_all_turbolanding_ids($group->{banners});
        my $existing_turbolandings = {};
        if (@$turbolanding_ids) {
            $existing_turbolandings = {
                map {$_->id => 1} @{Direct::TurboLandings->get_by(client_id => $options{ClientID}, id => $turbolanding_ids)->items}
            };
        }
        
        foreach my $banner (@{$group->{banners}}) {
            if (scalar keys %{$banner->{lines_by_field}{is_banner}} > 1) {
                my @rows = sort {scalar @$a <=> scalar @$b} values %{$banner->{lines_by_field}{is_banner}};
                push @errors,
                    iget('Строки %s: основное объявление в группе совпадает с одним из доп. объявлений',
                        array_to_compact_str(map {$rows[$_]->[0]} 0..min(10, $#rows)));
            }
            
            if ($banner->{turbolanding}) {
                push @errors, iget('Строка %s: задан несуществующий идентификатор Турбо-страницы объявления - %s',
                                   $banner->{line_number}, $banner->{turbolanding}->{id})
                    unless $existing_turbolandings->{$banner->{turbolanding}->{id}};
            }
            
            if ($banner->{sitelinks}) {
                foreach my $sl (@{$banner->{sitelinks}}){
                    next unless $sl->{turbolanding};
                    push @errors, iget('Строка %s: задан несуществующий идентификатор Турбо-страницы быстрой ссылки - %s',
                                       $banner->{line_number}, $sl->{turbolanding}->{id})
                        unless $existing_turbolandings->{$sl->{turbolanding}->{id}};
                }
            }
            
            my $bid_error_prefix = $banner->{bid} ? iget("Объявление М-%s:", $banner->{bid}) : "";
            foreach my $field (@BANNER_XLS_FIELDS, @TURBOLANDING_XLS_FIELDS) {
                if (scalar keys %{$banner->{lines_by_field}{$field}} > 1) {
                    my @rows = sort {scalar @$a <=> scalar @$b} values %{$banner->{lines_by_field}{$field}};
                    my @lines = map {$rows[$_]->[0]} 0..min(10, $#rows);
                    push @errors, $bid_error_prefix
                        . " " . iget("Значения в поле '%s' отличаются в строках %s. Указанные поля должны быть одинаковыми для одного и того же объявления.",
                        get_field_name($field), join ', ', @lines);
                }
            }
        }
        
        unless ($group->{is_filled}) {
            my @lines = map {$_->{line_number}} @{$group->{banners}};
            push @errors, @lines > 1
                ? iget("Строки %s: группа не может состоять только из доп. объявлений.", array_to_compact_str(@lines))
                : iget("Строка %s: группа не может состоять только из доп. объявлений.", $lines[0])
        } else {
            if (!length($group->{region}) && !$options{allow_empty_geo}) {
                push @geo_errors,
                    iget("Строка %s: регион не указан, укажите регион в файле или задайте ЕДИНЫЙ регион для всех объявлений.", $group->{line_number});
            }

            unless (length($group->{geo}) && ( scalar(split /,/, $group->{geo}) == scalar(split /,/, $group->{region}) )){
                push @geo_errors, iget("Строка %s: регион не соответствует классификатору: '%s'.", $group->{line_number}, $group->{region});
            }
        }
    }
    
    return (
        errors => \@errors,
        warnings => \@warnings,
        geo_errors => \@geo_errors
    );
}

sub _get_all_turbolanding_ids {
    my ($user_banners) = @_;

    my $t_ids;
    foreach my $user_banner (@$user_banners) {
        if (exists $user_banner->{turbolanding}) {
            $t_ids->{$user_banner->{turbolanding}->{id}} //= 1;
        }
        if (exists $user_banner->{sitelinks}) {
            foreach my $sitelink (@{$user_banner->{sitelinks}}){
                if (exists $sitelink->{turbolanding}) {
                    $t_ids->{$sitelink->{turbolanding}->{id}} //= 1;
                }
                elsif(exists $sitelink->{tl_id}){
                    $t_ids->{$sitelink->{tl_id}} //= 1;
                }
            }
        }
    }
    
    return [ grep {$_ > 0} keys %$t_ids];
}

sub _get_exists_groups {
    
    my ($cid, $groups) = @_;

    my (%exists_groups, @bids);
    foreach my $group (@$groups) {
        next unless $group->{pid};
        $exists_groups{$group->{pid}} = {map {
            push @bids, $_->{bid}; 
            ($_->{bid} => hash_cut $_, 'bid', 'banner_type', 'title_extension', 'title', 'body', 'href', 'sitelinks')
        } values %{$group->{banners}} }
    }
    # TODO adgroup: заменить на Model::AdGrop::
    my $rest_banners = get_all_sql(PPC(cid => $cid), ['SELECT pid, bid, type AS banner_type, title, title_extension, body, href, sitelinks_set_id FROM banners', 
                                            where => {cid => $cid, bid__not_in => \@bids}]);
    foreach my $banner (@$rest_banners) {
        $exists_groups{$banner->{pid}} = {} unless exists $exists_groups{$banner->{pid}};
        $exists_groups{$banner->{pid}}->{$banner->{bid}} = hash_cut $banner, 'bid', 'banner_type', 'title', 'title_extension', 'body', 'href', 'sitelinks'
    }
    
    return \%exists_groups;
}


sub _transform_raw_groups {
    
    my ($raw_groups, %options) = @_;

    my $xls = _parse_raw_groups($raw_groups, $options{currency}, $options{ClientID});
    my @parse_errors = @{$xls->{parse_errors}};
    my @parse_warnings = @{$xls->{parse_warnings}};
    
    # warnings for existing campaigns
    my (%groups, %warnings_for_campaign);

    if (!defined($xls->{campaign_type})) {
        $warnings_for_campaign{campaign_type_not_defined} = iget('Загружен устаревший шаблон файла. Пожалуйста, используйте новый шаблон.');
        $xls->{campaign_type} = 'text';
    }

    $xls->{mediaType} = $xls->{campaign_type};

    # Обрабатываем файл в два прохода. Во время первого прохода обрабатываем баннеры с указанными номерами.
    # Во время второго прохода пытаемся угадать номера для баннеров без таковых.
    # Угадываем на основе объявлений из загруженного файла (они могли измениться) и на основе тех объявлений
    # из БД, которые не засветились в файле.
    my $client_can_do_turbolandings = Client::ClientFeatures::get_is_featureTurboLandingEnabled(client_id => $options{ClientID});
    my %extract_groups_params = (ClientID => $options{ClientID}, campaign_type => $xls->{campaign_type}, client_can_do_turbolandings => $client_can_do_turbolandings);
    my ($errors, $warnings) = _extract_groups(\%groups, [grep {$_->{bid}} @{$xls->{records}}], %extract_groups_params);
    
    push @parse_errors, @$errors;
    push @parse_warnings, @$warnings;
    my @records = grep {!$_->{bid}} @{$xls->{records}};
    
    if ($xls->{cid}) {
        my $exists_groups = _get_exists_groups($xls->{cid}, [values %groups]);
        foreach my $record (@records) {
            if ($record->{pid} && $exists_groups->{$record->{pid}}) {
                $record->{bid} = _guess_bid_by_texts($record, $exists_groups->{$record->{pid}});
            } 
        }
    }
    ($errors, $warnings) = _extract_groups(\%groups, \@records, %extract_groups_params);
    push @parse_errors, @$errors;
    push @parse_warnings, @$warnings;
    my @exists_pids = grep {$_} map {int($_->{pid}||0)} values %groups;
    $xls->{cid} = _guess_cid_by_pids(@exists_pids) if !$xls->{cid} && @exists_pids;
    if (!@exists_pids) {
        $warnings_for_campaign{without_any_pids} = join " ",
            iget("В файле отсутствуют ID групп."),
            iget("Импорт такого файла может привести к дублированию существующих групп, объявлений и фраз."),
            iget("Вы уверены, что хотите продолжить импорт данного файла?");
    } elsif (my @lines_without_pids =  map {@{$_->{lines_without_pids}}} values %groups) {
        my $line_numbers_str = array_to_compact_str(@lines_without_pids);
        $warnings_for_campaign{without_pids} = join " ",
            iget("В строках %s отсутствуют ID групп.", $line_numbers_str),
            iget("Отсутствие ID у уже существующих групп может привести к их дублированию.");
    }

    my %header_keys = map {($_ => 1)} grep {$_} @{$xls->{data_header}};

    my @groups;
    foreach my $group (values %groups) {
        my ($main_banner, $banners) = part {$_->{is_main_banner} ? 0 : 1} values %{$group->{banners}};
        $group->{adgroup_type} = get_camp_supported_adgroup_types(type => $xls->{campaign_type})->[0];
        $group->{banners} = [];
        push @{$group->{banners}}, sort {$a->{line_number} <=> $b->{line_number}} @$main_banner if $main_banner;
        push @{$group->{banners}}, sort {$a->{line_number} <=> $b->{line_number}} @$banners if $banners;
        $group->{group_name} = undef if (!$group->{group_name} || length($group->{group_name}) == 0);
        delete $group->{minus_words} unless $header_keys{minus_words};
        push @groups, $group;
    }
    
    my %result = validate_xls_groups(\@groups, %options);
    push @parse_errors, @{$result{errors}};
    push @parse_warnings, @{$result{warnings}};
    
    foreach my $g (@groups) {
        delete @$g{qw/lines_by_field lines_without_pids phrases_ids/};
        delete $_->{lines_by_field} foreach @{$g->{banners}}
    }
    # sort groups by ID and number
    my $super_number = 1 + ((max grep {$_} map {$_->{number}} @groups) || 0);
    my $super_pid = 1 + ((max grep {$_} map {$_->{pid}} @groups) || 0);
    @groups = sort {
        ($a->{pid} || $super_pid) <=> ($b->{pid} || $super_pid)
            or ($a->{number} || $super_number) <=> ($b->{number} || $super_number)
    } @groups;
    
    delete $xls->{records};
    $xls->{groups} = \@groups;

    $xls->{header_keys} = \%header_keys;
    delete $xls->{data_header};

    return (
        xls => $xls,
        parse_errors => \@parse_errors,
        parse_warnings => \@parse_warnings,
        parse_warnings_for_exists_camp => \%warnings_for_campaign,
        geo_errors => $result{geo_errors}
    );
}

sub _transform_raw_contact {

    my $raw_contact = shift;
    my $contact_data = {};

    my %week_days = (iget('пн') => 0,
                     iget('вт') => 1,
                     iget('ср') => 2,
                     iget('чт') => 3,
                     iget('пт') => 4,
                     iget('сб') => 5,
                     iget('вс') => 6);

    my $i = 26;
    my $worktime = '';
    while (defined $raw_contact->[$i][1] && length($raw_contact->[$i][1]) > 0) {

        next if ! defined $raw_contact->[$i][1]
             || ! defined $raw_contact->[$i][2]
             || ! defined $raw_contact->[$i][4]
             || ! defined $raw_contact->[$i][5];

        my $d1 = $week_days{$raw_contact->[$i][1]};
        my $d2 = $week_days{$raw_contact->[$i][2]};
        my ($h1, $m1) = split /:/, $raw_contact->[$i][4];
        my ($h2, $m2) = split /:/, $raw_contact->[$i][5];

        next if ! defined $d1 || ! defined $d2 || ! defined $h1 || ! defined $h2 || ! defined $m1 || ! defined $m2;
        $worktime .= join('#', $d1, $d2, $h1, $m1, $h2, $m2) . ';';
    } continue {
        $i++;
    }
    chop $worktime; # remove last ';'

    # my $phone = join("#", map {str}  $raw_contact->[12][2], $raw_contact->[12][3] ,$raw_contact->[12][4], $raw_contact->[12][5]);

    $contact_data = {
        country_code  => $raw_contact->[12][2] || undef,
        city_code     => $raw_contact->[12][3] || undef,
        phone         => $raw_contact->[12][4],
        ext           => $raw_contact->[12][5],
        # phone         => $phone, 
        city          => $raw_contact->[10][2],
        country       => $raw_contact->[8 ][2],
        name          => $raw_contact->[16][1],
        contactperson => $raw_contact->[19][1] || undef,
        street        => $raw_contact->[22][1] || undef,
        house         => $raw_contact->[22][3] || undef,
        build         => $raw_contact->[22][4] || undef,
        apart         => $raw_contact->[22][5] || undef,
        worktime      => $worktime,
        
        contact_email => $raw_contact->[16][8] || undef,
        im_client     => $raw_contact->[19][8] || undef,
        im_login      => $raw_contact->[19][10] || undef,
        extra_message => $raw_contact->[22][8] || undef,
        ogrn          => $raw_contact->[12][8],
    };
    # клиент загрузил файл со старым шаблоном (нет ячейки для ОГРН) - не затираем существующий
    delete $contact_data->{ogrn} unless $raw_contact->[11][8];

    my $without_plus = defined $contact_data->{country_code} && defined $contact_data->{city_code} && (
           ( $contact_data->{country_code} == 8 && ($contact_data->{city_code} == 800 || $contact_data->{city_code} == 804 ))
        || ( $contact_data->{country_code} == 0 && $contact_data->{country_code} == 8 ) );

    if (defined $contact_data->{country_code} && $contact_data->{country_code} !~ /^\+/ and $contact_data->{country_code} ne '' && !$without_plus) {
        $contact_data->{country_code} = "+$contact_data->{country_code}";
    }
            
    smartstrip($_) for grep {defined $_} values %$contact_data;
    
    return $contact_data;
}

=head2 read_excel($type, $file)

    Постранично прочитать содержимое xls[x] файла
    
    @worksheet = read_excel(xls => $file)
    
    Параметры:        
        $type - тип excel файла (xls|xlsx)
        $file - содержимое файла
        
        для $type == xls параметр $file должен быть строкой с содержимым файла
        для $type == xlsx параметр $file - путь к файлу .xlsx
    
    Результат:
        @worksheet - массив всех прочитанных страниц из excel
            каждый элемент массива это страница в excel файле
                вложенный массив - построчное содержимое excel страницы

=cut

sub read_excel {
    
    my ($type, $file_content) = @_;
    
    my @worksheets;
    if ($type eq 'xls') {
        my $book = Spreadsheet::ParseExcel::Workbook->Parse(\$file_content, Yandex::Spreadsheet::ParseExcel::FmtUTF8->new());
        @worksheets = map { _read_xls_worksheet($_) } $book->worksheets;
    } elsif ($type eq 'xlsx') {
        my $r = Excel::Reader::XLSX->new;
        my $book = $r->read_file($file_content);
        @worksheets = map { _read_xlsx_worksheet($_) } $book->worksheets;
    }
    return @worksheets;
}

=head2 _read_xls_worksheet

    load raw data from excel
    my $data = _read_xls_worksheet($worksheet);
    
    $worksheet is Spreadsheet::ParseExcel::Worksheet
    
    result
        $data - arrayref, readed data from worksheet

=cut

sub _read_xls_worksheet
{
    my $oWkS = shift;

    my @data;
    for (my $iR = $oWkS->{MinRow}; defined $oWkS->{MaxRow} && $iR <= $oWkS->{MaxRow}; $iR++) {
        for (my $iC = $oWkS->{MinCol}; defined $oWkS->{MaxCol} && $iC <= $oWkS->{MaxCol}; $iC++) {
            my $oWkC = $oWkS->{Cells}[$iR][$iC];
            if (defined $oWkC) {
                # числовые ячейки получаем без учёта формата числа
                # иначе могут получаться всякие непонятности вроде 5,555, которые мы не умеем обрабатывать 
                # "For example a number such as 40177 might be formatted as 40,117, 40117.000 or even as the date 2009/12/30" (c) perldoc Spreadsheet::ParseExcel::Cell
                $data[$iR][$iC] = ($oWkC->type() eq 'Numeric') ? $oWkC->unformatted() : $oWkC->value();
            }
        }
    }
    
    return \@data;
}

=head2 _read_xlsx_worksheet

    load raw data from MS Excel 2007+
    
    result
        $data - arrayref, readed data from worksheet

=cut

sub _read_xlsx_worksheet {
    
    my $worksheet = shift;
    
    my @data;
    while (my $row = $worksheet->next_row) {
        while (my $cell = $row->next_cell) {
            $data[$cell->row][$cell->col] = $cell->value; 
        }
    }

    return \@data;
}

sub _get_internal_value_by_text {
    my $text = shift || '';

    my $dict = get_header_dict(Yandex::I18n::current_lang());

    my @texts = split /\s*,\s*/, $text;
    my @result = map {@{$device_type_targeting_fields_to_keys{$_} || $network_targeting_fields_to_keys{$_} || []}} grep {defined} map {$dict->{$_}} @texts;
    return \@result;
}

1;
