package YaCatalogApi;

# $Id$

=head1 NAME
    
    YaCatalogApi
    Функции по работе с рубриками Я.Каталога

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use Yandex::Clone qw/yclone/;

use YaCatalogLite;

use List::Util qw/max/;

use Yandex::HashUtils;
use Yandex::I18n;

use utf8;

use base qw/Exporter/;

our @EXPORT = qw/
            get_category_url
            get_category_path
            get_category_name
            get_category_level
            is_category_hidden
            add_children_to_categories
            process_categories
            get_filtered_categories
            validate_rubrics
            validate_one_rubric
/;

=head2 get_category_path

=cut
sub get_category_path {
    my $id = shift;
    my $cat = $YaCatalogLite::Trees{cat};
    return defined $cat->{ $id }
            ? [@{$cat->{ $id }->{path}}]
            : [];
}


=head2 get_category_url

    Получить урл рубрики по id

=cut
sub get_category_url {
    my $id = shift;
    my $cat = $YaCatalogLite::Trees{cat};

    my $BASE_CATALOG_URL = 'http://yaca.yandex.ru/yca/';

    return defined $cat->{ $id }
            ? $BASE_CATALOG_URL.join( '/',
                    map { $cat->{ $_ }->{url_name} }
                    grep { defined $_ }
                    @{ $cat->{ $id }->{path} }, $id
                ).'/'
            : '';
}

=head2 get_category_name

    Получить полное название категории из каталога по id

=cut
sub get_category_name {
    my $id = shift;

    my $catalog_key_name = {ru => 'name'
                            , en => 'short_engl_name'
                            # , ua => 'short_name_ua' # пока нет переводов
                           }->{$Yandex::I18n::cur_lang || 'ru'} || 'name';

    my $cat = $YaCatalogLite::Trees{cat};
    return defined $cat->{ $id }
            ? join( ' / ',
                    map { $cat->{ $_ }->{ $catalog_key_name } || $cat->{ $_ }->{name} || ''}
                    grep { $_ }
                    @{ $cat->{ $id }->{path} }, $id
                )
            : 'удалена';
}

=head2 is_category_hidden

    Скрыта ли рубрика (hide=1)

=cut
sub is_category_hidden
{
    my $id = shift;

    return defined $YaCatalogLite::Trees{cat}->{$id}
           ? $YaCatalogLite::Trees{cat}->{$id}->{hide}
           : 1;
}

=head2 get_category_level

    Получить уровень категории из каталога

=cut
sub get_category_level {
    my $id = shift;
    my $cat = $YaCatalogLite::Trees{cat};
    return defined $cat->{ $id } ? scalar( @{ $cat->{ $id }->{path} } ) : -1;
}

=head2 add_children_to_categories 

    Добавить массив потомков к каждой категории

=cut

sub add_children_to_categories($){
    my $cat_hash = shift;
    while(my($id, $cat) = each %$cat_hash) {
        my $parent_id = $cat->{path}[-1];
        next if !defined $parent_id;
        push @{$cat_hash->{$parent_id}->{childs}}, $id;
    }
}

=head2 process_categories
    
    добавляем уровень вложенности и глубину дерева для каждой категории

=cut

sub process_category {
    my ($cat_hash, $id, $lvl) = @_;
    $lvl||=0;
    my $cat = $cat_hash->{$id};
    
    $cat->{id} =$id;
    $cat->{lvl}=$lvl;
    my $max_depth=0;
    for my $ch_id (@{$cat->{childs}}) {
        my $ch_cat = $cat_hash->{$ch_id};
        $max_depth = max $max_depth, process_category($cat_hash, $ch_id, $lvl+1);
    }
    $cat->{childs} = [ sort {lc($cat_hash->{$a}{name}) cmp lc($cat_hash->{$b}{name})} @{$cat->{childs}}];
    return ($cat->{depth} = 1+$max_depth);
}

sub process_categories($){
    my($cat_hash) = @_;
    return process_category($cat_hash, 0, 0);
}

=head2 get_filtered_categories

    Возвращает отфильтрованный список категорий

=cut

sub get_filtered_categories
{
    my $categories_hash = yclone($YaCatalogLite::Trees{cat});

    # фильтруем пусте категории
    $categories_hash = hash_grep { scalar keys %$_ } $categories_hash;

    # Модифицируем дерево категорий для упрощения навигации
    add_children_to_categories($categories_hash);

    # Считаем глубину и уровень вложенности
    process_categories($categories_hash);

    _process_good_categories($categories_hash, 112);   # Работа

    my %hidden_rubrics;
    while( my ( $id, $cat ) = each %$categories_hash ) {
        $hidden_rubrics{$id} = 1 if $cat->{hide};
    }    

    while (my($id, $cat) = each %$categories_hash) {
        next unless defined $cat->{lvl};
        $cat->{checkable} = $cat->{lvl} > 2  || $cat->{is_good} ? 1 : 0;
        $cat->{hidden} = ($cat->{depth} + $cat->{lvl} < 4 && !$cat->{is_good}) ? 1 : 0;
        foreach (@{$cat->{path}}) {
            if ($hidden_rubrics{$_}) {
                $cat->{hidden_parent} = 1;
            }
        }
        $cat->{url} = get_category_url( $id );
    }

    while (my ($id, $cat) = each %$categories_hash) {
        $cat->{childs} = [ grep { !_is_hidden_category($categories_hash->{$_}) } @{$cat->{childs}}];
    }

    $categories_hash = hash_grep { !_is_hidden_category($_) } $categories_hash;

    return $categories_hash;
}

=head2 validate_rubrics

    проверяем наличие категории в каталоге
    На вход получает массив хэшей рубрик, из элементов которого нас в данный момент интересует только id
    Возвращает массив ошибок

=cut

{
my $categories_hash = undef;

sub validate_rubrics
{
    my ($rubrics, $old_rubrics) = @_;

    return unless $rubrics && @$rubrics;
    
    my @errors;
    if (!$categories_hash) {
        $categories_hash = get_filtered_categories();
    }

    foreach (@$rubrics) {
        if (! $categories_hash->{$_->{id}} && !$old_rubrics->{$_->{id}} ) {
            push @errors, iget('Невозможно добавить рубрику "%s", рубрики с таким id не существует.', $_->{id});
        }

        if ($categories_hash->{$_->{id}} && ! $categories_hash->{$_->{id}}->{checkable} && !$old_rubrics->{$_->{id}} ) {
            push @errors, iget('Невозможно добавить рубрику "%s".', $categories_hash->{$_->{id}}->{name});
        }
    }

    return @errors;
}

}

sub validate_one_rubric
{
    my ($rubric_id) = @_;
    my @res = validate_rubrics([{id => $rubric_id}], {});
    return $res[0];
}

=head2 _process_good_categories(\%cat, $id)

    Выставляет соответствующий флаг категориям, с переданным id

=cut

sub _process_good_categories {
    my ($cat_hash, $cat) = @_;

    $cat_hash->{$cat}->{is_good}=1;
    for my $ch_cat (map {$cat_hash->{$_} } @{$cat_hash->{$cat}->{childs}}) {
        _process_good_categories($cat_hash, $ch_cat->{id});        
    }
    return $cat_hash;
}

=head2 _is_hidden_category

    Скрытая ли категория

=cut

sub _is_hidden_category($) {
    my $cat = shift;
    return $cat->{hidden} || $cat->{hide} || $cat->{hidden_parent};
}

1;
