package Yandex::Audience;

# ABSTRACT: Yandex.Audience API client

=head1 SYNOPSIS

    # provide your oauth-token 
    my $api = Yandex::Audience->new(token => $token);


=head1 DESCRIPTION

Client for Yandex.Audience API.

API methods are mapped to internal object methods.

See api docs for parameters and response formats at
https://tech.yandex.ru/audience/doc/concept/about-docpage/

=cut

use Direct::Modern;

use Mouse;

use Path::Tiny;
use List::Util qw/first/;
use LWP::UserAgent;
use JSON;

use Yandex::HTTP;

use Log::Any '$log';


our $AUDIENCE_API_URL ||= 'https://api-audience.yandex.ru';
our $AUDIENCE_API_TOKEN;
our $AUDIENCE_API_TOKEN_FILE;

# базовый id для целей сегментов Аудиторий;
# $goal_id = $segment_id + $SEGMENT_GOAL_ID_SHIFT
# https://wiki.yandex-team.ru/jurijjgalickijj/Raznoe/goalid/
our $SEGMENT_GOAL_ID_SHIFT = 2500000000;


=attr token

Application token.

Can be provided directly or got from file.

Default is in $AUDIENCE_API_TOKEN

=cut

has token => (
    is => 'rw',
    isa => 'Str',
    lazy_build => 1,
    clearer => '_clear_token',
);


=attr token_file

File where the token should be taken from.

Default is in $AUDIENCE_API_TOKEN_FILE

=cut

has token_file => (
    is => 'rw',
    isa => 'Str',
    trigger => sub { shift()->_clear_token() } ,
);


sub _build_token {
    my ($self) = @_;

    my $token = $AUDIENCE_API_TOKEN;
    if ( my $file = $self->token_file || $AUDIENCE_API_TOKEN_FILE ) {
        $log->debug("Obtaining token from $file");
        $token = $self->_get_key_from_file($file);
    }
    croak 'Yandex.Audience API token is not provided'  if !$token;
    return $token;
}

sub _get_key_from_file {
    my ($self, $file) = @_;

    # first non-empty not commented string
    my $str = first {/^[^#]/} map {s/\s+$//r} map {s/^\s+//r} path($file)->lines;
    return $str;
}


=head1 INTERNAL METHODS

Internal methods with prefix _call_ is created for every record in %METHOD_ALIAS.
Return decoded response.

    my $respond = Yandex::Audience->new()->_call_get_experiments();

=cut

our %METHOD_ALIAS = (
    create_experiment   => [ POST => '/v1/management/experiments' ],
    get_experiments     => [ GET => '/v1/management/experiments' ],
    delete_experiment   => [ DELETE => '/v1/management/experiment/{experiment_id}' ],
    update_experiment   => [ PUT => '/v1/management/experiment/{experiment_id}' ],

    get_experiment_grants    => [ GET => '/v1/management/experiment/{experiment_id}/grants' ],
    set_experiment_grant     => [ PUT => '/v1/management/experiment/{experiment_id}/grant' ],
    delete_experiment_grant  => [ DELETE => '/v1/management/experiment/{experiment_id}/grant' ],

    # todo: define more methods here
);


=head2 new

    # provide token...
    my $api = Yandex::Audience->new(token => $token );

    # ... or token file path...
    my $api = Yandex::Audience->new(token_file => '/path/to/token');

    # ... or set variable preliminarily
    $Yandex::Audience::AUDIENCE_API_TOKEN_FILE = $token_file;
    my $api = Yandex::Audience->new();

Constructor

=head2 call

    my $result = $api->call(GET => $url, %params);
    my $result = $api->call(POST => $url, %params, $payload_data);

API method call

=cut

sub call {
    my $self = shift;

    my $http_method = shift;
    my $url_template = shift;
    my $payload_data = (@_ % 2) ? pop : undef;
    my %get_param = @_;

    my $url = $AUDIENCE_API_URL . $url_template;
    # resolve templates
    my $_take_away = sub {
        my ($param_name) = @_;
        my $param = delete $get_param{$param_name};
        if (!defined $param) {
            local $Carp::CarpLevel = $Carp::CarpLevel + 1;
            croak "Required parameter $param_name not provided for method $url_template";
        }
        return $param;
    };
    $url =~ s/\{(\w+)\}/$_take_away->($1)/ge;

    # concatenate url-params
    if (%get_param) {
        $url = Yandex::HTTP::make_url($url, \%get_param);
    }

    my %header = (
        "Authorization" => "OAuth " . $self->token(),
    );
    $log->trace("Call: $http_method $url")  if $log->is_trace();

    my $response;
    if ($http_method eq 'GET') {
        $response = LWP::UserAgent->new()->get($url, %header);
    } elsif ($http_method eq 'DELETE') {
        $response = LWP::UserAgent->new()->delete($url, %header);
    } elsif ($http_method eq 'POST' || $http_method eq 'PUT') {
        croak "Payload for $http_method is not provided"  if !$payload_data;
        my $payload = ref $payload_data ? encode_json($payload_data) : $payload_data;
        $log->trace("Payload: $payload")  if $log->is_trace();
        $header{"Content-Type"} = "application/json";
        if ($http_method eq 'POST') {
            $response = LWP::UserAgent->new()->post($url, %header, Content => $payload);
        } elsif ($http_method eq 'PUT') {
            $response = LWP::UserAgent->new()->put($url, %header, Content => $payload);
        }
    } else {
        croak "Unknown method $http_method";
    }

    # die on communication 5xx errors
    if ($response->code =~ /^5/) {
        my $msg = join ' ' => $response->code, $response->message;
        $log->trace("Error: $msg")  if $log->is_trace();
        die "Audience API call failed: $msg";
    }

    my $resp_content = $response->decoded_content();
    $log->trace("Response: $resp_content")  if $log->is_trace();

    my $result = decode_json $resp_content;
    return $result;
}


=head2 create_experiment

    my $experiment_params = {
        name => "A/B test",
        counter_ids => $counters,
        segments => [
            {name => "A", start => 0,  end => 50},
            {name => "B", start => 50, end => 100},
        ],
    }
    my $new_experiment = $api->create_experiment($experiment_params);

Create new experiment

=cut

sub create_experiment {
    my $self = shift;
    my ($experiment_params, $ulogin) = @_;

    my $result = $self->_call_create_experiment( ulogin => $ulogin, {experiment => $experiment_params} );
    die "Experiment creation failed: $result->{message}"  if $result->{errors};

    return $result->{experiment};
}


=head2 set_experiment_grant

    my $grant = $api->set_experiment_grant($experiment_id, $login, $grant);

Provide access to experiment. Default grant is 'view'

=cut

sub set_experiment_grant {
    my $self = shift;
    my ($experiment_id, $login, $grant) = @_;

    my $grant_request = {
        user_login => $login,
        permission => $grant || "view",
    };
    my $result = $self->_call_set_experiment_grant(experiment_id => $experiment_id, {grant => $grant_request});
    die "Grant setting failed: $result->{message}"  if $result->{errors};

    return $result->{grant};
}


# mapping object internal methods to external API calls
for my $alias (keys %METHOD_ALIAS) {
    my $sub = sub {
        my $self = shift;
        return $self->call(@{$METHOD_ALIAS{$alias}}, @_);
    };
    __PACKAGE__->meta->add_method("_call_$alias" => $sub);
}


__PACKAGE__->meta->make_immutable();

1;
