package Yandex::CallerStack;

use strict;
use warnings;

use Devel::StackTrace;
use Carp;
use JSON;

sub new {
    my $pkg = shift;

    my $self = bless {
        serialize  => JSON->new(),
        max_level  => 100,
        skip_level => 4,
        fields     => [qw/
            package 
            filename 
            line 
            subroutine 
            hasargs 
            wantarray 
            evaltext 
            is_require 
            hints 
            bitmask 
            args
        /],
        @_
    }, $pkg;

    return $self;
}

sub trace {
    my $self = shift;

    my $trace = Devel::StackTrace->new(); 
    for ( 0..$self->{skip_level}-1 ) { $trace->next_frame };

    my $max_l = $self->{max_level};
    my @stack; 
    while ($max_l-- and my $f = $trace->next_frame) {
        if ($f->subroutine ne '(eval)') {
            my %_r;
            for my $p ( @{ $self->{fields} } ) {
                my $method_name = 
                    $self->can( "_field_$p" ) 
                        ? "_field_$p" 
                        : "_field"
                    ;
                $self->$method_name( frame=>$f,  field=>$p, hash=>\%_r );
            }
            $_r{pid} = $$;
            push @stack, \%_r;
        }
    }


    return $self->{serialize}->encode( \@stack );
}

sub _field {
    my $self  = shift;
    my %args  = @_;

    my $frame = delete $args{frame};
    my $field = delete $args{field};
    my $hash  = delete $args{hash};

    if ( my $r = $frame->$field ) {
        $hash->{ $field } = $r;
    };
}

sub _field_args {
    my $self  = shift;
    my %args  = @_;

    my $frame = delete $args{frame};
    my $hash  = delete $args{hash};

    $hash->{args}=[map {Carp::format_arg($_)} $frame->args] if ( $frame->hasargs );
}

sub _field_bitmask {
    my $self  = shift;
    my %args  = @_;

    my $frame = delete $args{frame};
    my $hash  = delete $args{hash};

    $hash->{bitmask}=$self->bstream2dec($frame->bitmask)
}

sub bstream2dec { shift; 
    return unpack("N", pack("B32", substr("0" x 32 . unpack("B*", shift), -32))) 
}


1;
