package Yandex::DBQueue;
use Direct::Modern;

=head1 NAME

Yandex::DBQueue

=head1 DESCRIPTION

Объект для работы с очередью заданий. Один объект работает с одной базой и с заданиями одного типа.
Обычно потребителям надо работать с заданиями одного типа. Если окажется не так, интерфейс здесь надо будет поменять.
Исключение: mark_finished_and_replace_with на объекте Yandex::DBQueue::Job может ставить задания другого типа.

Точка входа. Чтобы начать работать с очередью заданий, надо создать объект и вызвать какую-то функцию из него.
поставить задание: вызвать insert_job
получить/захватить задание: grab_jobs
узнать информацию про очередь: count_jobs, find_jobs

=head1 SYNOPSIS

    ## процесс в веб-сервере понимает, что какую-то работу надо выполнить асинхронно, ставит задание в очередь
    my $ClientID = 123;
    my $job_type = 'eat_apple';
    my $queue = Yandex::DBQueue->new( PPC( ClientID => $ClientID ), $job_type );
    my $job_id = get_new_id('job_id');

    my $job = $queue->insert_job( {
        job_id => get_new_id('job_id'),
        ClientID => $client_id,
        args => { apple_id => 18110 },
        priority => 123,
    } );

    ## процесс скрипта работает над заданиями: обрабатывает одно, потом второе, потом третье
    # обязательно надо передать объект Yandex::Log, чтобы DBQueue писал, что он делает
    my $log = Yandex::Log->new( ... );
    $Yandex::DBQueue::LOG = $log;

    my $queue = Yandex::DBQueue->new( PPC( shard => 4 ), $job_type );
    while (!should_quit()) {
        my $job = $queue->grab_job();

        if ($job) {
            # как работать над работой, пример дальше
            work_on_job($job);
        } else {
            # работы сейчас нет, надо подождать, прежде чем спросим базу ещё раз
            sleep 1;
        }

        # между работой может потребоваться сделать ещё что-то
        # например, уведомить мониторинги, что скрипт работает
        ...;
    }

    ## в процессе скрипта задание обрабатывается
    sub work_on_job {
        my ($job) = @_;

        warn ref $job; # Yandex::DBQueue::Job -- там можно посмотреть список методов

        # больше 17 раз не пробуем
        #
        # trycount стоит проверять дважды: один раз перед началом работы, второй раз после ошибки
        # второй раз: больше информации, что пошло не так
        # первый раз: не всегда можно перехватить ошибку (segfault в библиотеке, например), надо гарантировать,
        #             что задание будет обработано не больше N раз
        #
        # текущая попытка в trycount уже засчитана
        #
        # TODO: если будет фреймворк для скриптов-обработчиков, двойную проверку стоит делать там
        if ( $job->trycount > 17 ) {
            # больше 17 раз не пробуем
            # попытка, которую делаем сейчас, в trycount уже засчитана
            $job->mark_failed_permanently( { error => "неизвестная проблема" } );
            return;
        }

        my $result = eval { eat_apple($job->args->{apple_id} ) };
        my $error = $@;

        if ($error) {
            # больше 17 раз не пробуем -- вторая поверка
            if ( $job->trycount >= 17 ) {
                $job->mark_failed_permanently( { error => $error } );
                return;
            }

            $job->mark_failed_once();
            return;
        }

        $job->mark_finished( { result => $result } );
    }

    ## в каком-то процессе надо сделать ограничение: клиент не может поставить больше 13 заданий
    # см. также комментарий в описании count_jobs про race condition
    if ( $queue->count_jobs( ClientID => 3 ) >= 13 ) {
        die "Too many tasks in the queue";
    }

    ## скрипт мониторинга отправляет числа про очередь (возраст, количество заданий) в мониторинг
    use List::Util 'max';
    my $stat = $queue->get_queue_statistics();
    monitor_values( { 'active_age' => max( $stat->{New}->{max_age}, $stat->{Grabbed}->{max_age} ) } );

    ## скрипт обслуживания удаляет старые задания, не нужны они больше
    my $lifetime_in_seconds = 24 * 60 * 60;
    $queue->delete_old_jobs($lifetime_in_seconds);
    # вызов удалил записи из базы -- у некоторых задач есть связанные файлы в MDS, их
    # стоит удалять отдельно вызовом remove_old_files в Direct::Storage

=head1 TODO

Юнит-тесты
- создать задание, проверить, что вернулось то, что создавали
- получить задание, "выполнить", сказать, что завершено
- получить задание два раза -- если в базе одно задание, второй раз ничего не вернётся
- typemap

Подумать, отчуждается ли от Директа

=cut

use Sys::Hostname 'hostname';
use Carp;

use Yandex::DBTools;
use Yandex::Overshard;
use Yandex::DBQueue::Job;
use Yandex::DBQueue::Typemap;
use Yandex::HashUtils;
use Yandex::Validate;

use JSON;

=head1 SUBROUTINES/METHODS/VARIABLES

=head2 $Yandex::DBQueue::LOG

Объект Yandex::Log, куда пишутся сообщения, что происходит в функциях, которые вызываются из скриптов:
* grab_jobs
* fill_job_types

Скрипты-обработчики передают туда свой объект Yandex::Log, так что DBQueue пишет лог в общий лог скрипта.
Если эта переменная не установлена, вызов функции упадёт.

=cut

our $LOG;

=head2 $Yandex::DBQueue::DEFAULT_GRAB_FOR

Как надолго по умочанию надо блокировать задачу, в секундах. Когда это время пройдёт, задачей
сможет заняться другой процесс.

В метод grab_jobs параметром grab_for можно передать другое значение, если значение
по умолчанию не подходит.

=cut

our $DEFAULT_GRAB_FOR ||= 15 * 60;

=head2 $Yandex::DBQueue::MAX_GRABBED_BY_LENGTH

Максимальная длина значения grabbed_by в базе -- на стороне клиента проверяется, что
что-то более длинное мы туда записать не пробуем.

=cut

our $MAX_GRABBED_BY_LENGTH = 40;

=head2 $Yandex::DBQueue::LOG_ARGS_LENGTH

При журналировании args, сериализованный вид которых превосходит LOG_ARGS_LENGTH по длине,
будут обрезаться до LOG_ARGS_LENGTH символов, после чего к ним будет приписана длина до обрезания

=cut

our $LOG_ARGS_LENGTH = 1000;

=head2 $job_fields

Поля, которые выбираются из таблицы dbqueue_jobs для формирования объекта Yandex::DBQueue::Job, записанные через запятую

=cut

my $job_fields = join(',', @Yandex::DBQueue::Job::JOB_FIELDS);

=head2 $job_archive_fields

Поля, которые выбираются из таблицы dbqueue_job_archive для формирования объекта Yandex::DBQueue::Job, записанные через запятую

=cut

my $job_archive_fields = join(',', @Yandex::DBQueue::Job::JOB_ARCHIVE_FIELDS);

=head2 Yandex::DBQueue->new( $db, $job_type )

Конструктор: надо вызывать снаружи, чтобы начать работать с DBQueue.

Параметры:
$db -- БД, которая подходит для запросов Yandex::DBTools
$job_type -- тип, зарегистрированный в typemap

=cut

sub new {
    my ( $class, $db, $job_type ) = @_;

    die 'missing required parameter: db' unless $db;
    die 'missing required parameter: job_type' unless $job_type;

    my $typemap = Yandex::DBQueue::Typemap->new($db);
    my $job_type_id = $typemap->type_to_id($job_type);

    return bless {
        _db => $db,
        _typemap => $typemap,
        _job_type => $job_type,
        _job_type_id => $job_type_id,
    }, $class;
}

=head2 $queue->grab_job(%opt)

grab single job

see grab_jobs

=cut

sub grab_job
{
    my ($self, %opt) = @_;
    return $self->grab_jobs(%opt, limit => 1);
}

=head2 $queue->grab_jobs(%opts)

Захватить задачу в очереди:
- записать в базу, что этот процесс с ней работает
- записать в базу, что другим процессам не надо брать эту задачу какое-то время
- записать в базу, что эту задачу пытались сделать ещё один раз (trycount++)

Что может быть в %opts:
- grab_for -- как надолго блокировать задачу, чтобы другой процесс не стал ей заниматься;
              если нет, используется $DEFAULT_GRAB_FOR
- limit    -- сколько записей выбрать. По-умолчанию - 1

В будущем в %opts можно будет задать параметры выборки, чтобы выбиралась не любая задача указанного типа.

TODO: когда в %opts будут параметры выборки, для них стоит сделать подструктуру:
$queue->grab_jobs( grab_for => 900, filters => { minimum_priority => 7 } );

=cut

sub grab_jobs {
    my ( $self, %opts ) = @_;

    my $LIMIT = $opts{limit} // 1;

    unless (is_valid_int($LIMIT, 1)) {
        croak "limit must be a positive integer";
    }

    my $tail = ':' . $$;
    my $hostname = hostname();
    my $global_process_id = $hostname . $tail;
    if (length($global_process_id) > $MAX_GRABBED_BY_LENGTH) {
        $global_process_id = substr($global_process_id, 0, $MAX_GRABBED_BY_LENGTH);
        substr($global_process_id, - length($tail), length($tail), $tail); # приписываем $$ ровно в конец
        die "global_process_id too long: $global_process_id" if length($global_process_id) > $MAX_GRABBED_BY_LENGTH;
    }

    my $global_process_id_quoted = sql_quote($global_process_id);

    $LOG->out( "trying to grab a jobs", { db => $self->{_db}, job_type => $self->{_job_type}, %opts } );

    my $grab_for = $opts{grab_for} // $DEFAULT_GRAB_FOR;
    unless ( is_valid_int( $grab_for, 1 ) ) {
        croak "grab_for must be a positive integer";
    }

    my $minimum_priority = (exists $opts{filters} && exists $opts{filters}->{minimum_priority}) ? $opts{filters}->{minimum_priority} : 0;

    my $client_id = (exists $opts{filters} && exists $opts{filters}->{ClientID} ? $opts{filters}->{ClientID} : 0);

    my $namespace = $Yandex::DBQueue::Job::NAMESPACE;

    my $affected_cnt = int do_sql( $self->{_db}, [
        qq{
            UPDATE dbqueue_jobs
            SET
                status = 'Grabbed',
                grabbed_by = $global_process_id_quoted,
                grabbed_at = NOW(),
                grabbed_until = NOW() + INTERVAL $grab_for SECOND,
                trycount = trycount + 1
        },
        WHERE => {
            job_type_id__int => $self->{_job_type_id},
            ( defined $namespace ? ( namespace => $namespace ) : ( namespace__is_null => 1 ) ),
            _OR => { status => 'New', grabbed_until__lt__dont_quote => 'NOW()' },
            ( $minimum_priority ? ( priority__ge => $minimum_priority ) : () ),
            ( $client_id ? (ClientID => $client_id) : () ),
        },
        'ORDER BY job_id',
        LIMIT => $LIMIT,
    ] );

    $LOG->out( { affected_cnt => $affected_cnt } );

    return unless $affected_cnt;

    my $job_rows = get_all_sql( $self->{_db},
        [
            qq{ SELECT $job_fields FROM dbqueue_jobs },
            WHERE => {
                job_type_id => $self->{_job_type_id},
                grabbed_by => $global_process_id,
            },
            LIMIT => $affected_cnt,
        ],
    );

    if (@$job_rows != $affected_cnt) {
        die "Fetched ".(scalar @$job_rows)." rows, but expected $affected_cnt";
    }

    my @jobs;
    for my $job_row (@$job_rows) {
        my $job = Yandex::DBQueue::Job->new_from_db_row( $self->{_db}, $self->{_typemap}, $job_row );

        my $job_copy = hash_cut($job, @Yandex::DBQueue::Job::JOB_FIELDS, '_db');
        if (exists $job_copy->{args}) {
            my $json_args = to_json($job_copy->{args}, {canonical => 1, allow_nonref => 1});

            my $len = length($json_args // '');
            if ($len > $LOG_ARGS_LENGTH) {
                $job_copy->{args} = substr($json_args, 0, $LOG_ARGS_LENGTH) . " [$len]";
            }
        }
        $LOG->out({job => $job_copy});
        
        push @jobs, $job;
    }

    return wantarray ? @jobs : $jobs[0];
}

=head2 $queue->count_jobs( ClientID => 17 )

Возвращает число -- количество задач.

ClientID -- необязательный параметр.

Задачи в dbqueue_job_archive не считаются.

Если использовать функцию, чтобы реализовать ограничение "у клиента не больше N задач в очереди", может быть
race condition: несколько процессов, которые работают одновременно, могут поставить задачи сверх ограничения.
Если race condition надо исключить, можно использовать средства вне DBQueue, например, sql_lock_guard
в Yandex::DBQueue.

=cut

sub count_jobs {
    my ( $self, %opts ) = @_;

    my $ClientID = $opts{ClientID};
    return int ($self->get_client_statistics($ClientID, uid => $opts{uid}, no_archive => 1)->{$ClientID}->{count} // 0);
}

=head2 $queue->find_jobs( ClientID => 17 )

Выбирает из базы задания и возвращает arrayref, в котором объекты Yandex::DBQueue::Job.

ClientID, status__not_in, job_id -- необязательные параметры.

skip_archived => 1 -- не выбирать архивные задания
limit => N, offset => N -- лимит, оффсет

Ищет задачи и в dbqueue_jobs, и в dbqueue_job_archive.

=cut

sub find_jobs {
    my ( $self, %opts ) = @_;

    my $ClientID = $opts{ClientID};
    my $uid = $opts{uid};
    my $status = $opts{status__not_in};
    my $job_id = $opts{job_id};

    return [] if $job_id && !@$job_id;

    my $where_cond = {
        job_type_id => $self->{_job_type_id},
        $ClientID ? ( ClientID => $ClientID ) : (),
        $uid ? ( uid => $uid ) : (),
        $status ? ( status__not_in => $status ) : (),
        $job_id ? ( job_id => $job_id ) : (),
    };

    my $job_rows = 
    overshard limit => $opts{limit}, offset => $opts{offset}, order => 'job_id',
    get_all_sql( $self->{_db}, [
        "SELECT $job_fields, NULL AS result, 0 AS job_in_archive FROM dbqueue_jobs", WHERE => $where_cond,
        ( $opts{skip_archived}
          ? () 
          : (
            'UNION ALL',
            "SELECT $job_archive_fields, 1 AS job_in_archive FROM dbqueue_job_archive", WHERE => $where_cond,
          ),
        )
    ] );

    my @jobs;
    for my $job_row (@$job_rows) {
        my $job_in_archive = delete $job_row->{job_in_archive};
        delete $job_row->{result} unless $job_in_archive;

        push @jobs, Yandex::DBQueue::Job->new_from_db_row( $self->{_db}, $self->{_typemap}, $job_row );
    }

    return \@jobs;
}

=head2 $queue->find_job_by_id($job_id)

=cut

sub find_job_by_id {
    my ( $self, $job_id ) = @_;

    my $where_cond = { job_id => $job_id, job_type_id => $self->{_job_type_id} };

    my $db_row =
        get_one_line_sql( $self->{_db}, [ "SELECT $job_fields FROM dbqueue_jobs", WHERE => $where_cond ] ) ||
        get_one_line_sql( $self->{_db}, [ "SELECT $job_archive_fields FROM dbqueue_job_archive", WHERE => $where_cond ] );

    return undef unless $db_row;
    return Yandex::DBQueue::Job->new_from_db_row( $self->{_db}, $self->{_typemap}, $db_row );
}

=head2 $queue->insert_job($job_hash)

Внутри $job_hash данные задачи, которые попадут в таблицу, кроме указания на тип.

Возвращает объект Yandex::DBQueue::Job с созданной задачей.

=cut

sub insert_job {
    my ( $self, $job_hash ) = @_;

    die "redundant field: job_type"    if exists $job_hash->{job_type};
    die "redundant field: job_type_id" if exists $job_hash->{job_type_id};

    # копируем входные данные, которые передали по ссылке
    my %job_hash_copy = %$job_hash;

    $job_hash_copy{namespace} =
        exists $job_hash_copy{namespace} ? $job_hash_copy{namespace}
                                         : $Yandex::DBQueue::Job::NAMESPACE;

    $job_hash_copy{job_type}    = $self->{_job_type};
    $job_hash_copy{job_type_id} = $self->{_job_type_id};

    my $job = Yandex::DBQueue::Job->new( $self->{_db}, $self->{_typemap}, \%job_hash_copy );

    do_insert_into_table( $self->{_db}, 'dbqueue_jobs', $job->as_db_row );

    return $job;
}

=head2 $queue->get_queue_statistics()

Возвращает структуру:
    {
        $status => { max_age => 123, count => 456 },
        $status => { count => 456 },
        ...
    }

max_age -- в секундах.

Из dbqueue_jobs выбирается длина и возраст. Из dbqueue_job_archive выбирается только длина.

=cut

sub get_queue_statistics {
    my ($self) = @_;

    my $job_type_id_quoted = sql_quote( $self->{_job_type_id} );

    my $stat_rows_by_status = {};

    hash_merge $stat_rows_by_status, get_hashes_hash_sql( $self->{_db}, qq{
        SELECT status, count(*) AS queue_length, UNIX_TIMESTAMP() - UNIX_TIMESTAMP(MIN(create_time)) AS queue_age
        FROM dbqueue_jobs
        WHERE job_type_id = $job_type_id_quoted
        GROUP BY status
    } );

    hash_merge $stat_rows_by_status, get_hashes_hash_sql( $self->{_db}, qq{
        SELECT status, count(*) AS queue_length
        FROM dbqueue_job_archive
        WHERE job_type_id = $job_type_id_quoted
        GROUP BY status
    } );

    my $result = {};
    for my $status ( @Yandex::DBQueue::Job::JOB_STATUSES ) {
        my $stat_row = $stat_rows_by_status->{$status} || {};

        $result->{$status} = {};
        $result->{$status}->{count} = $stat_row->{queue_length} // 0;

        if ( !$Yandex::DBQueue::Job::JOB_STATUS_IN_ARCHIVE{$status} ) {
            $result->{$status}->{max_age} = $stat_row->{queue_age} // 0;
        }

    }

    return $result;
}

=head2 get_client_statistics

Получить статистику очереди в разбивке по ClientID
Параметры:
    $client_ids -- $ClientID | [ $clid1, $clid2, ...]
    %opt:
        status => 'New' | [ 'New', 'Grabbed' ],
        uid => $uid
        no_archive => 1 -- не считать статистику по архивным заданиям

Возвращает ссылку на хеш со статистикой:
    {
        $ClientID => { count => 123 },
    }

=cut

sub get_client_statistics {
    my ($self, $client_ids, %opt) = @_;
    my %where = (
        job_type_id => $self->{_job_type_id},
        ClientID => $client_ids,
        ( $opt{uid} ? ( uid => $opt{uid} ) :  () ),
        ( $opt{status} ? ( status => $opt{status} ) : () ),
    );
    my $stat = get_hashes_hash_sql($self->{_db}, [
        "select ClientID, count(*) as `count` from dbqueue_jobs",
        where => \%where,
        "group by ClientID"
    ]);
    
    unless ($opt{no_archive}) {
        my $stat_arch = get_all_sql($self->{_db}, [
            "select ClientID, count(*) as `count` from dbqueue_job_archive",
            where => \%where,
            "group by ClientID"
        ]);
        for my $row (@$stat_arch) {
            $stat->{$row->{ClientID}}{count} += $row->{count};
        }
    }
    return $stat;
}

=head2 $queue->delete_old_jobs($lifetime_in_seconds; %O)

$lifetime_in_seconds должно быть по крайней мере 15.
Если указаны $O{callback} и $O{limit}, то перед удалением передать в функцию по этой ссылке массив из удовлетворяющих условию удаления объектов Yandex::DBQueue::Job,
размер массива и количество соответствующих удаляемых элементов при этом ограничены $O{limit}. Вызов не происходит, если объектов не нашлось.
$O{limit} так же может ограничивать удаление без $O{callback}. В любом случае, удаление идёт в первую очередь из dbqueue_jobs.
$O{status} - указать статус заданий, которые можно удалить

=cut

sub delete_old_jobs {
    my ($self, $lifetime_in_seconds, %O) = @_;

    die "Missing required parameter: lifetime_in_seconds" unless $lifetime_in_seconds;
    die "Invalid lifetime_in_seconds value: $lifetime_in_seconds" unless is_valid_int($lifetime_in_seconds, 15);

    my $deleted_jobs = 0;

    my %where = (
        create_time__lt__dont_quote => "now() - interval $lifetime_in_seconds second",
        job_type_id => $self->{_job_type_id},
        ($O{status} ? ( status => $O{status} ) : () ),
    );

    if (defined $O{callback}) {
        die "Bad callback given" unless ref $O{callback} eq 'CODE';
        if (defined $O{limit}) {
            my ($jobs_to_delete_from_archive, $jobs_for_callback);

            my $jobs_to_delete = get_all_sql($self->{_db}, [
                "select $job_fields from dbqueue_jobs",
                WHERE => \%where, 
                LIMIT => $O{limit}
            ]);
            push @$jobs_for_callback, map {Yandex::DBQueue::Job->new_from_db_row($self->{_db}, $self->{_typemap}, $_)} @$jobs_to_delete;

            if (scalar @$jobs_to_delete < $O{limit}) {
                my $rest_limit = $O{limit} - scalar @$jobs_to_delete;
                $jobs_to_delete_from_archive = get_all_sql($self->{_db}, [
                    "select $job_archive_fields from dbqueue_job_archive",
                    WHERE => \%where,
                    LIMIT => $rest_limit
                ]);
                push @$jobs_for_callback, map {Yandex::DBQueue::Job->new_from_db_row($self->{_db}, $self->{_typemap}, $_)} @$jobs_to_delete_from_archive;
            }

            $O{callback}->($jobs_for_callback) if @$jobs_for_callback;

            if (@$jobs_to_delete) {
                $deleted_jobs += do_delete_from_table($self->{_db},
                    'dbqueue_jobs', where => {job_id => [map {$_->{job_id}} @$jobs_to_delete]}
                );
            }
            
            if (@$jobs_to_delete_from_archive) {
                $deleted_jobs += do_delete_from_table($self->{_db},
                    'dbqueue_job_archive', where => {job_id => [map {$_->{job_id}} @$jobs_to_delete_from_archive]}
                );
            }
        } else {
            die "A set limit is mandatory to use a callback";
        }
    } elsif (defined $O{limit}) {
        $deleted_jobs += do_sql($self->{_db}, [
            'DELETE FROM dbqueue_jobs',
            WHERE => \%where, 
            LIMIT => $O{limit}
        ]);

        if ($deleted_jobs < $O{limit}) {
            $deleted_jobs += do_sql($self->{_db}, [
                'DELETE FROM dbqueue_job_archive',
                WHERE => \%where,
                LIMIT => $O{limit} - $deleted_jobs,
            ]);
        }
    } else {
        $deleted_jobs += do_delete_from_table($self->{_db}, 'dbqueue_jobs', where => \%where);
        $deleted_jobs += do_delete_from_table($self->{_db}, 'dbqueue_job_archive', where => \%where);
    }

    return $deleted_jobs;
}

1;
