package Yandex::DBQueue::Job;
use Direct::Modern;

=head1 NAME

Yandex::DBQueue::Job

=head1 DESCRIPTION

Объект-задание

Потребителям не надо подключать явно этот модуль или самим конструировать объекты Job.

Потребителям надо поставить переменную модуля NAMESPACE, если задания надо привязать к какой-то бете:
    # в SettingsDevTest.pm:
    $Yandex::DBQueue::Job::NAMESPACE = "beta:$BETA_PORT";

Объекты Job конструируются в методах Yandex::DBQueue и отдаются потребителям:
постановщику заданий -- из insert_job
исполнителю заданий -- из grab_jobs
надсмотрщику заданий -- из find_jobs

Что потребителю делать с этими объектами, см. SYNOPSIS в Yandex::DBQueue.

Что внутри объекта: строка таблицы dbqueue_jobs или dbqueue_job_archive как есть, с исключением:
args и result в объекте распакованы из json
job_type из dbqueue_job_types тоже есть
если задача ещё не в dbqueue_job_archive, result нет

Про trycount:
В момент обработки задания trycount включает текущую попытку; там может быть 0, только если задание
только в положили в базу и пока не пробовали обработать

=cut

use parent 'Class::Accessor';

use Encode qw( encode_utf8 decode_utf8 );
use JSON;

use Yandex::Compress;
use Yandex::DBTools;
use Yandex::Retry;

=head1 SUBROUTINES/METHODS/VARIABLES

=head2 @Yandex::DBQueue::Job::JOB_FIELDS

Какие поля в базе есть у активных заданий (которые ещё не сделаны)

=cut

our @JOB_FIELDS = qw(
    job_id
    ClientID
    job_type_id
    namespace
    status
    uid
    args
    priority
    create_time
    expiration_time
    grabbed_by
    grabbed_until
    grabbed_at
    trycount
    run_after
);

=head2 @Yandex::DBQueue::Job::JOB_ARCHIVE_FIELDS

Какие поля в базе есть у сделанных заданий

=cut

our @JOB_ARCHIVE_FIELDS = ( @JOB_FIELDS, qw( result ) );

=head2 @Yandex::DBQueue::Job::JOB_STATUSES

Какие статусы есть у заданий

=cut

our @JOB_STATUSES = qw(
    New
    Grabbed
    Finished
    Failed
    Revoked
);

=head2 %Yandex::DBQueue::Job::JOB_STATUS_IN_ARCHIVE

Означает ли какой-то статус, что задание должно лежать в таблице со сделанными

=cut

our %JOB_STATUS_IN_ARCHIVE = map { $_ => 1 } qw( Finished Failed Revoked );

=head2 $Yandex::DBQueue::Job::MAX_PACKED_ARGS_LENGTH

Максимальная длина значения args в базе -- DBQueue не пытается записать
в базу задания с args, которые туда не влезают, порождает исключение.

=cut

our $MAX_PACKED_ARGS_LENGTH = ( 1 << 24 ) - 1; # там mediumblob

=head2 $Yandex::DBQueue::Job::MAX_PACKED_RESULT_LENGTH

Максимальная длина значения result в базе -- DBQueue не пытается записать
в базу задания с result, которые туда не влезают, порождает исключение.

=cut

our $MAX_PACKED_RESULT_LENGTH = ( 1 << 24 ) - 1; # там mediumblob

=head2 $Yandex::DBQueue::Job::NAMESPACE

Пространство имён: к какому инстансу вызывающей системы относятся задания, с которыми работаем.

Зачем: в production не надо (в production везде NULL); в разработке надо, чтобы отделить друг от друга
задания с разных бет.

Ожидается, что это значение поставят из какого-то модуля конфигурации, например, из SettingsDevTest.

=cut

our $NAMESPACE;

## конструкторы (снаружи вызывать не надо, функции Yandex::DBQueue их вызывают, когда надо)

=head2 Yandex::DBQueue::Job->new( $db, $typemap, $fields )

=cut

sub new {
    my ( $class, $db, $typemap, $fields ) = @_;

    die "db is missing" unless $db;
    die "typemap is missing" unless $typemap;
    die "ClientID field is missing" unless defined $fields->{ClientID};

    die "Must specify either job_type or job_type_id in fields" unless $fields->{job_type} || $fields->{job_type_id};

    # копируем входные данные, которые передали по ссылке
    my %fields_copy = %$fields;

    $fields_copy{job_type}    //= $typemap->id_to_type( $fields_copy{job_type_id} );
    $fields_copy{job_type_id} //= $typemap->type_to_id( $fields_copy{job_type} );

    $fields_copy{status}   //= 'New';
    $fields_copy{args}     //= {};
    $fields_copy{priority} //= 0;

    my $self = bless \%fields_copy, $class;
    $self->{_db} = $db;
    $self->{_typemap} = $typemap;

    return $self;
}

=head2 Yandex::DBQueue::Job->new_from_db_row( $db, $typemap, $row )

=cut

sub new_from_db_row {
    my ( $class, $db, $typemap, $row ) = @_;

    Carp::confess "db is missing" unless $db;
    Carp::confess "typemap is missing" unless $typemap;
    Carp::confess "db row is missing" unless $row;
    Carp::confess "job_id is missing in db row" unless $row->{job_id};

    if ( my $args = delete $row->{args} ) {
        $row->{args} = Yandex::DBQueue::Job->_unpack_struct_from_db($args);
    }

    if ( exists $row->{result} ) {
        $row->{result} = Yandex::DBQueue::Job->_unpack_struct_from_db( $row->{result} );
    }

    return $class->new( $db, $typemap, $row );
}

## аксессоры
__PACKAGE__->mk_ro_accessors(@JOB_FIELDS);

=head2 $job->result

Нестандартный аксессор: если в задаче нет результата (потому что она ещё не сделана),
порождает исключение.

=cut

sub result {
    my ($self) = @_;
    die "No result for job $self->{job_id}" unless exists $self->{result};
    return $self->{result};
}

## помощники поверх аксессоров: вызывают функции-аксессоры и делают простую логику, чтобы не делать
## эту же простую логику снаружи

=head2 $job->has_result()

Стоит ли пытаться вызывать метод result

=cut

sub has_result {
    my ($self) = @_;
    return exists $self->{result};
}

=head2 $job->is_new()

=cut

sub is_new {
    my ($self) = @_;
    return $self->status eq 'New';
}

=head2 $job->is_grabbed()

=cut

sub is_grabbed {
    my ($self) = @_;
    return $self->status eq 'Grabbed';
}

=head2 $job->is_finished()

=cut

sub is_finished {
    my ($self) = @_;
    return $self->status eq 'Finished';
}

=head2 $job->is_failed()

=cut

sub is_failed {
    my ($self) = @_;
    return $self->status eq 'Failed';
}

=head2 $job->is_revoked()

=cut

sub is_revoked {
    my ($self) = @_;
    return $self->status eq 'Revoked';
}

## изменение состояния в базе: здесь, может быть, задачу переложат из dbqueue_jobs в dbqueue_job_archive
## функции возвращают значение: если задачу удалось переместить из dbqueue_jobs в dbqueue_job_archive, 1; иначе 0
## в $result надо передать ссылку, которую сможет сериализовать JSON

=head2 $job->mark_finished($result)

=cut

sub mark_finished {
    my ( $self, $result ) = @_;
    return $self->_job_done( 'Finished', $result );
}

=head2 $job->mark_finished_and_replace_with( $result, $new_job_type, $new_job_hash )

$result -- как у mark_finished
$new_job_hash -- как у DBQueue::insert_job

Если старого задания в очереди не оказалось, падает, в отличие от mark_finished и mark_failed_permanently.

=cut

sub mark_finished_and_replace_with {
    my ( $self, $result, $new_job_type, $new_job_hash ) = @_;

    ## no critic (ControlStructures::ProhibitUnreachableCode)
    die 'This implementation is not verified to be working';

    my $quoted_status = sql_quote('Finished');

    die "result must be a JSON-dumpable reference" unless $result && ref $result;

    my $packed_result = Yandex::DBQueue::Job->_pack_struct_for_db($result);
    die "result too long: " . length($packed_result) . " bytes"
        if length($packed_result) > $MAX_PACKED_RESULT_LENGTH;

    my $quoted_result = sql_quote($packed_result);

    die "redundant field: job_type"    if exists $new_job_hash->{job_type};
    die "redundant field: job_type_id" if exists $new_job_hash->{job_type_id};

    # копируем входные данные, которые передали по ссылке
    my %new_job_hash_copy = %$new_job_hash;

    $new_job_hash_copy{namespace} = exists $new_job_hash_copy{namespace} ? $new_job_hash_copy{namespace}
                                                                         : $NAMESPACE;

    $new_job_hash_copy{job_type}    = $new_job_type;
    $new_job_hash_copy{job_type_id} = $self->{_typemap}->type_to_id($new_job_type);

    my $new_job = Yandex::DBQueue::Job->new( $self->{_db}, $self->{_typemap}, \%new_job_hash_copy );

    my ($affected_by_insert, $affected_by_delete, $affected_by_insert_2);
    retry tries => 3, sub {
        do_in_transaction {
            # как работает этот insert <...> select:
            # выбираем из dbqueue_jobs всё, кроме статуса (он при перемещении теряется),
            # и в нагрузку $new_status и $result в качестве нового статуса и результата;
            # результат вставляем в dbqueue_job_archive
            my $copied_fields = join( ', ', grep { $_ ne 'status' } @JOB_FIELDS );

            $affected_by_insert = do_sql( $self->{_db}, [
                qq{
                    INSERT INTO dbqueue_job_archive ($copied_fields, status, result)
                    SELECT $copied_fields, $quoted_status, $quoted_result FROM dbqueue_jobs
                },
                WHERE => { job_id => $self->job_id },
                "FOR UPDATE", # сразу блокируем запись в dbqueue_jobs на X по избежание deadlock-а
            ] );

            if ($affected_by_insert > 0) {

                $affected_by_delete = do_sql($self->{_db},
                    [
                        'DELETE FROM dbqueue_jobs',
                        WHERE => {job_id => $self->job_id}
                    ]
                );
                die "Failed to delete old job" unless $affected_by_delete > 0;

                if ($affected_by_insert > 0) {
                    $affected_by_insert_2 = do_insert_into_table($self->{_db}, 'dbqueue_jobs', $new_job->as_db_row);
                }
            }
        }; # тут напоминание, что весь этот код скрыт за die и не протестирован
    };

    return;
}

=head2 $job->mark_failed_permanently($result)

=cut

sub mark_failed_permanently {
    my ( $self, $result ) = @_;
    return $self->_job_done( 'Failed', $result );
}

=head2 $job->mark_revoked()

Если задача уже в архиве -- меняет статус
Если задача ещё "активна" -- перемещает в архив с пустым хэшом в качестве результата, ставит статус

=cut

sub mark_revoked {
    my ($self) = @_;

    if ( $JOB_STATUS_IN_ARCHIVE{ $self->status } ) {
        # задача в архиве, надо только статус поменять
        my $updated_rows = do_sql( $self->{_db},
            'UPDATE dbqueue_job_archive SET status="Revoked" WHERE job_id = ?', $self->job_id );
        return ( $updated_rows > 0 );
    }

    return $self->_job_done( 'Revoked', {} );
}

=head2 $job->_job_done( $new_status, $result )

"Больше задачей заниматься не надо": перекладывает задачу в dbqueue_job_archive с новым статусом

=cut

sub _job_done {
    my ( $self, $new_status, $result ) = @_;

    my $quoted_status = sql_quote($new_status);

    die "result must be a JSON-dumpable reference" unless $result && ref $result;

    my $packed_result = Yandex::DBQueue::Job->_pack_struct_for_db($result);
    die "result too long: " . length($packed_result) . " bytes"
        if length($packed_result) > $MAX_PACKED_RESULT_LENGTH;

    my $quoted_result = sql_quote($packed_result);

    my ($affected_by_insert, $affected_by_delete, $moved_ok);
    retry tries => 3, sub {
        do_in_transaction {
            # как работает этот insert <...> select:
            # выбираем из dbqueue_jobs всё, кроме статуса (он при перемещении теряется),
            # и в нагрузку $new_status и $result в качестве нового статуса и результата;
            # результат вставляем в dbqueue_job_archive
            my $copied_fields = join( ', ', grep { $_ ne 'status' } @JOB_FIELDS );

            $affected_by_insert = do_sql($self->{_db}, [
                qq{
                    INSERT INTO dbqueue_job_archive ($copied_fields, status, result)
                    SELECT $copied_fields, $quoted_status, $quoted_result FROM dbqueue_jobs
                },
                WHERE => { job_id => $self->job_id },
                "FOR UPDATE", # сразу блокируем запись в dbqueue_jobs на X по избежание deadlock-а
            ]);

            $affected_by_delete = do_sql($self->{_db},
                [
                    'DELETE FROM dbqueue_jobs',
                    WHERE => {job_id => $self->job_id}
                ]
            );
        };
        $moved_ok = ($affected_by_insert > 0 && $affected_by_delete > 0);
    };


    if ($moved_ok) {
        $self->{status} = $new_status;
        $self->{result} = $result;
    }

    return $moved_ok;
}

=head2 $job->mark_failed_once(%args)

    $job->mark_failed_once( backoff => 15 * 60 );

    backoff -- необязательный параметр
    ВАЖНО: backoff не реализован. Ожидается, что его можно будет реализовать вместе с и через
    установку-обработку поля задачи run_after.

    "сейчас с задачей не получилось, но, может, получится в будущем":
    делает так, чтобы задачей немедленно или через время backoff мог заняться другой процесс

    Возвращает значение: 1, если задача нашлась в таблице dbqueue_jobs в базе; 0, если не нашлась
    (например, если кто-то её переместил в dbqueue_job_archive после обработки или потому что клиент её отозвал)

=cut

sub mark_failed_once {
    my ( $self, %args ) = @_;

    my $backoff = $args{backoff};
    die 'backoff is not implemented' if $backoff;

    my $affected_rows = do_sql( $self->{_db},
        [ 'UPDATE dbqueue_jobs SET grabbed_until = NULL, status = "New"', WHERE => { job_id => $self->job_id } ] );

    my $updated_ok = ( $affected_rows > 0 );

    if ($updated_ok) {
        $self->{grabbed_until} = undef;
        $self->{status} = 'New';
    }

    return $updated_ok;
}

## разные функции

=head2 $job->TO_JSON()

Эту функцию зовёт модуль JSON -- функция нужна для того, чтобы объект можно было сериализовать
в JSON и потом записать в лог, например.

=cut

sub TO_JSON {
    my ($self) = @_;
    return { %$self };
}

=head2 $job->as_db_row()

Как представить задачу в базе: результат можно передавать в do_insert_into_table, например.

=cut

sub as_db_row {
    my ($self) = @_;

    my $packed_args = Yandex::DBQueue::Job->_pack_struct_for_db( $self->{args} );
    die "args too long: " . length($packed_args) . " bytes"
        if length($packed_args) > $MAX_PACKED_ARGS_LENGTH;

    my $packed_result;
    if ( $self->{result} ) {
        $packed_result = Yandex::DBQueue::Job->_pack_struct_for_db( $self->{result} );
        die "result too long: " . length($packed_result) . " bytes"
            if length($packed_result) > $MAX_PACKED_RESULT_LENGTH;
    }

    return {
        job_id => $self->{job_id},
        ClientID => $self->{ClientID},
        job_type_id => $self->{job_type_id},
        namespace => $self->{namespace},
        status => $self->{status},
        uid => $self->{uid} // 0,
        args => $packed_args,
        priority => $self->{priority},
        expiration_time => $self->{expiration_time},
        run_after => $self->{run_after},

        ( $self->{create_time} ? ( create_time => $self->{create_time} ) : () ),
        ( $packed_result ? ( result => $packed_result ) : () ),
    };
}

=head2 $job->_pack_struct_for_db()

=cut

sub _pack_struct_for_db {
    my ( $class, $struct ) = @_;
    return mysql_compress( encode_utf8( to_json($struct) ) );
}

=head2 $job->_unpack_struct_from_db()

=cut

sub _unpack_struct_from_db {
    my ( $class, $packed_struct ) = @_;
    return from_json( decode_utf8( mysql_uncompress($packed_struct) ) );
}

1;
