package Yandex::DBQueue::Typemap;
use Direct::Modern;

=head1 NAME

Yandex::DBQueue::Typemap

=head1 DESCRIPTION

Компонент

Превращает ID типов заданий в строковые названия типов заданий. И наоборот.
Отображение хранится в YAML-файле и в базе. Файл считается первичным источником, таблица в базе
приводится в соответствие файлу.

Зачем: чтобы в таблице с заданиями тип был числовой, так быстрее работает.
Ещё зачем: чтобы можно было им обмениваться между DBQueue и Job. Им эти данные про отображение обоим нужны.
Зачем хранить в файле и в базе: в базе, чтобы поставить foreign key в таблицах с задачами; в файле, чтобы
не плодить релизные миграции для заполнения базы.

В какой момент записи копируются из файла в базу:
1. когда "на лету" обнаруживается, что их там не хватает -- в базу записываются только те строки, которые
в этот момент нужны
2. (опционально) когда запускается скрипт и вызывает функцию fill_job_types -- в базу записываются
все строки, которых там не хватает

Интерфейсы:
создать объект: $typemap = Typemap->new( PPC( shard => 4 ) )
превратить строковое название в ID: $typemap->type_to_id($job_type)
превратить ID в строковое название: $typemap->id_to_type($id)
заполнить базу: $typemap->fill_job_types( log => $log ) # $log -- объект Yandex::Log

Больше он ничего не должен делать.

Снаружи надо периодически вызывать fill_job_types из какого-нибудь отдельного скрипта.

Функции type_to_id, id_to_type вызываются из других частей DBQueue, снаружи они не нужны.

=cut

use List::MoreUtils 'uniq';

use Yandex::DBTools;
use Yandex::LiveFile::YAML;
use Yandex::Validate;

=head1 SUBROUTINES/METHODS/VARIABLES

=head2 $Yandex::DBQueue::Typemap::SOURCE_FILE_PATH

Путь к файлу, откуда заполняется таблица dbqueue_job_types. Пример файла:

    ---
    - id: 1
      type: 'example_type'

Значение переменной ставится один раз на проект, например, в Settings.pm.

=cut
our $SOURCE_FILE_PATH;

=head2 $SOURCE_FILE (private)

Объект Yandex::LiveFile для файла по пути $SOURCE_FILE_PATH. Создаётся лениво при первом обращении
к DBQueue.

=cut
my $SOURCE_FILE;

=head2 Yandex::DBQueue::Typemap->new($db)

=cut
sub new {
    my ( $class, $db ) = @_;

    $SOURCE_FILE //= Yandex::LiveFile::YAML->new( filename => $SOURCE_FILE_PATH );

    return bless {
        _db => $db,
        _type_to_id => {},
        _id_to_type => {},
    }, $class;
}

=head2 $typemap->_insert_file_entry_into_db($entry)

Вставляет в таблицу запись из YAML-файла и проверяет, что она там появилась.
В качестве entry принимает запись из файла: { type => 'type', id => 12 }
Возвращает структуру: { job_type => 'type', job_type_id => 12, inserted_rows => 0|1 }
Если вставить не получилось из-за того, что в базе была другая (отличающаяся) запись
с тем же job_type или job_type_id, порождает исключение.

=cut
sub _insert_file_entry_into_db {
    my ( $self, $entry ) = @_;

    # здесь используем do_sql, а не do_insert_into_table, потому что do_insert_into_table не возвращает
    # количество на самом деле вставленных строк
    my $inserted_rows = do_sql(
        $self->{_db},
        'INSERT IGNORE INTO dbqueue_job_types (job_type, job_type_id) VALUES (?, ?)',
        $entry->{type}, $entry->{id} );

    my $db_row = { job_type => $entry->{type}, job_type_id => $entry->{id} };

    # проверяем, что в базе запись появилась: вдруг там была конфликтующая строчка?
    # если была, скорее всего, стоит поправить файл $SOURCE_FILE_PATH и либо поправить id для этого type, либо
    # выбрать другой id для нового type
    my $inserted_db_row = get_one_line_sql( $self->{_db},
        [
            'SELECT job_type, job_type_id FROM dbqueue_job_types',
            WHERE => { job_type => $db_row->{job_type}, job_type_id => $db_row->{job_type_id} }
        ]
    );

    die "Failed to insert job_type: $entry->{type}, job_type_id: $entry->{id}" unless $inserted_db_row;

    return { %$inserted_db_row, inserted_rows => $inserted_rows + 0 };
}

=head2 $typemap->_find_file_entry( $filter_type, $filter_value )

Вставляет в таблицу запись из YAML-файла и проверяет, что она там появилась.
Запись из файла ищется по соответствию типа или по соответствию ID.
Возвращает запись из файла: запись из файла: { type => 'type', id => 12 }
Если запись не нашлась, порождает исключение.

    $typemap->_find_file_entry( type => 'type' );
    $typemap->_find_file_entry( id => 12 );

=cut
sub _find_file_entry {
    my ( $self, $filter_type, $filter_value ) = @_;

    my $source_file_content = $SOURCE_FILE->data;

    my $matching_entry;
    if ( $filter_type eq 'type' ) {
        ($matching_entry) = grep { $_->{type} eq $filter_value } @$source_file_content;
    } elsif ( $filter_type eq 'id' ) {
        ($matching_entry) = grep { $_->{id} == $filter_value } @$source_file_content;
    }

    die "Invalid $filter_type: $filter_value" unless $matching_entry;

    return $matching_entry;
}

=head2 $typemap->type_to_id($job_type)

=cut
sub type_to_id {
    my ( $self, $job_type ) = @_;

    if ( $self->{_type_to_id}->{$job_type} ) {
        return $self->{_type_to_id}->{$job_type};
    }

    my $job_type_id = get_one_field_sql( $self->{_db},
        [ 'SELECT job_type_id FROM dbqueue_job_types', WHERE => { job_type => $job_type } ] );

    if ($job_type_id) {
        $self->{_type_to_id}->{$job_type} = $job_type_id;
        $self->{_id_to_type}->{$job_type_id} = $job_type;

        return $self->{_type_to_id}->{$job_type};
    }

    my $entry = $self->_find_file_entry( type => $job_type );
    my $insert_result = $self->_insert_file_entry_into_db($entry);

    $self->{_type_to_id}->{$job_type} = $insert_result->{job_type_id};
    $self->{_id_to_type}->{ $insert_result->{job_type_id} } = $job_type;

    return $self->{_type_to_id}->{$job_type};
}

=head2 $typemap->id_to_type($id)

=cut
sub id_to_type {
    my ( $self, $job_type_id ) = @_;

    if ( $self->{_id_to_type}->{$job_type_id} ) {
        return $self->{_id_to_type}->{$job_type_id};
    }

    my $job_type = get_one_field_sql( $self->{_db},
        [ 'SELECT job_type FROM dbqueue_job_types', WHERE => { job_type_id => $job_type_id } ] );

    if ($job_type) {
        $self->{_type_to_id}->{$job_type} = $job_type_id;
        $self->{_id_to_type}->{$job_type_id} = $job_type;

        return $self->{_id_to_type}->{$job_type_id};
    }

    my $entry = $self->_find_file_entry( id => $job_type_id );
    my $insert_result = $self->_insert_file_entry_into_db($entry);

    $self->{_type_to_id}->{ $insert_result->{job_type} } = $job_type_id;
    $self->{_id_to_type}->{$job_type_id} = $insert_result->{job_type};

    return $self->{_id_to_type}->{$job_type_id};
}


=head2 get_job_types

    возвращает ссылку на массив строк -- имен всех типов заданий
    и из БД, и из файла
    список уникализирован и отсортирован

    проверка консистентности БД и файла по id типов НЕ делается

    параметров нет (если понадобится -- можно сделать выбор БД/файл)

=cut 
sub get_job_types
{
    my ( $self, %params ) = @_;
    my $types_in_db = get_one_column_sql( $self->{_db}, 'SELECT job_type FROM dbqueue_job_types' );

    my $source_file_content = $SOURCE_FILE->data;
    my $types_in_file = [ map { $_->{type} } @$source_file_content ];

    my @all_job_types = ( @$types_in_db, @$types_in_file );
    @all_job_types = uniq @all_job_types;
    @all_job_types = sort @all_job_types;

    return \@all_job_types;
}

=head2 $typemap->fill_job_types( log => $log )

Заполняет таблицу dbqueue_job_types в соответствии с файлом.
Возвращает количество добавленных в базу записей.
Если что-то пошло не так, порождает исключение.

Параметром log обязательно надо передать объект Yandex::Log, чтобы функция отчитывалась, что она делает.

Надо вызывать только в системе, где какой-то внешний процесс (например, решардинг)
сам манипулирует данными в dbqueue_jobs и dbqueue_job_archive. Такому процессу может быть
надо, чтобы при записи в dbqueue_jobs была нужная запись в dbqueue_job_types, потому что
в dbqueue_jobs есть внешний ключ.

В такой системе эту функцию надо вызывать периодически из какого-нибудь скрипта.

=cut
sub fill_job_types {
    my ( $self, %params ) = @_;

    my $log = $params{log} or die "Missing required parameter: log";

    ## в переменных map_in_* структура: { $job_type => $job_type_id, ... }
    my $map_in_db = get_hash_sql( $self->{_db}, 'SELECT job_type, job_type_id FROM dbqueue_job_types' );
    $log->out( { map_in_db => $map_in_db } );

    my $source_file_content = $SOURCE_FILE->data;
    my $map_in_file = { map { $_->{type} => $_->{id} } @$source_file_content };
    $log->out( { map_in_file => $map_in_file } );

    my @all_job_types = ( keys %$map_in_db, keys %$map_in_file );
    @all_job_types = uniq @all_job_types;
    @all_job_types = sort @all_job_types;

    my $total_inserted_rows = 0;

    for my $job_type (@all_job_types) {
        if ( exists $map_in_db->{$job_type} && exists $map_in_file->{$job_type} ) {
            $log->out(
                "$job_type present both in DB and source file; " .
                "job_type_id is $map_in_db->{$job_type} in DB, $map_in_file->{$job_type} in source file" );

            $log->die("invalid ID value for $job_type in DB") unless is_valid_id( $map_in_db->{$job_type} );
            $log->die("invalid ID value for $job_type in source file") unless is_valid_id( $map_in_file->{$job_type} );

            unless ( $map_in_db->{$job_type} == $map_in_file->{$job_type} ) {
                $log->die("inconsistent IDs for job type $job_type");
            }
        } elsif ( exists $map_in_file->{$job_type} ) {
            $log->out("$job_type only present in source file, trying to insert into DB");

            my $job_type_id = $map_in_file->{$job_type};
            $log->die("invalid ID value for $job_type in source file") unless is_valid_id($job_type_id);

            my $insert_result = eval { $self->_insert_file_entry_into_db( { type => $job_type, id => $job_type_id } ) };
            $log->die($@) if $@;

            $log->out( { inserted_rows => $insert_result->{inserted_rows} + 0 } );

            $total_inserted_rows += $insert_result->{inserted_rows};
        } else {
            $log->out("$job_type only present in DB");
        }
    }

    return $total_inserted_rows;
}

1;
