package Yandex::ORM::Helpers;

use Direct::Modern;

use Yandex::DateTime qw/now/;
use DateTime::Format::MySQL;

use base qw/Exporter/;

our @EXPORT_OK = qw(mysql_timestamp_around is_json_eq_without_quotes);

=head2 mysql_timestamp_around($attr_name)

Возвращает ссылку на функцию для использования в качестве around-модификатора атрибута.
Рекомендуется для Timestamp атрибутов.
Добавляет поддержку указания ключевого слова "now" для задания текущего времени.

    package Direct::Model::PerformanceFilter;
    ...
    around last_change => mysql_timestamp_around('last_change');
    ...
    1;

    my $filter = Direct::Model::PerformanceFilter->new();
    $filter->last_change('now');

При этом в БК при сохранениии в качестве значения атрибута будет указано неквотируемое NOW()

=cut

sub mysql_timestamp_around {
    my ($attr_name) = @_;

    return sub {
        my ($orig, $self, @args) = (shift, shift, @_);

        return $self->$orig() if !@args;

        my ($set_to_now, $keep);

        if (($args[0] // '') =~ /^now$/i) {
            $args[0] = DateTime::Format::MySQL->format_datetime(now());
            $set_to_now = 1;
        } elsif (($args[0] // '') =~ /^keep$/i) {
            @args = ();
            $keep = 1;
        }

        my $attr = $self->meta->find_attribute_by_name($attr_name);
        croak "invalid attribute `$attr_name`" if !$attr || !$attr->does('Yandex::ORM::Meta::Attribute::Trait::Column');

        my ($table, $column) = @{$attr->params}{qw/table column/};
        croak "attribute `$attr_name` does not mapped to db column" if !defined $table || !defined $column;

        my $res = $self->$orig(@args);

        if    ($set_to_now) { $self->set_db_column_value($table, $column, 'NOW()', dont_quote => 1); }
        elsif ($keep)       { $self->set_db_column_value($table, $column, $column, dont_quote => 1); }
        else                { $self->set_db_column_value($table, $column, undef, dont_quote => 1); }

        return $res;
    };
}

=head2 is_json_eq_without_quotes($old, $new)

Сравнивает два json без учета кавычек

=cut

sub is_json_eq_without_quotes {
    my ($old, $new) = @_;
    $old =~ s/(?<!\\)"//g;
    $new =~ s/(?<!\\)"//g;
    return $old eq $new;
}


1;
