package Yandex::ORM::Model::Base;

use Direct::Modern;

use Mouse -traits => ['Yandex::ORM::Meta::Class::Trait::TableSchema'];
use Mouse::Util::TypeConstraints;
use Mouse::Util qw/apply_all_roles/;

use Yandex::ORM::Types;

use List::MoreUtils qw//;
use Scalar::Util qw/blessed/;
use Storable qw/dclone/;

# _state и _db_state хранят внутреннее состояние объекта:
#   _state содержит информацию о установленных флагах и их значения, а также признаки измененности `additional` атрибутов
#   _db_state содержит данные в разрезе таблиц БД: признак измененности поля и, опционально, dont_quote значение
has '_state'    => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{} });
has '_db_state' => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{} });

# Признак того, что конструктор отработал (используется в триггерах на атрибуты)
has '_constructed' => (is => 'rw', isa => 'Bool', init_arg => undef, default => 0);

sub BUILD {
    my ($self) = @_;
    $self->_constructed(1);
}

my $_get_mouse_isa_constraint = sub {
    my ($type, %params) = @_;

    if ($type eq 'Enum' || $type eq 'Maybe[Enum]') {
        my $type_name = '_'.join('_', map { $_ =~ s/[^0-9a-z]//irg } sort @{$params{values}}).'_enum_t';
        enum($type_name, $params{values}) if !find_type_constraint($type_name);
        return $type =~ /Maybe/ ? "Maybe[${type_name}]" : $type_name;
    } elsif ($type =~ /^(\+)?Set$/ || $type =~ /^Maybe\[(\+)?Set\]$/) {
        my $plus = $1 // '';
        my $type_name = '_'.join('_', map { $_ =~ s/[^0-9a-z]//ir } sort @{$params{values}}).($plus ? '_nonempty' : '').'_set_t';

        if (!find_type_constraint($type_name)) {
            set($plus.$type_name, $params{values});
            if ($type =~ /Maybe/) {
                subtype("Maybe_${type_name}", as "Maybe[${type_name}]");
                $type_name = "Maybe_${type_name}";
            }
            coerce($type_name, from 'Str', via { [grep { length } split /,/] });
        }

        return $type_name;
    } else {
        return $type;
    }
};

sub _setup {
    my ($package, %schema) = @_;

    croak "is package-only method" if blessed($package);

    my $default_table = $schema{default_table};
    $package->meta->default_table($default_table) if defined $default_table;

    if (my $manager_class = $schema{manager_class}) {
        if (0 && !Mouse::Util::is_class_loaded($manager_class)) {
            eval { require(($manager_class =~ s{::}{/}gr).'.pm'); 1; } or do { croak "Cannot load class `$manager_class`"; };
        }
        my $try_to_load = 1;
        $package->meta->add_method(manager_class => sub {
            if ($try_to_load) {
                local $@;
                eval { require(($manager_class =~ s{::}{/}gr).'.pm'); 1; } if !Mouse::Util::is_class_loaded($manager_class);
                $try_to_load = 0;
            }
            return $manager_class;
        });
    }

    my @fields = @{$schema{fields} // []};

    my @additional_fields;
    for my $section (qw/additional relations/) {
        my @pairs = @{$schema{$section} // []};
        for (my $i = 0; $i < @pairs; $i += 2) {
            my ($field_name, $field_params) = @pairs[$i, $i+1];
            $field_params->{section} = $section;
            push @additional_fields, @pairs[$i, $i+1];
        }
    }

    my @state_flags = @{$schema{state_flags} // []};

    my %existent_attrs;

    # Описание атрибутов модели, соотнесённых с колонками в таблицах БД (@fields)
    my $pos = 0;
    for (my $i = 0; $i < @fields; $i += 2, $pos++) {
        my ($field_name, $field_params) = @fields[$i, $i+1];

        my $table  = $field_params->{table}  // $default_table;
        my $column = $field_params->{column} // $field_name;
        my $alias  = $field_params->{alias}  // $column;

        my $is_primary_key = $field_params->{primary_key};
        my $is_set_type = $field_params->{type} =~ /Set/;
        my $custom_trigger = delete $field_params->{trigger};
        my $custom_eq = delete $field_params->{custom_eq};
        my $is_volatile = $field_params->{volatile};

        # если поле должно быть в списке fields для класса, но для данного класса не имеет значения то 
        # задаем дефолтное значение в dummy values
        # Нужно, например при наследование, когда список полей для выборки из БД берем из родительского 
        # класса, а наслеников создаем фабрикой и в наследниках поле значимое (пример см в Direct::Model::Creative и Direct::Model::CanvasHtml5Creative)
        # может быть undef, т.к. ниже проверяем по exists в field_params
        my $dummy_value = $field_params->{dummy_value};

        croak "Attribute `$field_name` already exists in $package" if $existent_attrs{$field_name}++;

        # $field_name может быть приватным (начинаться с `_`), нужно это учитывать при генерации имен
        my ($field_prefix, $field_basename) = map { $_ // '' } ($field_name =~ /^(_?)(.+)$/);

        # Добавляем атрибут на класс + триггер для отслеживания изменений соответствующей колонки в таблице БД
        $package->meta->add_attribute($field_name => (
            traits     => [qw/Yandex::ORM::Meta::Attribute::Trait::Column/],
            is         => 'rw',
            isa        => $_get_mouse_isa_constraint->($field_params->{type}, %$field_params),
            coerce     => $is_set_type,

            lazy       => 1,
            ( exists $field_params->{dummy_value}
                ? ( default => sub { $dummy_value })
                : ( builder => sub { croak "Cannot get a value for unassigned attribute `$field_name`" } )
            ),
            predicate  => "${field_prefix}has_${field_basename}",
            clearer    => "${field_prefix}clear_${field_basename}",

            # Для главного ключа отслеживание изменений не требуется
            (!$is_primary_key ? (
            trigger    => sub {
                my ($self, $new, $old) = @_;

                # Внешний триггер
                $custom_trigger->(@_) if ref($custom_trigger) eq 'CODE';

                # Если триггер вызывается из конструктора, то ничего не делаем
                # Структуру _db_state будем заполнять уже по факту
                return if !$self->_constructed;

                # Array struct: 0 -> is_changed; 1 -> val__dont_quote;
                my $column_state = ($self->_db_state->{$table}->{$column} //= [0, undef]);

                $column_state->[0] = 1 if @_ == 2; # если нет old значения, то считаем что атрибут изменился
                $column_state->[0] = 1 if $is_volatile; # если выставлен флаг volatile, всегда считаем атрибут изменившимся (DIRECT-58018)
                return if $column_state->[0]; # атрибут уже был изменён

                if ($is_set_type) {
                    $old = join('', sort(List::MoreUtils::uniq(@$old))) if defined $old;
                    $new = join('', sort(List::MoreUtils::uniq(@$new))) if defined $new;
                }

                if ($field_params->{type} eq 'Bool' || $field_params->{type} eq 'Maybe[Bool]') {
                    $old = !!$old if defined $old;
                    $new = !!$new if defined $new;
                }

                $column_state->[0] = defined($new) && defined($old)
                    ? ($custom_eq ? !$custom_eq->($old, $new) : $new ne $old)
                    : defined($new) || defined($old);

                return;
            },
            ) : ()),

            params     => {%$field_params, table => $table, column => $column, alias => $alias},
        ));

        #if ($is_set_type) {
        #    $package->meta->add_around_method_modifier($field_name => sub {
        #        my $orig = shift;
        #        my $self = shift;
        #        return dclone($self->$orig(@_));
        #    });
        #}

        # Метод: is_{field}_changed
        $package->meta->add_method("${field_prefix}is_${field_basename}_changed" => sub {
            my $self = shift;
            if (@_) {
                $self->_db_state->{$table}->{$column} //= [0, undef];
                return $self->_db_state->{$table}->{$column}->[0] = $_[0] ? 1 : 0;
            } elsif (exists $self->_db_state->{$table} && exists $self->_db_state->{$table}->{$column}) {
                return $self->_db_state->{$table}->{$column}->[0];
            }
            return 0;
        }) if !$is_primary_key;

        # Фиксируем описание колонки в схеме таблиц БД
        $package->meta->table_schema->{$table}->{$column} = {
            attr_name => $field_name,
            alias     => $alias,
            length    => $field_params->{length},
            nullable  => $field_params->{type} =~ /^Maybe/ ? 1 : 0,
            (exists $field_params->{default} ? (default => $field_params->{default}) : ()),
            pos       => $pos,
        };
    }

    # Описание дополнительных атрибутов, которые могут быть инициализированы в конструкторе from_db_hash,
    # но не имеют хранимой колонки в таблице БД (например, вычисляются на лету в SQL запросе)
    # Тут же описываем связи модели с другими объектами (секция relations)
    for (my $i = 0; $i < @additional_fields; $i += 2) {
        my ($field_name, $field_params) = @additional_fields[$i, $i+1];

        my $section = $field_params->{section};
        my $track_changes = $field_params->{track_changes};
        my $is_set_type = $field_params->{type} =~ /Set/;
        my $custom_trigger = delete $field_params->{trigger};
        my $custom_eq = delete $field_params->{custom_eq};
        my $builder = delete $field_params->{builder};
        my $is = delete $field_params->{is};
        my $cmp_func = delete $field_params->{cmp_func}; 
        
        croak "Attribute `$field_name` already exists in $package" if $existent_attrs{$field_name}++;
        croak "Attribute `$field_name` requires `cmp_func`" if $section eq 'relations' && $track_changes && !$cmp_func;

        # Атрибут может быть приватным (начинаться с `_`), нужно это учитывать при генерации имен
        my ($field_prefix, $field_basename) = map { $_ // '' } ($field_name =~ /^(_?)(.+)$/);

        $package->meta->add_attribute($field_name => (
            is         => $is // 'rw',
            isa        => $_get_mouse_isa_constraint->($field_params->{type}, %$field_params),
            coerce     => $is_set_type || $field_params->{coerce},

            ($track_changes ? (
            trigger    => sub {
                my ($self, $new, $old) = @_;

                $custom_trigger->(@_) if ref($custom_trigger) eq 'CODE';

                return if !$self->_constructed;
                $self->_state->{changes}->{$field_name} = 1 if @_ == 2;
                return if $self->_state->{changes}->{$field_name};

                if ($cmp_func) {
                    $self->_state->{changes}->{$field_name} = $cmp_func->(@_);
                    return;
                }

                if ($is_set_type) {
                    $old = join('', sort(List::MoreUtils::uniq(@$old))) if defined $old;
                    $new = join('', sort(List::MoreUtils::uniq(@$new))) if defined $new;
                }

                if ($field_params->{type} eq 'Bool' || $field_params->{type} eq 'Maybe[Bool]') {
                    $old = !!$old if defined $old;
                    $new = !!$new if defined $new;
                }

                $self->_state->{changes}->{$field_name} = defined($new) && defined($old)
                    ? ($custom_eq ? !$custom_eq->($old, $new) : $new ne $old)
                    : defined($new) || defined($old);

                return;
            }) : (ref($custom_trigger) eq 'CODE' ? (
            trigger    => $custom_trigger,
            ) : ())),

            lazy       => 1,
            builder    => $builder // sub { croak "Cannot get a value for unassigned attribute `$field_name`" },
            predicate  => "${field_prefix}has_${field_basename}",
        ));

        if ($section eq 'relations') {
            $package->meta->add_method("${field_prefix}clear_${field_basename}" => sub {
                my $self = shift;
                $self->meta->find_attribute_by_name($field_name)->clear_value($self);
                delete $self->_state->{changes}->{$field_name} if $track_changes;
                return;
            });
        }

        if ($track_changes) {
            $package->meta->add_method("${field_prefix}is_${field_basename}_changed" => sub {
                my $self = shift;
                return $self->_state->{changes}->{$field_name} = $_[0] ? 1 : 0 if @_;
                return $self->_state->{changes}->{$field_name};
            });
        }
    }

    # Описание флагов (которыми задаётся состояние модели)
    for my $flag (@state_flags) {
        $package->meta->add_method("do_${flag}" => sub {
            my $self = shift;
            return ($self->_state->{flags}->{$flag} = $_[0]) if @_;
            return $self->_state->{flags}->{$flag};
        });
    }

    $package->meta->strict_constructor(1) unless defined $schema{strict_constructor} && !$schema{strict_constructor};
    $package->meta->make_immutable unless defined $schema{make_immutable} && !$schema{make_immutable};
}

#######################
#    Class methods    #
#######################

=head2 get_table_schema($table)

    my $schema = Direct::Model::Banner->get_table_schema("banners");

Возвращяет схему таблицы $table: возможные колонки и их свойства.

=cut

sub get_table_schema {
    my ($class, $table) = @_;

    croak "is class-only method" if blessed($class);
    croak "\$table should be defined" unless defined $table;

    my %merged_schema;
    for my $parent_class (reverse $class->meta->linearized_isa) {
        next unless $parent_class->meta->meta->has_attribute('table_schema');
        next unless exists $parent_class->meta->table_schema->{$table};

        my $schema = $parent_class->meta->table_schema->{$table};
        $merged_schema{$_} = $schema->{$_} for keys %$schema;
    }

    croak "cannot find schema for table `$table`" unless %merged_schema;

    return \%merged_schema;
}

=head2 get_db_columns_list($table, $fields, $exclude_fields)

    my @columns = Direct::Model::Banner->get_db_columns_list("banners");

Возвращяет список возможных колонок в таблице $table;
Дополнительно можно указать:
    $fields (ArrayRef) -- атрибуты объекта, по которым будет построен список колонок.
    $exclude_fields -- атрибуты объекта, которые будут исключены из списка
    

=cut

sub get_db_columns_list {
    my ($class, $table, $fields, $exclude_fields) = @_;

    croak "is class-only method" if blessed($class);

    my $table_schema = $class->get_table_schema($table);
    my @table_columns = sort { $table_schema->{$a}->{pos} <=> $table_schema->{$b}->{pos} } keys %$table_schema;

    my %exclude = map { $_ => undef } defined $exclude_fields ? @$exclude_fields : ();
    return grep { !exists $exclude{ $table_schema->{$_}->{attr_name} } } @table_columns if !defined $fields || !@$fields;

    my %unused_fields = map { $_ => 1 } defined $fields ? @$fields : ();
    my @columns_list;
    for my $column (@table_columns) {
        next unless delete $unused_fields{ $table_schema->{$column}->{attr_name} };
        next if exists $exclude{ $table_schema->{$column}->{attr_name} };
        push @columns_list, $column;
    }

    croak "cannot find definition for fields `".join(',', keys %unused_fields)."` for table `$table` in `$class`" if %unused_fields;

    return @columns_list;
}

=head2 get_db_columns($class, $table, $table_alias, %options)

    my $columns = get_db_columns(banners => 'b');
    my $banners = get_all_sql(PPC(bid => \@bids), ["SELECT $columns FROM banners b", where => {bid => SHARD_IDS}]));

Возвращяет список колонок из таблицы $table, разделенных через ','.
Используется преимущественно для построения SELECT запросов.

Параметры:

    $table       -> таблица БД, из которой будут выбираться данные
    $table_alias -> алиас таблицы

    %options:
        prefix -> префикс, который следует добавить ко всем колонкам; по умолчанию, если указан алиас таблицы = `table_alias`_
            b.bid AS b_bid
            b_ - префикс

        fields -> ARRAY; список полей модели, по которым выбирать колонки таблицы; по умолчанию выбираются все колонки в таблице
        exclude_fields -> ARRAY; массив полей модели, которые нужно исключить

=cut

sub get_db_columns {
    my ($class, $table, $table_alias, %options) = @_;

    croak "is class-only method" if blessed($class);

    my $prefix = $options{prefix} // (defined $table_alias ? $table_alias.'_' : '') // '';
    $table_alias = defined $table_alias ? $table_alias.'.' : '';

    my $table_schema = $class->get_table_schema($table);
    my @columns = map {
        $_ eq $prefix.$table_schema->{$_}->{alias}
        ? $table_alias.'`'.$_.'`'
        : $table_alias.'`'.$_.'`'.' AS '.'`'.$prefix.$table_schema->{$_}->{alias}.'`'
    } $class->get_db_columns_list($table, $options{fields}, $options{exclude_fields});

    return join(', ', @columns);
}

=head2 from_db_hash($row, $cache, %options)

    my $rows = get_all_sql(PPC(bid => \@bids), [q{
        SELECT
            b.bid AS b_bid, b.title AS b_title, b.body AS b_body, b.vcard_id AS b_vcard_id,
            vc.vcard_id AS vc_vcard_id, vc.phone AS vc_phone
        FROM banners b LEFT JOIN vcards vc ON (vc.vcard_id = b.vcard_id)
    }, WHERE => {'b.bid' => SHARD_IDS}]);
    my $cache;
    for my $row (@$rows) {
        my $banner = Direct::Model::Banner->from_db_hash($row, \$cache, prefix => 'b_');
        $banner->vcard(
            Direct::Model::VCard->from_db_hash($row, \$cache, prefix => 'vc_')
        ) if $banner->vcard_id;
    }

Создает объект по результату выборки из БД (get_all_sql).

Параметры:
    $row   -> хеш, представляющий одну строку выборки
    $cache -> вспомогательный кеш, хранит конфигурацию ключей в $row, для быстрой инициализации моделей
        Указывать его как \{} не рекомендуется, если объект инициализируется в цикле и с однаковыми ключами в $row;
        При использовании $cache, производительность вырастает более чем в 3 раза;

    %options:
        prefix -> префикс на ключах в $row, идентифицирующий ключи для моделей одного типа
            Например: prefix => 'b_' указывает, что ключи в $row, относящиеся к создаваемой модели, будут выглядеть как b_something
        with   -> роли, с которыми инициализировать объект
        delete_keys -> удалить из $row ключи, преобразованные в объект

=cut

sub from_db_hash {
    my ($class, $row, $cache, %options) = @_;

    croak "is class-only method" if blessed($class);
    croak "\$cache should be Ref" unless ref($cache) eq 'SCALAR' || ref($cache) eq 'REF';
    croak "invalid \$cache value" unless !defined $$cache || ref($$cache) eq 'HASH';

    if (!defined $$cache || !defined ${$cache}->{$class}) {
        my $state = ${$cache}->{$class} = {};

        # Роли, с которыми нужно создать каждый объект
        my $with = $options{with};
        my @roles_to_apply;
        for my $role_name (defined $with ? (ref($with) ? @$with : $with) : ()) {
            Carp::confess("Invalid role name found") if !defined($role_name) || !length($role_name);
            for my $class ($class->meta->linearized_isa) {
                my $meta = Mouse::Util::get_metaclass_by_name($class) or next;
                my $role_to_apply = $role_name =~ /::/ ? $role_name : "${class}::Role::${role_name}";

                my $is_applied_role = List::MoreUtils::any { $_->name eq $role_to_apply } @{$meta->roles};
                if (!$is_applied_role) {
                    # Попробуем понять, есть ли такая роль
                    my $class_to_load = $role_to_apply;
                    Carp::confess("Invalid class name: $class_to_load") unless Mouse::Util::is_valid_class_name($class_to_load);

                    my $is_existent_class;
                    if (Mouse::Util::is_class_loaded($class_to_load)) {
                        $is_existent_class = 1;
                    } else {
                        local $@;
                        $class_to_load =~ s{::}{/}g;
                        $class_to_load .= '.pm';
                        eval { require($class_to_load); 1; } and do { $is_existent_class = 1 };
                    }

                    push @roles_to_apply, $role_to_apply if $is_existent_class;
                }
            }
        }

        # Если требуется применение ролей, то создадим анонимный класс с нужным набором ролей.
        my $dst_class = @roles_to_apply ?
            Mouse::Meta::Class->create_anon_class(superclasses => [$class], roles => \@roles_to_apply, cache => 1)->name
            : $class;
        $state->{dst_class} = $dst_class;
        $state->{attr_by_prefix} = {};        
    }

    # Префикс, добавленный к ключам в $row
    my $prefix = $options{prefix} // '';
    if (!exists ${$cache}->{$class}->{attr_by_prefix}->{$prefix}) {
        my $dst_class = ${$cache}->{$class}->{dst_class};
        my $state = ${$cache}->{$class}->{attr_by_prefix}->{$prefix} = {};
        # attr_keys -> имена атрибутов для конструктора
        # row_keys  -> имена ключей, по которым искать значения в каждой строке $rows
        my (@attr_names, @row_keys);

        for my $attr ($dst_class->get_public_attributes) {
            #next unless $attr->does('Yandex::ORM::Meta::Attribute::Trait::Column');

            my $alias = $attr->does('Yandex::ORM::Meta::Attribute::Trait::Column') ? $attr->params->{alias} : undef;
            my $row_key = $prefix.($alias // $attr->name);
            if (exists $row->{$row_key}) {
                push @attr_names, $attr->name;
                push @row_keys, $row_key;
            }
        }
        $state->{attr_names} = \@attr_names;
        $state->{row_keys} = \@row_keys;
    }

    my ($dst_class, $attr) = (${$cache}->{$class}->{dst_class}, ${$cache}->{$class}->{attr_by_prefix}->{$prefix});
    my %args;
    @args{@{ $attr->{attr_names} }} = @$row{@{ $attr->{row_keys} }};
    if ($options{delete_keys}) {
        delete @$row{@{ $attr->{row_keys} }};
    }

    my $result = eval { $dst_class->new(%args); };
    if ($@) {
        # require JSON;
        # warn JSON::to_json(\%args, {pretty => 1, allow_blessed => 1, convert_blessed => 1});
        die $@;
    }
    return $result;
}

=head2 from_db_hash_multi($rows, %options)

    my $rows = get_all_sql(PPC(bid => \@bids), [q{SELECT bid,title,body,statusBsSynced FROM banners}, WHERE => {bid => SHARD_IDS}]);
    my $banners = Direct::Model::Banner->from_db_hash_multi($rows);

См. from_db_hash.

Создает массив моделей по результатам выборки get_all_sql (массив хешей).
Ключи хешей внутри $rows должны быть одинаковыми;

=cut

sub from_db_hash_multi {
    my ($class, $rows, %options) = @_;

    croak "is class-only method" if blessed($class);

    my $instances = [];
    return $instances unless @$rows;

    my $row0_keys = join('', sort keys %{$rows->[0]});
    croak "rows should have the same structure" if List::MoreUtils::any { $row0_keys ne join('', sort keys %$_) } @$rows;

    my $cache;
    push @$instances, $class->from_db_hash($_, \$cache, %options) for @$rows;

    return $instances;
}

##########################
#    Instance methods    #
##########################

=head2 is_changed

Глобальный метод is_changed на объект: возвращяет true если изменилось хотя бы одно поле.

=cut

sub is_changed {
    my ($self) = @_;

    for my $table (keys %{$self->_db_state}) {
        for my $column (keys %{$self->_db_state->{$table}}) {
            return 1 if $self->_db_state->{$table}->{$column}->[0];
        }
    }

    for my $field_name (keys %{$self->_state->{changes}}) {
        return 1 if $self->_state->{changes}->{$field_name};
    }

    return 0;
}

=head2 reset_state

Сброс состояния объекта: установленные флаги и признак изменённости.

=cut

sub reset_state {
    my ($self) = @_;
    %{$self->_state} = ();
    %{$self->_db_state} = ();
    return;
}

=head2 clone(%new_params)

Метод клонирования (создания полной копии) объекта.

=cut

sub clone {
    my ($self, %new_params) = @_;

    my $newobj = $self->meta->clone_object($self);

    for my $attr (grep { $_->has_value($self) } $self->meta->get_all_attributes) {
        my $val = $attr->get_value($self);

        if (blessed($val) && $val->can('clone')) {
            $attr->set_value($newobj, $val->clone);
        } elsif (ref($val) eq 'ARRAY') {
            $attr->set_value($newobj, [map {
                blessed($_) && $_->can('clone') ? $_->clone : (ref($_) ? dclone($_) : $_)
            } @$val]);
        } elsif (ref($val)) {
            $attr->set_value($newobj, dclone($val));
        }
    }

    return $newobj->alter(%new_params);
}

=head2 alter(%params)

Изменение атрибутов модели по списку ключ/значение

=cut

sub alter {
    my ($self, %params) = @_;

    while (my ($attr, $val) = each %params) {
        $self->$attr($val);
    }

    return $self;
}

=head2 to_db_hash

Дамп модели в качестве хеша с полями, соответствующими названию в схеме БД.

=cut

sub to_db_hash {
    my ($self) = @_;

    my $dump = {};
    for my $attr ($self->meta->get_all_attributes) {
        next unless $attr->does('Yandex::ORM::Meta::Attribute::Trait::Column') && $attr->has_value($self) && $attr->params->{table};
        my $attr_key = $attr->params->{alias} // $attr->name;
        my $val = $attr->get_value($self);
        $val = join(',', @$val) if $attr->params->{type} =~ /Set/ && defined $val;
        $dump->{$attr_key} = $val;
    }

    return $dump;
}

=head2 to_hash

Дамп модели в качестве хеша с полями, соответствующими названию атрибутов.

=cut

sub to_hash {
    my ($self) = @_;

    my $dump = {};
    for my $attr (grep { $_->has_value($self) } $self->get_public_attributes) {
        my $val = $attr->get_value($self);

        if (blessed($val) && $val->can('to_hash')) {
            $dump->{$attr->name} = $val->to_hash;
        } elsif (ref($val) eq 'ARRAY') {
            $dump->{$attr->name} = [map {
                blessed($_) && $_->can('to_hash') ? $_->to_hash : (ref($_) ? dclone($_) : $_)
            } @$val];
        } elsif (ref($val)) {
            $dump->{$attr->name} = dclone($val);
        } else {
            $dump->{$attr->name} = $val;
        }
    }

    return $dump;
}

=head2 get_db_column_value($self, $table, $column, %options)

    my $statusModerate = $banner->get_db_column_value('banners', 'statusModerate');
    my $main_domain_id = $adgroup_dynamic->get_db_column_value('adgroups_dynamic', 'main_domain_id');

Возвращяет значение атрибута, по названию соответствующей колонки в таблице БД.

Параметры:

    $table  -> таблица, содержащая колонку
    $column -> колонка, к которой привязан какой-либо атрибут
    %options:
        extended      -> (по умолчанию: 0) возвращать результат в виде хеша, с ключами val и val__dont_quote
            val -> текущее значение атрибута; val__dont_quote -> установленное "dont_quote" значение
        default_is_ok -> (по умолчанию: 0) если атрибут на имеет значения, возвращать значение по умолчанию (из схемы)
        strict        -> (по умолчанию: 1) падать, если нет подходящего значения; иначе возвращать undef

=cut

sub get_db_column_value {
    my ($self, $table, $column, %options) = @_;

    # Дополнительные параметры
    my $strict = $options{strict} // 1;
    my $default_is_ok = $options{default_is_ok};
    my $extended = $options{extended};

    my $table_schema = $self->meta->name->get_table_schema($table);

    croak "cannot find definition for column `$table`.`$column`" unless exists $table_schema->{$column};
    my $column_schema = $table_schema->{$column};
    my $column_state = exists $self->_db_state->{$table} && exists $self->_db_state->{$table}->{$column}
        ? $self->_db_state->{$table}->{$column}
        : [0, undef];

    my $attr = $self->meta->find_attribute_by_name($column_schema->{attr_name});
    my $is_set_attr = $attr->params->{type} =~ /Set/;
    if ($attr->has_value($self)) {
        my $val = $attr->get_value($self);
        $val = _serialize_set($val)  if $is_set_attr;
        # Значение соответствующего атрибута задано
        return !$extended
            ? $val
            : {val => $val, defined $column_state->[1] ? (val__dont_quote => $column_state->[1]) : ()};
    } elsif (defined $column_state->[1] && $extended) {
        # Особый случай - задано только dont_quote значение
        return {val__dont_quote => $column_state->[1]};
    }

    # Значение соответствующего атрибута не задано, вернем значение по умолчанию (если есть)
    if ($default_is_ok && ($column_schema->{nullable} || exists $column_schema->{default})) {
        my $default = $column_schema->{default};
        $default = $self->$default  if ref $default eq 'CODE';
        $default = _serialize_set($default)  if $is_set_attr;
        return !$extended
            ? $default
            : {val => $default};
    }

    return !$extended ? undef : {val => undef} if !$strict;
    croak "cannot find value for `${table}`.`${column}`";
}


sub _serialize_set {
    my $values = shift;
    return $values  if !ref $values;
    return join ',' => @$values;
}


=head2 set_db_column_value($self, $table, $column, $value, %options)

    if ($banner->do_update_status_post_moderate_unless_rejected) {
        $banner->is_status_post_moderate_changed(1);
        $banner->set_db_column_value('banners', 'statusPostModerate', sprintf(
            "IF(statusPostModerate = 'Rejected', 'Rejected', %s)", sql_quote($banner->status_post_moderate)
        ), dont_quote => 1);
    }

Устанавливает значение атрибута, по названию соответствующей колонки в таблице БД.

Параметры:

    $table  -> таблица, содержащая колонку
    $column -> колонка, к которой привязан какой-либо атрибут

    %options:
        dont_quote -> установить "dont_quote" значение.
            Фактическое значение атрибута при этом не поменяется, но при применении модели к БД, будет взято "dont_quote" значение;

=cut

sub set_db_column_value {
    my ($self, $table, $column, $value, %options) = @_;

    # Дополнительные параметры
    my $dont_quote = $options{dont_quote};

    my $table_schema = $self->meta->name->get_table_schema($table);

    croak "cannot find definition for column `$table`.`$column`" unless exists $table_schema->{$column};
    my $column_schema = $table_schema->{$column};
    my $column_state = ($self->_db_state->{$table}->{$column} //= [0, undef]);

    my $attr = $self->meta->find_attribute_by_name($column_schema->{attr_name});
    if (!$dont_quote) {
        $attr->set_value($self, $value);
    } else {
        $column_state->[1] = $value;
    }

    return;
}

=head2 _set_attr_raw_value($attr_name, $attr_value)

Устанавливает значение атрибута $attr_name без "побочных" эффектов,
таких как: триггер, контроль целостности, проверка существования атрибута, и т.д.

(!) Метод рекомендуется только для внутренного использования.

=cut

sub _set_attr_raw_value {
    my ($self, $attr_name, $attr_value) = @_;
    $self->{$attr_name} = $attr_value;
}

=head2 does_role($role_name)

    say "\$banner does role `Update`!" if $banner->does_role("Update");

Возвращает true, если объект поддерживает роль $role_name.

=cut

sub does_role {
    my ($self, $role_name) = @_;
    for my $class ($self->meta->linearized_isa) {
        my $meta = Mouse::Util::get_metaclass_by_name($class) or next;
        for my $role (@{$meta->roles}) {
            return 1 if $role_name =~ /::/ ? $role->name eq $role_name : $role->name =~ /::Role::\Q$role_name\E$/;
        }
    }
    return 0;
}

=head2 get_public_attributes

Возвращает список всех публичных атрибутов. Метод аналогичен $object->meta->get_all_attributes с пропуском
приватных атрибутов.

=cut

sub get_public_attributes {
    my ($self) = @_;
    return grep { $_->name !~ /^_(?:(?:(?:db_)?state)|constructed)$/ } $self->meta->get_all_attributes;
}

=head2 get_state_hash

Возвращает хеш с описанием состояния модели. Хеш, вида: {flags => {...}, changes => {...}}, где
    flags -> список установленных флагов и их значения,
    changes -> список полей модели, помеченных как измененные (при условии поддержки отслеживания изменений на поле)

=cut

sub get_state_hash {
    my ($self) = @_;

    my %state_hash = (
        flags => dclone($self->_state->{flags} // {}),
        changes => {},
    );

    while (my ($key, $is_changed) = each %{$self->_state->{changes} // {}}) {
        $state_hash{changes}->{$key} = $is_changed if $is_changed;
    }

    for my $attr ($self->meta->get_all_attributes) {
        next unless $attr->does('Yandex::ORM::Meta::Attribute::Trait::Column');
        if (%{$attr->params}) {
            my ($table, $column) = @{$attr->params}{qw/table column/};
            next unless exists $self->_db_state->{$table} && exists $self->_db_state->{$table}->{$column};
            my $is_changed = $self->_db_state->{$table}->{$column}->[0];
            $state_hash{changes}->{$attr->name} = $is_changed if $is_changed;
        }
    }

    return \%state_hash;
}

=head2 field2sql

Перевести имя поля из модели в его название в sql:

my $sql_field = Direct::Model::Foo->field2sql($foo_field) or die "unknown field $foo_field";

=cut

sub field2sql
{
    my ($class, $field) = @_;
    my $schema = $class->get_table_schema($class->meta->default_table);
    while (my ($column, $f) = each %$schema) {
        if ($f->{attr_name} eq $field || $f->{alias} eq $field) {
            return $column;
        }
    }
    return undef;
}

1;
