package Yandex::ORM::Model::Manager::Base;

use Direct::Modern;

use Mouse;

use Yandex::DBTools;
use Yandex::ListUtils qw//;
use List::MoreUtils qw//;
use Scalar::Util qw/blessed/;

has _max_items_to_insert => (is => 'ro', isa => 'Int', default => 10_000);
has _max_items_to_update => (is => 'ro', isa => 'Int', default => 500);

=head2 _insert_to_one_table_in_db($db, $table, $columns, $items)

    my @columns = Direct::Model::Banner->get_db_columns_list('banners');
    $self->_insert_to_one_table_in_db(PPC(shard => $shard), 'banners', \@columns, \@banners_in_shard);

Всталяет записи с колонками $columns для объектов $items в таблицу $table и возвращает количество вставленных записей.

Параметры:

    $db      -> база (в терминах Yandex::DBTools) с которой работаем
    $table   -> таблица, куда будем вставлять
    $columns -> список колонок, на основе которых будут формироваться записи
                (если undef, то список будет сформирован автоматически)
    $items   -> список объектов, у которых будут браться значения для соответствующих $columns
    %options:
        ignore_duplicates -> игнорировать ли дубликаты при вставке в таблицу (с помощью insert ignore)

=cut

sub _insert_to_one_table_in_db {
    my ($self, $db, $table, $columns, $items, %options) = @_;

    return unless @$items;

    state $columns_cache = {};
    if (!defined $columns) {
        my $class = blessed($items->[0]);
        $columns = ($columns_cache->{"${class}|${table}"} //= [$class->get_db_columns_list($table)]);
    }

    my @data_to_insert;
    for my $item (@$items) {
        push @data_to_insert, [map {
            my $val = $item->get_db_column_value($table, $_, default_is_ok => 1, strict => 1, extended => 1);
            defined $val->{val__dont_quote} ? $val->{val__dont_quote} : sql_quote($val->{val})
        } @$columns];
    }

    return do_mass_insert_sql($db, 
        sprintf('INSERT' . ($options{ignore_duplicates} ? ' IGNORE ' : ' ') . 'INTO %s (%s) VALUES %%s', $table, join(', ', map { sql_quote_identifier($_) } @$columns)), 
        \@data_to_insert, 
        { max_row_for_insert => $self->_max_items_to_insert, dont_quote => 1 });
}

=head2 _insert_to_secondary_table_in_db($db, $table, $table_pkey, $pkey, $columns, $items)

    my @columns = Direct::Model::AdGroupDynamic->get_db_columns_list('adgroups_dynamic');
    $self->_insert_to_secondary_table_in_db(PPC(shard => $shard), 'adgroups_dynamic', phrases => 'pid', \@columns, \@adgroups_in_shard);

Всталяет записи с колонками ($pkey, @$columns) для объектов $items в таблицу $table.
Значение первой вставляемой колонки ($pkey) ищется в модели в таблице $table_pkey.
Если в @$columns уже есть колонка $pkey, то она пропускается.

Параметры:

    $db         -> база (в терминах Yandex::DBTools) с которой работаем
    $table      -> таблица, куда будем вставлять данные
    $table_pkey -> таблица, из которой в модели будем читать значение первой колонки
    $pkey       -> название первой колонки (главный ключ)
    $columns    -> список остальных колонок, на основе которых будут формироваться записи
    $items      -> список объектов, у которых будут браться значения для соответствующих $columns
    $pkey_map   -> отображение названии колонок дочерней таблицы на родительскую (если они отличаются)

=cut

sub _insert_to_secondary_table_in_db {
    my ($self, $db, $table, $table_pkey, $pkey, $columns, $items, $pkey_map) = @_;

    return unless @$items;

    my @columns_pkey = (ref($pkey) eq 'ARRAY' ? @$pkey : $pkey);
    my %pkey_dict = map { $_ => 1 } @columns_pkey;
    my @columns_tail = (grep { !$pkey_dict{$_} } @$columns);
    my @data_to_insert;

    for my $item (@$items) {
        push @data_to_insert, [
            (map { $item->get_db_column_value($table_pkey, $pkey_map ? ($pkey_map->{$_} // $_) : $_, strict => 1) } @columns_pkey),
            map {
                my $val = $item->get_db_column_value($table, $_, default_is_ok => 1, strict => 1, extended => 1);
                defined $val->{val__dont_quote} ? $val->{val__dont_quote} : sql_quote($val->{val})
            } @columns_tail,
        ];
    }

    do_mass_insert_sql($db,
        sprintf('INSERT INTO %s (%s) VALUES %%s', $table, join(', ', map { sql_quote_identifier($_) } (@columns_pkey, @columns_tail))),
        \@data_to_insert,
        { max_row_for_insert => $self->_max_items_to_insert(), dont_quote => 1 },
    );
}

=head2 _update_one_table_in_db($db, $table, $pkey, $items, %options)

    $self->_update_one_table_in_db(PPC(shard => $shard), banners => 'bid', \@banners_in_shard);

Обновляет записи в таблице $table, для объектов $items с использованием primary-key $pkey.
Обновление происходит за один запрос, с использованием sql_case конструкции.
Для обновления выбираются только измененные колонки, по критерию is_{column}_changed.

Параметры:

    $db       -> база (в терминах Yandex::DBTools) с которой работаем
    $table    -> таблица, которую будем обновлять
    $pkey     -> главный ключ в $table, которому соответствует значение $item->id
                (или $item->$pk_field, где $pk_field = $options{pk_field} // 'id')
    $items    -> список объектов для обновления
    %options:
        where    -> hashref, дополнительное условие для sql update, в формате Yandex::DBTools
        pk_field -> имя атрибута сохраняемого объекта, которому соответствует главный ключ таблицы,
                    по-умолчанию - 'id'

=cut

sub _update_one_table_in_db {
    my ($self, $db, $table, $pkey, $items, %options) = @_;

    return unless @$items;

    my $where = $options{where} // {};
    my $id_field = $options{pk_field} // 'id';

    for my $items_chunk (Yandex::ListUtils::chunks($items, $self->_max_items_to_update)) {

        my (%item_by_id, %columns_to_update);
        for my $item (@$items_chunk) {
            next unless $item->is_changed;

            croak "duplicate item#".($item->$id_field)." found" if exists $item_by_id{$item->$id_field};
            $item_by_id{$item->$id_field} = $item;

            while (my ($column, $state) = each %{$item->_db_state->{$table}}) {
                next unless $state->[0]; # is_changed
                $columns_to_update{$column}->{$item->$id_field} = $item->get_db_column_value($table, $column, extended => 1);
            }
        }

        my %update_info;
        while (my ($column, $data) = each %columns_to_update) {
            # Все значения, в которые будет устанавливаться $column
            my @column_vals = values %$data;
            my $val0 = $column_vals[0];

            # Если $column для всех объектов устанавливается в одно и тоже значение -- обновим без использования sql_case
            my $_is_equal = sub { defined($_[0]) && defined($_[1]) ? $_[0] eq $_[1] : !defined($_[0]) && !defined($_[1]) };
            if (
                scalar(@column_vals) == scalar(keys %item_by_id) &&
                (List::MoreUtils::all { $_is_equal->($val0->{val__dont_quote}, $_->{val__dont_quote}) } @column_vals) &&
                (List::MoreUtils::all { $_is_equal->($val0->{val}, $_->{val}) } @column_vals)
            ) {
                if (defined $val0->{val__dont_quote}) {
                    $update_info{"${column}__dont_quote"} = $val0->{val__dont_quote};
                } else {
                    $update_info{$column} = $val0->{val};
                }
                next;
            }

            # Иначе - обновим через sql_case
            $update_info{"${column}__dont_quote"} = sql_case($pkey, {map {
                defined $data->{$_}->{val__dont_quote}
                ? ($_ => $data->{$_}->{val__dont_quote})
                : ($_ => sql_quote($data->{$_}->{val}))
            } keys %$data}, default__dont_quote => $column, dont_quote_value => 1);
        }

        do_update_table($db, $table, \%update_info, where => {$pkey => [keys %item_by_id], %$where}) if %update_info;
    }

    return;
}

__PACKAGE__->meta->make_immutable;

1;
