# $Id$

=head1 NAME

    Yandex::Subscriptions

=head1 DESCRIPTION

    Работа с рассылками через api Я.Доставлятора
    http://wiki.yandex-team.ru/Marketing/spec/podpiska/podkapotom/API

=cut

package Yandex::Subscriptions;


use strict;
use warnings;
use utf8;

use Encode;
use Carp;
use JSON;

use LWP::UserAgent;
use Yandex::Log;
use Yandex::CSV;




our $VERSION = '0.01';


use base qw/Exporter/;

our @EXPORT = qw(
    maillist_clear_subscribers
    maillist_add_subscribers
    maillist_send_message
    maillist_send_saved_message
    maillist_get_info
    maillist_get_templates
    maillist_get_letter_info
);

our @EXPORT_OK = qw(
);


our $SUBSCRIPTIONS_API ||= "http://subs-api.yandex-team.ru/api/1.0";
our $AUTHORIZATION_STR;

our $SUBSCRIPTIONS_LOG ||= 'maillist.log';

our $MAILLIST_BUSY = "ожидает отправки";




=head1 FUNCTIONS

=head2 maillist_clear_subscribers

    Очищает список подписчиков рассылки
    Возвращает ответ API в виде ссылки на хеш, или 0 в случае ошибки.

    maillist_clear_subscribers( $maillist_id ) or die;

=cut

sub maillist_clear_subscribers {
    my ( $slug ) = @_;

    return _maillist_command( $slug, 'clear' );
}


=head2 maillist_add_subscribers 

    Добавляет подписчиков в рассылку
    Возвращает ответ API в виде ссылки на хеш, или 0 в случае ошибки.

    maillist_add_subscribers( $maillist_id, 'aa@bb.cc', 'xx@yy.zz' );
    или
    maillist_add_subscribers( $maillist_id, { name => 'xx', email => 'xx@yy.zz' ... }, ... );

=cut

sub maillist_add_subscribers {
    my ( $slug, @emails ) = @_;

    my $post_data = {};
    if ( ref $emails[0] ) {
        # таблица
        my $first_rec = $emails[0];
        my $csv;
        if ( ref $first_rec eq 'HASH' ) {
            my $header = [ keys %$first_rec ];
            $csv = data2csv( [ $header, map { [ @$_{ @$header } ] } @emails ], { binmode => 'utf8' } );
        } else {
            $csv = data2csv( \@emails, { binmode => 'utf8' } );
        }

        $post_data->{emails_file} = [ undef, 'emails.csv', Content => $csv ];
    } else {
        # список
        $post_data->{emails} = join( q{,}, @emails );
    }
    return _maillist_command( $slug, 'subscribe-bulk', post => $post_data, multipart => 1 );
}


=head2 maillist_send_message

    Отправляет новое сообщение в список рассылки
    Возвращает ответ API в виде ссылки на хеш, или 0 в случае ошибки.

    maillist_send_message( $maillist_id, $subject, $text, $html ) or die;

=cut

sub maillist_send_message {
    my ( $slug, $subject, $text, $html ) = @_;

    my $post_data = { 
        subject => encode( 'utf8', $subject ),
        delay   => 0,
    };

    $post_data->{text} = encode( 'utf8', $text )    if $text;
    $post_data->{html} = encode( 'utf8', $html )    if $html;

    return _maillist_command( $slug, 'send', post => $post_data );
}


=head2 maillist_send_saved_message

    Отправляет сохранённое в доставляторе сообщение в список рассылки
    Возвращает ответ API в виде ссылки на хеш, или 0 в случае ошибки.

    maillist_send_saved_message( $letter_id ) or die;

=cut

sub maillist_send_saved_message {
    my ( $message_id ) = @_;
    
    return _maillist_command( $message_id, 'resend' );
}



=head2 _maillist_command 

    (for internal use)
    
    _maillist_command ( $id, $command, post => { option => $value } ) or die;
    
    Отправляет команду в API доставлятора.
    Возвращает ответ API в виде ссылки на хеш, или 0 в случае ошибки.

=cut

sub _maillist_command {
    my ( $slug, $command, %options ) = @_;

    my %command_base = (
        clear               => 'maillists',
        'subscribe-bulk'    => 'maillists',
        send                => 'maillists',
        resend              => 'letter',
    );

    croak "Unknown command $command"    unless exists $command_base{$command};
    
    my $log = Yandex::Log->new( log_file_name => $SUBSCRIPTIONS_LOG, date_suf => '%Y%m' );
    my $ua = LWP::UserAgent->new();
    $ua->default_header( Authorization => $AUTHORIZATION_STR )  if $AUTHORIZATION_STR;
    
    my $url = "$SUBSCRIPTIONS_API/$command_base{$command}/$slug/$command/";
    my $result = $ua->post( $url,
        ( delete $options{multipart} ? (Content_Type => 'form-data') : () ),
        Content => $options{post},
    );
    if ($result->is_success) {
        
        my $content = $result->content;
        
        my $decoded_result = eval { decode_json( $content ) };
        $log->out({ id => $slug, command => $command, result => $content, options => \%options, error => $@ });

        return 0 if $@;
        return 0 unless ref $decoded_result eq 'HASH';
        return 0 if exists $decoded_result->{error};
        return $decoded_result;
    } else {
        $log->out({ id => $slug, command => $command, error => $result->status_line, options => \%options });
        return 0
    }
}




=head2 maillist_get_info 

    my $info_ref = maillist_get_info( $id ) or die;

    Запрашивает информацию о рассылке.
    Вотвращает ответ API в виде хеша, или 0 в случае ошибки

=cut

sub maillist_get_info {
    my ( $slug, %options ) = @_;

    my $url = "$SUBSCRIPTIONS_API/maillists/$slug/";
    if ( ref $options{get} eq 'HASH' ) {
        $url .= q{?} . join( q{&}, map { "$_=$options{get}->{$_}" } keys %{$options{get}} );
    }

    my $ua = LWP::UserAgent->new();
    $ua->default_header( Authorization => $AUTHORIZATION_STR )  if $AUTHORIZATION_STR;
    my $result = $ua->get( $url )->content();

    my $log = Yandex::Log->new( log_file_name => $SUBSCRIPTIONS_LOG, date_suf => '%Y%m' );
    $log->out({ id => $slug, result => $result });
                            
    my $decoded_result = eval { decode_json( $result ) };

    return 0 if $@;
    return 0 unless ref $decoded_result eq 'HASH';
    return 0 if exists $decoded_result->{error};

    return $decoded_result;
}



=head2 maillist_get_templates

    my $templates = maillist_get_templates( $id ) or die;

    Запрашивает умолчальные шаблоны рассылки
    Возвращает хеш с шаблонами, или 0 в случае ошибки

=cut

sub maillist_get_templates {
    return maillist_get_info( @_, get => { fields => 'txt_template,html_template' } );
}



=head2 maillist_get_letter_info 

    my $info_ref = maillist_get_letter_info( $id ) or die;

    Запрашивает информацию о письме.
    Вотвращает ответ API в виде хеша, или 0 в случае ошибки

=cut

sub maillist_get_letter_info {
    my ( $slug ) = @_;

    my $ua = LWP::UserAgent->new();
    $ua->default_header( Authorization => $AUTHORIZATION_STR )  if $AUTHORIZATION_STR;
    my $result = $ua->get( "$SUBSCRIPTIONS_API/letter/$slug/" )->content();

    my $log = Yandex::Log->new( log_file_name => $SUBSCRIPTIONS_LOG, date_suf => '%Y%m' );
    $log->out({ id => $slug, result => $result });
                            
    my $decoded_result = eval { decode_json( $result ) };

    return 0 if $@;
    return 0 unless ref $decoded_result eq 'HASH';
    return 0 if exists $decoded_result->{error};

    return $decoded_result;
}

1;

