package YandexOffice;

# $Id$

=head1 NAME

    YandexOffice

=head1 DESCRIPTION

    Модуль для работы с табличкой офисов Яндекса

=cut

use strict;
use warnings;

use List::MoreUtils qw/any/;

use Settings;
use Yandex::DBTools;
use Yandex::I18n;
use GeoTools;
use geo_regions;

use base qw/Exporter/;
our @EXPORT = qw/
    get_yandex_offices
    get_manager_office
    get_office_by_geo
    get_officecity_by_geo_footer
    /;

use utf8;

=head2 get_yandex_offices(param => val, ...)

    Получить список офисов, соответствующих одределённому условию.
    Возможные параметры:
        office_id => XX
        office_nick => XX

=cut
sub get_yandex_offices {
    my %params = @_;
    # Формируем условие для запроса и список байндов
    my (@WHERE, @BINDS);
    while(my ($p, $v) = each %params) {
        if ($p eq 'office_id' || $p eq 'office_nick') {
            # по id
            next if !defined $v;
            push @WHERE, "$p = ?";
            push @BINDS, $v;
        } else {
            die "Unknown param '$p'";
        }
    }

    # Формируем и выполняем запрос
    my $SQL = "
            SELECT office_id, office_name,
                   office_nick, country_code, servicing_border,
                   office_short_name, serviced_regions,
                   office_phone, support_phone, sales_add_phone, office_email,
                   use_nds
              FROM yandex_offices
              ".(@WHERE ? " WHERE ".join(" AND ", @WHERE) : '')."";

    my @offices = @{get_all_sql(PPCDICT, $SQL, @BINDS)||[]};

    for my $office (@offices) {
        $office->{serviced_regions} = [ grep {length $_} map {m/(\d+)/; $1} split /\,/, $office->{serviced_regions}];
    }

    return @offices; 
}

=head2 get_manager_office($manager_uid)

    Получить офис, к которому принадлежит менеджер.
    Если офис не определён - вернуть default офис

=cut
sub get_manager_office {
    my $manager_uid = shift;
    return undef if !$manager_uid;
    my $office_id = get_one_field_sql(PPC(uid => $manager_uid), "SELECT manager_office_id FROM users_options WHERE uid = ?", $manager_uid);
    if ($office_id) {
        return [get_yandex_offices(office_id => $office_id)]->[0];
    } else {
        # по умолчанию - Москва
        return get_default_office();
    }
}

=head2 get_default_office()

    Получить "умолчательный" офис

=cut
sub get_default_office {
    # по умолчанию - Москва
    return [get_yandex_offices(office_nick => 'msk')]->[0];
}

{
my ($_office_list, $_office_by_geo);
sub get_office_by_geo {
    my $geo = shift;

    unless($_office_list) {
        $_office_list = [get_yandex_offices()];
        $_office_by_geo = {map {my $office = $_; map {$_ => $office} @{$_->{serviced_regions}} } @$_office_list};
    }
    $geo = 0 if ! exists $geo_regions::GEOREG{$geo}; 

   my $office_geo = get_geo_projection(
        $geo,
        {
            geo_list => [ keys %$_office_by_geo ],
            tree => 'ua', # TODO!!! транслокальность, после полного перехода, выбрать дерево
        }
    );

    return $_office_by_geo->{$office_geo};
}
}

=head2 get_officecity_by_geo_footer

    Метод перенесен на джава для DNA: ClientOfficeService.getOfficeContactForFooter Если правите тут, правьте и там.

    Функция, возвращающая по коду региона и домену строку, описывающую офис, 
    который обслуживает данный регион.

    По смыслу аналогична get_office_by_geo, но внутри заложена логика,
    используемая для отображения телефона в футере. К сожалению, она
    отличается от логики определения офиса для отчётов по таблице yandex_offices.
    Логика перенесена из шаблонов "как есть". В интерфейсе результат этой функции
    назывался officecity.

    Кроме того, логика показа контактов для футера также завязана на домен (на 
    национальных доменах показываем соответствующие контакты).

    $geo_region = 995; $yandex_domain = 'yandex.com';
    $office = get_officecity_by_geo_footer($geo_region, $yandex_domain);
    $office => 'ukraine'|'russia'|'msk'|'spb'|'ekb'|'ukr'|... (см. data/t/auto/office_contacts.html)

=cut

sub get_officecity_by_geo_footer {
    my ($geo_region, $yandex_domain) = @_;

    my $translocal_settings = {host => $yandex_domain};

    my $geo_city;
    if (is_targeting_in_region($geo_region, 20541, $translocal_settings)) {
        $geo_city = 'ua_odessa';
    } elsif (is_targeting_in_region($geo_region, 20544, $translocal_settings)) {
        $geo_city = 'ua_kiev';
    } elsif (is_targeting_in_region($geo_region, 187, $translocal_settings)) {
        $geo_city = 'ua';
    # Казахстан
    } elsif (is_targeting_in_region($geo_region, 159, $translocal_settings)) {
        $geo_city = 'kz';
    # Беларусь
    } elsif (is_targeting_in_region($geo_region, 149, $translocal_settings)) {
        $geo_city = 'by';
    # Россия
    } elsif (is_targeting_in_region($geo_region, 1, $translocal_settings)) {
        $geo_city = 'ru_moscow';
    } elsif (is_targeting_in_region($geo_region, 10174, $translocal_settings)) {
        $geo_city = 'ru_spb';
    } elsif (is_targeting_in_region($geo_region, 11162, $translocal_settings)) {
        $geo_city = 'ru_eburg';
    } elsif (is_targeting_in_region($geo_region, 11316, $translocal_settings)) {
        $geo_city = 'ru_novosib';
    } elsif (is_targeting_in_region($geo_region, 11079, $translocal_settings)) {
        $geo_city = 'ru_nnovgorod'; # Нижегородская область
    } elsif (is_targeting_in_region($geo_region, 3, $translocal_settings)
             && ! (is_targeting_in_region($geo_region, 10645, $translocal_settings)
                   || is_targeting_in_region($geo_region, 10672, $translocal_settings)
                  )
            )
    {
        $geo_city = 'ru_center'; # Центральный округ (минус Белгородская и Воронежская области);
    } elsif (is_targeting_in_region($geo_region, 17, $translocal_settings)) {
        $geo_city = 'ru_north'; # Северо-Запад
    } elsif (any {is_targeting_in_region($geo_region, $_, $translocal_settings)} qw/11119 11153 11131 11070 11148 11077 11156 11117/) {
        $geo_city = 'ru_tatarstan'; # Республика Татарстан, Ульяновская и Самарская области (плюс Кировская область, Удмуртия, Марий Эл, Чувашия, Мордовия)
    } elsif (any {is_targeting_in_region($geo_region, $_, $translocal_settings)} qw/52 11111 11108/) {
        $geo_city = 'ru_ural'; # Урал (плюс Башкортостан и Пермский край)
    } elsif (is_targeting_in_region($geo_region, 59, $translocal_settings) || is_targeting_in_region($geo_region, 73, $translocal_settings)) {
        $geo_city = 'ru_siberia'; # Сибирь и Дальний восток
    } elsif (is_targeting_in_region($geo_region, 40, $translocal_settings)
             && ! (any {is_targeting_in_region($geo_region, $_, $translocal_settings)} qw/11111 11108 11153 11131 11070 11148 11077 11156 11117/)
            )
    {
        $geo_city = 'ru_volga'; # Приволжский округ (минус Башкортостан, Пермский край, Ульяновская и Самарская области, Кировская, Удмуртия, Марий Эл, Чувашия, Мордовия)
    } elsif (any {is_targeting_in_region($geo_region, $_, $translocal_settings)} qw/26 10645 10672/) {
        $geo_city = 'ru_south'; # Южный округ (плюс Белгородская и Воронежская области) и Северный Кавказ (сейчас у нас Юг включает в себя Северный Кавказ: 11069, 11021, 11010, 11013, 11020, 11024, 11012)
    } elsif (is_targeting_in_region($geo_region, 225, $translocal_settings)) {
        $geo_city = 'ru';
    } else {
        $geo_city = '';
    }

    my $lang = Yandex::I18n::current_lang();
    my $officecity;
    if ($yandex_domain && $yandex_domain eq 'yandex.com.tr') {
        # Турция - отдельный продукт
        $officecity = 'tr';
    } elsif ($geo_city eq 'kz') {
        # Казахстанцев обстлуживает спец. агентство
        $officecity = 'kaz';
    } elsif ($geo_city eq 'by') {
        # Беларусь
        $officecity = 'blr';
    } elsif ($lang eq 'en') {
        $officecity = 'en'; # реально отвечают из Москвы
    } elsif ($lang eq 'ua') {
        if ($geo_city eq 'ua_kiev') {
            $officecity = 'kiev';
        } else {
            $officecity = 'ukr';
        }
    } elsif ($lang eq 'ru') {
        if ($geo_city eq 'ua' || $geo_city eq 'ua_odessa') {
            $officecity = 'ukr';
        } elsif ($geo_city eq 'ua_kiev') {
            $officecity = 'kiev';
        } elsif ($geo_city eq 'ru' || $geo_city eq 'ru_moscow' || $geo_city eq 'ru_center' || $geo_city eq 'ru_volga') {
            $officecity = 'msk';
        } elsif ($geo_city eq 'ru_spb' || $geo_city eq 'ru_north') {
            $officecity = 'spb';
        } elsif ($geo_city eq 'ru_eburg' || $geo_city eq 'ru_ural') {
            $officecity = 'ekb';
        } elsif ($geo_city eq 'ru_novosib' || $geo_city eq 'ru_siberia') {
            $officecity = 'nsk';
        } elsif ($geo_city eq 'ru_nnovgorod') {
            $officecity = 'nnovgorod';
        } elsif ($geo_city eq 'ru_south') {
            $officecity = 'rostov';
        } elsif ($geo_city eq 'ru_tatarstan') {
            $officecity = 'kazan';
        } else {
            $officecity = 'other';
        }
    } else {
        $officecity = 'other';
    }

    #print STDERR "lang=$lang, geo_city=$geo_city, yandex_domain=$yandex_domain -> officecity=$officecity\n";
    return $officecity;
}

1;
