#!/usr/bin/perl

=encoding utf8

=head1 METADATA

<crontab>
    time: */59 * * * *
    package: scripts-switchman
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 800
        </leases>
    </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:    2d
    sharded: 1
    tag: direct_group_internal_systems
</juggler>

=head1 NAME

    api5CalcUnitsLimit.pl

=head1 SYNOPSIS

    api5CalcUnitsLimit.pl --shard-id 1

=head1 DESCRIPTION

    Скрипт для начисления клиентам дневного лимита баллов API5 в зависимости от средних трат за 28 дней

=cut

use Direct::Modern;

use List::Util qw/max/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils qw/chunks/;
use Yandex::Retry;
use Yandex::TimeCommon qw/mysql_round_day today/;

use my_inc '..';

use API::Limits qw/get_limit_by_spent/;

use Property;

use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;

my $CHUNK_SIZE = 10_000;

run();



sub run {

    $log->out('start');

    extract_script_params();

    my $today = today();

    # script run once a day
    my $script_run_prop      = Property->new("DailyApi5UnitsLimit_shard_$SHARD");

    my $last_script_run_date = $script_run_prop->get() // '';
    if ( $last_script_run_date eq $today ) {
        $log->out('Already worked today, finish');
        juggler_ok(description => 'already worked today');
        exit(0);
    }

    # script should run after ImportEstimateFromYtJob
    my $last_stat_upload_datetime = Property->new('ClientsDailySpentEstimate')->get() // '';

    # convert 'YYYY-mm-ddTHH:MM:SSZ' to 'YYYYmmdd'
    my $last_stat_upload_date = mysql_round_day( $last_stat_upload_datetime );

    my $last_processed_date_prop = Property->new("Api5CalcUnitsLimitLastProcessedDate_shard_$SHARD");
    my $last_processed_date = $last_processed_date_prop->get() // '';
    if ( $last_stat_upload_date le $last_processed_date ) {
        $log->out('Stats not updated yet, finish');
        exit(0);
    }

    calc_api5_units();

    $script_run_prop->set($today);
    $last_processed_date_prop->set($last_stat_upload_date); # сохраняем новую дату обработанной статистики

    juggler_ok();

    $log->out('finish');
}

sub calc_api5_units {

    # gather old limits
    my $old_units_limit = get_hash_sql( PPC(shard => $SHARD),
                            'SELECT
                                ClientID,
                                api5_units_daily
                             FROM
                                clients_api_options
                             WHERE
                                api5_units_daily IS NOT NULL'
                          );
    $log->out('Old limit found for '. keys( %$old_units_limit ) .' clients');

    # gather spendings

    # NB: идея запроса - выбрать траты только тех клиентов, которые либо являются
    #     главными клиентами в бренде либо не входят в бренд. Причина, по которой
    #     баллы не начисляются клиентам, входящим в бренд и не являющихся там
    #     главными - баллы для таких клиентов списываются с корзины главного
    #     клиента в бренде.

    # NB: некоторые замечания по структуру запроса:
    #
    #    cs.type in ("brand", "agency")
    #       в таблице clients_stat для каждого конечного клиента по его ClientID
    #       лежит две строчки - с type = 'client' и с type = 'brand'
    #       (см. CalcEstimateJob)
    #
    #    cb.ClientID IS NULL OR cb.ClientID = cb.brand_ClientID
    #       выбираем только клиентов не входящих в бренд либо главных в бренде
    my $clients_stat = get_hashes_hash_sql( PPC(shard => $SHARD),
                        'SELECT
                            cs.ClientID,
                            cs.active_28days_sum_rub as spent,
                            cs.sum_rest_rub          as available
                         FROM
                            clients_stat cs
                            LEFT JOIN client_brands cb USING (ClientID)
                         WHERE
                            cs.type in ("brand", "agency")
                            AND (cs.active_28days_sum_rub > 0 OR cs.sum_rest_rub > 0)
                            AND (cb.ClientID IS NULL OR cb.ClientID = cb.brand_ClientID)'
                       );
    $log->out('Spendings found for '. keys( %$clients_stat ) .' clients');

    # calc new limits
    my @buf;
    my $flush_buf = sub {

        my $_rg = relaxed_guard times => 2;

        foreach_shard ClientID => \@buf, by => sub { $_->[0] }, with_undef_shard => 1, sub {
            my ( $shard, $chunk ) = @_;

            unless ( $shard ) { # 0 - resharding, undef - broken metabase
                $shard //= '';
                $log->out("Bad shard id [$shard] for ClientID: ". join ', ' => map { $_->[0] } @$chunk);
                return;
            }

            do_mass_insert_sql( PPC(shard => $shard), '
                        INSERT INTO
                            clients_api_options (ClientID, api5_units_daily)
                        VALUES
                            %s
                        ON DUPLICATE KEY UPDATE
                            api5_units_daily = VALUES(api5_units_daily)
                        ', $chunk);
        };

        @buf = ();
    };

    foreach my $ClientID ( keys %$clients_stat ) {

        my $stat = $clients_stat->{ $ClientID };

        # https://st.yandex-team.ru/DIRECT-46972
        my $sum = max( $stat->{spent} || 0, $stat->{available} || 0 );

        my $limit = get_limit_by_spent( $sum );

        push @buf, [ $ClientID, $limit ];

        delete $old_units_limit->{ $ClientID };

        if ( @buf >= $CHUNK_SIZE ) {
            $flush_buf->();
        }
    }
    $flush_buf->();
    $log->out('Limit set for '. keys( %$clients_stat ) .' clients');

    # set to NULL old limits
    my $old_limits_count = keys %$old_units_limit;

    # NB: seems that $old_limits_count will be small, so there is no need
    # for anonymous sub such the $flush_buf above
    if ( $old_limits_count ) {
        foreach my $chunk ( chunks( [ keys %$old_units_limit ], $CHUNK_SIZE ) ) {
            do_update_table( PPC(ClientID => $chunk), 'clients_api_options',
                { api5_units_daily => undef }, where => { ClientID => SHARD_IDS }
            );
        }
        $log->out('Limit set to NULL for '. $old_limits_count .' clients');
    }
}
