#!/usr/bin/perl

use Direct::Modern;

=head1 METADATA

<crontab>
    time: */2 * * * *
    package: scripts-switchman
    <switchman>
        group: scripts-other
    </switchman>
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 15m
    tag: direct_group_api
    <notification>
        template: on_status_change
        status: CRIT
        status: OK
        method: telegram
        login: APIMonitoring
    </notification>
</juggler>

<monrun>
    name: direct.api.errors.1000.rate
    juggler_host: checks_auto.direct.yandex.ru
    warn: 10
    crit: 30
    span: 20min
    expression: 'movingMaximum(sumSeries(transformNull(direct_one_min.db_configurations.production.api.errors.api5.*.*.1000,0)),10)'
    tag: direct_group_api
    <notification>
        template: on_status_change
        status: CRIT
        status: WARN
        status: OK
        method: telegram
        login: APIMonitoring
    </notification>
</monrun>

<monrun>
    name: direct.api.errors.1002.rate
    juggler_host: checks_auto.direct.yandex.ru
    warn: 150
    crit: 300
    span: 20min
    expression: 'movingMaximum(sumSeries(transformNull(direct_one_min.db_configurations.production.api.errors.api5.*.*.1002,0)),10)'
    tag: direct_group_api
    <notification>
        template: on_status_change
        status: CRIT
        status: WARN
        status: OK
        method: telegram
        login: APIMonitoring
    </notification>
</monrun>

=head1 NAME

apiErrorRequestsStat.pl

=head1 DESCRIPTION

Получает из ClickHouse статистику по ошибкам API за последние 20 минут и записывает в graphite.

=cut

use JSON;

use Yandex::Advmon;

use my_inc "..";

use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;
use Tools 'get_clickhouse_handler';

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {
    [
        qw( direct_one_min db_configurations ),
        $Settings::CONFIGURATION,
        qw/api errors/,
    ];
};

$log->out('start');

$log->out('get stat for last 20 minutes');

my $ch = get_clickhouse_handler('cloud');
my $query = qq{
    SELECT toUnixTimestamp(toStartOfMinute(log_time)) as unixtime,http_status,cmd,api_version,reqid
    FROM ppclog_api
    PREWHERE
        log_date >= toDate(now() - 1200) and log_date <= today()
        AND log_time >= minus(toStartOfMinute(now()), 1200)
        AND http_status != toUInt16('0')
    FORMAT JSON
};

my $results = $ch->query($query);

$log->out({requests => $results->json->{rows}});

my %uniq;
my %stat;
foreach my $result (@{$results->json->{data}}) {
    next if $uniq{$result->{reqid}}++;
    my ($service, $method) = split(/\./, $result->{cmd}, 2);
    my $api_version = $result->{api_version} == 5 ? 'api5' : 'api4';
    $stat{$result->{unixtime}}{$api_version}{$service // 'undefined'}{$method // 'undefined'}{$result->{http_status}}++;
}

foreach my $date_ts (keys %stat) {
    monitor_values($stat{$date_ts}, time => $date_ts);
    $log->out( { timestamp => $date_ts, values => $stat{$date_ts} } );
}

juggler_ok();

$log->out('finish');

