#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */3 * * * *
    package: scripts-switchman
    sharded: 1
    <switchman>
        group: scripts-other
    </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsMonitor.working
    sharded: 1
    raw_events:     scripts.apiReportsMonitor.working.shard_$shard
    ttl: 5m
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */3 * * * *
    sharded: 1
    only_shards: 1
    package: scripts-sandbox
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsMonitor.working.sandbox
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.apiReportsMonitor.working.sandbox.shard_$shard
    vars:           shard=1
    ttl:            10m
    tag: direct_group_internal_systems
</juggler>

#-------------------------------------------------------------------------------
# поверх этих двух проверок заведен juggler-агрегат, описанный в файле
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.stat.queue
    warn: 500
    crit: 700
    span: 30min
    expression: 'sumSeries(direct_one_min.db_configurations.production.flow.DBQueue.api_report.{New,Grabbed}.count.*)'
    tag: direct_queues
</monrun>
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.stat.maxtime
    warn: 1000
    crit: 2000
    span: 30min
    expression: 'maxSeries(direct_one_min.db_configurations.production.flow.DBQueue.api_report.{New,Grabbed}.max_age.*)'
    tag: direct_queues
</monrun>
<juggler>
    from_file: apiReportDaemon.yml
</juggler>
#-------------------------------------------------------------------------------
# поверх этих двух проверок заведен juggler-агрегат, описанный в файле
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.reports.queue
    warn: 500
    crit: 700
    span: 30min
    expression: 'sumSeries(direct_one_min.db_configurations.production.flow.DBQueue.api5_report.{New,Grabbed}.count.*)'
    tag: direct_queues
</monrun>
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.reports.maxtime
    warn: 1000
    crit: 2000
    span: 30min
    expression: 'maxSeries(direct_one_min.db_configurations.production.flow.DBQueue.api5_report.{New,Grabbed}.max_age.*)'
    tag: direct_queues
</monrun>
<juggler>
    from_file: apiReportsBuilder.yml
</juggler>
#-------------------------------------------------------------------------------
# поверх этих двух проверок заведен juggler-агрегат, описанный в файле
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.wordstat.queue
    warn: 100
    crit: 200
    span: 30min
    expression: 'sumSeries(direct_one_min.db_configurations.production.flow.DBQueue.api_wordstat.{New,Grabbed}.count.*)'
    tag: direct_queues
</monrun>
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.wordstat.maxtime
    warn: 900
    crit: 1800
    span: 30min
    expression: 'maxSeries(direct_one_min.db_configurations.production.flow.DBQueue.api_wordstat.{New,Grabbed}.max_age.*)'
    tag: direct_queues
</monrun>
<juggler>
    from_file: apiWordstatReportDaemon.yml
</juggler>
#-------------------------------------------------------------------------------
# поверх этих двух проверок заведен juggler-агрегат, описанный в файле
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.forecast.queue
    warn: 500
    crit: 1500
    span: 30min
    expression: 'sumSeries(direct_one_min.db_configurations.production.flow.DBQueue.api_forecast.{New,Grabbed}.count.*)'
    tag: direct_queues
</monrun>
<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.forecast.maxtime
    warn: 900
    crit: 1800
    span: 30min
    expression: 'maxSeries(direct_one_min.db_configurations.production.flow.DBQueue.api_forecast.{New,Grabbed}.max_age.*)'
    tag: direct_queues
</monrun>
<juggler>
    from_file: apiForecastDaemon.yml
</juggler>
#-------------------------------------------------------------------------------

<monrun>
 juggler_host:   checks_auto.direct.yandex.ru
    name: direct.api.stat.fails.reports
    warn: 10
    crit: 50
    span: 30min
    expression: 'movingAverage(sumSeries(direct_one_min.db_configurations.production.flow.DBQueue.api_report.Failed.count.*),"10min")'
    tag: direct_queues
    tag: direct_group_api
</monrun>

<monrun>
    name: direct.reports.queue
    juggler_host: checks_auto.direct.yandex.ru
    warn: 2000
    crit: 2500
    span: 10min
    expression: 'maxSeries(sumSeriesWithWildcards(direct_one_min.db_configurations.production.flow.DBQueue.{api_forecast,api_wordstat,api_report,api5_report}.{New,Grabbed}.count.*, 8))'
    tag: direct_group_api
    <notification>
        template: on_status_change
        status: CRIT
        status: WARN
        status: OK
        method: telegram
        login: APIMonitoring
    </notification>
</monrun>

=cut

=head1 NAME

apiReportsMonitor.pl

=head1 DESCRIPTION

НАЗВАНИЕ ОБМАНЫВАЕТ

На самом деле считает статистику не только по отчетам, но и вообще по всем типам заданий DBQueue

Мониторинги надо объявляють вручную для каждого типа заданий

Считает максимальный возраст и количество заданий в каждом из состояний
и посылает все данные в Графит

1) падения берём только за последние сутки
2) maxtime берём только для независших заданий

type -- необязательный параметр, значения -- типы из etc/dbqueue-types.yaml

TODO 
 - переименовать во что-то про dbqueue
 - запускать для ppcdict

=cut

use Direct::Modern;

use Time::HiRes ();

use Yandex::Advmon;
use Yandex::DBQueue;

use EnvTools;
use Settings;
use ScriptHelper 'Yandex::Log' => 'messages', sharded => 1, get_file_lock => ['dont_die'];

=head1 SUBROUTINES/METHODS/VARIABLES

=head2 $MIN_ITERATION_DURATION

    минимальное время итерации
    если итерация закончилась быстрее - делаем sleep
    кроме того используется для округления времени источника данных перед отправкой в графит

=cut

my $MIN_ITERATION_DURATION = 1 * 60;

=head2 $ITERATION_START_SECOND

    сдвиг относительно округленного до $MIN_ITERATION_DURATION времени, до которого будем спать,
    если отработали быстрее, чем $MIN_ITERATION_DURATION

=cut

my $ITERATION_START_SECOND = 26;

=head2 $REPORT_TYPE

    Для какого типа отчетов получаем статистику очереди

=cut

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION, qw/flow DBQueue/]};


my ($ONCE, $JOB_TYPE);
extract_script_params(
    once => \$ONCE,
    "type=s" => \$JOB_TYPE,
);

$log->out('START');

my $typemap = Yandex::DBQueue::Typemap->new( PPC( shard => $SHARD ) );

while (1) {
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Exiting.");
        exit 0;
    } else {
        restart_tracing();
    }

    # округляем вверх до $MIN_ITERATION_DURATION секунд
    my $queue_stats_time = time();
    my $time_for_graphite = int($queue_stats_time / $MIN_ITERATION_DURATION) * $MIN_ITERATION_DURATION;
    $time_for_graphite += $MIN_ITERATION_DURATION if $queue_stats_time % $MIN_ITERATION_DURATION;

    my $types = $JOB_TYPE ? [ $JOB_TYPE ] : $typemap->get_job_types();
    foreach my $type (@$types) {
        my $prefix_guard = $log->msg_prefix_guard("$type:");
        $log->out('get queue statistics');

        my $queue = Yandex::DBQueue->new( PPC( shard => $SHARD ), $type );
        my $stat = $queue->get_queue_statistics();

        $log->out($stat);

        # преобразуем данные для отправки в Графит, добавив в конец шард
        my %graphite_stat;
        for my $status (keys %$stat) {
            for my $value (keys %{ $stat->{$status} }) {
                $graphite_stat{$status}->{$value}->{"shard_$SHARD"} = $stat->{$status}->{$value};
            }
        }

        monitor_values({ $type => \%graphite_stat }, time => $time_for_graphite);
    }

    juggler_ok(service_suffix => (EnvTools::is_sandbox() ? 'sandbox' : undef));

    last if $ONCE;

    my $sleep_time = $time_for_graphite + $ITERATION_START_SECOND - Time::HiRes::time();
    if ($sleep_time > 0) {
        $log->out("sleep for $sleep_time seconds");
        Time::HiRes::sleep($sleep_time);
    }
}

$log->out('FINISH');
