#!/usr/bin/env perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */5 * * * *
    package: scripts-switchman
    sharded: 1
    <switchman>
        group: scripts-other
    </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsStatClean.working.api_report
    sharded:        1
    ttl:            15m
    raw_events:     scripts.apiReportsStatClean.working.api_report.shard_$shard
    tag: direct_group_internal_systems
</juggler>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsStatClean.working.api_wordstat
    sharded:        1
    ttl:            15m
    raw_events:     scripts.apiReportsStatClean.working.api_wordstat.shard_$shard
    tag: direct_group_internal_systems
</juggler>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsStatClean.working.api_forecast
    sharded:        1
    ttl:            15m
    raw_events:     scripts.apiReportsStatClean.working.api_forecast.shard_$shard
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */5 * * * *
    sharded: 1
    only_shards: 1
    package: scripts-sandbox
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsStatClean.working.sandbox.api_report
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.apiReportsStatClean.working.sandbox.api_report.shard_$shard
    vars:           shard=1
    tag: direct_group_internal_systems
</juggler>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsStatClean.working.sandbox.api_wordstat
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.apiReportsStatClean.working.sandbox.api_wordstat.shard_$shard
    vars:           shard=1
    tag: direct_group_internal_systems
</juggler>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsStatClean.working.sandbox.api_forecast
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.apiReportsStatClean.working.sandbox.api_forecast.shard_$shard
    vars:           shard=1
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */3 * * * *
    sharded: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    flock: 1
</crontab>

=cut

=head1 NAME

apiReportsStatClean.pl

=head1 DESCRIPTION

Скрипт удаления старых задач и файлов отчётов статистики в API.

=cut

use Direct::Modern;

use Yandex::DBQueue;

use Direct::Storage;
use EnvTools;
use Settings;
use ScriptHelper sharded => 1, get_file_lock => undef, 'Yandex::Log' => 'messages';
use LockTools qw/get_file_lock/;

use API::ReportCommon;

# сколько файлов можно удалить за одну итерацию
my $FILE_DELETION_LIMIT = 1_000;

my %REPORT_PARAMS = (
    report => {
        # время жизни отчета (секунд)
        REPORT_LIFE_TIME => 5 * 3_600,
        # время жизни файла отчета (секунд)
        REPORT_FILE_LIFE_TIME => 2 * 5 * 3_600,
        FILE_TYPE => ['api_report_stat'],
        # сколько записей можно удалить за одну итерацию
        DELETION_LIMIT => 500,
    },
    wordstat => {
        REPORT_LIFE_TIME => 5 * 3_600,
        REPORT_FILE_LIFE_TIME => 2 * 5 * 3_600,
        FILE_TYPE => ['api_wordstat_nameless', 'api_wordstat'],
        DELETION_LIMIT => 500,
    },
    forecast => {
        REPORT_LIFE_TIME => 5 * 3_600,
        REPORT_FILE_LIFE_TIME => 2 * 5 * 3_600,
        FILE_TYPE => ['api_forecast_nameless', 'api_forecast'],
        DELETION_LIMIT => 150,
    },
);

main();
exit;


sub main {
    $log->msg_prefix("shard=$SHARD");
    my ($ONCE, @REPORT_TYPES);
    extract_script_params(
        "once" => \$ONCE,
        "type=s" => \@REPORT_TYPES,
    );
    @REPORT_TYPES = sort keys %REPORT_PARAMS unless @REPORT_TYPES;
    if (my @err = grep {!exists $REPORT_PARAMS{$_}} @REPORT_TYPES) {
        $log->die('Incorrect report types: ' . join(', ', @err));
    }
    
    get_file_lock('dont_die', "apiReportsStatClean.$SHARD." . join('.', @REPORT_TYPES));
    $log->out('START');

    while(1) {
        restart_tracing();
        my $iter_start = time;
        for my $report_type (@REPORT_TYPES) {
            my $lg = $log->msg_prefix_guard("shard=$SHARD,type=$report_type");
    
            clean($report_type, $REPORT_PARAMS{$report_type});
        
            $log->out('Sending OK to juggler');
            my $suffix = $API::ReportCommon::JOB_TYPE_BY_REPORT_TYPE{$report_type};
            juggler_ok(service_suffix => (EnvTools::is_sandbox() ? 'sandbox.' . $suffix : $suffix));
        }
        if ($ONCE) {
            last;
        } elsif (my $reason = smart_check_stop_file()) {
            $log->out("$reason! Exiting.");
            last;
        }
        my $sleep = 60 - (time - $iter_start);
        my $profile = Yandex::Trace::new_profile('sleep');
        sleep $sleep if $sleep > 0;
    }    
    
    $log->out('FINISH');
}

sub clean {
    my ($report_type, $report_params) = @_;

    $log->out('removing old jobs');
    my $queue = Yandex::DBQueue->new( PPC( shard => $SHARD ), $API::ReportCommon::JOB_TYPE_BY_REPORT_TYPE{$report_type} );
    my $deleted_jobs = $queue->delete_old_jobs($report_params->{REPORT_LIFE_TIME}, limit => $report_params->{DELETION_LIMIT});
    $log->out({deleted_jobs => $deleted_jobs});
    
    $log->out('removing old files');
    my $storage = Direct::Storage->new();
    foreach my $file_type (@{$report_params->{FILE_TYPE}}) {
        my $deleted_files = $storage->remove_old_files($SHARD, $file_type, $report_params->{REPORT_FILE_LIFE_TIME}, limit => $FILE_DELETION_LIMIT);
        $log->out({deleted_files => scalar(@$deleted_files)});
    }
}
