#!/usr/bin/env perl

# $Id$

use my_inc "..";

=encoding utf-8

=head1 METADATA

# NB! при изменении числа инстансов - не забыть поменять juggler-проверку ниже

<crontab>
    time: */5 * * * *
    package: scripts-switchman
    sharded: 1
    params: --uniq 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 1500
        </leases>
    </switchman>
</crontab>
<crontab>
    time: */5 * * * *
    package: scripts-switchman
    sharded: 1
    params: --uniq 2
    <switchman>
        group: scripts-other
        <leases>
            mem: 1500
        </leases>
    </switchman>
</crontab>
<crontab>
    time: */5 * * * *
    package: scripts-switchman
    sharded: 1
    params: --uniq 3
    <switchman>
        group: scripts-other
        <leases>
            mem: 1500
        </leases>
    </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.apiReportsWordstat.working.uniq_$uniq.shard_$shard
    sharded:        1
    vars:           uniq=1,2,3
    ttl:            10m
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */3 * * * *
    params: --uniq 1
    sharded: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    flock: 1
</crontab>

<crontab>
    time: */5 * * * *
    params: --uniq 1
    sharded: 1
    only_shards: 1
    package: scripts-sandbox
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.apiReportsWordstat.working.sandbox
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.apiReportsWordstat.working.sandbox.uniq_$uniq.shard_$shard
    vars:           shard=1
    vars:           uniq=1
    ttl:            10m
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

apiReportsWordstat - cкрипт для создания Wordstat отчета, заказанного из API

=head1 DESCRIPTION

Параметры запуска:
    --uniq - обязательный параметр, число, позволяющее развести несколько
             скриптов на кластере ppcscripts (обрабатывается switchman)

    На машинах под switchman запускается несколько процессов для каждого шарда.

Алгоритм работы скрипта, обрабатывающего очередь apiReportsWordstat.pl:
    а. зная диапазон приоритетов (101 значение между 0 и 100 включительно), шаг группировки приоритетов и uniq номер воркера,
       определяем, сколько всего групп по приоритету и к какой группе относится воркер. Например, если шаг 50,
       то есть три группы: от 0 до 100, от 50 до 100 и ровно 100. Каждый третий воркер, начиная с первого, обрабатывает любое
       задание очереди. Каждый третий воркер, начиная со второго, обрабатывает задания с приоритетом 50 или выше, а так же
       может брать любое при отсутствии высокоприоритетных. Каждый третий воркер, начиная с третьего, обрабатывает задания с
       приоритетом 100, и он так же может брать любое при отсутствии высокоприоритетных.
    б. выбираем из базы запрос с наименьшим job_id, удовлетворяющий условию предыдущего пункта.
    в. строим отчёт и обновляем/перекладываем в архив запись в очереди.

=cut

use Direct::Modern;

use Time::HiRes qw/time/;
use Sys::Hostname qw/hostname/;
use Yandex::DBQueue;
use Yandex::ProcInfo qw/proc_memory proc_status_str/;

use Settings;
use ScriptHelper sharded => 1, get_file_lock => undef, 'Yandex::Log' => 'messages';

use API::ReportCommon qw/wordstat_common_report calculate_priority/;
use API::Settings;
use Direct::Storage;
use EnvTools;
use LockTools;

use YAML ();

use open ':std' => ':utf8';

# hostname как общий префикс id процесса
my $HOSTNAME = hostname() or die("Can't get hostname");

# максимальное количество попыток сгенерировать отчет
my $MAX_PROCESSING_TIMES = $API::Settings::MAX_PROCESSING_TIMES_REPORTS || 5;

my $PROCESS_START_TIME = time();

my $STORAGE = Direct::Storage->new;

my $JUGGLER_SERVICE_SUFFIX = '';

# счётчик итераций; когда дойдёт до $MAX_ITERATIONS, скрипт завершается
my $ITERATIONS_ALMOST_DONE = 0;

# максимальное количество итераций цикла "выбрать задачу - сделать задачу";
# итерации, в которых задачи не нашлось, считаются
my $MAX_ITERATIONS = 10000;

# Максимальное время жизни процесса (с)
my $PROCESS_LIFETIME = 3_600; # 1h

# максимальный размер памяти, отведенный под процесс
my $MAX_PROC_SIZE = 1_500_000_000; # 1.5 GB (IEC)

# сколько спать при отсутствии задания (с)
my $IDLE_SLEEP_TIME = 5;

# на сколько секунд процесс может захватить задание в очереди
my $GRAB_DURATION = 60 * 5;

my ($UNIQ, $ONCE);

extract_script_params(
    'uniq=s' => \$UNIQ,
    'once' => \$ONCE,
);
die '--uniq param is mandatory!' unless defined $UNIQ;

if ( !get_file_lock('dont die', get_script_name() . ".$UNIQ") ) {
    exit 0;
}

$log->msg_prefix("[shard_$SHARD,uniq_$UNIQ]");
$Yandex::DBQueue::LOG = $log;

$JUGGLER_SERVICE_SUFFIX .= 'sandbox.' if is_sandbox();
$JUGGLER_SERVICE_SUFFIX .= "uniq_$UNIQ";

# идентификация процесса в логах: "$hostname/$uniq/$$"
my $FULL_PROC_ID = "$HOSTNAME/$UNIQ/$$";

$log->out("start daemon $FULL_PROC_ID");

my $JOB_QUEUE = Yandex::DBQueue->new(PPC(shard => $SHARD), $API::ReportCommon::JOB_TYPE_BY_REPORT_TYPE{'wordstat'});

# Какой минимальный приоритет для задачи запрашивать из очереди
my $priority_gate = calculate_priority($UNIQ);

while (1) {
    last if (++$ITERATIONS_ALMOST_DONE > $MAX_ITERATIONS);

    my $idle = 1;

    if (my $job = $JOB_QUEUE->grab_job(grab_for => $GRAB_DURATION, filters => {minimum_priority => $priority_gate})) {
        $idle = 0;
        work_on_job($job);
    } elsif ($priority_gate) {
        $log->out("no high-priority jobs");
        if (my $job = $JOB_QUEUE->grab_job(grab_for => $GRAB_DURATION)) {
            $idle = 0;
            work_on_job($job);
        }
    }


    juggler_ok(service_suffix => $JUGGLER_SERVICE_SUFFIX);

    last if should_quit();
    idle() if $idle;

    restart_tracing('worker');
}

$log->out("end daemon $FULL_PROC_ID");

sub work_on_job {
    my ($job) = @_;

    my $job_id = $job->job_id;
    my $ClientID = $job->ClientID;

    if ($job->trycount > $MAX_PROCESSING_TIMES) {
        $log->out({error => "rank_exceed_limit", job_id => $job_id, trycount => $job->trycount});

        my $marked_ok = $job->mark_failed_permanently({error => 'Unknown error'});
        $log->out("failed to mark $job_id failed (permanently) - another process handled it?") unless $marked_ok;
        return;
    }

    $log->out("$FULL_PROC_ID process start for job $job_id");

    my $starttime = time();

    my ($data, $result) = wordstat_common_report($job->args->{Phrases}, $job->args->{GeoID});

    my $yaml = YAML::Dump($data);
    my $stored_result = {res => eval {$STORAGE->save('api_wordstat_nameless', $yaml, ClientID => $ClientID, filename => "wordstat_nameless_" . $job_id)}};

    if ($@ || !$stored_result->{res}) {
        my $error = $@;
        $stored_result = {error => $error};
        $log->out({'Failed to store file:' => $error, job_id => $job_id || 0, args => $job->args});
    } else {
        $stored_result->{ok} = 1;
        $log->out("Precision values for wordstat report $job_id: ".join(', ', map {sprintf('%0.2f', $_->{precision} || 0)} @{$result}));
    }

    my $report_generation_time = time() - $starttime;

    if ($stored_result->{ok}) {
        $log->out( 'Create report - '.$job_id );
        my $marked_ok = $job->mark_finished({
            filename => $stored_result->{res}->{filename},
            generation_time => $report_generation_time
        });

        $log->out("failed to mark $job_id finished") unless $marked_ok;
    } else {
        my $error = $stored_result->{error} || 'Unknown error';

        if ($job->trycount >= $MAX_PROCESSING_TIMES) {
            $log->out({ error => "rank_exceed_limit", job_id => $job_id, trycount => $job->trycount });

            my $marked_ok = $job->mark_failed_permanently( { error => $error } );
            $log->out( "failed to mark $job_id failed (permanently) - another process handled it?") unless $marked_ok;
        } else {
            my $marked_ok = $job->mark_failed_once;
            $log->out( "failed to mark $job_id failed once - another process handled it?") unless $marked_ok;
        }
    }

    $log->out("process time for job $job_id: " . sprintf("%.3f", ($report_generation_time))." sec, trycount = " . $job->trycount);
}

=head2 idle()

Log and do nothing for a set amount of seconds

=cut

sub idle {
    # спим, чтобы не пожрать все ресурсы, если отчетов для обработки нет
    $log->out("No report selected, sleeping for $IDLE_SLEEP_TIME seconds");
    my $profile = Yandex::Trace::new_profile("apiReportsWordstat:sleep");
    sleep $IDLE_SLEEP_TIME;
}

=head2 should_quit()

Проверки, что процесс должен завершиться, потому что:
    а) слишком много памяти ест
    б) слишком долго работает
    в) получил SIGTERM или версия Директа поменялась (использует стандартный smart_check_stop_file)
    г) сделали слишко много итераций
    д) нас просили сделать только одну итерацию

Возвращает 0 (надо работать дальше) или 1 (надо завершиться).

=cut

sub should_quit {
    return 1 if $ONCE;
    # перезапускаем процесс, если он начал занимать слишком много памяти
    if (proc_memory() > $MAX_PROC_SIZE) {
        $log->out("$FULL_PROC_ID uses too much memory, quitting: " . proc_status_str());
        return 1;
    # или процесс работает уже слишком долго (смерть от старости)
    } elsif (Time::HiRes::time() - $PROCESS_START_TIME > $PROCESS_LIFETIME) {
        $log->out("$FULL_PROC_ID is too old, quitting");
        return 1;
    # или уже сделали достаточно итераций
    } elsif ($ITERATIONS_ALMOST_DONE >= $MAX_ITERATIONS) {
        $log->out("$FULL_PROC_ID: done $MAX_ITERATIONS iterations, quitting");
        return 1;
    # или получен SIGTERM/изменилась версия Директа
    } elsif (my $reason = smart_check_stop_file()) {
        $log->out("$FULL_PROC_ID: $reason, quitting");
        return 1;
    }

    return 0;
}
