#!/usr/bin/perl
use Direct::Modern;

=head1 METADATA

<crontab>
    time: 0 * * * *
    package: scripts-switchman
    <switchman>
        group: scripts-other
    </switchman>
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 3h10m
    tag: direct_group_internal_systems
</juggler>

=head1 NAME

apiRequestStats.pl

=head1 SYNOPSIS

    # вызывается без параметров
    apiRequestStats.pl

=head1 DESCRIPTION

Получает из ClickHouse статистику по использованию API за последние 24 часа и записывает в graphite.
Гранулярность статистики: час.

=cut

use my_inc '..';

use POSIX 'strftime';

use Yandex::Advmon;
use Yandex::ListUtils;
use Yandex::TimeCommon 'mysql2unix';

use ScriptHelper 'Yandex::Log' => 'messages';
use Tools 'get_clickhouse_handler';

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {
    [
        qw( direct_one_min db_configurations ),
        $Settings::CONFIGURATION,
        qw( api requests ),
    ];
};

$log->out('start');

my $period_start = strftime( '%Y-%m-%d %H:%M:%S', localtime( time - 24 * 60 * 60 ) );
$log->out( { period_start => $period_start } );

my $clh = get_clickhouse_handler('cloud');

collect_stats();

juggler_ok();

$log->out('finish');

=head2 collect_stats

=cut

sub collect_stats {
    $log->out( { collect_stats => 'start' } );

    my $query = qq{
        SELECT
            toStartOfHour(log_time) AS time,
            api_version,
            cmd,
            count() AS count
        FROM ppclog_api
        WHERE log_time >= toStartOfHour(toDateTime('$period_start')) AND log_date >= toDate('$period_start')
        GROUP BY
            toStartOfHour(log_time),
            api_version,
            cmd
        FORMAT JSON
    };

    $log->out( { collect_stats => $query } );

    my $result = $clh->query($query);

    # time => {
    #     api_versions => { 104 => M, 4 => N, 5 => K, ... },
    #     v4 => { 'CreateOrUpdateBanners' => N, ... },
    #     v5_services => { 'Ads' => N, 'Campaigns' => N, ... },
    #     v5_methods => { 'Ads.get' => N, ... }
    # }
    my %values;

    for my $row ( @{ $result->json->{data} } ) {
        my $timestamp = mysql2unix( $row->{time} );
        $values{$timestamp} ||= {
            api_versions => {},
            v4 => {},
            v5_services => {},
            v5_methods => {},
        };

        my $api_version = $row->{api_version};

        $values{$timestamp}->{api_versions}->{$api_version} ||= 0;
        $values{$timestamp}->{api_versions}->{$api_version} += $row->{count};

        my $cmd = $row->{cmd};
        next if ! defined $cmd || $cmd eq '';

        if ( $api_version eq '4' || $api_version eq '40' || $api_version eq '104' ) {
            $values{$timestamp}->{v4}->{$cmd} ||= 0;
            $values{$timestamp}->{v4}->{$cmd} += $row->{count};
        }

        if ( $api_version eq '5' ) {
            my ($service) = split /\./, $cmd, 2;

            $values{$timestamp}->{v5_methods}->{$cmd} ||= 0;
            $values{$timestamp}->{v5_methods}->{$cmd} += $row->{count};

            $values{$timestamp}->{v5_services}->{$service} ||= 0;
            $values{$timestamp}->{v5_services}->{$service} += $row->{count};
        }
    }

    for my $timestamp ( nsort keys %values ) {
        monitor_values( $values{$timestamp}, time => $timestamp );
        $log->out( { timestamp => $timestamp, values => $values{$timestamp } } );
    }

    $log->out( { collect_stats => 'finish' } );
}
