#!/usr/bin/perl

use my_inc "..";


=encoding utf8

=head1 METADATA

<crontab>
    time: * * * * *
    sharded: 1
    <switchman>
        group:      scripts-other
        <leases>
            mem:   80
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
   time: */59 8-23/5 * * *
   <switchman>
       group:      scripts-other
       <leases>
           mem:   1000
       </leases>
   </switchman>
   package: scripts-switchman
   params: --full-sync
</crontab>
<crontab>
    time: */3 * * * *
    sharded: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>
<crontab>
    time: 18 2 * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    params: --full-sync
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    name:    scripts.balanceGetClientNDSDiscountSchedule.working.partial_sync
    sharded: 1
    tag:    direct_group_internal_systems
</juggler>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.balanceGetClientNDSDiscountSchedule.working.full_sync
    raw_events:     scripts.balanceGetClientNDSDiscountSchedule.working.full_sync
    ttl:            24h
    tag:    direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    balanceGetClientNDSDiscountSchedule.pl

=head1 DESCRIPTION

    Получает от Баланса график скидок и НДС по всем клиентам и складывает в PPC.

    При получении новых данных по клиенту, переотправляет в БК все кампании этого клиента.
    Здесь БАГ: переотправляет лишнего, нужного не переотправляет: DIRECT-89443: неправильно определяются кампании для переотправки при смене НДС

=head1 RUNNING

    Раз в минуту скрипт обрабатывает клиентов из таблицы clients_to_fetch_nds (новые недавно оплаченные клиенты).
    Раз в два часа скрипт получает полный набор данных о всех клиентах.
    В БД пишет только отличающиеся данные.
    При запуске с параметром --clientid и/или --login получает данные только по указанным клиентам.
    Получение графиков НДС всех клиентов занимает порядка 2 минут.

    LOG_TEE=1 ./protected/balanceGetClientNDSDiscountSchedule.pl --shard-id 1
    LOG_TEE=1 ./protected/balanceGetClientNDSDiscountSchedule.pl --shard-id 1 --login holodilnikru
    LOG_TEE=1 ./protected/balanceGetClientNDSDiscountSchedule.pl --shard-id 1 --clientid 12345
    LOG_TEE=1 ./protected/balanceGetClientNDSDiscountSchedule.pl --full-sync

=head1 INTERNALS

    Работа с НДС и скидками очень сходна, но имеет свои отличия. Отличия собраны в функциях и параметрах, которые задаются в
    C<%operations>.

    Работает скрипт так:
    1. Получает полный список графиков скидок/НДС для всех клиентов Баланса (кусочками), немного корректирует.
    2. Выбирает текущее состояние из PPC.
    3. Сериализует оба набора данных и сравнивает. Разницу пишет в БД.

=cut

use warnings;
use strict;

use ScriptHelper get_file_lock => undef, 'Yandex::Log' => 'messages';
use Settings;
use Yandex::DBTools;

use RBAC2::Extended;
use PrimitivesIds;
use Client::NDSDiscountSchedule;
use LockTools;

use utf8;
use open ':std' => ':utf8';

my $SHARD;
my $OPERATOR_UID = 1;
my $MAX_FETCH_TRIES = 3;
my $MAX_CLIENTS_TO_PROCESS = 1_000;

my @client_id_cmd_line;
my @client_login_cmd_line;
my $need_full_sync = 0;

extract_script_params(
    'login=s@' => \@client_login_cmd_line,
    'clientid=i@' => \@client_id_cmd_line,
    'shard-id=i' => \$SHARD,
    'full-sync' => \$need_full_sync,
);

my $SCRIPT = get_script_name(shardid => $SHARD);
get_file_lock(undef, $SCRIPT . ($need_full_sync ? '-full' : '-partial'));

die "--shard-id 1 or --full-sync should by specified" unless $need_full_sync || $SHARD;

$log->out('START');

if (@client_login_cmd_line){
    push @client_id_cmd_line, @{get_clientids(login => \@client_login_cmd_line)};
}

my $rbac = eval { RBAC2::Extended->get_singleton($OPERATOR_UID) }
    or $log->die("cannot get RBAC: $@");

my %sync_common_parameters = (
    log => $log,
    rbac => $rbac,
    timeout => 320,
    dont_die => 1,
);

my $has_errors = 0;
if (@client_id_cmd_line) {
    $need_full_sync = 0;
    $log->msg_prefix('[ClientIDs]');
    $log->out("Syncing NDS and discount for clients from command line", \@client_id_cmd_line);
    $has_errors = Client::NDSDiscountSchedule::sync_nds_schedule_for_clients(\@client_id_cmd_line, %sync_common_parameters, log_data_from_balance => 1);
} else {
    if ($need_full_sync) {
        $log->msg_prefix('[full]');
        $log->out("Doing full sync.");
        $has_errors = Client::NDSDiscountSchedule::sync_all_nds_schedules(%sync_common_parameters) || $has_errors;
        if ($has_errors) {
            juggler_warn(
                service_suffix => 'full_sync',
                description => 'Data from balance differs from stored in DB data, see balanceGetClientNDSDiscountSchedule.log',
            );
        } else {
        juggler_ok(service_suffix => 'full_sync');
        }
    } else {
        $log->msg_prefix("[shard $SHARD]");
        $log->out("Syncing only NDS for new clients: ");
        my $clients = get_all_sql(PPC(shard => $SHARD), 'SELECT ClientID, tries FROM clients_to_fetch_nds LIMIT ?', $MAX_CLIENTS_TO_PROCESS) || [];
        my (@clientids_to_delete, @clientids_to_try_again);
        for my $client (@$clients) {
            my $client_id = $client->{ClientID};
            my $tries = $client->{tries};
            $log->out("Fetching NDS graph for ClientID $client_id");
            my $success = eval {
                die "zero ClientID" if $client_id == 0;
                $has_errors = Client::NDSDiscountSchedule::sync_nds_schedule_for_clients([$client_id], %sync_common_parameters) || $has_errors;
                return 1;
            };
            if ($success) {
                $log->out("Successfully fetched NDS graph for ClientID $client_id");
                push @clientids_to_delete, $client_id;
            } else {
                $log->warn("Error fetching NDS graph for ClientID $client_id: $@");
                $tries++;
                if ($tries >= $MAX_FETCH_TRIES) {
                    push @clientids_to_delete, $client_id;
                } else {
                    push @clientids_to_try_again, $client_id;
                }
            }
        }
        if (@clientids_to_delete) {
            $log->out('Deleting ClientIDs from queue: ', \@clientids_to_delete);
            do_delete_from_table(PPC(shard => $SHARD), 'clients_to_fetch_nds', where => {ClientID => \@clientids_to_delete});
        }
        if (@clientids_to_try_again) {
            $log->out('Will try again with ClientIDs: ', \@clientids_to_try_again);
            do_update_table(PPC(shard => $SHARD), 'clients_to_fetch_nds', {tries__dont_quote => 'tries+1'}, where => {ClientID => \@clientids_to_try_again});
        }
        if ($has_errors) {
            juggler_warn(
                service_suffix => "shard_$SHARD",
                description => 'Data from balance differs from stored in DB data, see balanceGetClientNDSDiscountSchedule.log',
            );
        } else {
        juggler_ok(service_suffix => "shard_$SHARD");
    }
    }
}

$log->out('FINISH');

release_file_lock();
