#!/usr/bin/perl

use my_inc "../..";

=head1 METADATA

<crontab>
    time: * * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 250
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 15m
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    Скрипт разбирает логи обращения в интерфейс (ppclog_cmd) и 
    блокирует IP-адреса и UIDы пользователей, делающих слишком много
    запросов (роботов, как правило).

    $Id$

=cut

use warnings;
use strict;

use ScriptHelper;

use Yandex::TimeCommon;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Cbb;
use Yandex::Retry;

use Settings;
use IpTools;
use ShardingTools;
use User;
use RBAC2::Extended;
use RBACElementary;

use Tools 'get_clickhouse_handler';

use utf8;

our $FLAG = 13;
our $CHECK_PERIOD = 5 * 60; # check for last seconds
# максимальное количество произвольных хитов за $CHECK_PERIOD секунд
our $MAX_ANY_HITS = 1.5 * $CHECK_PERIOD;
# максимальное количество прогнозов хитов за $CHECK_PERIOD секунд
our $MAX_WORDSTAT_HITS = $CHECK_PERIOD / 1.5;
# время блокировки
our $BLOCK_PERIOD = 1 * 60 * 60;
# веса разных команд, по умолчанию - 1
my %CMD_WEIGHTS = (
    ajaxSpellCheck => 0.1,
    ajaxMinusPhrase => 0.1,
    ajaxGetSuggestion => 0.5,
    ajaxCheckUrl => 0.3,
    getGeoRestrictions => 0.1,
    showCaptcha => 1, # большое значение ставить нельзя - по ajax-запросам капчу могут не видеть
    );

=head2 %FORECAST_CMD_WEIGHTS

    Веса команд, обращающихся к ADVQ.
    Не указанные здесь команды в расчёте forecast_cnt не участвуют (т.е. имеют вес 0).

=cut

my %FORECAST_CMD_WEIGHTS = (
    calcForecast => 1,
    wordstat => 1,
    ForecastByWords => 1,
);

=head2 $FORECAST_SPECIAL_SQL

    SQL для обработки специальных случаев расчёта forecast_cnt
    используется sprintf:
      вместо %s поставится выражение для CASE, составленное по %FORECAST_CMD_WEIGHTS
      проценты надо удваивать (%% даст в результате %)

=cut

my $FORECAST_SPECIAL_SQL = q/if(cmd = 'ajaxGetSuggestion' and param LIKE '%%get_stat=1%%', 1, %s)/;

=head2 $ANONYMOUS_WEIGHT

    Весовой коэффициент, применяемый для запросов без авторизации

=cut

my $ANONYMOUS_WEIGHT = 1.5;

# если клиент потратил больше этой суммы - не блокируем клиента
my $SUM_SPENT_BORDER = 100;
# во сколько раз правила строже для uid по сравнению с ip
our $UID_COEF = 5;
our $CAPTCHA_FREQ = 30;
our $CAPTCHA_EXPIRES = 24 * 3600;
our $NOW = unix2mysql(time);

$log->out('START');

my $DEBUG = 0;
extract_script_params(
    "debug" => \$DEBUG,
    "now=s" => \$NOW,
);

# получаем данные, аггрегированные по ip и uid
my $log_data = get_log_data();

# находим подозрительные ip и блокируем их
my @bad_ips = get_suspicious_data($log_data, 'ip');
process_bad_ips(\@bad_ips);

# находим подозрительных uid и ставим им капчу
my @bad_uids = get_suspicious_data($log_data, 'uid', $UID_COEF);
process_bad_uids(\@bad_uids);

# удаляем старые капчи
clear_expired_captchas();

juggler_ok();

$log->out('FINISH');

# получить данные из лога, аггрегированные по ip/uid
sub get_log_data {

    my $WEIGHT_CASE_SQL = sql_case('cmd', \%CMD_WEIGHTS, default => 1, dont_quote_value => 1);
    my $FORECAST_WEIGHT_CASE_SQL = sprintf $FORECAST_SPECIAL_SQL, sql_case('cmd', \%FORECAST_CMD_WEIGHTS, default => 0, dont_quote_value => 1);
    my $ANONYMOUS_WEIGHT_SQL = qq/if(uid = 0, $ANONYMOUS_WEIGHT, 1)/;

    my $clh = get_clickhouse_handler('cloud');

    my $sql = $clh->format([
        "SELECT
            ip, uid, role,
            sum(($ANONYMOUS_WEIGHT_SQL) * ($WEIGHT_CASE_SQL)) AS cnt,
            sum(($ANONYMOUS_WEIGHT_SQL) * ($FORECAST_WEIGHT_CASE_SQL)) AS forecast_cnt
        FROM ppclog_cmd
        WHERE
                log_date = toDate(now())
            and log_time between now() - $CHECK_PERIOD and now()
            and cmd not like '\\_%'
        GROUP BY ip, uid, role"
    ]);

    $clh->query_format('JSON');

    my $res = $clh->query($sql)->json->{data};

    return $log_data;
}

# по предаггрованным данным возвращает набор uid или ip для блокировки
sub get_suspicious_data {
    my ($log_data, $field, $k) = @_;
    $k ||= 1;
    my %ROLE_K = (agency => 0.1);
    # аггрегация
    my %RES;
    for my $row (@$log_data) {
        # пропускаем внутренние сети
        next if is_internal_ip($row->{ip});
        # пропускаем некорректные данные
        next if !$row->{$field};
        my $role_k = $ROLE_K{$row->{role}||''} || 1;
        for my $data_key (qw/cnt forecast_cnt/) {
            $RES{$row->{$field}}->{$data_key} += $row->{$data_key} * $role_k;
        }
    }
    # фильтрация
    my @ret;
    while(my ($key, $val) = each %RES) {
        if ($val->{cnt} > $MAX_ANY_HITS / $k || $val->{forecast_cnt} > $MAX_WORDSTAT_HITS / $k) {
            push @ret, hash_merge({$field => $key}, $val) 
        }
    }
    return @ret;
}

# получаем плохие ip - блокируем в cbb, отправляем письма
sub process_bad_ips {
    my $bad_ips = shift;

    $bad_ips = [grep {!is_internal_ip($_->{ip})} @$bad_ips];
    return if !@$bad_ips;

    # пишем в cbb
    for my $row (@$bad_ips) {
        $log->out("block ip $row->{ip}: ($row->{cnt} hits, $row->{forecast_cnt} forecast hits in $CHECK_PERIOD sec.)");
        if (!$DEBUG) {
            retry tries => 3, pauses => [1,3], sub {
                cbb_add_range(flag => $FLAG, ip_from => $row->{ip}, ip_to => $row->{ip},
                              expire_period => $BLOCK_PERIOD, description => 'mass use direct');
            };
        }
    }
}

# получаем плохие uid - проверяем сколько денег они нам приносят, ставим капчу, отправляем письма
sub process_bad_uids {
    my $bad_uids_data = shift;
    
    my $rbac = RBAC2::Extended->get_singleton(1);

    # для каждого клиента смотрим, сколько он тратит
    for my $uid_data (@$bad_uids_data) {
        $uid_data->{role} = rbac_who_is($rbac, $uid_data->{uid});
        hash_merge $uid_data, get_user_data($uid_data->{uid}, ['is_captcha_amnested', 'captcha_freq', 'login']);
        $uid_data->{sum_spent} = get_one_field_sql(PPC(uid => $uid_data->{uid}), "
                            SELECT sum(sum_spent)
                              FROM users u1
                                   JOIN users u2 on u2.ClientID = u1.ClientID
                                   JOIN campaigns c ON c.uid = u2.uid
                             WHERE c.OrderID > 0
                               AND u1.ClientID > 0
                               AND u1.uid = ?
                            ", $uid_data->{uid}) || 0;
        $uid_data->{login_uid} = ($uid_data->{login}||'-')."/$uid_data->{uid}";
    }
    
    $bad_uids_data = [
        grep {
            $_->{sum_spent} < $SUM_SPENT_BORDER
            && $_->{role} =~ /^(empty|client)$/
            && !$_->{is_captcha_amnested}
            && !$_->{captcha_freq}
        } 
        @$bad_uids_data
        ];
    return if !@$bad_uids_data;
    
    for my $bad (@$bad_uids_data) {
        $log->out("mass use direct: $bad->{login_uid} (sum_spent: $bad->{sum_spent}) ($bad->{cnt} hits, $bad->{forecast_cnt} forecast hits in $CHECK_PERIOD sec.)");
    }

    if (!$DEBUG) {
        my $uids = [map {$_->{uid}} @$bad_uids_data];
        $log->out("set captcha to uids: ".join(',', @$uids));
        my $expires = unix2mysql(time + $CAPTCHA_EXPIRES);
        foreach_shard uid => $uids, chunk_size => 100, sub {
            my ($shard, $uids_chunk) = @_;
            do_update_table(PPC(shard => $shard), 'users', {captcha_freq => $CAPTCHA_FREQ}, where => {uid => $uids_chunk});
            do_mass_insert_sql(PPC(shard => $shard), "
                        INSERT INTO users_captcha (uid, captcha_expires)
                                VALUES %s
                                ON DUPLICATE KEY UPDATE captcha_expires = values(captcha_expires)",
                           [map {[$_, $expires]} @$uids_chunk]);
        };
    }
}

sub clear_expired_captchas {
    for my $shard (ppc_shards()) {
        my $uids = get_one_column_sql(PPC(shard => $shard), "
                    SELECT uid
                      FROM users_captcha
                     WHERE is_captcha_amnested = 0
                       AND captcha_expires < NOW()
                ") || [];
        next if !@$uids;
        $log->out("clear captcha, uids: ".join(',', @$uids));
        if (!$DEBUG) {
            do_update_table(PPC(shard => $shard), "users", {captcha_freq => 0}, where => {uid => $uids});
            do_delete_from_table(PPC(shard => $shard), "users_captcha", where => {uid => $uids});
        }
    }
}
